//+======================================================================
// $Source: /cvsroot/tango-cs/tango/api/java/fr/esrf/TangoApi/DeviceProxy.java,v $
//
// Project:   Tango
//
// Description:  java source code for the TANGO clent/server API.
//
// $Author: pascal_verdier $
//
// $Revision: 3.22 $
//
// $Log: DeviceProxy.java,v $
// Revision 3.22  2005/08/30 07:33:44  pascal_verdier
// Redundancy database connection between 2 TANGO_HOST added.
//
// Revision 3.21  2005/08/10 08:13:30  pascal_verdier
// The object returned by get_adm_dev() is the same object
// for all devices of a server (do not re-create).
//
// Revision 3.20  2005/04/26 13:27:09  pascal_verdier
// unexport added for command line.
//
// Revision 3.19  2005/03/01 08:21:45  pascal_verdier
// Attribute name set to lower case in subscribe event.
//
// Revision 3.18  2005/02/11 12:50:46  pascal_verdier
// DeviceInfo Object added (Start/Stop device dates).
//
// Revision 3.17  2005/02/10 14:07:33  pascal_verdier
// Build connection before get_attribute_list() if not already done.
//
// Revision 3.16  2005/01/19 10:17:00  pascal_verdier
// Convert NamedDevFailedList to DevFailed for single write_attribute().
//
// Revision 3.15  2005/01/06 10:16:39  pascal_verdier
// device_2 create a AttributeIfoExt on get_attribute_info_ex().
//
// Revision 3.14  2004/12/16 10:16:44  pascal_verdier
// Missing TANGO 5 features added.
//
// Revision 3.13  2004/12/07 14:29:30  pascal_verdier
// NonDbDevice exception management added.
//
// Revision 3.12  2004/12/07 09:30:29  pascal_verdier
// Exception classes inherited from DevFailed added.
//
// Revision 3.11  2004/11/30 13:06:19  pascal_verdier
// get_adm_dev() method added..
//
// Revision 3.10  2004/11/05 11:59:20  pascal_verdier
// Attribute Info TANGO 5 compatibility.
//
// Revision 3.9  2004/09/23 14:00:15  pascal_verdier
// Attribute for Devive_3Impl (Tango 5) implemented.
//
// Revision 3.8  2004/09/17 07:57:03  pascal_verdier
// Attribute for Devive_3Impl (Tango 5) implemented.
//
// Revision 3.7  2004/06/29 04:02:57  pascal_verdier
// Comments used by javadoc added.
//
// Revision 3.6  2004/05/14 14:21:33  pascal_verdier
// Add timeout at runtime.
// Some little bugs fixed.
//
// Revision 3.5  2004/03/12 13:15:23  pascal_verdier
// Using JacORB-2.1
//
// Revision 3.3  2004/03/08 11:35:40  pascal_verdier
// AttributeProxy and aliases management added.
// First revision for event management classes.
//
// Revision 3.2  2003/09/08 11:02:34  pascal_verdier
// *** empty log message ***
//
// Revision 3.1  2003/05/19 13:35:14  pascal_verdier
// Bug on transparency reconnection fixed.
// input argument modified for add_logging_target method.
//
// Revision 3.0  2003/04/29 08:03:26  pascal_verdier
// Asynchronous calls added.
// Logging related methods.
// little bugs fixed.
//
// Revision 2.0  2003/01/09 14:00:37  verdier
// jacORB is now the ORB used.
//
// Revision 1.8  2002/06/26 09:02:17  verdier
// tested with atkpanel on a TACO device
//
// Revision 1.7  2002/04/09 12:21:51  verdier
// IDL 2 implemented.
//
// Revision 1.6  2002/01/09 12:18:15  verdier
// TACO signals can be read as TANGO attribute.
//
// Revision 1.5  2001/12/10 14:19:42  verdier
// TACO JNI Interface added.
// URL syntax used for connection.
// Connection on device without database added.
//
// Revision 1.4  2001/07/04 14:06:05  verdier
// Attribute management added.
//
// Revision 1.3  2001/04/02 08:32:05  verdier
// TangoApi package has users...
//
// Revision 1.1  2001/02/02 13:03:46  verdier
// Initial revision
//
// Copyright 2001 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package fr.esrf.TangoApi;

import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import fr.esrf.TangoApi.events.*;

import java.util.*;

/**
 *	Class Description:
 *	This class manage device connection for Tango objects.
 *	It is an api between user and IDL Device object.
 *
 *	<Br><Br>
 *	<Br><b> Usage example: </b> <Br>
 *	<ul><i>
 *		String	status; <Br>
 *		DeviceProxy	dev = new DeviceProxy("sys/steppermotor/1"); <Br>
 *		try { <Br>	<ul>
 *			DeviceData	data = dev.command_inout("DevStatus"); <Br>
 *			status = data.extractString(); <Br>	</ul>
 *		} <Br>
 *		catch (DevFailed e) { <Br>	<ul>
 *			status = "Unknown status"; <Br>
 *			Except.print_exception(e); <Br>	</ul>
 *		} <Br>
 *	</ul></i>
 *
 * @author  verdier
 * @version  $Revision: 3.22 $
 */


public class DeviceProxy extends Connection implements ApiDefs
{
	
	static final private boolean	check_idl = false;

	/**
	 *	DbDevice object to make an agregat..
	 */
	private DbDevice	db_dev;

	private String		full_class_name;

	/**
	 *	Instance on administration device
	 */
	private	DeviceProxy	adm_dev = null;

	//===================================================================
	/**
	 *	Default DeviceProxy constructor. It will do nothing
	 */
	//===================================================================
	public DeviceProxy()	throws DevFailed
	{
	}
	//===================================================================
	/**
	 *	DeviceProxy constructor. It will import the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 */
	//===================================================================
	public DeviceProxy(String devname)	throws DevFailed
	{
		super(devname);
		full_class_name = "DeviceProxy("+name()+")";
	}
	//===================================================================
	/**
	 *	TangoDevice constructor. It will import the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 *	@param	ior		ior string used to import device
	 */
	//===================================================================
	public DeviceProxy(String devname, String ior)	throws DevFailed
	{
		super(devname, ior, FROM_IOR);
		full_class_name = "DeviceProxy("+name()+")";
	}
	//===================================================================
	/**
	 *	TangoDevice constructor. It will import the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 *	@param	host	host where database is running.
	 *	@param	port	port for database connection.
	 */
	//===================================================================
	public DeviceProxy(String devname, String host, String port)	throws DevFailed
	{
		super(devname, host, port);
		full_class_name = "DeviceProxy("+name()+")";
	}
	//===================================================================
	/**
	 *	Get connection on administration device.
	 */
	//===================================================================
	protected void import_admin_device(String origin)
				throws DevFailed
	{
		checkIfTango(origin);
		if (device==null && devname!=null)
			build_connection();

		//	Get connection on administration device
		if (adm_dev==null)
		{
			DServer	ds = DServer.get_instance();
			adm_dev = ds.get_adm_dev(adm_name());
			//adm_dev = new DeviceProxy(adm_name());
		}
	}
	//===========================================================
	/**
	 *	return the device name.
	 */
	//===========================================================
	public String name()
	{
		return get_name();
	}

	//===========================================================
	/**
	 *	return the device status read from CORBA attribute.
	 */
	//===========================================================
	public String status()	throws DevFailed
	{
		return status(ApiDefs.FROM_ATTR);
	}
	//===========================================================
	/**
	 *	return the device status.
	 *	@param	src	Source to read status.
	 *		It could be ApiDefs.FROM_CMD to read it from a command_inout or
	 *		ApiDefs.FROM_ATTR to read it from CORBA attribute.
	 */
	//===========================================================
	public String status(boolean src)	throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();

		if (url.protocol==TANGO)
		{
			String	status;
			try
			{
				if (src==ApiDefs.FROM_ATTR)
					status = device.status();
				else
				{
					DeviceData	argout = command_inout("Status");
					status = argout.extractString();
				}
			}
			catch(Exception e)
			{
				status = "Unknown";
				throw_dev_failed(e, "DeviceProxy.status()", false);
			}
			return status;
		}
		else
			return command_inout("DevStatus").extractString();
	}

	//===========================================================
	/**
	 *	return the device state read from CORBA attribute.
	 */
	//===========================================================
	public DevState state()	throws DevFailed
	{
		return state(ApiDefs.FROM_ATTR);
	}
	//===========================================================
	/**
	 *	return the device state.
	 *
	 *	@param	src	Source to read state.
	 *		It could be ApiDefs.FROM_CMD to read it from a command_inout or
	 *		ApiDefs.FROM_ATTR to read it from CORBA attribute.
	 */
	//===========================================================
	public DevState state(boolean src)	throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();

		if (url.protocol==TANGO)
		{
			DevState	state = DevState.UNKNOWN;
			int	retries = (transparent_reconnection)? 2 : 1;
			boolean done = false;
			for (int i=0 ; i<retries && done==false ; i++)
			{
				try
				{
					if (src==ApiDefs.FROM_ATTR)
						state = device.state();
					else
					{
						DeviceData	argout = command_inout("State");
						state  = argout.extractDevState();
					}
					done = true;
				}
				catch(Exception e)
				{
					//	Do reconnection ?
					if (e.toString().indexOf("org.omg.CORBA.TRANSIENT")>=0 && i==0)
					{
						//System.out.println("Retrying...");
						device = null;
						//ior    = null;
						build_connection();

						if (i==(retries-1))
						{
							state = DevState.UNKNOWN;
							throw_dev_failed(e, "DeviceProxy.state()", false);
						}
					}
					else
					{
						state = DevState.UNKNOWN;
						throw_dev_failed(e, "DeviceProxy.state()", false);
					}
				}
			}
			return state;
		}
		else
		{
			DeviceData	argout = command_inout("DevState");
			short		state  = argout.extractShort();
			return T2Ttypes.tangoState(state);
		}
	}

	//===========================================================
	/**
	 *	return the IDL device command_query for the specified command.
	 *
	 *	@param cmdname command name to be used for the command_query
	 *	@return the command information..
	 */
	//===========================================================
	public CommandInfo command_query(String cmdname)	throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();
		
		CommandInfo	info;
		if (url.protocol==TANGO)
		{
			//	Check IDL version
			if (device_2!=null)
			{
				DevCmdInfo_2	tmp = device_2.command_query_2(cmdname);
				info = new CommandInfo(tmp);
			}
			else
			{
				DevCmdInfo	tmp = device.command_query(cmdname);
				info = new CommandInfo(tmp);
			}
		}
		else
		{
			//	TACO device
			info = taco_device.commandQuery(cmdname);
		}
		return info;
	}










	//===========================================================
	//	The following methods are an agrega of DbDevice
	//===========================================================
	//==========================================================================
	/**
	 *	Set an alias for a device name
	 *	@param aliasname alias name.
	 */
	//==========================================================================
	public void put_alias(String aliasname)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").put_alias()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("put_alias");
		db_dev.put_alias(aliasname);
	}
	//==========================================================================
	/**
	 *	Get alias for the device
	 *	@return the alias name if found.
	 */
	//==========================================================================
	public String get_alias()
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_alias()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("get_alias");
		return db_dev.get_alias();
	}

	//==========================================================================
	/**
	 *	Query the database for the info of this device.
	 *	@return the information in a DeviceInfo.
	 */
	//==========================================================================
	public DeviceInfo get_info()
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").import_device()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		//checkIfTango("import_device");
		if (url.protocol==TANGO)
			return db_dev.get_info();
		else
			return null;
	}

	//==========================================================================
	/**
	 *	Query the database for the export info of this device.
	 *	@return the information in a DbDevImportInfo.
	 */
	//==========================================================================
	public DbDevImportInfo import_device()
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").import_device()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		//checkIfTango("import_device");
		if (url.protocol==TANGO)
			return db_dev.import_device();
		else
			return null;
	}
	//==========================================================================
	/**
	 *	Update the export info for this device in the database.
	 *	@param devinfo	Device information to export.
	 */
	//==========================================================================
	public void export_device(DbDevExportInfo devinfo)
				throws DevFailed
	{
		checkIfTango("export_device");
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").export_device()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		db_dev.export_device(devinfo);
	}
	//==========================================================================
	/**
	 *	Unexport the device in database.
	 */
	//==========================================================================
	public void unexport_device() throws DevFailed
	{
		checkIfTango("unexport_device");
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").unexport_device()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		db_dev.unexport_device();
	}
	//==========================================================================
	/**
	 *	Add/update this device to the database
	 *	@param devinfo The device name, class and server  specified in object.
	 */
	//==========================================================================
	public void add_device(DbDevInfo devinfo) throws DevFailed
	{
		checkIfTango("add_device");
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").add_device()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		db_dev.add_device(devinfo);
	}
	//==========================================================================
	/**
	 *	Delete this device from the database
	 */
	//==========================================================================
	public void delete_device() throws DevFailed
	{
		checkIfTango("delete_device");
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").add_device()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		db_dev.delete_device();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	@param wildcard	propertie's wildcard (* matches any charactere).
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public String[] get_property_list(String wildcard)
				throws DevFailed
	{
		checkIfTango("get_property_list");
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_property_list()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		return db_dev.get_property_list(wildcard);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device properties for this device.
	 *	@param propnames list of property names.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_property(String[] propnames)
				throws DevFailed
	{
		if (url.protocol==TANGO)
		{
			if (url.use_db==false)
				Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_property()");
		
			if (db_dev==null)
				db_dev = new DbDevice(devname, url.host, url.strport);
			return db_dev.get_property(propnames);
		}
		else
			return taco_device.get_property(propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a device property for this device.
	 *	@param propname property name.
	 *	@return property in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum get_property(String propname)
				throws DevFailed
	{
		if (url.protocol==TANGO)
		{
			if (url.use_db==false)
				Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_property()");
		
			if (db_dev==null)
				db_dev = new DbDevice(devname, url.host, url.strport);
			return db_dev.get_property(propname);
		}
		else
		{
			String[]	propnames = { propname };
			return taco_device.get_property(propnames)[0];
		}
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device properties for this device.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param properties list of property DbDatum objects.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_property(DbDatum[] properties)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("get_property");
		return db_dev.get_property(properties);
	}

	//==========================================================================
	/**
	 *	Insert or update a  property for this device
	 *	The property name and its value are specified by the DbDatum
	 *	
	 *	@param prop Property name and value
	 */
	//==========================================================================
	public void put_property(DbDatum prop)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").put_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("put_property");
		
		DbDatum[] properties = new DbDatum[1];
		properties[0] = prop;
		put_property(properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for this device
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_property(DbDatum[] properties)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").put_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("put_property");
		db_dev.put_property(properties);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for this device.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_property(String[] propnames)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_property");
		db_dev.delete_property(propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for this device.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_property(String propname)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_property");
		db_dev.delete_property(propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for this device.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_property(DbDatum[] properties)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_property");
		db_dev.delete_property(properties);
	}






	//============================================
	//	ATTRIBUTE PROPERTY MANAGEMENT
	//============================================

	//==========================================================================
	/**
	 *	Query the device for attribute config and returns names only.
	 *
	 *	@return attributes list for specified device
	 */
	//==========================================================================
	public String[] get_attribute_list() throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();
		if (url.protocol==TANGO)
		{
			//	Read All attribute config
			String[]		wildcard = new String[1];
			
			if (device_3!=null)
			{
				wildcard[0] = TangoConst.Tango_AllAttr_3;
			}
			else
				wildcard[0] = TangoConst.Tango_AllAttr;
			AttributeInfo[]	ac = get_attribute_config(wildcard);
			String[]		result = new String[ac.length];
			for (int i=0 ; i<ac.length ; i++)
				result[i] = ac[i].name;
			return result;
		}
		else
		{
			return taco_device.get_attribute_list();
		}
	}
	//==========================================================================
	/**
	 *	Insert or update a list of attribute properties for this device.
	 *	The property names and their values are specified by the DbAttribute array.
	 *	
	 *	@param attr attribute names and properties (names and values) array.
	 */
	//==========================================================================
	public void put_attribute_property(DbAttribute[] attr)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").put_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("put_attribute_property");
		db_dev.put_attribute_property(attr);
	}
	//==========================================================================
	/**
	 *	Insert or update an attribute properties for this device.
	 *	The property names and their values are specified by the DbAttribute array.
	 *	
	 *	@param attr attribute name and properties (names and values).
	 */
	//==========================================================================
	public void put_attribute_property(DbAttribute attr)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").put_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("put_attribute_property");
		db_dev.put_attribute_property(attr);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for this object.
	 *	@param attname attribute name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_attribute_property(String attname, String[] propnames)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_attribute_property");
		db_dev.delete_attribute_property(attname, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for this object.
	 *	@param attname attribute name.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_attribute_property(String attname, String propname)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_attribute_property");
		db_dev.delete_attribute_property(attname, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for this object.
	 *	@return properties in DbAttribute objects.
	 */
	//==========================================================================
	public void delete_attribute_property(DbAttribute attr)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_attribute_property");
		db_dev.delete_attribute_property(attr);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for this object.
	 *	@return properties in DbAttribute objects array.
	 */
	//==========================================================================
	public void delete_attribute_property(DbAttribute[] attr)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_attribute_property");
		db_dev.delete_attribute_property(attr);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device attribute
	 *	properties for this device.
	 *	@param attnames list of attribute names.
	 *	@return properties in DbAttribute objects array.
	 */
	//==========================================================================
	public DbAttribute[] get_attribute_property(String[] attnames)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("get_attribute_property");
		return db_dev.get_attribute_property(attnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a device attribute
	 *	property for this device.
	 *	@param attname attribute name.
	 *	@return property in DbAttribute object.
	 */
	//==========================================================================
	public DbAttribute get_attribute_property(String attname)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").get_attribute_property()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("get_attribute_property");
		return db_dev.get_attribute_property(attname);
	}

	//==========================================================================
	/**
	 *	Delete an attribute for this object.
	 *	@param attname attribute name.
	 */
	//==========================================================================
	public void delete_attribute(String attname)
				throws DevFailed
	{
		if (url.use_db==false)
			Except.throw_non_db_exception("Api_NonDatabaseDevice",
					"Device " + name() + " do not use database",
					"DeviceProxy(" + name() + ").delete_attribute()");
		
		if (db_dev==null)
			db_dev = new DbDevice(devname, url.host, url.strport);
		checkIfTango("delete_attribute");
		db_dev.delete_attribute(attname);
	}





	//===========================================================
	//	Attribute Methods
	//===========================================================
	//==========================================================================
	/**
	 *	Get the attributes configuration for the specified device.
	 *
	 *	@param attnames	attribute names to request config.
	 *	@return the attributes configuration.
	 */
	//==========================================================================
	public AttributeInfo[] get_attribute_info(String[] attnames) throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();

		try
		{
			AttributeInfo[]		result;
			AttributeConfig[]	ac   = null;
			AttributeConfig_2[]	ac_2 = null;
			if (url.protocol==TANGO)
			{
				//	Check IDL version
				if (device_2!=null)
				{
					ac_2 = device_2.get_attribute_config_2(attnames);
				}
				else
					ac = device.get_attribute_config(attnames);
			}
			else
				ac = taco_device.get_attribute_config(attnames);

			//	Convert AttributeConfig(_2)  object to AttributeInfo object
			int	size = (ac_2!=null)? ac_2.length : ac.length;
			result = new AttributeInfo[size];
			for (int i=0 ; i<size ; i++)
			{
				if (ac_2!=null)
					result[i] =  new AttributeInfo(ac_2[i]);
				else
					result[i] =  new AttributeInfo(ac[i]);
			}
			return result;
		}
		catch(DevFailed e)
		{
			throw e;
		}
		catch(Exception e)
		{
			e.printStackTrace();
			throw_dev_failed(e, "get_attribute_config", true);
			return null;
		}
	}
	//==========================================================================
	/**
	 *	Get the extended attributes configuration for the specified device.
	 *
	 *	@param attnames	attribute names to request config.
	 *	@return the attributes configuration.
	 */
	//==========================================================================
	public AttributeInfoEx[] get_attribute_info_ex(String[] attnames) throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();

		try
		{
			AttributeInfoEx[]	result = null;
			AttributeConfig_3[]	ac_3   = null;
			AttributeConfig_2[]	ac_2   = null;
			AttributeConfig[]	ac     = null;
			if (url.protocol==TANGO)
			{
				//	Check IDL version
				if (device_3!=null)
				{
					ac_3 = device_3.get_attribute_config_3(attnames);
				}
				else
				if (device_2!=null)
				{
					ac_2 = device_2.get_attribute_config_2(attnames);
				}
				else
					Except.throw_non_supported_exception("TangoApi_IDL_NOT_SUPPORTED",
							"Not supported by the IDL version used by device",
							new String(full_class_name+".get_attribute_info_ex()"));
			}
			else
				ac = taco_device.get_attribute_config(attnames);

			//	Convert AttributeConfig(_3)  object to AttributeInfo object
			int	size;
			if (ac_3!=null)
				size = ac_3.length;
			else
			if (ac_2!=null)
				size = ac_2.length;
			else
				size = ac.length;
			result = new AttributeInfoEx[size];
			for (int i=0 ; i<size ; i++)
				if (ac_3!=null)
					result[i] =  new AttributeInfoEx(ac_3[i]);
				else
				if (ac_2!=null)
					result[i] =  new AttributeInfoEx(ac_2[i]);
				else
				if (ac!=null)	//	Taco
					result[i] =  new AttributeInfoEx(ac[i]);
			return result;
		}
		catch(DevFailed e)
		{
			throw e;
		}
		catch(Exception e)
		{
			e.printStackTrace();
			throw_dev_failed(e, "get_attribute_config_ex", true);
			return null;
		}
	}
	//==========================================================================
	/**
	 *	Get the attributes configuration for the specified device.
	 *
	 *	@param attnames	attribute names to request config.
	 *	@return the attributes configuration.
	 *	@deprecated use get_attribute_info(String[] attnames)
	 */
	//==========================================================================
	public AttributeInfo[] get_attribute_config(String[] attnames) throws DevFailed
	{
		return get_attribute_info(attnames);
	}
	//==========================================================================
	/**
	 *	Get the attribute info for the specified device.
	 *
	 *	@param attname	attribute name to request config.
	 *	@return the attribute info.
	 */
	//==========================================================================
	public AttributeInfo get_attribute_info(String attname) throws DevFailed
	{
		String[]		attnames = ApiUtil.toStringArray(attname);
		AttributeInfo[]	ac = get_attribute_info(attnames);
		return ac[0];
	}
	//==========================================================================
	/**
	 *	Get the attribute info for the specified device.
	 *
	 *	@param attname	attribute name to request config.
	 *	@return the attribute info.
	 */
	//==========================================================================
	public AttributeInfoEx get_attribute_info_ex(String attname) throws DevFailed
	{
		String[]		attnames = ApiUtil.toStringArray(attname);
		AttributeInfoEx[]	ac = get_attribute_info_ex(attnames);
		return ac[0];
	}
	//==========================================================================
	/**
	 *	Get the attribute configuration for the specified device.
	 *
	 *	@param attname	attribute name to request config.
	 *	@return the attribute configuration.
	 *	@deprecated use get_attribute_info(String attname)
	 */
	//==========================================================================
	public AttributeInfo get_attribute_config(String attname) throws DevFailed
	{
		return get_attribute_info(attname);
	}
	//==========================================================================
	/**
	 *	Get all attributes info for the specified device.
	 *
	 *	@return all attributes info.
	 */
	//==========================================================================
	public AttributeInfo[] get_attribute_info() throws DevFailed
	{
		String[]		attnames = new String[1];
		if (device_3!=null)
			attnames[0] = TangoConst.Tango_AllAttr_3;
		else
			attnames[0] = TangoConst.Tango_AllAttr;

		AttributeInfo[]	ac = get_attribute_info(attnames);
		return ac;
	}
	//==========================================================================
	/**
	 *	Get all attributes info for the specified device.
	 *
	 *	@return all attributes info.
	 */
	//==========================================================================
	public AttributeInfoEx[] get_attribute_info_ex() throws DevFailed
	{
		String[]		attnames = new String[1];
		if (device_3!=null)
			attnames[0] = TangoConst.Tango_AllAttr_3;
		else
			attnames[0] = TangoConst.Tango_AllAttr;

		AttributeInfoEx[]	ac = get_attribute_info_ex(attnames);
		return ac;
	}
	//==========================================================================
	/**
	 *	Get all attributes configuration for the specified device.
	 *
	 *	@return all attributes configuration.
	 *	@deprecated use get_attribute_info()
	 */
	//==========================================================================
	public AttributeInfo[] get_attribute_config() throws DevFailed
	{
		return get_attribute_info();
	}
	//==========================================================================
	/**
	 *	Update the attributes info for the specified device.
	 *
	 *	@param attr the attributes info.
	 */
	//==========================================================================
	public void set_attribute_info(AttributeInfo[] attr) throws DevFailed
	{
		checkIfTango("set_attribute_config");
		if (device==null && devname!=null)
			build_connection();
		try
		{
			AttributeConfig[]	config = new AttributeConfig[attr.length];
			for (int i=0 ; i<attr.length ; i++)
				config[i] = attr[i].get_attribute_config_obj();
			device.set_attribute_config(config);
		}
		catch(DevFailed e)
		{
			throw e;
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "set_attribute_info", true);
		}
	}
	//==========================================================================
	/**
	 *	Update the attributes extended info for the specified device.
	 *
	 *	@param attr the attributes info.
	 */
	//==========================================================================
	public void set_attribute_info(AttributeInfoEx[] attr) throws DevFailed
	{
		checkIfTango("set_attribute_config");
		if (device==null && devname!=null)
			build_connection();
		try
		{
			if (device_3!=null)
			{
				AttributeConfig_3[]	config = new AttributeConfig_3[attr.length];
				for (int i=0 ; i<attr.length ; i++)
					config[i] = attr[i].get_attribute_config_obj_3();
				device_3.set_attribute_config_3(config);
			}
			else
			{
				AttributeConfig[]	config = new AttributeConfig[attr.length];
				for (int i=0 ; i<attr.length ; i++)
					config[i] = attr[i].get_attribute_config_obj();
				device.set_attribute_config(config);
			}
		}
		catch(DevFailed e)
		{
			throw e;
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "set_attribute_info", true);
		}
	}
	//==========================================================================
	/**
	 *	Update the attributes configuration for the specified device.
	 *
	 *	@param attr the attributes configuration.
	 *	@deprecated use set_attribute_info(AttributeInfo[] attr)
	 */
	//==========================================================================
	public void set_attribute_config(AttributeInfo[] attr) throws DevFailed
	{
		set_attribute_info(attr);
	}
	//==========================================================================
	/**
	 *	Read the attribute value for the specified device.
	 *
	 *	@param attname	attribute name to request value.
	 *	@return the attribute value.
	 */
	//==========================================================================
	public DeviceAttribute read_attribute(String attname)
					throws DevFailed
	{
		String[] names = ApiUtil.toStringArray(attname);
		DeviceAttribute[] attval = read_attribute(names);
		return attval[0];
	}
	//==========================================================================
	/**
	 *	return directly AttributeValue object without creation of 
	 *	DeviceAttribute object
	 */
	//==========================================================================
	//	used only by read_attribute_value() to do not re-create it every time.
	private String[]	attnames_array = null;
	public AttributeValue read_attribute_value(String attname)
					throws DevFailed
	{
		checkIfTango("read_attribute_value");
		if (device==null && devname!=null)
			build_connection();

		AttributeValue[]	attrval;
		if (attnames_array==null)
			attnames_array = new String[1];
		attnames_array[0] = attname;
		try {
 			if (device_2!=null)
				attrval = device_2.read_attributes_2(attnames_array, dev_src);
			else
				attrval = device.read_attributes(attnames_array);
			return attrval[0];
		}
		catch (DevFailed e) {
			Except.throw_connection_failed(e, "TangoApi_CANNOT_READ_ATTRIBUTE",
							"Cannot read attribute:   " + attname,
							new String(full_class_name+".read_attribute()"));
			return null;
		}
		catch (Exception e) {
			throw_dev_failed(e, "device.read_attributes()", false);
			return null;
		}
	}
	//==========================================================================
	/**
	 *	Read the attribute values for the specified device.
	 *
	 *	@param attnames	attribute names to request values.
	 *	@return the attribute values.
	 */
	//==========================================================================
	public DeviceAttribute[] read_attribute(String[] attnames)
					throws DevFailed
	{
		DeviceAttribute[]	attr;

		if (device==null && devname!=null)
			build_connection();
		//	Read attributes on device server
		AttributeValue[]	attrval = new AttributeValue[0];
		AttributeValue_3[]	attrval_3 = new AttributeValue_3[0];
		if (url.protocol==TANGO)
		{
		//	try 2 times for reconnection if requested
			boolean done = false;
			int	retries = (transparent_reconnection)? 2 : 1;
			for (int i=0 ; i<retries && done==false ; i++)
			{
				try {
					if (device_3!=null)
						attrval_3 = device_3.read_attributes_3(attnames, dev_src);
					else
					if (device_2!=null)
						attrval = device_2.read_attributes_2(attnames, dev_src);
					else
						attrval = device.read_attributes(attnames);
					done = true;
				}
				catch (DevFailed e) {
					//Except.print_exception(e);
					StringBuffer	sb = new StringBuffer(attnames[0]);
					for (int j=1 ; j<attnames.length ; j++)
						sb.append(", " + attnames[j]);
					Except.throw_connection_failed(e, "TangoApi_CANNOT_READ_ATTRIBUTE",
									"Cannot read attribute(s):   " + sb.toString(),
									new String(full_class_name+".read_attribute()"));
				}
				catch (Exception e)
				{
					if (e.toString().indexOf("org.omg.CORBA.TRANSIENT")>=0 && i==0)
					{
						device = null;
						ior    = null;
						build_connection();

						if (i==(retries-1))
							throw_dev_failed(e, "device.read_attributes()", false);
					}
					else
					{
						throw_dev_failed(e, "device.read_attributes()", false);
					}
				}
			}
			//	Build a Device Attribute Object
			//	Depends on Device_impl version
			if (device_3!=null)
			{
				attr = new DeviceAttribute[attrval_3.length];
				for (int i=0 ; i<attrval_3.length ; i++)
					attr[i] = new DeviceAttribute(attrval_3[i]);
			}
			else
			{
				attr = new DeviceAttribute[attrval.length];
				for (int i=0 ; i<attrval.length ; i++)
					attr[i] = new DeviceAttribute(attrval[i]);
			}
		}
		else
			attr = taco_device.read_attribute(attnames);
		return attr;
	}


	//==========================================================================
	/**
	 *	Write the attribute value for the specified device.
	 *
	 *	@param	devattr	attribute name and value.
	 */
	//==========================================================================
	public void write_attribute(DeviceAttribute devattr)
					throws DevFailed
	{
		checkIfTango("write_attribute");
		try
		{
			DeviceAttribute[] array = new DeviceAttribute[1];
			array[0] = devattr;
			write_attribute(array);
		}
		catch (NamedDevFailedList e) {
			NamedDevFailed	namedDF = e.elementAt(0);
			DevFailed		df = new DevFailed(namedDF.err_stack);
			//Except.print_exception(e);
			throw df;
		}
		catch (Exception e)
		{
			// e.printStackTrace();
			throw_dev_failed(e, "device.write_attributes()", false);
		}
	}
	//==========================================================================
	/**
	 *	Write the attribute values for the specified device.
	 *
	 *	@param	devattr	attribute names and values.
	 */
	//==========================================================================
	public void write_attribute(DeviceAttribute[] devattr)
					throws DevFailed
	{
		checkIfTango("write_attribute");
		if (device==null && devname!=null)
			build_connection();

		//	Build an AttributeValue IDL object array
		AttributeValue[]	attrval = new AttributeValue[devattr.length];
		for (int i=0 ; i<devattr.length ; i++)
			attrval[i] = devattr[i].getAttributeValueObject_2();
		//	write attributes on device server
		try {
			if (device_3!=null)
				device_3.write_attributes_3(attrval);
			else
				device.write_attributes(attrval);
		}
		catch (DevFailed e) {
			//Except.print_exception(e);
			throw e;
		}
		catch (MultiDevFailed e) {
			throw new NamedDevFailedList(e,
					name(), "DeviceProxy.write_attribute", "MultiDevFailed");
		}
		catch (Exception e)
		{
			 //e.printStackTrace();
			throw_dev_failed(e, "device.write_attributes()", false);
		}
	}


	//==========================================================================
	//==========================================================================
	public DeviceProxy get_adm_dev() throws DevFailed
	{
		if (adm_dev==null)
			import_admin_device("get_adm_dev");
		return adm_dev;
	}
	//==========================================================================
	/**
	 *	Polling commands.
	 */
	//==========================================================================
	//==========================================================================
	/**
	 *	Add a command to be polled for the device.
	 *	If already polled, update its polling period.
	 *
	 *	@param	objname	command/attribute name to be polled.
	 *	@param	objtype	command or attribute.
	 *	@param	period	polling period.
	 */
	//==========================================================================
	private void poll_object(String objname, String objtype, int period) throws DevFailed
	{
		DevVarLongStringArray	lsa = new DevVarLongStringArray();
		lsa.lvalue = new int[1];
		lsa.svalue = new String[3];
		lsa.svalue[0] = devname;
		lsa.svalue[1] = objtype;
		lsa.svalue[2] = objname;
		lsa.lvalue[0] = period;

		//	Send command on administration device.
		if (adm_dev==null)
			import_admin_device("poll_object");
		if (check_idl && adm_dev.get_idl_version()<2)	//	Check if IDL support command
			Except.throw_non_supported_exception("TangoApi_IDL_NOT_SUPPORTED",
							"Not supported by the IDL version used by device",
							new String(full_class_name+".poll_object()"));

		DeviceData	argin = new DeviceData();
		argin.insert(lsa);
		
		//	Try to add polling period.
		try {
			adm_dev.command_inout("AddObjPolling", argin);
		}
		catch (DevFailed e)
		{
			//	check : if already polled, just update period polling
			for (int i=0 ; i<e.errors.length ; i++)
			{
				if(e.errors[i].reason.equals("API_AlreadyPolled"))
				{
					adm_dev.command_inout("UpdObjPollingPeriod", argin);
					return;
				}
			}
			//	Not this exception then  re-throw it
			Except.throw_communication_failed(e, "TangoApi_CANNOT_POLL_OBJECT",
										"Cannot poll object " + objname,
										new String(full_class_name+".poll_object()"));
		}
	}
	//==========================================================================
	/**
	 *	Add a command to be polled for the device.
	 *	If already polled, update its polling period.
	 *
	 *	@param	cmdname	command name to be polled.
	 *	@param	period	polling period.
	 */
	//==========================================================================
	public void poll_command(String cmdname, int period) throws DevFailed
	{
		poll_object(cmdname, "command", period);
	}
	//==========================================================================
	/**
	 *	Add a attribute to be polled for the device.
	 *	If already polled, update its polling period.
	 *
	 *	@param	attname	attribute name to be polled.
	 *	@param	period	polling period.
	 */
	//==========================================================================
	public void poll_attribute(String attname, int period) throws DevFailed
	{
		poll_object(attname, "attribute", period);
	}
	//==========================================================================
	/**
	 *	Remove object of polled object list
	 *
	 *	@param	objname	object name to be removed of polled object list.
	 *	@param	objtype	object type (command or attribute).
	 */
	//==========================================================================
	private void remove_poll_object(String objname, String objtype) throws DevFailed
	{
		//	Send command on administration device.
		if (adm_dev==null)
			import_admin_device("remove_poll_object");
		if (check_idl && adm_dev.get_idl_version()<2)	//	Check if IDL support command
			Except.throw_non_supported_exception("TangoApi_IDL_NOT_SUPPORTED",
							"Not supported by the IDL version used by device",
							new String(full_class_name+".remove_poll_object()"));

		DeviceData	argin = new DeviceData();
		String[]	params = new String[3];
		params[0] = devname;
		params[1] = objtype;
		params[2] = objname;
		argin.insert(params);
		adm_dev.command_inout("RemObjPolling", argin);
	}
	//==========================================================================
	/**
	 *	Remove command of polled object list
	 *
	 *	@param	cmdname	command name to be removed of polled object list.
	 */
	//==========================================================================
	public void stop_poll_command(String cmdname) throws DevFailed
	{
		remove_poll_object(cmdname, "command");
	}
	//==========================================================================
	/**
	 *	Remove attribute of polled object list
	 *
	 *	@param	attname	attribute name to be removed of polled object list.
	 */
	//==========================================================================
	public void stop_poll_attribute(String attname) throws DevFailed
	{
		remove_poll_object(attname, "attribute");
	}
	//==========================================================================
	/**
	 *	Returns the polling status for the device.
	 */
	//==========================================================================
	public String[] polling_status() throws DevFailed
	{
		//	Send command on administration device.
		if (adm_dev==null)
			import_admin_device("polling_status");
		if (check_idl && adm_dev.get_idl_version()<2)	//	Check if IDL support command
			Except.throw_non_supported_exception("TangoApi_IDL_NOT_SUPPORTED",
							"Not supported by the IDL version used by device",
							new String(full_class_name+".polling_status()"));
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		DeviceData	argout = adm_dev.command_inout("DevPollStatus", argin);
		String[]	results = argout.extractStringArray();
		return results;
	}
	//==========================================================================
	/**
	 *	Return the history for command polled.
	 *
	 *	@param	cmdname	command name to read polled history.
	 *	@param	nb		nb data to read.
	 */
	//==========================================================================
	public DeviceDataHistory[] command_history(String cmdname, int nb) throws DevFailed
	{
		checkIfTango("command_history");
		if (device==null && devname!=null)
			build_connection();
		
		if (check_idl && get_idl_version()<2)	//	Check if IDL support command
			Except.throw_non_supported_exception("TangoApi_IDL_NOT_SUPPORTED",
							"Not supported by the IDL version used by device",
							new String(full_class_name+".command_history()"));

		DevCmdHistory[]		cmd_histo =
								device_2.command_inout_history_2(cmdname, nb);
		DeviceDataHistory[]	histo = new DeviceDataHistory[cmd_histo.length];
		for(int i=0 ; i<cmd_histo.length ; i++)
			histo[i] =  new DeviceDataHistory(cmdname, cmd_histo[i]);
		return histo;
	}
	//==========================================================================
	/**
	 *	Return the history for attribute polled.
	 *
	 *	@param	attname	attribute name to read polled history.
	 *	@param	nb		nb data to read.
	 */
	//==========================================================================
	public DeviceDataHistory[] attribute_history(String attname, int nb) throws DevFailed
	{
		checkIfTango("attribute_history");
		if (device==null && devname!=null)
			build_connection();
		
		if (check_idl && get_idl_version()<2)	//	Check if IDL support command
			Except.throw_non_supported_exception("TangoApi_IDL_NOT_SUPPORTED",
							"Not supported by the IDL version used by device",
							new String(full_class_name+".attribute_history()"));

		DeviceDataHistory[]	histo = new DeviceDataHistory[0];
		if (device_3!=null)
		{
			DevAttrHistory_3[]	att_histo =
				device_3.read_attribute_history_3(attname, nb);
			
			histo = new DeviceDataHistory[att_histo.length];
			for(int i=0 ; i<att_histo.length ; i++)
				histo[i] =  new DeviceDataHistory(att_histo[i]);
		}
		else	
		if (device_2!=null)
		{
			DevAttrHistory[]	att_histo =
				device_2.read_attribute_history_2(attname, nb);
			histo = new DeviceDataHistory[att_histo.length];
			for(int i=0 ; i<att_histo.length ; i++)
				histo[i] =  new DeviceDataHistory(att_histo[i]);
		}
		return histo;
	}

	//==========================================================================
	/**
	 *	Return the full history for command polled.
	 *
	 *	@param	cmdname	command name to read polled history.
	 */
	//==========================================================================
	public DeviceDataHistory[] command_history(String cmdname) throws DevFailed
	{
		int		hist_depth = 10;
		DbDatum	data = get_property("poll_ring_depth");
		if (data.is_empty()==false)
			hist_depth = data.extractLong();
		return command_history(cmdname, hist_depth);
	}
	//==========================================================================
	/**
	 *	Return the full history for attribute polled.
	 *
	 *	@param	attname	attribute name to read polled history.
	 */
	//==========================================================================
	public DeviceDataHistory[] attribute_history(String attname) throws DevFailed
	{
		int		hist_depth = 10;
		DbDatum	data = get_property("poll_ring_depth");
		if (data.is_empty()==false)
			hist_depth = data.extractLong();
		return attribute_history(attname, hist_depth);
	}





	//==========================================================================
	/**
	 * 	Asynchronous calls
	 */
	//==========================================================================
	//==========================================================================
	/**
	 *	Asynchronous command_inout.
	 *
	 *	@param	cmdname	command name.
	 *	@param	data_in	input argument command.
	 */
	//==========================================================================
	public int command_inout_asynch(String cmdname, DeviceData data_in)
				throws DevFailed
	{
		return command_inout_asynch(cmdname, data_in, false);
	}
	//==========================================================================
	/**
	 *	Asynchronous command_inout.
	 *
	 *	@param	cmdname	command name.
	 */
	//==========================================================================
	public int command_inout_asynch(String cmdname)
				throws DevFailed
	{
		return command_inout_asynch(cmdname, new DeviceData(), false);
	}
	//==========================================================================
	/**
	 *	Asynchronous command_inout.
	 *
	 *	@param	cmdname	command name.
	 *	@param	forget	forget the response if true
	 */
	//==========================================================================
	public int command_inout_asynch(String cmdname, boolean forget)
				throws DevFailed
	{
		return command_inout_asynch(cmdname, new DeviceData(), forget);
	}
	//==========================================================================
	/**
	 *	Asynchronous command_inout.
	 *
	 *	@param	cmdname	command name.
	 *	@param	data_in	input argument command.
	 *	@param	forget	forget the response if true
	 */
	//==========================================================================
	public int command_inout_asynch(String cmdname, DeviceData data_in, boolean forget)
				throws DevFailed
	{
		checkIfTango("command_inout_asynch");
		if (device==null && devname!=null)
			build_connection();
		
		//	Create the request object
		//----------------------------------
		Request request;
		if (idl_version<2)
			request = device._request("command_inout");
		else
			request = device_2._request("command_inout");

		ORB	orb = ApiUtil.get_orb();

		request.add_in_arg().insert_string(cmdname);
		request.add_in_arg().insert_any(data_in.any);
		request.set_return_type(orb.get_primitive_tc(org.omg.CORBA.TCKind.tk_any));
		request.exceptions().add(DevFailedHelper.type());

		//	send it (defered or just one way)
		int	id = 0;
		//	Else tango call
		boolean done = false;
		//	try 2 times for reconnection if requested
		int	retries = (transparent_reconnection)? 2 : 1;
		for (int i=0 ; i<retries && done==false ; i++)
		{
			try
			{
				if (forget)
					request.send_oneway();
				else
				{
					request.send_deferred();
					//	store request reference to read reply later
					String[]	names = new String[1];
					names[0] = cmdname;
					id = ApiUtil.put_async_request(new AsyncCallObject(request, this, CMD, names));
				}
				done = true;
			}
			catch(Exception e)
			{
				if (e.toString().indexOf("org.omg.CORBA.TRANSIENT")>=0 && i==0)
				{
					//System.out.println("Retrying...");
					device = null;
					//ior    = null;
					build_connection();

					if (i==(retries-1))
						throw_dev_failed(e, cmdname, true);
				}
				else
					throw_dev_failed(e, cmdname, true);
			}
		}
		return id;
	}
	//==========================================================================
	/**
	 *	Asynchronous command_inout using callback for reply.
	 *
	 *	@param	cmdname	Command name.
	 *	@param	argin	Input argument command.
	 *	@param	cb		a CallBack object instance.
	 */
	//==========================================================================
	public void command_inout_asynch(String cmdname, DeviceData argin, CallBack cb)
				throws DevFailed
	{
		int id = command_inout_asynch(cmdname, argin, false);
		ApiUtil.set_async_reply_model(id, CALLBACK);
		ApiUtil.set_async_reply_cb(id, cb);

		//	if push callback, start a thread to do it
		if (ApiUtil.get_asynch_cb_sub_model()==PUSH_CALLBACK)
		{
			AsyncCallObject aco = ApiUtil.get_async_object(id);
			new CallbackThread(aco).start();
		}
	}
	//==========================================================================
	/**
	 *	Asynchronous command_inout using callback for reply.
	 *
	 *	@param	cmdname	Command name.
	 *	@param	cb		a CallBack object instance.
	 */
	//==========================================================================
	public void command_inout_asynch(String cmdname, CallBack cb)
				throws DevFailed
	{
		command_inout_asynch(cmdname, new DeviceData(), cb);
	}



	//==========================================================================
	/**
	 *	Read Asynchronous command_inout reply.
	 *
	 *	@param	id	asynchronous call id (returned by command_inout_asynch).
	 *	@param	timeout	number of millisonds to wait reply before throw an excption.
	 */
	//==========================================================================
	public DeviceData command_inout_reply(int id, int timeout)
			throws DevFailed, AsynReplyNotArrived
	{
		return command_inout_reply(ApiUtil.get_async_object(id), timeout);
	}
	//==========================================================================
	/**
	 *	Read Asynchronous command_inout reply.
	 *
	 *	@param	request	asynchronous call Request instance
	 *	@param	timeout	number of millisonds to wait reply before throw an excption.
	 */
	//==========================================================================
	DeviceData command_inout_reply(AsyncCallObject aco, int timeout)
			throws DevFailed, AsynReplyNotArrived
	{
		DeviceData			argout = null;
		int					ms_to_sleep = 50;
		AsynReplyNotArrived	except = null;
		long	t0 = System.currentTimeMillis();
		long	t1 = t0;
		
		//System.out.println("command_inout_reply to= " + timeout + " ms");
		while (((t1-t0)<timeout || timeout==0 ) && argout==null)
		{
			try
			{
				argout = command_inout_reply(aco);
			}
			catch (AsynReplyNotArrived na)
			{
				except = na;
				//	Wait a bit before retry
				this.sleep(ms_to_sleep);
				t1 = System.currentTimeMillis();
				//System.out.println(" " + (t1-t0) + " ms");
			}
			catch (DevFailed e)
			{
				throw e;
			}
		}
		//	If reply not arrived throw last exception
		if (argout==null && except!=null)
			throw except;

		return argout;
	}
	//==========================================================================
	/**
	 *	Read Asynchronous command_inout reply.
	 *
	 *	@param	id	asynchronous call id (returned by command_inout_asynch).
	 */
	//==========================================================================
	public DeviceData command_inout_reply(int id)
			throws DevFailed, AsynReplyNotArrived
	{
		return command_inout_reply(ApiUtil.get_async_object(id));
	}
	//==========================================================================
	/**
	 *	Read Asynchronous command_inout reply.
	 *
	 *	@param	request	asynchronous call Request instance
	 */
	//==========================================================================
	DeviceData command_inout_reply(AsyncCallObject aco)
			throws DevFailed, AsynReplyNotArrived
	{
		DeviceData	data = null;

		check_asynch_reply(aco.request, aco.id, "command_inout");

		//	If no exception, extract the any from return value,
		Any	any = aco.request.return_value().extract_any();	

		//	And put it in a DeviceData object
		data = new DeviceData();
		data.any = any;
		ApiUtil.remove_async_request(aco.id);

		return data;
	}
	//==========================================================================
	/**
	 *	Asynchronous read_attribute.
	 *
	 *	@param	attname	Attribute name.
	 */
	//==========================================================================
	public int read_attribute_asynch(String attname)
				throws DevFailed
	{
		String[]	attnames = new String[1];
		attnames[0] = attname;
		return read_attribute_asynch(attnames);
	}
	//==========================================================================
	/**
	 *	Asynchronous read_attribute.
	 *
	 *	@param	attnames	Attribute names.
	 */
	//==========================================================================
	public int read_attribute_asynch(String[] attnames)
				throws DevFailed
	{
		checkIfTango("read_attributes_asynch");
		if (device==null && devname!=null)
			build_connection();
		
		//	Create the request object
		//----------------------------------
		Request request;
		if (idl_version<2)
			request = device._request("read_attributes");
		else
			request = device_2._request("read_attributes");

		Any	any = request.add_in_arg();
		DevVarStringArrayHelper.insert(any, attnames);

		request.set_return_type(AttributeValueListHelper.type());
		request.exceptions().add(DevFailedHelper.type());

		//	send it (defered or just one way)
		request.send_deferred();
		//	store request reference to read reply later
		int	id = ApiUtil.put_async_request(new AsyncCallObject(request, this, ATT_R, attnames));
		
		return id;
	}
	//==========================================================================
	/**
	 *	Retrieve the command/attribute arguments to build exception description.
	 */
	//==========================================================================
	protected String get_asynch_idl_cmd(Request request, String idl_cmd)
	{
		NVList	args = request.arguments();
		StringBuffer	sb = new StringBuffer();
		try
		{
			if (idl_cmd.equals("command_inout"))
				return args.item(0).value().extract_string();	//	get command name
			else
			{
				//	read_attribute, get attribute names
				String[]	s_array = DevVarStringArrayHelper.extract(args.item(0).value());
				for (int i=0 ; i<s_array.length ; i++)
				{
					sb.append(s_array[i]);
					if (i<s_array.length-1)
						sb.append(", ");
				}
			}
		}
		catch(org.omg.CORBA.Bounds e) {}
		return sb.toString();
	}
	//==========================================================================
	/**
	 *	Check Asynchronous call reply.
	 *
	 *	@param	id	asynchronous call id (returned by read_attribute_asynch).
	 */
	//==========================================================================
	protected void check_asynch_reply(Request request, int id, String idl_cmd)
		throws DevFailed, AsynReplyNotArrived
	{
		//	Check if request object has been found
		if (request==null)
			Except.throw_connection_failed(	"TangoApi_CommandFailed",
									"Asynchronous call id not found",
									new String(full_class_name+"."+idl_cmd + "_reply()"));

		if (request.operation().equals(idl_cmd)==false)
			Except.throw_connection_failed(	"TangoApi_CommandFailed",
									new String("Asynchronous call id not for " + idl_cmd),
									new String(full_class_name+"."+idl_cmd + "_reply()"));
		
		//	Reply arrived ? Throw exception if not yet arrived
		if (request.poll_response() == false)
		{
			//System.out.println("Response not yet arrived");
			Except.throw_asyn_reply_not_arrived("API_AsynReplyNotArrived",
					new String("Device " + devname + 
								": reply for asynchronous call (id = "+
								id + ") is not yet arrived"),
						new String(full_class_name+"."+ idl_cmd + "_reply()"));
		}
		else
		{
			//	Check if an exception has been thrown
			Exception	except = request.env().exception();
			if (except!=null)
			{
				//	Check if user exception (DevFailed).
				if (except instanceof org.omg.CORBA.UnknownUserException)
				{
					Any	any = ((org.omg.CORBA.UnknownUserException)except).except;	
					DevFailed	e = DevFailedHelper.extract(any);
					Except.throw_connection_failed(e,
										"TangoApi_CommandFailed",
										"Asynchronous command failed",
										new String(full_class_name+"."+idl_cmd + "_reply("+
												get_asynch_idl_cmd(request, idl_cmd)+")"));
				}
				else
				{
					ApiUtil.remove_async_request(id);
					//	Another exception -> re-throw it as a DevFailed
					throw_dev_failed(except,
							new String(full_class_name+"."+idl_cmd + "_reply("+
										get_asynch_idl_cmd(request, idl_cmd)+")"), false);
				}
			}
			//System.out.println("Reply is arrived !!!");
		}
	}
	//==========================================================================
	/**
	 *	Read Asynchronous read_attribute reply.
	 *
	 *	@param	id	asynchronous call id (returned by read_attribute_asynch).
	 *	@param	timeout	number of millisonds to wait reply before throw an excption.
	 */
	//==========================================================================
	public DeviceAttribute[] read_attribute_reply(int id, int timeout)
				throws DevFailed, AsynReplyNotArrived
	{
		DeviceAttribute[]	argout = null;
		int					ms_to_sleep = 50;
		AsynReplyNotArrived	except = null;
		long	t0 = System.currentTimeMillis();
		long	t1 = t0;

		while (((t1-t0)<timeout || timeout==0 ) && argout==null)
		{
			try
			{
				argout = read_attribute_reply(id);
			}
			catch(AsynReplyNotArrived na)
			{
				except = na;
				//	Wait a bit before retry
				this.sleep(ms_to_sleep);
				t1 = System.currentTimeMillis();
				//System.out.println(" " + (t1-t0) + " ms");
			}
			catch (DevFailed e)
			{
				throw e;
			}
		}
		//	If reply not arrived throw last exception
		if (argout==null && except!=null)
			throw except;

		return argout;
	}
	//==========================================================================
	/**
	 *	Read Asynchronous read_attribute reply.
	 *
	 *	@param	id	asynchronous call id (returned by read_attribute_asynch).
	 */
	//==========================================================================
	public DeviceAttribute[] read_attribute_reply(int id)
			throws DevFailed, AsynReplyNotArrived
	{
		DeviceAttribute[]	data = null;
		Request				request = ApiUtil.get_async_request(id);

		check_asynch_reply(request, id, "read_attributes");


		//	If no exception, extract the any from return value,
		Any	any = request.return_value();
		AttributeValue[]	attl = AttributeValueListHelper.extract(any);
		data = new DeviceAttribute[attl.length];
		for (int i=0 ; i<attl.length ; i++)
			data[i] = new DeviceAttribute(attl[i]);

		ApiUtil.remove_async_request(id);

		return data;
	}
	//==========================================================================
	/**
	 *	Asynchronous read_attribute using callback for reply.
	 *
	 *	@param	attname	attribute name.
	 *	@param	cb		a CallBack object instance.
	 */
	//==========================================================================
	public void read_attribute_asynch(String attname, CallBack cb)
				throws DevFailed
	{
		String[]	attnames = new String[1];
		attnames[0] = attname;
		read_attribute_asynch(attnames, cb);
	}
	//==========================================================================
	/**
	 *	Asynchronous read_attribute using callback for reply.
	 *
	 *	@param	attnames	attribute names.
	 *	@param	cb			a CallBack object instance.
	 */
	//==========================================================================
	public void read_attribute_asynch(String[] attnames, CallBack cb)
				throws DevFailed
	{
		int id = read_attribute_asynch(attnames);
		ApiUtil.set_async_reply_model(id, CALLBACK);
		ApiUtil.set_async_reply_cb(id, cb);

		//	if push callback, start a thread to do it
		if (ApiUtil.get_asynch_cb_sub_model()==PUSH_CALLBACK)
		{
			AsyncCallObject aco = ApiUtil.get_async_object(id);
			new CallbackThread(aco).start();
		}
	}

	//==========================================================================
	/**
	 *	Asynchronous write_attribute.
	 *
	 *	@param	attr	Attribute value (name, writing value...)
	 */
	//==========================================================================
	public int write_attribute_asynch(DeviceAttribute attr)
				throws DevFailed
	{
		return write_attribute_asynch(attr, false);
	}
	//==========================================================================
	/**
	 *	Asynchronous write_attribute.
	 *
	 *	@param	attr	Attribute value (name, writing value...)
	 *	@param	forget	forget the response if true
	 */
	//==========================================================================
	public int write_attribute_asynch(DeviceAttribute attr, boolean forget)
				throws DevFailed
	{
		DeviceAttribute[]	attribs = new DeviceAttribute[1];
		attribs[0] = attr;
		return write_attribute_asynch(attribs);
	}
	//==========================================================================
	/**
	 *	Asynchronous write_attribute.
	 *
	 *	@param	attribs	Attribute values (name, writing value...)
	 */
	//==========================================================================
	public int write_attribute_asynch(DeviceAttribute[] attribs)
				throws DevFailed
	{
		return write_attribute_asynch(attribs, false);
	}
	//==========================================================================
	/**
	 *	Asynchronous write_attribute.
	 *
	 *	@param	attribs	Attribute values (name, writing value...)
	 *	@param	forget	forget the response if true
	 */
	//==========================================================================
	public int write_attribute_asynch(DeviceAttribute[] attribs, boolean forget)
				throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();

		//	Build idl argin object
		AttributeValue[] attval  = new AttributeValue[attribs.length];
		String[]		attnames = new String[attribs.length];
		for (int i=0 ; i<attribs.length ; i++)
		{
			attval[i] = attribs[i].getAttributeValueObject_2();
			attnames[i] = attval[i].name;
		}
		
		//	Create the request object
		//----------------------------------
		Request request;
		if (idl_version<2)
			request = device._request("write_attributes");
		else
			request = device_2._request("write_attributes");

		Any	any = request.add_in_arg();
		AttributeValueListHelper.insert(any, attval);
		request.exceptions().add(DevFailedHelper.type());

		//	send it (defered or just one way)
		int	id = 0;
		if (forget)
			request.send_oneway();
		else
		{
			request.send_deferred();
			//	store request reference to read reply later
			id = ApiUtil.put_async_request(new AsyncCallObject(request, this, ATT_W, attnames));
		}

		return id;
	}
	//==========================================================================
	/**
	 *	check for Asynchronous write_attribute reply.
	 *
	 *	@param	id	asynchronous call id (returned by read_attribute_asynch).
	 */
	//==========================================================================
	public void write_attribute_reply(int id)
			throws DevFailed, AsynReplyNotArrived
	{
		DeviceAttribute[]	data = null;
		Request				request = ApiUtil.get_async_request(id);

		check_asynch_reply(request, id, "write_attributes");
	}
	//==========================================================================
	/**
	 *	check for Asynchronous write_attribute reply.
	 *
	 *	@param	id	asynchronous call id (returned by write_attribute_asynch).
	 *	@param	timeout	number of millisonds to wait reply before throw an excption.
	 */
	//==========================================================================
	public void write_attribute_reply(int id, int timeout)
			throws DevFailed, AsynReplyNotArrived
	{
		int					ms_to_sleep = 50;
		AsynReplyNotArrived	except = null;
		long	t0 = System.currentTimeMillis();
		long	t1 = t0;
		boolean	done = false;
		while (((t1-t0)<timeout || timeout==0 ) && !done)
		{
			try
			{
				write_attribute_reply(id);
				done = true;
			}
			catch(AsynReplyNotArrived na)
			{
				except = na;
				//	Wait a bit before retry
				this.sleep(ms_to_sleep);
				t1 = System.currentTimeMillis();
				//System.out.println(" " + (t1-t0) + " ms");
			}
			catch(DevFailed e)
			{
				throw e;
			}
		}
		//	If reply not arrived throw last exception
		if (except!=null && !done)
			throw except;
	}
	//==========================================================================
	/**
	 *	Asynchronous write_attribute using callback for reply.
	 *
	 *	@param	attr	Attribute values (name, writing value...)
	 *	@param	cb		a CallBack object instance.
	 */
	//==========================================================================
	public void write_attribute_asynch(DeviceAttribute attr, CallBack cb)
				throws DevFailed
	{
		DeviceAttribute[]	attribs = new DeviceAttribute[1];
		attribs[0] = attr;
		write_attribute_asynch(attribs, cb);
	}
	//==========================================================================
	/**
	 *	Asynchronous write_attribute using callback for reply.
	 *
	 *	@param	attribs	Attribute values (name, writing value...)
	 *	@param	cb		a CallBack object instance.
	 */
	//==========================================================================
	public void write_attribute_asynch(DeviceAttribute[] attribs, CallBack cb)
				throws DevFailed
	{
		int id = write_attribute_asynch(attribs);
		ApiUtil.set_async_reply_model(id, CALLBACK);
		ApiUtil.set_async_reply_cb(id, cb);

		//	if push callback, start a thread to do it
		if (ApiUtil.get_asynch_cb_sub_model()==PUSH_CALLBACK)
		{
			AsyncCallObject aco = ApiUtil.get_async_object(id);
			new CallbackThread(aco).start();
		}
	}
	//==========================================================================
	/**
	 *	return the still pending asynchronous call for a reply model.
	 *
	 *	@param	reply_model	ApiDefs.ALL_ASYNCH, POLLING or CALLBACK.
	 */
	//==========================================================================
	public int pending_asynch_call(int reply_model)
	{
		return ApiUtil.pending_asynch_call(this, reply_model);
	}
	//==========================================================================
	/**
	 *	Fire callback methods for all 
	 *	asynchronous requests(cmd and attr) with already arrived replies.
	 */
	//==========================================================================
	public void get_asynch_replies()
	{	
		ApiUtil.get_asynch_replies(this);
	}
	//==========================================================================
	/**
	 *	Fire callback methods for all 
	 *	asynchronous requests(cmd and attr) with already arrived replies.
	 */
	//==========================================================================
	public void get_asynch_replies(int timeout)
	{	
		ApiUtil.get_asynch_replies(this, timeout);
	}



	//==========================================================================
	/**
	 * 	Logging related methods
	 */
	//==========================================================================
	//==========================================================================
	/**
	 *	Adds a new logging target to the device.
	 *
	 *	@deprecated use add_logging_target(String target).
	 */
	//==========================================================================
	public void add_logging_target(String target_type, String target_name)
				throws DevFailed
	{
		//	Get connection on administration device
		if (adm_dev==null)
			import_admin_device("add_logging_target");

		//	Prepeare data
		String[]		target = new String[2];
		target[0] = get_name();
		target[1] = target_type + "::" + target_name;
		DeviceData	argin = new DeviceData();
		argin.insert(target);
		//	And send command
		adm_dev.command_inout("AddLoggingTarget", argin);
	}
	//==========================================================================
	/**
	 *	Adds a new logging target to the device.
	 *
	 *	@param target The target for logging (e.g. file::/tmp/logging_device).
	 */
	//==========================================================================
	public void add_logging_target(String target)
				throws DevFailed
	{
		//	Get connection on administration device
		if (adm_dev==null)
			import_admin_device("add_logging_target");

		//	Prepeare data
		
		String[]		str = new String[2];
		str[0] = get_name();
		str[1] = target;
		DeviceData	argin = new DeviceData();
		argin.insert(str);
		//	And send command
		adm_dev.command_inout("AddLoggingTarget", argin);
	}
	//==========================================================================
	/**
	 *	Removes a new logging target to the device.
	 */
	//==========================================================================
	public void remove_logging_target(String target_type, String target_name)
				throws DevFailed
	{
		//	Get connection on administration device
		if (adm_dev==null)
			import_admin_device("remove_logging_target");

		//	Prepeare data
		String[]		target = new String[2];
		target[0] = get_name();
		target[1] = target_type + "::" + target_name;
		DeviceData	argin = new DeviceData();
		argin.insert(target);
		//	And send command
		adm_dev.command_inout("RemoveLoggingTarget", argin);
	}
	//==========================================================================
	/**
	 *	get logging target from the device.
	 */
	//==========================================================================
	public String[] get_logging_target()
				throws DevFailed
	{
		//	Get connection on administration device
		if (adm_dev==null)
			import_admin_device("get_logging_target");

		//	Prepeare data
		DeviceData	argin = new DeviceData();
		argin.insert(get_name());
		//	And send command
		DeviceData	argout = adm_dev.command_inout("GetLoggingTarget", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	get logging level from the device.
	 *	@return device's logging level:
	 *		(ApiDefs.LOGGING_OFF, ApiDefs.LOGGING_FATAL, ApiDefs.LOGGING_ERROR,
	 *		ApiDefs.LOGGING_INFO, ApiDefs.LOGGING_DEBUG)
	 */
	//==========================================================================
	public int get_logging_level()
				throws DevFailed
	{
		//	Get connection on administration device
		if (adm_dev==null)
			import_admin_device("get_logging_level");
		
		//	Prepeare data
		String[]	target = new String[1];
		target[0] = get_name();
		DeviceData	argin = new DeviceData();
		argin.insert(target);
		//	And send command
		DeviceData	argout = adm_dev.command_inout("GetLoggingLevel", argin);
		DevVarLongStringArray	lsa = argout.extractLongStringArray();		
		return lsa.lvalue[0];
	}
	//==========================================================================
	/**
	 *	Set logging level from the device.
	 *	@param level device's logging level:
	 *		(ApiDefs.LOGGING_OFF, ApiDefs.LOGGING_FATAL, ApiDefs.LOGGING_ERROR,
	 *		ApiDefs.LOGGING_INFO, ApiDefs.LOGGING_DEBUG)
	 */
	//==========================================================================
	public void set_logging_level(int level)
				throws DevFailed
	{
		//	Get connection on administration device
		if (adm_dev==null)
			import_admin_device("set_logging_level");
		
		//	Prepeare data
		DevVarLongStringArray	lsa = new DevVarLongStringArray();
		lsa.lvalue = new int[1];
		lsa.svalue = new String[1];
		lsa.lvalue[0] = level;
		lsa.svalue[0] = get_name();
		DeviceData	argin = new DeviceData();
		argin.insert(lsa);
		//	And send command
		adm_dev.command_inout("SetLoggingLevel", argin);
	}




	//==========================================================================
	/**
	 * 	TACO commands
	 */
	//==========================================================================
	//==========================================================================
	/**
	 *	Returns  TACO device information.
	 *
	 *	@return  TACO device information as String array.
	 *	<li> Device name.
	 *	<li> Class name
	 *	<li> Device type
	 *	<li> Device server name
	 *	<li> Host name
	 */
	//==========================================================================
	public String[] dev_inform() throws DevFailed
	{
		checkIfTaco("dev_inform");
		if (device==null && devname!=null)
			build_connection();
		return taco_device.dev_inform();
	}

	//==========================================================================
	/**
	 *	Execute the dev_rpc_protocol TACO command to change RPC protocol mode.
	 *	@param	mode RPC protocol mode to be seted 
	 *		(TangoApi.TacoDevice.<b>D_TCP</b> or TangoApi.TacoDevice.<b>D_UDP</b>).
	 */
	//==========================================================================
	public void set_rpc_protocol(int mode)	throws DevFailed
	{
		checkIfTaco("dev_rpc_protocol");
		if (device==null && devname!=null)
			build_connection();
		taco_device.set_rpc_protocol(mode);
	}

	//==========================================================================
	/**
	 *	@returns	mode RPC protocol mode  used 
	 *		(TangoApi.TacoDevice.<b>D_TCP</b> or TangoApi.TacoDevice.<b>D_UDP</b>).
	 */
	//==========================================================================
	public int get_rpc_protocol()	throws DevFailed
	{
		checkIfTaco("get_rpc_protocol");
		if (device==null && devname!=null)
			build_connection();
		return taco_device.get_rpc_protocol();
	}
	
	//==========================================================================
	/**
	 *	Just a main method to check API methods.
	 */
	//==========================================================================
	public static void main (String args[])
	{
		String	devname = null;
		String	cmdname = null;
		try
		{
			cmdname = args[0];
			devname = args[1];
		}
		catch(Exception e)
		{
			if (cmdname==null)
			{
				System.out.println("Usage :");
				System.out.println("fr.esrf.TangoApi.DeviceProxy  cmdname devname");
				System.out.println("	- cmdname : command name (ping, state, status, unexport...)");
				System.out.println("	- devname : device name to send command.");
			}
			else
				System.out.println("Device name ?");
			System.exit(0);
		}
		try
		{
			//	Check if wildcard
			String[]		devnames;
			DeviceProxy[]	dev;
			if (devname.indexOf("*")<0)
			{
				devnames = new String[1];
				devnames[0] = devname;
			}
			else
				devnames = ApiUtil.get_db_obj().getDevices(devname);

			//	Create DeviceProxy Objects
			dev = new DeviceProxy[devnames.length];
			for (int i=0 ; i<devnames.length ; i++)
				dev[i] = new DeviceProxy(devnames[i]);
			
			if (cmdname.equals("ping"))
			{	
				while (true)
				{
					for (int i=0 ; i<dev.length ; i++)
					{
						try {
							long t = dev[i].ping();
							System.out.println(devnames[i] + " is alive  (" + t/1000 + " ms)");
						}
						catch(DevFailed e) {
							System.out.println(devnames[i] +
											"  " + e.errors[0].desc);
						}
					}
					System.out.println();
					try { Thread.sleep(1000); }
					catch(InterruptedException e) {}
				}
			}
			else
			if (cmdname.equals("status"))
				for (int i=0 ; i<dev.length ; i++)
				{
					try {
						System.out.println(devnames[i] + " - " +
								dev[i].status());
					}
					catch(DevFailed e) {
						System.out.println(devnames[i] +
										"  " + e.errors[0].desc);
					}
				}
			else
			if (cmdname.equals("state"))
			{
				for (int i=0 ; i<dev.length ; i++)
				{
					try {
						System.out.println(devnames[i] + " is " +
									ApiUtil.stateName(dev[i].state()));
					}
					catch(DevFailed e) {
						System.out.println(devnames[i] +
										"  " + e.errors[0].desc);
					}
				}
			}
			else
			if (cmdname.equals("unexport"))
			{
				for (int i=0 ; i<dev.length ; i++)
				{
					try {
						dev[i].unexport_device();
						System.out.println(devnames[i] + " unexported !");
					}
					catch(DevFailed e) {
						System.out.println(devnames[i] +
										"  " + e.errors[0].desc);
					}
				}
			}
			else
				System.out.println(cmdname + " ?   Unknow command !");
		}
		catch(DevFailed e)
		{
			Except.print_exception(e);
			//e.printStackTrace();
		}
	}
	//===============================================================
	//===============================================================
	private synchronized void sleep(long ms)
	{
		try
		{
			wait(ms);
		}
		catch(InterruptedException e) { System.out.println(e);}
	}
	//==========================================================================
	/**
	 *	Subscribe to an event.
	 *
	 *	@param	attr_name	attribute name.
	 *	@param	event		event name.
	 *	@param  callback	event callback.
	 */
	//==========================================================================
	public int subscribe_event(String attr_name, int event, CallBack callback, String[] filters)
                   throws DevFailed
	{
        if (ApiUtil.get_event_consumer() == null)
        {
            ApiUtil.create_event_consumer();
        }
        EventConsumer event_consumer = ApiUtil.get_event_consumer();
		return event_consumer.subscribe_event(this, attr_name.toLowerCase(), event, callback, filters);
	}
	//==========================================================================
	/**
	 *	Unsubscribe to an event.
	 *
	 *	@param	event_id	event identifier.
	 */
	//==========================================================================
	public void unsubscribe_event(int event_id)
                   throws DevFailed
    {
        if (ApiUtil.get_event_consumer() == null)
        {
            ApiUtil.create_event_consumer();
        }
        EventConsumer event_consumer = ApiUtil.get_event_consumer();
        event_consumer.unsubscribe_event(event_id);
    }
}
