//+======================================================================
// $Source: /cvsroot/tango-cs/tango/api/java/fr/esrf/TangoApi/DeviceDataHistory.java,v $
//
// Project:   Tango
//
// Description:  java source code for the TANGO clent/server API.
//
// $Author: pascal_verdier $
//
// $Revision: 3.9 $
//
// $Log: DeviceDataHistory.java,v $
// Revision 3.9  2005/06/13 09:05:18  pascal_verdier
// Minor bugs fixed.
//
// Revision 3.8  2004/12/16 10:16:44  pascal_verdier
// Missing TANGO 5 features added.
//
// Revision 3.7  2004/12/07 09:30:29  pascal_verdier
// Exception classes inherited from DevFailed added.
//
// Revision 3.6  2004/05/14 14:21:33  pascal_verdier
// Add timeout at runtime.
// Some little bugs fixed.
//
// Revision 3.5  2004/03/12 13:15:21  pascal_verdier
// Using JacORB-2.1
//
// Revision 3.0  2003/04/29 08:03:28  pascal_verdier
// Asynchronous calls added.
// Logging related methods.
// little bugs fixed.
//
// Revision 2.0  2003/01/09 14:00:37  verdier
// jacORB is now the ORB used.
//
// Revision 1.8  2002/06/26 09:02:17  verdier
// tested with atkpanel on a TACO device
//
// Revision 1.7  2002/04/09 12:21:51  verdier
// IDL 2 implemented.
//
//
// Copyright 2001 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package fr.esrf.TangoApi;

import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import java.io.*;
import java.util.*;


/**
 *	Class Description:
 *	This class manage data object for Tango device history Data access.
 *	<Br><Br>
 *	<Br><b> Usage example: </b> <Br>
 *	<ul><i>
 *		DeviceDataHistory[]	histo = dev.command_history("ReadCurrent", 10); <Br>
 *		for (int i=0 ; i < histo.length ; i++) <Br>
 *		{	<Br><ul>
 *			Date		d = new Date(histo[i].getTime()); <Br>
 *			double[]	values = histo[i].extractDoubleArray(); <Br></ul>
 *		} <Br>
 *	</ul></i>
 *
 * @author  verdier
 * @version  $Revision: 3.9 $
 */

public class DeviceDataHistory
{
	private	Any					any;
	private DevCmdHistory		cmd_histo;
	private AttributeValue_3	attrval;
	private TimeVal				tval;

	/**
	 *	History from command history.
	 */
	public static final int	COMMAND   = 0;
	/**
	 *	History from attribute history.
	 */
	public static final int	ATTRIBUTE = 1;

	/**
	 *	Data source DeviceDataHistory.COMMAND or DeviceDataHistory.ATTRIBUTE
	 */
	public int				source;
	/**
	 *	Command/Attribute name.
	 */
	public String			name;
	/**
	 *	true if command/attribute failed.
	 */
	public boolean			failed;
	/**
	 *	Error list if any in reading Command/Attribute.
	 */
	public DevError[]		errors;
	//===========================================
	/**
	 *	Constructor from a DevCmdHistory.
	 */
	//===========================================
	public DeviceDataHistory(String cmdname, DevCmdHistory cmd_hist) throws DevFailed
	{
		any       = cmd_hist.value;
		source    = COMMAND;
		cmd_histo = cmd_hist;
		name   = cmdname;
		tval   = cmd_histo.time;
		failed = cmd_histo.cmd_failed;
		errors = cmd_histo.errors;
	}
	//===========================================
	/**
	 *	Constructor from an AttributeValue.
	 */
	//===========================================
	public DeviceDataHistory(DevAttrHistory att_histo) throws DevFailed
	{
		any     = att_histo.value.value;
		source  = ATTRIBUTE;
		attrval = new AttributeValue_3(any, 
					att_histo.value.quality,
					att_histo.value.time,
					att_histo.value.name,
					new AttributeDim(att_histo.value.dim_x, att_histo.value.dim_y),
					new AttributeDim(0, 0),
					att_histo.errors);
		name    = att_histo.value.name;
		tval    = att_histo.value.time;
		failed  = att_histo.attr_failed;
		errors  = att_histo.errors;
	}

	//===========================================
	/**
	 *	Constructor from an AttributeValue for Device_3Impl.
	 */
	//===========================================
	public DeviceDataHistory(DevAttrHistory_3 att_histo) throws DevFailed
	{
		any     = att_histo.value.value;
		source  = ATTRIBUTE;
		attrval = att_histo.value;
		name    = att_histo.value.name;
		tval    = att_histo.value.time;
		failed  = att_histo.attr_failed;
		errors  = att_histo.value.err_list;
	}
		
	//===========================================
	/**
	 *	Return attribute time value.
	 */
	//===========================================
	public TimeVal getTimeVal()
	{
		return tval;
	}
	//===========================================
	/**
	 *	Return attribute time value in seconds since EPOCH.
	 */
	//===========================================
	public long getTimeValSec()
	{
		return (long)tval.tv_sec;
	}
	//===========================================
	/**
	 *	return time in milliseconds since EPOCH
	 *	to build a Date class.
	 */
	//===========================================
	public long getTime()
	{
		return  (long)tval.tv_sec * 1000 + tval.tv_usec/1000;
	}
	//===========================================
	/**
	 *	return AttrQuality if from attribute.
	 */
	//===========================================
	public AttrQuality getAttrQuality() throws DevFailed
	{
		if (source==COMMAND)
			Except.throw_non_supported_exception("TangoApi_NOT_AVAILABLE",
									"Method not avalaible for command",
									"DeviceDataHistory.getAttrQuality()");
		return attrval.quality;
	}
	//===========================================
	/**
	 *	Return attribute dim_x if from attribute.
	 */
	//===========================================
	public int getDimX() throws DevFailed
	{
		if (source==COMMAND)
			Except.throw_non_supported_exception("TangoApi_NOT_AVAILABLE",
									"Method not avalaible for command",
									"DeviceDataHistory.getDimX()");
		return attrval.r_dim.dim_x;
	}
	//===========================================
	/**
	 *	Return attribute dim_y if from attribute.
	 */
	//===========================================
	public int getDimY() throws DevFailed
	{
		if (source==COMMAND)
			Except.throw_non_supported_exception("TangoApi_NOT_AVAILABLE",
									"Method not avalaible for command",
									"DeviceDataHistory.getDimY()");
		return attrval.r_dim.dim_y;
	}



	//**********	Extract Methods for basic types	*********************


	//===========================================
	/**
	 *	extract method for a CORBA Any.
	 */
	//===========================================
	public Any extractAny()
	{
		return any;
	}
	//===========================================
	/**
	 *	extract method for a boolean.
	 */
	//===========================================
	public boolean extractBoolean()
	{
		return DevBooleanHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a short.
	 */
	//===========================================
	public short extractShort()
	{
		if (source==ATTRIBUTE)
		{
			short[]	array = extractShortArray();
			return array[0];
		}
		else
			return DevShortHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for an unsigned short.
	 */
	//===========================================
	public short extractUShort()
	{
		return DevUShortHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a long.
	 */
	//===========================================
	public int extractLong()
	{
		if (source==ATTRIBUTE)
		{
			int[]	array = extractLongArray();
			return array[0];
		}
		else
			return DevLongHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for an unsigned long.
	 */
	//===========================================
	public int extractULong()
	{
		return DevULongHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a float.
	 */
	//===========================================
	public float extractFloat()
	{
		return DevFloatHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a double.
	 */
	//===========================================
	public double extractDouble()
	{
		if (source==ATTRIBUTE)
		{
			double[]	array = extractDoubleArray();
			return array[0];
		}
		else
			return DevDoubleHelper.extract(any);
	}

	//===========================================
	/**
	 *	extract method for a String.
	 */
	//===========================================
	public String extractString()
	{
		if (source==ATTRIBUTE)
		{
			String[]	array = extractStringArray();
			return array[0];
		}
		else
			return DevStringHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a DevState.
	 */
	//===========================================
	public DevState extractDevState()
	{
		return DevStateHelper.extract(any);
	}



	//**********	Extract Methods for sequence types	*********************


	//===========================================
	/**
	 *	extract method for a byte Array.
	 */
	//===========================================
	public boolean[] extractBooleanArray()
	{
		return DevVarBooleanArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a byte Array.
	 */
	//===========================================
	public byte[] extractByteArray()
	{
		return DevVarCharArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a short Array.
	 */
	//===========================================
	public short[] extractShortArray()
	{
		return DevVarShortArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for an unsigned short Array.
	 */
	//===========================================
	public short[] extractUShortArray()
	{
		return DevVarUShortArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a long Array.
	 */
	//===========================================
	public int[] extractLongArray()
	{
		return DevVarLongArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for an unsigned long Array.
	 */
	//===========================================
	public int[] extractULongArray()
	{
		return DevVarULongArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a float Array.
	 */
	//===========================================
	public float[] extractFloatArray()
	{
		return DevVarFloatArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a double Array.
	 */
	//===========================================
	public double[] extractDoubleArray()
	{
		return DevVarDoubleArrayHelper.extract(any);
	}

	//===========================================
	/**
	 *	extract method for a String Array.
	 */
	//===========================================
	public String[] extractStringArray()
	{
		if (any==null)	System.out.println("any = null !!");
		return DevVarStringArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a DevVarLongStringArray.
	 */
	//===========================================
	public DevVarLongStringArray extractLongStringArray()
	{
		return DevVarLongStringArrayHelper.extract(any);
	}
	//===========================================
	/**
	 *	extract method for a DevVarDoubleStringArray.
	 */
	//===========================================
	public DevVarDoubleStringArray extractDoubleStringArray()
	{
		return DevVarDoubleStringArrayHelper.extract(any);
	}

	//===========================================
	/**
	 *	Returns the attribute errors list
	 */
	//===========================================
	public DevError[] getErrStack()
	{
		return attrval.err_list;
	}
	//===========================================
	/**
	 *	Returns the attribute type
	 */
	//===========================================
	public TypeCode type()
	{
		return any.type();
	}
	//===========================================
	/**
	 *	Return attribute name.
	 *	@throws	DevFailed in case of read_attribute failed
	 */
	//===========================================
	public String getName()
	{
		return attrval.name;
	}
	//===========================================
	//===========================================
	private int DIM_MINI(int x)
	{
		return (x==0) ? 1 : x;
	}
	//===========================================
	/**
	 *	Return number of data read.
	 *	@throws	DevFailed in case of read_attribute failed
	 */
	//===========================================
	public int getNbRead()
	{
		return attrval.r_dim.dim_x * DIM_MINI(attrval.r_dim.dim_y);
	}
	//===========================================
	/**
	 *	Return number of data written.
	 *	@throws	DevFailed in case of read_attribute failed
	 */
	//===========================================
	public int getNbWritten()
	{
		return attrval.w_dim.dim_x * DIM_MINI(attrval.w_dim.dim_y);
	}
	//===========================================
	/**
	 *	Return attribute written dim_x.
	 *	@throws	DevFailed in case of read_attribute failed
	 */
	//===========================================
	public int getWrittenDimX()
	{
		return attrval.w_dim.dim_x;
	}
	//===========================================
	/**
	 *	Return attribute written dim_y.
	 *	@throws	DevFailed in case of read_attribute failed
	 */
	//===========================================
	public int getWrittenDimY()
	{
		return attrval.w_dim.dim_y;
	}
	//===========================================
	/**
	 *	Returns attribute Tango type.
	 */
	//===========================================
	public int getType() throws DevFailed
	{
		int	type = -1;
		try {
			TypeCode	tc = attrval.value.type();
			//	Special case for test
			if (tc.kind().value()==TCKind._tk_enum)
				return TangoConst.Tango_DEV_STATE;
			
			TypeCode	tc_alias = tc.content_type();
			TypeCode	tc_seq   = tc_alias.content_type();
			TCKind		kind = tc_seq.kind();
			switch(kind.value())
			{
			case TCKind._tk_void:
				type = TangoConst.Tango_DEV_VOID;
				break;
			case TCKind._tk_boolean:
				type = TangoConst.Tango_DEV_BOOLEAN;
				break;
			case TCKind._tk_char:
				type = TangoConst.Tango_DEV_CHAR;
				break;
			case TCKind._tk_short:
				type = TangoConst.Tango_DEV_SHORT;
				break;
			case TCKind._tk_ushort:
				type = TangoConst.Tango_DEV_USHORT;
				break;
			case TCKind._tk_long:
				type = TangoConst.Tango_DEV_LONG;
				break;
			case TCKind._tk_ulong:
				type = TangoConst.Tango_DEV_ULONG;
				break;
			case TCKind._tk_float:
				type = TangoConst.Tango_DEV_FLOAT;
				break;
			case TCKind._tk_double:
				type = TangoConst.Tango_DEV_DOUBLE;
				break;
			case TCKind._tk_string:
				type = TangoConst.Tango_DEV_STRING;
				break;
			}
		}
		catch(org.omg.CORBA.TypeCodePackage.BadKind e)
		{
			Except.throw_exception("Api_TypeCodePackage.BadKind",
						"Bad or unknown type ",
						"DeviceDataHistory.getType()");
		}
		return type;
	}
}
