//+======================================================================
// $Source: /cvsroot/tango-cs/tango/api/java/fr/esrf/TangoApi/Database.java,v $
//
// Project:   Tango
//
// Description:  java source code for the TANGO clent/server API.
//
// $Author: pascal_verdier $
//
// $Revision: 3.16 $
//
// $Log: Database.java,v $
// Revision 3.16  2005/08/30 07:33:44  pascal_verdier
// Redundancy database connection between 2 TANGO_HOST added.
//
// Revision 3.15  2005/06/13 09:05:18  pascal_verdier
// Minor bugs fixed.
//
// Revision 3.14  2005/06/02 14:08:28  pascal_verdier
// *** empty log message ***
//
// Revision 3.13  2005/02/11 12:50:46  pascal_verdier
// DeviceInfo Object added (Start/Stop device dates).
//
// Revision 3.12  2004/12/07 09:30:29  pascal_verdier
// Exception classes inherited from DevFailed added.
//
// Revision 3.11  2004/11/05 11:59:20  pascal_verdier
// Attribute Info TANGO 5 compatibility.
//
// Revision 3.10  2004/10/11 12:25:42  pascal_verdier
// Multi TANGO_HOST bug fixed.
//
// Revision 3.9  2004/09/23 14:00:15  pascal_verdier
// Attribute for Devive_3Impl (Tango 5) implemented.
//
// Revision 3.8  2004/06/29 04:02:57  pascal_verdier
// Comments used by javadoc added.
//
// Revision 3.7  2004/06/21 12:23:35  pascal_verdier
// get_device_exported_for_class(String classname) method added
// and get_alias(String devname) bug fixed.
//
// Revision 3.6  2004/05/14 14:21:33  pascal_verdier
// Add timeout at runtime.
// Some little bugs fixed.
//
// Revision 3.5  2004/03/12 13:15:23  pascal_verdier
// Using JacORB-2.1
//
// Revision 3.2  2004/03/08 11:35:40  pascal_verdier
// AttributeProxy and aliases management added.
// First revision for event management classes.
//
// Revision 3.1  2003/07/22 14:15:35  pascal_verdier
// DeviceData are now in-methods objects.
// Minor change for TACO-TANGO common database.
//
// Revision 3.0  2003/04/29 08:03:29  pascal_verdier
// Asynchronous calls added.
// Logging related methods.
// little bugs fixed.
//
// Revision 2.0  2003/01/09 14:00:37  verdier
// jacORB is now the ORB used.
//
// Revision 1.8  2002/06/26 09:02:17  verdier
// tested with atkpanel on a TACO device
//
// Revision 1.7  2002/04/09 12:21:51  verdier
// IDL 2 implemented.
//
// Revision 1.6  2002/01/09 12:18:15  verdier
// TACO signals can be read as TANGO attribute.
//
// Revision 1.5  2001/12/10 14:19:42  verdier
// TACO JNI Interface added.
// URL syntax used for connection.
// Connection on device without database added.
//
// Revision 1.4  2001/07/04 14:06:05  verdier
// Attribute management added.
//
// Revision 1.3  2001/04/02 08:32:05  verdier
// TangoApi package has users...
//
// Revision 1.1  2001/02/02 13:03:46  verdier
// Initial revision
//
//
// Copyright 2001 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package fr.esrf.TangoApi;
 

import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import fr.esrf.TangoApi.events.*;
import java.io.*;
import java.util.*;

/** 
 *	Class Description:
 *	This class is the main class for TANGO database API.
 *	The TANGO database is implemented as a TANGO device server.
 *	To access it, the user has the CORBA interface command_inout().
 *	This expects and returns all parameters as ascii strings thereby making
 *	the database laborious to use for retreing device properties and information.
 *	In order to simplify this access, a high-level API has been implemented
 *	which hides the low-level formatting necessary to convert the
 *	command_inout() return values into binary values and all CORBA aspects
 *	of the TANGO.
 *	All data types are native java types e.g. simple types an arrays.
 *
 * @author  verdier
 * @version  $Revision: 3.16 $
 */


public class Database extends Connection
{
	//===================================================================
	/**
	 *	Database access constructor.
	 *
	 *	@throws DevFailed in case of environment not corectly set.
	 */
	//===================================================================
	public Database() throws DevFailed
	{
		super();
	}

	
	//===================================================================
	/**
	 *	Database access constructor.
	 *
	 *	@param	host	host where database is running.
	 *	@param	port	port for database connection.
	 *	@throws DevFailed in case of host or port not available
	 */
	//===================================================================
	public Database(String host, String port) throws DevFailed
	{
		super(host, port);
	}

	//==========================================================================
	/**
	 *	Convert a String array to a sting.
	 */
	//==========================================================================
	private String stringArray2String(String[] array)
	{
		StringBuffer	sb = new StringBuffer("");
		for(int i=0 ; i<array.length ; i++)
		{
			sb.append(array[i]);
			if (i<array.length-1)
				sb.append("\n");
		}
		return sb.toString();
	}







	//**************************************
	//       MISCELANEOUS MANAGEMENT
	//**************************************	


	//==========================================================================
	/**
	 *	Query the database for general info about the table in the database.
	 *	@return	the result of the query as String.
	 */
	//==========================================================================
	public String get_info() throws DevFailed
	{
		//	Query info from database
		DeviceData	argout = command_inout("DbInfo");
		String[] info = argout.extractStringArray();

		//	format result as string
		return stringArray2String(info);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of host registred.
	 *	@return	the list of all hosts registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_host_list() throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert("*");
		DeviceData	argout = command_inout("DbGetHostList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of host registred.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return	the list of the hosts registred in TANGO database 
	 *			with the specified wildcard.
	 */
	//==========================================================================
	public String[] get_host_list(String wildcard) throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetHostList", argin);
		return argout.extractStringArray();
	}






	//**************************************
	//       SERVERS MANAGEMENT
	//**************************************	

	//==========================================================================
	/**
	 *	Query the database for a list of classes instancied for a server.
	 *
	 *	@param	servname server name and instance name (ie.: Serial/i1).
	 *	@return	the list of all classes registred in TANGO database for 
	 *			servname except the DServer class
	 *			(existing on all Tango device server).
	 */
	//==========================================================================
	public String[] get_server_class_list(String servname) throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(servname);
		DeviceData	argout = command_inout("DbGetDeviceServerClassList", argin);
		String[]	list = argout.extractStringArray();
		//	Extract DServer class
		int 		nb_classes;
		if (list.length==0)
			nb_classes = 0 ;
		else
			nb_classes = list.length-1;
		String[]	classes = new String[nb_classes];
		for (int i=0, j=0 ; i<list.length ; i++)
			if (list[i].equals("DServer")==false)
				classes[j++] = list[i];
		return classes;
	}
	//==========================================================================
	/**
	 *	Query the database for a list of server names registred in the database.
	 *	@return	the list of all server names registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_server_name_list() throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert("*");
		DeviceData	argout = command_inout("DbGetServerNameList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of instance names
	 *		registred for specified server name.
	 *
	 *	@param	servname	server name.
	 *	@return	the list of all instance names for specified server name.
	 */
	//==========================================================================
	public String[] get_instance_name_list(String servname) throws DevFailed
	{
	
		DeviceData	argin = new DeviceData();
		argin.insert(servname);
		DeviceData	argout = command_inout("DbGetInstanceNameList", argin);		
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of servers registred in the database.
	 *	@return	the list of all servers registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_server_list() throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert("*");
		DeviceData	argout = command_inout("DbGetServerList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of servers registred in the database.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return	the list of all servers registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_server_list(String wildcard) throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetServerList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of servers registred on the specified host.
	 *	@param hostname	the specified host name.
	 *	@return	the list of the servers registred in TANGO database 
	 *			for the specified host.
	 */
	//==========================================================================
	public String[] get_host_server_list(String hostname) throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(hostname);
		DeviceData	argout = command_inout("DbGetHostServerList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for server information.
	 *	@param servname	The specified server name.
	 *	@return	The information found for the specified server
	 *				in a DBServInfo object.
	 */
	//==========================================================================
	public DbServInfo get_server_info(String servname) throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(servname);
		DeviceData	argout = command_inout("DbGetServerInfo", argin);
		String[]	info = argout.extractStringArray();
		return new DbServInfo(info);
	}

	//==========================================================================
	/**
	 *	Add/update server information in databse.
	 *	@param info	Server information for the specified server
	 *					in a DbServinfo object.
	 */
	//==========================================================================
	public void put_server_info(DbServInfo info) throws DevFailed
	{
		String[] array;
		array = new String[4];
		array[0] = info.name;
		array[1] = info.host;
		array[2] = (info.controlled) ? "1" : "0";
		array[3] = new String(""+info.startup_level);

		/*System.out.println("DbPutServerInfo:");
		for (int i=0 ; i<array.length ; i++)
			System.out.println("	"+array[i]);
		*/

		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbPutServerInfo", argin);
	}
	//==========================================================================
	/**
	 *	Delete server information in databse.
	 *	@param servname	Server name.
	 */
	//==========================================================================
	public void delete_server_info(String servname) throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(servname);
		command_inout("DbDeleteServerInfo", argin);
	}

	//**************************************
	//       DEVICES MANAGEMENT
	//**************************************	

	//==========================================================================
	/**
	 *	Add/update a device to the database
	 *	@param devinfo The device name, class and server  specified in object.
	 */
	//==========================================================================
	public void add_device(DbDevInfo devinfo) throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devinfo.toStringArray());
		command_inout("DbAddDevice", argin);
		//System.out.println(devinfo.name + " created");
	}

	//==========================================================================
	/**
	 *	Add/update a device to the database
	 *	@param devname		The device name
	 *	@param classname	The class.
	 *	@param servname		The server name.
	 */
	//==========================================================================
	public void add_device(String devname, String classname, String servname)
					throws DevFailed
	{
		DbDevInfo devinfo = new DbDevInfo(devname, classname, servname);
		DeviceData	argin = new DeviceData();
		argin.insert(devinfo.toStringArray());
		command_inout("DbAddDevice", argin);
		//System.out.println(devinfo.name + " created");
	}

	//==========================================================================
	/**
	 *	Delete the device of the specified name from the database
	 *	@param devname The device name.
	 */
	//==========================================================================
	public void delete_device(String devname) throws DevFailed
	{
		boolean	delete = true;
		try {
			//	Check if device alive before delete
			//------------------------------------------
			DeviceProxy	d = new DeviceProxy(devname);
			d.ping();
			//	If device alive do not delete
			//------------------------------------------
			delete = false;
		}
		catch(DevFailed e) {}

		if (delete)
		{
			DeviceData	argin = new DeviceData();
			argin.insert(devname);
			command_inout("DbDeleteDevice", argin);
			//System.out.println(devname + " deleted");
		}
		else
			Except.throw_connection_failed("TangoApi_DEVICE_ALIVE",
				"Cannot delete a device which is ALIVE.", "delete_device()");
	}

	//==========================================================================
	/**
	 *	Query the database for the export and more info of the specified device.
	 *	@param devname The device name.
	 *	@return the information in a DbGetDeviceInfo.
	 */
	//==========================================================================
	public DeviceInfo get_device_info(String devname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		DeviceData	argout = command_inout("DbGetDeviceInfo", argin);
		DevVarLongStringArray	info = argout.extractLongStringArray();
		return new DeviceInfo(info);
	}

	//==========================================================================
	/**
	 *	Query the database for the export info of the specified device.
	 *	@param devname The device name.
	 *	@return the information in a DbDevImportInfo.
	 */
	//==========================================================================
	public DbDevImportInfo import_device(String devname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		DeviceData	argout = command_inout("DbImportDevice", argin);
		DevVarLongStringArray	info = argout.extractLongStringArray();
		return new DbDevImportInfo(info);
	}
	//==========================================================================
	/**
	 *	Mark the specified server as unexported in the database.
	 *	@param devname The device name.
	 */
	//==========================================================================
	public void unexport_device(String devname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		command_inout("DbUnExportDevice", argin);
	}
	//==========================================================================
	/**
	 *	Update the export info fort this device in the database.
	 *	@param devinfo	Device information to export.
	 */
	//==========================================================================
	public void export_device(DbDevExportInfo devinfo)
				throws DevFailed
	{
		String[]	array = devinfo.toStringArray();
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbExportDevice", argin);
	}



	//**************************************
	//       Devices list MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for server devices and classes.
	 *	@param servname	The specified server name.
	 *	@return	The devices and classes (e.g. "id11/motor/1", "StepperMotor",
	 *			"id11/motor/2", "StepperMotor",....)
	 */
	//==========================================================================
	public String[] get_device_class_list(String servname) throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(servname);
		DeviceData	argout = command_inout("DbGetDeviceClassList", argin);
		return argout.extractStringArray();
	}


	//==========================================================================
	/**
	 *	Query the database for a list of devices served by the specified server
	 *	and of the specified class.
	 *	@param servname		The server name.
	 *	@param classname	The class name
	 *	@return the device names are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_name(String servname, String classname)
				throws DevFailed
	{
		String[]	array;
		array = new String[2];
		array[0] = servname;
		array[1] = classname;
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		DeviceData	argout = command_inout("DbGetDeviceList", argin);

		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of device domain names witch match
	 *	the wildcard provided.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device domain are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_domain(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetDeviceDomainList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of device family names witch match
	 *	the wildcard provided.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device family are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_family(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetDeviceFamilyList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of device member names witch match
	 *	the wildcard provided.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device member are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_member(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetDeviceMemberList", argin);
		return argout.extractStringArray();
	}


	//**************************************
	//       SERVERS MANAGEMENT
	//**************************************	

	//==========================================================================
	/**
	 *	Add a group of devices to the database.
	 *	@param servname	Server name for these devices.
	 *	@param devinfo	Devices and server information.
	 */
	//==========================================================================
	public void add_server(String servname, DbDevInfo[] devinfo)
				throws DevFailed
	{
		//	Convert data from DbDevInfos to a string array
		//----------------------------------------------
		//System.out.println("creating " + servname);
		String[]	array;
		array = new String[1 + 2*devinfo.length];
		
		array[0]   = servname;
		for (int i=0 ; i<devinfo.length ; i++)
		{
			array[2*i+1] = devinfo[i].name;
			array[2*i+2] = devinfo[i]._class;
		}

		//	Send command
		//-----------------------
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbAddServer", argin);
	}
	//==========================================================================
	/**
	 *	Delete the device server and its associated devices from the database.
	 *	@param devname the device name.
	 */
	//==========================================================================
	public void delete_server(String devname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		command_inout("DbDeleteServer", argin);
	}
	//==========================================================================
	/**
	 *	Add a group of devices to the database.
	 *	@param devinfo	Devices and server information.
	 */
	//==========================================================================
	public void export_server(DbDevExportInfo[] devinfo)
				throws DevFailed
	{
		//	Convert data from DbDevInfos to a string array
		//----------------------------------------------
		String[]	array;
		array = new String[6*devinfo.length];
		for (int i=0 ; i<devinfo.length ; i++)
		{
			String[]	one = devinfo[i].toStringArray();
			for (int j=0 ; j<6 ; j++)
				array[6*i+j] = new String(one[j]);
		}

		//	Send command
		//-----------------------
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbExportServer", argin);
	}
	//==========================================================================
	/**
	 *	Mark all devices exported for this device server as unexported.
	 *	@param devname the device name.
	 */
	//==========================================================================
	public void unexport_server(String devname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		command_inout("DbUnExportServer", argin);
	}



	
	//**************************************
	//       PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 * Convert Properties in DbDatnum array to a String array.
	 *	@param name	Object name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	private String[] dbdatum2StringArray(String name, DbDatum[] properties)
	{
		//	At first, search the array size
		//---------------------------------------
		int	size = 2;	//	Object Name and nb properties
		for (int i=0 ; i<properties.length ; i++)
		{
			size += 2;	//	Property Name and nb properties
			size += properties[i].size();
		}

		//	Format input parameters as strin array
		//--------------------------------------------
		String[] result;
		result = new String[size];
		result[0] = name;
		result[1] = String.valueOf(properties.length);
		for (int i=0, pnum=2 ; i<properties.length ; i++)
		{
			String[] prop = properties[i].toStringArray();
			for (int j=0 ; j<prop.length ; j++)
				result[pnum++] = prop[j];
		}
		return result;
	}
	//==========================================================================
	/**
	 * Convert Properties in DbDatnum array to a String array.
	 *	@param objname	Object name.
	 *	@param attname	attribute name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	private String[] dbdatum2StringArray(String objname, String attname, DbDatum[] properties)
	{
		//	At first, search the array size
		//---------------------------------------
		int	size = 4;	//	Object Name and nb properties
		for (int i=0 ; i<properties.length ; i++)
		{
			size += 2;	//	Property Name and nb properties
			size += properties[i].size();
		}

		//	Format input parameters as strin array
		//--------------------------------------------
		String[] result;
		result = new String[size];
		int		n   = 0;
		result[n++] = objname;	//	Object name
		result[n++] = "1";		//	One atribute
		result[n++] = attname;	//	attrbute name
		result[n++] = String.valueOf(properties.length);	//	Nb properties
		for (int i=0, pnum=n ; i<properties.length ; i++)
		{
			String[] prop = properties[i].toStringArray();
			for (int j=0 ; j<prop.length ; j++)
				result[pnum++] = prop[j];
		}
		return result;
	}
	//==========================================================================
	/**
	 * Convert a String array to a DbDatnum array for properties.
	 *	@param strprop Properties names and values array.
	 *	@return a DbDatum array specifying the properties fonud in string array.
	 */
	//==========================================================================
	private DbDatum[] 	stringArray2DbDatum(String[] strprop)
	{
		//	And format result as DbDatum array
		//---------------------------------------
		DbDatum[]	properties;
		int 		nb_prop = Integer.parseInt(strprop[1]);
		properties = new DbDatum[nb_prop];

		//	Skip obj name, nb prop found and name of first property.
		//-----------------------------------------------------------
		for (int i=2, pnum=0 ; i<strprop.length-1 ; )
		{
			int nb = Integer.parseInt(strprop[i+1]);

			//	if property exist, create Datnum object.
			//---------------------------------------------------
			int start_val = i+2;
			int	end_val   = i+2+nb;
			if (nb>0)
				properties[pnum++] = new DbDatum(strprop[i],
										strprop, start_val, end_val);
			else
			{
				//	no property  --> fields do not exist
				properties[pnum++] = new DbDatum(strprop[i]);

				//	If nb property is zero there is a property
				//	set to space char (!!!)
				//	if Object is device it is true but false for a class !!!
				//----------------------------------------------------------
				if (start_val+1 < strprop.length)
				{
					String s = strprop[start_val];
					if (s.length()==0 || s.equals(" "))
						end_val = start_val+1;
				}
			}
			i = end_val;
		}
		return properties;
	}

	//==========================================================================
	/**
	 *	Query the database for a list of object (ie. non-device, class or device)
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param properties list of property DbDatum objects.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	private DbDatum[] get_obj_property(String name, String type, DbDatum[] properties)
				throws DevFailed
	{
		//	Format input parameters as string array
		//--------------------------------------------
		String[]	array;
		array = new String[properties.length];
		for (int i=0 ; i<properties.length ; i++)
			array[i] = properties[i].name;
		return get_obj_property(name, type, array);
	}
	//==========================================================================
	/**
	 *	Query the database for an object (ie. non-device, class or device)
	 *	property for the pecified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propename lproperty name.
	 *	@return property in DbDatum objects.
	 */
	//==========================================================================
	private DbDatum get_obj_property(String name, String type, String propname)
				throws DevFailed
	{
		//	Format input parameters as string array
		//--------------------------------------------
		String[]	array;
		array = new String[1];
		array[0] = propname;
		DbDatum[]	data = get_obj_property(name, type, array);
		return data[0];
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (ie. non-device, class or device)
	 *	properties for thr dpecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propnames list of property names.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	private DbDatum[] get_obj_property(String name, String type, String[] propnames)
				throws DevFailed
	{
		//	Format input parameters as string array
		//--------------------------------------------
		String[]	array;
		array = new String[1 + propnames.length];
		array[0] = name;
		for (int i=1 ; i<propnames.length+1 ; i++)
			array[i] = propnames[i-1];

		//	Buid command name (depends on object type)
		//---------------------------------------------------
		String	cmd = new String("DbGet"+type+"Property");

		//	Read Database
		//---------------------
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		DeviceData	argout = command_inout(cmd, argin);
		String[]	result = argout.extractStringArray();

//if (type.equals("Class"))
//for (int i=0 ; i<result.length ; i++)
//	System.out.println(i + ": " +result[i]);

		//	And convert to DbDatum array before returning
		//-------------------------------------------------
		return stringArray2DbDatum(result);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param properties list of property DbDatum objects.
	 */
	//==========================================================================
	private void delete_obj_property(String name, String type, DbDatum[] properties)
				throws DevFailed
	{
		//	Format input parameters as strin array
		//--------------------------------------------
		String[]	array;
		array = new String[properties.length];
		for (int i=0 ; i<properties.length ; i++)
			array[i] = properties[i].name;

		delete_obj_property(name, type, array);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propname Property name.
	 */
	//==========================================================================
	private void delete_obj_property(String name, String type, String propname)
				throws DevFailed
	{
		//	Format input parameters as strin array
		//--------------------------------------------
		String[]	array;
		array = new String[1];
		array[0] = propname;

		delete_obj_property(name, type, array);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propnames Property names.
	 */
	//==========================================================================
	private void delete_obj_property(String name, String type, String[] propnames)
				throws DevFailed
	{
		//	Format input parameters as strin array
		//--------------------------------------------
		String[]	array;
		array = new String[propnames.length+1];
		array[0] = name;
		for (int i=0 ; i<propnames.length ; i++)
			array[i+1] = propnames[i];

		//	Buid command name (depends on object type)
		//---------------------------------------------------
		String	cmd = new String("DbDelete"+type+"Property");
		
		/****
		for (int i=0 ; i<array.length ; i++)
			System.out.println("array -> " + array[i]);
		System.out.println("cmd -> " + cmd);
		*********/

		//	Send it to  Database
		//------------------------------
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout(cmd, argin);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (ie non-device)
	 *	for which properties are defiend.
	 *	@param wildcard	wildcard (* matches any charactere).
	 *	@return objects for which properties are defiened list.
	 */
	//==========================================================================
	public String[] get_object_list(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetObjectList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (ie non-device)
	 *	for which properties are defiend.
	 *	@param objname	object name.
	 *	@param wildcard wildcard (* matches any charactere).
	 *	@return Property names..
	 */
	//==========================================================================
	public String[] get_object_property_list(String objname, String wildcard)
				throws DevFailed
	{
		String[]	array = new String[2];
		array[0] = objname;
		array[1] = wildcard;
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		DeviceData	argout = command_inout("DbGetPropertyList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (ie non-device)
	 *	properties for the pecified object.
	 *	@param name Object name.
	 *	@param propnames list of property names.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "";	//	No object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for an object (ie non-device)
	 *	property for the pecified object.
	 *	@param name Object name.
	 *	@param propname list of property names.
	 *	@return property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_property(String name, String propname)
				throws DevFailed
	{
		String	type = "";	//	No object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (ie non-device)
	 *	properties for thr dpecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Object name.
	 *	@param properties list of property DbDatum objects.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String	type = "";	//	No object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified object
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name Object name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbPutProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a  property for the specified object.
	 *	@param name Object name.
	 *	@param propname Property names.
	 */
	//==========================================================================
	public void delete_property(String name, String propname)
				throws DevFailed
	{
		String type = "";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "";
		delete_obj_property(name, type, properties);
	}

	//==========================================================================
	/**
	 *	Query the database for a list of class
	 *	properties for the pecified object.
	 *	@param classname	device name.
	 *	@param wildcard	propertie's wildcard (* matches any charactere).
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public String[] get_class_property_list(String classname, String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(classname);
		DeviceData	argout = command_inout("DbGetClassPropertyList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	@param devname	device name.
	 *	@param wildcard	propertie's wildcard (* matches any charactere).
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public String[] get_device_property_list(String devname, String wildcard)
				throws DevFailed
	{
		String[]	array = new String[2];
		array[0] = devname;
		array[1] = wildcard;		
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		DeviceData	argout = command_inout("DbGetDevicePropertyList", argin);
		return argout.extractStringArray();
	}







	
	//**************************************
	//       DEVICE PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	@param name device name.
	 *	@param propnames list of property names.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_device_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "Device";	//	Device object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a  device
	 *	property for the pecified object.
	 *	@param name device name.
	 *	@param propname property name.
	 *	@return property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_device_property(String name, String propname)
				throws DevFailed
	{
		String	type = "Device";	//	Device object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name device name.
	 *	@param properties list of property DbDatum objects.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_device_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String		type = "Device";	//	Device object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified device
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name device name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_device_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbPutDeviceProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Device name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_device_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "Device";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Device name.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_device_property(String name, String propname)
				throws DevFailed
	{
		String type = "Device";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Device name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_device_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "Device";
		delete_obj_property(name, type, properties);
	}






	
	//**************************************
	//      ATTRIBUTE PROPERTIES MANAGEMENT
	//**************************************
	
	//==========================================================================
	/**
	 *	Query the database for a list of device attributes
	 *	properties for the pecified object.
	 *	@param devname device name.
	 *	@param attnames attribute names.
	 *	@return properties in DbAttribute objects array.
	 */
	//==========================================================================
	public DbAttribute[] get_device_attribute_property(String devname, String[] attnames)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		DeviceData	argout = null;
		int			mode = 2;

		try
		{
					//	value is an array
			argin.insert(ApiUtil.toStringArray(devname, attnames));
			argout = command_inout("DbGetDeviceAttributeProperty2", argin);
		}
		catch(DevFailed e)
		{
			if (e.errors[0].reason.equals("API_CommandNotFound"))
			{
					//	Value is just one element
				argout = command_inout("DbGetDeviceAttributeProperty", argin);
				mode = 1;
			}
			else
				throw e;
		}
		return ApiUtil.toDbAttributeArray(argout.extractStringArray(), mode);
	}
	//==========================================================================
	/**
	 *	Query the database for device attribute
	 *	property for the pecified object.
	 *	@param devname device name.
	 *	@param attname attribute name.
	 *	@return property in DbAttribute object.
	 */
	//==========================================================================
	public DbAttribute get_device_attribute_property(String devname, String attname)
				throws DevFailed
	{
		String[]	attnames = new String[1];
		attnames[0] = attname;
		return get_device_attribute_property(devname, attnames)[0];
	}
	//==========================================================================
	/**
	 *	Insert or update a list of attribute properties for the specified device.
	 *	The property names and their values are specified by the DbAttribute array.
	 *	
	 *	@param devname	device name.
	 *	@param attr		attribute names, and properties (names and values).
	 */
	//==========================================================================
	public void put_device_attribute_property(String devname, DbAttribute[] attr)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		try
		{
					//	value is an array
			argin.insert(ApiUtil.toStringArray(devname, attr, 2));
			command_inout("DbPutDeviceAttributeProperty2", argin);
		}
		catch(DevFailed e)
		{
			if (e.errors[0].reason.equals("API_CommandNotFound"))
			{
					//	Value is just one element
				argin.insert(ApiUtil.toStringArray(devname, attr, 1));
				command_inout("DbPutDeviceAttributeProperty", argin);
			}
			else
				throw e;
		}
	}
	//==========================================================================
	/**
	 *	Insert or update a list of attribute properties for the specified device.
	 *	The property names and their values are specified by the DbAttribute.
	 *	
	 *	@param devname	device name.
	 *	@param attr	attribute name, and properties (names and values).
	 */
	//==========================================================================
	public void put_device_attribute_property(String devname, DbAttribute attr)
				throws DevFailed
	{
		DbAttribute[]	da = new DbAttribute[1];
		da[0] = attr;
		put_device_attribute_property(devname, da);
	}
	//==========================================================================
	/**
	 *	Delete an list of attributes properties for the specified object.
	 *	@param devname Device name.
	 *	@param attr	attribute name, and properties (names).
	 */
	//==========================================================================
	public void delete_device_attribute_property(String devname, DbAttribute attr)
				throws DevFailed
	{
		delete_device_attribute_property(devname, 
						attr.name, attr.get_property_list());
	}
	//==========================================================================
	/**
	 *	Delete a list of attributes properties for the specified object.
	 *	@param devname Device name.
	 *	@param attr	attribute names, and properties (names) in array.
	 */
	//==========================================================================
	public void delete_device_attribute_property(String devname, DbAttribute[] attr)
				throws DevFailed
	{
		for (int i=0 ; i<attr.length ; i++)
			delete_device_attribute_property(devname,
							attr[i].name, attr[i].get_property_list());
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param devname	Device name.
	 *	@param attname	Attribute name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_device_attribute_property(String devname, String attname, String[] propnames)
				throws DevFailed
	{
		if (propnames.length==0)
			return;

		//	Build a String array before command
		String[]	array = new String[2+propnames.length];
		array[0] = devname;
		array[1] = attname;
		for (int i=0 ; i<propnames.length ; i++)
			array[i+2] = propnames[i];		

		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbDeleteDeviceAttributeProperty", argin);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param devname	Device name.
	 *	@param attname	Attribute name.
	 *	@param propname	Property name.
	 */
	//==========================================================================
	public void delete_device_attribute_property(String devname, String attname, String propname)
				throws DevFailed
	{
		String[]	array = new String[1];
		array[0] = propname;
		delete_device_attribute_property(devname, attname, array);
	}


	//==========================================================================
	/**
	 *	Delete an attribute for the specified object.
	 *	@param devname	Device name.
	 *	@param attname	Attribute name.
	 */
	//==========================================================================
	public void delete_device_attribute(String devname, String attname) throws DevFailed
	{
		String[] array = new String[2];
		array[0] = devname;
		array[1] = attname;
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbDeleteDeviceAttribute", argin);
	}


	//**************************************
	//      CLASS PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a list of classes registred in the database.
	 *	@param servname	server name
	 *	@return	the list of all servers registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_class_list(String servname) throws DevFailed
	{
		//	Query info from database
		DeviceData	argin = new DeviceData();
		argin.insert(servname);
		DeviceData	argout = command_inout("DbGetClassList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of class
	 *	properties for the pecified object.
	 *	@param name Class name.
	 *	@param propnames list of property names.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_class_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "Class";	//	class object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a class
	 *	property for the pecified object.
	 *	@param name Class name.
	 *	@param propname list of property names.
	 *	@return property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_class_property(String name, String propname)
				throws DevFailed
	{
		String	type = "Class";	//	class object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of class
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Class name.
	 *	@param properties list of property DbDatum objects.
	 *	@return properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_class_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String		type = "Class";	//	Device object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified class.
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name Class name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_class_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbPutClassProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_class_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "Class";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Class name.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_class_property(String name, String propname)
				throws DevFailed
	{
		String type = "Class";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_class_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "Class";
		delete_obj_property(name, type, properties);
	}



	
	//**************************************
	//      CLASS Attribute PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a attributes defined for a class.
	 *	All attributes for a class attribute are returned.
	 *
	 *	@param classname	class name.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return attributes list for specified class
	 */
	//==========================================================================
	public String[] get_class_attribute_list(String classname, String wildcard) throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(ApiUtil.toStringArray(classname, wildcard));
		DeviceData	argout = command_inout("DbGetClassAttributeList", argin);
		return argout.extractStringArray();
	}
	
	//==========================================================================
	/**
	 *	Query the database for a attribute properties for trhe specified class.
	 *
	 *	@param classname	class name.
	 *	@param attname		attribute name
	 *	@return attribute properties for specified class and attribute.
	 */
	//==========================================================================
	public DbAttribute get_class_attribute_property(String classname, String attname) throws DevFailed
	{
		String[]	attnames = new String[1];
		attnames[0] = attname;
		return get_class_attribute_property(classname, attnames)[0];
	}
	
	//==========================================================================
	/**
	 *	Query the database for a list of class attributes
	 *	properties for the pecified object.
	 *	@param classname Class name.
	 *	@param attnames list of attribute names.
	 *	@return attribute properties for specified class and attributes.
	 */
	//==========================================================================
	public DbAttribute[] get_class_attribute_property(String classname, String[] attnames)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		DeviceData	argout;
		int			mode = 2;

		try
		{
					//	value is an array
			argin.insert(ApiUtil.toStringArray(classname, attnames));
			argout = command_inout("DbGetClassAttributeProperty2", argin);
		}
		catch(DevFailed e)
		{
			if (e.errors[0].reason.equals("API_CommandNotFound"))
			{
					//	Value is just one element
				argout = command_inout("DbGetClassAttributeProperty", argin);
				mode = 1;
			}
			else
				throw e;
		}
		return ApiUtil.toDbAttributeArray(argout.extractStringArray(), mode);
	}
	
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified class attribute.
	 *	The attribute name, the property names and their values
	 *	are specified by the DbAttribute.
	 *	
	 *	@param classname Class name.
	 *	@param attr		DbAttribute objects containing attribute names,
	 *					property names and property values.
	 */
	//==========================================================================
	public void put_class_attribute_property(String classname, DbAttribute[] attr)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(ApiUtil.toStringArray(classname, attr, 1));
		command_inout("DbPutClassAttributeProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified class attribute.
	 *	The attribute name, the property names and their values
	 *	are specified by the DbAttribute.
	 *	
	 *	@param classname Class name.
	 *	@param attr		DbAttribute object containing attribute name,
	 *					property names and property values.
	 */
	//==========================================================================
	public void put_class_attribute_property(String classname, DbAttribute attr)
				throws DevFailed
	{
		DbAttribute[]	da = new DbAttribute[1];
		da[0] = attr;
		put_class_attribute_property(classname, da);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Class name.
	 *	@param propname Property names.
	 */
	//==========================================================================
	public void delete_class_attribute_property(String name, String attname, String propname)
				throws DevFailed
	{
		String[]	array = new String[1];
		array[0] = propname;
		delete_class_attribute_property(name, attname, array);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param attname attribute name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_class_attribute_property(String name, String attname, String[] propnames)
				throws DevFailed
	{
		String[]	array = new String[2+propnames.length];
		array[0] = name;
		array[1] = attname;
		for (int i=0 ; i<propnames.length ; i++)
			array[i+2] = propnames[i];		

		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbDeleteClassAttributeProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Query database for list of exported devices.
	 *
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return	The list of exported devices
	 */
	//==========================================================================
	public String[] get_device_exported(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetDeviceExportedList", argin);		

		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query database for list of exported devices for the specified class name.
	 *
	 *	@param classname	class name to query the exported devices.
	 *	@return	The list of exported devices
	 */
	//==========================================================================
	public String[] get_device_exported_for_class(String classname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(classname);
		DeviceData	argout = command_inout("DbGetExportdDeviceListForClass", argin);		

		return argout.extractStringArray();
	}

	//==========================================================================
	//		Aliases management
	//==========================================================================

	//==========================================================================
	/**
	 *	Query the database for a list of aliases for the specified wildcard.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device aliases are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_alias_list(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetDeviceAliasList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for an alias for the specified device.
	 *	@param devname	device's name.
	 *	@return the device alias found.
	 */
	//==========================================================================
	public String get_device_alias(String devname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(devname);
		DeviceData	argout = command_inout("DbGetDeviceAlias", argin);
		return argout.extractString();
	}
	
	//==========================================================================
	/**
	 *	Query the database a device for the specified alias.
	 *	@param alias	The device name.alias
	 *	@return the device aliases are stored in an array of strings.
	 */
	//==========================================================================
	public String get_alias_device(String alias)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(alias);
		DeviceData	argout = command_inout("DbGetAliasDevice", argin);
		return argout.extractString();
	}
	
	//==========================================================================
	/**
	 *	Set an alias for a device name
	 *	@param devname device name.
	 *	@param aliasname alias name.
	 */
	//==========================================================================
	public void put_device_alias(String devname, String aliasname)
				throws DevFailed
	{
		String[]	array = new String[2];
		array[0] = devname;
		array[1] = aliasname;
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbPutDeviceAlias", argin);		
	}
	//==========================================================================
	/**
	 *	Query the database to delete alias for the specified device alias.
	 *	@param alias	device alias name.
	 */
	//==========================================================================
	public void delete_device_alias(String alias)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(alias);
		command_inout("DbDeleteDeviceAlias", argin);
	}



	//==========================================================================
	/**
	 *	Query the database for a list of aliases for the specified wildcard.
	 *	@param 	wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device aliases are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_attribute_alias_list(String wildcard)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(wildcard);
		DeviceData	argout = command_inout("DbGetAttributeAliasList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of aliases for the specified attribute.
	 *	@param	attname	The attribute name.
	 *	@return the device aliases are stored in an array of strings.
	 */
	//==========================================================================
	public String get_attribute_alias(String attname)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(attname);
		DeviceData	argout = command_inout("DbGetAttributeAlias", argin);
		return argout.extractString();
	}
	//==========================================================================
	/**
	 *	Set an alias for a attribute name
	 *	@param attname attribute name.
	 *	@param aliasname alias name.
	 */
	//==========================================================================
	public void put_attribute_alias(String attname, String aliasname)
				throws DevFailed
	{
		String[]	array = new String[2];
		array[0] = attname;
		array[1] = aliasname;
		DeviceData	argin = new DeviceData();
		argin.insert(array);
		command_inout("DbPutAttributeAlias", argin);		
	}
	//==========================================================================
	/**
	 *	Query the database to delete alias for the specified attribute alias.
	 *	@param alias	device alias name.
	 */
	//==========================================================================
	public void delete_attribute_alias(String alias)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(alias);
		command_inout("DbDeleteAttributeAlias", argin);
	}
	//==========================================================================
	//==========================================================================
	public String[] getDevices(String wildcard) throws DevFailed
	{
		//	Get each field of device name
		StringTokenizer stk = new StringTokenizer(wildcard, "/");
		Vector	vector = new Vector();
		for (int i=0 ; stk.hasMoreTokens() ; i++)
			vector.add(stk.nextToken());
		if (vector.size()<3)
			Except.throw_exception( "TangoApi_DeviceNameNotValid",
				"Device name not valid", "ATangoApi.Database.getDevices()");
		
		String	domain = (String)vector.elementAt(0);
		String	family = (String)vector.elementAt(1);
		String	member = (String)vector.elementAt(2);
		
		//System.out.println(domain + "/" + family + "/" + member);		
		String[]	domains = get_device_domain(domain);
		if (domains.length==0)
		{
			domains = new String[1];
			domains[0] = domain;
		}
		vector.clear();
		for (int i=0 ; i<domains.length ; i++)
		{
			String[]	families = get_device_family(domains[i] + "/" + family);
			if (families.length==0)
			{
				families = new String[1];
				families[0] = family;
			}
			for (int j=0 ; j<families.length ; j++)
			{
				String[]	members = get_device_member(domains[i] +
								"/" + families[j] + "/" + member);
				if (members.length==0)
				{
					members = new String[1];
					members[0] = member;
				}
				for (int k=0 ; k<members.length ; k++)
					vector.add(domains[i] +
								"/" + families[j] + "/" + members[k]);
			}
		}
		String[]	devices = new String[vector.size()];
		for (int i=0 ; i<vector.size() ; i++)
			devices[i] = (String)vector.elementAt(i);
		return devices;
	}








	//==========================================================================
	/**
	 *	Query the database for the export info of the specified event.
	 *	@param  channel_name	The event name.
	 *	@return the information in a DbEventImportInfo.
	 */
	//==========================================================================
	public DbEventImportInfo import_event(String channel_name)
				throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		argin.insert(channel_name);
		DeviceData	argout = command_inout("DbImportEvent", argin);
		DevVarLongStringArray	info = argout.extractLongStringArray();
		return new DbEventImportInfo(info);
	}
}
