//+======================================================================
// $Source: /cvsroot/tango-cs/tango/api/java/fr/esrf/TangoApi/Connection.java,v $
//
// Project:   Tango
//
// Description:  java source code for the TANGO clent/server API.
//
// $Author: pascal_verdier $
//
// $Revision: 3.19 $
//
// $Log: Connection.java,v $
// Revision 3.19  2005/08/30 07:33:44  pascal_verdier
// Redundancy database connection between 2 TANGO_HOST added.
//
// Revision 3.18  2005/08/11 07:17:15  pascal_verdier
// Bug fixed on Database reconnection.
//
// Revision 3.17  2005/08/10 08:09:05  pascal_verdier
// If connection failed, do not retry if less than one second.
//
// Revision 3.16  2005/04/26 13:29:03  pascal_verdier
// On DB_DeviceNotDefined exception, add database name on description.
//
// Revision 3.15  2005/03/03 09:10:10  pascal_verdier
// Method  public DevInfo_3  info_3() added.
// info() return DevInfo.
//
// Revision 3.14  2005/03/01 08:20:54  pascal_verdier
// dev_info() modied for Device_3Impl.
//
// Revision 3.13  2005/01/14 10:32:33  pascal_verdier
// jacorb.connection.client.connect_timeout property added and set to 5000.
//
// Revision 3.12  2004/12/07 09:30:30  pascal_verdier
// Exception classes inherited from DevFailed added.
//
// Revision 3.11  2004/12/02 13:59:44  pascal_verdier
// Throws a DevFailed exception if TANGO_HOST host name is not valid.
//
// Revision 3.10  2004/11/30 13:05:37  pascal_verdier
// Add no database syntax on adm_name.
//
// Revision 3.9  2004/11/05 12:05:34  pascal_verdier
// Use now JacORB_2_2_1.
//
// Revision 3.8  2004/09/17 07:57:02  pascal_verdier
// Attribute for Devive_3Impl (Tango 5) implemented.
//
// Revision 3.7  2004/08/17 08:37:45  pascal_verdier
// minor change.
//
// Revision 3.6  2004/05/14 14:21:33  pascal_verdier
// Add timeout at runtime.
// Some little bugs fixed.
//
// Revision 3.5  2004/03/12 13:15:22  pascal_verdier
// Using JacORB-2.1
//
// Revision 3.4  2004/03/08 11:35:40  pascal_verdier
// AttributeProxy and aliases management added.
// First revision for event management classes.
//
// Revision 3.3  2003/09/08 11:02:34  pascal_verdier
// *** empty log message ***
//
// Revision 3.2  2003/07/22 14:15:35  pascal_verdier
// DeviceData are now in-methods objects.
// Minor change for TACO-TANGO common database.
//
// Revision 3.1  2003/05/19 13:35:14  pascal_verdier
// Bug on transparency reconnection fixed.
// input argument modified for add_logging_target method.
//
// Revision 3.0  2003/04/29 08:03:28  pascal_verdier
// Asynchronous calls added.
// Logging related methods.
// little bugs fixed.
//
// Revision 2.0  2003/01/09 14:00:37  verdier
// jacORB is now the ORB used.
//
// Revision 1.8  2002/06/26 09:02:17  verdier
// tested with atkpanel on a TACO device
//
// Revision 1.7  2002/04/09 12:21:51  verdier
// IDL 2 implemented.
//
// Revision 1.6  2002/01/09 12:18:15  verdier
// TACO signals can be read as TANGO attribute.
//
// Revision 1.5  2001/12/10 14:19:42  verdier
// TACO JNI Interface added.
// URL syntax used for connection.
// Connection on device without database added.
//
// Revision 1.4  2001/07/04 14:06:05  verdier
// Attribute management added.
//
// Revision 1.3  2001/04/02 08:32:05  verdier
// TangoApi package has users...
//
// Revision 1.1  2001/02/02 13:03:46  verdier
// Initial revision
//
//
// Copyright 2001 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package fr.esrf.TangoApi;

import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import java.io.*;
import java.util.*;
import java.net.*;

import org.jacorb.orb.ParsedIOR;

/**
 *	Class Description:
 *	This class manage device connection for Tango objects.
 *	It is an api between user and IDL Device object.
 *
 * @author  verdier
 * @version  $Revision: 3.19 $
 */

public class Connection implements ApiDefs
{
	/**
	 *	Device IDL version number
	 */
	protected int	idl_version = 1;
	/**
	 *	Device IDL object used for TANGO device access
	 */
	protected Device	device = null;
	/**
	 *	Device IDL_2 object used for TANGO device access
	 */
	protected Device_2	device_2 = null;
	/**
	 *	Device IDL_3 object used for TANGO device access
	 */
	protected Device_3	device_3 = null;
	/**
	 *	TACO Device object used for TANGO interface.
	 */
	protected TacoTangoDevice	taco_device = null;
	/**
	 *	Device IDL object used for efective connection.
	 */
	private org.omg.CORBA.Object obj = null;
	/**
	 *	Device timeout value in ms.
	 */
	private int			dev_timeout = 0;
	/**
	 *	Device name;
	 */
	protected String	devname = null;
	/**
	 *	Used to know if it is a connection or a Re-connection.
	 */
	private boolean		already_connected = false;
	/**
	 *	Device connected is a database or a device proxy
	 */
	private	boolean		device_is_dbase;
	/**
	 *	IOR String only used if the device import is done from ior.
	 */
	protected String	ior = null;
	/**
	 *	URL like for connection:
	 */
	protected TangoUrl	url;
	/**
	 *	Data source (CACHE_DEV, CACHE or DEV)
	 */
	protected DevSource	dev_src = DevSource.CACHE_DEV;
	/**
	 *	if true -> try to reconnect the device before exception
	 */
	protected boolean transparent_reconnection = false;

	/**
	 *	Previous reconnection failed excpetionm.
	 */
	private DevFailed	prev_failed = null;
	/**
	 *	Previous reconnection failed time.
	 */
	private long		prev_failed_t0 = 0;
	/**
	 *	HashMap to manage database redundancy.
	 */
	static private DbRedundancy	db_redundancy;

	//===================================================================
	//===================================================================
	public Device get_device()
	{
		return device;
	}
	//===================================================================
	//===================================================================
	private String buildUrlName(String host, String port)
	{
		return new String("tango://" + host + ":" + port);
	}
	//===================================================================
	//===================================================================
	private String buildUrlName(String host, String port, String devname)
	{
		return new String("tango://" + host + ":" + port + "/" + devname);
	}
	//===================================================================
	/**
	 *	Connection constructor. It makes a connection on database server.
	 *
	 */
	//===================================================================
	protected Connection()	throws DevFailed
	{
		//	Check Tango Host Properties (host name, port number)
		//------------------------------------------------------------
		url = new TangoUrl();
		device_is_dbase = true;
		transparent_reconnection = true;	//	Always true for Database
		//url.trace();

		ApiUtil.get_orb();
		connect_to_dbase();
		devname = device.name();
		already_connected = true;
	}
	
	//===================================================================
	/**
	 *	Connection constructor. It makes a connection on database server.
	 *
	 *	@param	host	host where database is running.
	 *	@param	port	port for database connection.
	 */
	//===================================================================
	protected Connection(String host, String port)	throws DevFailed
	{
		url = new TangoUrl(buildUrlName(host, port));
		device_is_dbase = true;
		transparent_reconnection = true;	//	Always true for Database

		ApiUtil.get_orb();
		connect_to_dbase();
		devname = device.name();
		already_connected = true;
	}
	//===================================================================
	/**
	 *	Connection constructor. It makes a connection on database server.
	 *
	 *	@param	host	host where database is running.
	 *	@param	port	port for database connection.
	 *	@param	auto_reconnect	do not reconnect if false.
	 */
	//===================================================================
	protected Connection(String host, String port, boolean auto_reconnect)	throws DevFailed
	{
		url = new TangoUrl(buildUrlName(host, port));
		device_is_dbase = true;
		transparent_reconnection = auto_reconnect;

		ApiUtil.get_orb();
		connect_to_dbase();
		devname = device.name();
		already_connected = true;
	}

	//===================================================================
	/**
	 *	Connection constructor. It imports the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 */
	//===================================================================
	protected Connection(String devname) throws DevFailed
	{
		//System.out.println("DeviceProxy("+devname+")");
		url = new TangoUrl(devname);
		device_is_dbase = false;
		this.devname = url.devname;
		
		//	Check if connection is possible
		if (url.protocol==TANGO && url.use_db)
			ior = get_exported_ior();
	}
	//===================================================================
	/**
	 *	Connection constructor. It imports the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 *	@param	param	String parameter to import device.
	 *	@param	src		Source to import device (ior, dbase...)
	 */
	//===================================================================
	protected Connection(String devname, String param, int src) throws DevFailed
	{
		this.devname = devname;
		device_is_dbase = false;
		if (src==FROM_IOR)
		{
			ior = param;
			url = new TangoUrl();
			url.protocol = TANGO;
		}
		else
		{
			String	reason = "TangoApi_INVALID_ARGS";
			String	desc   = new String("Invalid argument");
			String	origin = new String("Connection.Connemction()");
			Except.throw_wrong_syntax_exception(reason, desc, origin);
		}
	}
	//===================================================================
	/**
	 *	Connection constructor. It imports the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 *	@param	host	host where database is running.
	 *	@param	port	port for database connection.
	 */
	//===================================================================
	protected Connection(String devname, String host, String port) throws DevFailed
	{
		url = new TangoUrl(buildUrlName(host, port, devname));
		this.devname = url.devname;
		device_is_dbase = false;

		//	Check if connection is possible
		if (url.protocol==TANGO)
			ior = get_exported_ior();
	}
	//===================================================================
	//===================================================================
	protected synchronized void build_connection() throws DevFailed
	{
		long	t = System.currentTimeMillis();
		long	delay = t - prev_failed_t0;
		boolean	try_reconnection = true;

		//	Check if connection on database or on a device
		if (device_is_dbase)
			connect_to_dbase();
		else
		try
		{
			//	Check if Tango or taco device
			if (url.protocol==TANGO)
			{
				//	Do not reconnect if to soon
				if (prev_failed!=null &&
					delay<ApiUtil.getReconnectionDelay())
				{
					try_reconnection = false;
					throw prev_failed;
				}
				//System.out.println(" Try reconnection on " + devname + ":	" + delay + " ms");
				prev_failed_t0 = t;

				//	Check if connection with databse or without
				if (url.use_db==true)
				{
					dev_import();
				}
				else
				{
					dev_import_without_dbase();
				}
				prev_failed = null;
			}
			else
			if (url.protocol==TACO)
				if (taco_device==null)
					taco_device = new TacoTangoDevice(devname);
		}
		catch(DevFailed e)
		{
			if (try_reconnection==true)
			{
				prev_failed_t0 = t;
				prev_failed = e;
			}
			ior = null;
			String	reason = "TangoApi_CANNOT_IMPORT_DEVICE";
			String	desc   = new String("Cannot import " + devname);
			String	origin = new String("Connection.build_connection(" + devname + ")");
			Except.throw_communication_failed(e, reason, desc, origin);
		}
	}

	//===================================================================
	/**
	 *	Ckeck the IDL revision and create the associated device.
	 *	@param	corba_str IOR or corbaloc string.
	 */
	//===================================================================
	private void createDevice(String corba_str) throws DevFailed
	{
		try
		{
			//	Build corba object
			ORB	orb = ApiUtil.get_orb();
			obj = orb.string_to_object(corba_str);
		}
		catch(RuntimeException e)
		{
			Except.throw_connection_failed("TangoApi_TANGO_HOST_NOT_VALID",
							e.toString().substring(e.toString().indexOf(":")+2),
							"Connection.createDevice()");
		}

		//	Get default timeout from env or fix it if not set
		String	strTimeout;
		if ((strTimeout=System.getProperty("TANGO_TIMEOUT"))==null)
			strTimeout = "3000";
		try { dev_timeout = Integer.parseInt(strTimeout); }
		catch(Exception e) { e.printStackTrace(); }

		build_obj_for_timout(dev_timeout);	
		long	t0 = System.currentTimeMillis();

boolean with_net_accesss = false;
if (with_net_accesss)
{
		//	Check IDL revision
		if (obj._is_a("IDL:Tango/Device_3:1.0") == true)
		{
			//System.out.println(devname + " IDL:Tango/Device_3:1.0");
			device_3 = Device_3Helper.narrow(obj);
			device_2 = Device_2Helper.narrow(obj);
			idl_version = 3;
		}
		else
		if (obj._is_a("IDL:Tango/Device_2:1.0") == true)
		{
			//System.out.println(devname + " IDL:Tango/Device_2:1.0");
			device_2 = Device_2Helper.narrow(obj);
			idl_version = 2;
		}
		else
		if (obj._is_a("IDL:Tango/Device:1.0") == true)
		{
			//System.out.println(devname + " IDL:Tango/Device:1.0");
		}
		else
			Except.throw_non_supported_exception("TangoApi_DEVICE_IDL_UNKNOWN",
					devname + " has an IDL revision not suported !",
					"Connection.createDevice("+ devname +")");

		//	Create the default (IDL 1) device
		device = DeviceHelper.narrow(obj);
		
		long	t1 = System.currentTimeMillis();
//		System.out.println(devname + " narrow elapsed time :	" + (t1-t0) + " ms");
}
else
{
		//	replace   obj.is_a()   method without network access
		ParsedIOR		pior =
				new ParsedIOR(corba_str, ApiUtil.get_orb(), null);
        org.omg.IOP.IOR	ior = pior.getIOR();
		String			type_id = ior.type_id;

		 //	 DB connection ?
		if (type_id==null  || type_id.equals("IDL:omg.org/CORBA/Object:1.0"))
		{
			//	Create the default (IDL 1) device
			//	This one use default narrow to check device access.
			device = DeviceHelper.narrow(obj);
		}
		else	//	DeviceProxy
		{
			//	Check IDL revision
			if (type_id.equals("IDL:Tango/Device_3:1.0") == true)
			{
				//System.out.println(devname + " IDL:Tango/Device_3:1.0");
				device_3 = narrow_3(obj);
				device_2 = narrow_2(obj);
				idl_version = 3;
			}
			else
			if (type_id.equals("IDL:Tango/Device_2:1.0") == true)
			{
				//System.out.println(devname + " IDL:Tango/Device_2:1.0");
				device_2 = narrow_2(obj);
				idl_version = 2;
			}
			else
			if (type_id.equals("IDL:Tango/Device:1.0") == true)
			{
				//System.out.println(devname + " IDL:Tango/Device:1.0");
				idl_version = 1;
			}
			else
				Except.throw_non_supported_exception("TangoApi_DEVICE_IDL_UNKNOWN",
						devname + " has an IDL revision not suported !",
						"Connection.createDevice("+ devname +")");

			//	Create the default (IDL 1) device
			//	This one use default narrow to check device access.
			device = DeviceHelper.narrow(obj);
			//device = narrow(obj);
		}

		long	t1 = System.currentTimeMillis();
		//System.out.println(devname + " narrow elapsed time :	" + (t1-t0) + " ms");
}
	}
	//===================================================================
	/**
	 *	Replace esrf.fr.Tango.Device_3Helper.narrow(obj)
	 *		method without network acces (do not use  obj.is_a()).
	 */
	//===================================================================
	protected Device_3 narrow_3(org.omg.CORBA.Object obj)
	{
		fr.esrf.Tango._Device_3Stub stub = new fr.esrf.Tango._Device_3Stub();
		stub._set_delegate(((org.omg.CORBA.portable.ObjectImpl)obj)._get_delegate());
		return stub;
	}
	//===================================================================
	/**
	 *	Replace esrf.fr.Tango.Device_2Helper.narrow(obj)
	 *		method without network acces (do not use  obj.is_a()).
	 */
	//===================================================================
	protected Device_2 narrow_2(org.omg.CORBA.Object obj)
	{
		fr.esrf.Tango._Device_2Stub stub = new fr.esrf.Tango._Device_2Stub();
		stub._set_delegate(((org.omg.CORBA.portable.ObjectImpl)obj)._get_delegate());
		return stub;
	}
	//===================================================================
	/**
	 *	Replace esrf.fr.Tango.DeviceHelper.narrow(obj)
	 *		method without network acces (do not use  obj.is_a()).
	 */
	//===================================================================
	protected Device narrow(org.omg.CORBA.Object obj)
	{
		fr.esrf.Tango._DeviceStub stub = new fr.esrf.Tango._DeviceStub();
		stub._set_delegate(((org.omg.CORBA.portable.ObjectImpl)obj)._get_delegate());
		return stub;
	}
	//===================================================================
	//===================================================================
	public String get_ior()	throws DevFailed
	{
		Database	db;
		if (url.host==null)
			db = ApiUtil.get_db_obj();
		else
			db = ApiUtil.get_db_obj(url.host, url.strport);
		DbDevImportInfo	info = db.import_device(devname);
		return info.ior;
	}
	//===================================================================
	//===================================================================
	private String get_exported_ior()	throws DevFailed
	{
		Database	db;
		if (url.host==null)
		{
			db = ApiUtil.get_db_obj();
		}
		else
		{
			db = ApiUtil.get_db_obj(url.host, url.strport);
		}

		DbDevImportInfo	info = db.import_device(devname);
		String result = null;
		if (info.exported)
			result = info.ior;

		//	check if TACO
		if (info.is_taco)
			url.protocol = TACO;

		return result;
	}
	//===================================================================
	/**
	 *	Really build the device connection.
	 */
	//===================================================================
	protected void dev_import()	throws DevFailed
	{
		Database	db;
		String		local_ior = null;
		if (url.host==null)
			db = ApiUtil.get_db_obj();
		else
			db = ApiUtil.get_db_obj(url.host, url.strport);

		//	Check if device must be imported directly from IOR
		//---------------------------------------------------------
		if (ior==null)
		{
			DbDevImportInfo	info = db.import_device(devname);
			if (info.exported==false)
				Except.throw_connection_failed("TangoApi_DEVICE_NOT_EXPORTED",
					devname + " Not Exported !", "Connection("+ devname +")");
			local_ior = info.ior;
		}
		else
			local_ior = ior;
		try
		{
			// Import the TANGO device
			//--------------------------------
			createDevice(local_ior);
			//device.ping();
			already_connected = true;
		}
		catch(DevFailed e)
		{
			device = null;
			ior    = null;
			throw e;
		}
		catch(Exception e)
		{
			device = null;
			ior    = null;
			String	reason = "TangoApi_CANNOT_IMPORT_DEVICE";
			String	s = (already_connected)? "Re-" : "";
			String	desc   = new String("Cannot " + s + "import " + devname + 
													" :\n\t" + e.toString());
			String	origin = new String("Connection.dev_import(" + devname + ")");
			//e.printStackTrace();
			Except.throw_connection_failed(reason, desc, origin);
		}
	}
	//===================================================================
	/**
	 *	Import the tango database.
	 *
	 */
	//===================================================================
	private void connect_to_dbase() throws DevFailed
	{
		if (device==null)
 		{
			try
			{
				//	Prepeare the connection string
				//----------------------------------
				String	db_corbaloc =  new String("corbaloc:iiop:" +
												url.host + ":" +
												url.strport + "/database" );
				//	And connect to database.
				//----------------------------
				createDevice(db_corbaloc);
System.out.println("Connected to " + db_corbaloc);
			}
			catch(SystemException ex)
			{
				if (transparent_reconnection)
				{
					try
					{
						DbRedundancy	dbr = DbRedundancy.get_instance();
						String	th2 = dbr.get(url.host + ":" + url.port);

						//	Prepeare the connection string
						//----------------------------------
						String	db_corbaloc =  new String("corbaloc:iiop:" +
														th2 + "/database" );
						//	And connect to database.
						//----------------------------
						createDevice(db_corbaloc);
	System.out.println("Connected to " + db_corbaloc);
					}
					catch(SystemException e)
					{
						//e.printStackTrace();
						device = null;
						ior    = null;
						Except.throw_connection_failed("TangoApi_DATABASE_CONNECTION_FAILED",
								"Connection to database failed  !\n" + e,
								"connect_to_dbase(" + url.host + "," + url.strport + ")");
					}
				}
				else
				{
					//e.printStackTrace();
					device = null;
					ior    = null;
					Except.throw_connection_failed("TangoApi_DATABASE_CONNECTION_FAILED",
							"Connection to database failed  !\n" + ex,
							"connect_to_dbase(" + url.host + "," + url.strport + ")");
				}
			}
		}
	}
	
	//===================================================================
	/**
	 *	Import the tango database.
	 *
	 */
	//===================================================================
	private void dev_import_without_dbase() throws DevFailed
	{
		if (device==null)
 		{
			try
			{
				//	Prepeare the connection string
				//----------------------------------
				String	db_corbaloc =  new String("corbaloc:iiop:" +
												url.host + ":" +
												url.strport + "/" +
												devname.toLowerCase() );
				//System.out.println("db_corbaloc=" + db_corbaloc);
				//	And connect to device.
				//----------------------------
				createDevice(db_corbaloc);
			}
			catch(SystemException e)
			{
				device = null;
				ior    = null;
				e.printStackTrace();
				Except.throw_connection_failed("TangoApi_DEVICE_CONNECTION_FAILED",
						"Connection to device without database failed  !\n" + e,
						"Connection.dev_import_without_dbase(" + url.host + "," + url.strport + ")");
			}
		}
	}
	//===================================================================
	//===================================================================
	private org.omg.CORBA.Object build_obj_for_timout(int  millis)
	{
		//	Change Jacorb policy for timeout
		org.omg.CORBA.Policy	p =
			new org.jacorb.orb.policies.RelativeRoundtripTimeoutPolicy(10000*millis);
		obj._set_policy_override( new Policy[] {p},
							org.omg.CORBA.SetOverrideType.ADD_OVERRIDE);

		//	Save new timeout value
		//---------------------------------
		dev_timeout = millis;
		//System.out.println("Timeout set to " + dev_timeout);
		return obj;
	}	
	//===================================================================
	/**
	 *	Change the timeout value for a device call.
	 *
	 *	@param	millis		New value of the timeout in milliseconds.
	 *	@throws	DevFaild	if orb.create_policy throws an
	 *						org.omg.CORBA.PolicyError exception.
	 */
	//===================================================================
	public void set_timeout_millis(int millis) throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();

		//	Check if TACO device
		if (url.protocol==TACO)
		{
			taco_device.set_rpc_timeout(millis);
			return;
		}

		//	Else TANGO device		
		org.omg.CORBA.Object	newObj = build_obj_for_timout(millis);
		if (device_3!=null)
			device_3 = narrow_3(newObj);
		if (device_2!=null)
			device_2 = narrow_2(newObj);
		device = narrow(newObj);
	}
	//===================================================================
	/**
	 *	return the timeout value for a device call.
	 *
	 *	@return the value of the timeout in milliseconds.
	 *	@deprecated use get_timeout_millis() instead
	 */
	//===================================================================
	public int get_timeout()
	{
		return dev_timeout;
	}
	//===================================================================
	/**
	 *	return the timeout value for a device call.
	 *
	 *	@return the value of the timeout in milliseconds.
	 */
	//===================================================================
	public int get_timeout_millis() throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();
		//	Check if TACO device
		if (url.protocol==TACO)
			return taco_device.get_rpc_timeout();
		else
			//	Else TANGO device
			return dev_timeout;
	}



	//===========================================================
	/**
	 *	Build reason and origin of the exception
	 *	And throw it into a DevFailed exception
	 */
	//===========================================================
	protected void throw_dev_failed(Exception e, String command, boolean from_inout_cmd)
						throws DevFailed
	{
		if (e instanceof DevFailed)
			throw (DevFailed)e;

		String	desc;
		String	origin = (from_inout_cmd) ?
					new String(devname + ".command_inout(" + command + ")")
					: new String(devname + "." + command + ")");

		if (e.toString().indexOf("org.omg.CORBA.NO_RESPONSE")>=0  ||
			e.toString().indexOf("org.omg.CORBA.TIMEOUT")>=0      ||
			e.toString().indexOf("org.omg.CORBA.IMP_LIMIT")>=0)
		{
			desc = new String("Device (" +  devname+
						") timed out (>" + dev_timeout +" ms)!");
			Except.throw_communication_failed(new String(e.toString()), desc, origin);
		}
		else
		if (e.toString().indexOf("org.omg.CORBA.TRANSIENT")        >=0  ||
			e.toString().indexOf("org.omg.CORBA.UNKNOWN")          >=0  ||
			e.toString().indexOf("org.omg.CORBA.COMM_FAILURE")     >=0  ||
			e.toString().indexOf("org.omg.CORBA.OBJECT_NOT_EXIST") >=0)
		{
			desc = new String("Lost Connection during command : " + command);
			device = null;
			ior    = null;
			Except.throw_connection_failed(new String(e.toString()), desc, origin);
		}
		else
		if (e.toString().startsWith("java.lang.RuntimeException") )
		{
			desc = new String("API has catched a RuntimeException" + command);
			device = null;
			ior    = null;
			Except.throw_connection_failed(new String(e.toString()), desc, origin);
		}
		else
		{
			System.out.println("API has catched an exception for " +
								origin + " : \n" + e);
			desc = new String(e.getMessage());
			e.printStackTrace();
			Except.throw_communication_failed(new String(e.toString()), desc, origin);
		}
	}
	//===========================================================
	/**
	 *	Send a command to a device server.
	 *
	 *	@param command	Command name to send to the device.
	 *	@param	argin	input command argument.
	 *	@return the output argument of the command.
	 *	@throws DevFailed	
	 */
	//===========================================================
	public DeviceData command_inout(String command, DeviceData argin)
			throws DevFailed
	{
		Any	received = null;
		if (device==null && devname!=null)
			build_connection();

		//	Check if taco device
		if (url.protocol==TACO)
			return taco_device.command_inout(command, argin);

		//	Else tango call
		boolean done = false;
		//	try 2 times for reconnection if requested
		int	retries = (transparent_reconnection)? 2 : 1;
		for (int i=0 ; i<retries && done==false ; i++)
		{
			try{
				if (device_2!=null)
					received = device_2.command_inout_2(command, argin.extractAny(), dev_src);
				else
					received = device.command_inout(command, argin.extractAny());
				done = true;
			}
			catch(DevFailed e)
			{
				String	reason = "TangoApi_CANNOT_EXECUTE_COMMAND";
				String	desc   = new String("Cannot execute command " + command +
											" on " + devname);
				String	origin = new String("Connection.command_inout()");

				//	Check if (from database) DeviceNotDefined exception
				//		then add witch database in the existing DevFailed.
				if (e.errors[0].reason.equals("DB_DeviceNotDefined"))
				{
					String	d = e.errors[0].desc;
					int		idx = d.lastIndexOf("!");
					d = d.substring(0, idx) +
						"  " + url.host + ":" + url.port + " !";
					e.errors[0].desc = d;
				}
				//e.printStackTrace();
				Except.throw_connection_failed(e, reason, desc, origin);
			}
			catch(Exception e)
			{
				if (e.toString().indexOf("org.omg.CORBA.TRANSIENT")>=0 && i==0)
				{
					//System.out.println("Retrying...");
					device = null;
					ior    = null;
					build_connection();

					if (i==(retries-1))
						throw_dev_failed(e, command, true);
				}
				else
				{
					//if (command.equals("DbImportDevice"))
					//	System.out.println("command_inout(DbImportDevice for "+
					//		argin.extractString());
					throw_dev_failed(e, command, true);
				}
			}
		}
		return new DeviceData(received);
	}
	//===========================================================
	/**
	 *	Send a command to a device server.
	 *
	 *	@param command 	Command name.
	 *	@return the output argument of the command.
	 *	@throws DevFailed	
	 */
	//===========================================================
	public DeviceData command_inout(String command)
			throws DevFailed
	{
		DeviceData	argin = new DeviceData();
		return command_inout(command, argin);
	}

	//===========================================================
	/**
	 *	Execute a ping command to a device server.
	 *
	 *	@return the elapsed time for ping command in microseconds.
	 */
	//===========================================================
	public long ping() throws DevFailed
	{
		checkIfTango("ping");
		if (device==null && devname!=null)
			build_connection();
		long	t0 = System.currentTimeMillis();
		try{
			device.ping();
		}
		catch(DevFailed e)
		{
			String	reason = "TangoApi_CANNOT_PING_DEVICE";
			String	desc   = new String("Cannot ping " + devname);
			String	origin = new String("Connection.ping()");
			Except.throw_connection_failed(e, reason, desc, origin);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "ping", false);
		}

		long	t1 = System.currentTimeMillis();
		return (int) (t1-t0)*1000;	//	Set as micro seconds
	}
	//===========================================================
	/**
	 *	Execute a info command to a device server.
	 */
	//===========================================================
	public String[] black_box(int length) throws DevFailed
	{
		checkIfTango("black_box");
		if (device==null && devname!=null)
			build_connection();
		try{
			String[] result = device.black_box(length);
			return result;
		}
		catch(DevFailed e)
		{
			String	reason = "TangoApi_CANNOT_READ_BLACK BOX";
			String	desc   = new String("Cannot read black box on " + devname);
			String	origin = new String("Connection.black_box()");
			Except.throw_connection_failed(e, reason, desc, origin);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "black_box", false);
		}
		return null;
	}
	//===========================================================
	/**
	 *	Execute a info command to a device server.
	 */
	//===========================================================
	public DevInfo_3 info_3() throws DevFailed
	{
		checkIfTango("info");
		if (device==null && devname!=null)
			build_connection();
		try{
			DevInfo_3	info_3;
			if (device_3!=null)
				info_3 = device_3.info_3();
			else
			{
				DevInfo info = device.info();
				info_3 = new DevInfo_3( info.dev_class,
										info.server_id,
										info.server_host,
										info.server_version,
										info.doc_url,
										"");
			}
			return info_3;
		}
		catch(DevFailed e)
		{
			String	reason = "TangoApi_CANNOT_READ _DEVICE_INFO";
			String	desc   = new String("Cannot read device info on " + devname);
			String	origin = new String("Connection.info()");
			Except.throw_connection_failed(e, reason, desc, origin);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "info", false);
		}
		return null;
	}
	//===========================================================
	/**
	 *	Execute a info command to a device server.
	 */
	//===========================================================
	public DevInfo info() throws DevFailed
	{
		checkIfTango("info");
		if (device==null && devname!=null)
			build_connection();
		try{
			DevInfo info = device.info();
			return info;
		}
		catch(DevFailed e)
		{
			String	reason = "TangoApi_CANNOT_READ _DEVICE_INFO";
			String	desc   = new String("Cannot read device info on " + devname);
			String	origin = new String("Connection.info()");
			Except.throw_connection_failed(e, reason, desc, origin);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "info", false);
		}
		return null;
	}
	//===========================================================
	/**
	 *	Execute a command_list_query command to a device server.
	 */
	//===========================================================
	public CommandInfo[] command_list_query() throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();
		try
		{
			CommandInfo[] result;
			if (url.protocol==TACO)
				result = taco_device.commandListQuery();
			else
			{
				//	Check IDL revision
				if (device_2!=null)
				{
					DevCmdInfo_2[]	tmp = device_2.command_list_query_2();
					result = new CommandInfo[tmp.length];
					for (int i=0 ; i<tmp.length ; i++)
						result[i] = new CommandInfo(tmp[i]);
				}
				else
				{
					DevCmdInfo[]	tmp = device.command_list_query();
					result = new CommandInfo[tmp.length];
					for (int i=0 ; i<tmp.length ; i++)
						result[i] = new CommandInfo(tmp[i]);
				}
			}
			return result;
		}
		catch(DevFailed e)
		{
			String	reason = "TangoApi_CANNOT__READ_CMD_LIST";
			String	desc   = new String("Cannot read command list for " + devname);
			String	origin = new String("Connection.command_list_query()");
			Except.throw_connection_failed(e, reason, desc, origin);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "command_list_query", false);
		}
		return null;
	}

	//==========================================================================
	/**
	 *	Returns the IDL version supported
	 *
	 *	@return  the IDL version supported .
	 */
	//==========================================================================
	public int get_idl_version() throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();
		return idl_version;
	}
	//==========================================================================
	/**
	 *	returns the device data source
	 *
	 *	@return  data source (CACHE_DEV, CACHE or DEV).
	 */
	//==========================================================================
	public DevSource get_source()
	{
		return dev_src;
	}
	//==========================================================================
	/**
	 *	Set the device data source
	 *
	 *	@param new_src	new data source (CACHE_DEV, CACHE or DEV).
	 */
	//==========================================================================
	public void set_source(DevSource new_src)
	{
		dev_src = new_src;
	}
	//==========================================================================
	/**
	 *	return the device connected name.
	 */
	//==========================================================================
	public String get_name()
	{
		return devname;
	}
	//==========================================================================
	/**
	 *	return the device connected dexcription.
	 */
	//==========================================================================
	public String description() throws DevFailed
	{
		if (device==null && devname!=null)
			build_connection();
		return device.description();
	}
	//==========================================================================
	/**
	 *	return the administartion device name.
	 */
	//==========================================================================
	public String adm_name() throws DevFailed
	{
		checkIfTango("adm_name");
		if (device==null && devname!=null)
			build_connection();
		String	name = null;
		try
		{
			name = device.adm_name();
		}
		catch(Exception e)
		{
			throw_dev_failed(e, "adm_name" , false);
		}
		
		//	If no database, add syntax to be used in DeviceProxy constructor.
		if (url.use_db==false)
			name = "tango://" + url.host + ":" + url.port +
									"/" + name + "#dbase=no";
		return name;
	}
	//==========================================================================
	/**
	 *	return the name of connection (host:port)
	 */
	//==========================================================================
	public String get_tango_host() throws DevFailed
	{
		checkIfTango("get_tango_host");
		return new String(url.host + ":" + url.strport);
	}

	//==========================================================================
	/**
	 *	return true if device is a taco device
	 */
	//==========================================================================
	public boolean is_taco()
	{
		return (url.protocol==TACO);
	}
	//==========================================================================
	/**
	 *	if not a TACO command then throw a DevFailed Exception.
	 *
	 *	@param cmdname	command name to be put inside reason and origin fields.
	 */
	//==========================================================================
	protected void checkIfTaco(String cmdname) throws DevFailed
	{
		if (is_taco()==false)
			Except.throw_non_supported_exception(
						"TangoApi_NOT_TANGO_CMD",
						new String(cmdname + " is NOT a TANGO command."),
						new String(cmdname + "()"));
	}
	//==========================================================================
	/**
	 *	if not a TACO command then throw a DevFailed Exception.
	 *
	 *	@param cmdname	command name to be put inside reason and origin fields.
	 */
	//==========================================================================
	protected void checkIfTango(String cmdname) throws DevFailed
	{
		if (is_taco()==true)
			Except.throw_non_supported_exception(
						"TangoApi_NOT_TACO_CMD",
						new String(cmdname + " is NOT a TACO command."),
						new String(cmdname + "()"));
	}
	//==========================================================================
	/**
	 *	return the value of transparent_reconnection
	 */
	//==========================================================================
	public boolean get_transparency_reconnection()
	{
		return transparent_reconnection;
	}
	//==========================================================================
	/**
	 *	set the value of transparent_reconnection
	 */
	//==========================================================================
	public void set_transparency_reconnection(boolean val)
	{
		transparent_reconnection = val;
	}
}
