// File:          FileUI.java
// Created:       2002-09-18 12:57:09, erik
// By:            <erik@assum.net>
// Time-stamp:    <2002-10-18 10:25:57, erik>
// 
// $Id: FileUI.java,v 1.2 2005/08/26 13:44:40 ounsy Exp $
// 
// Description:       

package explorer.ui;

import java.io.*;
import java.awt.Color;
import java.awt.JobAttributes;
import java.awt.PageAttributes;
import java.awt.PrintJob;
import java.awt.event.*;
import java.awt.image.BufferedImage;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;

/**
 * Usefull class to print
 * @author GIRARDOT
 */
class DTPrinter{
    /**
     * Displays the print dialog and sends a component snapshot to the printer.
     * Using the printerResolution can be usefull to print your frame bigger
     * or smaller. A screen typicaly has a resolution of ~100dpi. This method does
     * not support multiple page documents.
     * @param comp JFrame to be printed out.
     * @param title Title of the print dialog.
     * @param fitToPage True to fit the component to the page (printerResolution ignored).
     * @param printerResolution Printer resolution when fitToPage is not enabled.
     */
    public static void printFrame(JFrame comp,String title,boolean fitToPage,int printerResolution) {

      // Default

      PageAttributes pa = new PageAttributes();
      JobAttributes ja = new JobAttributes();
      pa.setPrintQuality(PageAttributes.PrintQualityType.HIGH);
      pa.setColor(PageAttributes.ColorType.COLOR);
      pa.setMedia(PageAttributes.MediaType.A4);
      if(fitToPage) {
        // Default resolution
        pa.setPrinterResolution(72);
      } else {
        pa.setPrinterResolution(printerResolution);
      }
      ja.setMaxPage(1);
      ja.setMinPage(1);
      ja.setDialog(JobAttributes.DialogType.COMMON);

      // Displays print window
      PrintJob printJob = java.awt.Toolkit.getDefaultToolkit().getPrintJob(comp, title, ja, pa);

      if (printJob != null) {

        // Get image dimension
        int w = comp.getSize().width;
        int h = comp.getSize().height;
        int tx,ty;

        // Make a screenshot of the graph
        BufferedImage img = new BufferedImage(w,h,BufferedImage.TYPE_INT_RGB);
        Color oldBackground = comp.getBackground();
        comp.setBackground(Color.WHITE);
        comp.printAll(img.getGraphics());
        comp.setBackground(oldBackground);

        try {

          // Fit the draw to the page by changing the printer resolution
          if (fitToPage) {

            // Get page dimension (should be given for 72dpi resolution)
            int wp = printJob.getPageDimension().width - 72; // 0.5inch margin
            int hp = printJob.getPageDimension().height - 72; // 0.5inch margin

            // Fit the graph to the page
            double ratioW = (double) w / (double) wp;
            double ratioH = (double) h / (double) hp;
            double nResolution;

            if (ratioW > ratioH) {

              // We get ratioW
              // We center verticaly
              nResolution = 72.0 * ratioW;
              tx = (int) (nResolution * 0.5);
              double cH = nResolution / 72.0 * (double) hp - (double) h;
              ty = (int) (0.5 * (nResolution + cH));

            } else {

              // We get ratioH
              // We center horizontaly
              nResolution = 72.0 * ratioH;
              double cW = nResolution / 72.0 * (double) wp - (double) w;
              tx = (int) (0.5 * (nResolution + cW));
              ty = (int) (nResolution * 0.5);

            }

            pa.setPrinterResolution((int) (nResolution + 0.5));

          } else {

            // 0.5 inch margin
            tx = printerResolution / 2;
            ty = printerResolution / 2;

          }

          // Print it
          java.awt.Graphics g = printJob.getGraphics();
          g.translate(tx, ty);
          g.setClip(0, 0, w, h);
          g.drawImage(img, 0, 0, null);
          g.dispose();
          printJob.end();

        } catch (Exception e) {

          e.printStackTrace();
          JOptionPane.showMessageDialog(comp, "Exception occured while printing\n" + e.getMessage(),
                                        title, JOptionPane.ERROR_MESSAGE);

        }

      }

    }
}


/**
 * <code>FileUI</code> is responsible for handling the file part of the
 * user interface. It will instansiate all menu- and toolbar items, and
 * handle their state. The items open, new, save, and save as are only
 * available in administrator mode. Save is only available when the
 * filename is set, either through open or save as. The icons are from
 * the java ui standard icon-package.
 *
 * @author <a href="mailto:erik@assum.net">Erik Assum</a>
 * @version $Revision: 1.2 $
 */
public class FileUI {
    UIBit saveBit;
    File file;
    JFileChooser fc;
    FileHandler main;
    JToolBar toolbar;
    
    public FileUI(FileHandler main, JToolBar toolbar, DTMenuBar menubar,
		  boolean isAdmin, final JFrame mainFrame) {
	UIBit saveAsBit;
	UIBit newBit;
	UIBit openBit;
	UIBit printBit;
	
	ImageIcon saveIcon, saveAsIcon, newIcon, openIcon, printIcon;

	this.main = main;
	this.toolbar = toolbar;
	
	saveIcon =
	    new ImageIcon(getClass().getResource("/fr/esrf/tangoatk/widget/util/Save16.gif"));
	saveAsIcon =
	    new ImageIcon(getClass().getResource("/fr/esrf/tangoatk/widget/util/SaveAs16.gif"));
	newIcon =
	    new ImageIcon(getClass().getResource("/fr/esrf/tangoatk/widget/util/New16.gif"));
	openIcon =
	    new ImageIcon(getClass().getResource("/fr/esrf/tangoatk/widget/util/Open16.gif"));
	printIcon = 
	    new ImageIcon(getClass().getResource("/explorer/ui/printer.gif"));

	saveBit = new UIBit("Save", new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    save();
		}
	    }, saveIcon);

	saveBit.setEnabled(false);
	saveBit.setAccelerator('S');


	saveAsBit = new UIBit("Save as..", "Save as", new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    saveAs();
		}
	    }, saveAsIcon);
	saveAsBit.setAccelerator('A');
	saveAsBit.setEnabled(isAdmin);
	newBit = new UIBit("New", new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    newFile();
		}
	    }, newIcon);
	newBit.setAccelerator('N');
	newBit.setEnabled(isAdmin);
	openBit = new UIBit("Open...", "Open file", new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    open();
		}
	    }, openIcon);
	
	openBit.setAccelerator('O');
	openBit.setEnabled(isAdmin);
	printBit = new UIBit("Print..", "Print...", new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    DTPrinter.printFrame(mainFrame,"DeviceTree",true,0);
		}
	    }, printIcon);
	menubar.add2FileMenu(openBit.getItem(), 0);
	menubar.add2FileMenu(newBit.getItem(), 1);
	menubar.add2FileMenu(new JSeparator(), 2);
	menubar.add2FileMenu(saveBit.getItem(), 3);
	menubar.add2FileMenu(saveAsBit.getItem(), 4);
	menubar.add2FileMenu(new JSeparator(),5);
	menubar.add2FileMenu(printBit.getItem(),6);
	
	toolbar.add(openBit.getButton());
	toolbar.add(newBit.getButton());
	toolbar.add(saveBit.getButton());
	toolbar.add(saveAsBit.getButton());
	toolbar.add(printBit.getButton());

	menubar.setQuitHandler(new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    quit();
		}
	    });
		
	fc = new JFileChooser(new File("."));

	fc.addChoosableFileFilter(new FileFilter() {

		public boolean accept(File f) {
		    if (f.isDirectory()) {
			return true;
		    }

		    String extension = getExtension(f);

		    if (extension != null && extension.equals("xml"))
			return true;
		    
		    return false;
		}

		public String getDescription() {
		    return "xml files ";
		}

	    });
    }

    /**
     * <code>getExtension</code> returns the extension of a given file,
     * that is the part after the last `.' in the filename.
     *
     * @param f a <code>File</code> value
     * @return a <code>String</code> value
     */
    public String getExtension(File f) {
	String ext = null;
	String s = f.getName();
	int i = s.lastIndexOf('.');
	
	if (i > 0 &&  i < s.length() - 1) {
	    ext = s.substring(i+1).toLowerCase();
	}
	return ext;
    }

    /**
     * <code>quit</code> calls its FileHandler <code>quit()</code> method
     */
    public void quit() {
	main.quit();
    }
    
    /**
     * <code>open</code> shows a file-selection dialog and calls its
     * FileHandler <code>open(file)</code> method.
     *
     */
    public void open() {
	int returnVal = fc.showOpenDialog(toolbar.getRootPane().getParent());

	if(returnVal == JFileChooser.APPROVE_OPTION) {
	    file = fc.getSelectedFile();
	    saveBit.setEnabled(true);
	    main.open(file);
	}
    }

    
    /**
     * <code>newFile</code> calls its FileHandler<code>newFile</code>
     * method.
     */
    public void newFile() {
	main.newFile();
    }
	
    /**
     * <code>save</code> calls its FileHandler <code>save(file)</code>
     * method.
     */
    public void save() {
	main.save(file);
    }

    /**
     * <code>confirm</code> a dialog which is used to make the user
     * confirm a save to an already existing file.
     * @return a <code>boolean</code> value
     */
    protected boolean confirm() {
	return JOptionPane.showConfirmDialog(null, "The file " +
					     file.toString() + " exists, " +
					     "do you want to overwrite it?",
					     "Alert",
					     JOptionPane.YES_NO_OPTION) ==
	    JOptionPane.YES_OPTION;
    }

    /**
     * <code>saveAs</code> shows a file selection dialog and calls
     * its FileHandler <code>save(file)</code> method.
     */
    public void saveAs() {
	int returnVal = fc.showSaveDialog(toolbar.getRootPane().getParent());

	if(returnVal == JFileChooser.APPROVE_OPTION) {
	    file = fc.getSelectedFile();

	    if (!file.exists() || confirm()) {
		if (getExtension(file) == null) {

		    file = new File(file.getAbsolutePath() + ".xml");
		}
		main.save(file);
		saveBit.setEnabled(true);
	    }
	}
    }
}
