// File:          CommandTable.java
// Created:       2002-09-13 12:47:34, erik
// By:            <erik@assum.net>
// Time-stamp:    <2003-01-16 15:50:20, erik>
// 
// $Id: CommandTable.java,v 1.2 2005/08/26 13:44:40 ounsy Exp $
// 
// Description:       

package explorer;

import java.awt.Component;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JMenuItem;
import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import explorer.ui.PreferencesDialog;
import fr.esrf.tangoatk.core.ICommand;
import fr.esrf.tangoatk.core.command.VoidVoidCommand;
import fr.esrf.tangoatk.widget.command.AnyCommandViewer;
import fr.esrf.tangoatk.widget.dnd.NodeFactory;
import fr.esrf.tangoatk.widget.util.HelpWindow;

/**
 * Class representing all things specific to the command table.
 * It extends <code>explorer.EntityTable</code>, the javadoc of which you can see
 * for further informations
 * @author Erik ASSUM
 */
public class CommandTable extends EntityTable {
    /** Adds the option to execute command in menu */
    JMenuItem execute;
    /** To view the command*/
    AnyCommandViewer commandViewer;
    /** The "execute" panel*/
    JDialog commandDialog;
    
    /**
     * Creates a new <code>CommandTable</code> instance.
     *
     * @param isAdmin a <code>boolean</code> value, which is true
     * if this is an administrator session.
     */
    public CommandTable(CommandTableModel model, Preferences prefs,
			boolean isAdmin) {
	preferences = prefs;
	initComponents(model, isAdmin);
	flavor = NodeFactory.MIME_COMMAND;
	setModel(model);
    }
    
    /**
     * Shows the "execute" panel
     */
    protected void showExecuteDialog() {
	ICommand command;

	if (getSelectedRow() == -1) return;

	command = (ICommand)model.getEntityAt(getSelectedRow());

	if (command instanceof VoidVoidCommand) {
	    command.execute();
	    return;
	}

	commandViewer = new AnyCommandViewer();
	commandDialog = new JDialog();
	commandViewer.initialize(command);
	commandViewer.setDeviceButtonVisible(false);
	commandViewer.setDescriptionVisible(true);
	//	commandViewer.setInfoButtonVisible(true);
	commandViewer.setBorder(null);
	String title = command.getAlias() != null ?
	    command.getAlias() :
	    command.getName();

	if (!command.takesInput()) {
	    command.execute();
	}

	commandDialog.setTitle(title);
	commandDialog.getContentPane().add(commandViewer);	
	commandDialog.pack();
	commandDialog.show();

    }

    /**
     * Initializes the "help" window
     */
    void initHelp() {
	helpUrl = "/explorer/html/CommandTableHelp.html";
	HelpWindow.getInstance().addCategory("Command Table", "Command table",
					     getClass().getResource(helpUrl));

    }

    /**
     * Menus initialization for the command
     */
    void initMenus() {
	execute = new JMenuItem("Execute...");
	execute.addActionListener(new ActionListener () {
		public void actionPerformed(ActionEvent evt) {
		    showExecuteDialog();
		}
	    });
	//	popup.add(execute);
    }
    
    /**
     * Presentation preferences initialization for the command table (visible fields)
     */
    void initPreferences() {
	PreferencesDialog.getInstance().addCategory("Command table",
						    "visible fields",
						    new ViewDialog(model));
    }

    /**
     * Manages the "drop" of a command in the table due to drag and drop
     * @param name the name of the command
     * @param flavor its <code>DataFlavor</code> representation for drag and drop
     */
    protected void dtHandleDrop(String name, DataFlavor flavor) {
	String mimeType = flavor.getMimeType();

	if (mimeType.startsWith(NodeFactory.MIME_COMMAND)) {
	    ((CommandTableModel)model).addCommand(name);
	    return;
	}
    }

    /**
     * Manages the right clicking over the command table to show
     * the corresponding menu
     * @param evt the <code>MouseEvent</code> for the "click"
     */
    protected void showPopup(MouseEvent evt) {
	int row = getRowAtPoint(evt.getPoint());

	if (row == -1) return;

	if (evt.isPopupTrigger()) {
	    setRowSelectionInterval(row, row);	
	    popup.show(evt.getComponent(), evt.getX(), evt.getY());
	}
    }

    /**
     * Manages the clicking over the command table to select
     * the command and launch the "execute" panel when necessary
     * @param evt the <code>MouseEvent</code> for the "click"
     */
    protected void entityTableMouseClicked(MouseEvent evt) {
	int row = getRowAtPoint(evt.getPoint());
	int column = 0;
	if (row == -1) return;
	setRowSelectionInterval(row, row);	
	ICommand command = (ICommand)model.getEntityAt(row);

	if (model.isExecuteColumn(getSelectedColumn()))
	    showExecuteDialog();
    }
    
    /**
     * Creates/adds a column in the command table
     * @param name the name of the column
     * @param i the index of the column in the model which is to be displayed 
     * by this TableColumn
     */
    TableColumn createTableColumn(String name, int i) {
	if (EntityTableModel.DEVICE == name) {// device column;
	    return new TableColumn(i, 75, deviceRenderer, null);
	}
	return new TableColumn(i, 75, new TableCellRenderer() {
		JButton renderer;

		public Component
		    getTableCellRendererComponent(JTable table,
						  Object value,
						  boolean isSelected,
						  boolean focus,
						  int row,
						  int column) {
		    if (renderer == null) renderer = new JButton();

		    try {
			String commandName =
			    model.getValueAt(row, column).toString();
			renderer.setText(commandName);
		    } catch (Exception e) {
			System.out.println("CommandTableModel " + e);
		    }
	
		    return renderer;
		}
	    }, null);
    }
}
