// File:          AttributeTableAdapter.jaav
// Created:       2002-09-10 13:53:19, erik
// By:            <erik@assum.net>
// Time-stamp:    <2003-01-30 11:23:20, erik>
// 
// $Id: AttributeTableModel.java,v 1.3 2005/08/26 13:44:40 ounsy Exp $
// 
// Description:       

package explorer;

import java.util.ArrayList;

import javax.swing.JComponent;
import javax.swing.JLabel;

import com.braju.format.Format;

import fr.esrf.TangoDs.AttrManip;
import fr.esrf.tangoatk.core.AttributeStateEvent;
import fr.esrf.tangoatk.core.ConnectionException;
import fr.esrf.tangoatk.core.ErrorEvent;
import fr.esrf.tangoatk.core.IAttribute;
import fr.esrf.tangoatk.core.IEntity;
import fr.esrf.tangoatk.core.IImageListener;
import fr.esrf.tangoatk.core.INumberImage;
import fr.esrf.tangoatk.core.INumberScalar;
import fr.esrf.tangoatk.core.INumberScalarListener;
import fr.esrf.tangoatk.core.INumberSpectrum;
import fr.esrf.tangoatk.core.ISpectrumListener;
import fr.esrf.tangoatk.core.IStateListener;
import fr.esrf.tangoatk.core.IStringScalar;
import fr.esrf.tangoatk.core.IStringScalarListener;
import fr.esrf.tangoatk.core.NumberImageEvent;
import fr.esrf.tangoatk.core.NumberScalarEvent;
import fr.esrf.tangoatk.core.NumberSpectrumEvent;
import fr.esrf.tangoatk.core.Property;
import fr.esrf.tangoatk.core.StateEvent;
import fr.esrf.tangoatk.core.StatusEvent;
import fr.esrf.tangoatk.core.StringScalarEvent;
import fr.esrf.tangoatk.widget.util.ATKConstant;

/**
 * Implements parts of the table model needed by AttributeTable
 * @author Erik ASSUM
 */
public class AttributeTableModel extends EntityTableModel {

    static String LABEL         = "Label";
    static String VALUE         = "Value";
    static String DISP_UNIT     = "Disp. Unit";
    static String MIN_VAL       = "Min value";
    static String MAX_VAL       = "Max value";
    static String MIN_ALARM     = "Min alarm";
    static String MAX_ALARM     = "Max alarm";
    static String UNIT          = "Unit";
    static String DATA_FORMAT   = "Data format";
    static String DATA_TYPE     = "Data type";
    static String DESCRIPTION   = "Description";
    static String WRITABLE      = "Writable";
    static String WRITABLE_NAME = "Writable name";
    static String MAX_Y         = "Max dim y";
    static String MAX_X         = "Max dim x";
    static String STD_UNIT      = "Standar unit";
    static String SET           = "Set";
    
    /**
     * Class constructor, initializer, with possibility to set status and preferences.
     * @param status the status
     * @param preferences the preferences
     */
    public AttributeTableModel(Status status, Preferences preferences) {
	this.status = status;
	entities = new fr.esrf.tangoatk.core.AttributeList();
	adapters = new ArrayList();
	columnIdentifiers = new ArrayList();
	this.preferences = preferences;
	setRefreshInterval(preferences.getInt("refreshInterval", 1000));
    }

    /**
     * Columns initialization. Sets the columns titles
     */
    void initColumns() {
	columnIdentifiers.add(DEVICE); // , true, deviceRenderer);
	columnIdentifiers.add(NAME); // , true);
	columnIdentifiers.add(LABEL); // , true);
	columnIdentifiers.add(VALUE); // , true, entityRenderer);
	columnIdentifiers.add(DISP_UNIT); // , true);
	columnIdentifiers.add(DESCRIPTION); // , true);
	columnIdentifiers.add(UNIT); // , true);
	columnIdentifiers.add(WRITABLE); // , true);
	columnIdentifiers.add(SET); // , true,

columnNames = new ArrayList(columnIdentifiers);

	columnNames.add(MIN_ALARM); // , false
	columnNames.add(MAX_ALARM); // , false
	columnNames.add(DATA_FORMAT); // , false
	columnNames.add(DATA_TYPE); // , false
	columnNames.add(MIN_VAL); // , false
	columnNames.add(MAX_VAL); // , false
	columnNames.add(WRITABLE_NAME); // , false
	columnNames.add(MAX_X); // , false
	columnNames.add(MAX_Y); // , false
	columnNames.add(LEVEL); // , false
	columnNames.add(STD_UNIT); // , false

	fireTableStructureChanged();
    }

    /**
     * Gives a prefix representing the name of the attribute table.
     * You might need this to reorganize the table (like adding/removing a column)
     * @return the <code>StringBuffer</code> containing the prefix
     */
    StringBuffer getPreferencePrefix() {
	return new StringBuffer("AttributeTable.");
    }
    
    /**
     * Returns the value to put in the specified row and column
     * @param row the row index
     * @param column the column index
     * @return the corresponding value, <code>null</code> when no value can correspond
     * with the specified row and column
     */
    public Object getValueAt(int row, int column) {
 
	Object o   = adapters.get(row);
	AttributeAdapter adapter = (AttributeAdapter)o;
	IAttribute attribute = adapter.getAttribute();
	
	String header = getColumnName(column); 

	if (DEVICE == header) 
	    return getDeviceAlias(attribute.getDevice());
	if (NAME == header)
	    return getEntityAlias(attribute);
	if (LABEL == header)
	    return attribute.getLabel();
	if (VALUE == header) 
	    return adapter;
	if (DISP_UNIT == header)
	    return attribute.getDisplayUnit();
	if (MIN_VAL == header)
	    if (attribute instanceof INumberScalar) 
		return new Double(((INumberScalar)attribute).getMinValue());
	    else 
		return new Double(Double.NaN);
	if (MAX_VAL == header)
	    if (attribute instanceof INumberScalar) 
		return new Double(((INumberScalar)attribute).getMaxValue());
	    else
		return new Double(Double.NaN);
	if (MIN_ALARM == header)
	    if (attribute instanceof INumberScalar) 
		return new Double(((INumberScalar)attribute).getMinAlarm());
	    else 
		return new Double(Double.NaN);
	if (MAX_ALARM == header)
	    if (attribute instanceof INumberScalar) 
		return new Double(((INumberScalar)attribute).getMinAlarm());
	    else 
		return new Double(Double.NaN);
	if (UNIT == header)
	    return attribute.getUnit();
	if (DATA_FORMAT == header)
	    return adapter.getDataFormat();
	if (DATA_TYPE == header)
	    return adapter.getDataType();
	if (DESCRIPTION == header)
	    return attribute.getDescription();
	if (WRITABLE == header)
	    return adapter.getWritable();

	if (WRITABLE_NAME == header)
	    return adapter.getWritableName();

	if (MAX_X == header)
	    return new Integer(attribute.getMaxXDimension());
	if (MAX_Y == header)
	    return new Integer(attribute.getMaxYDimension());
	if (LEVEL == header)
	    return adapter.getLevel();
	if (SET == header)
	    return attribute.isWritable() ? SET : "";
 	return null;
    }

    public void load(String name, String alias, String deviceAlias)
	throws ConnectionException {
	IAttribute attribute;

	attribute = (IAttribute)entities.add(name);
//	if (alias != null) attribute.setAlias(alias);
//	if (deviceAlias != null) attribute.getDevice().setAlias(deviceAlias);
	addAttribute(attribute);
    }	

    /**
     * Adds an attribute in the table. Uses <code>addAttribute(...)</code>
     * @param attribute the attribute to add
     */
    public void addEntity(IEntity attribute) {
	entities.add(attribute);
	addAttribute((IAttribute) attribute);
    }

    /**
     * Adds an attribute in the table if it is not already present.
     * @param attribute the attribute to add
     */
    public void addAttribute(IAttribute attribute) {
	if (exists(attribute)) return;

	adapters.add(new AttributeAdapter(attribute, adapters.size()));
	fireTableRowsInserted(adapters.size() - 1, adapters.size() - 1);
    }

    /**
     * Adds a number scalar attribute in the table. Uses <code>addAttribute(...)</code>
     * @param name the name of the attribute to add
     */
    protected void addNumberScalar(String name) {
	INumberScalar scalar;
	try {
	    scalar = (INumberScalar)entities.add(name);
	    addAttribute(scalar);
	} catch (ConnectionException e) {
	    //	    status.status(getRootPane().getParent(),
	    //			"Cannot load attribute " + name, e);

	    /*RG comment : TODO - Exception to manage */
	}
    }

    /**
     * Adds a string scalar attribute in the table. Uses <code>addAttribute(...)</code>
     * @param name the name of the attribute to add
     */
    protected void addStringScalar(String name) {
	IStringScalar scalar;
	try {
	    scalar = (IStringScalar)entities.add(name);
	    addAttribute(scalar);
	} catch (ConnectionException e) {
	    status.status("Cannot load attribute " + name, e);
	}
    }

    /**
     * Adds a number spectrum attribute in the table. Uses <code>addAttribute(...)</code>
     * @param name the name of the attribute to add
     */
    protected void addNumberSpectrum(String name) {
	INumberSpectrum spectrum;
	try {
	    spectrum = (INumberSpectrum)entities.add(name);
	    addAttribute(spectrum);
	} catch (ConnectionException e) {
	    status.status("Cannot load attribute " + name, e);
	}
    }

    /**
     * Adds a number image attribute in the table. Uses <code>addAttribute(...)</code>
     * @param name the name of the attribute to add
     */
    protected void addNumberImage(String name) {
	INumberImage image;
	try {
	    image = (INumberImage)entities.add(name);
	    addAttribute(image);
	} catch (ConnectionException e) {
	    status.status("Cannot load attribute " + name, e);
	}
    }


    /**
     * Method that returns a boolean representing if the specified column is
     * the "set" column or not
     * @param i the column index
     * @return <code>true</code> if the corresponding column is the "set" column,
     * <code>false</code> otherwise
     */
    boolean isExecuteColumn(int i) {
	return columnModel.getColumn(i).getHeaderValue() == SET;
    }

    /**
     * A listener for the attributes in table
     * @author Erik ASSUM
     */
    class AttributeAdapter implements IStringScalarListener,
				      INumberScalarListener,
				      ISpectrumListener,
				      IImageListener,
				      IStateListener,
				      EntityAdapter {
	int row;
	IAttribute attribute;
	String last = "See tab";
	String format = "";
	double doubleVal;

	/**
	 * returns the <code>IEntity</code> representation of the attribute
	 */
	public IEntity getEntity() {
	    return attribute;
	}

	/**
	 * Refreshes the properties of this attribute
	 */
	public void reloadProperties() {
	    fireTableRowsUpdated(row, row);
	}
	
	/**
	 * Class constructor, initializer. Associates the listener with an attribute 
	 * and a row in the table
	 * @param attribute the attribute
	 * @param row the row index
	 */
	public AttributeAdapter(IAttribute attribute, int row) {
	    this.row = row;
	    this.attribute = attribute;
	    attribute.getDevice().addStateListener(this);

	    if (attribute instanceof INumberScalar) {
		last = "----";
		((INumberScalar)attribute).addNumberScalarListener(this);
		format = attribute.getFormat();
	    }

	    if (attribute instanceof IStringScalar) {
		last = "----";
		((IStringScalar)attribute).addStringScalarListener(this);
	    }

	    if (attribute instanceof INumberSpectrum) {
		((INumberSpectrum)attribute).addSpectrumListener(this);
	    }

	    if (attribute instanceof INumberImage) {
		((INumberImage)attribute).addImageListener(this);
	    }
	}

	/**
	 * Method to remove listener from the associated attribute.
	 * Called when removing an attribute from table
	 */
	public void remove() {
	    attribute.getDevice().removeStateListener(this);
	    if (attribute instanceof INumberScalar) {
		((INumberScalar)attribute).removeNumberScalarListener(this);
	    } else 
		if (attribute instanceof IStringScalar) {
		    ((IStringScalar)attribute).removeStringScalarListener(this);
		} else
		    if (attribute instanceof INumberSpectrum) {
			((INumberSpectrum)attribute).removeSpectrumListener(this);
		    } else
			if (attribute instanceof INumberImage) {
			    ((INumberImage)attribute).removeImageListener(this);
			}
	}

	/**
	 * useless method (empty)
	 * @param evt unused
	 */
	public void statusChange(StatusEvent evt) {
	    /* RG comment : TODO - what was it for ? */
	}

	/**
	 * Changes the state of this listener according
	 * a <code>StateEvent</code>
	 * @param evt the <code>StateEvent</code>
	 */
	public void stateChange(StateEvent evt) {
	    JComponent r =
		getDeviceRenderer(getDeviceAlias(attribute.getDevice()));
	    if (r == null) return;
		
	    r.setBackground(ATKConstant.getColor4State(evt.getState()));
	    r.setToolTipText(evt.getState());
	    fireTableRowsUpdated(row, row);
	}

	/**
	 * Gets the value renderer for this listener. For more informations, see :
	 * <code>explorer.EntityTableModel.getRenderer(...)</code>
	 * @return the value renderer
	 */
	JComponent getValueRenderer() {
	    return getRenderer(this, VALUE);
	}

	/**
	 * useless method (empty)
	 * @param evt unused
	 */
	public void errorChange(ErrorEvent evt) {
	    /* RG comment : TODO - what was it for ? */
	}
    
	/**
	 * Changes the state of this listener according
	 * an <code>AttributeStateEvent</code>
	 * @param evt the <code>AttributeStateEvent</code>
	 */
	public void stateChange(AttributeStateEvent evt) {
	    JComponent r = getValueRenderer();
	    if (r == null) return;
	    
	    r.setToolTipText(evt.getState());
	    r.setBackground(ATKConstant.getColor4Quality(evt.getState()));
	    fireTableRowsUpdated(row, row);
	}

	/**
	 * Changes the state of this listener according
	 * an <code>StringScalarEvent</code>
	 * @param evt the <code>StringScalarEvent</code>
	 */
	public void stringScalarChange(StringScalarEvent evt) {
	    last = evt.getValue();
	    updateRenderer();
	}
	
	/**
	 * Changes the state of this listener according
	 * an <code>NumberScalarEvent</code>
	 * @param evt the <code>NumberScalarEvent</code>
	 */
	public void numberScalarChange(NumberScalarEvent evt) {
	    doubleVal = evt.getValue();
	    last = Double.toString(doubleVal);	    
	    updateRenderer();
	}

	/**
	 * Updates the renderer of this listener
	 */
	void updateRenderer() {
	    JComponent r = getValueRenderer();
	    if (r == null) return;
	    ((JLabel)r).setText(getLastValue());
	    fireTableRowsUpdated(row, row);
	}

	/**
	 * useless method (empty)
	 * @param evt unused
	 */
	public void spectrumChange(NumberSpectrumEvent evt) {
	    // we don't to anything on a spectrum change
	}

	/**
	 * useless method (empty)
	 * @param evt unused
	 */
	public void imageChange(NumberImageEvent evt) {
	    // nor on an imagechange
	}

	
	/**
	 * Returns the <code>IAttribute</code> representation of the attribute
	 */
	public IAttribute getAttribute() {
	    return attribute;
	}

	/**
	 * Returns the last value of the attribute
	 * @return the value of the attribute as a <code>String</code>
	 */
	public String getLastValue() {

	    if (format.equals("No format") || format == "") return last;
	    
	    if (format.indexOf('%') == -1) 
		return AttrManip.format(format, doubleVal);
	    Object[] o = { new Double(doubleVal)};
	    return Format.sprintf(format, o);
	}

	/**
	 * Same as <code>getLastValue()</code>
	 */
	public String toString() {
	    return getLastValue();
	}
	// The following methods are just convenience wrappers for the
	// getValueAt of the AttributeAdapter.
	
	public String getLevel() {
	    return ((Property)attribute.getPropertyMap().
		    get("level")).getPresentation();
	}

	public String getDataFormat() {
	    return ((Property)attribute.getPropertyMap().
		    get("data_format")).getPresentation();
	}

	public String getDataType() {
	    return ((Property)attribute.getPropertyMap().
		    get("data_type")).getPresentation();
	}

	public String getWritable() {
	    return ((Property)attribute.
		    getPropertyMap().get("writable")).getPresentation();
	}

	public String getWritableName() {
	    return ((Property)attribute.
		    getPropertyMap().get("writable_attr_name")).
		getPresentation();
	}
    }
}
