// File:          AttributeTable.java
// Created:       2002-09-10 10:04:14, erik
// By:            <erik@assum.net>
// Time-stamp:    <2003-01-16 15:48:31, erik>
// 
// $Id: AttributeTable.java,v 1.2 2005/08/26 13:44:40 ounsy Exp $
// 
// Description:       

package explorer;

import java.awt.Component;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import explorer.ui.PreferencesDialog;
import fr.esrf.tangoatk.core.IAttribute;
import fr.esrf.tangoatk.core.IEntity;
import fr.esrf.tangoatk.core.INumberScalar;
import fr.esrf.tangoatk.core.IScalarAttribute;
import fr.esrf.tangoatk.widget.attribute.Trend;
import fr.esrf.tangoatk.widget.dnd.NodeFactory;
import fr.esrf.tangoatk.widget.dnd.NumberScalarNode;
import fr.esrf.tangoatk.widget.util.HelpWindow;

/**
 * <code>AttributeTable</code> implements the table holding the
 * attributes.
 * @author <a href="mailto:erik@assum.net">Erik Assum</a>
 * @version $Revision: 1.2 $
 */
public class AttributeTable extends EntityTable {
    /** The attribute panel in which the table will be*/
    AttributePanel panel;
    /** Allows in menu to set an attribute*/
    JMenuItem set;
    /** Allows in menu to add an attribute in trend*/
//    JMenuItem trend;
    /** Allows in menu to add an attribute in global trend */
    JMenuItem globalTrendItem;
    /** Allows in menu to remove an attribute from trend*/
    JMenuItem removeTrend;
    /** Allows in menu to refresh attributes*/
    JMenuItem refresh;

    /** The trend*/
    Trend trendPanel;
    /** The global trend*/
    Trend globalTrend;
    
    /** The trend frame*/
    explorer.ui.Dialog trendFrame;

    /** The "set" dialog box for attributes*/
    SetDialog setDialog;

    /**
     * Creates a new <code>AttributeTable</code> instance
     * @param isAdmin a <code>boolean</code> value, is true if
     * the table is used in administrator mode.
     */
    public AttributeTable(AttributeTableModel model, Preferences prefs,
			  boolean isAdmin) {
	preferences = prefs;
	initComponents(model, isAdmin);
	flavor = NodeFactory.MIME_ATTRIBUTE;
	table.setTransferHandler(new fr.esrf.tangoatk.widget.dnd.TransferHandler() {
		protected Transferable createTransferable(JComponent comp) {
		    if (!(comp instanceof JTable)) return null;
			JTable t = (JTable) comp;
			IEntity entity =
			    ((AttributeTableModel)getModel()).
			    getEntityAt(getSelectedRow());
			if (!(entity instanceof INumberScalar)) return null;
			INumberScalar scalar = (INumberScalar)entity;
			return  new NumberScalarNode(scalar);
			}
		});

	table.setDragEnabled(true);
	setModel(model);
    }
    
    /**
     * <code>dtHandleDrop</code> handles the dropping of an object
     * onto this table. The object is an attribute.
     * @param name a <code>String</code> value
     * @param flavor a <code>DataFlavor</code> value
     * @see fr.esrf.tangoatk.widget.dnd
     */
    protected void dtHandleDrop(String name, DataFlavor flavor) {
	String mimeType = flavor.getMimeType();

	if (mimeType.startsWith(NodeFactory.MIME_STRINGSCALAR)) {
	    ((AttributeTableModel)model).addStringScalar(name);
	    return;
	}

	if (mimeType.startsWith(NodeFactory.MIME_NUMBERSCALAR)) {
	    ((AttributeTableModel)model).addNumberScalar(name);
	    return;
	}

	if (mimeType.startsWith(NodeFactory.MIME_NUMBERSPECTRUM)) {
	    ((AttributeTableModel)model).addNumberSpectrum(name);
	    return;
	}

	if (mimeType.startsWith(NodeFactory.MIME_NUMBERIMAGE)) {
	    ((AttributeTableModel)model).addNumberImage(name);
	    return;
	}
    }

    /**
     * Shows the "set" dialog box of the selected attribute in the table
     */
    protected void showSetDialog() {
	IAttribute attribute;
	if (getSelectedRow() == - 1) return;
	attribute = (IAttribute)model.getEntityAt(getSelectedRow());
			
	setDialog = new SetDialog((IScalarAttribute)attribute);
	setDialog.show();
    }

    /**
     * Refreshes informations (like value) about the selected attribute in the table
     */
    protected void refreshEntity() {
	IAttribute attribute;
	attribute = (IAttribute)model.getEntityAt(getSelectedRow());
	attribute.refresh();
	attribute.getDevice().refresh();
    }
    
    /**
     * Shows the trend panel
     */
    protected void showTrend() {
	IAttribute attribute;
	attribute = (IAttribute)model.getEntityAt(getSelectedRow());

	trendPanel = new Trend();
	trendFrame = new explorer.ui.Dialog(trendPanel);
	trendFrame.setTitle(attribute.getAlias());

	trendPanel.setButtonBarVisible(false);

	trendPanel.addAttribute((INumberScalar)attribute);
	trendFrame.show();
    }


    /**
     * Sets the trend to a specified trend
     * @param trend the trend
     */
    public void setGlobalTrend(Trend trend) {
	globalTrend = trend;
    }
    
    /**
     * Adds the selected attribute in the trend
     */
    protected void addGlobalTrend() {
	IAttribute attr;
	attr = (IAttribute)model.getEntityAt(getSelectedRow());
	globalTrend.addAttribute((INumberScalar)attr);
    }

    /**
     * Removes the selected attribute from trend
     */
    protected void removeTrend() {
	IAttribute attr;
	attr = (IAttribute)model.getEntityAt(getSelectedRow());
	
	globalTrend.removeAttribute((INumberScalar)attr);
    }

    /**
     * Initializes the help page
     */
    void initHelp() {
	helpUrl = "/explorer/html/AttributeTableHelp.html";
	
	HelpWindow.getInstance().addCategory("Attribute Table",
					     "Attribute table",
					     getClass().
					     getResource(helpUrl));
    }

    /**
     * Initializes the menu panel (obtained with right click)
     */
    void initMenus() {
	refresh = new JMenuItem("Refresh");
	set = new JMenuItem("Set...");
//	trend = new JMenuItem("Trend...");
	globalTrendItem = new JMenuItem("Add to trend");
	removeTrend = new JMenuItem("Remove from trend");
	

	refresh.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent evt) {
		    refreshEntity();
		}
	    });
	
	set.addActionListener(new ActionListener () {
		public void actionPerformed(ActionEvent evt) {
		    showSetDialog();
		}
	    });

/*	trend.addActionListener(new ActionListener () {
		public void actionPerformed(ActionEvent evt) {
		    showTrend();
		}
	    });*/
	
	globalTrendItem.addActionListener(new ActionListener () {
		public void actionPerformed(ActionEvent evt) {
		    addGlobalTrend();
		}
	    });

	removeTrend.addActionListener(new ActionListener () {
		public void actionPerformed(ActionEvent evt) {
		    removeTrend();
		}
	    });

	popup.add(refresh);
	popup.add(new JSeparator());
	//	popup.add(set);
//	popup.add(trend);
	popup.add(globalTrendItem);
	popup.add(removeTrend);
    }

    /**
     * Initializes the presentation preferences panel
     */
    void initPreferences() {
	PreferencesDialog.getInstance().addCategory("Attribute table",
						    "visible fields",
						    new ViewDialog(model));
    }

    /**
     * Shows the menu panel (obtained with right click)
     * @param evt the mouse event for the "click"
     */
    protected void showPopup(MouseEvent evt) {
	int row = getRowAtPoint(evt.getPoint());

	if (row == -1) return;

	if (evt.isPopupTrigger()) {
	    setRowSelectionInterval(row, row);

	    IAttribute attribute = (IAttribute)model.getEntityAt(row);

	    set.setEnabled(attribute.isWritable());
//	    trend.setEnabled(attribute instanceof INumberScalar);
	    removeTrend.setEnabled(attribute instanceof INumberScalar);
	    globalTrendItem.setEnabled(attribute instanceof INumberScalar);

	    popup.show(evt.getComponent(), evt.getX(), evt.getY());
	} 
    }
    
    /**
     * Manages the consequences of a mouse click on the table
     * @param evt the mouse "click" event
     */
    protected void entityTableMouseClicked ( MouseEvent evt ) 
    {
		int row = getRowAtPoint ( evt.getPoint () );
		if ( row == -1 ) return;
	
		IAttribute attribute = (IAttribute) model.getEntityAt( row );
		    
		if ( 
		     model.isExecuteColumn ( getSelectedColumn () )
		     && attribute.isWritable () 
		   )
		{
		   showSetDialog();
		}  
    }

    TableColumn createTableColumn(String name, int i) {
	final AttributeTableModel myModel = (AttributeTableModel)model;
	if (AttributeTableModel.DEVICE == name) {
	    return new TableColumn(i, 75, deviceRenderer, null);
	}
	if (AttributeTableModel.VALUE == name) {
	    return new TableColumn(i, 75, entityRenderer, null);
	}
	
	if (AttributeTableModel.SET == name) {	
	    return new TableColumn(i, 75, new TableCellRenderer() {
		    JButton renderer;
		    public Component
			getTableCellRendererComponent(JTable table,
						      Object value,
						      boolean select,
						      boolean focus,
						      int row,
						      int column) {
			if (renderer == null) renderer =
						  new JButton(AttributeTableModel.SET);
			
			IAttribute attribute =
			    (IAttribute)model.getEntityAt(row);
			renderer.setEnabled(attribute.isWritable());
			return renderer;
		    }
		} ,null);
	}

	return new TableColumn(i, 75);
    }
    
    /**
     * Sets preferences to a certain preferences value
     * @param preferences the preferences value
     */
    public void setPreferences(Preferences preferences) {
      super.setPreferences(preferences);
      String s=preferences.getString("globalGraphSettings",null);
      if( s!=null ) {
        String err = globalTrend.setSetting(s);
	if( err.length()>0 ) JOptionPane.showMessageDialog(null,"Failed apply trend configuration: " + err,"Error",JOptionPane.ERROR_MESSAGE);
      }
    }

    /**
     * Calls this models storePreferences
     */
    public void storePreferences() {
      super.storePreferences();

      // Save global trend settings
      preferences.putString("globalGraphSettings", globalTrend.getSettings() );
            
    }

}
