// File:          AdminMain.java
// Created:       2002-12-13 12:55:12, erik
// By:            <Erik Assum <erik@assum.net>>
// Time-stamp:    <2003-01-16 15:12:9, erik>
// 
// $Id: AdminMain.java,v 1.3 2005/08/26 13:44:40 ounsy Exp $
// 
// Description:       

package explorer;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeSelectionModel;

import fr.esrf.tangoatk.core.AttributeList;
import fr.esrf.tangoatk.core.CommandList;
import fr.esrf.tangoatk.core.ConnectionException;
import fr.esrf.tangoatk.core.IEntity;
import fr.esrf.tangoatk.core.INumberSpectrum;
import fr.esrf.tangoatk.widget.device.Tree;
import fr.esrf.tangoatk.widget.dnd.AttributeNode;
import fr.esrf.tangoatk.widget.dnd.CommandNode;
import fr.esrf.tangoatk.widget.dnd.EntityNode;

/**
 * Class for the administrator view of the device tree program
 * 
 * @author Erik ASSUM
 */
public class AdminMain extends Main {
    /** The device tree (the one you can see on the left of the screen in admin mode)*/
    JTree deviceTree;
    /** 
     * The multi window manager. It is here to split what's common in admin and user mode with
     * what's specific to admin mode
     */
    JSplitPane mainSplit;
    
    /**
     * Class constructor, initializer. This method is called when you run device tree from shell
     * 
     * @param args arguments given in shell command
     */
    public AdminMain(String [] args) {
	isAdmin = true; // this is a hack! is only used in super.initUI
	initComponents();
	initTree();

	//allowing tree refresh from "edit" menu
	JMenuItem refreshTreeItem = new JMenuItem("Refresh Tree...");
	refreshTreeItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    refreshTree();
		}
	    });
	Container menuContain = menuBar.getParent();
	menuContain.remove(menuBar);
	menuBar.add2EditMenu(refreshTreeItem);
	menuContain.add(menuBar);
	
	mainFrame.pack();

	mainFrame.show();
    }
    
    /**
     * Class constructor, initializer. This method is called through other classes, not from shell.
     */
    public AdminMain() {
    	runningFromShell=false;
	isAdmin = true; // this is a hack! is only used in super.initUI
	initComponents();
	initTree();
	
	//allowing tree refresh from "edit" menu
	JMenuItem refreshTreeItem = new JMenuItem("Refresh Tree...");
	refreshTreeItem.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    refreshTree();
		}
	    });
	Container menuContain = menuBar.getParent();
	menuContain.remove(menuBar);
	menuBar.add2EditMenu(refreshTreeItem);
	menuContain.add(menuBar);
	
	viewSplit.setLastDividerLocation(tableSplitHeight / 3);

	mainFrame.pack();
	viewSplit.setDividerLocation(1.0d);	    
	mainFrame.show();
    }

    /**
     * Sets up the device tree window with specific constraints
     * 
     * @param constraints the specific constraints
     */
    protected void specificSetup(GridBagConstraints constraints) {
	deviceTree = initTree();
	splash.setMessage("Adding tree...");	           
	JScrollPane treePane = new JScrollPane(deviceTree);
	treePane.setPreferredSize(new Dimension(300, 600));
	mainSplit = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
						  treePane, viewSplit);
	mainSplit.setDividerSize(9);
	mainSplit.setOneTouchExpandable(true);
	mainSplit.setDividerLocation(mainSplitDividerLocation);
	mainFrame.getContentPane().add(mainSplit, constraints);
	splash.setMessage("Adding tree..." + done);	           
    }

    /**
     * Prepares a new device tree file creation
     */
    public void newFile() {
	attributeTableModel.clear();
	commandTableModel.clear();
    }

    /**
     * Opens and loads a device tree file
     * 
     * @param file the device tree file to open
     */
    public void open(File file) {
	open(file.getAbsolutePath());
	mainSplit.setDividerLocation(mainSplitDividerLocation);
    }
    
    /**
     * Saves a device tree file
     * 
     * @param file the device tree file to save
     */
    public void save(File file) {
	String name = file.getAbsolutePath();
	mainFrame.setTitle(name);
	progress.setIndeterminate(true);
	status.setText("Saving to " + file.getAbsolutePath());
	
	try {
	    storePreferences();
	    fileManager.setAttributes((AttributeList)attributeTableModel.getList());
	    fileManager.setCommands((CommandList)commandTableModel.getList());
	    
	    fileManager.save(file);
	    status.setText("Saving ok");
	} catch (Exception e) {
	    status(mainFrame, "Could not save " + file, e);
 	}
	progress.setValue(progress.getMinimum());	
	progress.setIndeterminate(false);
    }

    /**
     * Initialization of the device tree (the tree on the left of the screen in admin mode)
     */
    protected JTree initTree() {
	String message = "Initializing device tree...";
	splash.setMessage(message);
	Tree tree = (Tree)treeInitialization();
	message += "done";
	splash.setMessage(message);
	return tree;
    }
    
    protected JTree treeInitialization(){
    	Tree tree = new Tree();
    	tree.addErrorListener(errorHistory);
    	tree.importFromDb();
    	treePopup = new JPopupMenu();


    	tree.addStatusListener(this);
    	tree.setShowEntities(true);

    	entityPopup = new JPopupMenu();
    	JMenuItem refreshItem = new JMenuItem("Refresh Tree...");
    	JMenuItem addItem = new JMenuItem("Add to table");

    	addItem.addActionListener(new ActionListener() {
    		public void actionPerformed(ActionEvent e) {
    		    add2Table();
    		}
    	    });

    	refreshItem.addActionListener(new ActionListener() {
    		public void actionPerformed(ActionEvent e) {
    		    refreshTree();
    		}
    	    });
    				      
    	treePopup.add(refreshItem);
    	entityPopup.add(addItem);

    	tree.getSelectionModel().
             setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

    	tree.addMouseListener(new MouseAdapter() {
    		public void mousePressed(MouseEvent evt) {
    		    treeClicked(evt);
    		}

    		public void mouseReleased(MouseEvent evt) {
    		    treeClicked(evt);
    		}
    	    });

    	tree.setDragEnabled(true);
    	return tree;
    }

    /**
     * Refreshes the device tree (the tree on the left of the screen in admin mode)
     */
    public void refreshTree() {
	//((Tree)deviceTree).refresh();
        status.setText("Refreshing Tree...");
        Container c = deviceTree.getParent();
        c.remove(deviceTree);
        ((Tree)deviceTree).removeListeners();
        deviceTree = null;
        deviceTree = treeInitialization();
        c.add(deviceTree);
        status.setText("Tree Refreshed");
    }

    /**
     * Manages a mouse click on the device tree (the tree on the left of the screen in admin mode).
     * This method is used to expand/collapse the tree branch over which the user clicked.
     * 
     * @param evt the mouse event for the "click"
     */
    public void treeClicked(MouseEvent evt) {
	int selectedRow =
	    deviceTree.getRowForLocation(evt.getX(), evt.getY());

	if (selectedRow != -1 && evt.isPopupTrigger()) {
	    Object n = deviceTree.getLastSelectedPathComponent();

	    if (n == null && !(n instanceof DefaultMutableTreeNode)) return;

	    Object node = ((DefaultMutableTreeNode)n).getUserObject();

	    if (node instanceof EntityNode) {
		entityPopup.show(evt.getComponent(), evt.getX(), evt.getY());
		return;
	    }

	    if (selectedRow == 0) {
		treePopup.show(evt.getComponent(), evt.getX(), evt.getY());
	    }
	}
    }

    /**
     * Method used to add an attribute/a command, in the tree, in the list of attributes/commands
     * to check/manage
     */
    public void add2Table() {
	Object n = deviceTree.getLastSelectedPathComponent();
	if (n == null && !(n instanceof DefaultMutableTreeNode)) return;
	Object node = ((DefaultMutableTreeNode)n).getUserObject();

	if (node instanceof AttributeNode) {
	    try {
		attributeTableModel.load(((AttributeNode)node).getFQName()); 
	    } catch (ConnectionException e) {
		Main.status(mainFrame, "Error loading attribute ", e);
	    }
	    return;
	}

	if (node instanceof CommandNode) {
	    try {
		commandTableModel.load(((CommandNode)node).getFQName());
	    } catch (ConnectionException e) {
		Main.status(mainFrame, "Error loading command ", e);
	    }
	    
	    return;
	}
    }

    /**
     * Stores the device tree window appearence preferences
     */
    public void storePreferences() {
    
	attributeTable.storePreferences();
	commandTable.storePreferences();
	preferences.putInt(WINDOW_WIDTH_KEY, mainFrame.getWidth());
	preferences.putInt(WINDOW_HEIGHT_KEY, mainFrame.getHeight());
	preferences.putInt(WINDOW_X_KEY, mainFrame.getX());
	preferences.putInt(WINDOW_Y_KEY, mainFrame.getY());
	preferences.putInt(TABLE_SPLIT_DIVIDER_LOCATION_KEY,
			   tableSplit.getDividerLocation());
	preferences.putInt(MAIN_SPLIT_DIVIDER_LOCATION_KEY,
			   mainSplit.getDividerLocation());
	preferences.putDouble(VIEW_SPLIT_DIVIDER_LOCATION_KEY,
			   viewSplit.getDividerLocation());
	preferences.putInt(TABLE_SPLIT_WIDTH_KEY,
			   tableSplit.getWidth());
	preferences.putInt(TABLE_SPLIT_HEIGHT_KEY,
			   tableSplit.getHeight());

	preferences.putBoolean(GLOBAL_TREND_VISIBLE_KEY,
			       viewSplit.getDividerLocation() == 1.0);
	fileManager.setPreferences(preferences);
	
        // Save trend settings for all specturm attribute
	EntityTableModel model = (EntityTableModel)attributeTable.getModel();	
        for(int i=0;i<model.getRowCount();i++) {
          IEntity att = model.getEntityAt(i);
	  if( att instanceof INumberSpectrum ) {
	    String keyname = att.getName() + ".GraphSettings";
            preferences.putString(keyname, attributePanel.getSpectrumGraphSettings(att) );
	  }
        }

    }

    /**
     * Exits program, stores preferences, saves the current device tree file and terminates 
     * the currently running Java Virtual Machine.
     */
    public void quit() {
	storePreferences();
	if ( (file != null) && (fileRecordable) ) {
	    save(file);
	}
	System.exit(0);
    }

    /**
     * Exits program, stores preferences and saves the current device tree file
     */
    public void exit() {
	storePreferences();
	if ( (file != null) && (fileRecordable) ) {
	    save(file);
	}
    }
}
