/*
 *  
 *   File          :   MainPanel.java
 *  
 *   Project       :   atkpanel generic java application
 *  
 *   Description   :   The main frame of atkpanel
 *  
 *   Author        :   Faranguiss Poncet
 *  
 *   Original      :   Mars 2002
 *  
 *   $Revision: 1.16 $				$Author: poncet $
 *   $Date: 2005/05/03 17:02:02 $					$State: Exp $
 *  
 *   $Log: MainPanel.java,v $
 *   Revision 1.16  2005/05/03 17:02:02  poncet
 *   Added an new attribute list for state and status attributes without events.
 *
 *   Revision 1.15  2005/04/15 12:16:28  poncet
 *   Added the BooleanScalarAttribute, stateAttributeViewer and statusAttributeViewer.
 *
 *   Revision 1.14  2004/11/24 14:11:55  poncet
 *   Added new constructor (Boolean arguments instead of boolean) to be used in synoptics.
 *
 *   Revision 1.13  2004/11/23 09:37:12  poncet
 *   Added ReadOnly mode for MainPanel which supresses all commands. Removed the
 *   TabbedPanel in the MainPanel when the device has no attribute at all. This
 *   Allows to have a smaller window for gauges for example.
 *
 *   Revision 1.12  2004/11/22 13:25:58  poncet
 *   Added two new flags for the constructor of atkpanel.MainPanel. The first is
 *   used to indicate if the property button is visible in the scalar attribute
 *   tab. The second one is used to start the atkpanel in read-only mode.
 *
 *   Revision 1.11  2004/10/12 12:49:51  poncet
 *   Used AttributePolledList for Trend.setModel method to be compatible ATK 1.9.0. and higher.
 *
 *   Revision 1.10  2003/12/16 17:56:56  poncet
 *   Added the handling of StringSpectrum Attributes in atkpanel.
 *
 *   Revision 1.9  2003/09/25 15:10:50  poncet
 *   Fixed a bug in the handling of keepStateRefresher flag. Stop state refresher
 *   when this flag is set to false and the menu bar command preferences->Stop
 *   refreshing is called.
 *
 *   Revision 1.8  2003/09/19 07:59:19  poncet
 *   Added the operator and expert modes handling. Scalars now displayed by
 *   ATK ScalarListViewer. The Image attributes now displayed by the new
 *   ATK viewer NumberImageViewer.
 *
 *   Revision 1.7  2003/08/05 09:32:14  poncet
 *   Added the flag keepStateRefresher in order to specify if we want to stop or to
 *   keep the device state and status referesher thread when atkpanel is stopped.
 *
 *   Revision 1.6  2003/03/20 13:28:14  poncet
 *   In non standalone mode the atkpanel mustn't do an system.exit(-1) when
 *   the device is not accessible. This feature has been added to atkpanel.
 *
 *   Revision 1.5  2003/02/03 15:48:10  poncet
 *   Fixed a bug in SpectrumPanel.java related to JTableAdapter.
 *
 *   Revision 1.4  2003/02/03 15:32:09  poncet
 *   Committed to have a coherent and identical revision numbers to
 *   avoid using tags.
 *
 *   Revision 1.2  2003/01/28 16:57:41  poncet
 *   Added bin and doc directory
 *
 *   Revision 1.1.1.1  2003/01/28 16:35:55  poncet
 *   Initial import
 *
 *  
 *   Copyright (c) 2002 by European Synchrotron Radiation Facility,
 *  		       Grenoble, France
 *  
 *                         All Rights Reserved
 *  
 *  
 */
 
package atkpanel;


/**
 *
 * @author  poncet
 */
import java.util.*;
import java.lang.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.Component;
import java.awt.event.*;
import fr.esrf.tangoatk.core.*;
import fr.esrf.tangoatk.widget.util.Splash;
import fr.esrf.tangoatk.widget.util.ErrorHistory;
import fr.esrf.tangoatk.widget.util.ErrorPopup;
import fr.esrf.tangoatk.widget.attribute.*;

public class MainPanel extends javax.swing.JFrame {

    private  final Splash                           splash = new Splash();
    private  final int                              MIN_WIDTH = 220;
    private  final int                              MAX_WIDTH = 570;
    private  final int                              MIN_HEIGHT = 220;
    private  final int                              MAX_HEIGHT = 570;
    
    
    private  boolean                                standAlone = false;
    private  boolean                                keepStateRefresher = true;
    private  boolean                                operatorView = true;
    private  boolean                                modifPropButton = true; // property button visible
    private  boolean                                roMode = false; // read only mode
    
    private  fr.esrf.tangoatk.core.AttributeList    all_scalar_atts; /* used in the scalar tab */
    private  fr.esrf.tangoatk.core.AttributeList    op_scalar_atts;  /* used for operator attributes */
    private  fr.esrf.tangoatk.core.AttributeList    exp_scalar_atts;  /* used for expert attributes */
    private  fr.esrf.tangoatk.core.AttributeList    state_status_atts; /* used for state and status attribute viewers */
    private  fr.esrf.tangoatk.core.AttributePolledList    number_scalar_atts; /* used in the global trend */

    private  fr.esrf.tangoatk.core.AttributeList    all_spectrum_atts; /* used in spectrum tabs */
    private  fr.esrf.tangoatk.core.AttributeList    op_spectrum_atts; /* used in spectrum tabs */
    private  List                                   all_spectrum_panels=null;

    private  fr.esrf.tangoatk.core.AttributeList    all_number_image_atts; /* used in number image tabs */
    private  fr.esrf.tangoatk.core.AttributeList    op_number_image_atts; /* used in number image tabs */
    private  List                                   all_image_panels=null;

    private  CommandList                            all_cmdl;
    private  CommandList                            op_cmdl;
    
    
    private  Device                                 panelDev;
    private  ErrorHistory                           errorHistory;
    private  ErrorPopup                             errorPopup;
    private  JFrame                                 trendFrame;
    private  ScalarListViewer                       allScalarListViewer;
    private  ScalarListViewer                       operatorScalarListViewer;
    private  IDevStateScalar                        stateAtt=null;
    private  IStringScalar                          statusAtt=null;


    /** Creates new form AtkPanel */
    public MainPanel(String  devName)
    {
	if (connectDevice(devName) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
	initComponents();
	startUp();
    }
    
    
    public MainPanel(String  args[])
    {
        this(args[0]);
    }
    
    public MainPanel(String  dev, boolean stda)
    {
        this.standAlone = stda;
	if (connectDevice(dev) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
        initComponents();
	startUp();
    }

    public MainPanel(String  args[], boolean stda)
    {       
        this.standAlone = stda;
	if (connectDevice(args[0]) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
        initComponents();
	startUp();
    }

    public MainPanel(String  dev, boolean stda, boolean keepStateRef)
    {       
        this.standAlone = stda;
	this.keepStateRefresher = keepStateRef;
	if (connectDevice(dev) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
        initComponents();
	startUp();
    }

    public MainPanel(String  dev, boolean stda, boolean keepStateRef, boolean modifProp)
    {       
        this.standAlone = stda;
	this.keepStateRefresher = keepStateRef;
	this.modifPropButton = modifProp;
	if (connectDevice(dev) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
        initComponents();
	startUp();
    }

    public MainPanel(String  dev, boolean stda, boolean keepStateRef,
                                  boolean modifProp, boolean ro)
    {       
        this.standAlone = stda;
	this.keepStateRefresher = keepStateRef;
	this.modifPropButton = modifProp;
	this.roMode = ro;
	if (ro == true)
	   this.modifPropButton = false;
	if (connectDevice(dev) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
        initComponents();
	startUp();
    }

    // This constructor is used in TangoSynopticHandler class of tangoatk
    public MainPanel(String  dev, Boolean stda, Boolean keepStateRef,
                                  Boolean modifProp, Boolean ro)
    {       
        this.standAlone = stda.booleanValue();
	this.keepStateRefresher = keepStateRef.booleanValue();
	this.modifPropButton = modifProp.booleanValue();
	this.roMode = ro.booleanValue();
	if (ro.booleanValue() == true)
	   this.modifPropButton = false;
	if (connectDevice(dev) == false)
	{
	   splash.setVisible(false);
	   return;		
        }
        initComponents();
	startUp();
    }
    
    
   /** Creates new form AtkPanel */
    private boolean connectDevice(String  devName)
    {
	
	splash.setTitle("AtkPanel  1.16");
	splash.setMessage("Waiting for device-name...");
	splash.initProgress();
        splash.setMaxProgress(12);
	
	if (devName == null)
	{
	    devName = javax.swing.JOptionPane.showInputDialog
		(null, "Please type device name");
	} 
		
	// Keeps all scalar attributes operator or expert
        all_scalar_atts = new fr.esrf.tangoatk.core.AttributeList();
        all_scalar_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (    (entity instanceof IScalarAttribute)
			            || (entity instanceof IBooleanScalar) )
			       {
                                  return true;
                               }
                               return false;
                            }
                         });
	
	
	// Keeps only operator scalar attributes
        op_scalar_atts = new fr.esrf.tangoatk.core.AttributeList();
        op_scalar_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (    (entity instanceof IScalarAttribute)
			            || (entity instanceof IBooleanScalar) )
			       {
				  if (entity.isOperator() == true)
				  {
				     String message = "Adding Operator scalar attributes(";
				     splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                     return true;
				  }
                               }
                               return false;
                            }
                         });
	
	
	// Keeps only expert scalar attributes
        exp_scalar_atts = new fr.esrf.tangoatk.core.AttributeList();
        exp_scalar_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (    (entity instanceof IScalarAttribute)
			            || (entity instanceof IBooleanScalar) )
			       {
				  if (entity.isOperator() == false)
				  {
				     String message = "Adding Expert scalar attributes(";
				     splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                     return true;
				  }
                               }
                               return false;
                            }
                         });
	
        state_status_atts = new fr.esrf.tangoatk.core.AttributeList();
	
	// Keep only number scalar attributes operator or expert
        number_scalar_atts = new fr.esrf.tangoatk.core.AttributePolledList();
        number_scalar_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (entity instanceof INumberScalar)
			       {
				  String message = "Adding scalar attributes(";
				  splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                  return true;
                               }
                               return false;
                            }
                         });

	
	// Keep all number spectrum attributes operator or expert
        all_spectrum_atts = new fr.esrf.tangoatk.core.AttributeList();
        all_spectrum_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (entity instanceof INumberSpectrum)
			       {
                        	  if (entity instanceof INumberScalar)
				  {
                                     return false;
                        	  }
				  else
				  {
				     String message = "Adding to all_spectrum_atts : ";
				     splash.setMessage(message + entity.getNameSansDevice() );
                                     return true;
				  }
                               }
			       else
			       {
			          if (entity instanceof IStringSpectrum)
				  {
				     String message = "Adding to all_spectrum_atts : ";
				     splash.setMessage(message + entity.getNameSansDevice() );
                                     return true;
				  }
			       }
                               return false;
                            }
                         });

	
	// Keep only number operator spectrum attributes
        op_spectrum_atts = new fr.esrf.tangoatk.core.AttributeList();
        op_spectrum_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (entity instanceof INumberSpectrum)
			       {
                        	  if (entity instanceof INumberScalar)
				  {
                                     return false;
                        	  }
				  else
				  {
				     if (entity.isOperator())
				     {
				        String message = "Adding to op_spectrum_atts : ";
					splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                	return true;
				     }
				  }
                               }
			       else
			       {
			          if (entity instanceof IStringSpectrum)
				  {
				     if (entity.isOperator())
				     {
					String message = "Adding to op_spectrum_atts : ";
					splash.setMessage(message + entity.getNameSansDevice() );
                                	return true;
				     }
				  }
			       }
                               return false;
                            }
                         });
                         
	
	
        all_number_image_atts = new fr.esrf.tangoatk.core.AttributeList();
	// Keep only number image attributes
        all_number_image_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (entity instanceof INumberImage)
			       {
                        	  if (entity instanceof INumberSpectrum)
				  {
                                     return false;
                        	  }
				  else
				  {
				     String message = "Adding image attributes(";
				     splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                     return true;
				  }
                               }
                               return false;
                            }
                         });
	
	
        op_number_image_atts = new fr.esrf.tangoatk.core.AttributeList();
	// Keep only number image attributes
        op_number_image_atts.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (entity instanceof INumberImage)
			       {
                        	  if (entity instanceof INumberSpectrum)
				  {
                                     return false;
                        	  }
				  else
				  {
				     if (entity.isOperator())
				     {
					String message = "Adding image attributes(";
					splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                	return true;
				     }
				  }
                               }
                               return false;
                            }
                         });
                         

        all_cmdl = new CommandList();

        op_cmdl = new CommandList();
	// Keep only operator commands
        op_cmdl.setFilter( new IEntityFilter () 
                         {
                            public boolean keep(IEntity entity)
			    {
                               if (entity instanceof ICommand)
			       {
				  if (entity.isOperator())
				  {
				     String message = "Adding Operator command (";
				     splash.setMessage(message + entity.getNameSansDevice() + ")...");
                                     return true;
				  }
                               }
                               return false;
                            }
                         });


        panelDev = null;
	
        errorHistory = new fr.esrf.tangoatk.widget.util.ErrorHistory();
	errorPopup = new fr.esrf.tangoatk.widget.util.ErrorPopup();
        all_scalar_atts.addErrorListener(errorHistory);
        all_scalar_atts.addSetErrorListener(errorPopup);
        all_scalar_atts.addSetErrorListener(errorHistory);
	
        state_status_atts.addErrorListener(errorHistory);
        

        all_cmdl.addErrorListener(errorHistory);
	all_cmdl.addErrorListener(errorPopup);
	
        try
        {
	   String message = "Getting device " + devName + "...";
	   splash.setMessage(message);
	   panelDev = DeviceFactory.getInstance().getDevice(devName);
	   splash.progress(1);
	   splash.setMessage(message + "done");
        }
        catch (Exception   e)
        {
	   javax.swing.JOptionPane.showMessageDialog(
	      null, "Cannot connect to the device.\n"
	           + "Check the device name you entered;"
		   + " Application will abort ...\n"
		   + "Connection Exception : " + e,
		   "Connection to device failed",
		   javax.swing.JOptionPane.ERROR_MESSAGE);
		   
           System.out.println("AtkPanel: Cannot connect to the device.");
           System.out.println("AtkPanel: Check the device name you entered;");
           System.out.println("AtkPanel: Application aborted....");
           System.out.println("AtkPanel: Connection Exception : " + e);
           //System.exit(-1);
	   abortAppli();
	   return false;
        }
	

	if (panelDev == null)
	{
	   javax.swing.JOptionPane.showMessageDialog(
	      null, "Cannot initialize the device object.\n"
	           + "Check the device name you entered;"
		   + " Application will abort ...\n",
		   "Device init failed",
		   javax.swing.JOptionPane.ERROR_MESSAGE);
		   
           System.out.println("AtkPanel: cannot initialize the device object.");
           System.out.println("AtkPanel: Check the device name you entered;");
           System.out.println("AtkPanel: Application aborted....");
           //System.exit(-1);
	   abortAppli();
	   return false;
	}
	
	
        try
        {
	   String message = "Importing attributes from " + devName + "...";
           try
           {
	      splash.setMessage(message);
	      op_scalar_atts.add(devName+"/*");
	      exp_scalar_atts.add(devName+"/*");
              number_scalar_atts.add(devName+"/*");
	      splash.progress(4);
              all_spectrum_atts.add(devName+"/*");
              op_spectrum_atts.add(devName+"/*");
	      splash.progress(5);
	      all_number_image_atts.add(devName+"/*");
	      op_number_image_atts.add(devName+"/*");
	      splash.progress(6);
	      splash.setMessage(message + "done");
           }
           catch (Exception   e)
           {
	      javax.swing.JOptionPane.showMessageDialog(
		 null, "Cannot build the attribute list.\n"
		      + "Application will abort ...\n"
		      + "Exception received : " + e,
		      "No Attribute Exception",
		      javax.swing.JOptionPane.ERROR_MESSAGE);

              System.out.println("AtkPanel: Cannot build attribute list.");
              System.out.println("AtkPanel: Application aborted....");
              System.out.println("AtkPanel: Connection Exception : " + e);
              //System.exit(-1);
	      abortAppli();
	      return false;
           }
	   message = "Importing commands from " + devName + "...";
	   all_cmdl.add(devName+"/*");
	   op_cmdl.add(devName+"/*");
	   splash.progress(7);
    	   splash.setMessage(message + "done");
        }
        catch (Exception   e)
        {
	   javax.swing.JOptionPane.showMessageDialog(
	      null, "Cannot build the command list.\n"
		   + "Application will abort ...\n"
		   + "Exception received : " + e,
		   "No Command Exception",
		   javax.swing.JOptionPane.ERROR_MESSAGE);

           System.out.println("AtkPanel: Cannot build the command list.");
           System.out.println("AtkPanel: Application aborted....");
           System.out.println("AtkPanel: Connection Exception : " + e);
           //System.exit(-1);
	   abortAppli();
	   return false;
        }
	
	this.setTitle("AtkPanel : "+devName);
	return true;
    }
    

    private void startUp()
    {	
	
	String message = "Initializing commands...";
	splash.setMessage(message);
		
	createAllCmdList();
	if (all_cmdl.getSize() > 0)
	{
	   if (operatorView == true) // display only operator level commands
              show_operator_commands();
	   else // display all commands (operator level + expert level)
              show_all_commands();
	}
	  
	if (roMode == true)
	{
	   commandComboViewer1.setVisible(false);
	}
	
	splash.progress(8);
	splash.setMessage(message + "done");
	message = "Initializing scalar attributes...";
	splash.setMessage(message);

	// Set scrolling 
       
        allScalarListViewer = null;
        operatorScalarListViewer = null;
	createAllScalarListViewers();
	
	if ( (all_scalar_atts.getSize() > 0) && (allScalarListViewer != null)
	                                      && (operatorScalarListViewer != null) )
	{
	   if (operatorView == true) // display only operator level scalar attributes
	   {
              show_operator_scalars();
	      expertCheckBoxMenuItem.setState(false);
	      operatorCheckBoxMenuItem.setState(true);
	   }
	   else // display all scalar attributes (operator level + expert level)
	   {
              show_all_scalars();
	      expertCheckBoxMenuItem.setState(true);
	      operatorCheckBoxMenuItem.setState(false);	
	   }
	}
	
	
	// Adding state and status attributes
        try
        {
	   stateAtt = (IDevStateScalar) state_status_atts.add(panelDev.getName()+"/State");
	   statusAtt = (IStringScalar) state_status_atts.add(panelDev.getName()+"/Status");
        }
        catch (Exception   e)
        {
           stateAtt=null;;
	   statusAtt=null;
        }
	
	if ((stateAtt == null) || (statusAtt == null))
	{
	   System.out.println("AtkPanel: Cannot get the State and / or Status attributes for the device.");
           System.out.println("AtkPanel: May be an old IDL 2 device server....");
	   devStateViewer.setVisible(true);
	   devStatusViewer.setVisible(true);
	   attStateViewer.setVisible(false);
	   attStatusViewer.setVisible(false);
	}
	else
	{
	   devStateViewer.setVisible(false);
	   devStatusViewer.setVisible(false);
	   
	   attStateViewer.setModel(stateAtt);
	   attStatusViewer.setModel(statusAtt);
	   attStateViewer.setVisible(true);
	   attStatusViewer.setVisible(true);
	   
	   //if no notify daemon (no events) then start refresher for state status atts
	   if (stateAtt.hasEvents() == false)
	   {
	      state_status_atts.startRefresher();
	   }
	}
	
	
	splash.progress(10);
	splash.setMessage(message + "done");
	message = "Initializing number spectrum attributes...";
	splash.setMessage(message);
        
		
        createAllSpectrumTabs();
	if (all_spectrum_atts.getSize() > 0)
	{
	   if (operatorView == true) // display only operator level spectrum attributes 
              show_operator_spectrums();
	   else // display all spectrum attributes (operator level + expert level)
              show_all_spectrums();
	}

	splash.progress(11);
	splash.setMessage(message + "done");
	message = "Initializing number image attributes...";
	splash.setMessage(message);
        
		
        createAllNumberImageTabs();
	if (all_number_image_atts.getSize() > 0)
	{
	   if (operatorView == true) // display only operator level image attributes 
              show_operator_images();
	   else // display all image attributes (operator level + expert level)
              show_all_images();
	}
	
	if (    (all_scalar_atts.getSize() <= 0) && (all_spectrum_atts.getSize() <= 0)
	     && (all_number_image_atts.getSize() <= 0) )
	{ // No attribute at all for this device
	   jTabbedPane1.setVisible(false);
	}

	   
	splash.progress(12);
	splash.setMessage(message + "done");
	splash.setMessage(message);
	
	
	splash.setVisible(false);
	
   	pack();
        setVisible(true);
	
	all_scalar_atts.startRefresher();
	all_spectrum_atts.startRefresher();
	all_number_image_atts.startRefresher();
	
	
        Trend   globalTrend = new Trend(trendFrame);
        globalTrend.setModel(number_scalar_atts);
	

        /* Put the globalTrend in a frame with a minimum height and width */
        trendFrame = null;
        trendFrame = new JFrame();

	javax.swing.JPanel jp= new javax.swing.JPanel();
        trendFrame.getContentPane().add(jp, java.awt.BorderLayout.CENTER);
	jp.setPreferredSize(new java.awt.Dimension(600, 300));
	jp.setLayout(new java.awt.GridBagLayout());
	
        java.awt.GridBagConstraints trendGbc;
        trendGbc = new java.awt.GridBagConstraints();
        trendGbc.gridx = 0;
        trendGbc.gridy = 0;
        trendGbc.fill = java.awt.GridBagConstraints.BOTH;
        trendGbc.weightx = 1.0;
        trendGbc.weighty = 1.0;
	
	jp.add(globalTrend, trendGbc);
	trendFrame.pack();
	//DeviceFactory.getInstance().setTraceMode(DeviceFactory.TRACE_REFRESHER);
 }

        
    
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
            jMenuBar1 = new javax.swing.JMenuBar();
            jMenu1 = new javax.swing.JMenu();
            jMenuItem1 = new javax.swing.JMenuItem();
	    
            jMenu3 = new javax.swing.JMenu();
            jMenuItem5 = new javax.swing.JMenuItem();
            jMenuItem3 = new javax.swing.JMenuItem();
            diagJMenuItem = new javax.swing.JMenuItem();
	    
            jMenu2 = new javax.swing.JMenu();
            jMenuItem6 = new javax.swing.JMenuItem();
            jMenuItem7 = new javax.swing.JMenuItem();
            jMenuItem2 = new javax.swing.JMenuItem();
            expertCheckBoxMenuItem = new javax.swing.JCheckBoxMenuItem();
            operatorCheckBoxMenuItem = new javax.swing.JCheckBoxMenuItem();
	    
            jMenu4 = new javax.swing.JMenu();
            jMenuItem4 = new javax.swing.JMenuItem();
	    
            jPanel1 = new javax.swing.JPanel();
            jPanel2 = new javax.swing.JPanel();
            commandComboViewer1 = new fr.esrf.tangoatk.widget.command.CommandComboViewer();
            devStateViewer = new fr.esrf.tangoatk.widget.device.StateViewer();
            devStatusViewer = new fr.esrf.tangoatk.widget.device.StatusViewer();
            attStateViewer = new fr.esrf.tangoatk.widget.attribute.StateViewer();
            attStatusViewer = new fr.esrf.tangoatk.widget.attribute.StatusViewer();
            jTabbedPane1 = new javax.swing.JTabbedPane();
            jScrollPane1 = new javax.swing.JScrollPane();
            jPanel3 = new javax.swing.JPanel();

            devStateViewer.setStateClickable(false);
            devStatusViewer.setPreferredSize(new java.awt.Dimension(50, 100));
            attStatusViewer.setPreferredSize(new java.awt.Dimension(50, 100));
 
            
            jMenu1.setText("File");
            jMenuItem1.setText("Quit");
            jMenuItem1.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    jMenuItem1ActionPerformed(evt);
                }
            });
            
            jMenu1.add(jMenuItem1);
            jMenuBar1.add(jMenu1);
          jMenu3.setText("View");
            jMenuItem5.setText("Trends ");
            jMenuItem5.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    viewTrendActionPerformed(evt);
                }
            });
            
            jMenu3.add(jMenuItem5);
            jMenuItem3.setText("Error History");
            jMenuItem3.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    viewErrHistoryActionPerformed(evt);
                }
            });
            
            jMenu3.add(jMenuItem3);
            diagJMenuItem.setText("Diagnostic ...");
            diagJMenuItem.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    fr.esrf.tangoatk.widget.util.ATKDiagnostic.showDiagnostic();
                }
            });
            jMenu3.add(diagJMenuItem);
	    
	    
            jMenuBar1.add(jMenu3);
          //jMenu2.setText("Options");
          jMenu2.setText("Preferences");
            jMenuItem6.setText("Stop   Refreshing");
            jMenuItem6.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    stopStartRefreshActionPerformed(evt);
                }
            });
            
            jMenu2.add(jMenuItem6);
            jMenuItem7.setText("Refresh  once");
            jMenuItem7.setEnabled(false);
            jMenuItem7.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    refreshOnceActionPerformed(evt);
                }
            });
            
            jMenu2.add(jMenuItem7);
            jMenuItem2.setText("Set refreshing period ...");
            jMenuItem2.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    refPeriodActionPerformed(evt);
                }
            });
            
            jMenu2.add(jMenuItem2);
	    
	    javax.swing.JSeparator  jsep = new javax.swing.JSeparator();
	    jMenu2.add(jsep);
	    
            operatorCheckBoxMenuItem.setSelected(true);
            operatorCheckBoxMenuItem.setText("Operator  View");
            operatorCheckBoxMenuItem.addActionListener(new java.awt.event.ActionListener() {
        	public void actionPerformed(java.awt.event.ActionEvent evt) {
                    operatorCheckBoxMenuItemActionPerformed(evt);
        	}
            });
            jMenu2.add(operatorCheckBoxMenuItem);
	    
            expertCheckBoxMenuItem.setSelected(false);
            expertCheckBoxMenuItem.setText("Expert  View");
            expertCheckBoxMenuItem.addActionListener(new java.awt.event.ActionListener() {
        	public void actionPerformed(java.awt.event.ActionEvent evt) {
                    expertCheckBoxMenuItemActionPerformed(evt);
        	}
            });
            jMenu2.add(expertCheckBoxMenuItem);
	    
	    
	    
            jMenuBar1.add(jMenu2);
          jMenu4.setText("Help");
            jMenuItem4.setText("On Version ...");
            jMenuItem4.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    helpVersionActionPerformed(evt);
                }
            });
            
            jMenu4.add(jMenuItem4);
            jMenuBar1.add(jMenu4);
          
            addWindowListener(new java.awt.event.WindowAdapter() {
                public void windowClosing(java.awt.event.WindowEvent evt) {
                    exitForm(evt);
                }
            });
            
            jPanel1.setLayout(new java.awt.GridBagLayout());
            java.awt.GridBagConstraints gridBagConstraints1;
            
            jPanel1.setMinimumSize(new java.awt.Dimension(22, 22));
            jPanel2.setLayout(new java.awt.GridBagLayout());
	    
	    JPanel    dummyPanel1 = new JPanel();
	    JPanel    dummyPanel2 = new JPanel();
            java.awt.GridBagConstraints gridBagConstraints2;
             
            gridBagConstraints2 = new java.awt.GridBagConstraints();
           
            gridBagConstraints2.gridx = 1;
            gridBagConstraints2.gridy = 0;
            gridBagConstraints2.insets = new java.awt.Insets(5, 5, 5, 5);
            jPanel2.add(attStateViewer, gridBagConstraints2);
	    devStateViewer.setModel(panelDev);
            jPanel2.add(devStateViewer, gridBagConstraints2);
           
            gridBagConstraints2.gridx = 2;
            gridBagConstraints2.gridy = 0;
            gridBagConstraints2.insets = new java.awt.Insets(5, 5, 5, 5);
            jPanel2.add(commandComboViewer1, gridBagConstraints2);
           
            gridBagConstraints2.gridx = 0;
            gridBagConstraints2.gridy = 0;
            gridBagConstraints2.insets = new java.awt.Insets(5, 5, 5, 5);
            gridBagConstraints2.fill = java.awt.GridBagConstraints.HORIZONTAL;
	    gridBagConstraints2.weightx = 0.5;
	    gridBagConstraints2.weighty = 0.0;
           jPanel2.add(dummyPanel1, gridBagConstraints2);
           
            gridBagConstraints2.gridx = 3;
            gridBagConstraints2.gridy = 0;
            gridBagConstraints2.insets = new java.awt.Insets(5, 5, 5, 5);
            gridBagConstraints2.fill = java.awt.GridBagConstraints.HORIZONTAL;
	    gridBagConstraints2.weightx = 0.5;
	    gridBagConstraints2.weighty = 0.0;
            jPanel2.add(dummyPanel2, gridBagConstraints2);
           
            gridBagConstraints2.gridx = 0;
            gridBagConstraints2.gridy = 1;
            gridBagConstraints2.insets = new java.awt.Insets(5, 5, 5, 5);
   gridBagConstraints2.weightx = 1.0;
   gridBagConstraints2.weighty = 1.0;
            gridBagConstraints2.fill = java.awt.GridBagConstraints.BOTH;
            gridBagConstraints2.gridwidth = 4;
            jPanel2.add(attStatusViewer, gridBagConstraints2);
            devStatusViewer.setModel(panelDev);
            jPanel2.add(devStatusViewer, gridBagConstraints2);



            
            gridBagConstraints1 = new java.awt.GridBagConstraints();
	    gridBagConstraints1.gridx = 0;
	    gridBagConstraints1.gridy = 0;
	    gridBagConstraints1.weightx = 0.01;
	    gridBagConstraints1.weighty = 0.01;
	    gridBagConstraints1.fill = java.awt.GridBagConstraints.BOTH;
	    gridBagConstraints1.insets = new java.awt.Insets(5, 5, 5, 5);
	    jPanel1.add(jPanel2, gridBagConstraints1);

	    jTabbedPane1.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);
              jPanel3.setLayout(new java.awt.GridBagLayout());
              java.awt.GridBagConstraints gridBagConstraints3;
              
              jScrollPane1.setViewportView(jPanel3);
              
              jTabbedPane1.addTab("Scalar", jScrollPane1);
            
            gridBagConstraints1 = new java.awt.GridBagConstraints();
          gridBagConstraints1.gridx = 0;
          gridBagConstraints1.gridy = 1;
          gridBagConstraints1.fill = java.awt.GridBagConstraints.BOTH;
          gridBagConstraints1.weightx = 1.0;
          gridBagConstraints1.weighty = 1.0;
          jPanel1.add(jTabbedPane1, gridBagConstraints1);
          
          getContentPane().add(jPanel1, java.awt.BorderLayout.CENTER);
        
        setJMenuBar(jMenuBar1);
        pack();
    }//GEN-END:initComponents

    private void refreshOnceActionPerformed(java.awt.event.ActionEvent evt)
    {//GEN-FIRST:event_refreshOnceActionPerformed
        // Add your handling code here:
	all_scalar_atts.refresh();
	all_spectrum_atts.refresh();
	all_number_image_atts.refresh();
	
	if (devStateViewer.isVisible() || devStatusViewer.isVisible())
	{
           DeviceFactory.getInstance().refresh();
	}
	else
	{
	   if ((stateAtt != null) && (statusAtt != null))
	   {
	      if (stateAtt.hasEvents() == false)
	      {
		 state_status_atts.refresh();
	      }
	   }
	}
    }//GEN-LAST:event_refreshOnceActionPerformed

    private void stopStartRefreshActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_stopStartRefreshActionPerformed
        // Add your handling code here:
	javax.swing.JMenuItem   menuButton;
	
	menuButton = (javax.swing.JMenuItem) evt.getSource();
	if (menuButton.getText().equalsIgnoreCase("Stop   Refreshing"))
	{
	   menuButton.setText("Start   Refreshing");
	   jMenuItem7.setEnabled(true);
	
	   all_scalar_atts.stopRefresher();
	   all_spectrum_atts.stopRefresher();
	   all_number_image_atts.stopRefresher();
	
	   if (devStateViewer.isVisible() || devStatusViewer.isVisible())
	   {
	      if (keepStateRefresher == false)
		 DeviceFactory.getInstance().stopRefresher();
	   }
	   else
	   {
	      if ((stateAtt != null) && (statusAtt != null))
	      {
		 if (stateAtt.hasEvents() == false)
		 {
		    state_status_atts.stopRefresher();
		 }
	      }
	   }
	}
	else
	{
	   menuButton.setText("Stop   Refreshing");
           jMenuItem7.setEnabled(false);
	   
	   all_scalar_atts.startRefresher();
	   all_spectrum_atts.startRefresher();
	   all_number_image_atts.startRefresher();
	
	   if (devStateViewer.isVisible() || devStatusViewer.isVisible())
	   {
	      DeviceFactory.getInstance().startRefresher();
	   }
	   else
	   {
	      if ((stateAtt != null) && (statusAtt != null))
	      {
		 if (stateAtt.hasEvents() == false)
		 {
		    state_status_atts.startRefresher();
		 }
	      }
	   }
	}
    }//GEN-LAST:event_stopStartRefreshActionPerformed

    private void helpVersionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_helpVersionActionPerformed
        // Add your handling code here:
        String    versionText, versNumber;
        int       colon_idx, dollar_idx;
        
        versionText = new String("$Revision: 1.16 $");
        
        colon_idx = versionText.lastIndexOf(":");
        dollar_idx = versionText.lastIndexOf("$");
        versNumber = versionText.substring(colon_idx+1, dollar_idx);
        try
        {
            javax.swing.JOptionPane.showMessageDialog(this, 
	    "\n\n   atkpanel   : "+versNumber
	   +"\n\n   ESRF  :   Computing  Services \n\n",
            "atkpanel   Version", javax.swing.JOptionPane.PLAIN_MESSAGE );
        }
        catch (Exception e)
        {
        }
        
    }//GEN-LAST:event_helpVersionActionPerformed

    private void viewTrendActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_viewTrendActionPerformed
        // Add your handling code here:
        trendFrame.setVisible(true);
    }//GEN-LAST:event_viewTrendActionPerformed

    private void refPeriodActionPerformed(java.awt.event.ActionEvent evt)
    {//GEN-FIRST:event_refPeriodActionPerformed
        // Add your handling code here:
	
	int  ref_period = -1;
	ref_period = all_scalar_atts.getRefreshInterval();
	if (ref_period == -1)
	   ref_period = all_spectrum_atts.getRefreshInterval();
	if (ref_period == -1)
	   ref_period = all_number_image_atts.getRefreshInterval();
//	System.out.println("Initial ref period = "+(Integer.toString(ref_period)));

        String refp_str = JOptionPane.showInputDialog(this,"Enter refresh interval (ms)",(Object) new Integer(ref_period));
	if ( refp_str != null )
	{
	    try
	    {
        	int it = Integer.parseInt(refp_str);
//		System.out.println("Set ref period to : "+it);

		all_scalar_atts.setRefreshInterval(it);
		all_spectrum_atts.setRefreshInterval(it);
		all_number_image_atts.setRefreshInterval(it);
	
		if ((devStateViewer.isVisible()==false) && (devStatusViewer.isVisible()==false))
		{
		   if ((stateAtt != null) && (statusAtt != null))
		   {
		      if (stateAtt.hasEvents() == false)
		      {
			 state_status_atts.setRefreshInterval(it);
		      }
		   }
		}
	    }
	    catch ( NumberFormatException e )
	    {
        	JOptionPane.showMessageDialog(this,"Invalid number !","Error",JOptionPane.ERROR_MESSAGE);
		return;
	    }
	}
	
    }//GEN-LAST:event_refPeriodActionPerformed


    private void viewErrHistoryActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_viewErrHistoryActionPerformed
        // Add your handling code here:
	errorHistory.setVisible(true);
    }//GEN-LAST:event_viewErrHistoryActionPerformed


    private void jMenuItem1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuItem1ActionPerformed
        // Add your handling code here:
	if (standAlone == true)
           System.exit(0);
	else
	{
	   all_scalar_atts.stopRefresher();
	   all_spectrum_atts.stopRefresher();
	   all_number_image_atts.stopRefresher();
	
	   if (devStateViewer.isVisible() || devStatusViewer.isVisible())
	   {
	      if (keepStateRefresher == false)
		 DeviceFactory.getInstance().stopRefresher();
	   }
	   else
	   {
	      if ((stateAtt != null) && (statusAtt != null))
	      {
		 if (stateAtt.hasEvents() == false)
		 {
		    state_status_atts.stopRefresher();
		 }
	      }
	   }
	   this.dispose();
	}
    }//GEN-LAST:event_jMenuItem1ActionPerformed


    
    private void expertCheckBoxMenuItemActionPerformed(java.awt.event.ActionEvent evt)
    {
	
	if (all_cmdl.getSize() > 0)
           show_all_commands();
	
	if (all_number_image_atts.getSize() > 0)
           show_all_images();

	if (all_spectrum_atts.getSize() > 0)
           show_all_spectrums();

        if (all_scalar_atts.getSize() > 0)
           show_all_scalars();
	   
	expertCheckBoxMenuItem.setState(true);
	operatorCheckBoxMenuItem.setState(false);	
    }

    
    private void operatorCheckBoxMenuItemActionPerformed(java.awt.event.ActionEvent evt)
    {
	
	if (all_cmdl.getSize() > 0)
           show_operator_commands();
	
	if (all_number_image_atts.getSize() > 0)
           show_operator_images();

	if (all_spectrum_atts.getSize() > 0)
           show_operator_spectrums();

        if (all_scalar_atts.getSize() > 0)
	   show_operator_scalars();
	   
	expertCheckBoxMenuItem.setState(false);
	operatorCheckBoxMenuItem.setState(true);
    }



    /** Exit the Application */
    private void exitForm(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_exitForm
        if (standAlone == true)
	   System.exit(0);
	else
	{
	   all_scalar_atts.stopRefresher();
	   all_spectrum_atts.stopRefresher();
	   all_number_image_atts.stopRefresher();
	
	   if (devStateViewer.isVisible() || devStatusViewer.isVisible())
	   {
	      if (keepStateRefresher == false)
		 DeviceFactory.getInstance().stopRefresher();
	   }
	   else
	   {
	      if ((stateAtt != null) && (statusAtt != null))
	      {
		 if (stateAtt.hasEvents() == false)
		 {
		    state_status_atts.stopRefresher();
		 }
	      }
	   }
	   this.dispose();
	}
    }//GEN-LAST:event_exitForm



    /* Abort the Application */
    private void abortAppli()
    {
        if (standAlone == true)
	   System.exit(-1);
	else
	{
	   all_scalar_atts.stopRefresher();
	   all_spectrum_atts.stopRefresher();
	   all_number_image_atts.stopRefresher();
	
	   if (devStateViewer.isVisible() || devStatusViewer.isVisible())
	   {
	      if (keepStateRefresher == false)
		 DeviceFactory.getInstance().stopRefresher();
	   }
	   else
	   {
	      if ((stateAtt != null) && (statusAtt != null))
	      {
		 if (stateAtt.hasEvents() == false)
		 {
		    state_status_atts.stopRefresher();
		 }
	      }
	   }
	   this.dispose();
	}
    }







    private void  createAllScalarListViewers()
    {
       java.awt.GridBagConstraints            gbc;

       
       gbc = new java.awt.GridBagConstraints();
       gbc.gridx = 0;
       gbc.gridy = 0;
       gbc.insets = new java.awt.Insets(5, 5, 5, 5);
       gbc.fill = java.awt.GridBagConstraints.BOTH;
       gbc.weightx = 1.0;
       gbc.weighty = 1.0;
       
       allScalarListViewer = null;
       operatorScalarListViewer = null;
       
       if ((op_scalar_atts.getSize() <= 0) && (exp_scalar_atts.getSize() <= 0))
       {
	  javax.swing.JLabel  noatt = new javax.swing.JLabel();
	  noatt.setText("No   attribute ");
	  noatt.setFont(new java.awt.Font("Helvetica", java.awt.Font.PLAIN, 18));
	  noatt.setForeground(java.awt.Color.black);
	  jPanel3.add(noatt, gbc);
       }
       else
       { 
	  
	  // Merge operator and expert attributes in all_scalar_atts
	  for (int ind=0; ind < op_scalar_atts.getSize(); ind++)
	  {
	     IEntity ie = (IEntity) op_scalar_atts.get(ind);
	     all_scalar_atts.add(ie);
	  }
	  for (int ind=0; ind < exp_scalar_atts.getSize(); ind++)
	  {
	     IEntity ie = (IEntity) exp_scalar_atts.get(ind);
	     all_scalar_atts.add(ie);
	  }
	  
	  allScalarListViewer = new ScalarListViewer();
	  operatorScalarListViewer = new ScalarListViewer();
	  
	  if (modifPropButton == false)
	  {
	     allScalarListViewer.setPropertyButtonVisible(false);
	     operatorScalarListViewer.setPropertyButtonVisible(false);
	  }
	  
	  if (roMode == true)
	  {
	     allScalarListViewer.setSetterVisible(false);
	     operatorScalarListViewer.setSetterVisible(false);
	  }
	  
          allScalarListViewer.setTheFont(new java.awt.Font("Dialog", java.awt.Font.PLAIN, 14));
	  allScalarListViewer.setModel(all_scalar_atts);

          operatorScalarListViewer.setTheFont(new java.awt.Font("Dialog", java.awt.Font.PLAIN, 14));
	  operatorScalarListViewer.setModel(op_scalar_atts);
	  
        }
    }

    
    
    
    private void show_all_scalars()
    {
	int             neededWidth, neededHeight;

	
	neededWidth = allScalarListViewer.getPreferredSize().width;
	neededHeight = allScalarListViewer.getPreferredSize().height;

	if (neededWidth < MIN_WIDTH)
	   neededWidth = MIN_WIDTH;
	if (neededWidth > MAX_WIDTH)
	   neededWidth = MAX_WIDTH;

	if (neededHeight < MIN_HEIGHT)
	   neededHeight = MIN_HEIGHT;
	if (neededHeight > MAX_HEIGHT)
	   neededHeight = MAX_HEIGHT;	      

	jScrollPane1.setViewportView(allScalarListViewer);
	jTabbedPane1.setPreferredSize(new java.awt.Dimension(neededWidth+30, neededHeight+30));
	pack();
    }

    
    
    
    private void show_operator_scalars()
    {
	int             neededWidth, neededHeight;

	
	neededWidth = operatorScalarListViewer.getPreferredSize().width;
	neededHeight = operatorScalarListViewer.getPreferredSize().height;

	if (neededWidth < MIN_WIDTH)
	   neededWidth = MIN_WIDTH;
	if (neededWidth > MAX_WIDTH)
	   neededWidth = MAX_WIDTH;

	if (neededHeight < MIN_HEIGHT)
	   neededHeight = MIN_HEIGHT;
	if (neededHeight > MAX_HEIGHT)
	   neededHeight = MAX_HEIGHT;	      

	jScrollPane1.setViewportView(operatorScalarListViewer);
	jTabbedPane1.setPreferredSize(new java.awt.Dimension(neededWidth+30, neededHeight+30));
	pack();
    }



    

    private void createAllSpectrumTabs()
    {
	int                                         nb_atts, idx;
        SpectrumPanel                               sp_panel;
        StringSpectrumPanel                         str_sp_panel;
	IEntity                                     spectrum_att = null;
	INumberSpectrum                             nb_spectrum_att = null;
	IStringSpectrum                             str_spectrum_att = null;
        fr.esrf.tangoatk.core.AttributeList         all_sorted_spectrum_atts;

	
        all_spectrum_panels = new Vector();
	nb_atts = all_spectrum_atts.getSize();
	
	if (nb_atts > 0)
	{
	   // Sort the operator and the expert attributes in all_spectrum_atts
	   all_sorted_spectrum_atts = new fr.esrf.tangoatk.core.AttributeList();
	   for (int ind=0; ind < op_spectrum_atts.getSize(); ind++)
	   {
	      IEntity ie = (IEntity) op_spectrum_atts.get(ind);
	      all_sorted_spectrum_atts.add(ie);
	   }
	   for (int ind=0; ind < all_spectrum_atts.getSize(); ind++)
	   {
	      IEntity ie = (IEntity) all_spectrum_atts.get(ind);
	      if (ie.isExpert())
		 all_sorted_spectrum_atts.add(ie);	         
	   }
	   all_spectrum_atts = all_sorted_spectrum_atts; // use the sorted list
	}
	
	nb_atts = all_spectrum_atts.getSize();
	for (idx=0; idx < nb_atts; idx++)
	{
            spectrum_att = (IEntity) all_spectrum_atts.getElementAt(idx);
            if (spectrum_att instanceof INumberSpectrum)
	    {
	       nb_spectrum_att = (INumberSpectrum) spectrum_att;
	       // Create one Number Spectrum Panel per Number Spectrum Attribute
               sp_panel = new SpectrumPanel(nb_spectrum_att);
               // Add the spectrum panel as a tab into the tabbed panel of the main frame
               jTabbedPane1.addTab(nb_spectrum_att.getNameSansDevice(), sp_panel);
	       all_spectrum_panels.add(idx, sp_panel);
	    }
	    else
	    {
	       if (spectrum_att instanceof IStringSpectrum)
	       {
		  str_spectrum_att = (IStringSpectrum) spectrum_att;
		  // Create one Number Spectrum Panel per Number Spectrum Attribute
        	  str_sp_panel = new StringSpectrumPanel(str_spectrum_att);
        	  // Add the spectrum panel as a tab into the tabbed panel of the main frame
        	  jTabbedPane1.addTab(str_spectrum_att.getNameSansDevice(), str_sp_panel);
		  all_spectrum_panels.add(idx, str_sp_panel);
	       }
	    }
        }
    }
    
    
    private void show_operator_spectrums()
    {
       int                  nb_tabs = 0;
       Component            specPanel = null;
       INumberSpectrum      nSpecAtt = null;
       IStringSpectrum      strSpecAtt = null;
       int                  ind, specInd;
       
       
       nb_tabs = jTabbedPane1.getTabCount();
       
       for (ind = 1; ind < nb_tabs; ind++)
       {
	  specPanel = jTabbedPane1.getComponentAt(ind);
	  if (specPanel instanceof SpectrumPanel)
	  {
	     SpectrumPanel sp = (SpectrumPanel) specPanel;
	     nSpecAtt = sp.getModel();
	     specInd = op_spectrum_atts.indexOf(nSpecAtt);
	     if (specInd < 0)
		jTabbedPane1.removeTabAt(ind);
	  }
	  else
	  {
	     if (specPanel instanceof StringSpectrumPanel)
	     {
		StringSpectrumPanel strsp = (StringSpectrumPanel) specPanel;
		strSpecAtt = strsp.getModel();
		specInd = op_spectrum_atts.indexOf(strSpecAtt);
		if (specInd < 0)
		   jTabbedPane1.removeTabAt(ind);
	     }
	  }
       }
    }

    
    
    private void show_all_spectrums()
    {
       int                  nb_spectrums = 0;
       Object               obj=null, spObj=null;
       SpectrumPanel        specPanel = null;
       StringSpectrumPanel  strspecPanel = null;
       INumberSpectrum      nSpecAtt = null;
       IStringSpectrum      strSpecAtt = null;
       int                  ind, specInd;
       
       
       nb_spectrums = all_spectrum_atts.getSize();      
       
       for (ind = 0; ind < nb_spectrums; ind++)
       {
	  obj = all_spectrum_atts.get(ind);
	  if (obj instanceof INumberSpectrum)
	  {
	      nSpecAtt = (INumberSpectrum) obj;
	      specInd = jTabbedPane1.indexOfTab(nSpecAtt.getNameSansDevice());
	      if (specInd < 0) // This spectrum is not currently in the tabbed pane
	      {
		spObj = all_spectrum_panels.get(ind);
		if (spObj instanceof SpectrumPanel)
		{
		   specPanel = (SpectrumPanel) spObj;
        	   jTabbedPane1.addTab(nSpecAtt.getNameSansDevice(), specPanel);
		}
	      }
	  }
	  else
	  {
	      if (obj instanceof IStringSpectrum)
	      {
		 strSpecAtt = (IStringSpectrum) obj;
		 specInd = jTabbedPane1.indexOfTab(strSpecAtt.getNameSansDevice());
		 if (specInd < 0) // This spectrum is not currently in the tabbed pane
		 {
		   spObj = all_spectrum_panels.get(ind);
		   if (spObj instanceof StringSpectrumPanel)
		   {
		      strspecPanel = (StringSpectrumPanel) spObj;
        	      jTabbedPane1.addTab(strSpecAtt.getNameSansDevice(), strspecPanel);
		   }
		 }
	      }
	  }
       }
    }

    

    

    private void createAllNumberImageTabs()
    {
	int                                         nb_atts, idx;
        ImagePanel                                  att_tab;
	INumberImage                                image_att = null;
        fr.esrf.tangoatk.core.AttributeList         all_sorted_image_atts;

	
        all_image_panels = new Vector();
	nb_atts = all_number_image_atts.getSize();
	
	if (nb_atts > 0)
	{
	   // Sort the operator and the expert attributes in all_number_image_atts
	   all_sorted_image_atts = new fr.esrf.tangoatk.core.AttributeList();
	   for (int ind=0; ind < op_number_image_atts.getSize(); ind++)
	   {
	      IEntity ie = (IEntity) op_number_image_atts.get(ind);
	      all_sorted_image_atts.add(ie);
	   }
	   for (int ind=0; ind < all_number_image_atts.getSize(); ind++)
	   {
	      IEntity ie = (IEntity) all_number_image_atts.get(ind);
	      if (ie.isExpert())
		 all_sorted_image_atts.add(ie);
	   }
	   all_number_image_atts = all_sorted_image_atts; // use the sorted list
	}
	
	nb_atts = all_number_image_atts.getSize();
	for (idx=0; idx < nb_atts; idx++)
	{
            image_att = (INumberImage) all_number_image_atts.getElementAt(idx);
            // Create one Image Panel per Number Imag Attribute
            att_tab = new ImagePanel(image_att);
            // Add the Image panel as a tab into the tabbed panel of the main frame
            jTabbedPane1.addTab(image_att.getNameSansDevice(), att_tab);
	    all_image_panels.add(idx, att_tab);
        }
    }
    
    
    private void show_operator_images()
    {
       int                  nb_tabs = 0;
       Component            imagePanel = null;
       INumberImage         nImageAtt = null;
       int                  ind, imageInd;
       
       
       nb_tabs = jTabbedPane1.getTabCount();
       
       for (ind = 1; ind < nb_tabs; ind++)
       {
	  imagePanel = jTabbedPane1.getComponentAt(ind);
	  if (imagePanel instanceof ImagePanel)
	  {
	     ImagePanel ip = (ImagePanel) imagePanel;
	     nImageAtt = ip.getModel();
	     imageInd = op_number_image_atts.indexOf(nImageAtt);
	     if (imageInd < 0)
		jTabbedPane1.removeTabAt(ind);
	  }
       }
    }

    
    
    private void show_all_images()
    {
       int                  nb_images = 0;
       Object               obj=null, imageObj=null;
       ImagePanel           imagePanel = null;
       INumberImage         nImageAtt = null;
       int                  ind, imageInd;
       
       
       nb_images = all_number_image_atts.getSize();      
       
       for (ind = 0; ind < nb_images; ind++)
       {
	  obj = all_number_image_atts.get(ind);
	  if (obj instanceof INumberImage)
	  {
	     nImageAtt = (INumberImage) obj;
	     imageInd = jTabbedPane1.indexOfTab(nImageAtt.getNameSansDevice());
	     if (imageInd < 0) // This image is not currently in the tabbed pane
	     {
		 imageObj = all_image_panels.get(ind);
		 if (imageObj instanceof ImagePanel)
		 {
		    imagePanel = (ImagePanel) imageObj;
        	    jTabbedPane1.addTab(nImageAtt.getNameSansDevice(), imagePanel);
		 }
	     }
	  }
       }
    }




    private void createAllCmdList()
    {
       int                 nb_cmds;
       CommandList         all_sorted_cmds;
       
       // Filter the commands to show here :
       
       nb_cmds = all_cmdl.getSize();
	
       if (nb_cmds > 0)
       {
	   // Sort operator and expert commands in all_cmdl
	   all_sorted_cmds = new fr.esrf.tangoatk.core.CommandList();
	   for (int ind=0; ind < op_cmdl.getSize(); ind++)
	   {
	      IEntity ie = (IEntity) op_cmdl.get(ind);
	      all_sorted_cmds.add(ie);
	   }
	   for (int ind=0; ind < all_cmdl.getSize(); ind++)
	   {
	      IEntity ie = (IEntity) all_cmdl.get(ind);
	      if (ie.isExpert())
		 all_sorted_cmds.add(ie);
	   }
	   all_cmdl = all_sorted_cmds; // use the sorted list

           commandComboViewer1.setModel(all_cmdl);
       }
       else
           commandComboViewer1.setModel(null);
    }
    
    
    private void show_operator_commands()
    {
	if (all_cmdl.getSize() > 0)
	{
           commandComboViewer1.setModel(op_cmdl);
	}
    }

    
    private void show_all_commands()
    {
	if (all_cmdl.getSize() > 0)
	{
           commandComboViewer1.setModel(all_cmdl);
	}
    }

    

    
     
    /**
    * @param args the command line arguments
    */
    public static void main(String args[])
    {
       // startup with standAlone=true, keepStateRefresher=false, modifPropButton=true, roMode=false
       if (args.length <= 0)
          new MainPanel((String) null, true, false); // modifPropButton and roMode = leur default values
       else // args.length > 0
       {
	  if (args.length == 1)
	  {
	     new MainPanel(args[0], true, false); // modifPropButton and roMode = leur default values
	  }
	  else // args.length > 1
	  {
	     if (args[0].equalsIgnoreCase("-ro"))
	        new MainPanel(args[1], true, false, false, true);
             else
	        if (args[1].equalsIgnoreCase("-ro"))
	           new MainPanel(args[0], true, false, false, true);
		else
		   new MainPanel(args[0], true, false);
	  }
       }
        
    }


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JMenuBar jMenuBar1;
    private javax.swing.JMenu jMenu1;
    private javax.swing.JMenuItem jMenuItem1;
    private javax.swing.JMenu jMenu3;
    private javax.swing.JMenuItem jMenuItem5;
    private javax.swing.JMenuItem jMenuItem3;
    private javax.swing.JMenuItem diagJMenuItem;
    private javax.swing.JMenu jMenu2;
    private javax.swing.JMenuItem jMenuItem6;
    private javax.swing.JMenuItem jMenuItem7;
    private javax.swing.JMenuItem jMenuItem2;
    private javax.swing.JCheckBoxMenuItem expertCheckBoxMenuItem;
    private javax.swing.JCheckBoxMenuItem operatorCheckBoxMenuItem;
    private javax.swing.JMenu jMenu4;
    private javax.swing.JMenuItem jMenuItem4;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private fr.esrf.tangoatk.widget.command.CommandComboViewer commandComboViewer1;
    private fr.esrf.tangoatk.widget.device.StateViewer devStateViewer;
    private fr.esrf.tangoatk.widget.device.StatusViewer devStatusViewer;
    private fr.esrf.tangoatk.widget.attribute.StateViewer attStateViewer;
    private fr.esrf.tangoatk.widget.attribute.StatusViewer attStatusViewer;
    private javax.swing.JTabbedPane jTabbedPane1;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JPanel jPanel3;
    // End of variables declaration//GEN-END:variables

}
