//+======================================================================
// $Source: /cvsroot/tango-cs/tango/tools/astor/ServArchitectureDialog.java,v $
//
// Project:   Tango Manager (Astor)
//
// Description:   Dialog Class to display server architecture.
//
// $Author: pascal_verdier $
//
// $Revision: 3.3 $
// $Log: ServArchitectureDialog.java,v $
// Revision 3.3  2005/09/27 12:45:59  pascal_verdier
// Expand button added.
//
// Revision 3.2  2005/09/15 13:44:04  pascal_verdier
// jive.MultiLineToolTipUI.initialize() call added.
//
// Revision 3.1  2005/09/15 08:26:36  pascal_verdier
// Server architecture display addded.
//
//
//
// Copyleft 2005 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package admin.astor;

import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import fr.esrf.TangoApi.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;


//===============================================================
/**
 *	Class Description: Basic Dialog Class to display info
 *
 *	@author  root
 */
//===============================================================


public class ServArchitectureDialog extends JDialog {
	private JDialog	parent;
	private int		retVal = JOptionPane.OK_OPTION;

	private String	servname;
	private boolean	from_appli = true;
	
	private ServInfoTree	tree;
	static public final boolean EXPAND_NOT_FULL = false;
	static public final boolean EXPAND_FULL     = true;

	//===============================================================
	/**
	 *	Creates new form ServArchitectureDialog
	 */
	//===============================================================
	public ServArchitectureDialog(JDialog parent, String servname) throws DevFailed
	{
		super(parent, false);
		this.parent   = parent;
		this.servname = servname;
		initComponents();

		titleLabel.setText(servname + "  Architecture");

		//	Show the result
		tree = new ServInfoTree();
		jScrollPane1.setViewportView (tree);

		//	Check if from an appli or from an empty JDialog
		if (parent.getWidth()==0)
			from_appli = false;

		jive.MultiLineToolTipUI.initialize();
		pack();
		tree.expandTree(EXPAND_NOT_FULL);
	}
	
	//===============================================================
	/**
	 *	Creates new form ServArchitectureDialog
	 */
	//===============================================================
	public ServArchitectureDialog(JDialog parent, DeviceProxy dev) throws DevFailed
	{
		this( parent, dev.name().substring("dserver/".length()) );
	}

	//===============================================================
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
	//===============================================================
    private void initComponents() {//GEN-BEGIN:initComponents
        jPanel1 = new javax.swing.JPanel();
        expandBtn = new javax.swing.JRadioButton();
        cancelBtn = new javax.swing.JButton();
        jPanel2 = new javax.swing.JPanel();
        titleLabel = new javax.swing.JLabel();
        jScrollPane1 = new javax.swing.JScrollPane();

        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });

        expandBtn.setText("Expand all properties");
        expandBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                expandBtnActionPerformed(evt);
            }
        });

        jPanel1.add(expandBtn);

        cancelBtn.setText("Dismiss");
        cancelBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelBtnActionPerformed(evt);
            }
        });

        jPanel1.add(cancelBtn);

        getContentPane().add(jPanel1, java.awt.BorderLayout.SOUTH);

        titleLabel.setFont(new java.awt.Font("Dialog", 1, 18));
        titleLabel.setText("Dialog Title");
        jPanel2.add(titleLabel);

        getContentPane().add(jPanel2, java.awt.BorderLayout.NORTH);

        getContentPane().add(jScrollPane1, java.awt.BorderLayout.CENTER);

        pack();
    }//GEN-END:initComponents

	//===============================================================
	//===============================================================
	private void expandBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_expandBtnActionPerformed

		if (expandBtn.getSelectedObjects()!=null)
			tree.expandTree(EXPAND_FULL);
		else
			tree.expandTree(EXPAND_NOT_FULL);

	}//GEN-LAST:event_expandBtnActionPerformed

	//===============================================================
	//===============================================================
	private void cancelBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelBtnActionPerformed
		retVal = JOptionPane.CANCEL_OPTION;
		doClose();
	}//GEN-LAST:event_cancelBtnActionPerformed

	//===============================================================
	/**
	 *	Closes the dialog
	 */
	//===============================================================
	private void closeDialog(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_closeDialog
		retVal = JOptionPane.CANCEL_OPTION;
		doClose();
	}//GEN-LAST:event_closeDialog

	//===============================================================
	/**
	 *	Closes the dialog
	 */
	//===============================================================
	private void doClose()
	{
		setVisible(false);
		dispose();
		if (from_appli==false)
			System.exit(0);
	}
	//===============================================================
	//===============================================================
	public int showDialog()
	{
		setVisible(true);
		return retVal;
	}

	//===============================================================
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel titleLabel;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JRadioButton expandBtn;
    private javax.swing.JButton cancelBtn;
    // End of variables declaration//GEN-END:variables
	//===============================================================







	//===============================================================
	/**
	 *	JTree Class
	 */
	//===============================================================
	class ServInfoTree extends JTree 
	{
		private DeviceProxy	dev     = null;
		private String[]	devlist = null;
		private DefaultTreeModel	treeModel;
		private DefaultMutableTreeNode root;
		//===============================================================
		//===============================================================
		public ServInfoTree() throws DevFailed
		{
			super();
			initComponent();
		}
		//===============================================================
		//===============================================================
		public ServInfoTree(DeviceProxy dev) throws DevFailed
		{
			super();
			this.dev = dev;
			initComponent();
		}
		//===============================================================
		//===============================================================
		void expandTree(boolean expand)
		{
			expandTree(root, expand);
		}
		//===============================================================
		/**
		 *	Expend tree from node (re-entring method)
		 *
		 *	@param node origin to start expanding.
		 */
		//===============================================================
		private void expandTree(DefaultMutableTreeNode node, boolean expand)
		{
			int	nb = node.getChildCount();
			for (int i=0 ; i<nb ; i++)
			{
				DefaultMutableTreeNode	child = 
						(DefaultMutableTreeNode)node.getChildAt(i);

				Object obj = child.getUserObject();
				//	Check if all must be expanded
				if (expand==EXPAND_FULL ||
					((obj instanceof TgProperty) == false) )
				{
					TreePath	path = new TreePath(child.getPath());
					expandPath(path);
					expandTree(child, expand);
				}

				//	Check if something must be collapsed
				if (expand==EXPAND_NOT_FULL &&
					((obj instanceof TgProperty) == true) )
				{
					TreePath	path = new TreePath(child.getPath());
					collapsePath(path);
				}
			}
		}
		//===============================================================
		//===============================================================
		private void initComponent() throws DevFailed
		{
			//	Create the nodes (root is the server).
			root = new DefaultMutableTreeNode(new TgServer(servname));

			createNodes(root);
			//	Create the tree that allows one selection at a time.
			getSelectionModel().setSelectionMode
        			(TreeSelectionModel.SINGLE_TREE_SELECTION);

			//	Create Tree and Tree model
			//------------------------------------
			treeModel = new DefaultTreeModel(root);
			setModel(treeModel);

			//	Enable tool tips.
			ToolTipManager.sharedInstance().registerComponent(this);

			//	Set the icon for leaf nodes.
			setCellRenderer(new TangoRenderer());
		}
		//===============================================================
		/**
		 *	Create the server tree
		 */
		//===============================================================
		private void createNodes(DefaultMutableTreeNode root) throws DevFailed
		{
			if (dev==null)
				dev = new DeviceProxy("dserver/"+servname);

			String[]	classnames = getClasses();
			DefaultMutableTreeNode[] classnodes =
				new DefaultMutableTreeNode[classnames.length];
			for (int c=0 ; c<classnames.length ; c++)
			{
				//	Display class part
				TgClass	_class = new TgClass(classnames[c]);
				classnodes[c] = new DefaultMutableTreeNode(_class);
				root.add(classnodes[c]);

				TgProperty[]	cl_prop = getProperties(classnames[c], "Class");
				for (int p=0 ; p<cl_prop.length ; p++)
				{
					cl_prop[p] = checkDbProperty(cl_prop[p]);
					DefaultMutableTreeNode	node = 
						new DefaultMutableTreeNode(cl_prop[p]);
					classnodes[c].add(node);
					node.add(new DefaultMutableTreeNode(cl_prop[p].value));
				}

				//	Display device part
				String[]	devnames = getDevices(classnames[c]);
				DefaultMutableTreeNode[] devnodes =
					new DefaultMutableTreeNode[devnames.length];
				TgProperty[]	dev_prop = getProperties(classnames[c], "Dev");
				for (int d=0 ; d<devnames.length ; d++)
				{
					devnodes[d] =
						new DefaultMutableTreeNode("Device: " + devnames[d]);
					classnodes[c].add(devnodes[d]);
					for (int p=0 ; p<dev_prop.length ; p++)
					{
						dev_prop[p] = checkDbProperty(dev_prop[p]);
						DefaultMutableTreeNode	node = 
							new DefaultMutableTreeNode(dev_prop[p]);
						devnodes[d].add(node);
						node.add(new DefaultMutableTreeNode(dev_prop[p].value));
					}
				}
			}
		}
		//===============================================================
		/**
		 *	Read database for property.
		 *	If exists, overwrite value.
		 */
		//===============================================================
		private TgProperty checkDbProperty(TgProperty prop)
		{
			try{
				DbDatum	data;
				if (prop.src.equals("Dev"))
					data = new DeviceProxy(prop.objname).get_property(prop.name);
				else
					data = new DbClass(prop.objname).get_property(prop.name);
				if (data.is_empty()==false)
				{
					String[]	tmp = data.extractStringArray();
					prop.value = "";

					for (int i=0 ; i<tmp.length ; i++)
					{
						prop.value += tmp[i];
						if (i<(tmp.length-1))
							prop.value += ", ";
					}
				}
			}
			catch(DevFailed e){}
			return prop;
		}
		//===============================================================
		//===============================================================
		private String[] getDevices(String classname) throws DevFailed
		{
			if (devlist==null)
			{
				DeviceData	argout  = dev.command_inout("QueryDevice");
				devlist = argout.extractStringArray();
			}
			Vector	v = new Vector();
			String	str = classname + "::";
			for (int i=0 ; i<devlist.length ; i++)
				if (devlist[i].startsWith(str))
					v.add(devlist[i].substring(str.length()));
			String[]	devnames = new String[v.size()];
			for (int i=0 ; i<v.size() ; i++)
				devnames[i] = (String) v.elementAt(i);
			return devnames;
		}
		//===============================================================
		//===============================================================
		private String[] getClasses() throws DevFailed
		{
			DeviceData	argout = dev.command_inout("QueryClass");
			return argout.extractStringArray();
		}
		//===============================================================
		//===============================================================
		private TgProperty[] getProperties(String classname, String source) throws DevFailed
		{
			DeviceData		argin  = new DeviceData();
			argin.insert(classname);
			String			cmd    = "QueryWizard" + source + "Property";
			DeviceData		argout = dev.command_inout(cmd, argin);
			String[]		str    = argout.extractStringArray();
			TgProperty[]	prop   = new TgProperty[str.length/3];
			for (int i=0, n=0 ; i<str.length ; n++, i+=3)
				prop[n] = new TgProperty(classname, source, str[i], str[i+1], str[i+2]);
			return prop;
		}
	}
	//===============================================================
	/**
	 *	Renderer Class
	 */
	//===============================================================
    private class TangoRenderer extends DefaultTreeCellRenderer
	{
			private ImageIcon	root_icon;
			private ImageIcon	class_icon;
			private ImageIcon	dev_icon;
			private ImageIcon	prop_icon;
			private ImageIcon	leaf_icon;
			private Font[]		fonts;

			private final int	TITLE      = 0;
			private final int	CLASS_DEV  = 1;
			private final int	PROP_NAME  = 2;
			private final int	PROP_DESC  = 3;
			public final String		img_path = "/admin/img/";

		//===============================================================
		//===============================================================
		public TangoRenderer()
		{
			root_icon = new ImageIcon(getClass().getResource(img_path + "network5.gif"));

			class_icon = new ImageIcon(getClass().getResource(img_path + "class.gif"));
			dev_icon = new ImageIcon(getClass().getResource(img_path + "device.gif"));
			prop_icon = new ImageIcon(getClass().getResource(img_path + "attleaf.gif"));
			leaf_icon = new ImageIcon(getClass().getResource(img_path + "uleaf.gif"));

			fonts = new Font[4];
			fonts[TITLE]     = new Font("helvetica", Font.BOLD, 18);
			fonts[CLASS_DEV] = new Font("helvetica", Font.BOLD, 16);
			fonts[PROP_NAME] = new Font("helvetica", Font.BOLD, 14);
			fonts[PROP_DESC] = new Font("helvetica", Font.PLAIN, 12);
		}

		//===============================================================
		//===============================================================
		public Component getTreeCellRendererComponent(
                            JTree tree,
                            Object obj,
                            boolean sel,
                            boolean expanded,
                            boolean leaf,
                            int row,
                            boolean hasFocus) {

            super.getTreeCellRendererComponent(
                            tree, obj, sel,
                            expanded, leaf, row,
                            hasFocus);
			String	tip = null;
			DefaultMutableTreeNode node = (DefaultMutableTreeNode)obj;
			Object	user_obj = node.getUserObject();
			if (row==0)
			{
				//	ROOT
				setIcon(root_icon);
				setFont(fonts[TITLE]);
				tip = ((TgServer)user_obj).desc;
			}
			else
			{
				if (user_obj instanceof TgClass)
				{
					setIcon(class_icon);
					setFont(fonts[CLASS_DEV]);
					tip = ((TgClass)user_obj).desc;
				}
				else
				if (user_obj instanceof TgProperty)
				{
					//	Property name
					setIcon(prop_icon);
					setFont(fonts[PROP_NAME]);
					tip = ((TgProperty)user_obj).desc;
				}
				else
				if (user_obj instanceof String)
				{
					String	str = user_obj.toString();

					if (str.startsWith("Device:"))
					{
						setIcon(dev_icon);
						setFont(fonts[CLASS_DEV]);
					}
					else
					{
						//	Property desc and value
						setIcon(leaf_icon);
						setFont(fonts[PROP_DESC]);
					}
				}
			}
			setToolTipText(tip);
            return this;
		}
	}
	//===============================================================
	/**
	 *	Class to define TANGO Server object
	 */
	//===============================================================
	class TgServer
	{
		String	name;
		String	desc;
		
		//===============================================================
		public TgServer(String name)
		{
			this.name  = name;
			this.desc  = "";
			
			try
			{
				String		admin = "dserver/" + name;
				IORdump 	id = new IORdump(admin);
				DeviceInfo	info = new DbDevice(admin).get_info();
				desc = id.toString() + "\nsince " + info.last_exported;
			}
			catch(DevFailed e){}
		}
		//===============================================================
		public String toString()
		{
			return name;
		}
	}
	//===============================================================
	/**
	 *	Class to define TANGO property object
	 */
	//===============================================================
	class TgProperty
	{
		String	objname;
		String	src;
		String	name;
		String	desc;
		String	value;
		
		//===============================================================
		public TgProperty(String objname, String src, String name, String desc, String value)
		{
			this.objname = objname;
			this.src     = src;
			this.name    = name;
			this.desc    = desc;
			this.value   = "";

			//	Take of '\n'
			int	idx;
			while ((idx=value.indexOf('\n'))>=0)
				value = value.substring(0, idx) + ", " +
					value.substring(idx + 1);
			this.value = value;
		}
		//===============================================================
		public String toString()
		{
			return name;
		}
	}
	//===============================================================
	/**
	 *	Class to define TANGO Class Object
	 */
	//===============================================================
	class TgClass
	{
		String	name;
		String	desc;
		//===============================================================
		public TgClass(String name)
		{
			this.name  = name;
			this.desc  = "No Description Found in Database";
			try
			{
				DbDatum data = new DbClass(name).get_property("Description");
				if (data.is_empty()==false)
				{
					String[]	array = data.extractStringArray();					
					this.desc = "";
					for (int i=0 ; i<array.length ; i++)
						this.desc += array[i] + "\n";
				}
			}
			catch(DevFailed e) {}
		}
		//===============================================================
		public String toString()
		{
			return "Class: " + name;
		}
	}
	//===============================================================
	//===============================================================










	//===============================================================
	/**
	* @param args the command line arguments
	*/
	//===============================================================
	public static void main(String args[]) {
		String		servname = "PowerSupply/pv";
		if (args.length>0)
			servname = args[0];
		try
		{
			//new ServArchitectureDialog(new javax.swing.JDialog(), servname).show();
			new ServArchitectureDialog(new javax.swing.JDialog(),
						new DeviceProxy("dserver/"+servname)).setVisible(true);
		}
		catch(DevFailed e)
		{
			app_util.PopupError.show(new javax.swing.JDialog(), e);
			System.exit(0);
		}

	}
}
