/*-
 * Copyright © 2009 Diamond Light Source Ltd.
 *
 * This file is part of GDA.
 *
 * GDA is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License version 3 as published by the Free
 * Software Foundation.
 *
 * GDA is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with GDA. If not, see <http://www.gnu.org/licenses/>.
 */

package gda.device.detector.pilatus;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import fr.esrf.Tango.DevFailed;
import fr.esrf.Tango.DevState;
import fr.esrf.TangoApi.DeviceAttribute;
import gda.device.DeviceException;
import gda.device.DummyTangoDeviceProxy;

import org.junit.Test;

public class TangoPilatusDetectorTest {

	private DummyTangoDeviceProxy dev;
	private TangoPilatusDetector detector = new TangoPilatusDetector();

	public TangoPilatusDetectorTest() {
		dev = new DummyTangoDeviceProxy("tango::12345");
		try {
			// setup some default values in the dummy device proxy
			dev.write_attribute(new DeviceAttribute("ExposureTime", 3.0));
			dev.write_attribute(new DeviceAttribute("ExposurePeriod", 10.0));
			dev.write_attribute(new DeviceAttribute("NbFrames", 25));
			dev.write_attribute(new DeviceAttribute("NbExposures", 3));
			dev.write_attribute(new DeviceAttribute("DelayTime", 7.5));
			dev.write_attribute(new DeviceAttribute("ShutterEnable", false));
			dev.write_attribute(new DeviceAttribute("TriggerMode", (short) 1));
			dev.write_attribute(new DeviceAttribute("FileDir", "/tmp/test"));
			dev.write_attribute(new DeviceAttribute("FilePrefix", "pre"));
			dev.write_attribute(new DeviceAttribute("FileStartNum", 7));
			dev.write_attribute(new DeviceAttribute("FilePostfix", "post"));
			dev.write_attribute(new DeviceAttribute("LastImageTaken", "lastImage"));
			dev.write_attribute(new DeviceAttribute("Threshold", 3));
			dev.write_attribute(new DeviceAttribute("Gain", (short) 2));
		} catch (DevFailed e) {
			fail("This should not occur");
		}
		detector.setTangoDeviceProxy(dev);
	}

	@Test
	public void testGetAttribute() {
		try {
			assertEquals(3.0, detector.getAttribute("ExposureTime"));
			assertEquals(10.0, detector.getAttribute("ExposurePeriod"));
			assertEquals(25, detector.getAttribute("NbFrames"));
			assertEquals(3, detector.getAttribute("NbExposures"));
			assertEquals(7.5, detector.getAttribute("DelayTime"));
			assertEquals(false, detector.getAttribute("ShutterEnable"));
			assertEquals((short)1, detector.getAttribute("TriggerMode"));
			assertEquals("/tmp/test", detector.getAttribute("FileDir"));
			assertEquals("pre", detector.getAttribute("FilePrefix"));
			assertEquals(7, detector.getAttribute("FileStartNum"));
			assertEquals("post", detector.getAttribute("FilePostfix"));
			assertEquals("lastImage", detector.getAttribute("LastImageTaken"));
			assertEquals(3, detector.getAttribute("Threshold"));
			assertEquals((short)2, detector.getAttribute("Gain"));
		} catch (Exception e) {
			fail("Exception from getAttribute()");
		}
	}

	@Test
	public void testState() {
		try {
			dev.setState(DevState.STANDBY);
			assertEquals("STANDBY", dev.status());
			assertEquals(7, detector.getStatus());
			dev.setState(DevState.UNKNOWN);
			assertEquals("UNKNOWN", dev.status());
			assertEquals(13, detector.getStatus());
			dev.setState(DevState.ON);
			assertEquals("ON", dev.status());
			assertEquals(0, detector.getStatus());
		} catch (DeviceException e) {
			fail("Exception from getStatus()");
		}
	}
	
//	@Test
	public void testSetAttribute() {
		try {
			detector.setAttribute("StepSize", 3.3);
			assertEquals(3.3, detector.getAttribute("StepSize"));
			detector.setAttribute("StepSize", 1.5);
			assertEquals(1.5, detector.getAttribute("StepSize"));

			detector.setAttribute("Acceleration", 18);
			assertEquals(18, detector.getAttribute("Acceleration"));
			detector.setAttribute("Acceleration", 15);
			assertEquals(15, detector.getAttribute("Acceleration"));

			detector.setAttribute("Backlash", 8.6);
			assertEquals(8.6, detector.getAttribute("Backlash"));
			detector.setAttribute("Backlash", 3.5);
			assertEquals(3.5, detector.getAttribute("Backlash"));

			detector.setAttribute("FirstVelocity", 14);
			assertEquals(14, detector.getAttribute("FirstVelocity"));
			detector.setAttribute("FirstVelocity", 10);
			assertEquals(10, detector.getAttribute("FirstVelocity"));

			detector.setAttribute("Steps_per_unit", 5.5);
			assertEquals(5.5, detector.getAttribute("Steps_per_unit"));
			detector.setAttribute("Steps_per_unit", 3.0);
			assertEquals(3.0, detector.getAttribute("Steps_per_unit"));

			detector.setAttribute("Home_position", 0.5);
			assertEquals(0.5, detector.getAttribute("Home_position"));
			detector.setAttribute("Home_position", -0.5);
			assertEquals(-0.5, detector.getAttribute("Home_position"));

			detector.setAttribute("PresetPosition", 5.5);
			assertEquals(5.5, detector.getAttribute("PresetPosition"));
			detector.setAttribute("PresetPosition", 5.0);
			assertEquals(5.0, detector.getAttribute("PresetPosition"));
		} catch (Exception e) {
			fail("Exception from getAttribute()");
		}
	}
}
