/*-
 * Copyright © 2009 Diamond Light Source Ltd.
 *
 * This file is part of GDA.
 *
 * GDA is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License version 3 as published by the Free
 * Software Foundation.
 *
 * GDA is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with GDA. If not, see <http://www.gnu.org/licenses/>.
 */

package gda.device.detector.pilatus;

import fr.esrf.Tango.DevFailed;
import fr.esrf.Tango.DevState;
import fr.esrf.TangoApi.DeviceAttribute;
import gda.analysis.ScanFileHolder;
import gda.analysis.io.ScanFileHolderException;
import gda.device.Detector;
import gda.device.DeviceException;
import gda.device.Scannable;
import gda.device.TangoDeviceProxy;
import gda.device.detector.DetectorBase;
import gda.device.detector.corba.impl.DetectorAdapter;
import gda.device.detector.corba.impl.DetectorImpl;
import gda.factory.Configurable;
import gda.factory.FactoryException;
import gda.factory.corba.util.CorbaAdapterClass;
import gda.factory.corba.util.CorbaImplClass;
import gda.io.PilatusEdfLoader;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@CorbaAdapterClass(DetectorAdapter.class)
@CorbaImplClass(DetectorImpl.class)
public class TangoPilatusDetector extends DetectorBase implements Detector, Scannable, Configurable {

	private static final Logger logger = LoggerFactory.getLogger(TangoPilatusDetector.class);
	private TangoDeviceProxy dev = null;

	private int width = 1;
	private int height = 1;
	private boolean createsOwnFile = true;
	private String detectorType = "Pilatus";
	private String detectorID = "Pilatus";
	private String detectorDescription = "Pilatus";
		
//	public static enum gainSettings {lowG, midG, highG, uhighG,}
//	public static enum triggerSettings {Internal, ExternalEnable, ExternalTrigger, MultipleExternalTrigger}
	
	private double exposureTime = 0.0;
	private double exposurePeriod = 0.0;
	private int nbFrames = 1;
	private int nbExposures = 1;
	private double delayTime = 0.0;
	private boolean shutterEnable = false;
	private short triggerMode = 0;
	private String fileDir = "";
	private String filePrefix = "";
	private int fileStartNum = 1;
	private String filePostfix = "";
	private String lastImageTaken;
	private int threshold = 0;
	private short gain = 0;

	@Override
	public void configure() throws FactoryException {
		try {
			if (dev != null)
				init();
		} catch (Exception e) {
			System.out.println(e);
			System.out.println(e.getMessage());
		}
	}

	@Override
	public void reconfigure() throws FactoryException {
		configured = false;
		logger.debug("NcdDetector reconfiguring " + getName());
		configure();
	}

	/**
	 * @return Returns the Tango device proxy.
	 */
	public TangoDeviceProxy getTangoDeviceProxy() {
		return dev;
	}

	/**
	 * @param dev The Tango device proxy to set.
	 */
	public void setTangoDeviceProxy(TangoDeviceProxy dev) {
		this.dev = dev;
	}

	public int getWidth() {
		return width;
	}

	public void setWidth(int width) {
		this.width = width;
	}

	public int getHeight() {
		return height;
	}

	public void setHeight(int height) {
		this.height = height;
	}

	public double getExposureTime() throws DeviceException {
		isAvailable();
		try {
			exposureTime = dev.read_attribute("ExposureTime").extractDouble();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get exposure time", e);
		}		
		return exposureTime;
	}

	public void setExposureTime(double exposureTime) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("ExposureTime", exposureTime));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set exposure time", e);
		}		
		this.exposureTime = exposureTime;
	}

	public double getExposurePeriod()  throws DeviceException {
		isAvailable();
		try {
			exposurePeriod = dev.read_attribute("ExposurePeriod").extractDouble();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get exposure period", e);
		}		
		return exposurePeriod;
	}

	public void setExposurePeriod(double exposurePeriod) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("ExposurePeriod", exposurePeriod));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set exposure period", e);
		}		
		this.exposurePeriod = exposurePeriod;
	}

	public int getNbFrames() throws DeviceException {
		isAvailable();
		try {
			nbFrames = dev.read_attribute("NbFrames").extractLong();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get nbFrames", e);
		}		
		return nbFrames;
	}

	public void setNbFrames(int nbFrames) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("NbFrames", nbFrames));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set nbFrames", e);
		}		
		this.nbFrames = nbFrames;
	}

	public int getNbExposures() throws DeviceException {
		isAvailable();
		try {
			nbExposures = dev.read_attribute("NbExposures").extractLong();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get nbExposures", e);
		}		
		return nbExposures;
	}

	public void setNbExposures(int nbExposures) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("NbExposures", nbExposures));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set nbExposures", e);
		}		
		this.nbExposures = nbExposures;
	}

	public double getDelayTime() throws DeviceException {
		isAvailable();
		try {
			delayTime = dev.read_attribute("DelayTime").extractDouble();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get delay time", e);
		}		
		return delayTime;
	}

	public void setDelayTime(double delayTime) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("DelayTime", delayTime));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set delay time", e);
		}		
		this.delayTime = delayTime;
	}

	public boolean isShutterEnable() throws DeviceException {
		isAvailable();
		try {
			shutterEnable = dev.read_attribute("ShutterEnable").extractBoolean();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get shutter enable", e);
		}		
		return shutterEnable;
	}

	public void setShutterEnable(boolean shutterEnable) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("ShutterEnable", shutterEnable));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set shutter enable", e);
		}		
		this.shutterEnable = shutterEnable;
	}

	public short getTriggerMode() throws DeviceException {
		isAvailable();
		try {
			triggerMode = dev.read_attribute("TriggerMode").extractShort();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get trigger mode", e);
		}		
		return triggerMode;
	}

	public void setTriggerMode(short triggerMode) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("TriggerMode", triggerMode));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set trigger mode", e);
		}		
		this.triggerMode = triggerMode;
	}

	public String getFileDir() throws DeviceException {
		isAvailable();
		try {
			fileDir = dev.read_attribute("FileDir").extractString();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get file dir", e);
		}		
		return fileDir;
	}

	public void setFileDir(String fileDir) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("FileDir", fileDir));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set file directory", e);
		}		
		this.fileDir = fileDir;
	}

	public String getFilePrefix() throws DeviceException {
		isAvailable();
		try {
			filePrefix = dev.read_attribute("FilePrefix").extractString();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get file prefix", e);
		}		
		return filePrefix;
	}

	public void setFilePrefix(String filePrefix) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("FilePrefix", filePrefix));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set file prefix", e);
		}		
		this.filePrefix = filePrefix;
	}

	public int getFileStartNum() throws DeviceException {
		isAvailable();
		try {
			fileStartNum = dev.read_attribute("FileStartNum").extractLong();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get file start number", e);
		}		
		return fileStartNum;
	}

	public void setFileStartNum(int fileStartNum) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("FileStartNum", fileStartNum));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set file start number", e);
		}		
		this.fileStartNum = fileStartNum;
	}

	public String getFilePostfix() throws DeviceException {
		isAvailable();
		try {
			filePostfix = dev.read_attribute("FilePostfix").extractString();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get file postfix", e);
		}		
		return filePostfix;
	}

	public void setFilePostfix(String filePostfix) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("FilePostfix", filePostfix));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set file postfix", e);
		}		
		this.filePostfix = filePostfix;
	}

	public int getThreshold() throws DeviceException {
		isAvailable();
		try {
			threshold = dev.read_attribute("Threshold").extractLong();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get threshold", e);
		}		
		return threshold;
	}

	public void setThreshold(int threshold) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("Threshold", threshold));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set threshold", e);
		}		
		this.threshold = threshold;
	}

	public short getGain() throws DeviceException {
		isAvailable();
		try {
			gain = dev.read_attribute("Gain").extractShort();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get gain", e);
		}		
		return gain;
	}

	public void setGain(short gain) throws DeviceException {
		isAvailable();
		try {
			dev.write_attribute(new DeviceAttribute("Gain", gain));
		} catch (DevFailed e) {
			throw new DeviceException("failed to set gain", e);
		}		
		this.gain = gain;
	}

	public String getLastImageTaken() throws DeviceException {
		isAvailable();
		try {
			lastImageTaken = dev.read_attribute("LastImageTaken").extractString();
		} catch (DevFailed e) {
			throw new DeviceException("failed to get last image taken", e);
		}		
		return lastImageTaken;
	}

	public void init() throws DeviceException {
		isAvailable();
		try {
			dev.command_inout("Init");
		} catch (DevFailed e) {
			throw new DeviceException("failed to Init", e);
		}		
	}

	@Override
	public void stop() throws DeviceException {
		isAvailable();
		try {
			dev.command_inout("StopAcq");
		} catch (DevFailed e) {
			throw new DeviceException("failed to stop", e);
		}
		
	}
		
	public void reset() throws DeviceException
	{
		isAvailable();
		try {
			dev.command_inout("Reset");
		} catch (DevFailed e) {
			throw new DeviceException("failed to reset", e);
		}
	}

	@Override
	public void collectData() throws DeviceException {
		isAvailable();
		try {
			dev.command_inout("StartStandardAcq");
		} catch (DevFailed e) {
			throw new DeviceException("failed to start", e);
		}		
		
	}

	@Override
	public boolean createsOwnFiles() throws DeviceException {
		return createsOwnFile;
	}

	@Override
	public String getDetectorType() throws DeviceException {
		return detectorType;
	}
	
	@Override
	public String getDescription() throws DeviceException {
		return detectorDescription;
	}

	@Override
	public String getDetectorID() throws DeviceException {
		return detectorID;
	}

	public String getDetectorDescription() {
		return detectorDescription;
	}

	public void setDetectorDescription(String detectorDescription) {
		this.detectorDescription = detectorDescription;
	}

	public void setDetectorType(String detectorType) {
		this.detectorType = detectorType;
	}

	public void setDetectorID(String detectorID) {
		this.detectorID = detectorID;
	}

	@Override
	public int getStatus() throws DeviceException {
		int state = DevState._FAULT;
		isAvailable();
		try {
			state = dev.state().value();
		} catch (DevFailed e) {
			throw new DeviceException("failed to reset", e);
		}
		return state;
	}

	@Override
	public Object readout() throws DeviceException {
//		String path = PathConstructor.createFromTemplate(dirTemplate);
//		String filename = path + "/" + getName() + "-" + tracker + "." + SUFFIX;

//		String filename = getLastImageTaken();
		String filename = "/home/grm84/users/data/LidiSi11_540h2_00000.edf";
		File file = new File(filename);

		while (!file.canRead()) {
			logger.info("waiting for " + filename + " to appear.");
			try {
				Thread.sleep(250);
			} catch (InterruptedException e) {
				throw new DeviceException("interrupted in readout waiting for file", e);
			}
		}

		ScanFileHolder sfh = new ScanFileHolder();
		try {
			sfh.load(new PilatusEdfLoader(filename));
		} catch (ScanFileHolderException e) {
			throw new DeviceException("problem getting data from disk", e);
		}
		double[] data = sfh.getAxis(0).getBuffer();
		// file.delete();
		return data;
	}

//	@Override
//	public Object readout() throws DeviceException {
		// creates its own files so just return the last image taken
//		return getLastImageTaken();
//	}

	@Override
	public void setAttribute(String attributeName, Object value) throws DeviceException {
		if ("ExposureTime".equals(attributeName)) {
			setExposureTime((Double)value);
		} else if ("ExposurePeriod".equals(attributeName)) {
			setExposurePeriod((Double) value);
		} else if ("NbFrames".equals(attributeName)) {
			setNbFrames((Integer) value);
		} else if ("NbExposures".equals(attributeName)) {
			setNbExposures((Integer) value);
		} else if ("DelayTime".equals(attributeName)) {
			setDelayTime((Integer) value);
		} else if ("ShutterEnable".equals(attributeName)) {
			setShutterEnable((Boolean) value);
		} else if ("TriggerMode".equals(attributeName)) {
			setTriggerMode((Short) value);
		} else if ("FileDir".equals(attributeName)) {
			setFileDir((String) value);
		} else if ("FilePrefix".equals(attributeName)) {
			setFilePrefix((String) value);
		} else if ("FileStartNum".equals(attributeName)) {
			setFileStartNum((Integer) value);
		} else if ("FilePostfix".equals(attributeName)) {
			setFilePostfix((String) value);
		} else if ("Threshold".equals(attributeName)) {
			setThreshold((Integer) value);
		} else if ("Gain".equals(attributeName)) {
			setGain((Short) value);
		}
	}
	
	@Override
	public Object getAttribute(String attributeName) throws DeviceException {
		Object object = null;
		if ("ExposureTime".equals(attributeName)) {
			object = getExposureTime();
		} else if ("ExposurePeriod".equals(attributeName)) {
			object = getExposurePeriod();
		} else if ("NbFrames".equals(attributeName)) {
			object = getNbFrames();
		} else if ("NbExposures".equals(attributeName)) {
			object = getNbExposures();
		} else if ("DelayTime".equals(attributeName)) {
			object = getDelayTime();
		} else if ("ShutterEnable".equals(attributeName)) {
			object = isShutterEnable();
		} else if ("TriggerMode".equals(attributeName)) {
			object = getTriggerMode();
		} else if ("FileDir".equals(attributeName)) {
			object = getFileDir();
		} else if ("FilePrefix".equals(attributeName)) {
			object = getFilePrefix();
		} else if ("FileStartNum".equals(attributeName)) {
			object = getFileStartNum();
		} else if ("FilePostfix".equals(attributeName)) {
			object = getFilePostfix();
		} else if ("LastImageTaken".equals(attributeName)) {
			object = getLastImageTaken();
		} else if ("Threshold".equals(attributeName)) {
			object = getThreshold();
		} else if ("Gain".equals(attributeName)) {
			object = getGain();
		}
		return object;
	}

	private void isAvailable() throws DeviceException {
		try {
			// Is the device still connected or just started or at fault
			if (dev.state().value() == DevState._FAULT)
				reset();
			setConfigured(true);
		} catch (DevFailed e) {
			// device has lost connection
			setConfigured(false);
			throw new DeviceException("Tango device server " + dev.get_name() + " failed");
		} catch (Exception e) {
			throw new DeviceException("Tango device server stuffed");			
		}
	}
}