/*-
 * Copyright © 2009 Diamond Light Source Ltd.
 *
 * This file is part of GDA.
 *
 * GDA is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License version 3 as published by the Free
 * Software Foundation.
 *
 * GDA is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with GDA. If not, see <http://www.gnu.org/licenses/>.
 */

package gda.device.detector.ncddetector.subdetector;

import java.util.ArrayList;
import java.util.List;

import gda.device.DeviceBase;
import gda.device.DeviceException;
import gda.device.detector.pilatus.TangoPilatusDetector;
import gda.device.detector.ncddetector.subdetector.INcdSubDetector;
import gda.device.detector.DataDimension;
import gda.factory.FactoryException;

public class TangoPilatusSubDetector extends DeviceBase implements INcdSubDetector {

	public static final String[] detectorTypes = new String[]{"SAXS", "WAXS", "CALIB", "TIMES", "SYS", "REDUCTION"};

	private TangoPilatusDetector pilatus;
	private String detectorType;
	private String filePrefix;
	private String filePostfix = "edf";
	private String directory;
	
	public TangoPilatusDetector getTangoPilatusDetector() {
		return pilatus;
	}

	public void setTangoPilatusDetector(TangoPilatusDetector pilatus) {
		this.pilatus = pilatus;
	}

	@Override
	public void configure() throws FactoryException {
		if (pilatus != null) {
			pilatus.reconfigure();
		}
	}

	@Override
	public void reconfigure() throws FactoryException {
		configured = false;
//		logger.debug("NcdDetector reconfiguring " + getName());
		configure();
	}

	@Override
	public void clear() throws DeviceException {
		// Not possible
	}

	@Override
	public int[] getDataDimensions() throws DeviceException {
		int[] dims = new int[2];
		dims[0] = pilatus.getWidth();
		dims[1] = pilatus.getHeight();
		return dims;
	}

	@Override
	public String getDetectorType() throws DeviceException {
		return detectorType;
	}

	@Override
	public int getMemorySize() throws DeviceException {
		return 0;
	}

	@Override
	public List<DataDimension> getSupportedDimensions() throws DeviceException {
		ArrayList<DataDimension> dataDims = new ArrayList<DataDimension>();
		dataDims.add(new DataDimension(pilatus.getWidth(), pilatus.getHeight()));
		return dataDims;
	}

	@Override
	public double[] output(int frames) throws DeviceException {
		return (double[]) pilatus.readout();
	}

	@Override
	public double[] read(int x, int y, int t, int dx, int dy, int dt) throws DeviceException {
		double[] data = new double[dx * dy * dt];
		if (t > 0 && !pilatus.isBusy()) {
			double[] rawData = (double[]) pilatus.readout();
			int width = pilatus.getWidth();
			int l = 0;
			for (int j = y; j < y + dy; j++) {
				for (int k = x; k < x + dx; k++) {
					data[l] = rawData[j * width + k];
					l++;
				}
			}
		}
		return data;
	}

	public double[] read(int frame) throws DeviceException {
		double[] rawData = null;
		if (frame > 0 && !pilatus.isBusy()) {
			rawData = (double[]) pilatus.readout();
		}
		return rawData;
	}

	@Override
	public void restart() throws DeviceException {
		// not possible
	}

	@Override
	public void setDataDimensions(int[] detectorSize) throws DeviceException {
		// not possible		
	}

	/**
	 * @param detectorType the detectorType to set
	 */
	public void setDetectorType(String detectorType) {
		for (String type : detectorTypes) {
			if (type.equalsIgnoreCase(detectorType)) {
				this.detectorType = detectorType;
				return;
			}
		}
		throw new IllegalArgumentException("attempt to set unrecognised detector type "+detectorType);
	}
	
	@Override
	public void start() throws DeviceException {
		pilatus.setFilePrefix(filePrefix);
		pilatus.setFilePostfix(filePostfix);
		pilatus.setFileDir(directory);
		pilatus.setFileStartNum(0);
		pilatus.collectData();
	}

	@Override
	public void stop() throws DeviceException {
		pilatus.stop();
	}

	@Override
	public void setAttribute(String attributeName, Object value) {
		if ("FilePrefix".equals(attributeName)) {
			filePrefix = (String) value;
		} else if ("Directory".equals(attributeName)) {
			directory = (String) value;
		}
	}
}