package gda.device.detector.ccd;

import fr.esrf.Tango.DevFailed;
import fr.esrf.Tango.DevState;
import fr.esrf.TangoApi.DeviceAttribute;
import gda.device.Detector;
import gda.device.detector.DetectorBase;
import gda.device.DeviceException;
import gda.device.TangoDeviceProxy;
import gda.factory.FactoryException;

import java.util.ArrayList;

public class TangoCcd extends DetectorBase implements Detector {

	private TangoDeviceProxy deviceProxy;
//	private CcdCallBack ccdCallBack;
	private boolean busy = false;

	@Override
	public void configure() throws FactoryException {
		try {
			deviceProxy.isAvailable();
		} catch (DeviceException e) {
			throw new FactoryException("Error connnecting to " + deviceProxy.get_name() + " " + e.getMessage());
		}
		setConfigured(true);
	}

	/**
	 * @return Returns the Tango device proxy.
	 */
	public TangoDeviceProxy getTangoDeviceProxy() {
		return deviceProxy;
	}

	/**
	 * @param deviceProxy The Tango device proxy to set.
	 */
	public void setTangoDeviceProxy(TangoDeviceProxy deviceProxy) {
		this.deviceProxy = deviceProxy;
	}

	@Override
	public void collectData() throws DeviceException {
		deviceProxy.isAvailable();
		try {
			busy= true;
			deviceProxy.command_inout("Start");
		} catch (DevFailed e) {
			throw new DeviceException("failed to start data collection");
		}
	}

	@Override
	public boolean createsOwnFiles() throws DeviceException {
		return false;
	}

	@Override
	public void endCollection() throws DeviceException {
		deviceProxy.isAvailable();
		try {
			deviceProxy.command_inout("Stop");
		} catch (DevFailed e) {
			throw new DeviceException("failed to stop data collection");
		}
	}

	@Override
	public String getDescription() throws DeviceException {
		return "ccd";
	}

	@Override
	public String getDetectorID() throws DeviceException {
		return "Abstract";
	}

	@Override
	public String getDetectorType() throws DeviceException {
		return "TangoCCD";
	}

	@Override
	public int getStatus() throws DeviceException {
		int status;
//		deviceProxy.isAvailable();
		try {
			DevState state = deviceProxy.state();
			switch (state.value()) {
			case DevState._ON:
				status = Detector.BUSY;
				break;
			case DevState._FAULT:
				status = Detector.FAULT;
				break;
			case DevState._OFF:
			default:
				status = Detector.IDLE;
				break;
			}
		} catch (DevFailed e) {
			throw new DeviceException("failed to get ccd state", e);
		}
		return status;
	}

	@Override
	public int[] getDataDimensions() throws DeviceException {
		int dims[] = {1,1};
		try {
			dims[0] = deviceProxy.read_attribute("Width").extractLong();
			dims[1] = deviceProxy.read_attribute("Height").extractLong();
			System.out.println("TangoCcd.getDataDimensions(): width*height "+dims[0]+" height "+dims[1]);
			
		} catch (DevFailed e) {
			e.printStackTrace();
		}
		return dims;
		
	}

	@Override
	public Object readout() throws DeviceException {
		byte[] b = null;
		int[] d = null;
		deviceProxy.isAvailable();
		try {
			b = deviceProxy.read_attribute("Image").extractCharArray();
			System.out.println("TangoCcd.readout(): read "+b.length+" bytes");
			d = new int[b.length/2];
			for (int i=0; i<b.length/2; i++) {
				d[i] = b[2*i+1]*256 + b[2*i];
			}
		} catch (DevFailed e) {
			e.printStackTrace();
		}
		return d;
	}
	
	@Override
	public void setCollectionTime(double time) {
		try {
			deviceProxy.isAvailable();
			deviceProxy.write_attribute(new DeviceAttribute("Exposure", time));
		} catch (DevFailed e) {
			e.printStackTrace();
		}
		catch (DeviceException e) {
			e.printStackTrace();
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public void setAttribute(String attributeName, Object value) throws DeviceException {
		deviceProxy.isAvailable();
		if ("Exposure".equalsIgnoreCase(attributeName)) {
			try {
				deviceProxy.write_attribute(new DeviceAttribute(attributeName, (Double) value));
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		if ("Trigger".equals(attributeName)) {
			try {
				deviceProxy.write_attribute(new DeviceAttribute(attributeName, (Short) value));
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("Frames".equals(attributeName)) {
			try {
				deviceProxy.write_attribute(new DeviceAttribute(attributeName, (Integer) value));
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("FileFormat".equals(attributeName)) {
			try {
				deviceProxy.write_attribute(new DeviceAttribute(attributeName, (String) value));
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("JpegQuality".equals(attributeName)) {
			try {
				deviceProxy.write_attribute(new DeviceAttribute(attributeName, (Short) value));
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("JpegCompression".equals(attributeName)) {
			try {
				deviceProxy.write_attribute(new DeviceAttribute(attributeName, (Boolean) value));
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("Roi".equals(attributeName)) {
			try {
				ArrayList<Integer> s = (ArrayList<Integer>) value;
				int[] ints = new int[s.size()];
				for (int i=0; i<s.size(); i++)
					ints[i] = s.get(i);
				DeviceAttribute devAttr = new DeviceAttribute(attributeName);
				devAttr.insert(ints);
				deviceProxy.write_attribute(devAttr);
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("Binning".equals(attributeName)) {
			try {
				ArrayList<Short> s = (ArrayList<Short>) value;
				short[] shorts = new short[s.size()];
				for (int i=0; i<s.size(); i++)
					shorts[i] = s.get(i);
				DeviceAttribute devAttr = new DeviceAttribute(attributeName);
				devAttr.insert(shorts);
				deviceProxy.write_attribute(devAttr);
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
		else if ("FileParams".equals(attributeName)) {
			try {
				ArrayList<String> s = (ArrayList<String>) value;
				String[] strings = null;
				strings = s.toArray(strings);
				DeviceAttribute devAttr = new DeviceAttribute(attributeName);
				devAttr.insert(strings);
				deviceProxy.write_attribute(devAttr);
			} catch (DevFailed e) {
				e.printStackTrace();
			}
			
		}
	}
	
	@Override
	public Object getAttribute(String attributeName) throws DeviceException {
		deviceProxy.isAvailable();
		if ("Exposure".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractDouble();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Trigger".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractShort();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Width".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractLong();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Height".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractLong();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Depth".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractShort();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Frames".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractLong();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("FileFormat".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractString();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("ImageFormat".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractString();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("ImageCounter".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractLong();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("JpegQuality".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractShort();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("JpegCompression".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractBoolean();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Roi".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractLongArray();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("Binning".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractShortArray();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("FileParams".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractStringArray();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		else if ("JpegImage".equals(attributeName)) {
			try {
				return deviceProxy.read_attribute(attributeName).extractUCharArray();
			} catch (DevFailed e) {
				e.printStackTrace();
			}
		}
		return null;
	}
	
	@Override
	public String toString() {
		return "SimCcd.toString()";
	}
}
