/*-
 * Copyright © 2009 Diamond Light Source Ltd.
 *
 * This file is part of GDA.
 *
 * GDA is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License version 3 as published by the Free
 * Software Foundation.
 *
 * GDA is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with GDA. If not, see <http://www.gnu.org/licenses/>.
 */

package gda.device;

import fr.esrf.Tango.DevFailed;
import fr.esrf.Tango.DevState;
import fr.esrf.TangoApi.AttributeInfo;
import fr.esrf.TangoApi.DbDatum;
import fr.esrf.TangoApi.DeviceAttribute;
import fr.esrf.TangoApi.DeviceData;
import fr.esrf.TangoDs.TangoConst;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DummyTangoDeviceProxy extends TangoDeviceProxy {
	private static final Logger logger = LoggerFactory.getLogger(DummyTangoDeviceProxy.class);
	private String deviceName;
	private boolean usedb = false;
	private HashMap<String, DeviceAttribute> attributeList = new HashMap<String, DeviceAttribute>();
	private HashMap<String, DbDatum> propertyList = new HashMap<String, DbDatum>();
	private DevState state;
	private String status;
	private static boolean fail = false;
		
	public DummyTangoDeviceProxy(String deviceName) {
		super();
		this.deviceName = deviceName;
		usedb = (deviceName.contains("#dbase=no")) ? false : true;
		setState(DevState.ON);
	}

	@Override
	public String get_name() {
		return deviceName;
	}

	@Override
	public DeviceData command_inout(String cmd) throws DevFailed {
		if (fail) {
			fail = false;
			throw new DevFailed ("Command " + cmd + " failed", null);
		}
		logger.info("Command " + cmd + " executed");
		return null;
	}
	
	@Override
	public DevState state() {
		return state;		
	}
	
	@Override
	public String status() {
		return status;
	}
	
	@Override
	public void write_attribute(DeviceAttribute attr) throws DevFailed {
		String key = attr.getName();
		if (attributeList.containsKey(key)) {
			attributeList.remove(key);
		}
		attributeList.put(key, attr);
		logger.debug("Attribute " + key + " written");
	}
	
	@Override
	public DeviceAttribute read_attribute(String attributeName) {
		DeviceAttribute devAttr = null;
		if (attributeList.containsKey(attributeName)) {
			devAttr = attributeList.get(attributeName);
		}
		return devAttr;
	}
	@Override
	public String[] get_attribute_list() throws DevFailed {
		Set<String> set = attributeList.keySet();
		return set.toArray(new String[1]);
	}

	@Override
	public AttributeInfo get_attribute_info(String attributeName) throws DevFailed {
		DeviceAttribute devAttr = null;
		AttributeInfo attrInfo = null;
		if (attributeList.containsKey(attributeName)) {
			devAttr = attributeList.get(attributeName);	
			attrInfo = new AttributeInfo(attributeName,
                    null,
                    devAttr.getDataFormat(),
                    devAttr.getType(),
                    devAttr.getDimX(),
                    devAttr.getDimY(),
                    "description",
                    "label",
                    "unit",
                    "standard_unit",
                    "display_unit",
                    "format",
                    "min_value",
                    "max_value",
                    "min_alarm",
                    "max_alarm",
                    "writable_attr_name",
                    null);
		}
		return attrInfo;
	}

	@Override
	public boolean use_db() {
		return usedb;
	}
	
	@Override
	public DbDatum get_property(String propertyName) throws DevFailed {
		DbDatum datum = null;
		if (usedb) {
			if (propertyList.containsKey(propertyName)) {
				datum = propertyList.get(propertyName);			
			}
		}
		return datum; // TODO
	}

	@Override
	public void put_property(DbDatum property) throws DevFailed {
		String key = property.name;
		if (propertyList.containsKey(key)) {
			propertyList.remove(key);
		}
		propertyList.put(key, property);
		logger.debug("Property " + key + " written");
	}

	@Override
	public void isAvailable() throws DeviceException {
	}

	// Methods used to setup for junit testing
	
	public void setState(DevState state) {
		this.state = state;
		status = TangoConst.Tango_DevStateName[state.value()];
	}
	
	public void setFail() {
		fail = true;
	}
	

	public ArrayList<DummyDeviceAttribute> getAttributeList() {
		ArrayList<DummyDeviceAttribute> holderList = new ArrayList<DummyDeviceAttribute>();
		for (DeviceAttribute attribute : attributeList.values()) {
			holderList.add(new DummyDeviceAttribute(attribute));
		}
		return holderList;
	}

	public void setAttributeList(ArrayList<DummyDeviceAttribute> attributes) throws DevFailed{
		for (DummyDeviceAttribute attribute : attributes) {
			write_attribute(attribute.getDeviceAttribute());
		}
	}
}

