static const char *RcsId = "$Header: /cvsroot/tango-cs/tango/bindings/python/src/device_proxy.cpp,v 1.14 2006/06/28 09:06:13 taurel Exp $";
//+=============================================================================
//
// file :         device_proxy.cpp
//
// description :  C++ source for the light DeviceProxy class
//                intented to be exported to python world.
//
// project :      Python Binding
//
// $Author: taurel $
//
// $Revision: 1.14 $
//
// $Log: device_proxy.cpp,v $
// Revision 1.14  2006/06/28 09:06:13  taurel
// - Fix bug in event management. Noe use the deviceProxyUtils instance of the
// binding DeviceProxy instead of creating a new one for each received event
//
// Revision 1.13  2006/04/13 13:40:03  taurel
// Several changes:
// - Add wrapping of the new set/get_transparency_reconnection AttributeProxy class methods
// - Add wrapping of the new AttributeProxy class ctor (with DeviceProxy)
// - Fix bug: Now all Pthon exceptions inherit from PyTango.DevFailed
// - Exception name from WrongSynTax to WrongNameSyntax and Communication to CommunicationFailed to be coherent with C++
// - Change the way attribute data type are managed in AttributeProxy class to speed up read() and write() methods
// - Change the R/W type for some elements of the AttributeInfo class. All of them were read only
//
// Revision 1.12  2006/03/27 08:50:47  taurel
// - Add the Deviceproxy.alisa() method
// - Fix a 32/64 bits problem in the wrapping of the "at" method  (for DS)
//
// Revision 1.11  2006/03/22 09:08:37  taurel
// - Commit after some changes from Vincenzo in device_proxy files
//
// Revision 1.10  2006/01/25 15:49:52  taurel
// - Add the DEV_UCHAR case in the DeviceAttribute to a Python string. It is not
// yet completely implemented. Will come soon
//
// Revision 1.9  2005/10/31 09:03:01  dusdan
// fixed bug with python2.4 and state attribute subscribe_event;
// added new Database object constructor
//
// Revision 1.8  2005/07/19 12:05:51  dusdan
// by Vincenzo Forchi`
//
// fixed bug in error handling in push_event (segfault)
// unified format of deverrorlist (tuple of dictionaries)
//
// Revision 1.7  2005/06/02 17:10:54  ounsy
// First introduction of group related methods
//
// Revision 1.6  2005/04/14 08:33:21  dusdan
// Vincenzo Forchi`:
// - fields device and event in EventData are filled in push_event
// - unsubscribe_event now decrements reference count for python callback
//
// Revision 1.5  2005/04/08 11:49:23  dusdan
// Vincenzo Forchi`
// - now all tango types _should_ be supported
// - modified the get_scalar_value in templates.cpp: now it tries to get a suitable c++ type from the string representation of the python object (if the boost::python::extract fails)
// - I grouped all the utility functions in a file called conversion_utils.cpp to avoid duplicated code
// - attribute_proxy_utils.cpp and device_proxy_utils.cpp aren't needed anymore
// - added a test_att_proxy.py to test the new object
// - added tango_client.py: a small command line client
//
// Revision 1.4  2005/03/29 11:41:41  taurel
// - Add try/catch block in the subscribe_event and unsubscribe_event call
//
// Revision 1.3  2005/03/24 13:19:42  taurel
// - Added Vincenzo changes for the Tango V5 port
// - Fix bug on Linux in the client_request function (default  argument)
//
// Revision 1.2  2004/01/27 09:59:57  ounsy
// Added asynchronous related methods
//
// Revision 1.1  2003/10/23 13:50:55  ounsy
//
//
// Added Python binding for TANGO
//
//
// copyleft :     Synchrotron SOLEIL
//			L'Orme des Merisiers, Saint-Aubin BP 48
//			91192 GIF SUR YVETTE Cedex
//
//-=============================================================================

#include <device_proxy.h>
#include <templates.cpp>
#include <pystate.h>
#include "conversion_utils.h"
#include <boost/python/errors.hpp>

#include <iostream>
using namespace std;


void
PythonCommandCallback::cmd_ended(Tango::CmdDoneEvent *cmd_event)
{
	CmdDoneEvent py_cmd_event;
	py_cmd_event.device = device->getself();

	PyThreadState *tstate = 0;
	tstate = PyThreadState_New(interp);
	PyEval_AcquireThread(tstate);
	
	py_cmd_event.cmd_name = cmd_event->cmd_name;
	if (cmd_event->err) {
		py_cmd_event.err = 1;
		const Tango::DevErrorList &dev_error_list = cmd_event->errors;
		/*CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			py_cmd_event.errors.append( dev_error_list[i] );
		}*/
		py_cmd_event.errors = Utils::translate_exception_value(dev_error_list);
	} else {
		py_cmd_event.err = 0;
		py_cmd_event.argout = Utils::translate_from_device_data(cmd_event->argout,argout_type);
	}
	boost::python::call_method<void>(python_callback.ptr(),"cmd_ended", py_cmd_event); 
	PyThreadState_Clear(tstate);
	PyThreadState_DeleteCurrent();
}

void  
PythonAttrReadCallback::attr_read(Tango::AttrReadEvent *read_event)
{
	AttrReadEvent py_read_event;

	PyThreadState *tstate = 0;
	tstate = PyThreadState_New(interp);
	PyEval_AcquireThread(tstate);
	
	py_read_event.attr_names = get_list_from_vector<std::string>(read_event->attr_names);
	if (read_event->err) {
		py_read_event.err = 1;
		const Tango::DevErrorList &dev_error_list = read_event->errors;
		/*CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			py_read_event.errors.append( dev_error_list[i] );
		}*/
		py_read_event.errors = Utils::translate_exception_value(dev_error_list);
	} else {
		py_read_event.err = 0;
		DeviceProxyUtils dev_utils(read_event->device);
		std::vector<Tango::DeviceAttribute> *dev_attr = read_event->argout;
		boost::python::list pyDevAttrList;
		std::vector<Tango::DeviceAttribute>::iterator i;
		long arg_type;
		Tango::AttrDataFormat arg_format;
		for (i = dev_attr->begin() ; i < dev_attr->end() ; i++ )
		{
			arg_type = (dev_utils.get_attribute_type( i->get_name() )).first;
			arg_format = (dev_utils.get_attribute_type( i->get_name() )).second;
			pyDevAttrList.append( Utils::translate_from_device_attribute( *i , arg_type , arg_format ) );
		}
		py_read_event.argout = pyDevAttrList;
	}
	boost::python::call_method<void>(python_callback.ptr(),"attr_read",py_read_event); 
	PyThreadState_Clear(tstate);
	PyThreadState_DeleteCurrent();
}

void  
PythonAttrWriteCallback::attr_write(Tango::AttrWrittenEvent *write_event)
{	
	AttrWrittenEvent py_write_event;
	py_write_event.device = device->getself();

	PyThreadState *tstate = 0;
	tstate = PyThreadState_New(interp);
	PyEval_AcquireThread(tstate);
	
	py_write_event.attr_names = get_list_from_vector<std::string>(write_event->attr_names);
	if (write_event->err) {
		py_write_event.err = 1;
		/* VF */ 
		/* commented out to compile in Tango V5: check */
		/*const Tango::DevErrorList &dev_error_list = write_event->errors;
		CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			py_write_event.errors.append( dev_error_list[i] );
		}*/
	} else {
		py_write_event.err = 0;
	}
	boost::python::call_method<void>(python_callback.ptr(),"attr_write",py_write_event); 
	PyThreadState_Clear(tstate);
	PyThreadState_DeleteCurrent();
}

/* VF 21/03/2005 */	
void  
PythonPushEventCallback::push_event(Tango::EventData *event)
{
	EventData py_event;
	DeviceProxyUtils *dev_utils = device->get_dev_utils();

	py_event.device = device->getself();
	py_event.event = event->event;
	py_event.attr_name = event->attr_name;

	/* VF
	PyThreadState *tstate = 0;
	tstate = PyThreadState_New(interp);
	PyEval_AcquireThread(tstate);*/

	if (tstate == 0)
		tstate = PyThreadState_New(interp);
	PyEval_AcquireLock();
	PyThreadState_Swap(tstate);
		
	if (event->err) {
		py_event.err = 1;
		const Tango::DevErrorList &dev_error_list = event->errors;
		/*CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			py_event.errors.append( dev_error_list[i] );
		}*/
		/*boost::python::list exc_value;
		CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			boost::python::dict err;
			err["reason"] = dev_error_list[i].reason.in();
			std::string severity;
			switch (dev_error_list[i].severity) {
				case Tango::WARN :
					severity = "WARNING";
					break;
				case Tango::ERR :
					severity = "ERROR";
					break;
				case Tango::PANIC :
					severity = "PANIC";
					break;
			}
			err["severity"] = severity;
			err["desc"] = dev_error_list[i].desc.in();
			err["origin"] = dev_error_list[i].origin.in();
			exc_value.append(err);
		}
		py_event.errors = boost::python::tuple(exc_value);	*/
		py_event.errors = Utils::translate_exception_value(dev_error_list);
	} else {
		py_event.err = 0;
		Tango::DeviceAttribute *dati = event->attr_value;
		int pos = event->attr_name.find_last_of('/');
		std::string name = event->attr_name.substr(pos + 1, event->attr_name.size() - pos - 1);
		long arg_type = (dev_utils->get_attribute_type(name)).first;
		Tango::AttrDataFormat arg_format = (dev_utils->get_attribute_type(name)).second;
		py_event.attr_value = (boost::python::object)Utils::translate_from_device_attribute((*dati), arg_type, arg_format);
	}
	
	try {
		boost::python::call_method<void>(python_callback.ptr(), "push_event", py_event); 
	} catch(boost::python::error_already_set) {
		std::cout << "error_already_set" << std::endl;
	}

	PyThreadState_Swap(NULL);
	PyEval_ReleaseLock();
		
/*	PyThreadState_Clear(tstate);
	PyThreadState_DeleteCurrent();*/
}
/* */

DeviceProxyUtils::DeviceProxyUtils(Tango::DeviceProxy *dev) : deviceProxy(dev)
{
}

std::pair<long,long>
DeviceProxyUtils::get_command_args(std::string name)
{
    if ( command_args.count(name) == 0 ) {
	Tango::CommandInfo commandInfo = deviceProxy->command_query(name);
        command_args[name] = pair<long,long>( commandInfo.in_type , commandInfo.out_type );	
    }
    return command_args[name]; 
}

std::pair<long,Tango::AttrDataFormat>
DeviceProxyUtils::get_attribute_type(std::string name)
{
    transform(name.begin(),name.end(),name.begin(),::tolower);
    if ( attribute_types.count(name) == 0 ) {
	Tango::AttributeInfo attributeInfo = deviceProxy->attribute_query(name);
        attribute_types[name] = pair<long,Tango::AttrDataFormat>( attributeInfo.data_type , attributeInfo.data_format );	
    }
    return attribute_types[name];
}

DeviceProxy::DeviceProxy(std::string name) : deviceName(name), was_allocated(true)
{
  try { 
	deviceProxy = new Tango::DeviceProxy(deviceName);
	devname = (boost::python::object) deviceProxy->dev_name();
	deviceProxyUtils = new DeviceProxyUtils(deviceProxy);
  } catch(const Tango::WrongNameSyntax &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}

DeviceProxy::DeviceProxy(Tango::DeviceProxy *dev) : deviceProxy(dev), was_allocated(false)
{
  try {
  	deviceName = deviceProxy->dev_name(); 
	devname = (boost::python::object) deviceProxy->dev_name();
	deviceProxyUtils = new DeviceProxyUtils(deviceProxy);
  } catch(const Tango::WrongNameSyntax &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


DeviceProxy::~DeviceProxy()
{
	if (was_allocated) 
    	delete deviceProxy;
	delete deviceProxyUtils;
}

boost::python::object 
DeviceProxy::getattr(std::string command)
{	
	return boost::python::object ( ClientRequestHandler(*this,command) ); 
}

boost::python::object 
DeviceProxy::getself()
{
//	    boost::shared_ptr<DeviceProxy>  self =  shared_from_this();
//		assert(self != 0);
		//return boost::python::object(self);
//		return (boost::python::object) self;
	return devname;
//	return (boost::python::object) deviceProxy->dev_name();
}

boost::python::object 
DeviceProxy::state()
{ 
  try { 
    return (boost::python::object)(deviceProxy->state()); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } 
} 


std::string 
DeviceProxy::status()
{ 
  try { 
    return deviceProxy->status(); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } 
}

int 
DeviceProxy::ping()
{ 
  try { 
    return deviceProxy->ping(); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } 
}     

void 
DeviceProxy::set_timeout_millis(int timeout)
{ 
    deviceProxy->set_timeout_millis(timeout); 
}     

int 
DeviceProxy::get_timeout_millis()
{ 
    return deviceProxy->get_timeout_millis(); 
}     

int 
DeviceProxy::get_idl_version()
{ 
    return deviceProxy->get_idl_version(); 
}     

void 
DeviceProxy::set_source(Tango::DevSource source)
{ 
    deviceProxy->set_source(source); 
}     

boost::python::object 
DeviceProxy::get_source()
{ 
    return (boost::python::object)(deviceProxy->get_source()); 
}     

boost::python::tuple 
DeviceProxy::black_box(int n)
{ 
  try { 
    std::vector<string> *comHist = deviceProxy->black_box(n);
    return Utils::translate_black_box(comHist);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

std::string 
DeviceProxy::name()
{ 
  try { 
    return deviceProxy->name(); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  }
}

std::string 
DeviceProxy::alias()
{ 
  try { 
    return deviceProxy->alias(); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}

std::string 
DeviceProxy::adm_name()
{ 
  try { 
    return deviceProxy->adm_name(); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  }
}

std::string 
DeviceProxy::dev_name()
{ 
    return deviceProxy->dev_name(); 
}

std::string 
DeviceProxy::description()
{ 
  try { 
    return deviceProxy->description(); 
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  }
}

boost::python::object
DeviceProxy::info()
{ 
  try {
      return (boost::python::object)(deviceProxy->info());
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

boost::python::object 
DeviceProxy::import_info()
{ 
  try {
     return (boost::python::object)(deviceProxy->import_info()); 
  } catch(const Tango::NonDbDevice &e) {
    throw e;
  } 
} 

boost::python::object 
DeviceProxy::command_query(std::string command)
{ 
  try { 
    return (boost::python::object)(deviceProxy->command_query(command));
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

boost::python::list 
DeviceProxy::command_list_query()
{ 
  try { 
    Tango::CommandInfoList *commandInfoList = deviceProxy->command_list_query();
    boost::python::list py_list = get_list_from_vector<Tango::CommandInfo>(*commandInfoList);
    delete commandInfoList;
    return py_list;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

boost::python::object 
DeviceProxy::command_inout(std::string name, boost::python::object pyData)
{
  try { 
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(name);
     Tango::DeviceData tangoOutData;
     if (pyData.ptr() == Py_None) {
         tangoOutData = deviceProxy->command_inout(name);
     } else {
	Tango::DeviceData tangoInData;
	Utils::translate_to_device_data(tangoInData,pyData,arg_types.first);
	tangoOutData = deviceProxy->command_inout(name,tangoInData);
     }
     boost::python::object result = Utils::translate_from_device_data(tangoOutData,arg_types.second);
     return result;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  }
} 

void 
DeviceProxy::command_inout_asynch(std::string name,boost::python::object callback)
{
  try {
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(name);
     PythonCommandCallback python_callback(this,callback,arg_types.second);
	 deviceProxy->command_inout_asynch(name,python_callback);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
} 

void 
DeviceProxy::command_inout_asynch(std::string name,boost::python::object pyData, boost::python::object callback)
{
  try {
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(name);
	 Tango::DeviceData tangoInData;
     Utils::translate_to_device_data(tangoInData,pyData,arg_types.first);
     PythonCommandCallback python_callback(this,callback,arg_types.second);
	 deviceProxy->command_inout_asynch(name,tangoInData,python_callback);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}
 

long 
DeviceProxy::command_inout_asynch(std::string name,int forget)
{
  try {
	 long id = deviceProxy->command_inout_asynch(name,forget);
	 id_to_name[id] = name;
	 return id;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}

long 
DeviceProxy::command_inout_asynch(std::string name,boost::python::object pyData,int forget)
{
  try {
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(name);
	 Tango::DeviceData tangoInData;
     Utils::translate_to_device_data(tangoInData,pyData,arg_types.first);
	 return deviceProxy->command_inout_asynch(name,tangoInData,forget);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}
 

boost::python::object 
DeviceProxy::command_inout_reply(long id)
{
  try { 
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(id_to_name[id]);
     Tango::DeviceData tangoOutData;
	 tangoOutData = deviceProxy->command_inout_reply(id);
     boost::python::object result = Utils::translate_from_device_data(tangoOutData,arg_types.second);
     return result;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}


boost::python::object 
DeviceProxy::command_inout_reply(long id,long timeout)
{
  try { 
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(id_to_name[id]);
     Tango::DeviceData tangoOutData;
	 tangoOutData = deviceProxy->command_inout_reply(id,timeout);
     boost::python::object result = Utils::translate_from_device_data(tangoOutData,arg_types.second);
     return result;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}


boost::python::list 
DeviceProxy::command_history(std::string name, int depth)
{
  try { 
     pair<long,long> arg_types = deviceProxyUtils->get_command_args(name);
     std::vector<Tango::DeviceDataHistory> *dev_data_hist = deviceProxy->command_history(name,depth);
     boost::python::list pyDevDataList;
     std::vector<Tango::DeviceDataHistory>::iterator i;
     for (i = dev_data_hist->begin() ; i < dev_data_hist->end() ; i++ )
     {
       pyDevDataList.append(Utils::translate_from_data_history(*i, arg_types.second));
     }
     delete dev_data_hist;
     return pyDevDataList;
  } catch(const Tango::NonSupportedFeature &e) {
    throw e;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  }
} 

boost::python::object 
DeviceProxy::attribute_history(std::string name, int depth)
{
  try { 
     long arg_type = (deviceProxyUtils->get_attribute_type(name)).first;
     Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(name)).second;
     std::vector<Tango::DeviceAttributeHistory> *dev_attr_hist = deviceProxy->attribute_history(name,depth);
     boost::python::list pyDevAttrList;
     std::vector<Tango::DeviceAttributeHistory>::iterator i;
     for (i = dev_attr_hist->begin() ; i < dev_attr_hist->end() ; i++ )
     {
       pyDevAttrList.append(Utils::translate_from_attribute_history( *i , arg_type , arg_format ) );
     }
     delete dev_attr_hist;
     return pyDevAttrList;
  } catch(const Tango::NonSupportedFeature &e) {
    throw e;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  }
} 

boost::python::object 
DeviceProxy::attribute_query(std::string attribute)
{ 
  try { 
     Tango::AttributeInfo attributeInfo = deviceProxy->attribute_query(attribute);
     return boost::python::object ( Utils::translate_attribute_info(&attributeInfo) );
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

boost::python::list 
DeviceProxy::attribute_list_query()
{ 
  try { 
     Tango::AttributeInfoList *attributeInfoList = deviceProxy->attribute_list_query();
     return Utils::translate_attribute_list(attributeInfoList);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

boost::python::list 
DeviceProxy::get_attribute_list()
{ 
  try { 
      std::vector<std::string> *attrList = deviceProxy->get_attribute_list();
      boost::python::list pyAttrList = get_list_from_vector<std::string>(*attrList);
      delete attrList;
      return pyAttrList;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e; 
  } 
} 


boost::python::object
DeviceProxy::get_attribute_config(std::string attribute)
{ 
  try { 
      Tango::AttributeInfo attributeInfo = deviceProxy->get_attribute_config(attribute);
      return boost::python::object ( Utils::translate_attribute_info(&attributeInfo) );
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 


boost::python::list 
DeviceProxy::get_attribute_config(boost::python::list att_list)
{ 
  try { 
    std::vector<std::string> vstr_array;
    get_array_value<std::string>(vstr_array,"String",
		(boost::python::object)att_list);
    Tango::AttributeInfoList *attributeInfoList = deviceProxy->get_attribute_config(vstr_array);
    return Utils::translate_attribute_list(attributeInfoList);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
} 

void 
DeviceProxy::set_attribute_config(boost::python::list att_info_list)
{ 
  try { 
    Tango::AttributeInfoList attributeInfoList;
    int s_len = boost::python::extract<int>(att_info_list.attr("__len__")()) ;
    for (int i = 0 ; i < s_len; i++ )
    {
    	AttributeInfo att_info =  boost::python::extract<AttributeInfo>(att_info_list[i]) ;
        attributeInfoList.push_back( Utils::translate_to_attribute_info(att_info) );		  		  
    }
    deviceProxy->set_attribute_config(attributeInfoList);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}
 
boost::python::object 
DeviceProxy::read_attribute(std::string name)
{
  try {
    long arg_type = (deviceProxyUtils->get_attribute_type(name)).first;
    Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(name)).second;
    Tango::DeviceAttribute tangoOutData = deviceProxy->read_attribute(name);
    boost::python::object result = 
		(boost::python::object)Utils::translate_from_device_attribute(tangoOutData,arg_type,arg_format);
    return result;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}
 

boost::python::list 
DeviceProxy::read_attributes(boost::python::list name_list)
{
  try { 
    std::vector<std::string> vstr_array;
    get_array_value<std::string>(vstr_array,"String",
		(boost::python::object)name_list);
    std::vector<Tango::DeviceAttribute> *dev_attr = deviceProxy->read_attributes(vstr_array);
    boost::python::list pyDevAttrList;
    std::vector<Tango::DeviceAttribute>::iterator i;
    long arg_type;
    Tango::AttrDataFormat arg_format;
    for (i = dev_attr->begin() ; i < dev_attr->end() ; i++ )
    {
       arg_type = (deviceProxyUtils->get_attribute_type( i->get_name() )).first;
       arg_format = (deviceProxyUtils->get_attribute_type( i->get_name() )).second;
       pyDevAttrList.append( Utils::translate_from_device_attribute( *i , arg_type , arg_format ) );
    }
    delete dev_attr;
    return pyDevAttrList;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}

void 
DeviceProxy::write_attributes(boost::python::list attr_list)
{
  try { 
    std::vector<Tango::DeviceAttribute> dev_attr_array;
    Tango::DeviceAttribute dev_attr;
    int s_len = boost::python::extract<int>(attr_list.attr("__len__")()) ;
    for (int i = 0 ; i < s_len; i++ )
    {
    	AttributeValue attr_val = boost::python::extract<AttributeValue>(attr_list[i]);
	long arg_type = (deviceProxyUtils->get_attribute_type(attr_val.name)).first;
	Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(attr_val.name)).second;
	Utils::translate_to_device_attribute(dev_attr, attr_val, arg_type, arg_format);
        dev_attr_array.push_back( dev_attr );		  		  
    }
    deviceProxy->write_attributes(dev_attr_array);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}

void 
DeviceProxy::write_attribute(AttributeValue attr_val)
{
  try { 
     Tango::DeviceAttribute dev_attr;
     	long arg_type = (deviceProxyUtils->get_attribute_type(attr_val.name)).first;
	Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(attr_val.name)).second;
	Utils::translate_to_device_attribute(dev_attr, attr_val, arg_type, arg_format);
        deviceProxy->write_attribute( dev_attr );
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


long 
DeviceProxy::read_attribute_asynch(std::string name)
{
  try {
	 long id = deviceProxy->read_attribute_asynch(name);
	 return id;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}
 

long 
DeviceProxy::read_attributes_asynch(boost::python::list name_list)
{
  try {
    std::vector<std::string> vstr_array;
    get_array_value<std::string>(vstr_array,"String",
		(boost::python::object)name_list);
	 long id = deviceProxy->read_attributes_asynch(vstr_array);
	 return id;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}


boost::python::object 
DeviceProxy::read_attribute_reply(long id)
{
  try { 
    Tango::DeviceAttribute *tangoOutData = deviceProxy->read_attribute_reply(id);
	std::string name = tangoOutData->get_name();
    long arg_type = (deviceProxyUtils->get_attribute_type(name)).first;
    Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(name)).second;
    boost::python::object result = 
		(boost::python::object)Utils::translate_from_device_attribute(*tangoOutData,arg_type,arg_format);
    return result;
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


boost::python::object 
DeviceProxy::read_attribute_reply(long id,long timeout)
{
  try { 
    Tango::DeviceAttribute *tangoOutData = deviceProxy->read_attribute_reply(id,timeout);
	std::string name = tangoOutData->get_name();
    long arg_type = (deviceProxyUtils->get_attribute_type(name)).first;
    Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(name)).second;
    boost::python::object result = 
		(boost::python::object)Utils::translate_from_device_attribute(*tangoOutData,arg_type,arg_format);
    return result;
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


boost::python::list 
DeviceProxy::read_attributes_reply(long id)
{
  try { 
    std::vector<Tango::DeviceAttribute> *dev_attr = deviceProxy->read_attributes_reply(id);
    boost::python::list pyDevAttrList;
    std::vector<Tango::DeviceAttribute>::iterator i;
    long arg_type;
    Tango::AttrDataFormat arg_format;
    for (i = dev_attr->begin() ; i < dev_attr->end() ; i++ )
    {
       arg_type = (deviceProxyUtils->get_attribute_type( i->get_name() )).first;
       arg_format = (deviceProxyUtils->get_attribute_type( i->get_name() )).second;
       pyDevAttrList.append( Utils::translate_from_device_attribute( *i , arg_type , arg_format ) );
    }
    delete dev_attr;
    return pyDevAttrList;
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


boost::python::list 
DeviceProxy::read_attributes_reply(long id,long timeout)
{
  try { 
    std::vector<Tango::DeviceAttribute> *dev_attr = deviceProxy->read_attributes_reply(id,timeout);
    boost::python::list pyDevAttrList;
    std::vector<Tango::DeviceAttribute>::iterator i;
    long arg_type;
    Tango::AttrDataFormat arg_format;
    for (i = dev_attr->begin() ; i < dev_attr->end() ; i++ )
    {
       arg_type = (deviceProxyUtils->get_attribute_type( i->get_name() )).first;
       arg_format = (deviceProxyUtils->get_attribute_type( i->get_name() )).second;
       pyDevAttrList.append( Utils::translate_from_device_attribute( *i , arg_type , arg_format ) );
    }
    delete dev_attr;
    return pyDevAttrList;
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


long 
DeviceProxy::write_attribute_asynch(AttributeValue attr_val)
{
  try { 
     Tango::DeviceAttribute dev_attr;
     long arg_type = (deviceProxyUtils->get_attribute_type(attr_val.name)).first;
	Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(attr_val.name)).second;
	Utils::translate_to_device_attribute(dev_attr, attr_val, arg_type, arg_format);
     long id = deviceProxy->write_attribute_asynch( dev_attr );
	 return id;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}


long 
DeviceProxy::write_attributes_asynch(boost::python::list attr_list)
{
  try { 
    std::vector<Tango::DeviceAttribute> dev_attr_array;
    Tango::DeviceAttribute dev_attr;
    int s_len = boost::python::extract<int>(attr_list.attr("__len__")()) ;
    for (int i = 0 ; i < s_len; i++ )
    {
    	AttributeValue attr_val = boost::python::extract<AttributeValue>(attr_list[i]);
	long arg_type = (deviceProxyUtils->get_attribute_type(attr_val.name)).first;
	Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(attr_val.name)).second;
	Utils::translate_to_device_attribute(dev_attr, attr_val, arg_type, arg_format);
        dev_attr_array.push_back( dev_attr );		  		  
    }
    long id = deviceProxy->write_attributes_asynch(dev_attr_array);
	return id;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}


void 
DeviceProxy::write_attribute_reply(long id)
{
  try { 
    deviceProxy->write_attribute_reply(id);
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


void 
DeviceProxy::write_attribute_reply(long id,long timeout)
{
  try { 
    deviceProxy->write_attribute_reply(id,timeout);
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


void 
DeviceProxy::write_attributes_reply(long id)
{
  try { 
    deviceProxy->write_attributes_reply(id);
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


void 
DeviceProxy::write_attributes_reply(long id,long timeout)
{
  try { 
    deviceProxy->write_attributes_reply(id,timeout);
  } catch(const Tango::AsynCall &e) {
    throw e;
  } catch(const Tango::AsynReplyNotArrived &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


void 
DeviceProxy::read_attribute_asynch(std::string name,boost::python::object callback)
{
  try {
     PythonAttrReadCallback python_callback(this,callback);
	 deviceProxy->read_attribute_asynch(name,python_callback);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}
 

void 
DeviceProxy::read_attributes_asynch(boost::python::list name_list,boost::python::object callback)
{
  try {
     PythonAttrReadCallback python_callback(this,callback);
    std::vector<std::string> vstr_array;
    get_array_value<std::string>(vstr_array,"String",
		(boost::python::object)name_list);
	 deviceProxy->read_attributes_asynch(vstr_array,python_callback);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}


void 
DeviceProxy::write_attribute_asynch(AttributeValue attr_val,boost::python::object callback)
{
  try {
     PythonAttrWriteCallback python_callback(this,callback);
     Tango::DeviceAttribute dev_attr;
     	long arg_type = (deviceProxyUtils->get_attribute_type(attr_val.name)).first;
	Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(attr_val.name)).second;
	Utils::translate_to_device_attribute(dev_attr, attr_val, arg_type, arg_format);
     deviceProxy->write_attribute_asynch( dev_attr , python_callback);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}
 

void 
DeviceProxy::write_attributes_asynch(boost::python::list attr_list,boost::python::object callback)
{
  try { 
    std::vector<Tango::DeviceAttribute> dev_attr_array;
    Tango::DeviceAttribute dev_attr;
    int s_len = boost::python::extract<int>(attr_list.attr("__len__")()) ;
    for (int i = 0 ; i < s_len; i++ )
    {
    	AttributeValue attr_val = boost::python::extract<AttributeValue>(attr_list[i]);
	long arg_type = (deviceProxyUtils->get_attribute_type(attr_val.name)).first;
	Tango::AttrDataFormat arg_format = (deviceProxyUtils->get_attribute_type(attr_val.name)).second;
	Utils::translate_to_device_attribute(dev_attr, attr_val, arg_type, arg_format);
        dev_attr_array.push_back( dev_attr );		  		  
    }
    PythonAttrWriteCallback python_callback(this,callback);
    deviceProxy->write_attributes_asynch(dev_attr_array,python_callback);
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  }
}



boost::python::object 
DeviceProxy::is_command_polled(std::string command)
{ 
     return boost::python::object( deviceProxy->is_command_polled(command) );
} 

boost::python::object 
DeviceProxy::is_attribute_polled(std::string attribute)
{ 
     return boost::python::object( deviceProxy->is_attribute_polled(attribute) );
}
 
int 
DeviceProxy::get_command_poll_period(std::string command)
{ 
     return deviceProxy->get_command_poll_period(command);
}

int 
DeviceProxy::get_attribute_poll_period(std::string attribute)
{ 
     return deviceProxy->get_attribute_poll_period(attribute);
}


boost::python::list 
DeviceProxy::polling_status()
{
    std::vector<std::string> *vstr_array = deviceProxy->polling_status();
    boost::python::list pyStatList = get_list_from_vector<std::string>(*vstr_array);
    delete vstr_array;
    return pyStatList;
}

void 
DeviceProxy::poll_command(std::string command,int period)
{ 
    deviceProxy->poll_command(command,period);
}

void 
DeviceProxy::poll_attribute(std::string attribute,int period)
{ 
    deviceProxy->poll_attribute(attribute,period);
}

void 
DeviceProxy::stop_poll_command(std::string command)
{ 
    deviceProxy->stop_poll_command(command);
}

void 
DeviceProxy::stop_poll_attribute(std::string attribute)
{ 
    deviceProxy->stop_poll_attribute(attribute);
}


long 
DeviceProxy::pending_asynch_call(boost::python::object req_type)
{
	Tango::asyn_req_type r_type = boost::python::extract<Tango::asyn_req_type>(req_type) ;
	return deviceProxy->pending_asynch_call(r_type);
}

void 
DeviceProxy::get_asynch_replies()
{
	deviceProxy->get_asynch_replies();
}


void 
DeviceProxy::get_asynch_replies(long timeout)
{
	try {
		deviceProxy->get_asynch_replies(timeout);
	} catch(const Tango::AsynReplyNotArrived &e) {
		throw e;
	}
}



boost::python::dict 
DeviceProxy::get_property(boost::python::list list_prop)
{
  try { 
    Tango::DbData db_data;
    std::vector<std::string> vstr_array;
    get_array_value<std::string>(vstr_array,"String",
		(boost::python::object)list_prop);
    deviceProxy->get_property(vstr_array,db_data);
    std::vector<Tango::DbDatum>::iterator i;
    boost::python::dict py_list_prop;
    int k;
    for ( k=0,i=db_data.begin() ; i < db_data.end() ; k++,i++)
    {
	boost::python::list py_list_val = get_list_from_vector<std::string>(i->value_string);
	py_list_prop[ vstr_array[k] ] = py_list_val; 	
    }
    return py_list_prop;
  } catch(const Tango::NonDbDevice &e) {
    throw e;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}


void 
DeviceProxy::put_property(boost::python::dict py_list_prop)
{
  try { 
    Tango::DbData db_data;
    std::string prop_name;
    std::string i_string;
    boost::python::list list_prop = py_list_prop.keys();
    int p_len = boost::python::extract<int>(list_prop.attr("__len__")()) ;

    for (int i = 0 ; i < p_len; i++ )
    {
        get_scalar_value<std::string>(prop_name,"String",list_prop[i]);
        Tango::DbDatum db_datum(prop_name);
        get_array_value<std::string>(db_datum.value_string,"String list",py_list_prop[prop_name]);
        db_data.push_back(db_datum);    
    }
    deviceProxy->put_property(db_data);
  } catch(const Tango::NonDbDevice &e) {
    throw e;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}

void 
DeviceProxy::delete_property(boost::python::list list_prop)
{
  try { 
    std::vector<std::string> vstr_array;
    get_array_value<std::string>(vstr_array,"String list",
		(boost::python::object)list_prop);
    deviceProxy->delete_property(vstr_array);
  } catch(const Tango::NonDbDevice &e) {
    throw e;
  } catch(const Tango::ConnectionFailed &e) {
    throw e;
  } catch(const Tango::CommunicationFailed &e) {
    throw e;
  } catch(const Tango::DevFailed &e) {
    throw e;
  } 
}

/* VF */
int DeviceProxy::subscribe_event(std::string attr_name, Tango::EventType event, boost::python::object call, boost::python::list filters)
{
   PyThreadState *tstate = 0;
   try
   {
	 // boost::weak_ptr<boost::python::object> temp_call(call);
	std::vector<std::string> str_array;
	get_array_value<std::string>(str_array, "String", filters);
	PythonPushEventCallback *event_callback =  new PythonPushEventCallback(this, call);
	PyEval_InitThreads();
	tstate = PyEval_SaveThread();
	event_callback->interp = tstate->interp;
	int event_id = deviceProxy->subscribe_event(attr_name, (Tango::EventType) event, (Tango::CallBack *) event_callback, (const std::vector<string>&) str_array);
	event_list.insert(std::pair<int, PythonPushEventCallback *>(event_id, event_callback));
	PyEval_RestoreThread(tstate);
	return event_id;
   }
   catch(const Tango::DevFailed &e) {
    PyEval_RestoreThread(tstate);
    throw e;
   }
}

void DeviceProxy::unsubscribe_event(int event_id)
{
   try
   {
	   std::map<int, PythonPushEventCallback *>::iterator it = event_list.find(event_id);
	   if (it != event_list.end())
	   {
		   delete (*it).second;
		   event_list.erase(it);
	   }
	deviceProxy->unsubscribe_event(event_id);
   }
   catch(const Tango::DevFailed &e) {
    throw e;
   }
}
