#include "conversion_utils.h"
#include <templates.cpp>

boost::python::tuple Utils::translate_exception_value(const Tango::DevErrorList &dev_error_list)
{
    	boost::python::list exc_value;
    	CORBA::ULong i;
    	for (i=0; i < dev_error_list.length() ; i++)
    	{
		boost::python::dict err;
		err["reason"] = dev_error_list[i].reason.in();

        	std::string severity;
        	switch (dev_error_list[i].severity)
		{
			case Tango::WARN :
				severity = "WARNING";
				break;
			case Tango::ERR :
				severity = "ERROR";
				break;
			case Tango::PANIC :
				severity = "PANIC";
				break;
		}
		err["severity"] = severity;
		err["desc"] = dev_error_list[i].desc.in();
		err["origin"] = dev_error_list[i].origin.in();
		exc_value.append(err);
    	}
    	return boost::python::tuple(exc_value);
}

boost::python::tuple Utils::translate_black_box(std::vector<string> *comHist) 
{
	boost::python::list pyComHist = get_list_from_vector<std::string>(*comHist);
	delete comHist;
	return boost::python::tuple(pyComHist);
}

void Utils::translate_to_device_data(Tango::DeviceData &deviceData, boost::python::object pyData, long arg_type)
{
	switch (arg_type) {
	case Tango::DEV_BOOLEAN :
	{
		if ((pyData.attr("__str__")() == "false") || (pyData.attr("__str__")() == "False") || (pyData.attr("__str__")() == "FALSE") || (pyData.attr("__str__")() == "0"))
			deviceData << false;
		else 
			deviceData << true;
		break;
	}
	case Tango::DEV_SHORT :
	{
		Tango::DevShort short_val;
		get_scalar_value<Tango::DevShort>(short_val,"Tango::DevShort",pyData);
		deviceData << short_val;
		break;
	}
	case Tango::DEV_LONG :
	{
		Tango::DevLong long_val;
		get_scalar_value<Tango::DevLong>(long_val,"Tango::DevLong",pyData);
		deviceData << long_val;
		break;
	}
	case Tango::DEV_FLOAT :
	{
		Tango::DevFloat float_val;
		get_scalar_value<Tango::DevFloat>(float_val,"Tango::DevFloat",pyData);
		deviceData << float_val;
		break;
	}
	case Tango::DEV_DOUBLE :
	{
		Tango::DevDouble double_val;
		get_scalar_value<Tango::DevDouble>(double_val,"Tango::DevDouble",pyData);
		deviceData << double_val;
		break;
	}
	case Tango::DEV_USHORT :
	{
		Tango::DevUShort ushort_val;
		get_scalar_value<Tango::DevUShort>(ushort_val,"Tango::DevUShort",pyData);
		deviceData << ushort_val;
		break;
	}
	case Tango::DEV_ULONG :
	{
		Tango::DevULong ulong_val;
		get_scalar_value<Tango::DevULong>(ulong_val,"Tango::DevULong",pyData);
		deviceData << ulong_val;
		break;
	}
	case Tango::DEV_STRING :
	case Tango::CONST_DEV_STRING :
	{
		std::string string_val;
		get_scalar_value<std::string>(string_val,"String",pyData);
		deviceData << string_val;
		break;
	}
	case Tango::DEV_STATE :
	{
		Tango::DevState state_val;
		boost::python::extract<Tango::DevState> val(pyData);
		if (val.check())
		{
			state_val = (Tango::DevState) val();
			deviceData << state_val;
		}
		else
		{
			Tango::Except::throw_exception((const char *)"PyDs_WrongDatatype",
						       (const char *)"Data coming from Python is not a State",
						       (const char *)"Python_binding::translate_to_device_data()");
		}
		break;
	}
	case Tango::DEVVAR_CHARARRAY :
	{
		std::vector<unsigned char> vchar_array;
		get_array_value<unsigned char>(vchar_array,"unsigned char list",pyData);
		deviceData << vchar_array;
		break;
	}
	case Tango::DEVVAR_SHORTARRAY :
	{	
		std::vector<Tango::DevShort> vshort_array;
		get_array_value<Tango::DevShort>(vshort_array,"Tango::DevShort list",pyData);
		deviceData << vshort_array;
		break;    
	}
	case Tango::DEVVAR_LONGARRAY :
	{
		std::vector<Tango::DevLong> vlong_array;
		get_array_value<Tango::DevLong>(vlong_array,"Tango::DevLong list",pyData);
		deviceData << vlong_array;
		break;    
	}
	case Tango::DEVVAR_FLOATARRAY :
	{
		std::vector<Tango::DevFloat> vfloat_array;
		get_array_value<Tango::DevFloat>(vfloat_array,"Tango::DevFloat list",pyData);
		deviceData << vfloat_array;
		break;    
	}
	case Tango::DEVVAR_DOUBLEARRAY :
	{
		std::vector<Tango::DevDouble> vdouble_array;
		get_array_value<Tango::DevDouble>(vdouble_array,"Tango::DevDouble list",pyData);
		deviceData << vdouble_array;
		break;    
	}
	case Tango::DEVVAR_USHORTARRAY :
	{
		std::vector<Tango::DevUShort> vushort_array;
		get_array_value<Tango::DevUShort>(vushort_array,"Tango::DevUShort list",pyData);
		deviceData << vushort_array;
		break;    
	}
	case Tango::DEVVAR_ULONGARRAY :
	{
		std::vector<Tango::DevULong> vulong_array;
		get_array_value<Tango::DevULong>(vulong_array,"Tango::DevULong list",pyData);
		deviceData << vulong_array;
		break;    
	}
	case Tango::DEVVAR_STRINGARRAY :
	{
		std::vector<std::string> vstr_array;
		get_array_value<std::string>(vstr_array,"String list",pyData);
		deviceData << vstr_array;
		break;
	}
	case Tango::DEVVAR_LONGSTRINGARRAY :
	{
		std::vector<Tango::DevLong> vlong_array;
		std::vector<std::string> vstr_array;
		get_array_value<Tango::DevLong>(vlong_array,"Tango::DevLong list",pyData[0]);
		get_array_value<std::string>(vstr_array,"String list",pyData[1]);
		deviceData.insert(vlong_array,vstr_array);
		break;
	}
	case Tango::DEVVAR_DOUBLESTRINGARRAY :
	{
		std::vector<Tango::DevDouble> vdouble_array;
		std::vector<std::string> vstr_array;
		get_array_value<Tango::DevDouble>(vdouble_array,"Tango::DevDouble list",pyData[0]);
		get_array_value<std::string>(vstr_array,"String list",pyData[1]);
		deviceData.insert(vdouble_array,vstr_array);
		break;
	}
	default :
		break;
	}
}
	
boost::python::object Utils::translate_from_device_data(Tango::DeviceData &deviceData, long arg_type)
{
	boost::python::object def ;
	switch (arg_type) {
	case Tango::DEV_BOOLEAN :
	{
		Tango::DevBoolean val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::DEV_SHORT :
	{
		Tango::DevShort val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::DEV_LONG :
	{
		Tango::DevLong val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::DEV_FLOAT :
	{	Tango::DevFloat val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::DEV_DOUBLE :
	{
		Tango::DevDouble val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::DEV_USHORT :
	{
		Tango::DevUShort val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::DEV_ULONG :
	{
		Tango::DevULong val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	case Tango::CONST_DEV_STRING :
	case Tango::DEV_STRING :
	{
		std::string val;
		deviceData >> val;
		return boost::python::object(val);
		break;
	}
	/* VF */
	case Tango::DEV_STATE:
	{
		Tango::DevState dev_state;
		deviceData >> dev_state;
		return boost::python::object(dev_state);
		break;
	}
	case Tango::DEVVAR_CHARARRAY :
	{
		const Tango::DevVarCharArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarCharArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_SHORTARRAY :
	{
		const Tango::DevVarShortArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarShortArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_LONGARRAY :
	{
		const Tango::DevVarLongArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarLongArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_FLOATARRAY :
	{
		const Tango::DevVarFloatArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarFloatArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_DOUBLEARRAY :
	{
		const Tango::DevVarDoubleArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarDoubleArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_USHORTARRAY :
	{
		const Tango::DevVarUShortArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarUShortArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_ULONGARRAY :
	{
		const Tango::DevVarULongArray *tmp_ptr;
		deviceData >> tmp_ptr;
		return get_list_from_const_sequence<Tango::DevVarULongArray>(tmp_ptr);
		break;
	}
	case Tango::DEVVAR_STRINGARRAY :
	{
		std::vector<std::string> vstring;
		deviceData >> vstring;
		return get_list_from_vector<std::string>(vstring);
		break;
	}
	case Tango::DEVVAR_LONGSTRINGARRAY :
	{
		const Tango::DevVarLongStringArray *tmp_ptr;
		deviceData >> tmp_ptr;
		boost::python::list l_val = get_list_from_const_sequence<Tango::DevVarLongArray>(&(tmp_ptr->lvalue));
		boost::python::list s_val = get_list_from_const_sequence<Tango::DevVarStringArray>(&(tmp_ptr->svalue));
		return boost::python::make_tuple(l_val,s_val);
		break;
	}
	case Tango::DEVVAR_DOUBLESTRINGARRAY :
	{
		const Tango::DevVarDoubleStringArray *tmp_ptr;
		deviceData >> tmp_ptr;
		boost::python::list d_val = get_list_from_const_sequence<Tango::DevVarDoubleArray>(&(tmp_ptr->dvalue));
		boost::python::list s_val = get_list_from_const_sequence<Tango::DevVarStringArray>(&(tmp_ptr->svalue));
		return boost::python::make_tuple(d_val,s_val);
		break;
	}
	default :
		break;
	}
	return def;  
}

DeviceDataHistory Utils::translate_from_data_history(Tango::DeviceDataHistory &deviceDataHistory,long arg_type)
{
	DeviceDataHistory py_data_history ;
	py_data_history.time = deviceDataHistory.date();
	boost::python::object value;
	boost::python::list errors;
	if (deviceDataHistory.failed()) {
		py_data_history.cmd_failed = 1;
		const Tango::DevErrorList &dev_error_list = deviceDataHistory.errors();
		CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			errors.append( dev_error_list[i] );
		}
	} else {
		py_data_history.cmd_failed = 0;
		value = translate_from_device_data(deviceDataHistory,arg_type);
	}
	py_data_history.value = value;
	py_data_history.errors = errors;
	return py_data_history;
}

DeviceAttributeHistory Utils::translate_from_attribute_history(Tango::DeviceAttributeHistory &deviceAttrHistory,long arg_type,Tango::AttrDataFormat arg_format)
{
	DeviceAttributeHistory py_attr_history ;
	AttributeValue value;
	boost::python::list errors;
	if (deviceAttrHistory.failed()) {
		py_attr_history.attr_failed = 1;
		/* commented out to compile in Tango V5: check */
		const Tango::DevErrorList &dev_error_list = deviceAttrHistory.get_err_stack();
		CORBA::ULong i;
		for (i=0; i < dev_error_list.length() ; i++)
		{
			errors.append( dev_error_list[i] );
		}
	} else {
		py_attr_history.attr_failed = 0;
		value = translate_from_device_attribute(deviceAttrHistory,arg_type,arg_format);
	}
	py_attr_history.value = value;
	py_attr_history.errors = errors;
	return py_attr_history;
}

AttributeInfo Utils::translate_attribute_info(Tango::AttributeInfo *attributeInfo) 
{
	AttributeInfo pyAttributeInfo;
	pyAttributeInfo.name = attributeInfo->name;
	pyAttributeInfo.writable = attributeInfo->writable;
	pyAttributeInfo.data_format = attributeInfo->data_format;
	pyAttributeInfo.data_type = attributeInfo->data_type;
	pyAttributeInfo.max_dim_x = attributeInfo->max_dim_x;
	pyAttributeInfo.max_dim_y = attributeInfo->max_dim_y;
	pyAttributeInfo.description = attributeInfo->description;
	pyAttributeInfo.label = attributeInfo->label;
	pyAttributeInfo.unit = attributeInfo->unit;
	pyAttributeInfo.standard_unit = attributeInfo->standard_unit;
	pyAttributeInfo.display_unit = attributeInfo->display_unit;
	pyAttributeInfo.format = attributeInfo->format;
	pyAttributeInfo.min_value = attributeInfo->min_value;
	pyAttributeInfo.max_value = attributeInfo->max_value;
	pyAttributeInfo.min_alarm = attributeInfo->min_alarm;
	pyAttributeInfo.max_alarm = attributeInfo->max_alarm;
	pyAttributeInfo.writable_attr_name = attributeInfo->writable_attr_name;
	pyAttributeInfo.extensions = get_list_from_vector<std::string>(attributeInfo->extensions);
	pyAttributeInfo.disp_level = attributeInfo->disp_level;
	return pyAttributeInfo;
}

Tango::AttributeInfo Utils::translate_to_attribute_info(AttributeInfo pyAttributeInfo) 
{
	Tango::AttributeInfo attributeInfo;
	get_scalar_value<std::string>(attributeInfo.name,"String",
			(boost::python::object)(pyAttributeInfo.name));
	attributeInfo.writable = pyAttributeInfo.writable;
	attributeInfo.data_format = pyAttributeInfo.data_format;
	get_scalar_value<int>(attributeInfo.data_type,"int",
			(boost::python::object)(pyAttributeInfo.data_type));
	get_scalar_value<int>(attributeInfo.max_dim_x,"int",
			(boost::python::object)(pyAttributeInfo.max_dim_x));
	get_scalar_value<int>(attributeInfo.max_dim_y,"int",
			(boost::python::object)(pyAttributeInfo.max_dim_y));
	get_scalar_value<std::string>(attributeInfo.description,"String",
			(boost::python::object)(pyAttributeInfo.description));
	get_scalar_value<std::string>(attributeInfo.label,"String",
			(boost::python::object)(pyAttributeInfo.label));
	get_scalar_value<std::string>(attributeInfo.unit,"String",
			(boost::python::object)(pyAttributeInfo.unit));
	get_scalar_value<std::string>(attributeInfo.standard_unit,"String",
			(boost::python::object)(pyAttributeInfo.standard_unit));
	get_scalar_value<std::string>(attributeInfo.display_unit,"String",
			(boost::python::object)(pyAttributeInfo.display_unit));
	get_scalar_value<std::string>(attributeInfo.format,"String",
			(boost::python::object)(pyAttributeInfo.format));
	get_scalar_value<std::string>(attributeInfo.min_value,"String",
			(boost::python::object)(pyAttributeInfo.min_value));
	get_scalar_value<std::string>(attributeInfo.max_value,"String",
			(boost::python::object)(pyAttributeInfo.max_value));
	get_scalar_value<std::string>(attributeInfo.min_alarm,"String",
			(boost::python::object)(pyAttributeInfo.min_alarm));
	get_scalar_value<std::string>(attributeInfo.max_alarm,"String",
			(boost::python::object)(pyAttributeInfo.max_alarm));
	get_scalar_value<std::string>(attributeInfo.writable_attr_name,"String",
			(boost::python::object)(pyAttributeInfo.writable_attr_name));
	boost::python::list extensions = pyAttributeInfo.extensions;
	get_array_value<std::string>(attributeInfo.extensions,"String list",
			(boost::python::object)extensions);
	attributeInfo.disp_level = pyAttributeInfo.disp_level;
	return attributeInfo;
}

boost::python::list Utils::translate_attribute_list(Tango::AttributeInfoList *attrHist) 
{
	boost::python::list pyAttrHist;
	std::vector<Tango::AttributeInfo>::iterator i;
	for (i = attrHist->begin() ; i < attrHist->end() ; i++ )
		pyAttrHist.append( translate_attribute_info( &(*i) ) );
	delete attrHist;
	return pyAttrHist;
}

AttributeValue Utils::translate_from_device_attribute(Tango::DeviceAttribute &deviceAttr, long arg_type, Tango::AttrDataFormat arg_format)
{
	AttributeValue result ;
	result.quality = deviceAttr.quality;
	result.time = deviceAttr.time;
	result.name = deviceAttr.name;
	result.dim_x = deviceAttr.dim_x;
	result.dim_y = deviceAttr.dim_y;
	switch (arg_type) {
	case Tango::DEV_STATE:
	{
		Tango::DevState val;
		deviceAttr >> val;
		result.value = (boost::python::object) val;
		break;
	}
	case Tango::DEV_SHORT :
		switch (arg_format)  {
		case Tango::SCALAR :
		{
			Tango::DevShort val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default :
		{
			Tango::DevVarShortArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarShortArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
	case Tango::DEV_LONG :
		switch (arg_format)  {
		case Tango::SCALAR :
		{
			Tango::DevLong val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default :
		{
			Tango::DevVarLongArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarLongArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
	case Tango::DEV_DOUBLE :
		switch (arg_format)  {
		case Tango::SCALAR :
		{
			Tango::DevDouble val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default :
		{
			Tango::DevVarDoubleArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarDoubleArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
	case Tango::DEV_FLOAT :
		switch (arg_format)  
		{
		case Tango::SCALAR:
		{
			Tango::DevFloat val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default:
		{
			Tango::DevVarFloatArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarFloatArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
	case Tango::DEV_BOOLEAN :
		switch (arg_format)  
		{
		case Tango::SCALAR:
		{
			Tango::DevBoolean val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default:
		{	
			Tango::DevVarBooleanArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarBooleanArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
		
	case Tango::DEV_USHORT :
		switch (arg_format)  
		{
		case Tango::SCALAR:
		{
			Tango::DevUShort val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default:
		{
			Tango::DevVarUShortArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarUShortArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
	
	case Tango::DEV_ULONG :
		switch (arg_format)  
		{
		case Tango::SCALAR:
		{
			Tango::DevLong val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default:
		{
			Tango::DevVarLongArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			result.value = (boost::python::object)get_list_from_sequence<Tango::DevVarLongArray>(tmp_ptr);
			delete tmp_ptr;
			break;           
		}
		}
		break;
	
	case Tango::DEV_STRING :
		switch (arg_format) {
		case Tango::SCALAR:
		{
			std::string val;
			deviceAttr >> val;
			result.value = (boost::python::object) val;
			break;
		}
		default :
		{
			std::vector<std::string> vstring;
			deviceAttr >> vstring;
			result.value = (boost::python::object)get_list_from_vector<std::string>(vstring);
			break;           
		}
		}
		break;
	
	case Tango::DEV_UCHAR :
		switch (arg_format) {
		case Tango::SCALAR:
		{
			unsigned char val;
			deviceAttr >> val;
			result.value = (boost::python::object)val;
			break;
		}
		default :
		{			
			Tango::DevVarCharArray *tmp_ptr;
			deviceAttr >> tmp_ptr;
			unsigned long nb_data = tmp_ptr->length();
			
			CORBA::Octet *co = tmp_ptr->get_buffer();
			const char *ch_ptr = (const char *)co;
			boost::python::str py_str(ch_ptr,(size_t)nb_data);
			delete tmp_ptr;
			result.value = py_str;

			break;        
		}
		}
		break;
	
	default :
		result.value = boost::python::object();
		break;
	}
	return result;  
}

void Utils::translate_to_device_attribute(Tango::DeviceAttribute &deviceAttr, AttributeValue attr_val, long arg_type, Tango::AttrDataFormat arg_format)
{
	std::string name = attr_val.name;
	//    get_scalar_value<std::string>(name,"String",attr_val[0]);
	deviceAttr.set_name(name);
	deviceAttr.quality = attr_val.quality;
	deviceAttr.time = attr_val.time;
	deviceAttr.name = attr_val.name;
	deviceAttr.dim_x = attr_val.dim_x;
	deviceAttr.dim_y = attr_val.dim_y;
	switch (arg_type) {
	case Tango::DEV_SHORT :
		switch (arg_format)  {
		case Tango::SCALAR :
		{
			Tango::DevShort val;
			get_scalar_value<Tango::DevShort>(val,"Tango::DevShort",attr_val.value);
			deviceAttr << val;
			break;
		}
		default :
		{
			std::vector<Tango::DevShort> vshort;
			get_array_value<Tango::DevShort>(vshort,"Tango::DevShort",attr_val.value);
			deviceAttr << vshort;
			break;           
		}
		}
		break;
	case Tango::DEV_LONG :
		switch (arg_format)  {
		case Tango::SCALAR :
		{
			Tango::DevLong val;
			get_scalar_value<Tango::DevLong>(val,"Tango::DevLong",attr_val.value);
			deviceAttr << val;
			break;
		}
		default :
		{
			std::vector<Tango::DevLong> vlong;
			get_array_value<Tango::DevLong>(vlong,"Tango::DevShort",attr_val.value);
			deviceAttr << vlong;
			break;           
		}
		}
		break;
	case Tango::DEV_DOUBLE :
		switch (arg_format)  {
		case Tango::SCALAR :
		{
			Tango::DevDouble val;
			get_scalar_value<Tango::DevDouble>(val,"Tango::DevDouble",attr_val.value);
			deviceAttr << val;
			break;
		}
		default :
		{
			std::vector<Tango::DevDouble> vdouble;
			get_array_value<Tango::DevDouble>(vdouble,"Tango::DevDouble",attr_val.value);
			deviceAttr << vdouble;
			break;           
		}
		}
		break;
	case Tango::DEV_FLOAT :
		switch (arg_format) {
		case Tango::SCALAR:
		{
			Tango::DevFloat val;
			get_scalar_value<Tango::DevFloat>(val,"Tango::DevFloat",attr_val.value);
			deviceAttr << val;
			break;
		}
		default:
		{
			std::vector<Tango::DevFloat> vec;
			get_array_value<Tango::DevFloat>(vec,"Tango::DevFloat",attr_val.value);
			deviceAttr << vec;
			break;           
		}
		}
		break;
		
	case Tango::DEV_BOOLEAN :
		switch (arg_format) { 
		case Tango::SCALAR:
		{
			Tango::DevBoolean val;
			get_scalar_value<Tango::DevBoolean>(val,"Tango::DevBoolean",attr_val.value);
			deviceAttr << val;
			break;
		}
		default:
		{
			std::vector<Tango::DevBoolean> vec;
			get_array_value<Tango::DevBoolean>(vec,"Tango::DevBoolean",attr_val.value);
			deviceAttr << vec;
			break;           
		}
		}
		break;
		
	case Tango::DEV_USHORT :
		switch (arg_format){
		case Tango::SCALAR:
		{
			Tango::DevUShort val;
			get_scalar_value<Tango::DevUShort>(val,"Tango::DevUShort",attr_val.value);
			deviceAttr << val;
			break;
		}
		default:
		{
			std::vector<Tango::DevUShort> vec;
			get_array_value<Tango::DevUShort>(vec,"Tango::DevUShort",attr_val.value);
			deviceAttr << vec;
			break;           
		}
		}
		break;
		
	case Tango::DEV_ULONG :
		switch (arg_format){
		case Tango::SCALAR:
		{
			Tango::DevLong val;
			get_scalar_value<Tango::DevLong>(val,"Tango::DevLong",attr_val.value);
			deviceAttr << val;
			break;
		}
		default:
		{
			std::vector<Tango::DevLong> vec;
			get_array_value<Tango::DevLong>(vec,"Tango::DevLong",attr_val.value);
			deviceAttr << vec;
			break;           
		}
		}
		break;
		
	case Tango::DEV_STRING :
		switch (arg_format){
		case Tango::SCALAR:
		{
			std::string val;
			get_scalar_value<std::string>(val,"String",attr_val.value);
			deviceAttr << val;
			break;
		}
		default:
		{
			std::vector<std::string> vec;
			get_array_value<std::string>(vec,"String",attr_val.value);
			deviceAttr << vec;
			break;           
		}
		}
		break;
	
	default :
		break;
	}
}
