static const char *RcsId = "$Header: /segfs/tango/cvsroot/cppserver/machine/powersupply/KiKe/KiKe.cpp,v 1.1.1.1 2008/07/10 08:41:16 meyer Exp $";
//+=============================================================================
//
// file :         KiKe.cpp
//
//  _  _  ____  _  _  ____    _  _  ___     ___  
// ( )/ )(_  _)( )/ )( ___)  ( \/ )(__ \   / _ \ 
//  )  (  _)(_  )  (  )__)    \  /  / _/  ( (_) )
// (_)\_)(____)(_)\_)(____)    \/  (____)()\___/ 
//
// description :  C++ source for the KiKe and its commands. 
//                The class is derived from Device. It represents the
//                CORBA servant object which will be accessed from the
//                network. All commands which can be executed on the
//                KiKe are implemented in this file.
//
// project :      TANGO Device Server
//
// $Author: vedder $
//
// $Revision: 1.1.1.1 $
//
// $Log: KiKe.cpp,v $
// Revision 1.1.1.1  2008/07/10 08:41:16  meyer
// Imported using TkCVS
//
//
// copyleft :     European Synchrotron Radiation Facility
//                BP 220, Grenoble 38043
//                FRANCE
//
//-=============================================================================
//
//  		This file is generated by POGO
//	(Program Obviously used to Generate tango Object)
//
//         (c) - Software Engineering Group - ESRF
//=============================================================================



//===================================================================
//
//	The following table gives the correspondence
//	between commands and method name.
//
//  Command name |  Method name
//	----------------------------------------
//  State        |  dev_state()
//  Status       |  dev_status()
//  On           |  on()
//  Off          |  off()
//  Reset        |  reset()
//  Standby      |  standby()
//  DBG_On       |  dbg__on()
//  DBG_Off      |  dbg__off()
//  DBG_Reset    |  dbg__reset()
//  DBG_Standby  |  dbg__standby()
//
//===================================================================

#include <tango.h>
#include <KiKe.h>
#include <KiKeClass.h>
#include <KiKeThread.h>

namespace KiKe_ns
{

//+----------------------------------------------------------------------------
//
// method : 		KiKe::KiKe(string &s)
// 
// description : 	constructor for simulated KiKe
//
// in : - cl : Pointer to the DeviceClass object
//      - s : Device name 
//
//-----------------------------------------------------------------------------
KiKe::KiKe(Tango::DeviceClass *cl,string &s)
:PowerSupply_ns::PowerSupply(cl,s.c_str())
{
	// Init here and not in init_device() is wanted:
	// the constructor is called only once per device
	sequenceRunning = false;
	sequenceFailed  = false;
	sequenceError = "";
	init_device();
}

KiKe::KiKe(Tango::DeviceClass *cl,const char *s)
:PowerSupply_ns::PowerSupply(cl,s)
{
	// Init here and not in init_device() is wanted:
	// the constructor is called only once per device
	sequenceRunning = false;
	sequenceFailed  = false;
	sequenceError = "";

	init_device();
}

KiKe::KiKe(Tango::DeviceClass *cl,const char *s,const char *d)
:PowerSupply_ns::PowerSupply(cl,s,d)
{
	// Init here and not in init_device() is wanted:
	// the constructor is called only once per device
	sequenceRunning = false;
	sequenceFailed  = false;
	sequenceError = "";

	init_device();
}
//+----------------------------------------------------------------------------
//
// method : 		KiKe::delete_device()
// 
// description : 	will be called at device destruction or at init command.
//
//-----------------------------------------------------------------------------
void KiKe::delete_device()
{
	//	Delete device allocated objects
	delete modbus_obj;
}

//+----------------------------------------------------------------------------
//
// method : 		KiKe::init_device()
// 
// description : 	will be called at device initialization.
//
//-----------------------------------------------------------------------------
void KiKe::init_device()
{
	INFO_STREAM << "KiKe::KiKe() create device " << device_name << endl;

	// Initialise variables to default values
	//--------------------------------------------
	get_device_property();
	
	// connect to PLC vis the modbus protocol
	modbus_obj = new Tango::DeviceProxy(modbusDevice);
}


//+----------------------------------------------------------------------------
//
// method : 		KiKe::get_device_property()
// 
// description : 	Read the device properties from database.
//
//-----------------------------------------------------------------------------
void KiKe::get_device_property()
{
	//	Initialize your default values here (if not done with  POGO).
	//------------------------------------------------------------------
	modbusDevice        = "";
	registerBaseAddress = -1;
	modbus_obj          = NULL;
	
	last_read_time      = 0;
	last_current_write	= 0;
	
	current_set			= 0;
	current_read        = 0;
	voltage             = 0;
	status_word         = 0;
	fault_word1         = 0;
	fault_word2         = 0;	
	alarm_word          = 0;
	
	kicker_state        = Tango::UNKNOWN;
	
	//	Read device properties from database.(Automatic code generation)
	//------------------------------------------------------------------
	Tango::DbData	dev_prop;
	dev_prop.push_back(Tango::DbDatum("ModbusDevice"));
	dev_prop.push_back(Tango::DbDatum("RegisterBaseAddress"));

	//	Call database and extract values
	//--------------------------------------------
	if (Tango::Util::instance()->_UseDb==true)
		get_db_device()->get_property(dev_prop);
	Tango::DbDatum	def_prop, cl_prop;
	KiKeClass	*ds_class =
		(static_cast<KiKeClass *>(get_device_class()));
	int	i = -1;

	//	Try to initialize ModbusDevice from class property
	cl_prop = ds_class->get_class_property(dev_prop[++i].name);
	if (cl_prop.is_empty()==false)	cl_prop  >>  modbusDevice;
	else {
		//	Try to initialize ModbusDevice from default device value
		def_prop = ds_class->get_default_device_property(dev_prop[i].name);
		if (def_prop.is_empty()==false)	def_prop  >>  modbusDevice;
	}
	//	And try to extract ModbusDevice value from database
	if (dev_prop[i].is_empty()==false)	dev_prop[i]  >>  modbusDevice;

	//	Try to initialize RegisterBaseAddress from class property
	cl_prop = ds_class->get_class_property(dev_prop[++i].name);
	if (cl_prop.is_empty()==false)	cl_prop  >>  registerBaseAddress;
	else {
		//	Try to initialize RegisterBaseAddress from default device value
		def_prop = ds_class->get_default_device_property(dev_prop[i].name);
		if (def_prop.is_empty()==false)	def_prop  >>  registerBaseAddress;
	}
	//	And try to extract RegisterBaseAddress value from database
	if (dev_prop[i].is_empty()==false)	dev_prop[i]  >>  registerBaseAddress;



	//	End of Automatic code generation
	//------------------------------------------------------------------

}
//+----------------------------------------------------------------------------
//
// method : 		KiKe::always_executed_hook()
// 
// description : 	method always executed before any command is executed
//
//-----------------------------------------------------------------------------
void KiKe::always_executed_hook()
{
	
}
//+----------------------------------------------------------------------------
//
// method : 		KiKe::read_attr_hardware
// 
// description : 	Hardware acquisition for attributes.
//
//-----------------------------------------------------------------------------
void KiKe::read_attr_hardware(vector<long> &attr_list)
{
	DEBUG_STREAM << "KiKe::read_attr_hardware(vector<long> &attr_list) entering... "<< endl;
	//	Add your own code here

	// read the PLC only once a second 
	
	time_t			current_time;
	double			time_difference;
	
	time (&current_time);
	time_difference = (double) difftime(current_time, last_read_time);
   // cout << "difftime = " << time_difference << endl;
	if ( time_difference > 1 )
	{
		status_word = 0;
		fault_word1 = 0;
		fault_word2 = 0;
		alarm_word  = 0;
		pulse_number = 0;
		pulse_number_read = 0;
		
		// read registers from PLC
		Tango::DeviceData argin;
 		Tango::DeviceData argout;

		vector<short> address;
		address.push_back (registerBaseAddress + READING_OFFSET);
		address.push_back (0x10);
		argin << address;
		
		argout = modbus_obj->command_inout ("ReadHoldingRegisters", argin);
		
		vector<short> registers;
		argout >> registers;

		current_set  = (double) registers[0x4];
		current_read = (double) registers[0x5];
		voltage      = (double) registers[0x6];
		pulse_number_read = (unsigned short)registers[0x7];		
		status_word  = registers[0x8];
		fault_word1  = registers[0x9];
		fault_word2  = registers[0xA];		
		alarm_word   = registers[0xB];

/*
		cout << "current set  = " << current_set << endl;
		cout << "current read = " << current_read << endl;
		cout << "voltage      = " << voltage << endl;
		cout << "pulse number read = " << pulse_number << endl;		
		cout << "status = " << status_word << endl;
		cout << "fault1 = " << fault_word1 << endl;
		cout << "fault2 = " << fault_word2 << endl;
		cout << "alarm  = " << alarm_word << endl;
*/		
		last_read_time = (time_t)current_time;
	}
}
//+----------------------------------------------------------------------------
//
// method : 		KiKe::write_PulseNumber
// 
// description : 	Write PulseNumber attribute values to hardware.
//
//-----------------------------------------------------------------------------
void KiKe::write_PulseNumber(Tango::WAttribute &attr)
{
	DEBUG_STREAM << "KiKe::write_PulseNumber(Tango::WAttribute &attr) entering... "<< endl;

	attr.get_write_value (pulse_number);

	// write the pulse set point 
	
	vector<short> address;
	address.push_back (registerBaseAddress + PULSE_SET_OFFSET);
	address.push_back (1);
	address.push_back (pulse_number);
	
	Tango::DeviceData argin;
	argin << address;
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);
	
	// apply the pulse set point
	address.clear();
	address.push_back (registerBaseAddress + CONTROL_WORD_OFFSET);
	address.push_back (1);
	address.push_back (APPLY_SETTING);
	
	argin << address;
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);		
}

//+----------------------------------------------------------------------------
//
// method : 		KiKe::read_PulseNumber
// 
// description : 	Extract real attribute values for PulseNumber acquisition result.
//
//-----------------------------------------------------------------------------
void KiKe::read_PulseNumber(Tango::Attribute &attr)
{
	DEBUG_STREAM << "KiKe::read_PulseNumber(Tango::Attribute &attr) entering... "<< endl;
	attr.set_value(&pulse_number_read);
}


//+----------------------------------------------------------------------------
//
// method : 		KiKe::write_Voltage
// 
// description : 	Write Voltage attribute values to hardware.
//
//-----------------------------------------------------------------------------
void KiKe::write_Voltage(Tango::WAttribute &attr)
{
	DEBUG_STREAM << "KiKe::write_Voltage(Tango::WAttribute &attr) entering... "<< endl;

	Tango::Except::throw_exception ((const char *)"KiKe error",
	                                (const char *)"A voltage set-point is not available for this power supply!",
	                                (const char *)"KiKe::write_Voltage");
}


//+----------------------------------------------------------------------------
//
// method : 		KiKe::read_Current
// 
// description : 	Extract real attribute values for Current acquisition result.
//
//-----------------------------------------------------------------------------
void KiKe::read_Current(Tango::Attribute &attr)
{
	DEBUG_STREAM << "KiKe::read_Current(Tango::Attribute &attr) entering... "<< endl;

	attr.set_value(&current_read);
}

//+----------------------------------------------------------------------------
//
// method : 		KiKe::write_Current
// 
// description : 	Write Current attribute values to hardware.
//
//-----------------------------------------------------------------------------
void KiKe::write_Current(Tango::WAttribute &attr)
{
	DEBUG_STREAM << "KiKe::write_Current(Tango::WAttribute &attr) entering... "<< endl;
	
	// Memorise write value.
	attr.get_write_value (attr_Current_write);

	//if status word contains bit ON or OFF set...
	if ( KIKE_IS_ON || KIKE_IN_STANDBY)
	{
		set_current_on_PLC();
	}
}

//+----------------------------------------------------------------------------
//
// method : 		KiKe::read_Voltage
// 
// description : 	Extract real attribute values for Voltage acquisition result.
//
//-----------------------------------------------------------------------------
void KiKe::read_Voltage(Tango::Attribute &attr)
{
	DEBUG_STREAM << "KiKe::read_Voltage(Tango::Attribute &attr) entering... "<< endl;

	attr.set_value(&voltage);
}

//+----------------------------------------------------------------------------
//
// method : 		KiKe::read_CurrentSetPoint
// 
// description : 	Extract real attribute values for CurrentSetPoint acquisition result.
//
//-----------------------------------------------------------------------------
void KiKe::read_CurrentSetPoint(Tango::Attribute &attr)
{
	DEBUG_STREAM << "KiKe::read_CurrentSetPoint(Tango::Attribute &attr) entering... "<< endl;

	attr.set_value(&current_set);
}



//+------------------------------------------------------------------
/**
 *	method:	KiKe::on
 *
 *	description:	method to execute "On"
 *	Switch powersupply ON.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::on()
{
	DEBUG_STREAM << "KiKe::on(): entering... !" << endl;
	//	Add your own code to control device here

	// Execute  KiKeThread.
	omni_thread *dt = new KiKeThread(this, mutex);
}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::off
 *
 *	description:	method to execute "Off"
 *	Switch powersupply OFF.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::off()
{
	DEBUG_STREAM << "KiKe::off(): entering... !" << endl;
	//	Add your own code to control device here

	Tango::DevState s = dev_state();

	if (KIKE_IS_ON)
	{
		standby_on_PLC();
		sleep(2);
		off_on_PLC();
		return;
	}
	
	if (KIKE_IN_STANDBY)
	{
		off_on_PLC();
		return;
	}
}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::reset
 *
 *	description:	method to execute "Reset"
 *	Reset the powersupply to a well known state.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::reset()
{
	DEBUG_STREAM << "KiKe::reset(): entering... !" << endl;

	//	Add your own code to control device here
	reset_on_PLC();

}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::standby
 *
 *	description:	method to execute "Standby"
 *	Set the kicker to standby
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::standby()
{
	DEBUG_STREAM << "KiKe::standby(): entering... !" << endl;
	Tango::DevState s = dev_state();
	
	if (KIKE_IS_ON)
	{
		standby_on_PLC();
		return;
	}

	if (KIKE_IS_OFF)
	{
		standby_on_PLC();
		sleep(2);
		set_current_on_PLC();
		return;
	}
}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::dev_state
 *
 *	description:	method to execute "State"
 *	This command gets the device state (stored in its <i>device_state</i> data member) and returns it to the caller.
 *
 * @return	State Code
 *
 */
//+------------------------------------------------------------------
Tango::DevState KiKe::dev_state()
{
	Tango::DevState	argout = DeviceImpl::dev_state();
	DEBUG_STREAM << "KiKe::dev_state(): entering... !" << endl;

	//	Add your own code to control device here
	
	// reset the internal kicker powersupply state
	
	kicker_state = Tango::UNKNOWN;
	
	// get the actual state
	if ((sequenceRunning != true) && (sequenceFailed != true))
	{
		if ( status_word & FAULT_STATE )
		{
			argout       = Tango::FAULT;
			kicker_state = Tango::FAULT;
		}
		else
		{
			if ( status_word & ON_STATE )
			{
				if (KIKE_IN_ALARM)
				{
					argout = Tango::ALARM;
				}
				else
				{
					argout = Tango::ON;
				}
				kicker_state = Tango::ON;
			}
			else
			{
				if ( status_word & STANDBY_STATE )
				{
					argout       = Tango::STANDBY;
					kicker_state = Tango::STANDBY;
				}
				else
				{
					if ( status_word & OFF_STATE )
					{
						argout       = Tango::OFF;
						kicker_state = Tango::OFF;
					}
					else
					{
						argout = Tango::UNKNOWN;
					}
				}
			}
		}
	}
	else
	{
		if (sequenceFailed == true)
		{
			argout = Tango::FAULT;
			kicker_state = Tango::FAULT;
		}
		else
		{
			//KiKeThread is running a sequence. Let's state to Moving.	
			argout = Tango::MOVING;
			kicker_state = Tango::MOVING;
		}
	}
	set_state(argout);
	return argout;
}


//+------------------------------------------------------------------
/**
 *	method:	KiKe::dev_status
 *
 *	description:	method to execute "Status"
 *	This command gets the device status (stored in its <i>device_status</i> data member) and returns it to the caller.
 *
 * @return	Status description
 *
 */
//+------------------------------------------------------------------
Tango::ConstDevString KiKe::dev_status()
{
	DEBUG_STREAM << "KiKe::dev_status(): entering... !" << endl;

	//	Add your own code to control device here
	string status = "";
	Tango::DevState ps_state = get_state();
	
	if (sequenceFailed == true)
	{
		status = "Sequence Failed :" + sequenceError + "\n";
	}
	
	switch (ps_state)
	{
		case Tango::FAULT:
			status += "PLC Fault word:\n";
			if ( fault_word1 & OIL_GROUP_FAULT )
				status += "\tOil group fault\n";
			if ( fault_word1 & OIL_LEVEL_LOW )
				status += "\tOil level too low\n";
			if ( fault_word1 & OIL_PRESSURE_HIGH )
				status += "\tOil pressure too high\n";
			if ( fault_word1 & OIL_TEMP_HIGH )
				status += "\tOil temperature too high\n";	
			if ( fault_word1 & OIL_FLOW_LOW )
				status += "\tOil flow too low\n";	
			if ( fault_word1 & THYRATRON_HEATERS )
				status += "\tThyratron heaters fault\n";
			if ( fault_word1 & BREAKER )
				status += "\tBreaker fault\n";
			if ( fault_word1 & PLC_24V_PS_FAILURE )
				status += "\t24 V PLC PS failure\n";
			if ( fault_word1 & CTRL_UNIT_PS_FAILURE )
				status += "\tControl unit PS failure\n";
			if ( fault_word1 & MAGNET_VACUUM_FAULT )
				status += "\tMagnet vacuum fault\n";
			if ( fault_word1 & MAG_SF6_PRESS_LOW )
				status += "\tMagnet SF6 pressure low\n";
			if ( fault_word1 & MAG_SF6BIS_PRESS_LOW )
				status += "\tMagnet SF6BIS pressure low (KI only)\n";
			if ( fault_word1 & MACHINE_INTERLOCK )
				status += "\tMachine interlock (KE only)\n";
			if ( fault_word1 & PSS_INTERLOCK )
				status += "\tPSS interlock (KE only)\n";
			if ( fault_word2 & FLT_TRIG_10HZ_FAILURE )
				status += "\t10 Hertz trigger failure\n";
			if ( fault_word2 & FLT_TRIG_1_10TH_HZ_FAILURE )
				status += "\t1/10Hertz trigger failure\n";
			if ( fault_word2 & FLT_SYNCHRO_FAULT )
				status += "\tSynchro fault\n";
			if ( fault_word2 & FLT_IMAG_LOW )
				status += "\tImag too low\n";
			if ( fault_word2 & FLT_IMAG_HIGH )
				status += "\tImag too high\n";
			if ( fault_word2 & FLT_VPFN_LOW )
				status += "\tVpfn too low\n";
			if ( fault_word2 & FLT_VPFN_HIGH )
				status += "\tVpfn too high\n";					
		break;
			
		case Tango::ALARM:
			status += "PLC Alarm word:\n";
			if (kicker_state == Tango::ON)
			{
				status += "\tOutput ON.\n";
			}
			else 
			if (kicker_state == Tango::OFF)
			{
				status += "\tOutput OFF.\n";
			}
			else
			if (kicker_state == Tango::ON)
			{
				status += "\tOutput STANDBY.\n";
			}

			if ( alarm_word & TRIG_10HZ_MISSING )
				status += "\t10 Hertz trigger missing\n";
			if ( alarm_word & TRIG_1_10TH_HZ_MISSING )
				status += "\t1/10 Hertz trigger missing\n";
			if ( alarm_word & IMAG_LOW )
				status += "\tImag too low\n";
			if ( alarm_word & IMAG_HIGH )
				status += "\tImag too high\n";
			if ( alarm_word & VPFN_LOW )
				status += "\tVpfn too low\n";
			if ( alarm_word & VPFN_HIGH )
				status += "\tVpfn too high\n";					
			if ( alarm_word & T1_TOO_SHORT )
				status += "\tT1 too short\n";
			if ( alarm_word & T1_TOO_LONG )
				status += "\tT1 too long\n";
			if ( alarm_word & T2_TOO_SHORT )
				status += "\tT1 too short\n";
			if ( alarm_word & T2_TOO_LONG )
				status += "\tT1 too long\n";			
		break;

		case Tango::MOVING:
			status = "Running a sequence.";
		break;

		default:
			status = "Device OK.";
		break;
	}
	set_status(status);
	return DeviceImpl::dev_status();
}




//+------------------------------------------------------------------
/**
 *	method:	KiKe::dbg__on
 *
 *	description:	method to execute "DBG_On"
 *	Send a ON command to the PLC, without any timing or state control. This is a debug function.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::dbg__on()
{
	DEBUG_STREAM << "KiKe::dbg__on(): entering... !" << endl;

	//	Add your own code to control device here
	on_on_PLC();
}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::dbg__off
 *
 *	description:	method to execute "DBG_Off"
 *	Send a OFF command to the PLC, without any timing or state control. This is a debug function.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::dbg__off()
{
	DEBUG_STREAM << "KiKe::dbg__off(): entering... !" << endl;

	//	Add your own code to control device here
	off_on_PLC();
}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::dbg__reset
 *
 *	description:	method to execute "DBG_Reset"
 *	Send a Reset command to the PLC, without any timing or state control. This is a debug function.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::dbg__reset()
{
	DEBUG_STREAM << "KiKe::dbg__reset(): entering... !" << endl;

	//	Add your own code to control device here
	reset_on_PLC();
}

//+------------------------------------------------------------------
/**
 *	method:	KiKe::dbg__standby
 *
 *	description:	method to execute "DBG_Standby"
 *	Send a Standby command to the PLC, without any timing or state control. This is a debug function.
 *
 *
 */
//+------------------------------------------------------------------
void KiKe::dbg__standby()
{
	DEBUG_STREAM << "KiKe::dbg__standby(): entering... !" << endl;

	//	Add your own code to control device here
	standby_on_PLC();
}



/*
 *	Write Memorised current value into the PLC. This Method also store the
 *  time of the operation into the last_current_write variable.
 */
void KiKe::set_current_on_PLC()
{
	// write the current set point 
	vector<short> address;
	address.push_back (registerBaseAddress + CURRENT_SET_OFFSET);
	address.push_back (1);
	address.push_back ((short)attr_Current_write);
	Tango::DeviceData argin;
	argin << address;
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);
	
	// apply the current set point
	address.clear();
	address.push_back (registerBaseAddress + CONTROL_WORD_OFFSET);
	address.push_back (1);
	address.push_back (APPLY_SETTING);
	
	argin << address;
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);
	
	// Store the date/time of last current write on PLC.
	time(&last_current_write);
	//cout << "set_current_on_PLC with value :"<< (short)attr_Current_write <<endl;	
}


/*
 * Simply send reset order on PLC.
 */
void KiKe::reset_on_PLC()
{
	vector<short> address;
	address.push_back (registerBaseAddress + CONTROL_WORD_OFFSET);
	address.push_back (1);
	address.push_back (RESET_CMD);
	
	Tango::DeviceData argin;
	argin << address;
		
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);

	// Clear error messages.
	omni_mutex_lock l(mutex);
	{
		sequenceError = "";
		sequenceFailed = false;
		sequenceRunning = false;
	}	
	//cout << "reset_on_PLC" <<endl;
}


/*
 * Simply send standby order on PLC.
 */
void KiKe::standby_on_PLC()
{
	vector<short> address;
	address.push_back (registerBaseAddress + CONTROL_WORD_OFFSET);
	address.push_back (1);
	address.push_back (STANDBY_CMD);
	
	Tango::DeviceData argin;
	argin << address;
		
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);
	//cout << "standby_on_PLC" <<endl;	
}

/*
 * Simply send ON order on PLC.
 */
void KiKe::on_on_PLC()
{

	vector<short> address;
	address.push_back (registerBaseAddress + CONTROL_WORD_OFFSET);
	address.push_back (1);
	address.push_back (ON_CMD);
	
	Tango::DeviceData argin;
	argin << address;
		
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);
	//cout << "on_on_PLC" <<endl;	
}

/*
 * Simply send OFF order on PLC.
 */
void KiKe::off_on_PLC()
{
	vector<short> address;
	address.push_back (registerBaseAddress + CONTROL_WORD_OFFSET);
	address.push_back (1);
	address.push_back (OFF_CMD);
	
	Tango::DeviceData argin;
	argin << address;
		
	modbus_obj->command_inout ("PresetMultipleRegisters", argin);
	//cout << "off_on_PLC" <<endl;
}



}	//	namespace
