/********************************************************/
/* RGA monitoring aplication                            */
/* File : xrga_gui.c                                    */
/* Descr: This module contains procedure to manage the  */
/*        Graphic User Interface                        */
/* Date : 06/00                                         */
/********************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <Xm/Xm.h>
#include <Xm/MwmUtil.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/CascadeB.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/BulletinB.h>
#include <Xm/Frame.h>
#include <Xm/Frame.h>
#include <Xm/ScrollBar.h>
#include <Xm/XrtGraph.h>
#include "xrga_gui.h"
#include "xrga_dev.h"
#include "xrga_tools.h"

#define MAX_ERR 100

/*
 * Convenience functions from utilities file.
  */
  extern void RegisterBxConverters(XtAppContext);
  extern XtPointer BX_CONVERT(Widget, char *, char *, int, Boolean *);
  extern XtPointer BX_DOUBLE(double);
  extern XtPointer BX_SINGLE(float);
  extern void BX_MENU_POST(Widget, XtPointer, XEvent *, Boolean *);
  extern Pixmap XPM_PIXMAP(Widget, char **);
  extern void BX_SET_BACKGROUND_COLOR(Widget, ArgList, Cardinal *, Pixel);

/******************************************************/
/* External declaration                               */
/******************************************************/
extern Widget errwin;
extern Widget errwin_txt;
extern Widget mesgwin;
extern Widget mesgwin_txt;
extern Widget mwin;
extern Widget mainsb;
extern Widget topLevelShell;
extern TRGA_channel RGA_channel[];
extern int current_rga_mode;

extern Widget gopt_win;
extern Widget gopt_xmin_text;
extern Widget gopt_ymin_text;
extern Widget gopt_xtitle_text;
extern Widget gopt_ytitle_text;
extern Widget gopt_xmax_text;
extern Widget gopt_ymax_text;
extern Widget gopt_header_text;
extern Widget gopt_xscale_frame;
extern Widget gopt_yscale_frame;
extern Widget gopt_xautoscale_tb;
extern Widget gopt_yautoscale_tb;
extern Widget gopt_ygrid_tb;
extern Widget gopt_xgrid_tb;

/******************************/
/* Global variable for module */
/******************************/

Cursor Arrow;
Cursor Rolex;
int NbError=0;
char *ErrorHistory[MAX_ERR];
Widget xrt_graph;
XrtData *graph_data;

XrtDataStyle MyDataStyle1[] = {
{XRT_LPAT_SOLID,XRT_FPAT_75_PERCENT,"Blue" ,1,XRT_POINT_NONE,"Black" , XRT_DEFAULT_POINT_SIZE },
{XRT_LPAT_SOLID,XRT_FPAT_SOLID,"Red",1,XRT_POINT_NONE,"Red"   , XRT_DEFAULT_POINT_SIZE }
};

XrtDataStyle *MyDataStyle1Ptr[] = { &MyDataStyle1[0],
                                    &MyDataStyle1[1],
                                    NULL };

static XrtTextHandle scan_text_handle=NULL;
static XrtTextHandle value_text_handle=NULL;
static XrtTextHandle zoom_text_handle=NULL;
int zoom_arm=0;
static  int     zoom_x1, zoom_y1, zoom_x2, zoom_y2 ;
static  float   old_maxx,old_maxy,old_minx,old_miny;
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define MAX(a, b) ((a) > (b) ? (a) : (b))
XmFontList  gfont;
Boolean     gfont_ok;

Widget graphwin;
static float *ys[2];

char translations[] = "#override        \n\
Shift<Btn2Down>:              ModifyStart(Translate)      \n\
Ctrl<Btn2Down>:               ModifyStart(Scale)      \n\
~Shift~Ctrl<Btn2Down>:        ModifyStart(Rotate)      \n\
Ctrl<Btn1Down>:               ModifyStart(Zoom) start_zoom()      \n\
<Key>r:                       ModifyStart(Reset) ViewReset() ModifyEnd(Reset) un_zoom()\n\
Shift<Btn2Motion>:            TranslateUpdate()      \n\
Ctrl<Btn2Motion>:             ScaleUpdate()      \n\
~Shift~Ctrl<Btn2Motion>:      RotateUpdate()      \n\
Ctrl<Btn1Motion>:             ZoomUpdate()      \n\
Shift<Btn2Up>:                ModifyEnd(Translate)    \n\
Ctrl<Btn2Up>:                 ModifyEnd(Scale)        \n\
~Shift~Ctrl<Btn2Up>:          ModifyEnd(Rotate)       \n\
Ctrl<Btn1Up>:                 ModifyEnd(Zoom) end_zoom()      \n\
Button1<Key>osfCancel:        ZoomCancel() ModifyEnd(Zoom)    \n\
<Btn1Down>:                   show_value()    \n\
<Btn1Up>:                     destroy_value() \n";

void show_value();
void destroy_value();
void start_zoom();
void end_zoom();
void un_zoom();

static XtActionsRec     my_actions[] = {
        {"show_value",          (XtActionProc) show_value              },
        {"destroy_value",       (XtActionProc) destroy_value           },
        {"start_zoom",          (XtActionProc) start_zoom              },
        {"end_zoom",            (XtActionProc) end_zoom                },
        {"un_zoom",             (XtActionProc) un_zoom                 },
        };


/*************************/
/* Graphg options        */
/*************************/
char  header[256];
char  xtitle[256];
char  ytitle[256];
Boolean y_autoscale=TRUE;
Boolean x_autoscale=TRUE;
float x_min=0.0;
float y_min=0.0;
float x_max=0.0;
float y_max=0.0;
Boolean x_grid=TRUE;
Boolean y_grid=TRUE;

Pixel grey_60;
Pixel green;
Pixel orange;
Pixel red;

/**********************************************************/
/* xrga_gui_init : Module initialisation                  */
/**********************************************************/
void xrga_gui_init() {
   Boolean ok;

   Arrow     = XCreateFontCursor( XtDisplay(mwin) , XC_top_left_arrow );
   Rolex     = XCreateFontCursor( XtDisplay(mwin) , XC_watch );
  
   gfont=(XmFontList)BX_CONVERT(mwin, "Courb12",
         XmRFontList, 0, &gfont_ok);

   strcpy(header,"");
   strcpy(xtitle,"mass");
   strcpy(ytitle,"Partial Pres");

   XtVaSetValues(mainsb,XmNminimum,0,NULL);
   XtVaSetValues(mainsb,XmNmaximum,10,NULL);
   XtVaSetValues(mainsb,XmNvalue,0,NULL);
   XtVaSetValues(mainsb,XmNpageIncrement,10,NULL);
   XtVaSetValues(mainsb,XmNincrement,10,NULL);
   XtVaSetValues(mainsb,XmNsliderSize,10,NULL);

   grey_60 = (Pixel)BX_CONVERT(mwin, "Grey60", XmRPixel, 0, &ok);
   green   = (Pixel)BX_CONVERT(mwin, "Green", XmRPixel, 0, &ok);
   orange  = (Pixel)BX_CONVERT(mwin, "Orange", XmRPixel, 0, &ok);
   red     = (Pixel)BX_CONVERT(mwin, "Red", XmRPixel, 0, &ok);

}

/***********************************************/
/* xrga_gui_set_list : Fill a scrolled list   */
/***********************************************/
void xrga_gui_set_list(Widget List,char **items,int nb_item)
{
  XmStringTable list;
  int i,nb;
  Dimension w1,h1;
  Dimension w2,h2;

  XtVaGetValues( XtParent(List) , XmNwidth , &w1 ,
                                  XmNheight , &h1 , NULL );
  XtVaGetValues( XtParent(XtParent(XtParent(List))) , XmNwidth , &w2 ,
                                  XmNheight , &h2 , NULL );

  XmListDeselectAllItems(List);

  if(nb_item>0)
  {
    list=(XmStringTable)malloc(nb_item*sizeof(XmString));
    for(i=0;i<nb_item;i++)
      list[i]=XmStringCreateLocalized(items[i]);

    XtVaSetValues( List , XmNitemCount , nb_item ,
                          XmNitems     , list , NULL);

    for(i=0;i<nb_item;i++)
      XmStringFree(list[i]);
    free(list);

  } else {
    XtVaSetValues( List , XmNitemCount , 0 , NULL);
  }

  XtVaSetValues( XtParent(List) , XmNwidth , w1 ,
                                  XmNheight , h1 , NULL );

  XtVaSetValues( XtParent(XtParent(XtParent(List))) , XmNwidth , w2 ,
                                  XmNheight , h2 , NULL );

}

void AddError(char *s)
{
  int i;

   if(NbError<MAX_ERR)
    {
      ErrorHistory[NbError]=(char *)malloc(strlen(s)+1);
      strcpy(ErrorHistory[NbError],s);
      NbError++;
    }else
    {
      free(ErrorHistory[0]);
      for(i=0;i<=MAX_ERR-2;i++) ErrorHistory[i]=ErrorHistory[i+1];
      ErrorHistory[MAX_ERR-1]=(char *)malloc(strlen(s)+1);
      strcpy(ErrorHistory[MAX_ERR-1],s);
    }
}

/************************************/
/* Show the error window            */
/************************************/
void xrga_gui_show_err()
{
  char tmp[32768];
  int  i;
  XmTextPosition pos;

  tmp[0]='\0';
  for(i=0;i<NbError;i++)
  {
    strcat(tmp,ErrorHistory[i]);
    strcat(tmp,"\n");
  }
  XtVaSetValues(errwin_txt,XmNvalue,tmp,NULL);
  pos=XmTextGetLastPosition( errwin_txt );
  XtVaSetValues(errwin_txt,XmNcursorPosition,pos-1,NULL);

  XtManageChild(errwin);
}

/************************************/
/* Show the message window          */
/************************************/
void xrga_gui_show_message(char *m)
{
  XtVaSetValues(mesgwin_txt,XmNvalue,m,NULL);
  XtManageChild(mesgwin);
}

/************************************/
/* Add an error and show the window */
/************************************/
void xrga_gui_adderr_man( char *s )
{
  AddError(s);
  xrga_gui_show_err();
}

/*******************************************/
/* Add an error whitout showing the window */
/*******************************************/
void xrga_gui_adderr( char *s )
{
  AddError(s);
}

/****************************************************/
/* xrga_gui_watch_cursor : Show watch cursor        */
/****************************************************/
void xrga_gui_watch_cursor(Widget w) {
  XDefineCursor( XtDisplay(w) , XtWindow(w) , Rolex );
  XFlush(XtDisplay(mwin));
}

/****************************************************/
/* xrga_gui_watch_arrow : Show arrow cursor         */
/****************************************************/
void xrga_gui_arrow_cursor(Widget w) {
  XDefineCursor( XtDisplay(w) , XtWindow(w) , Arrow );
  XFlush(XtDisplay(mwin));
}

/****************************************************/
/* Set scroll bar parameters                        */
/****************************************************/
void xrga_gui_set_scrollbar(int value,int pginc,int inc,int size,int max) {

  XtVaSetValues(mainsb,XmNvalue,0,NULL);

  XtVaSetValues(mainsb,XmNmaximum,max
                      ,XmNsliderSize,size
                      ,XmNpageIncrement,pginc
                      ,XmNincrement,inc
                      ,XmNvalue,value,NULL);
}

/****************************************************/
/* xrga_gui_set_background : Set the background     */
/* according to the state color                     */
/****************************************************/
void xrga_gui_set_relay_background(Widget w,int state)
{

  switch(state) {
    case 1:
      XtVaSetValues(w,XmNbackground,green,NULL);
      break;
    case 2:
      XtVaSetValues(w,XmNbackground,orange,NULL);
      break;
    case 3:
      XtVaSetValues(w,XmNbackground,red,NULL);
      break;
    default:
      XtVaSetValues(w,XmNbackground,grey_60,NULL);
      break;
  }

}

/****************************************************/
/* xrga_gui_set_background : Set the background     */
/* according to the state color                     */
/****************************************************/
void xrga_gui_set_cond_background(Widget w,int state)
{

  switch(state) {
    case 1:
      XtVaSetValues(w,XmNbackground,green,NULL);
      break;
    case 2:
      XtVaSetValues(w,XmNbackground,red,NULL);
      break;
    default:
      XtVaSetValues(w,XmNbackground,grey_60,NULL);
      break;
  }

}


/****************************************************/
/* Update the scroll bar according to xmin and xmax */
/****************************************************/
void xrga_gui_update_scrollbar()
{
  float xmin;
  float xmax;
  int nb;

  nb = graph_data->a.npoints;

  if( nb==0 ) {
    xrga_gui_set_scrollbar( 0,10,10,10,10 );
    return;
  }

  if( x_autoscale ) {
    xmin = 0.0;
    xmax = graph_data->a.data.xp[nb-1];
  } else {
    xmin = x_min;
    xmax = y_min;
  }

}

/****************************************************/
/* xrga_set_label : Set label with the given text   */
/****************************************************/
void xrga_gui_set_label(Widget w,char *m)
{
  XmString s;
  XmString s_old;

  XtVaGetValues(w,XmNlabelString,&s_old,NULL);
  s = XmStringCreateLocalized(m);

  if( !XmStringCompare( s,s_old ) ) XtVaSetValues(w,XmNlabelString,s,NULL);

  XmStringFree(s);
  XmStringFree(s_old);

}

/***************************/
/* Position option menu pb */
/***************************/
void xrga_gui_om_set(Widget om,int pos)
{
  Widget *lfils=NULL;
  Widget pdm;
  int fils;

  XtVaGetValues(om,XmNsubMenuId,&pdm,NULL);

  XtVaGetValues(pdm,XtNnumChildren,&fils,NULL);
  XtVaGetValues(pdm,XtNchildren,&lfils,NULL);

  if (pos < fils )
    XtVaSetValues( om , XmNmenuHistory , lfils[pos] , NULL );
  else {
    printf("Sorry, Fatal Error in OptionMenu set [%s]\n",XtName(om));
    exit(0);
  }
}

/***************************************/
/* Return Index of the positioned menu */
/***************************************/
int xrga_gui_om_pos(Widget om)
{
  Widget *lfils=NULL;
  Widget pdm;
  int fils;
  int found=0;
  int i=0;
  Widget pb;

  XtVaGetValues(om,XmNsubMenuId,&pdm,NULL);

  XtVaGetValues(pdm,XtNnumChildren,&fils,NULL);
  XtVaGetValues(pdm,XtNchildren,&lfils,NULL);
  XtVaGetValues(om ,XmNmenuHistory,&pb , NULL );

  while( !found && i<fils ) {
    found=(pb==lfils[i]);
    if(!found) i++;
  }

  if( found ) return i;
  else {
    printf("Sorry, Fatal Error in OptionMenu pos [%s]\n",XtName(om));
    exit(0);
  }
}

/****************************************************/
/* xrga_set_app_title : Set apliication title       */
/****************************************************/
void xrga_gui_set_app_title(char *m)
{
  char *h_strings[2];

  strcpy(header,m);
  h_strings[0] = header;
  h_strings[1] = NULL;
  XtVaSetValues(xrt_graph, XtNxrtHeaderStrings, h_strings, NULL);
}

/*******************************************************/
/* xrga_gui_set_bar : Set the porgress bar (0 to 100%) */
/*            Arg in: float p within [0,1]             */
/*******************************************************/
void xrga_gui_set_bar(Widget f,Widget b,float p)
{
  Dimension w;
  int       w2;

  XtVaGetValues(f,XmNwidth,&w,NULL);

  if(p<0.0) p=0.0;
  if(p>1.0) p=1.0;

  w2 = (int)( p*(float)w )-2;
  if(w2<1) w2=1;

  XtVaSetValues(b,XmNwidth,w2,NULL);
}

/*****************************************************/
/* Set the graph in Bar mode                         */
/*****************************************************/
void xrga_gui_set_graph_bar()
{
  XtVaSetValues(xrt_graph,XtNxrtType, XRT_TYPE_BAR,NULL);
}

/*****************************************************/
/* Set the graph in plot mode                        */
/*****************************************************/
void xrga_gui_set_graph_plot()
{
  XtVaSetValues(xrt_graph,XtNxrtType, XRT_TYPE_AREA,NULL);
}

/*****************************************/
/* Function called when graph is resized */
/*****************************************/
void xrga_gui_resize_graph(Widget w, XtPointer client_dat, XtPointer call_data )
{
  Dimension l;
  Dimension h;
  int x,y;

  XtVaGetValues( graphwin , XmNwidth  , &l , NULL);
  XtVaGetValues( graphwin , XmNheight , &h , NULL);

  if( scan_text_handle != NULL )
  {
    x = l/2 ;
    y = h/2 ;

    XrtTextAreaSetValues(scan_text_handle,
                         XRT_TEXT_ATTACH_PIXEL_X,       x,
                         XRT_TEXT_ATTACH_PIXEL_Y,       y,
			 NULL );
  }

}

/***************************************************/
/* Create the Xrt graph widget                     */
/***************************************************/
void xrga_gui_create_graph(Widget w)
{
  Arg         args[512];
  Cardinal    argcnt;
  Boolean     argok;


  
  graphwin=w;

  XtInitializeWidgetClass(xtXrtGraphWidgetClass);
  argcnt = 0;
  XtSetArg(args[argcnt],XmNx, 0); argcnt++;
  XtSetArg(args[argcnt],XmNy, 0); argcnt++;
  XtSetArg(args[argcnt],XmNwidth, 739); argcnt++;
  XtSetArg(args[argcnt],XmNheight, 517); argcnt++;
  XtSetArg(args[argcnt],XtNxrtType, XRT_TYPE_BAR); argcnt++;
  XtSetArg(args[argcnt],XtNxrtType2, XRT_TYPE_PLOT); argcnt++;
  XtSetArg(args[argcnt],XtNxrtXAnnotationMethod, XRT_ANNO_VALUES); argcnt++;
  XtSetArg(args[argcnt],XtNxrtBackgroundColor, "white"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtForegroundColor, "black"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtHeaderBackgroundColor, "white"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtHeaderForegroundColor, "black"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtFooterBackgroundColor, "white"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtFooterForegroundColor, "black"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtLegendBackgroundColor, "white"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtLegendForegroundColor, "black"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtDataAreaBackgroundColor, "white"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtDataStyles  ,MyDataStyle1Ptr ); argcnt++;
  XtSetArg(args[argcnt],XtNxrtYAxisLabelFormat, "0e-00"); argcnt++;
  XtSetArg(args[argcnt],XtNxrtYAxisLogarithmic,TRUE);argcnt++;
  XtSetArg(args[argcnt],XtNxrtXGridUseDefault, TRUE);argcnt++;
  XtSetArg(args[argcnt],XtNxrtYGridUseDefault, TRUE);argcnt++;
  XtSetArg(args[argcnt],XtNxrtXTitle, "mass");argcnt++;
  XtSetArg(args[argcnt],XtNxrtYTitle, "Partial Pres");argcnt++;
  XtSetArg(args[argcnt],XtNtranslations,
                        XtParseTranslationTable(translations)); argcnt++;
  XtSetArg(args[argcnt],XtNxrtLegendAnchor, XRT_ANCHOR_SOUTH ); argcnt++;

  xrt_graph = XtCreateWidget("graph",
                   xtXrtGraphWidgetClass,
                   w,
                   args,
                   argcnt);

  XtManageChild(xrt_graph);

  XtAppAddActions(XtWidgetToApplicationContext(w), my_actions, XtNumber(my_actions));
  XtAddCallback(xrt_graph, XtNxrtResizeCallback, xrga_gui_resize_graph , (XtPointer)0);

  graph_data=XrtMakeData(XRT_DATA_ARRAY, 1, 0, TRUE);
}

/************************************/
/* Display the graph options window */
/************************************/
void xrga_gui_display_graph_options()
{
  char tmp[128];
 
  /* Y params */
 
  XtVaSetValues( gopt_yautoscale_tb , XmNset , y_autoscale, NULL );
  sprintf(tmp,"%g",y_min);
  XtVaSetValues( gopt_ymin_text , XmNvalue , tmp , NULL);
  sprintf(tmp,"%g",y_max);
  XtVaSetValues( gopt_ymax_text , XmNvalue , tmp , NULL);

  if(y_autoscale) {
    XtSetSensitive(gopt_yscale_frame,False);
  } else {
    XtSetSensitive(gopt_yscale_frame,True);
  }
  XtVaSetValues( gopt_ytitle_text , XmNvalue , ytitle , NULL);
  XtVaSetValues( gopt_ygrid_tb , XmNset , y_grid, NULL );
 
  /* X params */
 
  XtVaSetValues( gopt_xautoscale_tb , XmNset , x_autoscale, NULL );
  sprintf(tmp,"%f",x_min);
  XtVaSetValues( gopt_xmin_text , XmNvalue , tmp , NULL);
  sprintf(tmp,"%f",x_max);
  XtVaSetValues( gopt_xmax_text , XmNvalue , tmp , NULL);

  if(x_autoscale) {
    XtSetSensitive(gopt_xscale_frame,False);
  } else {
    XtSetSensitive(gopt_xscale_frame,True);
  }
  XtVaSetValues( gopt_xtitle_text , XmNvalue , xtitle , NULL);
  XtVaSetValues( gopt_xgrid_tb , XmNset , x_grid, NULL );
 
  /*****************************************/
  /* Commona param                         */
  /*****************************************/
  
  XtVaSetValues( gopt_header_text , XmNvalue , header , NULL);

  XtManageChild( gopt_win );
}

/*********************************************************/
/* Scale the graph to the given range                    */
/*********************************************************/
void xrga_gui_apply_scale( float xmin,float xmax,
			   float ymin,float ymax )
{

  y_autoscale = FALSE;
  y_min = ymin;
  y_max = ymax;
  x_autoscale = FALSE;
  x_min = xmin;
  x_max = xmax;

  XtVaSetValues(xrt_graph, XtNxrtYAxisMax , XrtFloatToArgVal(y_max) , NULL);
  XtVaSetValues(xrt_graph, XtNxrtYAxisMin , XrtFloatToArgVal(y_min) , NULL);
  XtVaSetValues(xrt_graph, XtNxrtXAxisMax , XrtFloatToArgVal(x_max) , NULL);
  XtVaSetValues(xrt_graph, XtNxrtXAxisMin , XrtFloatToArgVal(x_min) , NULL);

}

/*********************************************************/
/* Get options from the grpah optins window and apply it */
/*********************************************************/
void xrga_gui_apply_graph_options()
{
  char *v;
  XmString s;
  float f;
  char *h_strings[2];

  /****************************/
  /* Update Y axis parameters */
  /****************************/

  XtVaGetValues( gopt_yautoscale_tb , XmNset , &y_autoscale, NULL );

  v=XmTextGetString( gopt_ymax_text );
  if( sscanf(v,"%f",&y_max) == 0 )
  {
    xrga_gui_adderr_man("Invalid y max value");
    y_autoscale=TRUE;
  }
  XtFree(v);

  v=XmTextGetString( gopt_ymin_text );
  if( sscanf(v,"%f",&y_min) == 0 )
  {
    xrga_gui_adderr_man("Invalid Y min value");
    y_autoscale=TRUE;
  }
  XtFree(v);

  if( !y_autoscale && y_max<=y_min ) {
    xrga_gui_adderr_man("Ymin must be lower than Ymax");
    y_autoscale=TRUE;
  }

  if(y_autoscale) {
    XtVaSetValues(xrt_graph, XtNxrtYAxisMinUseDefault, True, NULL);
    XtVaSetValues(xrt_graph, XtNxrtYAxisMaxUseDefault, True, NULL);
  } else {
    XtVaSetValues(xrt_graph, XtNxrtYAxisMax , XrtFloatToArgVal(y_max) , NULL);
    XtVaSetValues(xrt_graph, XtNxrtYAxisMin , XrtFloatToArgVal(y_min) , NULL);
  }

  v=XmTextGetString( gopt_ytitle_text );
  strcpy( ytitle , v);
  XtFree(v);
  XtVaSetValues(xrt_graph, XtNxrtYTitle , ytitle , NULL);
 
  XtVaGetValues( gopt_ygrid_tb , XmNset , &y_grid, NULL );
  if (y_grid)
    XtVaSetValues(xrt_graph, XtNxrtYGridUseDefault, TRUE, NULL) ;
  else
    XtVaSetValues(xrt_graph, XtNxrtYGrid, XrtFloatToArgVal(0.0), NULL) ;
 
  /*************************************/
  /* Update X axis parameters          */
  /*************************************/

  XtVaGetValues( gopt_xautoscale_tb , XmNset , &x_autoscale , NULL );

  v=XmTextGetString( gopt_xmax_text );
  if( sscanf(v,"%f",&x_max) == 0 )
  {
    xrga_gui_adderr_man("Invalid x max value");
    x_autoscale=TRUE;
  }
  XtFree(v);

  v=XmTextGetString( gopt_xmin_text );
  if( sscanf(v,"%f",&x_min) == 0 )
  {
    xrga_gui_adderr_man("Invalid X min value");
    x_autoscale=TRUE;
  }
  XtFree(v);

  if( !x_autoscale && x_max<=x_min ) {
    xrga_gui_adderr_man("Xmin must be lower than Xmax");
    x_autoscale=TRUE;
  }

  if(x_autoscale) {
    XtVaSetValues(xrt_graph, XtNxrtXAxisMinUseDefault, True, NULL);
    XtVaSetValues(xrt_graph, XtNxrtXAxisMaxUseDefault, True, NULL);
  } else {
    XtVaSetValues(xrt_graph, XtNxrtXAxisMax , XrtFloatToArgVal(x_max) , NULL);
    XtVaSetValues(xrt_graph, XtNxrtXAxisMin , XrtFloatToArgVal(x_min) , NULL);
  }

  v=XmTextGetString( gopt_xtitle_text );
  strcpy( xtitle , v);
  XtFree(v);
  XtVaSetValues(xrt_graph, XtNxrtXTitle , xtitle , NULL);

  XtVaGetValues( gopt_xgrid_tb , XmNset , &x_grid, NULL );
  if (x_grid)
    XtVaSetValues(xrt_graph, XtNxrtXGridUseDefault, TRUE, NULL) ;
  else
    XtVaSetValues(xrt_graph, XtNxrtXGrid, XrtFloatToArgVal(0.0), NULL) ;
  
  /*****************************************/
  /* Commona param                         */
  /*****************************************/
  
  v=XmTextGetString( gopt_header_text );
  strcpy( header , v);
  XtFree(v);
  h_strings[0] = header;
  h_strings[1] = NULL;
  XtVaSetValues(xrt_graph, XtNxrtHeaderStrings, h_strings, NULL);

}

/****************************************************************************
*  Function Name:  xrga_gui_update_graph
*
*  Description:    This function update xrt graph.
*  Args:           float *mass   mass values
*                  float *values pressure values
*                  int nb number of value
****************************************************************************/
void  xrga_gui_update_graph(int nb,float *mass,float *values)
{
  ys[0]=values;
  
  if (scan_text_handle != NULL)
  {
    XrtTextDestroy(xrt_graph, scan_text_handle);
    scan_text_handle=NULL;
  }

  graph_data->a.npoints=nb;
  graph_data->a.data.xp=mass;
  graph_data->a.data.yp=ys;

  XtVaSetValues(xrt_graph,XtNxrtData , graph_data , NULL);
}

/**************************************************************/
/* Clear the grpah                                            */
/**************************************************************/
void xrga_gui_clear_graph()
{
  un_zoom(xrt_graph,NULL);
  graph_data->a.npoints=0;
  XtVaSetValues(xrt_graph,XtNxrtData , graph_data , NULL);
}

/**************************************************************/
/* Display a text box indicating that the scan is in progress */
/**************************************************************/
void xrga_gui_scan_in_progress()
{

  static  char   *txt1="Scanning";
  static  char   *txt2="in progress";
  static  char   *txts[3];

  if (scan_text_handle != NULL)
  {
    XrtTextDestroy(xrt_graph, scan_text_handle);
    scan_text_handle=NULL;
  }

  if (value_text_handle != NULL)
  {
    XrtTextDestroy(xrt_graph, value_text_handle);
    value_text_handle=NULL;
  }

  txts[0]=txt1;
  txts[1]=txt2;
  txts[2]=NULL;

  scan_text_handle = XrtTextAreaCreate(xrt_graph);

  XrtTextAreaSetValues(scan_text_handle,
                             XRT_TEXT_NAME,                 "area_0",
                             XRT_TEXT_ATTACH_TYPE,          XRT_TEXT_ATTACH_PIXEL,
                             XRT_TEXT_ATTACH_PIXEL_X,       -100,
                             XRT_TEXT_ATTACH_PIXEL_Y,       -100,
                             XRT_TEXT_STRINGS,              txts,
                             XRT_TEXT_ANCHOR,               XRT_ANCHOR_HOME,
                             XRT_TEXT_IS_CONNECTED,         FALSE,
                             XRT_TEXT_FOREGROUND_COLOR,     "Black",
                             XRT_TEXT_BACKGROUND_COLOR,     "Light Grey",
                             XRT_TEXT_BORDER,               XRT_BORDER_BEVEL,
                             XRT_TEXT_BORDER_WIDTH,         5,
                             NULL);

  xrga_gui_resize_graph(NULL,NULL,NULL);
}

/******************************************************************/
/* Translation procedures (Zoom , value display)                  */ 
/******************************************************************/

void show_value(Widget w, XEvent *event)
{
  XrtRegion             pr ;
  XrtPickResult         p ;
  XButtonEvent          *be = (XButtonEvent *) event;
  float                 valx;
  float                 valy;
  float                 min_x;
  float                 min_y;
  float                 max_x;
  float                 max_y;
  float                 mil_x;
  float                 mil_y;
  int                   id;

static char *jpl="\n";
static char valuex[64];
static char valuey[64];
static char mult[64];
static char enable[64];
static char range[64];
static char arange[64];

static char *valuestr[10];

  sprintf(mult,  "Multiplier :---");
  sprintf(enable,"Channel    :---");
  sprintf(range, "Range      :---");

  if (value_text_handle != NULL)
  {
    XrtTextDestroy(xrt_graph, value_text_handle);
    value_text_handle= NULL;
  }

  pr = XrtPick(xrt_graph, XRT_DATASET1 , be->x, be->y, &p,
                                XRT_XFOCUS | XRT_YFOCUS);

  if ((pr == XRT_RGN_IN_GRAPH) && (p.set != XRT_RGN_NOWHERE)) {

    XtVaGetValues(w, XtNxrtXAxisMin, &min_x,
                     XtNxrtYAxisMin, &min_y,
                     XtNxrtXAxisMax, &max_x,
                     XtNxrtYAxisMax, &max_y,
                     NULL);

    valx=graph_data->a.data.xp[p.point];
    valy=graph_data->a.data.yp[p.set][p.point];

    sprintf(valuex,  "Mass : %g",valx);
    sprintf(valuey,  "Pres : %g",valy);

    valuestr[0]=valuex;
    valuestr[1]=valuey;

    switch(current_rga_mode) {
      case 3: /* Peak Jump */
        
	id = xrga_dev_get_mass_indx((int)valx);

	if( id!=-1 ) {

	  sprintf(mult,"Multiplier :");
	  if( RGA_channel[id].Multiplier )
	    strcat(mult,"On");
	  else
	    strcat(mult,"Off");

	  sprintf(enable,"Channel    :");
	  if( RGA_channel[id].Enable )
	    strcat(enable,"Enable");
  	  else
	    strcat(enable,"Disable");

	  sprintf(range,"Range      :%d",RGA_channel[id].Range );

	  sprintf(arange,"Auto Range :");
	  if( RGA_channel[id].AutoRange )
	    strcat(arange,"Yes");
  	  else
	    strcat(arange,"No");
	}
	valuestr[2]=jpl;
	valuestr[3]=mult;
	valuestr[4]=enable;
	valuestr[5]=range;
	valuestr[6]=arange;
	valuestr[7]=NULL;
	break;

      default:
	valuestr[2]=NULL;
    }
  
    value_text_handle = XrtTextAreaCreate(xrt_graph);


    if( gfont_ok ) {
        XrtTextAreaSetValues(value_text_handle,
            XRT_TEXT_FONT, gfont,
            NULL);
    }

    mil_y = ( log10(min_y) + log10(max_y) ) / 2.0;
    mil_x = ( min_x + max_x ) / 2.0;

    if( log10(valy) > mil_y )
    {
      if( valx > mil_x ) {
        XrtTextAreaSetValues(value_text_handle,
            XRT_TEXT_ANCHOR, XRT_ANCHOR_SOUTHWEST,
            NULL);
      } else {
        XrtTextAreaSetValues(value_text_handle,
            XRT_TEXT_ANCHOR, XRT_ANCHOR_SOUTHEAST,
            NULL);
      }
    } else {
      if( valx > mil_x ) {
        XrtTextAreaSetValues(value_text_handle,
            XRT_TEXT_ANCHOR, XRT_ANCHOR_NORTHWEST,
            NULL);
      } else {
        XrtTextAreaSetValues(value_text_handle,
            XRT_TEXT_ANCHOR, XRT_ANCHOR_NORTHEAST,
            NULL);
      }
    }

    XrtTextAreaSetValues(value_text_handle,
                         XRT_TEXT_NAME,                 "area_1",
                         XRT_TEXT_ATTACH_TYPE,          XRT_TEXT_ATTACH_DATA,
                         XRT_TEXT_ATTACH_SET,           p.set,
                         XRT_TEXT_ATTACH_POINT,         p.point,
                         XRT_TEXT_STRINGS,              valuestr,
                         XRT_TEXT_IS_CONNECTED,         TRUE,
                         XRT_TEXT_FOREGROUND_COLOR,     "Black",
                         XRT_TEXT_BACKGROUND_COLOR,     "Light Grey",
                         XRT_TEXT_BORDER,               XRT_BORDER_ETCHED_IN,
                         XRT_TEXT_BORDER_WIDTH,         3,
                         XRT_TEXT_OFFSET,               10,
                         XRT_TEXT_ADJUST,               XRT_ADJUST_LEFT,
                         NULL);
  }
}


void destroy_value(Widget w, XEvent *event)
{
  if (value_text_handle != NULL)
  {
    XrtTextDestroy(xrt_graph, value_text_handle) ;
    value_text_handle=NULL;
  }
}


void un_zoom(Widget w, XEvent *event)
{
  if (zoom_text_handle != NULL)
  {
    XrtTextDestroy(xrt_graph, zoom_text_handle);
    zoom_text_handle=NULL;
  }

  if (zoom_arm) {

	if(x_autoscale) {
          XtVaSetValues(w,
                  XtNxrtXAxisMinUseDefault , True ,
                  XtNxrtXAxisMaxUseDefault , True ,
                  NULL);
        } else {
          XtVaSetValues(w,
                XtNxrtXAxisMin, XrtFloatToArgVal(old_minx),
                XtNxrtXAxisMax, XrtFloatToArgVal(old_maxx),
                NULL);
        }

        if(y_autoscale) {
          XtVaSetValues(w,
                XtNxrtYAxisMinUseDefault,True,
                XtNxrtYAxisMaxUseDefault,True,
                NULL);
        } else {
          XtVaSetValues(w,
                XtNxrtYAxisMin, XrtFloatToArgVal(old_miny),
                XtNxrtYAxisMax, XrtFloatToArgVal(old_maxy),
                NULL);
        }

        zoom_arm=0;
  }
}



void start_zoom(Widget w, XEvent *event)
{
   XButtonEvent    *be = (XButtonEvent *) event;

   zoom_x1 = be->x;
   zoom_y1 = be->y;
}


/*======================================================================
 Function:      void end_zoom()

 Description:   Applies the zoom. Also memorizes the situation before the
                zoom so that it is possible to go back to it.
 =======================================================================*/
void end_zoom(Widget w, XEvent *event)
{
        XrtMapResult    min1, max1 ;
        XrtMapResult    min2, max2 ;
        int             top, bottom, left, right ;
        XButtonEvent    *be = (XButtonEvent *) event;
        static  char   *zoom_help1="[r] for zoom back";
        static  char   *zoom_help[2];

        zoom_help[0]=zoom_help1;
        zoom_help[1]=NULL;

   if( graph_data->a.npoints>0 ) {

        min2.x=0;
        min2.y=0;
        max2.x=0;
        max2.y=0;

        zoom_x2 = be->x;
        zoom_y2 = be->y;

        left    = MIN(zoom_x1, zoom_x2) ;
        right   = MAX(zoom_x1, zoom_x2) ;
        top     = MIN(zoom_y1, zoom_y2) ;
        bottom  = MAX(zoom_y1, zoom_y2) ;

        XrtMap(w, 1, left, bottom, &min1) ;
        XrtMap(w, 1, right, top, &max1) ;

        if( min1.x>1e30 || min1.y>1e30 ||
            min2.x>1e30 || min2.y>1e30 ||
            max1.x>1e30 || max1.y>1e30 ||
            max2.x>1e30 || max2.y>1e30 )
        {
            xrga_gui_adderr_man("Zoom scaling is out of range");
            return;
        }

        if(!zoom_arm)
        {
          XtVaGetValues(w,
                XtNxrtXAxisMin, &old_minx,
                XtNxrtYAxisMin, &old_miny,
                XtNxrtXAxisMax, &old_maxx,
                XtNxrtYAxisMax, &old_maxy,
                NULL);
        }

        XtVaSetValues(w,
                XtNxrtXAxisMin, XrtFloatToArgVal(min1.x),
                XtNxrtXAxisMax, XrtFloatToArgVal(max1.x),
                NULL);

        XtVaSetValues(w,
                XtNxrtYAxisMin, XrtFloatToArgVal(min1.y),
                XtNxrtYAxisMax, XrtFloatToArgVal(max1.y),
                NULL);


      if( zoom_text_handle==NULL ) {

        zoom_text_handle = XrtTextAreaCreate(xrt_graph);

        XrtTextAreaSetValues(zoom_text_handle,
                         XRT_TEXT_NAME,                 "area_2",
                         XRT_TEXT_ATTACH_TYPE,          XRT_TEXT_ATTACH_PIXEL,
                         XRT_TEXT_ATTACH_PIXEL_X,       70,
                         XRT_TEXT_ATTACH_PIXEL_Y,       20,
                         XRT_TEXT_STRINGS,              zoom_help,
                         XRT_TEXT_IS_CONNECTED,         FALSE,
                         XRT_TEXT_FOREGROUND_COLOR,     "Black",
                         XRT_TEXT_BACKGROUND_COLOR,     "Light Grey",
                         XRT_TEXT_BORDER,               XRT_BORDER_ETCHED_IN,
                         XRT_TEXT_BORDER_WIDTH,         2,
                         XRT_TEXT_OFFSET,               0,
                         XRT_TEXT_ADJUST,               XRT_ADJUST_LEFT,
                         NULL);
      }

      zoom_arm=1;
   }
}

