/*----------------------------------------------------------------------
 File : xrga_dev.c
 Description : This module contains procedures for RGA and PPAS device 
               management.

 Author : JL PONS
 Date :   06/2000
 Project : RGA and PPAS application

-----------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include "xrga_dev.h"
#include "xrga_gui.h"
#include "xrga_tools.h"
#include <DevRgaCmds.h>

/***************************************************************/
/* Global for module                                           */
/***************************************************************/
/* Tree for rga device browsing */
static struct node  *ExportedTree=NULL;

/* Device server pointer */
devserver RGAds=NULL;
devserver PPASds=NULL;

/* RGA channel parameters */
TRGA_channel RGA_channel[12];

/* Current scannig mode (RGA) */
int current_rga_mode;

/* Current parameters */

int cur_multiplier;
int cur_accuracy;
int cur_firstmass;
int cur_lastmass;
int cur_range;
int cur_autorange;
int cur_multvolt;
int cur_multgain;
int cur_filament;

/**************************************************************/
/* Initialise sgmon_dev module                                */
/**************************************************************/
void xrga_dev_init()
{
  long error=0;
  char *str_err;

  /* Make connection to Data base server */

  if ( db_import (&error) < 0 )
  {
    str_err=dev_error_str(error);
    xrga_gui_adderr_man(str_err);
    free(str_err);
  }

}

/**********************************************************/
/* init_dev_list : Initialise the tree of exported device */
/*                 device (return -1 when failed else 0)  */
/**********************************************************/
long xrga_dev_init_dev_list() {

long error=0;
char tmp[256];
char *err_str;
unsigned int i;
char **dev_names=NULL;
unsigned int dev_nb=0;
db_resource rga_l[1];
DevVarStringArray rga_list={0,NULL}; 


  FreeTree(ExportedTree);
  ExportedTree=NULL;

  /* Get list of exported device */

  if( db_getdevexp( "*/v-rga/*" , &dev_names , &dev_nb , &error )<0 )
  {
      err_str=dev_error_str(error);
      sprintf(tmp,"db_devgetexp() failed %s",
                   err_str);
      xrga_gui_adderr_man(tmp);
      free(err_str);
      return -1;
  }

  for( i=0 ; i<dev_nb ; i++) AddTree( &ExportedTree,dev_names[i] );

  db_freedevexp( dev_names );

  /* Also add devices coming from sys/xrga/default/rgalist */
  rga_l[0].resource_adr  = &( rga_list );
  rga_l[0].resource_type = D_VAR_STRINGARR;
  rga_l[0].resource_name = "rgalist";

  if( db_getresource( "sys/xrga/default" , rga_l , 1 , &error ) < 0 )
  {
      err_str=dev_error_str(error);
      sprintf(tmp,"db_getresource() failed %s",
                   err_str);
      xrga_gui_adderr_man(tmp);
      free(err_str);
      return -1;
  }

  for( i=0 ; i<rga_list.length ; i++) AddTree( &ExportedTree,rga_list.sequence[i] );

  return 1;
}

/****************************************************************/
/*dev_get_domain_list : Return the list of domain               */
/*                      Allocated memory must be free by caller */
/****************************************************************/
void xrga_dev_get_domain_list(char ***domain_list,
                               long *domain_nb)
{
  char *s[1024];
  int  nb=0;
  int  i;
  struct node *p_domain;

  p_domain = ExportedTree;
  while( p_domain !=NULL )
  {
    s[nb]=(char *)malloc( strlen(p_domain->name) + 1);
    strcpy(s[nb],p_domain->name);
    nb++;
    p_domain=p_domain->next;
  }

  *domain_list=(char **)malloc( nb * sizeof(char *) );
  for(i=0;i<nb;i++) (*domain_list)[i]=s[i];
  *domain_nb=nb;

}

/****************************************************************/
/*    get_family_list : Return the list of family               */
/*                      Allocated memory must be free by caller */
/*                      return -1 when failed else 0            */
/****************************************************************/
long xrga_dev_get_family_list(char *domain,
                      char ***family_list,
                        long *family_nb)
{
  char *s[1024];
  int  nb=0;
  int  i,found=0;
  struct node *p_domain;
  struct node *p_family;
  long error;

  p_domain = ExportedTree;
  found=0;

  while( p_domain !=NULL && !found )
  {
    found = (strcasecmp(p_domain->name,domain)==0);
    if(!found) p_domain=p_domain->next;
  }

  if( !found ) {
    xrga_gui_adderr_man("Invalid domain name.");
    return -1;
  }

  p_family = p_domain->sub_list;

  while( p_family != NULL )
  {
    s[nb]=(char *)malloc( strlen(p_family->name) + 1);
    strcpy(s[nb],p_family->name);
    nb++;
    p_family=p_family->next;
  }

  *family_list=(char **)malloc( nb * sizeof(char *) );
  for(i=0;i<nb;i++) (*family_list)[i]=s[i];
  *family_nb=nb;
  return 0;
}

/****************************************************************/
/*    get_member_list : Return the list of member               */
/*                      Allocated memory must be free by caller */
/*                      return -1 when failed else 0            */
/****************************************************************/
long xrga_dev_get_member_list(char *domain,
                        char *family,
                      char ***member_list,
                        long *member_nb)
{
  char *s[1024];
  int  nb=0;
  int  i,found=0;
  struct node *p_domain;
  struct node *p_family;
  struct node *p_member;
  long error;

  p_domain = ExportedTree;
  found=0;

  while( p_domain !=NULL && !found )
  {
    found = (strcasecmp(p_domain->name,domain)==0);
    if(!found) p_domain=p_domain->next;
  }

  if( !found ) {
    xrga_gui_adderr_man("Invalid domain name.");
    return -1;
  }

  p_family = p_domain->sub_list;
  found=0;

  while( p_family!=NULL && !found )
  {
    found = (strcasecmp(p_family->name,family)==0);
    if(!found) p_family=p_family->next;
  }

  if( !found ) {
    xrga_gui_adderr_man("Invalid family name.");
    return -1;
  }

  p_member=p_family->sub_list;

  while( p_member != NULL )
  {
    s[nb]=(char *)malloc( strlen(p_member->name) + 1);
    strcpy(s[nb],p_member->name);
    nb++;
    p_member=p_member->next;
  }

  *member_list=(char **)malloc( nb * sizeof(char *) );
  for(i=0;i<nb;i++) (*member_list)[i]=s[i];
  *member_nb=nb;
  return 0;
}

/*********************************************************/
/* import_rga() : Import the RGA device server and check */
/* if an associated ppas exists                          */
/*********************************************************/
void xrga_dev_import_rga(char *d,char *f,char *m)
{
  char rga_name[128];
  char ppas_name[128];
  char tmp[256];
  char *str_err=NULL;
  long error=0;
  struct timeval timeout;
  DevVarShortArray param={0,NULL};
  DevVarShortArray channels={0,NULL};
  char *nethost;

  /*********************************/
  /* Free OLD pointer              */
  /*********************************/
  if( RGAds!=NULL )  
    { dev_free(RGAds , &error);RGAds=NULL; }
  if( PPASds!=NULL ) 
    { dev_free(PPASds , &error);PPASds=NULL; }


  nethost = GetNethost(ExportedTree , d);
  if( nethost==NULL ) {
    sprintf(rga_name,"%s/%s/%s",d,f,m);
    sprintf(ppas_name,"%s/v-ppas/%s",d,m);
  } else {
    sprintf(rga_name,"//%s/%s/%s/%s",nethost,d,f,m);
    sprintf(ppas_name,"//%s/%s/v-ppas/%s",nethost,d,m);
  }

  /**************************/
  /* Import RGA             */
  /**************************/

  if( dev_import( rga_name , 0 , &RGAds , &error) <0 )
  {
    str_err=dev_error_str(error);
    sprintf(tmp,"dev_import(%s) failed %s",rga_name,str_err);
    free(str_err);
    xrga_gui_adderr_man(tmp);
    RGAds=NULL;
    return;
  }

  timeout.tv_sec=6;
  timeout.tv_usec=0;

  dev_rpc_timeout ( RGAds , CLSET_TIMEOUT , &timeout, &error);
  
  if( dev_putget( RGAds , DevGetOperationMode , NULL , D_VOID_TYPE ,
		 &param , D_VAR_SHORTARR , &error) <0 )
  {
    str_err=dev_error_str(error);
    sprintf(tmp,"DevGetOperationMode(%s) failed %s",RGAds->device_name,str_err);
    free(str_err);
    xrga_gui_adderr(tmp);
    return;
  }
  current_rga_mode=param.sequence[0];

  /* Initialise parameters if we are in Peak Jump Mode */

  if( current_rga_mode == 3 || current_rga_mode == 9 )
  {
   
    if( dev_putget( RGAds , DevGetChannelParameters , NULL , D_VOID_TYPE ,
                    &channels , D_VAR_SHORTARR , &error )<0 )
    {
      str_err=dev_error_str(error);
      sprintf(tmp,"DevGetChannelParameters(%s) failed %s",
	      RGAds->device_name,str_err);
      free(str_err);
      xrga_gui_adderr_man(tmp);
      return;
    }

    memcpy( RGA_channel , channels.sequence , 12*sizeof(TRGA_channel) );
    dev_xdrfree(D_VAR_FLOATARR , &channels , &error);

  }

  dev_xdrfree( D_VAR_SHORTARR , &(param) , &error);

  /*****************************************/
  /* Import PPAS                           */
  /*****************************************/
  if( dev_import( ppas_name , 0 , &PPASds , &error) <0 )
  {
    PPASds=NULL;
    return;
  }

  sprintf(tmp,"%s PPAS found.",ppas_name);
  xrga_gui_show_message(tmp);

}

/****************************************************/
/* get_rga_status : Return status of the RGA device */
/*                  All string must be preallocated */
/****************************************************/
void xrga_dev_get_rga_status( char *mode ,
			      int  *filament1 ,
			      int  *filament2 ,
			      int  *multiplier ,
			      char *multiplier_volt ,
			      char *multiplier_gain ,
			      int  *autorange ,
                              char *range,
			      char *accuracy ,
			      char *start_mass ,
			      char *end_mass )
{
  long error=0;
  DevVarShortArray param={0,NULL};
  char tmp[256];
  char *str_err;
  DevShort volt;

  if( RGAds==NULL ) {
    return;
  }

  if( dev_putget( RGAds , DevGetOperationMode , NULL , D_VOID_TYPE ,
		 &param , D_VAR_SHORTARR , &error) <0 )
  {
    str_err=dev_error_str(error);
    sprintf(tmp,"DevGetOperationMode(%s) failed %s",RGAds->device_name,str_err);
    free(str_err);
    xrga_gui_adderr(tmp);
    return;
  }
  current_rga_mode=param.sequence[0];


  /* RGA MODE */

  switch( current_rga_mode ) {
  case 0:
    strcpy( mode , "Scanning OFF" ); break;
  case 1:
    strcpy( mode , "Leak Check" ); break;
  case 2:
    strcpy( mode , "Bar Chart" ); break;
  case 3:
    strcpy( mode , "Peak Jump" ); break;
  case 4:
    strcpy( mode , "Analog" ); break;
  case 5:
    strcpy( mode , "Single Peak" ); break;
  case 6:
    strcpy( mode , "Fast Scan" ); break;
  case 7:
    strcpy( mode , "Degas" ); break;
  case 9:
    strcpy( mode , "PPAS mode" ); break;
  }

  /* RGA Multiplier GAIN */
  sprintf(multiplier_gain,"%d",cur_multgain=param.sequence[1]);
  *multiplier = (int)param.sequence[4]+1;
  cur_multiplier = *multiplier-1;

  /* Autorange and range */

  switch( current_rga_mode ) {
    case 0:
    case 3:
      break;
    case 9:
      sprintf(start_mass,"%d",cur_firstmass);
      sprintf(end_mass  ,"%d",cur_lastmass);
      break;
    default:
      sprintf(start_mass,"%d",cur_firstmass=param.sequence[7]);
      sprintf(end_mass  ,"%d",cur_lastmass=param.sequence[8]);
      sprintf(range     ,"%d",cur_range=param.sequence[9]);
      *autorange=(cur_autorange=(int)param.sequence[10])+1;
  }


  /* Accuracy */
  sprintf(accuracy,"%d",cur_accuracy=param.sequence[6]);

  dev_xdrfree( D_VAR_SHORTARR , &(param) , &error);
  param.length=0;
  param.sequence=NULL;

  /*************************************/
  /* Filament state                    */
  /*************************************/

  if( dev_putget( RGAds , DevFilamentState , NULL , D_VOID_TYPE ,
		 &param , D_VAR_SHORTARR , &error) <0 )
  {
    str_err=dev_error_str(error);
    sprintf(tmp,"DevFilamentState(%s) failed %s",RGAds->device_name,str_err);
    free(str_err);
    xrga_gui_adderr(tmp);
    return;
  }

  if( param.sequence[1]==2 )
  {

    if( param.sequence[0]==1 ) {
     *filament1=2;
     *filament2=1;
     cur_filament=1;
    } else {
     *filament1=1;
     *filament2=2;
     cur_filament=2;
    }

  } else {
     *filament1=1;
     *filament2=1;
     cur_filament=0;
  }

  /**************************************************/
  /* multiplier voltage                             */
  /**************************************************/

  if( dev_putget( RGAds , DevReadVoltage , NULL , D_VOID_TYPE ,
		 &volt , D_SHORT_TYPE , &error) <0 )
  {
    str_err=dev_error_str(error);
    sprintf(tmp,"DevReadVoltage(%s) failed %s",RGAds->device_name,str_err);
    free(str_err);
    xrga_gui_adderr(tmp);
    return;
  }

  sprintf(multiplier_volt,"%d",cur_multvolt=volt);

}


/****************************************************************************
 *  Function Name:  xrga_dev_get_data
 *
 *  Description:    Return the data from RGA server acording to mode
 *  Args in:        none
 *  Args out:       int   *nb     : number of values
 *		    float *mass   : time vector or freq vector [256 val max]
 *                                  must be allocated
 *                  float *values : values [256 val max] must be allocated
 *  return 1 when success else 0
 ****************************************************************************/
int xrga_dev_get_data(int *nb,float *mass,float *values)
{
  long error=0;
  char tmp[256];
  char *err_str;
  int i;
  float f=0.0;
  float nbv=0.0;
  double limit;
  DevVarFloatArray argout={0,NULL};
  DevShort m;


  if( current_rga_mode != 9 ) {
  
    if(RGAds==NULL) {
     *nb=0;
      return 1;
    }

    if( dev_putget( RGAds , DevReadValues , NULL , D_VOID_TYPE ,
                         &argout , D_VAR_FLOATARR , &error)<0)
    {
     err_str=dev_error_str(error);
     sprintf(tmp,"DevReadValues(%s) failed %s",RGAds->device_name,err_str);
     free(err_str);
     xrga_gui_adderr_man(tmp);
     return 0;
    }

    *nb=argout.length;

    /* Supress the last signal (state) when Peak Jump */
    if( *nb==13 ) *nb=*nb-1;
    nbv=(float)*nb;

    sprintf(tmp,"5.0e-%d",cur_range+2);
    limit = atof(tmp);

    for(i=0;i<*nb;i++)
    {

      switch( current_rga_mode ) {
        case 3: /* Peak jump */
        case 9: /* PPAS mode */
          mass[i]=RGA_channel[i].mass;
          values[i]=argout.sequence[i];
          break;
        case 4: /* Analog */
          mass[i]=cur_firstmass + (float)i/nbv * (float)cur_lastmass;

	  if( argout.sequence[i] <= (float)limit )
	    values[i]=0.0;
	  else
            values[i]=argout.sequence[i];

  	  break;
        default:/* Bar chart */
          mass[i]=(float)i+cur_firstmass;
          values[i]=argout.sequence[i];
          break;
      }

    }

  } else {
    
    if(PPASds==NULL) {
     *nb=0;
      return 1;
    }
    m=3;

    if( dev_putget( PPASds , DevReadValue , &m , D_SHORT_TYPE ,
                            &argout , D_VAR_FLOATARR , &error)<0)
    {
     err_str=dev_error_str(error);
     sprintf(tmp,"DevReadValue(3,%s) failed %s",PPASds->device_name,err_str);
     free(err_str);
     xrga_gui_adderr_man(tmp);
     return 0;
    }

    *nb=argout.length-2;
    nbv=(float)(*nb);

    cur_firstmass = argout.sequence[0];
    cur_lastmass  = argout.sequence[1];
    
    for(i=0;i<*nb;i++)
    {
      mass[i]=cur_firstmass + (float)i/nbv * (float)cur_lastmass;
      values[i]=argout.sequence[i+2];
    }


  }
    
  dev_xdrfree(D_VAR_FLOATARR , &argout , &error);
  return 1;
}

/*****************************************************/
/* xrga_dev_set_rga_mode() : Set the mode of the RGA */
/*   Arg in : int mode : 2 => Bar Chart              */
/*                       3 => Peak Jump              */
/*                       4 => Analog                 */
/*****************************************************/
int xrga_dev_set_rga_mode(int mode) {

  DevShort m;
  long error;
  char *err_str;
  char tmp[256];
  DevVarShortArray channels={0,NULL};

  if(RGAds==NULL) {
    return 0;
  }

  m=(short)mode;

  if( dev_putget( RGAds , DevSetMode , &m , D_SHORT_TYPE ,
                         NULL , D_VOID_TYPE , &error)<0)
  {
   err_str=dev_error_str(error);
   sprintf(tmp,"DevSetMode(%s) failed %s",RGAds->device_name,err_str);
   free(err_str);
   xrga_gui_adderr_man(tmp);
   return 0;
  }


  /* Get the channels configuration for peak jump mode */

  if(mode==3 || mode==9) {

    if( dev_putget( RGAds , DevGetChannelParameters , NULL , D_VOID_TYPE ,
                    &channels , D_VAR_SHORTARR , &error )<0 )
    {
      err_str=dev_error_str(error);
      sprintf(tmp,"DevGetChannelParameters(%s) failed %s",RGAds->device_name,err_str);
      free(err_str);
      xrga_gui_adderr_man(tmp);
      return 0;
    }

    memcpy( RGA_channel , channels.sequence , 12*sizeof(TRGA_channel) );
    dev_xdrfree(D_VAR_FLOATARR , &channels , &error);
  }

  return 1;
}

/************************************/
/* Return the current scan position */
/************************************/
void xrga_dev_get_rga_scan(int *s)
{
  DevShort m;
  long error;
  char *err_str;
  char tmp[256];
  DevVarFloatArray farr={0,NULL};

  *s=-1;

  if( current_rga_mode != 9 ) {

    if(RGAds==NULL) return;

    if( dev_putget( RGAds , DevIndicator , NULL, D_VOID_TYPE , 
			  &m , D_SHORT_TYPE ,&error)<0)
    {
     err_str=dev_error_str(error);
     sprintf(tmp,"DevIndicator(%s) failed %s",RGAds->device_name,err_str);
     free(err_str);
     xrga_gui_adderr_man(tmp);
    }

    *s=(int)m;

  } else {
  
    if(PPASds==NULL) return;
    m=4;

    if( dev_putget( PPASds , DevReadValue , &m , D_SHORT_TYPE , 
		            &farr , D_VAR_FLOATARR ,&error)<0)
    {
     err_str=dev_error_str(error);
     sprintf(tmp,"DevReadValue(1,%s) failed %s",PPASds->device_name,err_str);
     free(err_str);
     xrga_gui_adderr_man(tmp);
    }

    *s=(int)farr.sequence[0];
    dev_xdrfree( D_VAR_FLOATARR , &farr , &error );

  }

}

/*************************************************/
/* Return True if a PPAS is present              */
/*************************************************/
int xrga_dev_is_ppas_present()
{
  return PPASds!=NULL;
}

/***************************************************/
/* Return the channel number according to its mass */
/* -1 is returned otherwise                        */
/***************************************************/
int xrga_dev_get_mass_indx(int mass)
{
  int found=0;
  int i=0;

  if( current_rga_mode == 3 || current_rga_mode == 9 )
  {

    while(i<12 & !found) {
      found=RGA_channel[i].mass==mass;
      if(!found) i++;
    }

    if( found ) return i;
    else        return -1;

  } else
    return -1;
}

/**************************************************/
/* Set one param on the RGA                       */
/**************************************************/
void xrga_dev_set_one_rga_param(int num,int p)
{
  long error;
  char *err_str;
  char tmp[256];
  DevVarShortArray params={0,NULL};
  short param[2];

  param[0]=(short)num;
  param[1]=(short)p;
  params.length=2;
  params.sequence=param;

  if( dev_putget( RGAds , DevSetParam , &params , D_VAR_SHORTARR ,
                         NULL , D_VOID_TYPE , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevSetParam(%s) failed %s",RGAds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
  }

}

/**************************************************/
/* Reset  the RGA                                 */
/**************************************************/
void xrga_dev_reset_rga()
{
  long error;
  char *err_str;
  char tmp[256];

  if(RGAds==NULL) return;

  if( dev_putget( RGAds , DevReset , NULL , D_VOID_TYPE ,
                         NULL , D_VOID_TYPE , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevReset(%s) failed %s",RGAds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
  }

}

/****************************************************/
/* Set parameters of RGA                            */
/****************************************************/
void xrga_dev_set_rga_param( int multiplier ,
			     int accuracy ,
			     int firstmass ,
			     int lastmass ,
			     int range ,
			     int autorange,
			     int multvolt,
			     int multgain,
			     int filament )
{
  DevShort m;
  long error;
  char *err_str;
  char tmp[256];
  DevShort fila;

  if( RGAds==NULL ) return;

  if(multiplier) multiplier=1;
  if(autorange)  autorange=1;

  /**********************************************/
  /* Set mode dependant parameters              */
  /**********************************************/

  switch( current_rga_mode ) {
    case 3: /* Peak Jump */
    case 9: /* Peak Jump */

      if( cur_accuracy!=accuracy )
	xrga_dev_set_one_rga_param(3,accuracy);

      break;

    case 2: /* Bar chart mode */
    case 4: /* Analog mode    */

      if( cur_multiplier!=multiplier )
	xrga_dev_set_one_rga_param(1,multiplier);
      
      if( cur_accuracy!=accuracy )
	xrga_dev_set_one_rga_param(3,accuracy);

      if( cur_firstmass!=firstmass )
	xrga_dev_set_one_rga_param(4,firstmass);

      if( cur_lastmass!=lastmass ) {
        if( current_rga_mode==4 && 
	    lastmass!=8 && lastmass!=16 &&
	    lastmass!=32 && lastmass!=64 )
	{
	  xrga_gui_adderr_man("Mass zoom parameter is incorrect : valid values are 8,16,32 or 64");
	} else {
	  xrga_dev_set_one_rga_param(5,lastmass);
	}
      }

      if( cur_range!=range )
	xrga_dev_set_one_rga_param(6,range);

      if( cur_autorange!=autorange )
	xrga_dev_set_one_rga_param(7,autorange);

      break;
  }

  /*********************************************/
  /* Set up filament                           */
  /*********************************************/

  if( cur_filament!=filament ) {
    if( filament==0 ) {

      if( dev_putget( RGAds , DevFilamentOff , NULL , D_VOID_TYPE ,
                    NULL , D_VOID_TYPE , &error)<0)
      {
       err_str=dev_error_str(error);
       sprintf(tmp,"DevFilamentOff(%s) failed %s",RGAds->device_name,err_str);
       free(err_str);
       xrga_gui_adderr_man(tmp);
       return;
      }

    } else {

      fila=(short)filament;
      if( dev_putget( RGAds , DevFilamentOn , &fila , D_SHORT_TYPE ,
                         NULL , D_VOID_TYPE , &error)<0)
      {
       err_str=dev_error_str(error);
       sprintf(tmp,"DevFilamentOn(%s) failed %s",RGAds->device_name,err_str);
       free(err_str);
       xrga_gui_adderr_man(tmp);
       return;
      }

    }
  }

  /***********************************************/
  /* Set multiplier voltage and gain             */
  /***********************************************/
  m=(short)multvolt;
  if( dev_putget( RGAds , DevSetVoltage , &m , D_SHORT_TYPE ,
                    NULL , D_VOID_TYPE , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevSetVoltage(%s) failed %s",RGAds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return;
  }

  m=(short)multgain;
  if( dev_putget( RGAds , DevSetChanGain , &m , D_SHORT_TYPE ,
                    NULL , D_VOID_TYPE , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevSetChanGain(%s) failed %s",RGAds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return;
  }

}

/*********************************************************************/
/* xrga_dev_get_ppas_status : Return all relay and conditions status */
/*********************************************************************/
void xrga_dev_get_ppas_status(int *cond,int *relay,char *mode) {

  long error;
  char *err_str;
  char tmp[256];
  DevVarLongArray conds={0,NULL};
  int i,k;

  strcpy(mode,"Bypass");
  if( PPASds == NULL || current_rga_mode!=9 ) return;


  if( dev_putget( PPASds , DevReadSigValues , NULL , D_VOID_TYPE ,
                    &conds , D_VAR_LONGARR , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevReadSigValues(%s) failed %s",PPASds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return;
  }

  k=0;
  for(i=0;i<4;i++,k++) relay[i]=conds.sequence[k];
  
  if( conds.sequence[k]==DEVON ) strcpy(mode,"Control");

  k++;
  for(i=0;i<12;i++,k++) cond[i]=conds.sequence[k]+1;

  dev_xdrfree(D_VAR_LONGARR , &conds , &error);

}

/*******************************************************************/
/* xrga_dev_get_ppas_status : Return all relay , conditions status */
/* and spectrum data of the the alarm                              */
/*******************************************************************/
int xrga_dev_get_last_alarm(int *cond,int *relay,
			     int *startmass,int *lastmass,
			     float *mass,float *data,int *nb
			     )
{
  long error;
  char *err_str;
  char tmp[256];
  DevShort argin;
  DevVarFloatArray dat;
  int i,k;
  float nbv;

  if( PPASds == NULL ) return;

  if( current_rga_mode != 9 ) {
    xrga_gui_adderr_man("This command can be executed only in CONTROL mode");
    return 0;
  }

  /* Read last alarm extended (relay and condition status ) */

  argin = 2;
  dat.length=0;
  dat.sequence=NULL;
  if( dev_putget( PPASds , DevReadValue , &argin , D_SHORT_TYPE ,
                 &dat , D_VAR_FLOATARR , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevReadValue(2,%s) failed %s",PPASds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return 0;
  }

  k=0;
  for(i=0;i<4;i++,k++)  relay[i]=(int)dat.sequence[k];
  for(i=0;i<12;i++,k++) cond[i]=(int)dat.sequence[k]+1;
  dev_xdrfree(D_VAR_FLOATARR , &dat , &error);

  /* Read last alarm specturm */
  argin = 1;
  dat.length=0;
  dat.sequence=NULL;
  if( dev_putget( PPASds , DevReadValue , &argin , D_SHORT_TYPE ,
                 &dat , D_VAR_FLOATARR , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevReadValue(1,%s) failed %s",PPASds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return 0;
  }

  *nb=dat.length-2;
  nbv = (float)(*nb);
  *startmass=(int)dat.sequence[0];
  *lastmass=(int)dat.sequence[1];

  for(i=0;i<*nb;i++) {
    data[i]=dat.sequence[i+2];
    mass[i]=(float)(*startmass) + (float)i/nbv * (float)(*lastmass);
  }
  dev_xdrfree(D_VAR_FLOATARR , &dat , &error);

  return 1;
}

/*******************************************************************/
/* xrga_dev_set_ppas_mode : Set the ppas in control or bypass mode */
/*******************************************************************/
void xrga_dev_set_ppas_mode(int mode)
{
  long error;
  char *err_str;
  char tmp[256];
   
  
  switch(mode) {

  case 1: /* Bypass */

  if( dev_putget( PPASds , DevOff , NULL , D_VOID_TYPE ,
                    NULL , D_VOID_TYPE , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevOff(%s) failed %s",PPASds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return;
  }
  break;

  case 2: /* Control Mode */
  
  if( dev_putget( PPASds , DevOn , NULL , D_VOID_TYPE ,
                    NULL , D_VOID_TYPE , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevOn(%s) failed %s",PPASds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return;
  }
  break;
  }

}


/*********************************************************************/
/* xrga_dev_get_ppas_conditions : Return the text of each conditions */
/*********************************************************************/
void xrga_dev_get_ppas_conditions( char **r,char **t)
{
  long error;
  char *err_str;
  char tmp[256];
  int  i,k;
  DevVarStringArray conds={0,NULL};

  if( PPASds == NULL || current_rga_mode!=9 ) return;


  if( dev_putget( PPASds , DevReadParam , NULL , D_VOID_TYPE ,
                    &conds , D_VAR_STRINGARR , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevReadParam(%s) failed %s",PPASds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return;
  }

  k=0;
  for(i=0;i<4;i++,k++)  strcpy(r[i],conds.sequence[k]);
  for(i=0;i<12;i++,k++) strcpy(t[i],conds.sequence[k]);

  dev_xdrfree(D_VAR_STRINGARR , &conds , &error);
}


/************************************************************/
/* xrga_dev_build_sgmon_config : Build a configuration file */
/* for sgmon                                                */
/************************************************************/
int xrga_dev_build_sgmon_config_file( char *filename ) {

  char tmp[256];
  FILE *f;
  DevVarStringArray sig_conf={0,NULL};
  int nb_sig;
  int nu_of_prop;
  int i;
  long error;
  char *err_str;
  char *domain;
  char *family;
  char *member;
  char *nethost;

  if(RGAds==NULL) return 0;

  /* Get the signal configuration for RGA */

  if( dev_putget( RGAds , DevGetSigConfig , NULL , D_VOID_TYPE ,
                    &sig_conf , D_VAR_STRINGARR , &error)<0)
  {
    err_str=dev_error_str(error);
    sprintf(tmp,"DevGetSigConfig(%s) failed %s",RGAds->device_name,err_str);
    free(err_str);
    xrga_gui_adderr_man(tmp);
    return 0;
  }

  nu_of_prop = atoi(sig_conf.sequence[0]);
  nb_sig = sig_conf.length / nu_of_prop;

  if( (f=fopen(filename,"w")) == NULL) {
    sprintf(tmp,"Failed to open %s",filename);
    xrga_gui_adderr_man(tmp);
    dev_xdrfree( D_VAR_STRINGARR , &sig_conf , &error);
    return 0;
  }

  DevExtractField(RGAds->device_name,
		      &nethost,
                      &domain,
                      &family,
                      &member);


  fprintf(f,"#-------------------------------------------\n");
  fprintf(f,"# sgmon configuration file generated by xrga\n");
  fprintf(f,"#-------------------------------------------\n");
  fprintf(f,"\n");
  fprintf(f,"# Signal definition\n");
  fprintf(f,"\n");
  fprintf(f,"signal {\n");

  for(i=0;i<nb_sig-1;i++)
    if( nethost==NULL )
      fprintf(f,"  dev:%s\n",sig_conf.sequence[i*nu_of_prop+1]);
    else
      fprintf(f,"  dev://%s/%s\n",nethost,sig_conf.sequence[i*nu_of_prop+1]);

  fprintf(f,"}\n");
  fprintf(f,"\n");
  fprintf(f,"selection_Y1 {\n");
  for(i=0;i<nb_sig-1;i++)
    if( nethost==NULL )
      fprintf(f,"  dev:%s\n",sig_conf.sequence[i*nu_of_prop+1]);
    else
      fprintf(f,"  dev://%s/%s\n",nethost,sig_conf.sequence[i*nu_of_prop+1]);
  
  fprintf(f,"}\n");
  fprintf(f,"\n");

  fprintf(f,"# General option\n");
  fprintf(f,"\n");
  fprintf(f,"general_option {\n");
  fprintf(f,"  Y1 {\n");
  fprintf(f,"    min:0.000000\n");
  fprintf(f,"    max:0.000000\n");
  fprintf(f,"    autoscale:TRUE\n");
  fprintf(f,"    mode:log\n");
  fprintf(f,"    scientific:TRUE\n");
  fprintf(f,"    scientific_format:\"0.0e-00\"\n");
  fprintf(f,"    title:\"Partial pressure\"\n");
  fprintf(f,"  }\n");
  fprintf(f,"  common {\n");
  fprintf(f,"    duration:600.000000\n");
  fprintf(f,"    refresh:5.000000\n");
  fprintf(f,"    xgrid:FALSE\n");
  fprintf(f,"    ygrid:FALSE\n");
  fprintf(f,"    y0_auto:TRUE\n");
  fprintf(f,"    y0:0.000000\n");
  fprintf(f,"    time_unit:second\n");
  fprintf(f,"    legend:\"none\"\n");
  fprintf(f,"    xlegend:\"\"\n");
  fprintf(f,"    footer:\"\"\n");
  fprintf(f,"    header:\"RGA %s monitoring\"\n",member);
  fprintf(f,"  }\n");
  fprintf(f,"}\n");
  fclose(f);

  free(domain);
  free(family);
  free(member);

  return 1;
}
