/*================================================================
 *
 *  File:       gui.cpp
 *  Author:		JL PONS
 *  Project:	CTRM Video synopsis
 *  Content:    Graphics User Interface routines
 *  Date:		June 1998
 *
 ******************************************************************/
#include "stdafx.h"
#undef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#include <stdio.h>
#include <time.h>
#include <ddraw.h>

#include "graphics.h"
#include "gif.h"
#include "dev.h"
#include "Log.h"


// Operator message management vars
char  last_opmesg[1024];
int   nb_line;
int   cur_line;
char *opmesg_ln[20];
LPDIRECTDRAWSURFACE     lpDDSOpMesg;   // DirectDraw primary surface

//********************************************************
// gui_init  : initialise all gui object
// HWND hwnd : Handle to main window
// err_str   : countains error when routine fails
// Return 0 when fails
//********************************************************
int gui_init( HWND hwnd , char *err_str )
{
	_startdbgProc("gui_init");
	int i;

	if( graphics_init( hwnd , err_str ) == 0 ) {
		_enddbgProc();
		return 0;
	}

	strcpy( last_opmesg,"" );
	for(i=0;i<20;i++) opmesg_ln[i]=NULL;
	nb_line=0;
	lpDDSOpMesg=NULL;

	_enddbgProc();
    return 1;
}

//**************************************
// Draw a bl control
//**************************************
long gui_draw_bl_control(LPDIRECTDRAWSURFACE lpDD,int x,int y)
{
	_startdbgProc("gui_draw_bl_control");
	long err = DD_OK;

	EK(err) err=Line( lpDD ,x  ,y  ,x+44,y  ,LIGHT_GREY );
    EK(err) err=Line( lpDD ,x  ,y+1,x+43,y+1,LIGHT_GREY );
    EK(err) err=Line( lpDD ,x  ,y+2,x+42,y+2,LIGHT_GREY );

	EK(err) err=Line( lpDD ,x  ,y  ,x  ,y+29, LIGHT_GREY );
    EK(err) err=Line( lpDD ,x+1,y  ,x+1,y+28, LIGHT_GREY );
    EK(err) err=Line( lpDD ,x+2,y  ,x+2,y+27, LIGHT_GREY );

	EK(err) err=Line( lpDD ,x+44,y  ,x+44,y+29 , DARK_GREY );
    EK(err) err=Line( lpDD ,x+43,y+1,x+43,y+29 , DARK_GREY );
    EK(err) err=Line( lpDD ,x+42,y+2,x+42,y+29 , DARK_GREY );

	EK(err) err=Line( lpDD ,x+2,y+27,x+44,y+27 , DARK_GREY );
    EK(err) err=Line( lpDD ,x+1,y+28,x+44,y+28 , DARK_GREY );
    EK(err) err=Line( lpDD ,x  ,y+29,x+44,y+29 , DARK_GREY );

	EK(err) err=FillRect( lpDD , x+3 , y+3 , x+42 , y+27 , GREY );

	_enddbgProc();
	return err;
}
//********************************************************
// Draw a shadowed field bordure
//********************************************************
long gui_draw_shadow(LPDIRECTDRAWSURFACE pdds,
				     int x1,int y1,
				     int x2,int y2)
{
  _startdbgProc("gui_draw_shadow");
  long err=DD_OK;

  EK(err) err=Line( pdds ,x1,y1,x2,y1 ,DARK_GREY );
  EK(err) err=Line( pdds ,x2,y1,x2,y2 ,LIGHT_GREY );
  EK(err) err=Line( pdds ,x2,y2,x1,y2 ,LIGHT_GREY );
  EK(err) err=Line( pdds ,x1,y2,x1,y1 ,DARK_GREY );

  EK(err) err=Line( pdds ,x1+1,y1+1,x2-1,y1+1,DARK_GREY );
  EK(err) err=Line( pdds ,x2-1,y1+1,x2-1,y2-1 ,LIGHT_GREY );
  EK(err) err=Line( pdds ,x2-1,y2-1,x1+1,y2-1 ,LIGHT_GREY );
  EK(err) err=Line( pdds ,x1+1,y2-1,x1+1,y1+1 ,DARK_GREY );

  _enddbgProc();
  return err;
}

long gui_draw_rect(LPDIRECTDRAWSURFACE pdds,
				   int x1,int y1,
				   int x2,int y2,
				   DWORD color)
{
  _startdbgProc("gui_draw_shadow");
  long err=DD_OK;

  EK(err) err=Line( pdds ,x1,y1,x2,y1 ,color );
  EK(err) err=Line( pdds ,x2,y1,x2,y2 ,color );
  EK(err) err=Line( pdds ,x2,y2,x1,y2 ,color );
  EK(err) err=Line( pdds ,x1,y2,x1,y1 ,color );

  _enddbgProc();
  return err;
}

//********************************************************
// Draw BM and ID
//********************************************************
long gui_draw_fixed(LPDIRECTDRAWSURFACE lpDD)
{
  _startdbgProc("gui_draw_fixed");
  int i,j;
  long err=DD_OK;

  //****************************
  // Draw edge
  //****************************

  // BL egde ***************************

  EK(err) err=Line( lpDD ,400,5,795,5,DARK_GREY );
  EK(err) err=Line( lpDD ,795,5,795,295,LIGHT_GREY );
  EK(err) err=Line( lpDD ,795,295,400,295,LIGHT_GREY );
  EK(err) err=Line( lpDD ,400,295,400,5,DARK_GREY );

  EK(err) err=Line( lpDD ,401,6,794,6,DARK_GREY );
  EK(err) err=Line( lpDD ,794,6,794,294,LIGHT_GREY );
  EK(err) err=Line( lpDD ,794,294,401,294,LIGHT_GREY );
  EK(err) err=Line( lpDD ,401,294,401,6,DARK_GREY );

  EK(err) err=Line( lpDD ,402,7,793,7,LIGHT_GREY );
  EK(err) err=Line( lpDD ,793,7,793,293,DARK_GREY );
  EK(err) err=Line( lpDD ,793,293,402,293,DARK_GREY );
  EK(err) err=Line( lpDD ,402,293,402,7,LIGHT_GREY );
  
  EK(err) err=Line( lpDD ,403,8,792,8,LIGHT_GREY );
  EK(err) err=Line( lpDD ,792,8,792,292,DARK_GREY );
  EK(err) err=Line( lpDD ,792,292,403,292,DARK_GREY );
  EK(err) err=Line( lpDD ,403,292,403,8,LIGHT_GREY );

  // Date edge ******************

  EK(err) err=Line( lpDD ,5,5,390,5,DARK_GREY );
  EK(err) err=Line( lpDD ,5,6,390,6,DARK_GREY );
  EK(err) err=Line( lpDD ,5,7,390,7,LIGHT_GREY );
  EK(err) err=Line( lpDD ,5,8,390,8,LIGHT_GREY );

  EK(err) err=Line( lpDD ,5,55,390,55,DARK_GREY );
  EK(err) err=Line( lpDD ,5,56,390,56,DARK_GREY );
  EK(err) err=Line( lpDD ,5,57,390,57,LIGHT_GREY );
  EK(err) err=Line( lpDD ,5,58,390,58,LIGHT_GREY );

  // Current edge

  EK(err) err=Line( lpDD ,5,61,390,61,DARK_GREY );
  EK(err) err=Line( lpDD ,5,62,389,62,DARK_GREY );
  EK(err) err=Line( lpDD ,5,63,388,63,DARK_GREY );
  EK(err) err=Line( lpDD ,5,64,387,64,DARK_GREY );

  EK(err) err=Line( lpDD ,5,61,5,151,DARK_GREY);
  EK(err) err=Line( lpDD ,6,61,6,150,DARK_GREY);
  EK(err) err=Line( lpDD ,7,61,7,149,DARK_GREY);
  EK(err) err=Line( lpDD ,8,61,8,148,DARK_GREY);

  EK(err) err=Line( lpDD ,387,64,387,151,LIGHT_GREY);
  EK(err) err=Line( lpDD ,388,63,388,151,LIGHT_GREY);
  EK(err) err=Line( lpDD ,389,62,389,151,LIGHT_GREY);
  EK(err) err=Line( lpDD ,390,61,390,151,LIGHT_GREY);

  EK(err) err=Line( lpDD ,5,151,390,151,LIGHT_GREY);
  EK(err) err=Line( lpDD ,6,150,390,150,LIGHT_GREY);
  EK(err) err=Line( lpDD ,7,149,390,149,LIGHT_GREY);
  EK(err) err=Line( lpDD ,8,148,390,148,LIGHT_GREY);

  EK(err) err=FillRect( lpDD , 9 , 65 , 387 , 148 , SYN_BLUE );

  // Botton edge (under Filling mode)

  EK(err) err=Line( lpDD ,5,293,390,293,DARK_GREY );
  EK(err) err=Line( lpDD ,5,294,390,294,DARK_GREY );
  EK(err) err=Line( lpDD ,5,295,390,295,LIGHT_GREY );
  EK(err) err=Line( lpDD ,5,296,390,296,LIGHT_GREY );

  // *******************************
  // Draw text
  // *******************************

  EK(err) err=gprint( lpDD , "ID" ,
		  487, 10, def_font,GREY,BLACK);
  EK(err) err=gprint( lpDD , "Bendings" ,
		  650, 10, def_font,GREY,BLACK);
  EK(err) err=gprint_trans( lpDD , "Filling mode" , 
	      5 , 165 , def_font , BLACK );
  EK(err) err=gprint_trans( lpDD , "Lifetime" , 
	      5 , 213 , def_font , BLACK );

  //************************
  // Draw control
  //************************

  for( i=412 ; i < 592 ; i+=45 )
	for ( j=45 ; j<285 ; j+=30 )
      EK(err) err=gui_draw_bl_control( lpDD , i , j );

  for( i=605 ; i < 785 ; i+=45 )
	for ( j=45 ; j<285 ; j+=30 )
      EK(err) err=gui_draw_bl_control( lpDD , i , j );

  _enddbgProc();
  return err;
}

//*************************************************
// Drawing current history
//*************************************************
char *Heure( long tm) {
  _startdbgProc("Heure");
  static char ret[128];
  char *ct;

  ct=ctime((time_t *)&tm);

  strncpy(ret,&(ct[11]),5);
  ret[6]=0;

  _enddbgProc();
  return ret;
}

long gui_draw_current_hist( LPDIRECTDRAWSURFACE pdds , 
					   HFONT fnt , HFONT fnt2,
					   long *tm,float *history , int nb )
{
	_startdbgProc("gui_draw_current_hist");
	float max;
	long i,xe,ye;
	char tmp[128];
	double x_width;
	long first_tm;
	int inc_cur;
	long err=DD_OK;

	HRGN rgn;
	POINT *pts;

	// Draw edge and background

	EK(err) err=Line( pdds ,5  ,302,795,302 ,DARK_GREY );
    EK(err) err=Line( pdds ,795,302,795,595 ,LIGHT_GREY );
    EK(err) err=Line( pdds ,795,595,5  ,595 ,LIGHT_GREY );
    EK(err) err=Line( pdds ,5  ,595,5  ,302 ,DARK_GREY );

	EK(err) err=Line( pdds ,6  ,303,794,303 ,DARK_GREY );
    EK(err) err=Line( pdds ,794,303,794,594 ,LIGHT_GREY );
    EK(err) err=Line( pdds ,794,594,6  ,594 ,LIGHT_GREY );
    EK(err) err=Line( pdds ,6  ,594,6  ,303 ,DARK_GREY );

	EK(err) err=Line( pdds ,7  ,304,793,304 ,LIGHT_GREY );
    EK(err) err=Line( pdds ,793,304,793,593 ,DARK_GREY );
    EK(err) err=Line( pdds ,793,593,7  ,593 ,DARK_GREY );
    EK(err) err=Line( pdds ,7  ,593,7  ,304 ,LIGHT_GREY );

	EK(err) err=Line( pdds ,8  ,305,792,305 ,LIGHT_GREY );
    EK(err) err=Line( pdds ,792,305,792,592 ,DARK_GREY );
    EK(err) err=Line( pdds ,792,592,8  ,592 ,DARK_GREY );
    EK(err) err=Line( pdds ,8  ,592,8  ,305 ,LIGHT_GREY );

    EK(err) err=FillRect( pdds , 9 , 306 , 791 , 592 , SYN_BLUE );

	// Don't trace anything if less than 2 points
	if( nb < 2 ) {
		_enddbgProc();
		return DD_OK;
	}

	// Search max to compute scaling and round it

	max=history[1];
	for( i=0 ; i<nb ; i++ )
	  if( history[i]>max ) max=history[i];
	max = (float)((((int)max/25)+1)*25);
	
	if(max<=25.0)              inc_cur=5;
	if(max>25 && max<=50.0)    inc_cur=10;
	if(max>50.0 && max<=100.0) inc_cur=25;
    if(max>100.0)              inc_cur=50;

	// Initialise vars to compute x scaling
	x_width=(double)(tm[nb-1]-tm[0]);
	first_tm = ((tm[0] / 3600)+1) * 3600;

	// Draw current

	pts=(POINT *)malloc(sizeof(POINT)*(nb+2));

	for( i=0 ; i<nb ; i++ )
	{
		xe=50 +(int)(700.0*(tm[i]-tm[0])/x_width);
		ye=560-(int)((250.0*history[i])/max);

		pts[i].x=xe;
		pts[i].y=ye;
	}

	pts[i].x=750;
	pts[i].y=560;
	i++;
	pts[i].x=50;
	pts[i].y=560;
	i++;
	rgn  = CreatePolygonRgn( pts , i , WINDING );

	EK(err) err=Fill( pdds , rgn , SYN_YELLOW );

	free(pts);
	if( !DeleteObject( rgn ) )
		err = DDERR_UNSUPPORTED;

    // Draw Y axis label
	for( i=0 ; i<=(int)max ; i+=inc_cur) {

		sprintf(tmp,"%d",i);
		ye=560-(int)((250.0*i)/max);
		xe=12;
        EK(err) err=LineDot(pdds,xe+35,ye,xe+740,ye, SYN_YELLOW );

		// Draw Y axis label on left with left justification

		if( i<10 )
		{
		  if( ye > 320 )
		  {
		    EK(err) err=gprint_trans(pdds,tmp,xe+24,ye-15,fnt2, SYN_YELLOW );
		  } else {
		    EK(err) err=gprint_trans(pdds,tmp,xe+24,ye-5,fnt2, SYN_YELLOW );
		  }
		}
		if( i>=10 && i<100 )
		{
		  if( ye > 320 ) 
		  {
		    EK(err) err=gprint_trans(pdds,tmp,xe+12,ye-15,fnt2, SYN_YELLOW );
		  } else {
		    EK(err) err=gprint_trans(pdds,tmp,xe+12,ye-5,fnt2, SYN_YELLOW );
		  }
		}
		if( i>=100 )
		{
		  if( ye > 320 )
		  {
		    EK(err) err=gprint_trans(pdds,tmp,xe,ye-15,fnt2, SYN_YELLOW );
		  } else {
		    EK(err) err=gprint_trans(pdds,tmp,xe,ye-5,fnt2, SYN_YELLOW );
		  }
		}

		// Draw Y axis on right

		if( ye > 320 )
		{
		  EK(err) err=gprint_trans(pdds,tmp,755,ye-15,fnt2, SYN_YELLOW );
		} else {
		  EK(err) err=gprint_trans(pdds,tmp,755,ye-5,fnt2, SYN_YELLOW );
		}

	}

	// Draw X axis label
	for( i=first_tm ; i<=tm[nb-1] ; i+=3600 )
	{
		xe=50+(int)(700.0*(i-tm[0])/x_width);
		ye=560;
        EK(err) err=LineDot(pdds,xe,ye+4,xe,ye-250, SYN_YELLOW );
        EK(err) err=gprint(pdds,Heure(i),xe-25,ye+10,fnt, SYN_BLUE , SYN_YELLOW );
	}

	// Draw axes

	EK(err) err=Line(pdds,50,560,50,310,  SYN_YELLOW );
	EK(err) err=Line(pdds,750,560,750,310,  SYN_YELLOW );
	EK(err) err=Line(pdds,50,560,750,560, SYN_YELLOW );


	_enddbgProc();
	return err;
}

//*************************************************
// Update current field 
//*************************************************
long gui_update_current(LPDIRECTDRAWSURFACE lpDD , char *c)
{
	_startdbgProc("gui_update_current");

	long err = DD_OK;

	EK(err) err=FillRect( lpDD , 9 , 65 , 387 , 148 , SYN_BLUE );
    EK(err) err=gprint_trans( lpDD , c ,
  		  10, 50, current_font,SYN_YELLOW);

	_enddbgProc();
	return err;
}

//*************************************************
// Update BM fields 
//*************************************************


long gui_update_bm( LPDIRECTDRAWSURFACE lpDD , long *BM_st )
{
  _startdbgProc("gui_update_bm");
  int i,j,k;
  char tmp[10];
  long err = DD_OK;

  k=1;
  for ( j=45 ; j<285 ; j+=30 )
    for( i=605 ; i < 785 ; i+=45 )
	{
	  if( BM[k] != 0 )
	  {
  	    EK(err) err=FillRect( lpDD , i+3 , j+3 , i+42 , j+27 , FEcolor( BM_st[k] ) );
        sprintf(tmp,"%d",k);
		if( k>=10 )
		{
          EK(err) err=gprint_trans( lpDD , tmp , i+5 , j , bl_font , BLACK );
		} else {
          EK(err) err=gprint_trans( lpDD , tmp , i+14 , j , bl_font , BLACK );
		}
	  }
	  k++;
	}

  _enddbgProc();
  return err;
}

long gui_update_id( LPDIRECTDRAWSURFACE lpDD , long *ID_st )
{
  _startdbgProc("gui_update_id");
  int i,j,k;
  char tmp[10];
  long err = DD_OK;

  k=1;
  for ( j=45 ; j<285 ; j+=30 )
    for( i=412 ; i < 592 ; i+=45 )
	{
	  if( ID[k] != 0 )
	  {
  	    EK(err) err=FillRect( lpDD , i+3 , j+3 , i+42 , j+27 , FEcolor( ID_st[k] ) );
        sprintf(tmp,"%d",k);
		if( k>=10 )
		{
          EK(err) err=gprint_trans( lpDD , tmp , i+5 , j , bl_font , BLACK );
		} else {
          EK(err) err=gprint_trans( lpDD , tmp , i+14 , j , bl_font , BLACK );
		}
	  }
	  k++;
	}

  _enddbgProc();
  return err;
}

//*******************************************
// Update MODE (USM ,MDT ,SHUTDOWN,SAFETY)
//*******************************************
long gui_update_mode(LPDIRECTDRAWSURFACE lpDD ,char *mode)
{
  _startdbgProc("gui_update_mode");
  long err = DD_OK;

  EK(err) err=FillRect( lpDD , 9 , 195 , 387 , 292 , GREY );
  
  EK(err) err=gprint( lpDD , mode ,
		  10, 195, mode_font,GREY,BLACK);

 _enddbgProc();
 return err;
}

//*****************************************
// Wrap long line
//*****************************************
void SplitString(char *s,int *nb)
{
   _startdbgProc("SplitString");
   int LastSpace;
   int i,j;
 
   i=0;
   j=0;
   *nb=1;
   LastSpace=-1;
 
   while( s[i]!=0 )
   {
     if(s[i]==' ') LastSpace=i;
     if( (j>=43) && (LastSpace!=-1) )
     {
       j=i-LastSpace;
       s[LastSpace]='\0';
       LastSpace=-1;
       *nb=*nb+1;
     }
     i++;j++;
   }

   	_enddbgProc();
}

// Initialise message for scrolling
long gui_update_init_opmesg(char *msg) {
	_startdbgProc("gui_update_init_opmesg");
	int i;char *s;
    long err = DD_OK;

	// Free old string
	for(i=0;i<nb_line;i++) {
		free(opmesg_ln[i]);
		opmesg_ln[i]=NULL;
	}
	
	for(i=0;i<20;i++) opmesg_ln[i]=NULL;
	nb_line=0;

	// Make string array
    SplitString(msg , &nb_line );
	s=msg;

    for( i=0; i<nb_line ;i++ )
	{
      opmesg_ln[i]=strdup(s);
	  s+=strlen(s);
	  s++;
	}
	cur_line=0;

	if(nb_line==1) {
		nb_line++;
		opmesg_ln[1]=strdup(" ");
	}

	// Build the DD surface
	if( lpDDSOpMesg ) {
		lpDDSOpMesg->Release();
        lpDDSOpMesg=NULL;
	}

	DDSURFACEDESC       ddsd;

	ZeroMemory(&ddsd, sizeof(ddsd));

    ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    ddsd.dwWidth  = 786;
    ddsd.dwHeight = 32*nb_line;

    err = lpDD->CreateSurface( &ddsd, &lpDDSOpMesg, NULL );
	if(err!=DD_OK) {
		_enddbgProc();
		return err;
	}

	// Write the text
    EK(err) err=FillRect( lpDDSOpMesg , 0 , 0 , 786 , 32*nb_line , SYN_BLUE );

	for(i=0;i<nb_line;i++)
	  EK(err) err=gprint_trans( lpDDSOpMesg , opmesg_ln[i] , 0,32*i-5,text_font,SYN_YELLOW);

	_enddbgProc();
	return err;
}

long gui_start_opmesg()
{
  _startdbgProc("gui_start_opmesg");
  long err = DD_OK;

  // Draw text on the back surface
  HDC hdc_dest;
  HDC hdc_src;
  cur_line=0;

  EK(err) err=lpDDSOpMesg->GetDC(&hdc_src);
  EK(err) err=lpDDSBack->GetDC(&hdc_dest);

  EK(err) BitBlt( hdc_dest,7,532,788,62, 
          hdc_src,0,0, SRCCOPY );

  lpDDSBack->ReleaseDC(hdc_dest);
  lpDDSOpMesg->ReleaseDC(hdc_src);

  _enddbgProc();
  return err;
}

long gui_update_opmesg()
{
  _startdbgProc("gui_update_opmesg");
  long err = DD_OK;
  int  i;
  int nb_step=20;

  // Draw and anim text on the primary surface
  HDC hdc_dest;
  HDC hdc_src;


  EK(err) err=lpDDSOpMesg->GetDC(&hdc_src);

  if( nb_line<=2 ) {


    EK(err) err=lpDDSPrimary->GetDC(&hdc_dest);

    BitBlt( hdc_dest,7,532,788,62, 
          hdc_src,0,0, SRCCOPY );

    lpDDSPrimary->ReleaseDC(hdc_dest);
  
  } else {

	cur_line++;
    if( cur_line+1 >= nb_line ) {
	  
	  // Scroll back
	  if( lpDDSPrimary->GetDC(&hdc_dest)==DD_OK ) {

	    for(i=0;i<=nb_step;i++) {
		  double w;
		  w = (double)(nb_line-2)*((double)(nb_step-i)/(double)nb_step);
		  BitBlt( hdc_dest,7,532,788,62, 
            hdc_src,0,(int)(w*32.0), SRCCOPY );
		  lpDD->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN ,NULL);
		}

	    lpDDSPrimary->ReleaseDC(hdc_dest);
	  }
	  cur_line=0;

	} else {

	  // Scroll down of 1 line  
	  if( lpDDSPrimary->GetDC(&hdc_dest)==DD_OK ) {

	    for(i=0;i<=nb_step;i++) {
		  double w;
		  w = (double)(cur_line-1)+((double)i/(double)nb_step);
		  BitBlt( hdc_dest,7,532,788,62, 
            hdc_src,0,(int)(w*32.0), SRCCOPY );
		  lpDD->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN ,NULL);
		}

	    lpDDSPrimary->ReleaseDC(hdc_dest);
	  }

	}

  }


  lpDDSOpMesg->ReleaseDC(hdc_src);

  _enddbgProc();
  return err;
}

//******************************************************
// Update Filling mode
//******************************************************
long gui_update_filling_mode( LPDIRECTDRAWSURFACE pdds , char *msg )
{
  _startdbgProc("gui_update_filling_mode");
  long err = DD_OK;

  // Draw edge and background

  EK(err) err=gui_draw_shadow( pdds , 160 , 158 , 390 , 197 );
  EK(err) err=FillRect( pdds , 162 , 160 , 388 , 196 , SYN_BLUE );
  EK(err) err=gprint_trans( pdds , msg , 163 , 158 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}

//******************************************************
// Update lifetime
//******************************************************
long gui_update_lifetime( LPDIRECTDRAWSURFACE pdds , char *msg )
{
  _startdbgProc("gui_update_lifetime");
  long err = DD_OK;

  // Draw edge and background

  EK(err) err=gui_draw_shadow( pdds , 160 , 204 , 390 , 243 );
  EK(err) err=FillRect( pdds , 162 , 204 , 388 , 242 , SYN_BLUE );
  EK(err) err=gprint_trans( pdds , msg , 165 , 204 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}

//******************************************************
// Update since message
//******************************************************
long gui_update_since( LPDIRECTDRAWSURFACE pdds , char *msg )
{
  _startdbgProc("gui_update_since");
  long err = DD_OK;

  // Draw edge and background

  EK(err) err=gui_draw_shadow( pdds , 5 , 250 , 390 , 289 );
  EK(err) err=FillRect( pdds , 7 , 252 , 388 , 288 , SYN_BLUE );
  EK(err) err=gprint_trans( pdds , msg , 9 , 250 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}


// *******************************************************************
// Orbit
// *******************************************************************
long gui_update_orbit( LPDIRECTDRAWSURFACE pdds ,char *ormsh , char *ormsv,
					                             char *opeakh,char *opeakv)
{
  _startdbgProc("gui_update_orbit");
  long err=DD_OK;

  EK(err) err=gprint_trans(  lpDDSBack , "Orbit (RMS)" ,60, 298, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "H" ,15, 325, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "V" ,15, 370, def_font,BLACK);

  EK(err) err=gui_draw_shadow( lpDDSBack , 35 , 324 , 225 , 364 );
  EK(err) err=FillRect( lpDDSBack , 37 , 326 , 224 , 363 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , ormsh , 39 , 326 , text_font , SYN_YELLOW );

  EK(err) err=gui_draw_shadow( lpDDSBack , 35 , 368 , 225 , 408 );
  EK(err) err=FillRect( lpDDSBack , 37 , 370 , 224 , 407 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , ormsv , 39 , 368 , text_font , SYN_YELLOW );

  EK(err) err=gprint_trans(  lpDDSBack , "Orbit (Peak)" ,60, 406, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "H" ,15, 437, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "V" ,15, 486, def_font,BLACK);
  
  EK(err) err=gui_draw_shadow( lpDDSBack , 35 , 435 , 225 , 480 );
  EK(err) err=FillRect( lpDDSBack , 37 , 437 , 224 , 479 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , opeakh , 39 , 435 , text_font , SYN_YELLOW );


  EK(err) err=gui_draw_shadow( lpDDSBack , 35 , 484 , 225 , 524 );
  EK(err) err=FillRect( lpDDSBack , 37 , 486 , 224 , 523 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , opeakv , 39 , 484 , text_font , SYN_YELLOW );
  
  _enddbgProc();
  return err;
}

// *******************************************************************
// Emitances
// *******************************************************************
long gui_update_emit( LPDIRECTDRAWSURFACE pdds ,char *d9x , char *d9z ,
					                            char *id8x,char *id8z)

{
  _startdbgProc("gui_update_emit");
  long err=DD_OK;

  EK(err) err=gprint_trans(  lpDDSBack , "H Emittance" ,310, 298, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "H" ,270, 325, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "H" ,270, 370, def_font,BLACK);

  EK(err) err=gui_draw_shadow( lpDDSBack , 295 , 324 , 485 , 364 );
  EK(err) err=FillRect( lpDDSBack , 297 , 326 , 484 , 363 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , id8x , 299 , 326 , text_font , SYN_YELLOW );

  EK(err) err=gui_draw_shadow( lpDDSBack , 295 , 368 , 485 , 408 );
  EK(err) err=FillRect( lpDDSBack , 297 , 370 , 484 , 407 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , d9x , 299 , 368 , text_font , SYN_YELLOW );

  EK(err) err=gprint_trans(  lpDDSBack , "V Emittance" ,310, 406, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "V" ,270, 437, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "V" ,270, 486, def_font,BLACK);
  
  EK(err) err=gui_draw_shadow( lpDDSBack , 295 , 435 , 485 , 480 );
  EK(err) err=FillRect( lpDDSBack , 297 , 437 , 484 , 479 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , id8z , 299 , 435 , text_font , SYN_YELLOW );


  EK(err) err=gui_draw_shadow( lpDDSBack , 295 , 484 , 485 , 524 );
  EK(err) err=FillRect( lpDDSBack , 297 , 486 , 484 , 523 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , d9z , 299 , 484 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}

// *******************************************************************
// Tunes
// *******************************************************************
long gui_update_tune(LPDIRECTDRAWSURFACE pdds ,char *th , char *tv ) {
   _startdbgProc("gui_update_tune");
  long err=DD_OK;

  EK(err) err=gprint_trans(  lpDDSBack , "Tunes" ,555, 298, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "H" ,    515, 325, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "V" ,    515, 370, def_font,BLACK);

  EK(err) err=gui_draw_shadow( lpDDSBack ,   540 , 324 , 778 , 364 );
  EK(err) err=FillRect( lpDDSBack ,          542 , 326 , 777 , 363 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , th , 544 , 326 , text_font , SYN_YELLOW );

  EK(err) err=gui_draw_shadow( lpDDSBack ,   540 , 368 , 778 , 408 );
  EK(err) err=FillRect( lpDDSBack ,          542 , 370 , 777 , 407 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , tv , 544 , 368 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}

// *******************************************************************
// Dipole Field
// *******************************************************************
long gui_update_field(LPDIRECTDRAWSURFACE pdds ,char *nmr , char *hall ) {
   _startdbgProc("gui_update_field");
  long err=DD_OK;

  EK(err) err=gprint_trans(  lpDDSBack , "Dipoles Field" ,570, 406, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "nmr" , 510, 437, def_font,BLACK);
  EK(err) err=gprint_trans(  lpDDSBack , "hall" ,510, 486, def_font,BLACK);
  
  EK(err) err=gui_draw_shadow( lpDDSBack , 555 , 435 , 765 , 480 );
  EK(err) err=FillRect( lpDDSBack ,        557 , 437 , 764 , 479 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , nmr , 559 , 435 , text_font , SYN_YELLOW );


  EK(err) err=gui_draw_shadow( lpDDSBack , 555 , 484 , 765 , 524 );
  EK(err) err=FillRect( lpDDSBack ,        557 , 486 , 764 , 523 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , hall , 559 , 484 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}

// *******************************************************************
// Average Pressure
// *******************************************************************
long gui_update_avpress(LPDIRECTDRAWSURFACE pdds ,char *press ) {
  _startdbgProc("gui_update_avpress");  
  long err=DD_OK;

  EK(err) err=gprint_trans(  lpDDSBack , "Average Pressure" ,560, 406, def_font,BLACK);
  //EK(err) err=gprint_trans(  lpDDSBack , "nmr" , 510, 437, def_font,BLACK);
  
  EK(err) err=gui_draw_shadow( lpDDSBack ,      540 , 435 , 778 , 480 );
  EK(err) err=FillRect( lpDDSBack ,             542 , 437 , 777 , 479 , SYN_BLUE );
  EK(err) err=gprint_trans( lpDDSBack , press , 544 , 435 , text_font , SYN_YELLOW );

  _enddbgProc();
  return err;
}

//******************************************************
// Draw shutdown mode
//******************************************************
long gui_draw_shutdown(LPDIRECTDRAWSURFACE pdds ,PIXEL *img,int x,int y) {
  _startdbgProc("gui_draw_shutdown");  

  char rdate[256];
  long err=DD_OK;
  int wd,hd;
  int wm,hm;
  
  char *txt = "Machine shutdown until";

  dev_get_nextrundate(rdate);	

  if(img!=NULL) {

    int x1,y1,x2,y2;
    
	EK(err) err=FillSurface(pdds,img,0,0,800,600);

    if(x>0 && y>0) {

	  EK(err) err=measure_text(pdds,rdate,current_font,&wd,&hd);
      EK(err) err=measure_text(pdds,txt,text_font,&wm,&hm);
      x1 = x-5;
      y1 = y-5;
      x2 = x + max(wm,wd) + 5;
      y2 = y+5+hm+hd;
    
	  EK(err) err=DarkenRect(pdds,x1,y1,x2,y2);
    
	  EK(err) err=gprint_trans(pdds , txt    ,x + (wd-wm)/2, y    , text_font   ,WHITE);
      EK(err) err=gprint_trans(pdds , rdate  ,x            , y+hm , current_font,WHITE);

      EK(err) err=Line(pdds,x1,y1,x2,y1, RED);
      EK(err) err=Line(pdds,x2,y1,x2,y2, RED);
      EK(err) err=Line(pdds,x2,y2,x1,y2, RED);
      EK(err) err=Line(pdds,x1,y2,x1,y1, RED);

	}

  } else {

	int wr,hr,x1,y1;

	EK(err) err=ClearSurface(pdds,0xAF);
	EK(err) err=FillRect(pdds ,   0 ,   0 , 800 ,   5 , DARK_GREY);
	EK(err) err=FillRect(pdds ,   0 ,   0 ,   8 , 300 , DARK_GREY);
	EK(err) err=FillRect(pdds ,   0 , 292 , 800 , 300 , DARK_GREY);
	EK(err) err=FillRect(pdds , 792 ,   0 , 800 , 300 , DARK_GREY);
	EK(err) err=FillRect(pdds ,   0 , 300 , 800 , 600 , WHITE);

    EK(err) err=measure_text(pdds,txt,mode_font,&wm,&hm);
	EK(err) err=measure_text(pdds,rdate,current_font,&wd,&hd);

	wr = max(wm,wd);
	hr = hm+hd;
	x1 = (800-wr)/2;
	y1 = (300-hr)/2;

    EK(err) err=gprint_trans(pdds , txt    ,x1             , y1    , mode_font   ,BLACK);
    EK(err) err=gprint_trans(pdds , rdate  ,x1 + (wm-wd)/2 , y1+hm , current_font,BLACK);

  }

  _enddbgProc();
  return err;

}


//******************************************************
// Clone high part of surface
//******************************************************
long gui_clone()
{
  _startdbgProc("gui_clone");
  DDSURFACEDESC surf;
  BYTE *add_src;
  BYTE *add_dest;
  int i;
  BYTE *tmp;
  HRESULT dderr;

  surf.dwSize = sizeof(surf);
  tmp=(BYTE *)malloc( 240000*sizeof(PIXEL) );

  //******************************
  // Get data from primary surface
  //******************************

  while( (dderr = lpDDSPrimary->Lock( NULL , &surf , 0 , NULL ))
	                      == DDERR_WASSTILLDRAWING);

  if( dderr!=DD_OK ) {
      _enddbgProc();
	  return dderr;
  }

  add_src=(BYTE *)surf.lpSurface;
  add_dest=tmp;

  for(i=0;i<300;i++)
  {
	  memcpy( add_dest , add_src , 2400 );  
	  add_src  = add_src + surf.lPitch;
	  add_dest = add_dest + 2400;
  }

  lpDDSPrimary->Unlock( NULL );  

  //******************************
  // Put data on second surface
  //******************************

  while( ( dderr = lpDDSBack->Lock( NULL , &surf , 0 , NULL ))
	                      == DDERR_WASSTILLDRAWING);
  if( dderr!=DD_OK ) {
	  free(tmp);
      _enddbgProc();
	  return dderr;
  }

  add_src=tmp;
  add_dest=(BYTE *)surf.lpSurface ;

  for(i=0;i<300;i++)
  {
	  memcpy( add_dest , add_src , 2400 );  
	  add_src  = add_src  + 2400;
	  add_dest = add_dest + surf.lPitch;
  }

  lpDDSBack->Unlock( NULL );

  //***
  
  free(tmp);

  _enddbgProc();
  return DD_OK;
}

//******************************************************
// Filp the surface
//******************************************************
long gui_flip()
{
  _startdbgProc("gui_flip");
  HRESULT ddrval;
  
  // Flip the surfaces
  while( ( ddrval = lpDDSPrimary->Flip( NULL, 0 ) )
	     == DDERR_WASSTILLDRAWING );

  _enddbgProc();
  return ddrval;
}

//******************************************************
// Restore surface
//******************************************************
void gui_restore()
{
  _startdbgProc("gui_restore");
  lpDDSPrimary->Restore();
  lpDDSBack->Restore();
  _enddbgProc();
}

//******************************************************
// Return true when surface are lost
//******************************************************
long gui_is_lost()
{
  return ( (lpDDSPrimary->IsLost() != DD_OK )
	    || (lpDDSBack->IsLost() != DD_OK ) );
}


//********************************************************
// gui_init_draw  : Routine called one time when appli starts
//********************************************************
long gui_init_draw()
{
	_startdbgProc("gui_init_draw");

  long err = DD_OK;

  EK(err) err=ClearSurface(lpDDSBack,0xAF);
  EK(err) err=gui_draw_fixed(lpDDSBack);
  gui_flip();
  EK(err) err=ClearSurface(lpDDSBack,0xAF);
  EK(err) err=gui_draw_fixed(lpDDSBack);

  _enddbgProc();
  return err;
}

//********************************************************
// gui_update  : update Graphic User Interface high screen
//********************************************************
long gui_update_high()
{
	_startdbgProc("gui_update_high");
  long BM_st[33];
  long ID_st[33];
  char tmp[128];
  long err = DD_OK;

  dev_update_signals();

  dev_get_current( tmp );
  EK(err) err=gui_update_current( lpDDSBack , tmp );

  dev_get_sys_time( tmp );
  EK(err) err=gprint( lpDDSBack , tmp ,
		  10, 15, text_font,GREY,BLACK);

  dev_get_BM_status( BM_st );
  EK(err) err=gui_update_bm( lpDDSBack , BM_st );

  dev_get_ID_status( ID_st );
  EK(err) err=gui_update_id( lpDDSBack , ID_st );

  dev_get_filling_mode( tmp );
  EK(err) err=gui_update_filling_mode( lpDDSBack , tmp );

  dev_get_lifetime( tmp );
  EK(err) err=gui_update_lifetime( lpDDSBack , tmp );

  dev_get_since( tmp );
  EK(err) err=gui_update_since( lpDDSBack , tmp );

  _enddbgProc();
  return err;
}

//********************************************************
// gui_update  : update Graphic User Interface low screen1
//********************************************************
long gui_update_low1()
{
  _startdbgProc("gui_update_low1");
  long err = DD_OK;

  EK(err) err=FillRect( lpDDSBack , 0 , 300 , 800 , 600 , GREY );
  EK(err) err=gui_draw_current_hist(lpDDSBack , small_font , def_font , 
	                    Hist_Tm , Hist_Vl , Hist_Nb );

  _enddbgProc();
  return err;
}

//********************************************************
// gui_update : update Graphic User Interface low extra screen
//********************************************************
long gui_update_low3(PIXEL *data)
{
  _startdbgProc("gui_update_low3");
  long err = DD_OK;
  EK(err) err=FillSurface(lpDDSBack,data,0,300,800,300);
  _enddbgProc();
  return err;
}

//********************************************************
// gui_update  : update Graphic User Interface low screen2
//********************************************************
long gui_update_low2()
{
  _startdbgProc("gui_update_low2");
  char tmp[1024];
  long err = DD_OK;
  char s1[32],s2[32],s3[32],s4[32];

  EK(err) err=FillRect( lpDDSBack , 0 , 300 , 800, 600 , GREY );
  // Draw edge and background of operator message
  EK(err) err=gui_draw_shadow( lpDDSBack , 5 , 530 , 797 , 595 );
  EK(err) err=FillRect( lpDDSBack , 7 , 532 , 795 , 593 , SYN_BLUE );

  dev_get_opp_message( tmp );
  if( strcmp(last_opmesg,tmp)!=0 ) {
	strcpy(last_opmesg,tmp);
    EK(err) err=gui_update_init_opmesg(tmp);
  }

  dev_get_tuneh(s1);
  dev_get_tunev(s2);
  EK(err) err=gui_update_tune(lpDDSBack,s1,s2);

  dev_get_orbitrmsh(s1);
  dev_get_orbitrmsv(s2);
  dev_get_orbitpeakh(s3);
  dev_get_orbitpeakv(s4);
  EK(err) err=gui_update_orbit(lpDDSBack,s1,s2,s3,s4);

  dev_get_emitd9x(s1);
  dev_get_emitd9z(s2);
  dev_get_emitid8x(s3);
  dev_get_emitid8z(s4);
  EK(err) err=gui_update_emit(lpDDSBack,s1,s2,s3,s4);

  //dev_get_dipolenmr(s1);
  //dev_get_dipolehall(s2);
  //EK(err) err=gui_update_field(lpDDSBack,s1,s2);

  dev_get_avpress(s1);
  EK(err) err=gui_update_avpress(lpDDSBack,s1);


  //dev_get_avpress(char *ret);

  _enddbgProc();
  return err;
}