/*================================================================
 *
 *  File:       graphics.cpp
 *  Author:		JL PONS
 *  Project:	CTRM Video synopsis
 *  Content:    Graphics low level routines
 *  Date:		May 1999
 *
 ******************************************************************/
#include "stdafx.h"
#undef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#include <stdio.h>
#include "graphics.h"
#include "gif.h"
#include "Log.h"
#include "resize.h"

extern char *GetDDErrorMsg(int code);

LPDIRECTDRAW            lpDD;           // DirectDraw object
LPDIRECTDRAWSURFACE     lpDDSPrimary;   // DirectDraw primary surface
LPDIRECTDRAWSURFACE     lpDDSBack;      // DirectDraw back surface
LPDIRECTDRAWSURFACE     lpDDSImage;     // DirectDraw temporary surface

HBITMAP				    back_bm;		// Background bitmap
HFONT					def_font;		// default font
HFONT					current_font;	// Current filed font
HFONT					text_font;		// Text font
HFONT					mode_font;		// mode font
HFONT					small_font;		// font for graph time axis
HFONT					bl_font;		// font for bm and id

unsigned char bmp_head[54]={ 
 66, 77, 54,249, 21,  0,  0,  0,  0,  0, 54,  0,  0,  0, 40,  0,  0,  0, 32,  3,
  0,  0, 88,  2,  0,  0,  1,  0, 24,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
 };


//********************************************************
// graphics_init  : initialise all graphics object
// HWND hwnd : Handle to main window
// err_str   : countains error when routine fails
// Return 0 when fails
//********************************************************
int graphics_init( HWND hwnd , char *err_str )
{
	_startdbgProc("graphics_init");

	DDSURFACEDESC       ddsd;
    DDSCAPS             ddscaps;
    HRESULT             ddrval;
	LOGFONT				lf;
	/*
     * create the main DirectDraw object
     */
    ddrval = DirectDrawCreate( NULL, &lpDD, NULL );
    if( ddrval != DD_OK )
    {
		sprintf(err_str,"DirectDraw init failed:%s",GetDDErrorMsg(ddrval));
		_enddbgProc();
        return 0;
    }

    // Get exclusive mode
    ddrval = lpDD->SetCooperativeLevel( hwnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN );
    if( ddrval != DD_OK )
    {
		sprintf(err_str,"DirectDraw SetCooperativeLevel() failed:%s",GetDDErrorMsg(ddrval));
		_enddbgProc();
        return 0;
    }
	

    // Set the video mode to 800x600 True color

	
	#ifdef _24BITS
    ddrval = lpDD->SetDisplayMode(800,600,24);
    #else 
    ddrval = lpDD->SetDisplayMode(800,600,32);
    #endif
	


	if( ddrval != DD_OK )
    {
		sprintf(err_str,"DirectDraw SetDisplayMode() failed:%s",GetDDErrorMsg(ddrval));
		_enddbgProc();
        return 0;
    }


    // Create the primary surface with 1 back buffer
	ZeroMemory(&ddsd, sizeof(ddsd));

    ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | 
                          DDSCAPS_FLIP |
                          DDSCAPS_COMPLEX;

    ddsd.dwBackBufferCount = 1;
    ddrval = lpDD->CreateSurface( &ddsd, &lpDDSPrimary, NULL );
    if( ddrval != DD_OK )
    {
		sprintf(err_str,"DirectDraw CreateSurface() failed:%s",GetDDErrorMsg(ddrval));
		_enddbgProc();
        return 0;
    }

    ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
    ddrval = lpDDSPrimary->GetAttachedSurface(&ddscaps, &lpDDSBack);
    if( ddrval != DD_OK )
    {
		sprintf(err_str,"DirectDraw GetAttachedSurface() failed:%s",GetDDErrorMsg(ddrval));
		_enddbgProc();
        return 0;
    }

#ifdef DDRAW_RESIZE
    // Create a temporay surface
    ZeroMemory( &ddsd, sizeof(ddsd) );
    ddsd.dwSize         = sizeof(ddsd);
    ddsd.dwFlags        = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    ddsd.dwWidth		= 400;
    ddsd.dwHeight		= 150;

    ddrval = lpDD->CreateSurface( &ddsd, &lpDDSImage, NULL );
    if( ddrval != DD_OK )
    {
		sprintf(err_str,"DirectDraw CreateSurface() failed:%s",GetDDErrorMsg(ddrval));
		_enddbgProc();
        return 0;
    }
#endif

	//
	// Initialise fonts
	//

	lf.lfHeight=30;
	lf.lfWidth =12;
	lf.lfEscapement=0;
	lf.lfOrientation=0;
	lf.lfWeight=900;
	lf.lfItalic=FALSE;
	lf.lfUnderline=FALSE;
	lf.lfStrikeOut=FALSE;
	lf.lfCharSet=DEFAULT_CHARSET;
	lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	lf.lfQuality=DEFAULT_QUALITY;
	lf.lfPitchAndFamily=FIXED_PITCH;
	strcpy(lf.lfFaceName,"Courrier");

	def_font = CreateFontIndirect( &lf );

	lf.lfHeight=35;
	lf.lfWidth =18;
	lf.lfEscapement=0;
	lf.lfOrientation=0;
	lf.lfWeight=900;
	lf.lfItalic=FALSE;
	lf.lfUnderline=FALSE;
	lf.lfStrikeOut=FALSE;
	lf.lfCharSet=DEFAULT_CHARSET;
	lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	lf.lfQuality=DEFAULT_QUALITY;
	lf.lfPitchAndFamily=FIXED_PITCH;
	strcpy(lf.lfFaceName,"Courrier");

	bl_font = CreateFontIndirect( &lf );

	lf.lfHeight=110;
	lf.lfWidth =35;
	lf.lfEscapement=0;
	lf.lfOrientation=0;
	lf.lfWeight=800;
	lf.lfItalic=FALSE;
	lf.lfUnderline=FALSE;
	lf.lfStrikeOut=FALSE;
	lf.lfCharSet=DEFAULT_CHARSET;
	lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	lf.lfQuality=DEFAULT_QUALITY;
	lf.lfPitchAndFamily=DEFAULT_PITCH;
	strcpy(lf.lfFaceName,"Times");

	current_font = CreateFontIndirect( &lf );

	lf.lfHeight=40;
	lf.lfWidth =18;
	lf.lfEscapement=0;
	lf.lfOrientation=0;
	lf.lfWeight=900;
	lf.lfItalic=FALSE;
	lf.lfUnderline=FALSE;
	lf.lfStrikeOut=FALSE;
	lf.lfCharSet=DEFAULT_CHARSET;
	lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	lf.lfQuality=DEFAULT_QUALITY;
	lf.lfPitchAndFamily=FIXED_PITCH;
	strcpy(lf.lfFaceName,"Courrier");

	text_font = CreateFontIndirect( &lf );

	lf.lfHeight=90;
	lf.lfWidth =25;
	lf.lfEscapement=0;
	lf.lfOrientation=0;
	lf.lfWeight=600;
	lf.lfItalic=FALSE;
	lf.lfUnderline=FALSE;
	lf.lfStrikeOut=FALSE;
	lf.lfCharSet=DEFAULT_CHARSET;
	lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	lf.lfQuality=DEFAULT_QUALITY;
	lf.lfPitchAndFamily=DEFAULT_PITCH;
	strcpy(lf.lfFaceName,"Times");

	mode_font = CreateFontIndirect( &lf );

	lf.lfHeight=20;
	lf.lfWidth =10;
	lf.lfEscapement=0;
	lf.lfOrientation=0;
	lf.lfWeight=300;
	lf.lfItalic=FALSE;
	lf.lfUnderline=FALSE;
	lf.lfStrikeOut=FALSE;
	lf.lfCharSet=DEFAULT_CHARSET;
	lf.lfOutPrecision=OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	lf.lfQuality=DEFAULT_QUALITY;
	lf.lfPitchAndFamily=DEFAULT_PITCH;
	strcpy(lf.lfFaceName,"Times");

	small_font = CreateFontIndirect( &lf );


    _enddbgProc();
	return 1;
}

//*******************************************
// ClearSurface : Clear a DD surface
//*******************************************
long ClearSurface(LPDIRECTDRAWSURFACE lpDD,BYTE graylevel)
{
  _startdbgProc("ClearSurface");
  DDSURFACEDESC surf;
  BYTE *add_surf;
  int sl,j;
  HRESULT dderr;

  // Lock and Get address of surfaces

  surf.dwSize = sizeof(surf);

  while( ( dderr = lpDD->Lock( NULL , &surf , 0 , NULL ) )
         == DDERR_WASSTILLDRAWING);

  if( dderr != DD_OK ) {
      _enddbgProc();
	  return dderr;
  }

  add_surf = (BYTE *)surf.lpSurface;

  #ifdef _24BITS
    sl=surf.lPitch;
    for( j=0 ; j<600 ; j++,add_surf += sl )
      memset(add_surf,graylevel,2400 ); // 800*3
  #else
    sl=surf.lPitch;
    for( j=0 ; j<600 ; j++,add_surf += sl)
      memset(add_surf,graylevel,3200 ); // 800*4
  #endif

  
  dderr = lpDD->Unlock( NULL );

  _enddbgProc();
  return dderr;
}

//*******************************************
// FillSurface : Fill surface with data
//*******************************************
long FillSurface(LPDIRECTDRAWSURFACE lpDD,PIXEL *src,int x,int y,int w,int h)
{
  _startdbgProc("FillSurface");

  DDSURFACEDESC surf;
  BYTE *add_surf;
  int sl,j;
  HRESULT dderr;

  // Lock and Get address of surfaces

  surf.dwSize = sizeof(surf);

  while( ( dderr = lpDD->Lock( NULL , &surf , 0 , NULL ) )
         == DDERR_WASSTILLDRAWING);

  if( dderr != DD_OK ) {
      _enddbgProc();
	  return dderr;
  }

  add_surf = (BYTE *)surf.lpSurface + y*surf.lPitch;

  sl=surf.lPitch;
  for( j=0 ; j<h ; j++,add_surf += sl,src+=w )
    memcpy(add_surf,src,w*sizeof(PIXEL) );

  dderr = lpDD->Unlock( NULL );

  _enddbgProc();
  return dderr;
}

//********************************************
// DarkenRect : Mare the specified rect darken
//********************************************
long DarkenRect(LPDIRECTDRAWSURFACE lpDD,int x1,int y1,int x2,int y2)
{
  _startdbgProc("FillSurface");

  DDSURFACEDESC surf;
  BYTE *add_surf;
  int sl,i,j;
  HRESULT dderr;

  // Lock and Get address of surfaces

  surf.dwSize = sizeof(surf);

  while( ( dderr = lpDD->Lock( NULL , &surf , 0 , NULL ) )
         == DDERR_WASSTILLDRAWING);

  if( dderr != DD_OK ) {
      _enddbgProc();
	  return dderr;
  }

  add_surf = (BYTE *)surf.lpSurface + y1*surf.lPitch;

  sl=surf.lPitch;
  for( j=0 ; j<(y2-y1+1) ; j++,add_surf += sl ) {
	 for( i=x1;i<=x2;i++ ) {	   
       add_surf[i*sizeof(PIXEL)  ]/=2;
       add_surf[i*sizeof(PIXEL)+1]/=2;
       add_surf[i*sizeof(PIXEL)+2]/=2;
	 }
  }

  dderr = lpDD->Unlock( NULL );

  _enddbgProc();
  return dderr;
}

//***********************************
// Print a string on a surface
//***********************************
long gprint(LPDIRECTDRAWSURFACE lpDDS, char *msg ,
			int x , int y , HFONT fnt,
			DWORD bkcolor , DWORD frcolor) 
{
  _startdbgProc("gprint");
  HDC                 hdc;
  HRESULT             hr;

  if (lpDDS == NULL) {
      _enddbgProc();
      return DDERR_INVALIDPARAMS;
  }

  //
  // get DC and output str.
  //

  if ((hr = lpDDS->GetDC(&hdc)) == DD_OK)
  {

	SetBkMode(hdc, OPAQUE);
	SetBkColor(hdc, bkcolor);
	SetTextColor( hdc , frcolor );
	SelectObject( hdc , fnt );
	TextOut( hdc, x, y, msg, strlen(msg) );
    hr=lpDDS->ReleaseDC(hdc);
  }

  _enddbgProc();
  return hr;
}

//***********************************
// Print a string on a surface
//***********************************
long gprint_trans(LPDIRECTDRAWSURFACE lpDDS , char *msg ,
			      int x , int y , HFONT fnt,
			      DWORD frcolor) 
{
  _startdbgProc("gprint_trans");
  HDC                 hdc;
  HRESULT             hr;

  if (lpDDS == NULL) {
      _enddbgProc();
      return DDERR_INVALIDPARAMS;
  }

  //
  // get DC and output str.
  //

  if ((hr = lpDDS->GetDC(&hdc)) == DD_OK)
  {

	SetBkMode(hdc, TRANSPARENT);
	SetTextColor( hdc , frcolor );
	SelectObject( hdc , fnt );
	TextOut( hdc, x, y, msg, strlen(msg) );
    hr=lpDDS->ReleaseDC(hdc);
  }

  _enddbgProc();
  return hr;
}

long measure_text(LPDIRECTDRAWSURFACE lpDDS , char *msg , HFONT fnt,int *widht,int *height) {

  _startdbgProc("measure_text");
  HDC                 hdc;
  HRESULT             hr;

  if (lpDDS == NULL) {
      _enddbgProc();
      return DDERR_INVALIDPARAMS;
  }

  //
  // get DC and output str.
  //

  if ((hr = lpDDS->GetDC(&hdc)) == DD_OK)
  {
	SIZE r;
	SelectObject( hdc , fnt );
	GetTextExtentPoint(hdc,msg,strlen(msg),&r);
	*widht  = r.cx;
	*height = r.cy;
    hr=lpDDS->ReleaseDC(hdc);
  }

  _enddbgProc();
  return hr;

}

//**************************************************
// Draw a line
//**************************************************
long Line(LPDIRECTDRAWSURFACE lpDDS, 
		  int x1, int y1 ,int x2, int y2,DWORD color)
{
	_startdbgProc("Line");
    HDC                 hdc;
    HRESULT             hr;
	LOGPEN				lp;
	HPEN				hp;
	POINT				oldp;

    if (lpDDS == NULL) {
	    _enddbgProc();
        return DDERR_INVALIDPARAMS;
	}

	//
	// Create brush
	//
	lp.lopnStyle=PS_SOLID;
	lp.lopnColor=color;
	lp.lopnWidth.x=1;
	lp.lopnWidth.y=1;
	hp=CreatePenIndirect(&lp);
    //
    // get DC and draw line.
    //

    if ((hr = lpDDS->GetDC(&hdc)) == DD_OK)
    {
		SetBkMode(hdc, TRANSPARENT);
		SelectObject( hdc , hp );
		MoveToEx( hdc, x1, y1 ,&oldp);
		LineTo( hdc, x2, y2);
        hr=lpDDS->ReleaseDC(hdc);
    }

	if( !DeleteObject(hp) ) {
	    _enddbgProc();
		return DDERR_UNSUPPORTED;
	}

    _enddbgProc();
	return hr;
}

//**************************************************
// Draw a dottef line
//**************************************************
long LineDot(LPDIRECTDRAWSURFACE lpDDS, 
		  int x1, int y1 ,int x2, int y2,DWORD color)
{
	_startdbgProc("LineDot");
    HDC                 hdc;
    HRESULT             hr;
	LOGPEN				lp;
	HPEN				hp;
	POINT				oldp;

    if (lpDDS == NULL) {
	    _enddbgProc();
        return DDERR_INVALIDPARAMS;
	}

	//
	// Create brush
	//
	lp.lopnStyle=PS_DOT;
	lp.lopnColor=color;
	lp.lopnWidth.x=1;
	lp.lopnWidth.y=1;
	hp=CreatePenIndirect(&lp);
    //
    // get DC and draw line.
    //

    if ((hr = lpDDS->GetDC(&hdc)) == DD_OK)
    {
		SetBkMode(hdc, TRANSPARENT);
		SelectObject( hdc , hp );
		MoveToEx( hdc, x1, y1 ,&oldp);
		LineTo( hdc, x2, y2);
        hr=lpDDS->ReleaseDC(hdc);
    }

	if( !DeleteObject(hp) ) {
	    _enddbgProc();
		return DDERR_UNSUPPORTED;
	}

    _enddbgProc();
	return hr;
}

//**************************************************
// Fill a rectangle
//**************************************************
long FillRect(LPDIRECTDRAWSURFACE pdds, 
			  int x1, int y1 ,int x2, int y2,DWORD color)
{
	_startdbgProc("FillRect");
    HDC                 hdc;
    HRESULT             hr;
	LOGBRUSH			lb;
	HBRUSH				hb;
	HRGN rgn;

    if (pdds == NULL) {
	    _enddbgProc();
        return DDERR_INVALIDPARAMS;
	}

	//
	// Create brush
	//
	lb.lbStyle=BS_SOLID;
	lb.lbColor=color;
	lb.lbHatch=NULL;

	hb=CreateBrushIndirect(&lb);

    //
    // get DC and fill.
    //
    rgn = CreateRectRgn( x1,y1,x2,y2 );

    if ((hr = pdds->GetDC(&hdc)) == DD_OK)
    {
		SetBkMode(hdc, TRANSPARENT);
		SelectObject( hdc , hb );
		PaintRgn( hdc , rgn );
        hr=pdds->ReleaseDC(hdc);
    }

	if(!DeleteObject(hb)) {
	    _enddbgProc();
		return DDERR_UNSUPPORTED;
	}
	if(!DeleteObject(rgn)) {
	    _enddbgProc();
		return DDERR_UNSUPPORTED;
    }

    _enddbgProc();
	return hr;
}

/*
 *  Fill
 *
 *  Fill zone within DirectDrawSurface
 *
 */
long Fill(LPDIRECTDRAWSURFACE pdds, 
		  HRGN rgn,DWORD color)
{
	_startdbgProc("Fill");
    HDC                 hdc;
    HRESULT             hr;
	LOGBRUSH			lb;
	HBRUSH				hb;

    if (pdds == NULL) {
	    _enddbgProc();
        return DDERR_INVALIDPARAMS;
	}

	//
	// Create brush
	//
	lb.lbStyle=BS_SOLID;
	lb.lbColor=color;
	lb.lbHatch=NULL;

	hb=CreateBrushIndirect(&lb);

    //
    // get DC and fill.
    //

    if ((hr = pdds->GetDC(&hdc)) == DD_OK)
    {
		SetBkMode(hdc, TRANSPARENT);
		SelectObject( hdc , hb );
		PaintRgn( hdc , rgn );
        hr=pdds->ReleaseDC(hdc);
    }

	if(!DeleteObject(hb)) {
	    _enddbgProc();
		return DDERR_UNSUPPORTED;
	}

	_enddbgProc();
	return hr;
}

/*
 *  SaveBitmap
 *
 *  Make a bmp file from directdraw surface
 *  from y1 to y2.
 *
 */

#ifdef DDRAW_RESIZE
PIXEL				raw[800*600];
#endif

#if 0
long SaveBitmap(IDirectDrawSurface *pdds,int y1,int y2,char *filename)
{
    HDC                 hdcImage;
    HDC                 hdc;
    DDSURFACEDESC       ddsd;
    HRESULT             hr;
	HBITMAP				hbm;
	HANDLE				bmp_file;
	int					lng;
	unsigned long		wrt;
	int					FlagOK=1;
	int					i,j;


    if (pdds == NULL)
        return E_FAIL;

    //
    // make sure this surface is restored.
    //
    pdds->Restore();

    //
    //  select bitmap into a memoryDC so we can use it.
    //
    hdcImage = CreateCompatibleDC(NULL);

	
    #ifdef _24BITS
	hbm=CreateBitmap(800,y2-y1,1,24,NULL);
    #else
	hbm=CreateBitmap(800,y2-y1,1,32,NULL);
    #endif
    SelectObject(hdcImage, hbm);

    //
    // get size of surface.
    //
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_HEIGHT | DDSD_WIDTH;
    pdds->GetSurfaceDesc(&ddsd);
	
    if ((hr = pdds->GetDC(&hdc)) == DD_OK)
    {
        BitBlt(hdcImage, 0, 0, ddsd.dwWidth, y2-y1, hdc, 0, y1, SRCCOPY);
        pdds->ReleaseDC(hdc);

		#ifdef _24BITS
		GetBitmapBits(hbm,800*(y2-y1)*3,
		(void *)(raw) );
        #else
		GetBitmapBits(hbm,800*(y2-y1)*4,
		(void *)(raw) );
        #endif

    } else {
	  FlagOK=0;
	}

	

	bmp_file=CreateFile(filename,
		                GENERIC_WRITE, 
						0,
						NULL,
						CREATE_ALWAYS,
					    FILE_ATTRIBUTE_NORMAL,
					    NULL);

	// Write dimensions
	bmp_head[22]=(y2-y1) & 0xFF;
	bmp_head[23]=(y2-y1) / 0x100;

	#ifdef _24BITS

	  if(bmp_file!=INVALID_HANDLE_VALUE) {

	    lng=sizeof(bmp_head);
        WriteFile(bmp_file,bmp_head,lng,&wrt,NULL);
	    lng=800*(y2-y1)*3;
        WriteFile(bmp_file,raw,lng,&wrt,NULL);

	  } else {
	    FlagOK=0;
	  }

    #else

	  if(bmp_file!=INVALID_HANDLE_VALUE) {

	    lng=sizeof(bmp_head);
        WriteFile(bmp_file,bmp_head,lng,&wrt,NULL);
 	    lng=3;

		for(j=(y2-y1)-1;j>=0;j--) {
		  for(i=0;i<800;i++) {
            WriteFile(bmp_file,&(raw[i+800*j]),lng,&wrt,NULL);
		  }
		}

	  } else {
	    FlagOK=0;
	  }

    #endif

	CloseHandle(bmp_file);

    DeleteDC(hdcImage);
	if(!DeleteObject(hbm))
		return DDERR_UNSUPPORTED;


    return FlagOK;
}
#endif

// Fast cmap reduction

int AddColor(GIF_IMAGE *img,PIXEL p) {
	_startdbgProc("AddColor");

	int i=0;
	int found=0;

	// Simplify color
	p.bits[0] &= 0xF0;
	p.bits[1] &= 0xF0;
	p.bits[2] &= 0xF0;

	// Search for color
	while(i<img->used_color && !found) {
		found = ( (img->Red[i]==p.bits[2]) && (img->Green[i]==p.bits[1]) && (img->Blue[i]==p.bits[0]) );
		if( !found ) i++;
	}

	if( !found ) {
		if( img->used_color<255 ) {
             img->Red[img->used_color] = p.bits[2];
             img->Green[img->used_color] = p.bits[1];
             img->Blue[img->used_color] = p.bits[0];
			 img->used_color++;
		}
	}

    _enddbgProc();
	return i;
}



void computeFastPalette(GIF_IMAGE *img,BYTE *dr,BYTE *dg,BYTE *db) {

  int w = img->width;
  int h = img->height;
  int i,j;

#ifdef DDRAW_RESIZE

  for(j=0;j<h;j++) {
    for(i=0;i<w;i++) {
      img.data[i+j*w]=AddColor(img,raw[i+w*j]);
	}
  }

#else

  //
  // Compute palette
  //
  for(j=0;j<h;j++) {
    for(i=0;i<w;i++) {
 	  PIXEL p;
	  int id=i+j*w;
	  p.bits[2]=dr[id];
	  p.bits[1]=dg[id];
	  p.bits[0]=db[id];
      img->data[id]=AddColor(img,p);
	}
  }

#endif

}

void computeGreyPalette(GIF_IMAGE *img,BYTE *dg) {

  int w = img->width;
  int h = img->height;
  int i,j;

  //
  // Compute palette
  //
  for(j=0;j<h;j++) {
    for(i=0;i<w;i++) {
	  int id=i+j*w;
      img->data[id]=dg[id];
	}
  }

  for(i=0;i<256;i++) {
	img->Red[i] = (BYTE)i;
	img->Green[i] = (BYTE)i;
	img->Blue[i] = (BYTE)i;
  }

  img->used_color=255;

}


// HQ cmap reduction

int AddColor24(int *nbcol,PIXEL *cmap,PIXEL p) {

	int b = p.bits[0];
	int g = p.bits[1];
	int r = p.bits[2];
	int i = 0;
	int found=FALSE;

	DWORD c = p.bits[0]*65536L + p.bits[1]*256L + p.bits[2];
	DWORD cm;

	// Search for color

	while(i<*nbcol && !found) {
        cm = cmap[i].bits[0]*65536L + cmap[i].bits[1]*256L + cmap[i].bits[2];
		found = (c == cm);
		if( !found ) i++;
	}

	if( !found ) {
	  
	  // New color
	  cmap[*nbcol] = p;
	  *nbcol = *nbcol + 1;
	  return (*nbcol)-1;

	} else {

	  return i;

	}

}

int _max(int i1,int i2,int i3) {
  int r = i1;
  if( i2>r ) r=i2;
  if( i3>r ) r=i3;
  return r;
}

void computeHqPalette(GIF_IMAGE *img,BYTE *dr,BYTE *dg,BYTE *db) {

  int w = img->width;
  int h = img->height;
  int l=w*h;
  int i,j,d,dmin,imin,jmin,vr,vg,vb;
  char tmp[256];
 
  int nbcol=0;
  PIXEL *cmap=(PIXEL *)malloc(sizeof(PIXEL)*l);
  int   *tmpi=(int *)malloc(sizeof(int)*l);

  DWORD t1 = GetTickCount();

  // ========= Parse picture

  for(j=0;j<h;j++) {
    for(i=0;i<w;i++) {
 	  PIXEL p;
	  int id=i+j*w;
	  p.bits[2]=dr[id];
	  p.bits[1]=dg[id];
	  p.bits[0]=db[id];
      tmpi[id]=AddColor24(&nbcol,cmap,p);
	}
  }

  DWORD t2 = GetTickCount();

  sprintf(tmp,"Got %d colors (%d ms)",nbcol,t2-t1);
  MessageBox(NULL,tmp,"Info",MB_OK);

  t2 = GetTickCount();

  // ========= Reduce color

  while(nbcol>256) {

	 //  -------------- Search for the 2 nearest colors -------------- 
	 dmin = 1048576;

	 for(i=0;i<nbcol;i++) {
	   for(j=i+1;j<nbcol;j++)
	   {

		  vr = (int)cmap[i].bits[2] - (int)cmap[j].bits[2];
		  vg = (int)cmap[i].bits[1] - (int)cmap[j].bits[1];
		  vb = (int)cmap[i].bits[0] - (int)cmap[j].bits[0];		  
		  d = _max( vr*vr , vg*vg , vb*vb );

		  if( d<dmin ) {
			imin = i;
			jmin = j;
			dmin = d;
		  }

	   }
	 }

	 // -------------- Replace reference to jmin add update image -
	 for(i=0;i<l;i++) {
		 if(tmpi[i]>=jmin) {
		   if(tmpi[i]==jmin) tmpi[i]=imin;
		   else              tmpi[i]--;
		 }
	 }

     // -------------- Remove jmin color -------------- 
     nbcol--;
	 for(i=jmin;i<nbcol;i++) cmap[i] = cmap[i+1];


  }

  DWORD t3 = GetTickCount();
  sprintf(tmp,"Reduced to %d colors (%d ms)",nbcol,t3-t2);
  MessageBox(NULL,tmp,"Info",MB_OK);

  // ========= Build image color

  for( i=0;i<l;i++ ) img->data[i]=tmpi[i];
  for( i=0;i<nbcol;i++ ) {
    img->Red[i]   = cmap[i].bits[2];
    img->Green[i] = cmap[i].bits[1];
    img->Blue[i]  = cmap[i].bits[0];
  }

  img->used_color = nbcol;

  free(cmap);
  free(tmpi);

}


/*
 *  SaveGifFromRGB
 *
 *  Make a gif file from a rectangle of a directdraw surface 
 *  xs,ys,ws,hs are coordinates of rectangle to resize in the DDsurface.
 *  Resize rectangle to w,h
 *  filename is the name of the GIF file.
 *  how = resize method (see resize.cpp)
 *  nbstep : number of sizing step
 *  palmode = Color reduction quality (0=fast, 1=HQ, 2=Grey Scale)
 *  Return 0 when failed, GifErrorMessage contains error text.
 */
long SaveGifFromRGB(IDirectDrawSurface *pdds,
					int xs,int ys,
					int ws,int hs,
					int w,int h,
					char *filename,
					int nbstep,
					int how,
					int palmode)
{
	_startdbgProc("SaveGifFromRGB");


    if (pdds == NULL) {
        _enddbgProc();
		return FALSE;
	}
    pdds->Restore();


    DDSURFACEDESC       ddsd;
	int					FlagOK=1;
	int					i,j;
	GIF_IMAGE			img;
	HRESULT				hr;

	BYTE *dstred;
	BYTE *dstgreen;
	BYTE *dstblue;
	BYTE *srcred;
	BYTE *srcgreen;
	BYTE *srcblue;
	BYTE *ptr;

	//
	// ------- Allocate frame
	//

	srcred  =(BYTE *)malloc(ws*hs);
	srcgreen=(BYTE *)malloc(ws*hs);
	srcblue =(BYTE *)malloc(ws*hs);
	
    dstred  =(BYTE *)malloc(ws*hs);
    dstgreen=(BYTE *)malloc(ws*hs);
    dstblue =(BYTE *)malloc(ws*hs);

	//
	// ------- Get data
	//

    ZeroMemory( &ddsd, sizeof( ddsd ) );
    ddsd.dwSize = sizeof( ddsd );

    hr=pdds->Lock( NULL, &ddsd, 0, NULL);
	if(hr!=DD_OK)
	{ 
	   strcpy(GifErrorMessage,"Failed to lock the surface\n");
	   strcat(GifErrorMessage,GetDDErrorMsg(hr));
       _enddbgProc();
       return FALSE;
	}


	for (j=ys;j<ys+hs;j++)
	{
	    ptr = (unsigned char *)ddsd.lpSurface  + j*ddsd.lPitch;
		for(i=xs;i<xs+ws;i++) {

			int id=(i-xs)+ws*(j-ys);
			#ifdef _24BITS
			  srcblue[id] =ptr[3*i+0];
			  srcgreen[id]=ptr[3*i+1];
			  srcred[id]  =ptr[3*i+2];
			#else
			  srcblue[id] =ptr[4*i+0];
			  srcgreen[id]=ptr[4*i+1];
			  srcred[id]  =ptr[4*i+2];
			#endif

		}
	}

	hr=pdds->Unlock(NULL);
	if(hr!=DD_OK)
	{ 
	   strcpy(GifErrorMessage,"Failed to unlock the surface\n");
	   strcat(GifErrorMessage,GetDDErrorMsg(hr));
       _enddbgProc();
	   return FALSE;
	}

	//
	// ------- Resize
	//

	int wt,ht;
	int lwt,lht;
	lwt = ws;
	lht = hs;

	for( i=1 ; i<=nbstep ; i++ ) {	  
	  wt = (int)( (double)ws + (double)i*(double)(w-ws)/(double)(nbstep) );
	  ht = (int)( (double)hs + (double)i*(double)(h-hs)/(double)(nbstep) );
      InitFilters(lwt,lht,wt,ht);
      //char tmp[256];
	  //sprintf(tmp,"InitFilters(%d,%d,%d,%d)",lwt,lht,wt,ht);
	  //Log(tmp,0);
	  ResizeFrame(srcblue,dstblue,how);
	  ResizeFrame(srcgreen,dstgreen,how);
	  ResizeFrame(srcred,dstred,how);
	  if( i<nbstep ) {
	    memcpy(srcblue,dstblue,wt*ht);
	    memcpy(srcgreen,dstgreen,wt*ht);
	    memcpy(srcred,dstred,wt*ht);
	  }
	  lwt=wt;
	  lht=ht;
	}


	//
	//  ------- Build the img structure
	//

	strcpy(img.FileName,filename);
	img.width=w;
	img.height=h;
	img.data=(char *)malloc(w*h);
	img.transparent=-1;
	img.format=CONTINOUS_FORMAT;
	img.background=0;
	img.used_color=0;
	img.pixel=8;

	switch(palmode) {
	case 0:
	  computeFastPalette(&img,dstred,dstgreen,dstblue);
	  break;
	case 1:
	  computeHqPalette(&img,dstred,dstgreen,dstblue);
	  break;
	case 2:
	  computeGreyPalette(&img,dstgreen);
	  break;
	}


	//
	//  Free all
	//
	free(srcred);
	free(srcgreen);
	free(srcblue);
	free(dstred);
	free(dstgreen);
	free(dstblue);

	//
	// Save file
	//
	FlagOK=SaveGifImage(img);
	free(img.data);

    _enddbgProc();
    return FlagOK;
}