namespace eval Axe {
  namespace export AxeCreate AxeStartMonitoring AxeStopMonitoring AxeSetUpdatePeriod 
 
proc AxeCreate {parent  maxeid axeid axename} {
#+############################################################################
#NAME:     AxeCreate  
#SYNOPSIS: AxeCreate parent_window axe_type maxe_device name axe_number axe_mnemonic
#DESCRIPTION: Create an axis object that includes currently used commands to deal with
#             an axis based on the Maxe->Vpapaxe ESRF classes.
#             Commands that are not often used can be sent using the menubar->command->Tcl Console
#              (see CreatMainWindow doc for more details about this general facility).
#ARGIN:    parent: The widget name of its parent (generally a frame) in wich all 
#                    widgets are created. The caller have to pack this parent
#          maxeid: the maxe device name to wich the axis belongs
#          axeid: axis number [0 7] into the maxe device
#          axename: A free mnemonic to identify this axis by its name 
#ARGOUT:   NONE
#RETURN:   NONE
#ERROR :   Treated by the interpreter if the creation is impossible (ex. parent
#             already existing). 
#GLOBAL_CREATED: AXE_CTRL AXE_MON_ST AXE_MON_PE AXE_MON_AF . These globals are used
#                within the scope of the object and _MUST_ not be accessed (r or w)
#                from outside this scope. The caller must not define the same globals
#                in his scope. (a future version using the namespace facility will
#                be written to avoid these conflicts).
#GLOBAL_USED: The generic ones
#GLOBAL_MODIFIED: NONE
#PROCEDURE_CALLED: execpt private procs, AxeCreate creates and uses 3 public procs:
#                  AxeStartMonitoring AxeStopMonitoring AxeSetUpdatePeriod
#                  The caller must only use these public procs to deal with the axe
#                  object when it is created.
#SEE ALSO: AxeStartMonitoring AxeStopMonitoring AxeSetUpdatePeriod
#SOURCE FILE: AxeCreate.tcl
#$Author: pepellin $ 
#$Date: 98/08/03 11:53:49 $
#$Revision: 2.3 $
#-############################################################################
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }

### Dynamically create globals to manage this axis
global AXE_CTRL ;# An array to hold the control mode status (true or false)
global AXE_MON_ST ;# An array to hold monitoring status (true or false)
global AXE_MON_PE ;# An array to hold monitoring period in ms
global AXE_MON_AF ;# An array to hold the after pid
set AXE_CTRL($axename) $TRUE
set AXE_MON_ST($axename) $FALSE
set AXE_MON_PE($axename) 2000
set AXE_MON_AF($axename) -1

### create main frames:
#   ftitle: to hold axis name (mnemonic) and axe status
#   fpos: to hold motor and encoder position
#   frel:   to hold buutons for relative motion and stop
#   fcmd:   to hold extras commands
set ftitle [frame $parent.ftitle]
set fpos   [frame $parent.fpos]
set fdelta [frame $parent.fdelta]
set frel   [frame $parent.frel]
set fcmd   [frame $parent.fcmd]


### fill ftitle
set lnames [ label $ftitle.lname -text "$axename" -width 8]
bind $lnames <Button-3> "UpdateStatusbar \"$maxeid   $axeid\""
set lstatus [label $ftitle.lstatus -text "Status: ????" -width 14]
pack $lnames -side left -anchor w
pack $lstatus -side right -anchor e 

### fill fpos
# A little font
set font "*helvetica-bold-r-*-*-6*"
# Motor
set fmpos [frame $fpos.fmpos]
set lmottitle [label $fmpos.ltitle -text "Mot"]
set lmotpos [label $fmpos.lmot -text "???" -fg Red -bg white]
bind $lmotpos <Control-Shift-Button-3> "BalloonHelp %x %y %W %W"
set labs [label $fmpos.labs -text "mv Abs "]
set eabs [entry $fmpos.eabs -width 8]
bind $eabs <Key-Return> "Axe::axeMoveAbs $eabs $maxeid $axeid "
bind $eabs <Button-3> "BalloonHelp %x %y %W \"Move the motor to this absolute position\""
pack $lmottitle $lmotpos -side left -padx 1
pack $labs $eabs -side left -padx 1

# encoder
set fepos [frame $fpos.fepos]
set lenctitle [label $fepos.ltitle -text "Enc" ]
set enctype [axeGetEncType $maxeid $axeid]
if { $enctype != "NONE" } {
  set lencpos [label $fepos.lenc -text "???" -fg Red -bg white]
} else {
    set lencpos [label $fepos.lenc -text "None" -fg Red -bg white]
}
#Units
set lunit [label $fepos.lunit -text ""]
set bunit [menubutton $fepos.bunit -text "???" -menu $fepos.bunit.menu \
                               -font $font -padx 1 -pady 1 \
                               -width 9 ]
bind $bunit <Control-Shift-Button-3> "BalloonHelp %x %y %W %W"                               
menu $fepos.bunit.menu                               
$fepos.bunit.menu add command -label "steps" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"STEPS\""
$fepos.bunit.menu add command -label "mm" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"MM\""
$fepos.bunit.menu add command -label "microm" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"MICRON\""
$fepos.bunit.menu add command -label "rotation" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"ROTATION\""
$fepos.bunit.menu add command -label "degree" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"DEGREE\""
$fepos.bunit.menu add command -label "arcsec" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"ARCSEC\""
$fepos.bunit.menu add command -label "user" -command "Axe::axeChangeUnit $bunit $maxeid $axeid \"USERUNIT\""

pack $lenctitle $lencpos  -side left -padx 1
pack $bunit $lunit -side right -anchor ne -padx 3 -pady 5
pack $fmpos  $fepos  -anchor w -fill x 

### Fill fres whit delta widgets
set ldelta [label $frel.ldelta -text "Rel delta: " -font $font ]
set edelta [entry $frel.delta -width 8 ]
$edelta insert 0 "1"
bind $edelta <Key-Return> "Axe::axeChangeDelta $edelta $maxeid $axeid $axename"
bind $edelta <Button-3> "BalloonHelp %x %y %W \"increment value for relative motion. MUST be >0\""

### fill frel with the following buttons
#   br: relative positive move
#   bl relative positive move
#   bs : stop

set br  [button $frel.br  -text "+"   -relief raised -font $font -padx 1 -pady 1\
                                      -command [list Axe::axeMoveRel $maxeid $axeid $edelta 1]]
bind $br <Button-3> "BalloonHelp %x %y %W \"Positive rel motion of Rel_delta increment. Red if pos switch\""

set bl  [button $frel.bl  -text "-"   -relief raised -font $font -padx 1 -pady 1 \
                                      -command [list Axe::axeMoveRel $maxeid $axeid $edelta -1]]
bind $bl <Button-3> "BalloonHelp %x %y %W \"Negative rel motion of Rel_delta increment. Red if neg switch\""

set bs  [button $frel.bs  -text "Stop"   -relief raised -font $font -padx 1 -pady 1 \
                                         -command [list Axe::axeAbortMotion $maxeid $axeid] ]
bind $bs <Button-3> "BalloonHelp %x %y %W \"Abort the current motion\""

pack  $bl $bs $br   -side left -padx 2
pack $ldelta $edelta -side left

### Fill fcmd
set bcmd [menubutton $fcmd.bcmd -text "Cmd" -menu $fcmd.bcmd.menu -padx 1 -pady 1]
set rbmon [radiobutton $fcmd.rbmon -text "Mon" \
                                 -font $font \
                                 -variable AXE_CTRL($axename) \
                                 -value $FALSE \
                                 -command [list Axe::axeUpdateCtrl $axename $eabs $edelta $br $bs $bl $bunit $bcmd]]
bind $rbmon <Button-3> "BalloonHelp %x %y %W \"Monitoring enabled, control disabled\""                                
bind $rbmon <Control-Shift-Button-3> "BalloonHelp %x %y %W %W"
set rbctrl [radiobutton $fcmd.rbctrl -text "Ctrl" \
                                     -font $font \
                                     -variable AXE_CTRL($axename) \
                                     -value $TRUE \
                                     -command [list Axe::axeUpdateCtrl $axename $eabs $edelta $br $bs $bl $bunit $bcmd]] 
bind $rbctrl <Button-3> "BalloonHelp %x %y %W \"Monitoring enabled, control enabled\""
bind $rbctrl <Control-Shift-Button-3> "BalloonHelp %x %y %W %W"
set bres [button $fcmd.bres -text "Res" -relief raised -font $font \
                            -padx 1 -pady 1 \
                            -command "Axe::axeGetRes $maxeid $axeid $axename"]
bind $bres <Button-3> "BalloonHelp %x %y %W \"Display resources\""
bind $bres <Control-Shift-Button-3> "BalloonHelp %x %y %W %W"

# create and fill a menu button for axe object extra commands
#menubutton $fcmd.bcmd -text "Cmd" -menu $fcmd.bcmd.menu -padx 1 -pady 1
bind $fcmd.bcmd <Button-3> "BalloonHelp %x %y %W \"Extra commands for this axe object\""
bind $fcmd.bcmd <Control-Shift-Button-3> "BalloonHelp %x %y %W %W"
menu $fcmd.bcmd.menu
$fcmd.bcmd.menu add command -label "Goto Reference (DevMoveReference)" \
                            -command "Axe::axeMoveReference $maxeid $axeid $axename"
$fcmd.bcmd.menu add command -label "Set Motor Counter (DevLoadPosition)" \
                            -command "Axe::axeLoadPosition $maxeid $axeid $axename"
$fcmd.bcmd.menu add command -label "Set Encoder Counter (DevLoadEncPos)" \
                            -command "Axe::axeLoadEncPos $maxeid $axeid $axename $lencpos"  
$fcmd.bcmd.menu add command -label "Set Velocity (DevSetVelocity)" \
                            -command "Axe::axeSetVelocity $maxeid $axeid $axename "                                                      
pack $rbmon $rbctrl $bres $fcmd.bcmd -side left -padx 2                                   
                                                                    
### Pack main frames widgets into $parent
pack $ftitle -fill x 
pack $fpos 
pack $frel  -pady 5
pack $fcmd -after $frel -pady 3

### Start update
UpdateAxe $lmotpos $lencpos $lstatus $bunit $br $bl $maxeid $axeid $axename

} ;# ends proc AxeCreate

proc axeGetEncType {maxeid axeid} {
# return the encoder type of this axis from the resource file
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
 set vaxe [db_io_get s e $maxeid axe_ident$axeid]
 set enctype [db_io_get s e $vaxe enc_type]
 return $enctype
} ;#ends proc axeGetEncType

proc axeChangeUnit { w maxe axe unit } {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
global UNITS2N
UpdateCommandSent "Change unit $maxe $axe $UNITS2N($unit)"
 set argin [list $axe $UNITS2N($unit)]
 dev_io s e $maxe DevSetUnits argin
 if {$s == $DS_OK} {
   $w configure -text $unit
 } else {
  AddNewError "[GetDate] Set unit ($unit) on $maxe  axe: $axe failed"
  AddNewError $e 
 }
}


proc axeChangeDelta {w maxe axeid axename } {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt}
set newdelta [$w get]
if {[CheckEntry $newdelta float 0 *] == $FALSE } {
  UpdateCommandSent "Change delta $maxe $axename to $newdelta"
  $w configure -bg Orange
} else {$w configure -bg grey80}
}

proc UpdateAxe {wmot wenc wstatus wunit wbr wbl maxe axe axename} {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
global UNITN2S AXE_MON_ST AXE_MON_PE AXE_MON_AF

set callback  [namespace code [list Axe::UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]]

# Check if the monitoring is enabled
if {$AXE_MON_ST($axename) == $FALSE } {
  $wmot configure -text "????" 
  if { [$wenc cget -text] == "motenc" } {$wenc configure -text  "????"}
  $wstatus configure -text "Status: ????"
  $wunit configure -text "????"
  #set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
  set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]
  return
}
set argin $axe

### get and update  status
set res [dev_io s e $maxe DevReadState argin]
if {$s != $DS_OK} {
AddNewError "updateAxe: DevReadState on device $maxe $axe $axename failed"
AddNewError $e
#set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]

return
}
switch [lindex $res 0] {
  2       {set status "ON"}
  9       {set status "MOVING"}
  23      {set status "FAULT"}
  default {set status "????"}
}
$wstatus configure -text "Status: $status"

### get units
set resunit [dev_io s e $maxe DevReadUnits argin]
if {$s != $DS_OK} {
AddNewError "updateAxe: DevReadUnits on device $maxe $axe $axename failed"
AddNewError $e
#set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]

return
}
$wunit configure -text $UNITN2S($resunit)
# Set the display format accordeing to the unit
switch $UNITN2S($resunit) {
  STEPS    -
  MICRON   -
  ROTATION {set dformat "%.0f"}
  MM       -
  DEGREE   -
  USERUNIT {set dformat "%.3f"}
  default  {set dformat "???" }
}
### Get motor position
set res [dev_io s e $maxe DevReadPosition argin]
if {$s != $DS_OK} {
AddNewError "updateAxe: DevReadPosition on device $maxe $axe $axename failed"
AddNewError $e
#set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]

return
}
set resformat [format $dformat $res]
$wmot configure -text $resformat -fg black

### Get encoder position if encoder isrequested (type != None)
if { [$wenc cget -text] != "None" } {
  set res [dev_io s e $maxe DevReadEncPos argin]
  if {$s != $DS_OK} {
    AddNewError "updateAxe: DevReadEncPos on device $maxe $axe $axename failed"
    AddNewError $e
    #set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
    set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]

    return
  }
  set resformat [format $dformat $res]
  $wenc configure -text $resformat -fg black
}

### Read switches
  set sw [dev_io s e $maxe DevReadSwitches argin]
  if {$s != $DS_OK} {
    AddNewError "updateAxe: DevReadSwitches on device $maxe $axe $axename failed"
    AddNewError $e
    #set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
    set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]

    return
  }
  switch $sw {
    0 {$wbr configure -fg Black ; $wbl configure -fg Black }
    1 {$wbr configure -fg Black ; $wbl configure -fg Red }
    2 {$wbr configure -fg Red   ; $wbl configure -fg Black } 
    3 {$wbr configure -fg Red   ; $wbl configure -fg Red } 
    default { AddNewError "updateAxe: DevReadSwitches returns $sw" }
  }
### restart the proc
#set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) UpdateAxe $wmot $wenc $wstatus $wunit $wbr $wbl $maxe $axe $axename]
set AXE_MON_AF($axename) [after $AXE_MON_PE($axename) eval $callback]


} ;#ends proc UpdateAxe

proc axeMoveRel {maxe axe wdelta sign} {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }

set value [expr [$wdelta get] * $sign]
set argin [list $axe $value]
UpdateCommandSent "dev_io s e $maxe DevMoveRelative $argin"
dev_io s e $maxe DevMoveRelative argin
if {$s != $DS_OK} {
AddNewError "DevMoveRelative on device $maxe axe# $axe failed"
AddNewError $e
return
}
} ;# ends proc proc axeMoveRel

proc axeMoveAbs { w maxe axe } {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
set value [$w get]
if { ![CheckEntry $value float]} {
  $w configure -bg Orange
  return
}
$w configure -bg grey80
set argin [list $axe $value]
UpdateCommandSent "dev_io s e $maxe DevMoveAbsolute $argin"
dev_io s e $maxe DevMoveAbsolute argin
if {$s != $DS_OK} {
AddNewError "updateAxe: DevMoveAbsolute on device $maxe axe# $axe failed"
AddNewError $e
return
}
}

proc axeAbortMotion {maxe axe} {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
set argin $axe
dev_io s e $maxe DevAbortCommand argin
if {$s != $DS_OK} {
AddNewError "updateAxe: DevAbortCommand on device $maxe axe# $axe failed"
AddNewError $e
return
}
}

proc axeGetRes {maxe axe axename} {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
# get and display the resources for an axis object

# Get maxe resources into list l0
  set pipe [open "|dbm_devres $maxe"]
  while { [gets $pipe line] >= 0} {lappend l0 $line}
    catch {close $pipe} ; unset pipe
  if { [info exist l0] != 1 } {
    tk_messageBox -message "Resources not accessibles from your host"\
                            -type ok -icon error
    return
  }
  set l0 [lsort $l0]
  set l0 [linsert $l0 0 "+++++++++++ $maxe   Maxe resources:"] 
   
# Build the vaxe device name and get its resource into l1 
  set vaxe [db_io_get s e $maxe axe_ident$axe]
  set pipe [open "|dbm_devres $vaxe"]
  while { [gets $pipe line] >= 0} {lappend l1 $line}
    catch {close $pipe} ; unset pipe
  if { [info exist l1] != 1 } {
    tk_messageBox -message "Resources not accessibles from your host"\
                            -type ok -icon error
    return
  }
  set l1 [lsort $l1]
  set l1 [linsert $l1 0 "+++++++++++ $vaxe   Axe resources:"]
  
# Build the motor device name and get its resource into l2
  set mot [db_io_get s e $vaxe vpap_ident]
  set pipe [open "|dbm_devres $mot"]
  while { [gets $pipe line] >= 0} {lappend l2 $line}
  catch {close $pipe} ; unset pipe
  set l2 [lsort $l2]
  set l2 [linsert $l2 0 "+++++++++++ $mot   Motor resources:" ] 
  
# Build the encoder device name and get its resource into l3
  set enc [db_io_get s e $vaxe enc_ident]
  set pipe [open "|dbm_devres $enc"]
  while { [gets $pipe line] >= 0} {lappend l3 $line}
  catch {close $pipe} ; unset pipe
  set l3 [lsort $l3]
  set l3 [linsert $l3 0 "+++++++++++ $enc   Encoder resources:"]  
  
# Build a top level scrolled window and fill it with l0 l1 l2 l3
  set wtext [TopScrolledTextWindow .wres$axename]
  foreach line $l0 { $wtext insert end $line\n}
  $wtext insert end \n
  foreach line $l1 { $wtext insert end $line\n}
  $wtext insert end \n
  foreach line $l2 { $wtext insert end $line\n}
  $wtext insert end \n
  foreach line $l3 { $wtext insert end $line\n}
  $wtext configure -state disabled
} ;# ends axeGetRes


proc axeUpdateCtrl {axename weabs wedelta br bs bl bunit bcmd} {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
global AXE_CTRL
DisplayDebug "axeUpdateCtrl: $weabs $wedelta"
switch $AXE_CTRL($axename) {
   0 { DisplayDebug "axeUpdateCtrl: disabled"
       $weabs configure -state disabled
       $wedelta configure -state disabled
       $br configure -state disabled
       $bl configure -state disabled
       $bs configure -state disabled
       $bunit configure -state disabled
       $bcmd configure -state disabled
     }
   1 { DisplayDebug "axeUpdateCtrl: enabled"
       $weabs configure -state normal
       $wedelta configure -state normal
       $br configure -state normal
       $bl configure -state normal
       $bs configure -state normal
       $bunit configure -state normal
       $bcmd configure -state normal
     }
   default {DisplayDebug "axeUpdateCtrl: default"}
}
} ;# ends proc axeUpdateCtrl

proc axeMoveReference {maxeid axeid axename } {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
  set argin [list $axeid 0]
  dev_io s e $maxeid DevMoveReference argin
  if {$s != $DS_OK } {
    AddNewError "[GetDate] DevMoveReference $axename failed"
    AddNewError $e
  }
}

proc axeLoadPosition {maxeid axeid axename } {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
  set value "cancelled"
  TopEntryWindow  . "New motor value" value 0 float args 
  if {$value == "cancelled" } return
  set argin [list $axeid $value]
  dev_io s e $maxeid DevLoadPosition argin
  if {$s != $DS_OK } {
    AddNewError "[GetDate] DevLoadPosition $axename failed"
    AddNewError $e
  }
}

proc axeSetVelocity {maxeid axeid axename } {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
global UNITN2S
  set argin $axeid
  set unitn [dev_io s e $maxeid DevReadUnits argin]
  if {$s != $DS_OK} {
    AddNewError "updateAxe: DevReadUnits on device $maxeid $axeid $axename failed"
    AddNewError $e
    return
  }
  set unit $UNITN2S($unitn) 
  set vel [dev_io s e $maxeid DevReadVelocity argin]
  if {$s != $DS_OK} {
    AddNewError "updateAxe: DevReadVelocity on device $maxeid $axeid $axename failed"
    AddNewError $e
    return
  }
  set value "cancelled"
  TopEntryWindow  . " Current velocity: $vel $unit/s \nNew velocity in $unit/s" value $vel float 1 *
  if {$value == "cancelled" } return
  set argin [list $axeid $value]
  dev_io s e $maxeid DevSetVelocity argin
  if {$s != $DS_OK } {
    AddNewError "[GetDate] DevLoadPosition $axename failed"
    AddNewError $e
  }
} ;# ends axeSetVelocity

proc axeLoadEncPos {maxeid axeid axename wenc} {
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
  if { [$wenc cget -text] != "None" } {
    set value "cancelled"
    TopEntryWindow  . "New motor value" value 0 float args 
    if {$value == "cancelled" } return
    set argin [list $axeid $value]
    dev_io s e $maxeid DevLoadEncPos argin
    if {$s != $DS_OK } {
      AddNewError "[GetDate] DevLoadEncPos $axename failed"
      AddNewError $e
    }
  } else {
     tk_messageBox -message "This axe (maxe: $maxeid axe#: $axeid axe name: $axename) has no encoder"\
                            -type ok -icon error
  }
} ;# ends proc axeLoadEncPos


proc AxeStartMonitoring {axename} {
#+############################################################################
#NAME:     AxeStartMonitoring  s
#SYNOPSIS: AxeStartMonitoring axe_mnemonic
#DESCRIPTION: AxeStartMonitoring must only be used to deal with an axis object
#             created with AxeCreate. When the axis is created the monitoring
#             procedure to update the axis panel is started as an event timer
#             but does nothing. The caller has to enable the panel updating
#             at the end of its main body.
#ARGIN:    axename: the axis mnemonic used when calling AxeCreate
#ARGOUT:   NONE
#RETURN:   NONE
#ERROR :   NONE
#GLOBAL_CREATED: NONE
#GLOBAL_USED: AXE_MON_ST
#GLOBAL_MODIFIED: AXE_MON_ST
#PROCEDURE_CALLED: NONE
#SEE ALSO:
#SOURCE FILE: AxeCreate.tcl
#$Author: pepellin $ 
#$Date: 98/08/03 11:53:49 $
#$Revision: 2.3 $
#-############################################################################
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
global AXE_MON_ST 
 set AXE_MON_ST($axename) $TRUE
} ;# ends proc AxeStartMonitoring

proc AxeStopMonitoring {axename} {
#+############################################################################
#NAME:     AxeStopMonitoring
#SYNOPSIS: AxeStopMonitoring axe_mnemonic
#DESCRIPTION: AxeStopMonitoring must only be used to deal with an axis object
#             created with AxeCreate. It has to be used to disable the axis
#             panel update.
#ARGIN:    axename: the axis mnemonic used when calling AxeCreate
#ARGOUT:   NONE
#RETURN:   NONE
#ERROR :   NONE
#GLOBAL_CREATED: NONE
#GLOBAL_USED: AXE_MON_ST
#GLOBAL_MODIFIED: AXE_MON_ST
#PROCEDURE_CALLED: NONE
#SEE ALSO:
#SOURCE FILE: AxeCreate.tcl
#$Author: pepellin $ 
#$Date: 98/08/03 11:53:49 $
#$Revision: 2.3 $
#-############################################################################
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }
global AXE_MON_ST 
 set AXE_MON_ST($axename) $FALSE
} ;# ends proc AxeStopMonitoring

proc AxeSetUpdatePeriod {axename period} {
#+############################################################################
#NAME:     AxeSetUpdatePeriod
#SYNOPSIS: AxeSetUpdatePeriod axe_mnemonic new_update_period
#DESCRIPTION: AxeStartMonitoring must only be used to deal with an axis object
#             created with AxeCreate. When the axis is created the monitoring
#             period to update the axis panel is set to 2000ms. 
#ARGIN:    axename: the axis mnemonic used when calling AxeCreate
#          period: a  positive integer in milliseconds
#ARGOUT:   NONE
#RETURN:   OK or NOTOK if the new period is not a positive integer
#            or a positive integer less than 100 milliseconds
#ERROR :   NONE
#GLOBAL_CREATED: NONE
#GLOBAL_USED: AXE_MON_PE
#GLOBAL_MODIFIED: AXE_MON_PE
#PROCEDURE_CALLED: NONE
#SEE ALSO:
#SOURCE FILE: AxeCreate.tcl
#$Author: pepellin $ 
#$Date: 98/08/03 11:53:49 $
#$Revision: 2.3 $
#-############################################################################
global ARR_GLOB1; foreach elt [array names ARR_GLOB1] { global $elt }

global AXE_MON_PE
if { [CheckEntry $period integer 100 *] != 1 } {return $NOTOK }
set AXE_MON_PE($axename) $period
return $OK
} ;# ends proc AxeSetUpdatePeriod

} ;# ends namespaxe Axe
### Create the package rrap
package provide rrap 2.3
