//=============================================================================
//
// file :               Tango_utils.h
//
// description :        Include for utility functions or classes  
//
// project :            TANGO
//
// author(s) :          A.Gotz + E.Taurel
//
// $Revision: 1.3 $
//
// $Log: tango_utils.h,v $
// Revision 1.3  2000/04/13 10:40:43  taurel
// Added attribute support
//
// Revision 1.2  2000/02/04 11:00:17  taurel
// Just update revision number
//
// Revision 1.1.1.1  2000/02/04 10:58:28  taurel
// Imported sources
//
//
// copyleft :           European Synchrotron Radiation Facility
//                      BP 220, Grenoble 38043
//                      FRANCE
//
//=============================================================================

#ifndef _TANGO_UTILS_H
#define _TANGO_UTILS_H

#include <tango_ds.h>
#include <new>

#ifndef WIN32
#include <unistd.h>
#endif /* WIN32 */

//
// For debug purpose
//

#define	cout1	if (TangoUtil::_tracelevel >= 1) cout 
#define cout2  	if (TangoUtil::_tracelevel >= 2) cout
#define cout3	if (TangoUtil::_tracelevel >= 3) cout
#define cout4	if (TangoUtil::_tracelevel >= 4) cout


//=============================================================================
//
//			The TangoUtil class
//
// description :	This class contains all properties and methods
//			which a device server process requires only once e.g. 
//			the orb and boa pointers....
//			This class is a singleton ( The constructor is 
//			protected and the _instance data member is static)
//
//			This class must be created at the beginning of each
//			device server process
//
//=============================================================================

class DeviceImpl;
class DServer;

/**
 * This class is a used to store TANGO device server process data and to provide
 * the user with a set of utilities method. This class is implemented using
 * the singleton design pattern. Therefore a device server process can have only
 * one instance of this class and its constructor is not public.
 *
 * @author	$Author: taurel $
 * @version	$Revision: 1.3 $
 */
 
class TangoUtil
{
public:

/**@name Singleton related methods
 * These methods follow the singleton designa pattern (only one instance
 * of a class) */
//@{

/**
 * Create and get the singleton object reference.
 *
 * This method returns a reference to the object of the TangoUtil class.
 * If the class singleton object has not been created, it will be 
 * instanciated
 *
 * @param argc The process command line argument number
 * @param argv The process commandline arguments
 * @return The TangoUtil object reference
 */
	static TangoUtil *init(int argc,char *argv[]);
	
/**
 * Get the singleton object reference.
 *
 * This method returns a reference to the object of the TangoUtil class.
 * If the class has not been initialised with it's init method, this method
 * print a message and abort the device server process
 *
 * @return The TangoUtil object reference
 */
	static TangoUtil *instance();
//@}

/**@name Destructor
 * Only one desctructor is defined for this class */
//@{
/**
 * The class desctructor.
 */
	~TangoUtil() {};
//@}

/**@name Get/Set instance data */
//@{
/**
 * Get a reference to the CORBA ORB
 *
 * @return The CORBA ORB
 */		
	CORBA_ORB_ptr get_orb() {return CORBA_ORB::_duplicate(orb);}
	
/**
 * Get a reference to the CORBA BOA
 *
 * @return The CORBA BOA
 */
	CORBA_BOA_var get_boa() {return boa;}

//
// process output control methods
//

/**
 * Set the process trace level.
 *
 * @param level The new process level
 */	
	void set_trace_level(long level) {_tracelevel = level;}
	
/**
 * Get the process trace level.
 *
 * @return The process trace level
 */
	long get_trace_level() {return _tracelevel;}

/**
 * Set the process trace output .
 *
 * @param level The new process output
 */	
	void set_trace_output(string &trace) {trace_output = trace;}
	
/**
 * Get the process trace output.
 *
 * @return The process trace output
 */	
	string &get_trace_output() {return trace_output;}	

/**
 * Get the temporary process output print stream
 *
 * @return The process output print stream
 */	
	TangoSys_Cout &get_out() {return cout_tmp;}
	
/**
 * Set the process trace output stream.
 *
 * @param in The new process output stream
 */	
	void set_trace_output_stream(ofstream *in) {file_stream = in;}
	
/**
 * Get the process output print stream
 *
 * @return The process output print stream
 */
	ofstream *get_trace_output_stream() {return file_stream;}

/**
 * Get the device server instance name.
 *
 * @return The device server instance name
 */	
	string &get_ds_inst_name() {return ds_instance_name;}
	
/**
 * Get the device server executable name.
 *
 * @return The device server executable name
 */
	string &get_ds_exec_name() {return ds_exec_name;}
	
/**
 * Get the device server name.
 *
 * The device server name is the device server executable name/the device
 * server instance name
 * @return The device server name
 */
	string &get_ds_name() {return ds_name;}

/**
 * Get the host name where the device server process is running.
 *
 * @return The host name
 */	
	string &get_host_name() {return hostname;}
	
/**
 * Get the device server process identifier as a String
 *
 * @return The device server process identifier as a string
 */
	string &get_pid_str() {return pid_str;}
	
/**
 * Get the device server process identifier
 *
 * @return The device server process identifier
 */
	TangoSys_Pid get_pid() {return pid;}

/**
 * Get the device server TANGO version.
 *
 * @return The device server version
 */	
	string &get_version_str() {return version_str;}	
//@}

/**@name Exception related method */
//@{	
/**
 * Print a TANGO exception.
 *
 * Print all the details of a TANGO exception.
 *
 * @param ex The exception object reference
 */	
	static void print_exception(CORBA_Exception &ex);

//@}

/**@name Database related methods */
//@{
/**
 * Connect the process to the TANGO database.
 *
 * If the connection to the database failed, a message is displayed on the
 * screen and the process is aborted
 */
	void connect_db();
		
/**
 * Get the TANGO database device reference
 *
 * @return The database device reference
 */
	Tango_Device_var &get_db_device() {return db_device;}
	
/**
 * Unregister a device server process from the TANGO database.
 *
 * If the database call fails, a message is displayed on the screen and the
 * process is aborted
 */	
	void unregister_server();
//@}

/** @name Device reference related methods */
//@{
/**
 * Get the list of device references for a given TANGO class.
 *
 * Return the list of references for all devices served by one implementation
 * of the TANGO device pattern implemented in the  process
 *
 * @param class_name The TANGO device class name
 * @return The device reference list
 * @exception DevFailed If in the device server process there is no TANGO 
 * device pattern implemented the TANGO device class given as parameter
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
	vector<DeviceImpl *> &get_device_list_by_class(const string &class_name);
	
/**
 * Get the list of device references for a given TANGO class.
 *
 * Return the list of references for all devices served by one implementation
 * of the TANGO device pattern implemented in the  process
 *
 * @param class_name The TANGO device class name
 * @return The device reference list
 * @exception DevFailed If in the device server process there is no TANGO 
 * device pattern implemented the TANGO device class given as parameter
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
	vector<DeviceImpl *> &get_device_list_by_class(const char *class_name);
	
/**
 * Get a device reference from its name
 *
 * @param dev_name The TANGO device name
 * @return The device reference
 * @exception DevFailed If in the device is not served by one device pattern
 * implemented in this process.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
	DeviceImpl *get_device_by_name(const string &dev_name);
	
/**
 * Get a device reference from its name
 *
 * @param dev_name The TANGO device name
 * @return The device reference
 * @exception DevFailed If in the device is not served by one device pattern
 * implemented in this process.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
	DeviceImpl *get_device_by_name(const char *dev_name);
	
/**
 * Get a reference to the dserver device attached to the device server process
 *
 * @return A reference to the dserver device
 */
	DServer *get_dserver_device();
//@}

/** @name Device pattern related methods */
//@{
/**
 * Initialise all the device server pattern(s) embedded in a device server
 * process.
 *
 * @exception DevFailed If the device pattern initialistaion failed
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */	
	void server_init();
//@}

/**@name Class data members */
//@{
/**
 * The process trace level
 */	
	static long		_tracelevel;
/**
 * The database use flag (Use with extreme care. Implemented mainly for
 * the database device server)
 */
	static bool		_UseDb;
//@}	
	
protected:
/**
 * Constructs a newly allocated TangoUtil object.
 *
 * This constructor is protected following the singleton pattern
 *
 * @param argc The process command line argument number
 * @param argv The process commandline arguments
 *
 */
	TangoUtil(int argc,char *argv[]);
		
private:
	static TangoUtil	*_instance;

	CORBA_ORB_var 		orb;
	CORBA_BOA_var		boa;
	
	string			ds_instance_name;	// The instance name
	string			ds_exec_name;		// The server exec. name
	string			ds_name;		// The server name
	
	string			hostname;		// The host name
	string			pid_str;		// The process PID (as string)
	TangoSys_Pid		pid;			// The process PID

	string			version_str;		// Tango version
	
	string			trace_output;
	TangoSys_Cout		cout_tmp;
	ofstream		*file_stream;

	string			db_host;		// The db server host
	unsigned short		db_port;		// The db server port
	Tango_Device_var	db_device;		// The db proxy

	void read_env();
	void misc_init();
	void server_already_running();	
	void print_usage(char *);
	void check_args(int, char *[]);
};


// Declaration and definition of inline functions

//+-------------------------------------------------------------------------
//
// function : 		throw_exception 
// 
// description : 	throw a Tango_DevFailed exception
//
// arguments : 		in : - reason : the exception reason
//			     - origin : the function/method where the exception
//				        is generated
//			     - code : The error code (default = -1)
//			     - fac : The facility (default = CONTROL)
//			     - sever : The severity (default = ERROR)
//
//--------------------------------------------------------------------------

/**@name Throw exception inline functions */
//@{
/**
 * Generate and throw a TANGO DevFailed exception.
 *
 * The exception is created with a single DevError
 * object. A default value of -1 is defined for the DevError code
 * field. A default value "Tango_ERROR" is defined for the DevError
 * severity field. A default value "Tango_CONTROL" is also defined
 * for the DevError facility field.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 * @param reason The exception DevError object reason field
 * @param origin The exception DevError object origin field
 * @param code The exception DevError object code field
 * @param fac The exception DevError object facility field
 * @param sever The exception DevError object severity field
 * @exception DevFailed The thrown exception.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline void throw_exception(const char *reason,const char *origin,long code = -1,
		            Tango_ErrFacility fac = Tango_CONTROL,
		            Tango_ErrSeverity sever = Tango_ERROR)
{
	Tango_DevErrorList errors(1);
	
	errors.length(1);
	errors[0].code = code;
	errors[0].severity = sever;
	errors[0].facility = fac;
	errors[0].reason = CORBA_string_dup(reason);
	errors[0].origin = CORBA_string_dup(origin);
	
	throw Tango_DevFailed(errors);
}

/**
 * Generate and throw a TANGO DevFailed exception.
 *
 * The exception is created with a single DevError
 * object. A default value of -1 is defined for the DevError code
 * field. A default value "Tango_ERROR" is defined for the DevError
 * severity field. A default value "Tango_CONTROL" is also defined
 * for the DevError facility field.
 * <p>
 * This method de-allocate the origin parameter.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 * @param reason The exception DevError object reason field
 * @param origin The exception DevError object origin field
 * @param code The exception DevError object code field
 * @param fac The exception DevError object facility field
 * @param sever The exception DevError object severity field
 * @exception DevFailed The thrown exception.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline void throw_exception(const char *reason,char *origin,long code = -1,
		            Tango_ErrFacility fac = Tango_CONTROL,
		            Tango_ErrSeverity sever = Tango_ERROR)
{
	Tango_DevErrorList errors(1);
	
	errors.length(1);
	errors[0].code = code;
	errors[0].severity = sever;
	errors[0].facility = fac;
	errors[0].reason = CORBA_string_dup(reason);
	errors[0].origin = CORBA_string_dup(origin);
	delete[] origin;
	
	throw Tango_DevFailed(errors);
}

/**
 * Generate and throw a TANGO DevFailed exception.
 *
 * The exception is created with a single DevError
 * object. A default value of -1 is defined for the DevError code
 * field. A default value "Tango_ERROR" is defined for the DevError
 * severity field. A default value "Tango_CONTROL" is also defined
 * for the DevError facility field.
 * <p>
 * This method de-allocate the reason parameter.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 * @param reason The exception DevError object reason field
 * @param origin The exception DevError object origin field
 * @param code The exception DevError object code field
 * @param fac The exception DevError object facility field
 * @param sever The exception DevError object severity field
 * @exception DevFailed The thrown exception.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline void throw_exception(char *reason,const char *origin,long code = -1,
		            Tango_ErrFacility fac = Tango_CONTROL,
		            Tango_ErrSeverity sever = Tango_ERROR)
{
	Tango_DevErrorList errors(1);
	
	errors.length(1);
	errors[0].code = code;
	errors[0].severity = sever;
	errors[0].facility = fac;
	errors[0].reason = CORBA_string_dup(reason);
	delete[] reason;
	errors[0].origin = CORBA_string_dup(origin);
	
	throw Tango_DevFailed(errors);
}

/**
 * Generate and throw a TANGO DevFailed exception.
 *
 * The exception is created with a single DevError
 * object. A default value of -1 is defined for the DevError code
 * field. A default value "Tango_ERROR" is defined for the DevError
 * severity field. A default value "Tango_CONTROL" is also defined
 * for the DevError facility field.
 * <p>
 * This method de-allocate the reason and origin parameters.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 * @param reason The exception DevError object reason field
 * @param origin The exception DevError object origin field
 * @param code The exception DevError object code field
 * @param fac The exception DevError object facility field
 * @param sever The exception DevError object severity field
 * @exception DevFailed The thrown exception.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline void throw_exception(char *reason,char *origin,long code = -1,
		            Tango_ErrFacility fac = Tango_CONTROL,
		            Tango_ErrSeverity sever = Tango_ERROR)
{
	Tango_DevErrorList errors(1);
	
	errors.length(1);
	errors[0].code = code;
	errors[0].severity = sever;
	errors[0].facility = fac;
	errors[0].reason = CORBA_string_dup(reason);
	delete[] reason;
	errors[0].origin = CORBA_string_dup(origin);
	delete[] origin;
	
	throw Tango_DevFailed(errors);
}

/**
 * Generate and throw a TANGO DevFailed exception.
 *
 * The exception is created with a single DevError
 * object. A default value of -1 is defined for the DevError code
 * field. A default value "Tango_ERROR" is defined for the DevError
 * severity field. A default value "Tango_CONTROL" is also defined
 * for the DevError facility field.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 * @param reason The exception DevError object reason field
 * @param origin The exception DevError object origin field
 * @param code The exception DevError object code field
 * @param fac The exception DevError object facility field
 * @param sever The exception DevError object severity field
 * @exception DevFailed The thrown exception.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline void throw_exception(const char *reason,string &origin,long code = -1,
		            Tango_ErrFacility fac = Tango_CONTROL,
		            Tango_ErrSeverity sever = Tango_ERROR)
{
	Tango_DevErrorList errors(1);
	
	errors.length(1);
	errors[0].code = code;
	errors[0].severity = sever;
	errors[0].facility = fac;
	errors[0].reason = CORBA_string_dup(reason);
	errors[0].origin = CORBA_string_dup(origin.c_str());
	
	throw Tango_DevFailed(errors);
}

/**
 * Generate and throw a TANGO DevFailed exception.
 *
 * The exception is created with a single DevError
 * object. A default value of -1 is defined for the DevError code
 * field. A default value "Tango_ERROR" is defined for the DevError
 * severity field. A default value "Tango_CONTROL" is also defined
 * for the DevError facility field.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 * @param reason The exception DevError object reason field
 * @param origin The exception DevError object origin field
 * @param code The exception DevError object code field
 * @param fac The exception DevError object facility field
 * @param sever The exception DevError object severity field
 * @exception DevFailed The thrown exception.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline void throw_exception(string &reason,const char *origin,long code = -1,
		            Tango_ErrFacility fac = Tango_CONTROL,
		            Tango_ErrSeverity sever = Tango_ERROR)
{
	Tango_DevErrorList errors(1);
	
	errors.length(1);
	errors[0].code = code;
	errors[0].severity = sever;
	errors[0].facility = fac;
	errors[0].reason = CORBA_string_dup(reason.c_str());
	errors[0].origin = CORBA_string_dup(origin);
	
	throw Tango_DevFailed(errors);
}
//@}

//+-------------------------------------------------------------------------
//
// function : 		return_empty_any
// 
// description : 	Return from a command when the command does not have
//			any output argument
//
// arguments : 		in : - cmd : The command name
//
//--------------------------------------------------------------------------

/**
 * Create and return an empty CORBA Any object.
 *
 * Create an empty CORBA Any object. Could be used by command which does
 * not return anything to the client. This method also prints a message on
 * screen (level 4) before it returns
 *
 * @param cmd The cmd name which use this empty Any. Only used to create the
 * thrown exception (in case of) and in the displayed message
 * @return The empty CORBA Any
 * @exception DevFailed If the Any object creation failed.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
inline CORBA_Any *return_empty_any(const char *cmd)
{

	CORBA_Any *out_any;
	try
	{
		out_any = new CORBA_Any();
	}
	catch (bad_alloc)
	{
		TangoSys_MemStream o;
		
		o << cmd << "::execute";
		cout3 << "Bad allocation while in " << cmd << "::execute()" << endl;
		throw_exception("Can't allocate memory in server",o.str(),
				Tango_DevErr_MemoryAllocation,Tango_APILIB);
	}
	
	cout4 << "Leaving " << cmd << "::execute()" << endl;	
	return(out_any);
	
}

		     
#endif /* TANGO_UTILS */
