//+======================================================================
// $Source: /segfs/tango/tools/TangoApi/RCS/Connection.java,v $
//
// Project:   Tango
//
// Description:  java source code for the TANGO clent/server API.
//
// $Author: verdier $
//
// $Revision: 1.1 $
//
// $Log: Connection.java,v $
// Revision 1.1  2001/02/02 13:03:46  verdier
// Initial revision
//
//
// Copyright 1995 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package fr.esrf.TangoApi;

import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import TangoDs.*;
import java.io.*;
import java.util.*;


/**
 *	Class Description:
 *	This class manage device connection for Tango objects.
 *	It is an api between user and IDL Device object.
 *
 * @author  verdier
 * @Revision 
 */

public class Connection
{
	/**
	 *	Device IDL object used for TANGO device access
	 */
	protected Device	device = null;
	/**
	 *	Device IDL object used for efective connection.
	 */
	private org.omg.CORBA.Object obj = null;
	/**
	 *	Device timeout value in ms.
	 */
	private int			dev_timeout = 0;
	/**
	 *	The tango host and port from environment variable.
	 */
	protected String	tango_host;
	/**
	 *	The tango host from environment variable.
	 */
	private String		dbase_host;
	/**
	 *	The tango port from environment variable.
	 */
	private String		dbase_port;
	/**
	 *	Device name;
	 */
	private String		devname = null;
	/**
	 *	Used to know if it is a connection or a Re-connection.
	 */
	private boolean		already_connected = false;


	//===================================================================
	/**
	 *	Connection constructor. It makes a connection on database server.
	 *
	 */
	//===================================================================
	public Connection()	throws DevFailed
	{
		//	Check Tango Host Properties (host name, port number)
		//------------------------------------------------------------
		if ((tango_host=System.getProperty("TANGO_HOST"))==null)
			TangoUtil.throw_exception(
				"Property \"TANGO_HOST\" not exported", "DbConnection()");
		int	idx;
		if ((idx=tango_host.indexOf(":"))<0)
			TangoUtil.throw_exception(
				"Unknown \"TANGO_HOST\" property " + tango_host, "DbConnection()");

		dbase_host = tango_host.substring(0,idx);
		dbase_port = tango_host.substring(idx+1);
		ApiUtil.get_orb();
		connect_to_dbase();
		devname = device.name();
		already_connected = true;
System.out.println("Database Connetion done " + device.name());
	}

	//===================================================================
	/**
	 *	Connection constructor. It imports the device.
	 *
	 *	@param	devname	name of the device to be imported.
	 */
	//===================================================================
	public Connection(String devname)
	{
		this.devname = devname;
	}
	//===================================================================
	/**
	 *	Really build the device connection.
	 */
	//===================================================================
	protected void dev_import()	throws DevFailed
	{
		Database	db = ApiUtil.get_db_obj();
		DbDevImportInfo	info = db.import_device(devname);

		if (info.exported==false)
			TangoUtil.throw_exception(devname + " Not Exported !",
									"Connection("+ devname +")");
		try
		{
			// Import the TANGO device
			//--------------------------------
			obj =
			  ((com.ooc.CORBA.ORB)ApiUtil.get_orb()).string_to_object(info.ior);

			//	Get the default value for timeout
			//	and change Timeout policy for object.
			//---------------------------------------------
			if (dev_timeout==0)
			{
				try
				{
					DbClass	tgDev = new DbClass("TangoDevice");
					DbDatum datum = tgDev.get_property("DefaultTimeout");
					dev_timeout = 2000;
					if (!datum.is_empty())
						dev_timeout = (int)datum.extractLong();
				}
				catch(DevFailed e)
				{
					dev_timeout = 2000;
				}
			}
			org.omg.CORBA.Object newObj = set_timeout(dev_timeout);

			device = Tango.DeviceHelper.narrow(newObj);
			device.ping();
			already_connected = true;
		}
		catch(Exception e)
		{
			device = null;
			String	s = (already_connected)? "Re-" : "";
			String	reason = new String("Cannot " + s + "import " + devname);
			String	origin = new String("Connection.dev_import(" + devname + ")");
			//e.printStackTrace();
			TangoUtil.throw_exception(reason, origin);
		}
	}
	//===================================================================
	/**
	 *	Import the tango database.
	 *
	 */
	//===================================================================
	private void connect_to_dbase() throws DevFailed
	{
		if (device == null)
 		{
			try
			{
				//	Prepeare the connection string
				//----------------------------------
				String	db_corbaloc =  new String("corbaloc:iiop:" +
												dbase_host + ":" +
												dbase_port + "/database" );
				//	And connect to database.
				//----------------------------
				obj = ((com.ooc.CORBA.ORB)ApiUtil.get_orb()).string_to_object(db_corbaloc);
				device = Tango.DeviceHelper.narrow(obj);
			}
			catch(SystemException ex)
			{
				device = null;
				TangoUtil.throw_exception("Connection to database failed  !",
						"get_dbase(" + dbase_host + "," + dbase_port + ")");
			}
		}
	}
	
	//==========================================================================
	/**
	 *	return the CORBA connection reference object.
	 */
	//==========================================================================
	//public org.omg.CORBA.Object get_corba_obj() throws DevFailed
	//{
	//	return obj;
	//}

	//===================================================================
	/**
	 *	Change the timeout value for a device call.
	 *
	 *	@param	millis		New value of the timeout in milliseconds.
	 *	@throws	DevFaild	if orb.create_policy throws an
	 *						org.omg.CORBA.PolicyError exception.
	 */
	//===================================================================
	public Device set_timeout(int millis) throws DevFailed
	{

		try
		{
			//	Change policy for new value of timeout
			//--------------------------------------------------
			org.omg.CORBA.Any	longAny	= ApiUtil.get_orb().create_any();
			longAny.insert_ulong(millis);
			org.omg.CORBA.Policy[]	policies = new org.omg.CORBA.Policy[1];
			policies[0] = ApiUtil.get_orb().create_policy(
						com.ooc.OB.TIMEOUT_POLICY_ID.value, longAny);

			//	Create a new object with the new policy
			//---------------------------------------------
			org.omg.CORBA.Object	newObj = 
				obj._set_policy_override(policies,
							org.omg.CORBA.SetOverrideType.ADD_OVERRIDE);
			//	Save new timeout value
			//---------------------------------
			dev_timeout = millis;
			
			return Tango.DeviceHelper.narrow(newObj);
		}
		catch(org.omg.CORBA.PolicyError e)
		{
			TangoUtil.throw_exception(
				"org.omg.CORBA.PolicyError" ,"TangoDevice.set_time_out()");
			return Tango.DeviceHelper.narrow(obj);
		}
	}
	//===================================================================
	/**
	 *	return the timeout value for a device call.
	 *
	 *	@return the value of the timeout in milliseconds.
	 */
	//===================================================================
	public int get_timeout()
	{
		return dev_timeout;
	}



	//===========================================================
	/**
	 *	Build reason and origin of the exception
	 *	And throw it into a DevFailed exception
	 */
	//===========================================================
	protected void throw_dev_failed(Exception e, String command) throws DevFailed
	{
		String	reason;
		String	origin =
					new String(devname + ".command_inout(" + command + ")");
		if (e.toString().indexOf("org.omg.CORBA.NO_RESPONSE")>=0)
		{
			reason = new String("Device (" +  devname+
						") timed out (>" + dev_timeout +" ms)!");
		}
		else
		if (e.toString().indexOf("org.omg.CORBA.TRANSIENT")>=0)
		{
			reason = new String("org.omg.CORBA.TRANSIENT:\n" +
								"Lost Connection during command : " + command);
			device = null;
		}
		else
		if (e.toString().indexOf("org.omg.CORBA.COMM_FAILURE")>=0)
		{
			reason = new String("org.omg.CORBA.COMM_FAILURE:\n" +
								"Lost Connection during command : " + command);
			device = null;
		}
		else
		{
			System.out.println(e);
			reason = new String(e.getMessage());
		}
		TangoUtil.throw_exception(reason, origin);
	}
	//===========================================================
	/**
	 *	Send a command to a device server.
	 *
	 *	@param command	Command name to send to the device.
	 *	@param	data	argin management object.
	 *	@throws DevFailed	
	 */
	//===========================================================
	public DeviceData command_inout(String command, DeviceData argin)
			throws DevFailed
	{
		if (device==null && devname!=null)
			dev_import();
		try{
			Any received = device.command_inout(command, argin.extractAny());
			return new DeviceData(received);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, command);
			return null;
		}
	}
	//===========================================================
	/**
	 *	Send a command to a device server.
	 *
	 *	@param device 	Device instance to send the command.
	 *	@throws DevFailed	
	 */
	//===========================================================
	public DeviceData command_inout(String command)
			throws DevFailed
	{
		if (device==null && devname!=null)
			dev_import();
		try{
			Any received = device.command_inout(command, ApiUtil.get_orb().create_any());
			return new DeviceData(received);
		}
		catch(Exception e)
		{
			throw_dev_failed(e, command);
			return null;
		}
	}

	//===========================================================
	/**
	 *	Execute a ping command to a device server.
	 */
	//===========================================================
	public void ping() throws DevFailed
	{
		if (device==null && devname!=null)
			dev_import();
		device.ping();
	}
	//===========================================================
	/**
	 *	Execute a info command to a device server.
	 */
	//===========================================================
	public DevInfo info() throws DevFailed
	{
		if (device==null && devname!=null)
			dev_import();
		return device.info();
	}
	//===========================================================
	/**
	 *	Execute a command_list_query command to a device server.
	 */
	//===========================================================
	public DevCmdInfo[] command_list_query() throws DevFailed
	{
		if (device==null && devname!=null)
			dev_import();
		return device.command_list_query();
	}

	//==========================================================================
	/**
	 *	return the device connected name.
	 */
	//==========================================================================
	public String get_name()
	{
		return device.name();
	}
	//==========================================================================
	/**
	 *	return the name of connection (host:port)
	 */
	//==========================================================================
	public String get_tango_host()
	{
		return tango_host;
	}
}
