//+======================================================================
// $Source: /segfs/tango/tools/api/fr/esrf/TangoApi/RCS/Database.java,v $
//
// Project:   Tango
//
// Description:  java source code for the TANGO clent/server API.
//
// $Author: verdier $
//
// $Revision: 1.3 $
//
// $Log: Database.java,v $
// Revision 1.3  2001/04/02 08:32:05  verdier
// TangoApi package has users...
//
// Revision 1.1  2001/02/02 13:03:46  verdier
// Initial revision
//
//
// Copyright 1995 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-======================================================================

package fr.esrf.TangoApi;
 

import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import java.io.*;
import java.util.*;

/** 
 *	Class Description:
 *	This class is the main class for TANGO database API.
 *	The TANGO database is implemented as a TANGO device server.
 *	To access it, the user has the CORBA interface command_inout().
 *	This expects and returns all parameters as ascii strings thereby making
 *	the database laborious to use for retreing device properties and information.
 *	In order to simplify this access, a high-level API has been implemented
 *	which hides the low-level formatting necessary to convert the
 *	command_inout() return values into binary values and all CORBA aspects
 *	of the TANGO.
 *	All data types are native java types e.g. simple types an arrays.
 *
 * @author  verdier
 * @Revision 
 */


public class Database extends Connection
{
	private	static DeviceData	argin, argout;
	
	//===================================================================
	/**
	 *	Database access constructor.
	 *
	 *	@throws DevFailed in case of environment not corectly set.
	 */
	//===================================================================
	public Database() throws DevFailed
	{
		super();
		argin = new DeviceData(ApiUtil.get_orb());
		argout= new DeviceData(ApiUtil.get_orb());
	}

	
	//===================================================================
	/**
	 *	Database access constructor.
	 *
	 *	@param	host	host where database is running.
	 *	@param	port	port for database connection.
	 *	@throws DevFailed in case of host or port not available
	 */
	//===================================================================
	public Database(String host, String port) throws DevFailed
	{
		super(host, port);
		argin = new DeviceData(ApiUtil.get_orb());
		argout= new DeviceData(ApiUtil.get_orb());
	}

	//==========================================================================
	/**
	 *	Convert a String array to a sting.
	 */
	//==========================================================================
	private String stringArray2String(String[] array)
	{
		StringBuffer	sb = new StringBuffer("");
		for(int i=0 ; i<array.length ; i++)
		{
			sb.append(array[i]);
			if (i<array.length-1)
				sb.append("\n");
		}
		return sb.toString();
	}







	//**************************************
	//       MISCELANEOUS MANAGEMENT
	//**************************************	


	//==========================================================================
	/**
	 *	Query the database for general info about the table in the database.
	 *	@return	the result of the query as String.
	 */
	//==========================================================================
	public String get_info() throws DevFailed
	{
		//	Query info from database
		argout = command_inout("DbInfo", argin);
		String[] info = argout.extractStringArray();

		//	format result as string
		return stringArray2String(info);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of host registred.
	 *	@return	the list of all hosts registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_host_list() throws DevFailed
	{
		//	Query info from database
		argin.insert("*");
		argout = command_inout("DbGetHostList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of host registred.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return	the list of the hosts registred in TANGO database 
	 *			with the specified wildcard.
	 */
	//==========================================================================
	public String[] get_host_list(String wildcard) throws DevFailed
	{
		//	Query info from database
		argin.insert(wildcard);
		argout = command_inout("DbGetHostList", argin);
		return argout.extractStringArray();
	}






	//**************************************
	//       SERVERS MANAGEMENT
	//**************************************	

	//==========================================================================
	/**
	 *	Query the database for a list of servers registred in the database.
	 *	@return	the list of all servers registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_server_list() throws DevFailed
	{
		//	Query info from database
		argin.insert("*");
		argout = command_inout("DbGetServerList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of servers registred in the database.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return	the list of all servers registred in TANGO database.
	 */
	//==========================================================================
	public String[] get_server_list(String wildcard) throws DevFailed
	{
		//	Query info from database
		argin.insert(wildcard);
		argout = command_inout("DbGetServerList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of servers registred on the specified host.
	 *	@param hostname	the specified host name.
	 *	@return	the list of the servers registred in TANGO database 
	 *			for the specified host.
	 */
	//==========================================================================
	public String[] get_host_server_list(String hostname) throws DevFailed
	{
		//	Query info from database
		argin.insert(hostname);
		argout = command_inout("DbGetHostServerList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for server information.
	 *	@param servname	The specified server name.
	 *	@return	The information found for the specified server
	 *				in a DBServInfo object.
	 */
	//==========================================================================
	public DbServInfo get_server_info(String servname) throws DevFailed
	{
		//	Query info from database
		argin.insert(servname);
		argout = command_inout("DbGetServerInfo", argin);
		String[]	info = argout.extractStringArray();
		return new DbServInfo(info);
	}

	//==========================================================================
	/**
	 *	Add/update server information in databse.
	 *	@param info	Server information for the specified server
	 *					in a DbServinfo object.
	 */
	//==========================================================================
	public void put_server_info(DbServInfo info) throws DevFailed
	{
		String[] array;
		array = new String[4];
		array[0] = info.name;
		array[1] = info.host;
		array[2] = (info.controled) ? "1" : "0";
		array[3] = new String(""+info.startup_level);

		//	Query info from database
		argin.insert(array);
		command_inout("DbPutServerInfo", argin);
	}


	//**************************************
	//       DEVICES MANAGEMENT
	//**************************************	

	//==========================================================================
	/**
	 *	Add/update a device to the database
	 *	@param devinfo The device name, class and server  specified in object.
	 */
	//==========================================================================
	public void add_device(DbDevInfo devinfo) throws DevFailed
	{
		argin.insert(devinfo.toStringArray());
		command_inout("DbAddDevice", argin);
		System.out.println(devinfo.name + " created");
	}

	//==========================================================================
	/**
	 *	Add/update a device to the database
	 *	@param devname		The device name
	 *	@param classnmame	The class.
	 *	@param servname		The server name.
	 */
	//==========================================================================
	public void add_device(String devname, String classname, String servname)
					throws DevFailed
	{
		DbDevInfo devinfo = new DbDevInfo(devname, classname, servname);
		argin.insert(devinfo.toStringArray());
		command_inout("DbAddDevice", argin);
		System.out.println(devinfo.name + " created");
	}

	//==========================================================================
	/**
	 *	Delete the device of the specified name from the database
	 *	@param devname The device name.
	 */
	//==========================================================================
	public void delete_device(String devname) throws DevFailed
	{
		argin.insert(devname);
		command_inout("DbDeleteDevice", argin);
		System.out.println(devname + " deleted");
	}


	//==========================================================================
	/**
	 *	Query the database for the export info of the specified device.
	 *	@param devname The device name.
	 *	@return the information in a DbDevImportInfo.
	 */
	//==========================================================================
	public DbDevImportInfo import_device(String devname)
				throws DevFailed
	{
		argin.insert(devname);
		argout = command_inout("DbImportDevice", argin);
		DevVarLongStringArray info = argout.extractLongStringArray();

		return new DbDevImportInfo(info);
	}
	//==========================================================================
	/**
	 *	Mark the specified server as unexported in the database.
	 *	@param devname The device name.
	 */
	//==========================================================================
	public void unexport_device(String devname)
				throws DevFailed
	{
		argin.insert(devname);
		command_inout("DbUnExportDevice", argin);
	}
	//==========================================================================
	/**
	 *	Update the export info fort this device in the database.
	 *	@param devinfo	Device information to export.
	 */
	//==========================================================================
	public void export_device(DbDevExportInfo devinfo)
				throws DevFailed
	{
		String[]	array = devinfo.toStringArray();
		argin.insert(array);
		command_inout("DbExportDevice", argin);
	}



	//**************************************
	//       Devices list MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for server devices and classes.
	 *	@param servname	The specified server name.
	 *	@return	The devices and classes (e.g. "id11/motor/1", "StepperMotor",
	 *			"id11/motor/2", "StepperMotor",....)
	 */
	//==========================================================================
	public String[] get_device_class_list(String servname) throws DevFailed
	{
		//	Query info from database
		argin.insert(servname);
		argout = command_inout("DbGetDeviceClassList", argin);
		return argout.extractStringArray();
	}


	//==========================================================================
	/**
	 *	Query the database for a list of devices served by the specified server
	 *	and of the specified class.
	 *	@param servname	The server name.
	 *	@param clasname	The class name
	 *	@return the device names are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_name(String servname, String classname)
				throws DevFailed
	{
		String[]	array;
		array = new String[2];
		array[0] = servname;
		array[1] = classname;
		argin.insert(array);
		argout = command_inout("DbGetDeviceList", argin);

		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of aliases for the specified device.
	 *	@param devvname	The server name.
	 *	@return the device aliases are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_alias(String devname)
				throws DevFailed
	{
		argin.insert(devname);
		argout = command_inout("DbGetDeviceAliasList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of device domain names witch match
	 *	the wildcard provided.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device domain are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_domain(String wildcard)
				throws DevFailed
	{
		argin.insert(wildcard);
		argout = command_inout("DbGetDeviceDomainList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of device family names witch match
	 *	the wildcard provided.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device family are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_family(String wildcard)
				throws DevFailed
	{
		argin.insert(wildcard);
		argout = command_inout("DbGetDeviceFamilyList", argin);
		return argout.extractStringArray();
	}

	//==========================================================================
	/**
	 *	Query the database for a list of device member names witch match
	 *	the wildcard provided.
	 *	@param wildcard	Wildcard char is '*' and matches wildvcard characters.
	 *	@return the device member are stored in an array of strings.
	 */
	//==========================================================================
	public String[] get_device_member(String wildcard)
				throws DevFailed
	{
		argin.insert(wildcard);
		argout = command_inout("DbGetDeviceMemberList", argin);
		return argout.extractStringArray();
	}


	//**************************************
	//       SERVERS MANAGEMENT
	//**************************************	

	//==========================================================================
	/**
	 *	Add a group of devices to the database.
	 *	@param servname	Server name for these devices.
	 *	@param devinfo	Devices and server information.
	 */
	//==========================================================================
	public void add_server(String servname, DbDevInfo[] devinfos)
				throws DevFailed
	{
		//	Convert data from DbDevInfos to a string array
		//----------------------------------------------
		String[]	array;
		array = new String[1 + 2*devinfos.length];
		
		array[0]   = servname;
		for (int i=0 ; i<devinfos.length ; i++)
		{
			array[2*i+1] = devinfos[i].name;
			array[2*i+2] = devinfos[i]._class;
		}

		//	Send command
		//-----------------------
		argin.insert(array);
		command_inout("DbAddServer", argin);
	}
	//==========================================================================
	/**
	 *	Delete the device server and its associated devices from the database.
	 *	@param devname the device name.
	 */
	//==========================================================================
	public void delete_server(String devname)
				throws DevFailed
	{
		argin.insert(devname);
		command_inout("DbDeleteServer", argin);
	}
	//==========================================================================
	/**
	 *	Add a group of devices to the database.
	 *	@param servname	server name for these devices.
	 *	@param devinfo	Devices and server information.
	 */
	//==========================================================================
	public void export_server(String devname, DbDevExportInfo[] devinfos)
				throws DevFailed
	{
		//	Convert data from DbDevInfos to a string array
		//----------------------------------------------
		String[]	array;
		array = new String[6*devinfos.length];
		for (int i=0 ; i<devinfos.length ; i++)
		{
			String[]	one = devinfos[i].toStringArray();
			for (int j=0 ; j<6 ; j++)
				array[6*i+j] = new String(one[j]);
		}

		//	Send command
		//-----------------------
		argin.insert(array);
		command_inout("DbExportServer", argin);
	}
	//==========================================================================
	/**
	 *	Mark all devices exported for this device server as unexported.
	 *	@param devname the device name.
	 */
	//==========================================================================
	public void unexport_server(String devname)
				throws DevFailed
	{
		argin.insert(devname);
		command_inout("DbUnExportServer", argin);
	}



	
	//**************************************
	//       PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 * Convert Propeties in DbDatnum array to a String array.
	 *	@param name	Object name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	private String[] dbdatum2StringArray(String name, DbDatum[] properties)
	{
		//	At first, search the array size
		//---------------------------------------
		int	size = 2;	//	Object Name and nb properties
		for (int i=0 ; i<properties.length ; i++)
		{
			size += 2;	//	Property Name and nb properties
			size += properties[i].size();
		}

		//	Format input parameters as strin array
		//--------------------------------------------
		String[] result;
		result = new String[size];
		result[0] = name;
		result[1] = String.valueOf(properties.length);
		for (int i=0, pnum=2 ; i<properties.length ; i++)
		{
			String[] prop = properties[i].toStringArray();
			for (int j=0 ; j<prop.length ; j++)
				result[pnum++] = prop[j];
		}
		return result;
	}
	//==========================================================================
	/**
	 * Convert a String array to a DbDatnum array for properties.
	 *	@param strprop Properties names and values array.
	 *	@return a DbDatum array specifying the properties fonud in string array.
	 */
	//==========================================================================
	private DbDatum[] 	stringArray2DbDatum(String[] strprop)
	{
		//	And format result as DbDatum array
		//---------------------------------------
		DbDatum[]	properties;
		int 		nb_prop = Integer.parseInt(strprop[1]);
		properties = new DbDatum[nb_prop];
		for (int i=2, pnum=0 ; i<strprop.length ; )
		{
			int	nb = Integer.parseInt(strprop[i+1]);
			//	if property exist, create Datnum object.
			//---------------------------------------------------
			int start_val = i+2;
			int	end_val   = i+2+nb;
			if (nb>0)
				properties[pnum++] = new DbDatum(strprop[i],
										strprop, start_val, end_val);
			else
				properties[pnum++] = new DbDatum(strprop[i]);
			i = end_val;
			
			//	Even if nb property is zero there is a property
			//	set to space char (!!!)
			//--------------------------------------------------------
			if(nb==0) i++;
		}
		return properties;
	}

	//==========================================================================
	/**
	 *	Query the database for a list of object (i.e. non-device, class or device)
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param properties list of property DbDatum objects.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	private DbDatum[] get_obj_property(String name, String type, DbDatum[] properties)
				throws DevFailed
	{
		//	Format input parameters as string array
		//--------------------------------------------
		String[]	array;
		array = new String[properties.length];
		for (int i=0 ; i<properties.length ; i++)
			array[i] = properties[i].name;
		return get_obj_property(name, type, array);
	}
	//==========================================================================
	/**
	 *	Query the database for an object (i.e. non-device, class or device)
	 *	property for the pecified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propename lproperty name.
	 *	@retun property in DbDatum objects.
	 */
	//==========================================================================
	private DbDatum get_obj_property(String name, String type, String propname)
				throws DevFailed
	{
		//	Format input parameters as string array
		//--------------------------------------------
		String[]	array;
		array = new String[1];
		array[0] = propname;
		DbDatum[]	data = get_obj_property(name, type, array);
		return data[0];
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (i.e. non-device, class or device)
	 *	properties for thr dpecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propnames list of property names.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	private DbDatum[] get_obj_property(String name, String type, String[] propnames)
				throws DevFailed
	{
		//	Format input parameters as string array
		//--------------------------------------------
		String[]	array;
		array = new String[1 + propnames.length];
		array[0] = name;
		for (int i=1 ; i<propnames.length+1 ; i++)
			array[i] = propnames[i-1];

		//	Buid command name (depends on object type)
		//---------------------------------------------------
		String	cmd = new String("DbGet"+type+"Property");

		//	Read Database
		//---------------------
		argin.insert(array);
		argout = command_inout(cmd, argin);
		String[]	result = argout.extractStringArray();

//for (int i=0 ; i<result.length ; i++)
//	System.out.println(i + ": " +result[i]);

		//	And convert to DbDatum array before returning
		//-------------------------------------------------
		return stringArray2DbDatum(result);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param properties list of property DbDatum objects.
	 */
	//==========================================================================
	private void delete_obj_property(String name, String type, DbDatum[] properties)
				throws DevFailed
	{
		//	Format input parameters as strin array
		//--------------------------------------------
		String[]	array;
		array = new String[properties.length];
		for (int i=0 ; i<properties.length ; i++)
			array[i] = properties[i].name;

		delete_obj_property(name, type, array);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propname Property name.
	 */
	//==========================================================================
	private void delete_obj_property(String name, String type, String propname)
				throws DevFailed
	{
		//	Format input parameters as strin array
		//--------------------------------------------
		String[]	array;
		array = new String[1];
		array[0] = propname;

		delete_obj_property(name, type, array);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param type	Object type (nothing, class, device..)
	 *	@param propnames Property names.
	 */
	//==========================================================================
	private void delete_obj_property(String name, String type, String[] propnames)
				throws DevFailed
	{
		//	Format input parameters as strin array
		//--------------------------------------------
		String[]	array;
		array = new String[propnames.length+1];
		array[0] = name;
		for (int i=0 ; i<propnames.length ; i++)
			array[i+1] = propnames[i];

		//	Buid command name (depends on object type)
		//---------------------------------------------------
		String	cmd = new String("DbDelete"+type+"Property");

		//	Send it to  Database
		//------------------------------
		argin.insert(array);
		command_inout(cmd, argin);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (i.e. non-device)
	 *	properties for the pecified object.
	 *	@param name Object name.
	 *	@param propnames list of property names.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "";	//	No object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for an object (i.e. non-device)
	 *	property for the pecified object.
	 *	@param name Object name.
	 *	@param propname list of property names.
	 *	@retun property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_property(String name, String propname)
				throws DevFailed
	{
		String	type = "";	//	No object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of object (i.e. non-device)
	 *	properties for thr dpecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Object name.
	 *	@param properties list of property DbDatum objects.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String	type = "";	//	No object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified object
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name Object name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		argin.insert(array);
		command_inout("DbPutProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a  property for the specified object.
	 *	@param name Object name.
	 *	@param propname Property names.
	 */
	//==========================================================================
	public void delete_property(String name, String propname)
				throws DevFailed
	{
		String type = "";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Object name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "";
		delete_obj_property(name, type, properties);
	}

	//==========================================================================
	/**
	 *	Query the database for a list of class
	 *	properties for the pecified object.
	 *	@param classname	device name.
	 *	@param wildcard	propertie's wildcard (* matches any charactere).
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public String[] get_class_property_list(String classname, String wildcard)
				throws DevFailed
	{
		String[]	array = new String[2];
		array[0] = classname;
		array[1] = wildcard;		
		//argin.insert(array);
		argin.insert(classname);
		argout = command_inout("DbGetClassPropertyList", argin);
		return argout.extractStringArray();
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	@param devname	device name.
	 *	@param wildcard	propertie's wildcard (* matches any charactere).
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public String[] get_device_property_list(String devname, String wildcard)
				throws DevFailed
	{
		String[]	array = new String[2];
		array[0] = devname;
		array[1] = wildcard;		
		argin.insert(array);
		argout = command_inout("DbGetDevicePropertyList", argin);
		return argout.extractStringArray();
	}







	
	//**************************************
	//       DEVICE PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	@param name device name.
	 *	@param propnames list of property names.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_device_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "Device";	//	Device object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a  device
	 *	property for the pecified object.
	 *	@param name device name.
	 *	@param propname property name.
	 *	@retun property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_device_property(String name, String propname)
				throws DevFailed
	{
		String	type = "Device";	//	Device object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name device name.
	 *	@param properties list of property DbDatum objects.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_device_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String		type = "Device";	//	Device object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified device
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name device name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_device_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		argin.insert(array);
		command_inout("DbPutDeviceProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Device name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_device_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "Device";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Device name.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_device_property(String name, String propname)
				throws DevFailed
	{
		String type = "Device";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Device name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_device_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "Device";
		delete_obj_property(name, type, properties);
	}






	
	//**************************************
	//      ATTRIBUTE PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a list of device attribute
	 *	properties for the pecified object.
	 *	@param name device name.
	 *	@param propnames list of property names.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_device_attribute_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "DeviceAttribute";	//	Device object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for device attribute
	 *	property for the pecified object.
	 *	@param name device name.
	 *	@param propname  property name.
	 *	@retun property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_device_attribute_property(String name, String propname)
				throws DevFailed
	{
		String	type = "DeviceAttribute";	//	Device object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of device attribute
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name device name.
	 *	@param properties list of property DbDatum objects.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_device_attribute_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String		type = "DeviceAttribute";	//	Device object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified device attribute.
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name device name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_device_attribute_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		argin.insert(array);
		command_inout("DbPutDeviceAttributeProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Device name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_device_attribute_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "DeviceAttribute";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Device name.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_device_attribute_property(String name, String propname)
				throws DevFailed
	{
		String type = "DeviceAttribute";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Device name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_device_attribute_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "DeviceAttribute";
		delete_obj_property(name, type, properties);
	}



	
	//**************************************
	//      CLASS PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a list of class
	 *	properties for the pecified object.
	 *	@param name Class name.
	 *	@param propnames list of property names.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_class_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "Class";	//	class object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a class
	 *	property for the pecified object.
	 *	@param name Class name.
	 *	@param propname list of property names.
	 *	@retun property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_class_property(String name, String propname)
				throws DevFailed
	{
		String	type = "Class";	//	class object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of class
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Class name.
	 *	@param properties list of property DbDatum objects.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_class_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String		type = "Class";	//	Device object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified class.
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name Class name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_class_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		argin.insert(array);
		command_inout("DbPutClassProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_class_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "Class";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Class name.
	 *	@param propname Property name.
	 */
	//==========================================================================
	public void delete_class_property(String name, String propname)
				throws DevFailed
	{
		String type = "Class";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_class_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "Class";
		delete_obj_property(name, type, properties);
	}



	
	//**************************************
	//      CLASS Attribute PROPERTIES MANAGEMENT
	//**************************************	
	//==========================================================================
	/**
	 *	Query the database for a list of class attribute
	 *	properties for the pecified object.
	 *	@param name Class name.
	 *	@param propnames list of property names.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_class_attribute_property(String name, String[] propnames)
				throws DevFailed
	{
		String	type = "ClassAttribute";	//	class Attribute object type
		return get_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Query the database for a class attribute
	 *	property for the pecified object.
	 *	@param name Class name.
	 *	@param propname property name.
	 *	@retun property in DbDatum object.
	 */
	//==========================================================================
	public DbDatum get_class_attribute_property(String name, String propname)
				throws DevFailed
	{
		String	type = "ClassAttribute";	//	class Attribute object type
		return get_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Query the database for a list of class attribute
	 *	properties for the pecified object.
	 *	The property names are specified by the DbDatum array objects.
	 *	@param name Class name.
	 *	@param properties list of property DbDatum objects.
	 *	@retun properties in DbDatum objects.
	 */
	//==========================================================================
	public DbDatum[] get_class_attribute_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String		type = "ClassAttribute";	//	Device object type
		return get_obj_property(name, type, properties);
	}
	//==========================================================================
	/**
	 *	Insert or update a list of properties for the specified class attribute.
	 *	The property names and their values are specified by the DbDatum array.
	 *	
	 *	@param name Class name.
	 *	@param properties Properties names and values array.
	 */
	//==========================================================================
	public void put_class_attribute_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String[]	array = dbdatum2StringArray(name, properties);
		argin.insert(array);
		command_inout("DbPutClassAttributeProperty", argin);		
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param propnames Property names.
	 */
	//==========================================================================
	public void delete_class_attribute_property(String name, String[] propnames)
				throws DevFailed
	{
		String type = "ClassAttribute";
		delete_obj_property(name, type, propnames);
	}
	//==========================================================================
	/**
	 *	Delete a property for the specified object.
	 *	@param name Class name.
	 *	@param propname Property names.
	 */
	//==========================================================================
	public void delete_class_attribute_property(String name, String propname)
				throws DevFailed
	{
		String type = "ClassAttribute";
		delete_obj_property(name, type, propname);
	}
	//==========================================================================
	/**
	 *	Delete a list of properties for the specified object.
	 *	@param name Class name.
	 *	@param properties Property DbDatum objects.
	 */
	//==========================================================================
	public void delete_class_attribute_property(String name, DbDatum[] properties)
				throws DevFailed
	{
		String type = "ClassAttribute";
		delete_obj_property(name, type, properties);
	}
}
