static char RcsId[] = "@(#) $Header: AGPowerSupply.c,v 1.1 91/05/02 08:25:31 goetz Exp $ ";

/*********************************************************************

 File:		AGPowerSupply.c

 Project:	Device Servers

 Description:	Code for implementing the AG Power Supply class
		The AG Power Supply is a simulation of a typical
		power supply at the ESRF. This means it has two
		main state DEVON and DEVOFF, DEVSTANDBY is unknown.
		All the common power supply commands are implemented.
		The simulation runs under OS9 and Unix. It has been
		developed for application program developers who want to
		test their applications without accessing real devices

 Author(s);	A. Goetz 

 Original:	March 1991

 $Log:	AGPowerSupply.c,v $
 * Revision 1.1  91/05/02  08:25:31  08:25:31  goetz (Andy Goetz)
 * Initial revision
 * 

 Copyleft (c) 1991 by European Synchrotron Radiation Facility, 
                      Grenoble, France



 *********************************************************************/

#include <API.h>
#include <DevServer.h>
#include <DevErrors.h>
#include <DevServerP.h>
/*
#include <db_setup.h>
 */
#include <PowerSupply.h>
#include <AGPowerSupplyP.h>
#include <AGPowerSupply.h>

/*
 * public methods
 */

static long class_initialise();
static long object_create();
static long object_initialise();
static long state_handler();

static DevMethodListEntry methods_list[] = {
 {DevMethodClassInitialise, class_initialise},
 {DevMethodCreate, object_create},
 {DevMethodInitialise, object_initialise},
 {DevMethodStateHandler, state_handler},
};

AGPowerSupplyClassRec aGPowerSupplyClassRec = {
   /* n_methods */     sizeof(methods_list)/sizeof(DevMethodListEntry),
   /* methods_list */  methods_list,
};

AGPowerSupplyClass aGPowerSupplyClass = 
                   (AGPowerSupplyClass)&aGPowerSupplyClassRec;

/*
 * public commands 
 */

static long dev_off();
static long dev_on();
static long dev_state();
static long dev_setvalue();
static long dev_readvalue();
static long dev_reset();
static long dev_error();
static long dev_local();
static long dev_remote();
static long dev_status();
static long dev_update();

static DevCommandListEntry commands_list[] = {
 {DevOff, dev_off, D_VOID_TYPE, D_VOID_TYPE},
 {DevOn, dev_on, D_VOID_TYPE, D_VOID_TYPE},
 {DevState, dev_state, D_VOID_TYPE, D_SHORT_TYPE},
 {DevSetValue, dev_setvalue, D_FLOAT_TYPE, D_VOID_TYPE},
 {DevReadValue, dev_readvalue, D_VOID_TYPE, D_FLOAT_READPOINT},
 {DevReset, dev_reset, D_VOID_TYPE, D_VOID_TYPE},
 {DevStatus, dev_status, D_VOID_TYPE, D_STRING_TYPE},
 {DevError, dev_error, D_VOID_TYPE, D_VOID_TYPE},
 {DevLocal, dev_local, D_VOID_TYPE, D_VOID_TYPE},
 {DevRemote, dev_remote, D_VOID_TYPE, D_VOID_TYPE},
 {DevUpdate, dev_update, D_VOID_TYPE, D_STATE_FLOAT_READPOINT},
};

static long n_commands = sizeof(commands_list)/sizeof(DevCommandListEntry);

/*
 * a template copy of the default powersupply that normally gets created
 * by the DevMethodCreate. it is initialised in DevMethodCLassInitialise 
 * to default values. these defaults can also be specified in the resource 
 * file or via an admin command.
 */

static AGPowerSupplyRec aGPowerSupplyRec;
static AGPowerSupply aGPowerSupply =
                     (AGPowerSupply)&aGPowerSupplyRec;

/*
 * template resource table used to access the static database
 */
  
   db_resource res_table[] = {
               {"state",D_LONG_TYPE},
               {"set_val",D_FLOAT_TYPE},
               {"channel",D_SHORT_TYPE},
               {"n_ave",D_SHORT_TYPE},
               {"conv_unit",D_STRING_TYPE},
               {"set_offset",D_FLOAT_TYPE},
               {"read_offset",D_FLOAT_TYPE},
               {"set_u_limit",D_FLOAT_TYPE},
               {"set_l_limit",D_FLOAT_TYPE},
               {"polarity",D_SHORT_TYPE},
                             };
   int res_tab_size = sizeof(res_table)/sizeof(db_resource);


/*======================================================================
 Function:      static long class_initialise()

 Description:	Initialise the AGPowerSupplyClass, is called once for
		this class per process. class_initialise() will initialise
		the class structure (aGPowerSupplyClass) and the default 
		powersupply device (aGPowerSupply). At present default
		values are hardwired in the include and source code files.

 Arg(s) In:	none

 Arg(s) Out:	long *error - pointer to error code if routine fails.
 =======================================================================*/

static long class_initialise(error)
long *error;
{
   int iret=0;
   AGPowerSupply ps;
   int state;
  
/*
 * AGPowerSupplyClass is a subclass of PowerSupplyClass
 */

   aGPowerSupplyClass->devserver_class.superclass = (DevServerClass)powerSupplyClass;
   aGPowerSupplyClass->devserver_class.class_name = (char*)malloc(sizeof("AGPowerSupplyClass"));
   sprintf(aGPowerSupplyClass->devserver_class.class_name,"AGPowerSupplyClass");

/*
 * commands implemented for the AG PowerSUpply class
 */

   aGPowerSupplyClass->devserver_class.n_commands = n_commands;
   aGPowerSupplyClass->devserver_class.commands_list = commands_list;

   aGPowerSupplyClass->devserver_class.class_inited = 1;
/*
 * initialise the template powersupply so that DevMethodCreate has
 * default values for creating a powersupply, these values will be
 * overridden by the static database (if defined there). 
 */

   aGPowerSupply->devserver.class_pointer = (DevServerClass)aGPowerSupplyClass;
/*
 * default is to start with powersupply switched OFF; the state 
 * variable gets (ab)used during initialisation to interpret the
 * initial state of the powersupply: 0==DEVOFF, 1==DEVON. this is
 * because the database doesn't support the normal state variables
 * like DEVON, DEVSTANDBY, DEVINSERTED, etc.
 */
   aGPowerSupply->devserver.state = 0;
   aGPowerSupply->devserver.n_state = aGPowerSupply->devserver.state;
   aGPowerSupply->powersupply.set_val = 0.0;
   aGPowerSupply->powersupply.read_val = 0.0;
   aGPowerSupply->powersupply.channel = 1;
   aGPowerSupply->powersupply.n_ave = 1;
   aGPowerSupply->powersupply.conv_unit = (char*)malloc(sizeof("AMP"));
   sprintf(aGPowerSupply->powersupply.conv_unit,"AMP");
   aGPowerSupply->powersupply.set_offset = 0.0,
   aGPowerSupply->powersupply.read_offset = 0.0;
   aGPowerSupply->powersupply.set_u_limit = AG_MAX_CUR;
   aGPowerSupply->powersupply.set_l_limit = AG_MIN_CUR;
   aGPowerSupply->powersupply.polarity = 1.0;

/*
 * interrogate the static database for default values 
 */
  
   ps = aGPowerSupply;
   res_table[0].resource_adr = &(ps->devserver.state);
   res_table[1].resource_adr = &(ps->powersupply.set_val);
   res_table[2].resource_adr = &(ps->powersupply.channel);
   res_table[3].resource_adr = &(ps->powersupply.n_ave);
   res_table[4].resource_adr = &(ps->powersupply.conv_unit);
   res_table[5].resource_adr = &(ps->powersupply.set_offset);
   res_table[6].resource_adr = &(ps->powersupply.read_offset);
   res_table[7].resource_adr = &(ps->powersupply.set_u_limit);
   res_table[8].resource_adr = &(ps->powersupply.set_l_limit);
   res_table[9].resource_adr = &(ps->powersupply.polarity);

   if(db_getresource("CLASS/AGPS/DEFAULT",res_table,res_tab_size,error))
   {
      printf("class_initialise(): db_getresource() failed, error %d\n",error);
      return(-1);
   }
   else
   {
      printf("default values after searching the static database\n\n");
      printf("CLASS/AGPS/DEFAULT/state         D_LONG_TYPE    %6d\n",ps->devserver.state);
      printf("CLASS/AGPS/DEFAULT/set_val       D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_val);
      printf("CLASS/AGPS/DEFAULT/channel       D_SHORT_TYPE   %6d\n",ps->powersupply.channel);
      printf("CLASS/AGPS/DEFAULT/n_ave         D_SHORT_TYPE   %6d\n",ps->powersupply.n_ave);
      printf("CLASS/AGPS/DEFAULT/conv_unit     D_STRING_TYPE  %6s\n",ps->powersupply.conv_unit);
      printf("CLASS/AGPS/DEFAULT/set_offset    D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_offset);
      printf("CLASS/AGPS/DEFAULT/read_offset   D_FLOAT_TYPE   %6.0f\n",ps->powersupply.read_offset);
      printf("CLASS/AGPS/DEFAULT/set_u_limit   D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_u_limit);
      printf("CLASS/AGPS/DEFAULT/set_l_limit   D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_l_limit);
      printf("CLASS/AGPS/DEFAULT/polarity      D_SHORT_TYPE   %6d\n",ps->powersupply.polarity);
   }

   printf("returning from class_initialise()\n");
   return(iret);
}

/*======================================================================
 Function:      static long object_create()

 Description:	create a AGPowerSupply object. This involves allocating
		memory for this object and initialising its name.

 Arg(s) In:	char *name - name of object.

 Arg(s) Out:	DevServer *ds_ptr - pointer to object created.
		long *error - pointer to error code (in case of failure)
 =======================================================================*/

static long object_create(name, ds_ptr, error)
char *name;
DevServer *ds_ptr;
long *error;
{
   int iret = 0;
   AGPowerSupply ps;

   printf("arrived in object_create(), name %s\n",name);

   ps = (AGPowerSupply)malloc(sizeof(AGPowerSupplyRec));

/*
 * initialise server with template
 */

   *(AGPowerSupplyRec*)ps = *(AGPowerSupplyRec*)aGPowerSupply;

/*
 * finally initialise the non-default values
 */

   ps->devserver.name = (char*)malloc(strlen(name));
   sprintf(ps->devserver.name,"%s",name);
   
   *ds_ptr = (DevServer)ps;

   printf("leaving object_create() and all OK\n");

   return(iret);
}
 
static long object_initialise(ps,error)
AGPowerSupply ps;
long *error;
{
   printf("arrived in object_initialise()\n");
/*
 * initialise powersupply with values defined in database
 */

   res_table[0].resource_adr = &(ps->devserver.state);
   res_table[1].resource_adr = &(ps->powersupply.set_val);
   res_table[2].resource_adr = &(ps->powersupply.channel);
   res_table[3].resource_adr = &(ps->powersupply.n_ave);
   res_table[4].resource_adr = &(ps->powersupply.conv_unit);
   res_table[5].resource_adr = &(ps->powersupply.set_offset);
   res_table[6].resource_adr = &(ps->powersupply.read_offset);
   res_table[7].resource_adr = &(ps->powersupply.set_u_limit);
   res_table[8].resource_adr = &(ps->powersupply.set_l_limit);
   res_table[9].resource_adr = &(ps->powersupply.polarity);

   if(db_getresource(ps->devserver.name,res_table,res_tab_size,error))
   {
      printf("class_initialise(): db_getresource() failed, error %d\n",error);
      return(-1);
   }
   else
   {
      printf("initial values after searching the static database for %s\n\n",ps->devserver.name);
      printf("state         D_LONG_TYPE    %6d\n",ps->devserver.state);
      printf("set_val       D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_val);
      printf("channel       D_SHORT_TYPE   %6d\n",ps->powersupply.channel);
      printf("n_ave         D_SHORT_TYPE   %6d\n",ps->powersupply.n_ave);
      printf("conv_unit     D_STRING_TYPE  %6s\n",ps->powersupply.conv_unit);
      printf("set_offset    D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_offset);
      printf("read_offset   D_FLOAT_TYPE   %6.0f\n",ps->powersupply.read_offset);
      printf("set_u_limit   D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_u_limit);
      printf("set_l_limit   D_FLOAT_TYPE   %6.0f\n",ps->powersupply.set_l_limit);
      printf("polarity      D_SHORT_TYPE   %6d\n",ps->powersupply.polarity);

/*
 * interpret the initial state of the powersupply
 */
      if (ps->devserver.state == 1)
      {
         printf("switching ON\n");
         dev_on(ps,NULL,NULL,error);
/*
 * if switched ON then set the current too
 */
         dev_setvalue(ps,&(ps->powersupply.set_val),NULL,error);
      }
      else
      {
         printf("switching OFF\n");
/*
 * default is to assume the powersupply is OFF
 */
         dev_off(ps,NULL,NULL,error);
      }
   }

   return(0);
   }

/*======================================================================
 Function:      static long state_handler()

 Description:	Check if the command to be executed does not violate
		the present state of the device.

 Arg(s) In:	AGPowerSupply ps - device to execute command to.
		DevCommand cmd - command to be executed.

 Arg(s) Out:	long *error - pointer to error code (in case of failure).
 =======================================================================*/

static long state_handler( ps, cmd, error)
AGPowerSupply ps;
DevCommand cmd;
long *error;
{
   long iret = 0;
   long int p_state, n_state;

   p_state = ps->devserver.state;

/*
 * before checking out the state machine assume that the state
 * doesn't change i.e. new state == old state
 */
   n_state = p_state;

   switch (p_state) {
   
   case (DEVOFF) :
   {
      switch (cmd) {
      
      case (DevOn) : n_state = DEVON;
                     break;

      case (DevError) : n_state = DEVFAULT;
                        break;

      case (DevLocal) : n_state = DEVLOCAL;
                        break;

/* following commands are ignored in this state */

      case (DevSetValue) :
      case (DevReadValue) : iret = -1;
                            *error = DevErr_CommandIgnored;
                            break;
/* following commands don't change the state machine */

      case (DevReset) : 
      case (DevRemote) : 
      case (DevOff) : 
      case (DevState) : 
      case (DevUpdate) :
      case (DevStatus) : break;

      default : break;
      }
      
      break;
   }

   case (DEVON) :
   {
      switch (cmd) {

      case (DevOff) : n_state = DEVOFF;
                      break;
      case (DevError) : n_state = DEVFAULT;
                        break;
      case (DevLocal) : n_state = DEVLOCAL;
                        break;

/* following commands violate the state machine */

      case (DevRemote) : 
      case (DevReset) : iret = -1;
                        (*error) = DevErr_AttemptToViolateStateMachine;
                        break;

/* the following commands don't change the state of the machine */

      case (DevState) : 
      case (DevStatus) : 
      case (DevReadValue) : 
      case (DevSetValue) :
      case (DevUpdate) :
      case (DevRun) : break;

      default : break;
      }

      break;
   }
   case (DEVLOCAL) :
   {
      switch (cmd) {

      case (DevRemote) : n_state = DEVOFF;
                         break;

/* the following commands violate the state machine */

      case (DevOn) :
      case (DevOff) :
      case (DevRun) :
      case (DevReset) :
      case (DevStandby) :
      case (DevError) : iret = -1;
                        (*error) = DevErr_AttemptToViolateStateMachine;
                        break;

/* following commands are ignored */
      case (DevSetValue) : iret = -1;
                           *error = DevErr_CommandIgnored;
                           break;

/* the following commands don't change the state of the machine */

      case (DevState) :
      case (DevStatus) :
      case (DevLocal) :
      case (DevUpdate) :
      case (DevReadValue) : break;

      default : break;

      }

      break;
   }
   case (DEVFAULT) :
   {
      switch (cmd) {

      case (DevReset) : n_state = DEVOFF;
                        break;
      
/* the following commands violate the state machine */

      case (DevOff) :
      case (DevRemote) :
      case (DevOn) :
      case (DevLocal) : iret = -1;
                        (*error) = DevErr_AttemptToViolateStateMachine;
                        break;

/* following commands are ignored */

      case (DevSetValue) : 
      case (DevReadValue) : iret = -1;
                            *error = DevErr_CommandIgnored;
                            break;

/* the following commands don't change the stae of the machine */

      case (DevState) : 
      case (DevStatus) : 
      case (DevUpdate) :
      case (DevError) : break;

      default : break;

      }
      break;
   }

   default : break;
   }

/*
 * update powersupply's private variable n_state so that other methods
 * can use it too.
 */

   ps->devserver.n_state = n_state;

/* printf("state_handler(): p_state %2d n_state %2d, iret %2d\n",
          p_state,n_state, iret); */

   return(iret);
}

/*======================================================================
 Function:      static long dev_off()

 Description:	switch the simulated power supply off. the read value 
		and set value get set to zero - this is a convention
		adopted at the ESRF for all powersupplies.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error code in case of failure.
 =======================================================================*/

static long dev_off (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

	ps->powersupply.read_val = 0.0;
	ps->powersupply.set_val = 0.0;
	ps->devserver.state = DEVOFF;

	return (iret);
}
	
/*======================================================================
 Function:      static long dev_on()

 Description:	switch simulated powersupply ON

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error code (in case of failure)
 =======================================================================*/

static long dev_on (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

	ps->devserver.state = DEVON;

	return (iret);
}

/*======================================================================
 Function:      static long dev_state()

 Description:	return state of simulated power supply.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - S_data branch contains status as short
		long *error - pointer to error code (in case routine fails)
 =======================================================================*/

static long dev_state (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevShort *argout;
long *error;
{
	long iret = 0;

/* this command can be always executed independent of the state  */

	*argout = ps->devserver.state;

	return (iret);
}

/*======================================================================
 Function:      static long dev_setvalue()

 Description:	Routine to set current of simulated power supply. This
		version does not check the limits of the set_value.

 Arg(s) In:	AGPowerSupply ps - object to execute command on.
		DevUnion *argin - F_data branch of union contains set value

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error code (in the case of failure)
 =======================================================================*/

static long dev_setvalue (ps, argin, argout, error)
AGPowerSupply ps;
DevFloat *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

	ps->powersupply.set_val = *argin;

	return (iret);
}

/*======================================================================
 Function:      static long dev_readvalue()

 Description:	Return read value of simulated power supply. The read
		value is the last set value with some simulated noise
		on it. The noise is taken from the time() and is therefore
		repetitive.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - F_data branch contains read value.
		long *error - pointer to error code (in the case of failure)
 =======================================================================*/

static long dev_readvalue (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevFloatReadPoint *argout;
long *error;
{
	long iret = 0;
        float per_error;

	argout->set = ps->powersupply.set_val;

/*
 * the last 4 bits of time() provide some random noise on this
 * simulated reading - andy 11apr90
 */
	per_error = AG_PER_ERROR/100.*2.*
                    (((float)(time(NULL)&0xf)/15.0)-0.5);
/*	            printf("ReadValue(): percent error %6.3f\n",
                            per_error);*/
        argout->read = ps->powersupply.set_val*
                                (1.-per_error);
        ps->powersupply.read_val = argout->read;

	return (iret);
}

/*======================================================================
 Function:      static long dev_remote()

 Description:	Switch simulated power supply from LOCAL mode to socalled
		REMOTE mode i.e. a mode in which the power supply can receive
		commands.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error code (in case of failure)
 =======================================================================*/

static long dev_remote (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

	ps->devserver.state = ps->devserver.n_state;
	
	return (iret);
}

/*======================================================================
 Function:      static long dev_reset()

 Description:	Reset simulated power supply to a well known state.
		Used to recover from errors mostly. All set and read 
		points are reset at the same time.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error in the case of failure.
 =======================================================================*/

static long dev_reset (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

	ps->powersupply.set_val = 0.0;
        ps->powersupply.read_val = 0.0;
        ps->powersupply.fault_val = 0;
        ps->powersupply.cal_val = 0.0;
        ps->powersupply.conv_val = 1.0;
        ps->devserver.state = ps->devserver.n_state;

	return (iret);
}

/*======================================================================
 Function:      static long dev_error()

 Description:	Simulate an error condition on the simulated power supply.
		The system call time() is used to generate any one of 8
		possible errors.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error code, in the case of failure.
 =======================================================================*/

static long dev_error (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

/*	printf("DevError():\n");*/

/*
 * use the last 16 bits of the system time to generate an error - andy
 */

	ps->powersupply.fault_val = (unsigned short)(time(NULL)&0xff);
	ps->devserver.state = ps->devserver.n_state;
/*	printf("DevError(): fault_val %2d\n",
	        ps->powersupply.fault_val);*/

	return (iret);
}

/*======================================================================
 Function:      static long dev_status()

 Description:	Return the state as an ASCII string. Interprets the error
		flag as well if the status is FAULT.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command.
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - STR_data branch of union contains string.
		long *error - pointer to error code (in the case of failure)
 =======================================================================*/

static long dev_status (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevString *argout;
long *error;
{
	static char mess[1024];
	int fault = ps->powersupply.fault_val;
	long iret = 0;
	long p_state;


	p_state = ps->devserver.state;

	switch (p_state) {

	case (DEVOFF) : sprintf(mess,"%s","Off");
	                break;

	case (DEVON) : sprintf(mess,"%s","On");
	                    break;

	case (DEVLOCAL) : sprintf(mess,"%s","Local");
	                  break;

	case (DEVFAULT) : sprintf(mess,"%s","Fault\n");
	                  break;

	default : sprintf(mess,"%s","Unknown");
	          break;
	}

/* translate fault into a string */

	if ((fault != 0) && (p_state == DEVFAULT))
	{
	if ((fault & AG_OVERTEMP) != 0) 
	{
	  sprintf(mess+strlen(mess)," %s","Overtemp");
	}
	if ((fault & AG_NO_WATER) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","No Cooling");
	}
	if ((fault & AG_CROWBAR) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","Crowbar");
	}
	if ((fault & AG_RIPPLE) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","Ripple");
	}
	if ((fault & AG_MAINS) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","Mains");
	}
	if ((fault & AG_LOAD) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","Load");
	}
	if ((fault & AG_TRANSFORMER) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","Transformer");
	}
	if ((fault & AG_THYRISTOR) != 0)
	{
	   sprintf(mess+strlen(mess)," %s","Thyristor");
	}
	}

	*argout = mess;

	return(iret);
}

/*======================================================================
 Function:      static long dev_local()

 Description:	Switch simulated power supply to LOCAL mode. In this mode
		the power supply does not respond to any commands until
		the next DevRemote command is executed.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - void.
		long *error - pointer to error code in the case of failure.
 =======================================================================*/

static long dev_local (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevVoid *argout;
long *error;
{
	long iret = 0;

	ps->devserver.state = ps->devserver.n_state;

	return(iret);
}

/*======================================================================
 Function:      static long dev_update()

 Description:	Return the state and the read and set points. This command
		is a combination of the DevState and the DevReadValue commands.

 Arg(s) In:	AGPowerSupply ps - object on which to execute command
		DevUnion *argin - void.

 Arg(s) Out:	DevUnion *argout - SRFP_data branch contains the three values.
		long *error - pointer to error code (in the case of failure)
 =======================================================================*/

static long dev_update (ps, argin, argout, error)
AGPowerSupply ps;
DevVoid *argin;
DevStateFloatReadPoint *argout;
long *error;
{
	DevShort darg_short;
	DevFloatReadPoint darg_frp;

/*
 * update state
 */
	dev_state(ps, NULL, &darg_short, error);
	argout->state = darg_short;

/*
 * get latest set and read values
 */
	dev_readvalue(ps, NULL, &darg_frp, error);
	argout->set = darg_frp.set;
	argout->read = darg_frp.read;

/*
	printf("dev_update(): state %d set %f read %f\n",
	        argout->state,
	        argout->set,
	        argout->read);
 */

	return(0);
}
/*======================================================================
 Function:      static long ()

 Description:

 Arg(s) In:

 Arg(s) Out:
 =======================================================================*/

