static char RcsId[] = 
"@(#)$Header: util_api.c,v 8.18 2000/11/12 13:24:01 goetz Rel $";

/*+*******************************************************************

 File       :	util_api.c

 Project    :	Device Servers with SUN-RPC

 Description:	Application Programmers Interface 
		
            :	Utilities for the interface to access and 
		handle remote devices.

 Author(s)  :	Jens Meyer

 Original   :	April 1993

 $Revision: 8.18 $
 $Date: 2000/11/12 13:24:01 $

 $Author: goetz $

 $Log:	util_api.c,v $
 * Revision 8.18  2000/11/12  13:24:01  13:24:01  goetz (Andy Goetz)
 * added dynamic error handling; improved TANGO error treatment
 * 
 Revision 8.17  1900/09/26 13:36:22  goetz
 tango_dev_putget() and tango_dev_putget_raw() implement immediate reconnection

 * Revision 8.16  2000/09/25  14:50:12  14:50:12  goetz (Andy Goetz)
 * tango_api implements TANGO_HOST, stateless import and reimport
 * 
 * Revision 8.15  20/0./7.  1.:6.:9.  1.:6.:9.  goetz (Andy Goetz)
 * changed from tango database to host mysql, added support for float and double arrays
 * 
 * Revision 8.14  20/0./6.  2.:4.:0.  2.:4.:0.  goetz (Andy Goetz)
 * fixed bug in dev_event_listen() which caused server to crash
 * 
 Revision 8.13  2000/06/02 21:56:37  goetz
 _DEVICE_H now _TANGO_DEVICE_H in Device.H; (char*)taco_tango

 Revision 8.12  2000/06/02 15:57:55  goetz
 dev_event_fire() now device specific; ported to SuSE V6.4

 Revision 8.11  2000/05/31 07:47:42  goetz
 tango_api has local copies of argc and argv, ported to HP-UX

 * Revision 8.10  2000/05/29  21:39:09  21:39:09  goetz (Andrew GOETZ)
 * fixed prototyping problems with C++ on HP-UX
 * 
 * Revision 8.9  2000/05/29  18:13:19  18:13:19  goetz (Andrew GOETZ)
 * fixed bug in dev_import_timeout() which prevented timeouts < 2s
 * 
 Revision 8.8  2000/05/02 15:32:04  goetz
 added prototype for tsleep() for OS9 C++

 * Revision 8.7  2000/05/02  14:19:27  14:19:27  goetz (Andy Goetz)
 * removed check for device server running on different host
 * 
 * Revision 8.6  2000/05/02  13:28:05  13:28:05  goetz (Andy Goetz)
 * added random sleep to gettransient + 3 retries
 * 
 Revision 8.5  2000/03/27 17:13:41  goetz
 declared pmap_getmaps() as external C function for OS9 C++

 Revision 8.4  2000/03/13 14:49:57  goetz
 import timeout now programmable using dev_import_timeout()

 Revision 8.3  2000/03/13 10:31:31  goetz
 added Jens' modifications to fix bug in local putget

 Revision 8.2  2000/03/10 17:01:18  goetz
 dev_synch() now excludes udp clients and new imports

 Revision 8.1  2000/01/18 16:48:37  goetz
 tango_dev_import() now only called in dev_import if -DTANGO

 Revision 8.0  1999/12/28 14:18:21  goetz
 added TANGO support for TACO dev_xxx() calls via -DTANGO for C++

 Revision 7.10  1999/11/25 08:27:09  goetz
 replaced fprintf(stderr) with printf; made startup() C++ compatible

 Revision 7.9  1999/11/22 20:18:55  goetz
 removed const from gettransient() prototype

 Revision 7.8  1999/11/21 20:45:34  goetz
 included all M.Diehl's patches (major changes to gettransient() + main())

 Revision 7.7  1999/11/18 22:28:51  goetz
 fixed event multi-client bug, free client handle after event_listen, timeout in dev_synch()

 Revision 7.6  1999/11/01 19:05:14  goetz
 make local copy of user name returned by getpwuid() in sec_api.c

 Revision 7.5  1999/08/06 17:44:01  goetz
 removed usage of varargs for g++ on solaris and hpux

 * Revision 7.4  99/07/09  05:15:25  05:15:25  goetz (Andy Goetz)
 * added M.Diehl's patch to DevServerSig.c to exit() after calling unregister_server()
 * 
 Revision 7.3  1999/06/07 15:26:50  goetz
 fixed bug with multi-nethost reimport, device name stored with nethost

 Revision 7.2  1999/05/12 15:21:48  goetz
 changed dev_event_fire() to void; fixed bug in dev_event_unlisten()

 * Revision 7.1  99/05/11  15:59:36  15:59:36  goetz (Andy Goetz)
 * replace static declaration of event_client[] array by malloc()
 * 
 Revision 7.0  1999/04/26 07:30:31  goetz
 implemented user events (added event_api.c)

 Revision 6.15  1999/03/27 09:00:10  goetz
 asynch_cleanup() only checks for asynch clients; C++ returns output arguments if DS_WARNING

 * Revision 6.14  99/02/27  15:21:10  15:21:10  goetz (Andy Goetz)
 * fixed reimport bug; disabled reimport for rpc version 1
 * 
 Revision 6.13  1999/02/05 00:50:17  goetz
 fixed reimport socket+memory leak; ported to aCC on HPUX; suppressed rpc errors

 * Revision 6.12  99/01/18  21:38:53  21:38:53  goetz (Andy Goetz)
 * suppressed test for RPC_TIMEDOUT in svc_check()
 * 
 * Revision 6.11  99/01/14  23:14:42  23:14:42  goetz (Andy Goetz)
 * dbase timeout = 4 s; unregister ver. 1; svc_check() handles RPC_TIMEDOUT
 * 
 * Revision 6.10  98/12/22  10:29:36  10:29:36  taurel (E.Taurel)
 * Added support for Solaris and HP C++ native compiler. Remove K_R C support
 * 
 * Revision 6.9  98/12/15  14:04:52  14:04:52  goetz (Andy Goetz)
 * unregister RPC; use pgm no. 1 as flag; dev_pending() checks timeout; extern C prototype
 * 
 * Revision 6.8  98/11/20  14:35:27  14:35:27  goetz (Andy Goetz)
 * unregister API_VERSION (4) correctly now + check return value of svc_register()
 * 
 * Revision 6.7  98/11/20  10:34:52  10:34:52  goetz (Andy Goetz)
 * improved gettrans_ut(); multi-config_flags=0; removde static client_data; fixed linux select() bug; only register pgm # 0,4,5; portmap address == 111
 * 
 * Revision 6.6  98/09/24  09:25:03  09:25:03  goetz (Andy Goetz)
 * fixed stupid type which reintroduced asynchronous memory leak
 * 
 * Revision 6.5  98/08/26  16:20:27  16:20:27  goetz (Andy Goetz)
 * added dynamic NETHOST allocation, fixed bug in db_setup_multi, ported to Irix 6.4
 * 
 * Revision 6.4  98/04/29  10:21:17  10:21:17  goetz (Andy Goetz)
 * asynchronous - memory leak fixed, OS9 bug fixed, stale handle cleanup, pending=300
 * 
 * Revision 6.3  97/12/10  15:28:25  15:28:25  meyer (J.Meyer)
 * Changed select() timeout in rpc_check_host() to 200ms
 * 
 * Revision 6.2  97/12/01  12:59:07  12:59:07  goetz (Andy Goetz)
 * intermediate checkin before opening 5.14.1 branch
 * 
 * Revision 6.1  97/11/20  21:49:17  21:49:17  goetz (Andy Goetz)
 * main() continue even if cannot register asynch rpc
 * 
 * Revision 6.0  97/11/03  17:22:19  17:22:19  goetz (Andy Goetz)
 * merged synchronous and asynchronous branches
 * 
 Revision 5.15  1997/10/24 15:26:55  klotz
 ported to Windows NT 4.0 + NMSDK 5.0 ; tested with TextTalkds ; new DevServerMain.c

 * Revision 5.14  97/09/25  17:36:35  17:36:35  goetz (Andy Goetz)
 * fixed
 * 
 Revision 5.13  1997/09/03 17:34:28  meyer
 Added casting for ucc++ in DevServerSig.c

 * Revision 5.12  97/09/03  15:54:04  15:54:04  meyer (J.Meyer)
 * Changed OS9 signal handling from intercept() to signal().
 * 
 * Revision 5.11  97/08/05  14:19:17  14:19:17  goetz (Andy Goetz)
 * ported to VxWork 5.2, pgm. no. based on CRC, -m suppressed, -ve errors bug fix
 * 
 * Revision 5.10  97/03/14  17:26:06  17:26:06  dserver ()
 * fixed bug with open file handles after free
 * 
 * Revision 5.8.1.4  97/03/14  17:24:42  17:24:42  goetz (Andy Goetz)
 * fixed bug with open file handles after free
 * 
 * Revision 5.9  97/03/13  11:17:09  11:17:09  klotz (W.D. Klotz)
 * first part of NT4.0 port
 * 
 * Revision 5.8  97/01/31  12:06:16  12:06:16  goetz (Andy Goetz)
 * dev_protocol(), Linux+security bug fixes, dev_import() ignores timeout on NULL procedure, svc_run for OS9
 * 
 * Revision 5.7  97/01/14  08:57:15  08:57:15  goetz (Andy Goetz)
 * added stateless to dev_free(),_raw(),_asyn(),_protocol(),_timeout()
 * 
 * Revision 5.6  97/01/12  18:33:35  18:33:35  goetz (Andy Goetz)
 * dev_import() now stateless i.e. import works even if server is not running
 * 
 * Revision 5.5  97/01/08  17:02:22  17:02:22  goetz (Andy Goetz)
 * Ported to Linux (2.0.9)
 * 
 * Revision 5.4  96/11/27  09:58:08  09:58:08  goetz (Andy Goetz)
 * removed rpc admin interface from DevServerMain.c; putenv() suppressed for _UCC
 * 
 * Revision 5.3  96/11/26  16:24:35  16:24:35  goetz (Andy Goetz)
 * fixed bug (i_nethost=0) in dev_import_local()
 * 
 * Revision 5.2  96/11/26  13:30:56  13:30:56  goetz (Andy Goetz)
 * OICDevice (C++ wrapper for OIC) added + fixed multi-nethost memory leaks
 * 
 * Revision 5.1  96/10/31  18:02:21  18:02:21  goetz (Andy Goetz)
 * added multi-nethost support
 * 
 * Revision 5.0  96/10/31  17:59:13  17:59:13  goetz (Andy Goetz)
 * added multi-nethost support
 * 

 Copyright (c) 1990 by European Synchrotron Radiation Facility, 
                       Grenoble, France

********************************************************************-*/

#include <API.h>
#include <ApiP.h>
#include <DevServer.h>
#include <API_xdr_vers3.h>

#include <DserverTeams.h>

#include <Admin.h>
#include <DevErrors.h>

#if !defined _NT
#if  ( OSK | _OSK )
#include <inet/socket.h>
#include <inet/netdb.h>
#else
#ifdef sun
#include <sys/filio.h>
#endif /* sun */
#include <sys/socket.h>
#if !defined vxworks
#include <netdb.h>
#endif /* !vxworks */
#endif /* OSK | _OSK */
#endif  /* _NT */

/* pointer to global error message */
extern char *dev_error_string;

static long get_cmd_string PT_( (devserver ds, long cmd, char *cmd_str, long *error) );


/****************************************
 *          Globals	                *
 ****************************************/

/*
 *  Types for global state flags for the current
 *  server or client status and for Database Server
 *  information are defined in API.h
 */

/*
 *  Configuration flags
 */

extern configuration_flags      config_flags;

/*
 * Global structure for multiple control systems, setup by
 * setup_config_multi() but used by all multi-nethost functions
 */

extern nethost_info *multi_nethost;
extern long  max_nethost;

/*  
 *  Structure for the adnministration of open RPC connections.
 */

extern server_connections	svr_conns [];


/*
 *  Table of command CLASS resources, read for dev_cmd_query()
 */

db_resource   res_tab [] = {
	{(char *)"In_Type",  D_STRING_TYPE, NULL},
	{(char *)"Out_Type", D_STRING_TYPE, NULL},
};


/*+**********************************************************************
 Function   :	extern long dev_putget_raw()

 Description:	dev_putget_raw execute commands on a device
		by passing input data and retrieving 
		output data in a raw opaque format.

 Arg(s) In  :	devserver ds       - handle to access the device.
	    :   long cmd           - command to be executed.
            :   DevArgument argin  - pointer to input arguments.
            :   DevType argin_type - data type of input arguments.

 Arg(s) Out :	DevOpaque argout    - pointer to DevOpaque structure.
            :   DevType argout_type - data type of output arguments,
				      as specified for the command.
            :	long *error         - Will contain an appropriate error 
				      code if the corresponding call 
				      returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc dev_putget_raw (devserver ds, long cmd, DevArgument argin,
			      DevType argin_type, DevOpaque *argout,
			      DevType argout_type, long *error)
{
	_server_data		server_data;
	_client_raw_data	client_data;
	enum clnt_stat		clnt_stat;
	long			client_id = 0;
	long 			status;

	*error = 0;
#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\ndev_putget_raw() : entering routine\n");
#endif /* EBUG */

	if (dev_error_string != NULL)
	{
		free(dev_error_string);
		dev_error_string = NULL;
	}
#ifdef TANGO
	if (ds->rpc_protocol == D_IIOP)
	{
                status = tango_dev_putget_raw(ds, cmd, argin, argin_type,
                                              argout, argout_type, error);
		return(status);
	}
#endif /* TANGO */
	/*
	 *  check data types
	 */

	if (  argin_type < 0 || argout_type < 0)
	{
		*error = DevErr_DevArgTypeNotRecognised;
		return(-1);
	}

	/*
         * Verify the RPC connection.
         */

	if ( dev_rpc_connection (ds, error)  == DS_NOTOK )
	{
		return (DS_NOTOK);
	}

	/*
	 * If the security system is configured, 
	 * verify the security key
	 */

	if ( config_flags.security == True )
	{
		if ( verify_sec_key (ds, &client_id, error) == DS_NOTOK )
		{
			return (DS_NOTOK);
		}
	}

	/*
	 *  fill in data to rpc transfer structures server_data
	 *  and client_data.
	 */

	server_data.ds_id	= ds->ds_id;
	server_data.client_id	= client_id;
	server_data.access_right= ds->dev_access;
	server_data.cmd		= cmd;
	server_data.argin_type	= argin_type;
	server_data.argout_type	= argout_type;
	server_data.argin	= (char *) argin;

	server_data.var_argument.length   = 0;
	server_data.var_argument.sequence = NULL;


	memset ((char *)&client_data, 0, sizeof (client_data));
	client_data.argout	= (char *) argout;

#ifdef EBUG
	dev_printdebug (DBG_API,
	    "dev_putget() : server data -> \n");
	dev_printdebug (DBG_API,
	    "ds_id=%d  cmd=%d  intype=%d  outtype=%d\n",
	    server_data.ds_id, server_data.cmd,
	    server_data.argin_type,
	    server_data.argout_type);
#endif /* EBUG */

	/*
	 *  call RPC_DEV_PUTGET_RAW at the correct device server
	 */

	/*
	 * Call a device with the current version number >1.
   	 */
	if ( ds->vers_number > DEVSERVER_VERS)
	{
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_PUTGET_RAW,
		    (xdrproc_t)xdr__server_data, (caddr_t) &server_data,
		    (xdrproc_t)xdr__client_raw_data, (caddr_t) &client_data, TIMEVAL(timeout));
	}
	else
	{
		/*
	    * Call a device from an old version server.
	    */
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_PUTGET_RAW,
		    (xdrproc_t)xdr__server_data_3, (caddr_t) &server_data,
		    (xdrproc_t)xdr__client_raw_data_3, (caddr_t) &client_data, TIMEVAL(timeout));
	}


	/*
         * Check for errors on the RPC connection.
         */

	if ( dev_rpc_error (ds, clnt_stat, error) == DS_NOTOK )
	{
		return (DS_NOTOK);
	}

	/*
         * Free the variable arguments in the client_raw_data
         * structure, coming from the server.
         */

	xdr_free ((xdrproc_t)xdr_DevVarArgumentArray,
	    (char *)&(client_data.var_argument));


	/*
	 * return error code and status from device server
	 */

	argout->length =(u_int) client_data.xdr_length;

	*error = client_data.error;
	return (client_data.status);
}


/*+**********************************************************************
 Function   :	extern long dev_put_asyn()

 Description:	application interface to execute commands on a device
		with only the possibility to pass input data. 
		The function will return before the command
		is executed. No errors or status of the command
		will be returned.

 Arg(s) In  :	devserver ds       - handle to access the device.
	    :   long cmd           - command to be executed.
            :   DevArgument argin  - pointer to input arguments.
            :   DevType argin_type - data type of input arguments.

 Arg(s) Out :	long *error        - Will contain an appropriate error 
				     code if the corresponding call 
				     returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/


long _DLLFunc dev_put_asyn (devserver ds, long cmd, DevArgument argin,
			    DevType argin_type, long *error )
{
	_server_data		server_data;
	_client_data     	client_data;
	enum clnt_stat		clnt_stat;
	long			client_id = 0;

	*error = 0;

#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\ndev_put_asyn() : entering routine\n");
#endif /* EBUG */

#ifdef TANGO
	if (ds->rpc_protocol == D_IIOP)
	{
/* TODO - implement dev_put_asyn() */
		*error = DevErr_CommandNotImplemented;
		return(DS_NOTOK);
	}
#endif /* TANGO */
	/*
         * Verify the RPC connection.
         */

	if ( dev_rpc_connection (ds, error)  == DS_NOTOK )
	{
		return (DS_NOTOK);
	}

	/*
	 * If the security system is configured, 
	 * verify the security key
	 */

	if ( config_flags.security == True )
	{
		if ( verify_sec_key (ds, &client_id, error) == DS_NOTOK )
		{
			return (DS_NOTOK);
		}
	}

	/*
	 *  fill in data to rpc transfer structures server_data
	 *  and client_data.
	 */

	server_data.ds_id	= ds->ds_id;
	server_data.client_id	= client_id;
	server_data.access_right= ds->dev_access;
	server_data.cmd		= cmd;
	server_data.argin_type	= argin_type;
	server_data.argout_type	= D_VOID_TYPE;
	server_data.argin	= (char *) argin;

	server_data.var_argument.length   = 0;
	server_data.var_argument.sequence = NULL;

#ifdef EBUG
	dev_printdebug (DBG_API,
	    "dev_put_asyn() : server data -> \n");
	dev_printdebug (DBG_API,
	    "ds_id=%d  cmd=%d  intype=%d\n",
	    server_data.ds_id, server_data.cmd,
	    server_data.argin_type);
#endif /* EBUG */

	/*
	 *  call RPC_DEV_PUT_ASYN at the correct device server
	 */

	/*
	 * Call a device with the current version number >1.
   	 */
	if ( ds->vers_number > DEVSERVER_VERS)
	{
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_PUT_ASYN,
		    (xdrproc_t)xdr__server_data, (caddr_t) &server_data,
		    (xdrproc_t)xdr__client_data, (caddr_t) &client_data, TIMEVAL(timeout));
	}
	else
	{
		/*
	    * Call a device from an old version server.
	    */
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_PUT_ASYN,
		    (xdrproc_t)xdr__server_data_3, (caddr_t) &server_data,
		    (xdrproc_t)xdr__client_data_3, (caddr_t) &client_data, TIMEVAL(timeout));
	}

	/*
         * Check for errors on the RPC connection.
         */

	if ( dev_rpc_error (ds, clnt_stat, error) == DS_NOTOK )
	{
		return (DS_NOTOK);
	}

	/*
         * Free the variable arguments in the client_data
         * structure, coming from the server.
         */

	xdr_free ((xdrproc_t)xdr_DevVarArgumentArray,
	    (char *)&(client_data.var_argument));

	*error = client_data.error;
	return (client_data.status);
}


/*+**********************************************************************
 Function   : 	extern long dev_cmd_query()

 Description:	Returns a sequence of structures containig all
		available commands, their names, their input and
		output data types, and type describtions for one
		device.
            :	Commands and data types are read from the command
		list in the device server by calling 
		RPC_DEV_CMD_QUERY.
            :	Command names are read from the command name list,
		defined in DevCmds.h.
            :	Data type describtions have to be specified as 
		CLASS resources as:
            :		CLASS/class_name/cmd_name/IN_TYPE:
            :		CLASS/class_name/cmd_name/OUT_TYPE:

 Arg(s) In  :	ds - client handle for the associated device.

 Arg(s) Out :	varcmdarr - sequence of DevCmdInfo structures.
		error     - Will contain an appropriate error code if the
			    corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc dev_cmd_query (devserver ds, DevVarCmdArray *varcmdarr, long *error)
{
	_dev_query_in	dev_query_in;
	_dev_query_out	dev_query_out;
	enum clnt_stat  clnt_stat;
	char		class_name[SHORT_NAME_SIZE];
	char		cmd_name[SHORT_NAME_SIZE];
	long		length;
	long		ret_stat;
	char		res_path [LONG_NAME_SIZE];
	int		i;
	u_int 		res_tab_size = sizeof(res_tab) / 
	    sizeof(db_resource);
	long		status;

	*error = 0;
#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\ndev_cmd_query() : entering routine\n");
#endif /* EBUG */

#ifdef TANGO
	if (ds->rpc_protocol == D_IIOP)
	{
		status = tango_dev_cmd_query(ds, varcmdarr, error);
		return(status);
	}
#endif /* TANGO */
	/*
         * Verify the RPC connection.
         */

	if ( dev_rpc_connection (ds, error)  == DS_NOTOK )
	{
		return (DS_NOTOK);
	}


	/*
         *  fill in data transfer structures dev_query_in
         *  and dev_query_out.
         */

	dev_query_in.ds_id = ds->ds_id;

	dev_query_in.var_argument.length   = 0;
	dev_query_in.var_argument.sequence = NULL;


	/*
 *  Call the rpc entry point RPC_DEV_CMD_QUERY at the specified device server.
 */

	memset ((char *)&dev_query_out, 0, sizeof (dev_query_out));

	/*
	 * Query a device with the current version number >1.
   	 */
	if ( ds->vers_number > DEVSERVER_VERS)
	{
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_CMD_QUERY,
		    (xdrproc_t)xdr__dev_query_in,  (caddr_t) &dev_query_in,
		    (xdrproc_t)xdr__dev_query_out, (caddr_t) &dev_query_out, TIMEVAL(timeout));
	}
	else
	{
		/*
	    * Query a device from an old version server.
	    */
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_CMD_QUERY,
		    (xdrproc_t)xdr__dev_query_in_3,  (caddr_t) &dev_query_in,
		    (xdrproc_t)xdr__dev_query_out_3, (caddr_t) &dev_query_out, TIMEVAL(timeout));
	}


	/*
         * Check for errors on the RPC connection.
         */

	if ( dev_rpc_error (ds, clnt_stat, error) == DS_NOTOK )
	{
		return (DS_NOTOK);
	}

	/*
         * Free the variable arguments in the dev_query_out
         * structure, coming from the server.
         */

	xdr_free ((xdrproc_t)xdr_DevVarArgumentArray,
	    (char *)&(dev_query_out.var_argument));


	/*
 * Allocate memory for a sequence of DevCmdInfo structures
 * returned with varcmdarr.
 */

	varcmdarr->length   = dev_query_out.length;
	varcmdarr->sequence = (DevCmdInfo *) malloc
	    (varcmdarr->length * sizeof (DevCmdInfo));
	if ( varcmdarr->sequence == NULL )
	{
		*error  = DevErr_InsufficientMemory;
		return (-1);
	}
	memset ((char *)varcmdarr->sequence, 0,
	    (varcmdarr->length * sizeof (DevCmdInfo)));

	/*
 * Now get command and types name strings for the returned
 * command sequence. Command names are retrieved from the
 * global command-name-list and name strings for the data types
 * are searched in the resource CLASS table of the object class.
 * 
 * Undefined names will be initialised with NULL.
 */

	for ( i=0; (u_long)i<varcmdarr->length; i++ )
	{
		/*
	    * initialise varcmdarr->sequence[i] with command and
	    * argument types, returned with dev_query_out from the
	    * device servers command list.
	    */

		varcmdarr->sequence[i].cmd      = dev_query_out.sequence[i].cmd;
		varcmdarr->sequence[i].in_type  = dev_query_out.sequence[i].in_type;
		varcmdarr->sequence[i].out_type = dev_query_out.sequence[i].out_type;

		/*
	    * get command name string from the resource database
	    */

		if ( (ret_stat = get_cmd_string (ds, varcmdarr->sequence[i].cmd, 
		    varcmdarr->sequence[i].cmd_name, 
		    error )) == DS_NOTOK )
		{
			/*
	       * An error will be only returned if the database
	       * access fails.
	       */
			return (DS_NOTOK);
		}


		/*
	    *  Check wether command name was found.
	    *  If the name was not found, get_cmd_string() returns
	    *  DS_WARNING.
	    */

		if ( ret_stat != DS_WARNING )
		{
			/*
	       * Limit the class_name and the command_name
	       * strings to 19 characters. This is the limit
	       * of the static database name fields.
	       */

			length = strlen (dev_query_out.class_name);
			if ( length > MAX_RESOURCE_FIELD_LENGTH )
				length = MAX_RESOURCE_FIELD_LENGTH;
			strncpy (class_name, dev_query_out.class_name, 
			    MAX_RESOURCE_FIELD_LENGTH);
			class_name[(_Int)length] = '\0';

			length = strlen (varcmdarr->sequence[i].cmd_name);
			if ( length > MAX_RESOURCE_FIELD_LENGTH )
				length = MAX_RESOURCE_FIELD_LENGTH;
			strncpy (cmd_name, varcmdarr->sequence[i].cmd_name,
			    MAX_RESOURCE_FIELD_LENGTH);
			cmd_name[(_Int)length] = '\0';

/*
 *  setup resource path to read information about
 *  data types from the CLASS resource table.
 *
 * but first check to see whether the device belongs to another
 * nethost domain i.e. i_nethost != 0
 */

			if (ds->i_nethost > 0)
			{
				sprintf(res_path, "//%s/CLASS/%s/%s",
				   get_nethost_by_index(ds->i_nethost, error),
				   class_name, cmd_name);
			}
/*
 * use default nethost
 */
			else
			{
				sprintf (res_path, "CLASS/%s/%s", 
				   class_name, cmd_name);
			}

			/*
	       *  read CLASS resources from database
	       */

			res_tab[0].resource_adr = &(varcmdarr->sequence[i].in_name);
			res_tab[1].resource_adr = &(varcmdarr->sequence[i].out_name);

			if (db_getresource (res_path, res_tab, res_tab_size, error) < 0)
			{
				return (-1);
			}
		}
	}

	/*
	 *  free dev_query_out 
	 */

	xdr_free ((xdrproc_t)xdr__dev_query_out,
	    (char *)&dev_query_out);

	/*
 * Return error code and status from device server.
 */

	*error = dev_query_out.error;
	return (dev_query_out.status);
}


/*+**********************************************************************
 Function   :	static long get_cmd_string()

 Description:   Read the command name as a string from the
		resource database.
		The rsource name is:
		CMD/team_no/server_no/cmd_ident:
		DS_WARNING is returned, if the function was
		executed correctly, but no command name
		string was found in the database.

 Arg(s) In  :   long cmd      - error number

 Arg(s) Out :   char *cmd_str - command name as a string.
		long *error   - Will contain an appropriate error
			        code if the corresponding call
		    	        returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK or DS_WARNING
***********************************************************************-*/

static long get_cmd_string (devserver ds, long cmd, char *cmd_str, long *error)
{
	char		res_path[LONG_NAME_SIZE];
	char		res_name[SHORT_NAME_SIZE];
	char		*ret_str = NULL;
	db_resource 	res_tab;
	unsigned long 	cmd_number_mask = 0x3ffff;
	unsigned short 	team;
	unsigned short 	server;
	unsigned short 	cmds_ident;

#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\nget_cmd_string() : entering routine\n");
#endif /* EBUG */

	*error = 0;

	/*
 * Decode the command nuber into the fields:
 * team, server and cmds_ident.
 */
	team   = (_Int)(cmd >> DS_TEAM_SHIFT);
	team   = team & DS_TEAM_MASK;
	server = (_Int)(cmd >> DS_IDENT_SHIFT);
	server = server & DS_IDENT_MASK;
	cmds_ident = (_Int)(cmd & cmd_number_mask);

	/*
 * Create the resource path and the resource structure.
 *
 * first check to see whether the device belongs to another
 * nethost domain i.e. i_nethost != 0
 */

	if (ds->i_nethost > 0)
	{
		sprintf(res_path, "//%s/CMDS/%d/%d", 
		   get_nethost_by_index(ds->i_nethost, error),
		   team, server);
	}
/*
 * use default nethost
 */
	else
	{
		sprintf(res_path, "CMDS/%d/%d", team, server);
	}

	sprintf (res_name, "%d", cmds_ident);
#ifdef EBUG
	dev_printdebug (DBG_API,
	    "get_cmds_string() : res_path = %s\n", res_path);
	dev_printdebug (DBG_API,
	    "get_cmds_string() : res_name = %s\n", res_name);
#endif /* EBUG */

	res_tab.resource_name = res_name;
	res_tab.resource_type = D_STRING_TYPE;
	res_tab.resource_adr  = &ret_str;

	/*
 * Read the command name string from the database.
 */

	if (db_getresource (res_path, &res_tab, 1, error) == DS_NOTOK)
	{
#ifdef EBUG
		dev_printdebug (DBG_API | DBG_ERROR,
		    "get_cmd_string() : db_getresource failed with error %d\n", *error);
#endif /* EBUG */

		return (DS_NOTOK);
	}

/*
 * If the variable ret_str is still NULL, no resource value was found
 * in the database, but the function was executed without error.
 * In this case return the value DS_WARNING.
 */
	if ( ret_str == NULL )
	{
		return (DS_WARNING);
	}

	sprintf (cmd_str, "%s", ret_str);
	free (ret_str);
	return (DS_OK);
}


/*+**********************************************************************
 Function   :	extern long dev_inform()

 Description:	Returns to the user a structure of device
		information for every specified device client handle.
            :   The information structure contains:
		- the name of the device,
		- the class name,
		- the device type,
		- the device server name,
		- the host name of the device server
 
 Arg(s) In  :	devserver *clnt_handles - pointer to a table of client handles.
	    :   long num_devices        - number of devices.

 Arg(s) Out :	DevInfo **dev_info      - pinter to information structures.
            :	error - Will contain an appropriate error code if the
		        corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc dev_inform (devserver *clnt_handles, long num_devices,
			  DevInfo * *dev_info, long *error)
{
	devserver	*clnt_ptr;
	DevInfo		*info_ptr;
	long 		i;


	*error = 0;

#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\ndev_inform() : entering routine\n");
#endif /* EBUG */

/*
 * Allocate memory for the array of information structures
 * only if the pointer "dev_info" is initialised to NULL.
 */
	if (*dev_info == NULL)
	{
		*dev_info = (DevInfo *) malloc (num_devices * sizeof(DevInfo));
		if (*dev_info == NULL)
		{
			*error  = DevErr_InsufficientMemory;
			return (DS_NOTOK);
		}
	}

	/* 
 * Read information from the client handles and store it in
 * the information structures.
 */

	clnt_ptr = (devserver *) clnt_handles;
	info_ptr = (DevInfo *)   *dev_info;

	for (i=0; i<num_devices; i++, clnt_ptr++, info_ptr++)
	{
		sprintf (info_ptr->device_name,  "%s", (*clnt_ptr)->device_name);
		sprintf (info_ptr->device_class, "%s", (*clnt_ptr)->device_class);
		sprintf (info_ptr->device_type,  "%s", (*clnt_ptr)->device_type);
		sprintf (info_ptr->server_name,  "%s", (*clnt_ptr)->server_name);
		sprintf (info_ptr->server_host,  "%s", (*clnt_ptr)->server_host);

#ifdef EBUG
		dev_printdebug (DBG_API,
		    "dev_inform() : information data\n");
		dev_printdebug (DBG_API,
		    "name  = %s\n",info_ptr->device_name);
		dev_printdebug (DBG_API,
		    "class  = %s\n",info_ptr->device_class);
		dev_printdebug (DBG_API,
		    "type  = %s\n",info_ptr->device_type);
		dev_printdebug (DBG_API,
		    "svc_name = %s\n",info_ptr->server_name);
		dev_printdebug (DBG_API,
		    "svc_host = %s\n",info_ptr->server_host);
#endif /* EBUG */
	}

	return (DS_OK);
}

/*+**********************************************************************
 Function   :	long dev_rpc_timeout()

 Description:	Sets or reads the timeout for an UDP connection to
		a server.
		A request to set the timeout has to be asked
		with CLSET_TIMEOUT. The timeout will be set
		without any retry.
		A request to read the timeout has to be asked with
		CLGET_TIMEOUT.

 Arg(s) In  :	devserver ds           - handle to device.
	    :	long request	       - indicates whether the timeout
					 should be set or only read.
	    :	struct timeval dev_timeout - timeout structure.

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc dev_rpc_timeout (devserver ds, long request, 
			       struct timeval *dev_timeout, long *error)
{
	*error = 0;

#ifdef TANGO
	if (ds->rpc_protocol == D_IIOP)
	{
/* TODO - implement dev_rpc_timeout() */
		return(DS_OK);
	}
#endif /* TANGO */
/*
 * in the stateless import case the device might not be imported yet
 * and therefore has not client handle. In this case if the client
 * wants to set the timeout (CLSET_TIMEOUT) save the timeout for later 
 * when the device is imported by rpc_reinstall_connection(). If the 
 * client wants to get the timeout (CLGET_TIMEOUT) then return the 
 * default timeout (api_timeout) or requested one if there is one.
 */
	if (ds->clnt == NULL)
	{
		switch (request)
		{
			case (CLSET_TIMEOUT) : 
				ds->rpc_timeout.tv_sec = dev_timeout->tv_sec;
				ds->rpc_timeout.tv_usec = dev_timeout->tv_usec;
				break;

			case (CLGET_TIMEOUT) :
				if ((dev_timeout->tv_sec == 0) &&
				    (dev_timeout->tv_usec == 0))
				{
					dev_timeout->tv_sec = api_timeout.tv_sec;
					dev_timeout->tv_usec = api_timeout.tv_usec;
				}
				else
				{
					dev_timeout->tv_sec = ds->rpc_timeout.tv_sec;
					dev_timeout->tv_usec = ds->rpc_timeout.tv_usec;
				}
				break;
				
        		default:
                		*error = DevErr_UnknownInputParameter;
                		return (DS_NOTOK);

		}
		return(DS_OK);
	}
		
	/*
	 * to be sure to work with the correct client handle
	 * initialise the pointer again.
	 */

	ds->clnt = svr_conns[ds->no_svr_conn].clnt;

	switch (request)
	{
		/*
	    * Set the timeout for the connection.
	    */
	case CLSET_TIMEOUT:
		clnt_control (ds->clnt, CLSET_RETRY_TIMEOUT, (char *) dev_timeout);
		clnt_control (ds->clnt, CLSET_TIMEOUT, (char *) dev_timeout);
		break;

		/*
	    * Set the timeout of the connection.
	    */
	case CLGET_TIMEOUT:
		clnt_control (ds->clnt, CLGET_RETRY_TIMEOUT, (char *) dev_timeout);
		clnt_control (ds->clnt, CLGET_TIMEOUT, (char *) dev_timeout);
		break;

	default:
		*error = DevErr_UnknownInputParameter;
		return (DS_NOTOK);
	}

	return (DS_OK);
}


#if defined(_NT)
#if defined (__STDC__)
size_t _DLLFunc msize(void* pointer, int* error) 
#else
/**/
size_t msize(pointer, error) 
/*+**********************************************************************
 Function   :	size_t msize()

 Description:	walks the heap until allocated block pointed to
      by pointer id found and checks if heap is still OK.

 Arg(s) In  :	void* pointer    - pointer to allocated block on heap.

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a zero value.

 Return(s)  :	size of allocated block on heap
***********************************************************************-*/
void* pointer;
int*  error;
#endif /* __STDC__ */
{
   struct _heapinfo info;  /* Heap info structure */
   int    heap_status;     /* Heap status */
   size_t ret_val;         /* msize() return value */

   info._pentry= NULL;     /* Initialize heap info structure */

   do {
      heap_status= _heapwalk(&info);   /* Step through heap until */
   } while ((pointer != info._pentry)  /* sesired entry is found */
     && (heap_status == _HEAPOK));     /* or a heap error occurs */

   if(heap_status == _HEAPOK)
      if(info._useflag != _FREEENTRY)
         ret_val= info._size;          /* no errors */
      else {
         ret_val= 0;                   /* Entry has been freed */
         *error= _FREEENTRY;
      }
   else {
      ret_val= 0;                      /* A heap error occurred */
      *error= heap_status;
   }
   return(ret_val);
}
#endif   /* _NT */

/*+**********************************************************************
 Function   :	long get_i_nethost_by_device_name()

 Description:	Get the index for the nethost from the device
		name. The nethost is specified in the device name
		as "//nethost/domain/family/member". If the nethost
		is not specified then return the index for the
		default nethost (0). If the nethost is specified
		but not imported then return -1.

 Arg(s) In  :	char *device_name - device name

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc get_i_nethost_by_device_name (char *device_name, long *error)
{
	long i_nethost, i;
	char *nethost;
	

/*
 * is the nethost specified ?
 */

	if (strncmp(device_name,"//",2) == 0)
	{
/*
 * make a copy of the nethost name without the "//" and
 * removing the device name which follows
 */
		nethost = (char*)malloc(strlen(device_name)+1);
		strcpy(nethost,device_name+2);
		for (i=0; i<(int)strlen(nethost); i++)
		{
			if (nethost[i] == '/')
			{
				nethost[i] = 0;
				break;
			}
		}
/*
 * check to see whether the nethost has been imported
 */
	
		if ((i_nethost = get_i_nethost_by_name(nethost,error)) < 0)
/*
 * the specified nethost is not in the list of imported nethosts,
 * (config_setup_multi() should be called to add it)
 */
		{
			i_nethost = -1;
		}
		free(nethost);
	}
/*
 * nethost not specified in the device name, assume the nethost of
 * the control system by default (i_nethost = 0)
 */
	else
	{
		i_nethost = 0;
	}

	return(i_nethost);
}

/*+**********************************************************************
 Function   :	long get_i_nethost_by_name()

 Description:	Get the index for the nethost from the nethost
		name. 

 Arg(s) In  :	char *device_name - device name

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc get_i_nethost_by_name (char *nethost, long *error)
{
	long i;
	
/*
 * loop through the array of imported nethosts looking for the 
 * specified nethost
 */

	for (i=0; i<max_nethost; i++)
	{
/*
 * first check to see whether this nethost has been imported
 */
		if( (multi_nethost[i].config_flags.configuration == True) &&
		    (multi_nethost[i].nethost != NULL))
		{
			if (strcmp(nethost, multi_nethost[i].nethost) == 0)
			{
				return(i);
			}
		}
	}
	
	return(-1);
}


/*+**********************************************************************
 Function   :	long get_i_nethost_by_name()

 Description:	Get the nethost name associated with this index

 Arg(s) In  :	long i_nethost - nethost index

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

char* _DLLFunc get_nethost_by_index (long i_nethost, long *error)
{
	char *nethost;
	

/*
 * first check to see whether the requested nethost has been imported
 */
	if (multi_nethost[i_nethost].config_flags.configuration)
	{
		nethost = multi_nethost[i_nethost].nethost;
	}
	else
	{
		nethost = NULL;
	}

	return(nethost);
}


/*+**********************************************************************
 Function   :	long extract_device_name()

 Description:	Extract the domain/family/member part of name from the 
		full device name "//nethost/domain/family/member" ,
		return a pointer

 Arg(s) In  :	char *full_name - device name

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a non-zero value.

 Return(s)  :	pointer to string containing only "domain/family/member"
***********************************************************************-*/
	
char* _DLLFunc extract_device_name (char *full_name, long *error)
{
	char *device_name;
	
/*
 * assume full_name == device_name to start off with
 */
	device_name = full_name;
/*
 * if nethost is specified in the device name "remove" it
 */
	if (strncmp(device_name,"//",2) == 0)
	{
		for (device_name +=2; *device_name != '/'; device_name++)
		{ 
			/* do nothing */ 
		};
		device_name++;
	}

	return(device_name);
}

/*
 * global arrays required for multi-nethost support, memory is
 * allocated for them by nethost_alloc()
 */
extern struct _devserver *msg_ds;
extern struct _devserver *db_ds;
extern short *auth_flag;

/*+**********************************************************************
 Function   :	long nethost_alloc()

 Description:	alloc place for MIN_NETHOST to global array which contains
		list of nethosts

 Arg(s) In  :	none

 Arg(s) Out :	error  - Will contain an appropriate error code if the
		         corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc nethost_alloc (long *error)
{
	static long first=1;
	long i, new_max_nethost;

/*
 * first time round allocate space for only one NETHOST, because
 * there is a high probability that the client only ones to work
 * with one control system i.e. $NETHOST
 */
	if (first)
	{
		multi_nethost = (nethost_info*)malloc((sizeof(nethost_info)));
		msg_ds = (struct _devserver*)malloc((sizeof(struct _devserver)));
		db_ds = (struct _devserver*)malloc((sizeof(struct _devserver)));
		auth_flag = (short*)malloc((sizeof(short)));
		first = 0;
		new_max_nethost = 1;
	}
/*
 * if more the one NETHOST is being used allocate space for an additional
 * MIN_NETHOST's at each call to avoid calling realloc() too often
 */
	else
	{
		multi_nethost = (nethost_info*)realloc(multi_nethost,(sizeof(nethost_info))*(max_nethost+MIN_NETHOST));
		msg_ds = (struct _devserver*)realloc(msg_ds,(sizeof(struct _devserver))*(max_nethost+MIN_NETHOST));
		db_ds = (struct _devserver*)realloc(db_ds,(sizeof(struct _devserver))*(max_nethost+MIN_NETHOST));
		auth_flag = (short*)realloc(auth_flag,(sizeof(short))*(max_nethost+MIN_NETHOST));
		new_max_nethost = max_nethost + MIN_NETHOST;
	}
/* 
 * update db_info and msg_info pointers to take into account realloc() of db_ds and msg_ds
 * which could have moved in memory
 */
	for (i=0; i<max_nethost; i++)
	{
		multi_nethost[i].db_info = (devserver)&db_ds[i];
		multi_nethost[i].msg_info = (devserver)&msg_ds[i];
	}
/* 
 * set new entries to False to indicate that they are free to be used
 */
	for (i=max_nethost; i<new_max_nethost; i++)
	{
		multi_nethost[i].config_flags.configuration = False;
/*
 * initialise all flags to zero so that no-one (e.g. ET) tries to use
 * the database (or any other server) as long as it is not imported
 */
		memset((char*)&multi_nethost[i].config_flags, 0, sizeof(multi_nethost[i].config_flags));
		auth_flag[i] = False;
	}
	max_nethost = new_max_nethost;
#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
			"nethost_alloc(): (re)allocate space for nethost, new max = %d\n",
			 max_nethost);
#endif /* EBUG */

	return(DS_OK);
}

/*+**********************************************************************
 Function   : 	extern long dev_ping()

 Description:	Pings the device server to find out if this device
		is being served. 

 Arg(s) In  :	ds - client handle for the associated device.

 Arg(s) Out : error     - Will contain an appropriate error code if the
			    corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc dev_ping (devserver ds, long *error)
{
	_dev_import_in	dev_import_in;
	_dev_import_out	dev_import_out;
	enum clnt_stat  clnt_stat;

	*error = 0;
#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\ndev_ping() : entering routine\n");
#endif /* EBUG */

#ifdef TANGO
        if (ds->rpc_protocol == D_IIOP)
        {
		tango_dev_ping(ds, error);
                return(DS_OK);
        }
#endif /* TANGO */
	/*
         * Verify the RPC connection.
         */

	if ( dev_rpc_connection (ds, error)  == DS_NOTOK )
	{
		return (DS_NOTOK);
	}


	/*
         *  fill in data transfer structures dev_import_in
         *  and dev_import_out.
         */

	dev_import_in.device_name = ds->device_name;
	dev_import_in.access_right = 0;
	dev_import_in.client_id = 0;
	dev_import_in.connection_id = 0;

	dev_import_in.var_argument.length   = 0;
	dev_import_in.var_argument.sequence = NULL;


/*
 *  Call the rpc entry point RPC_DEV_PING at the specified device server.
 */

	memset ((char *)&dev_import_out, 0, sizeof (dev_import_out));

	/*
	 * Query a device with the current version number >1.
   	 */
	if ( ds->vers_number > DEVSERVER_VERS)
	{
		clnt_stat = clnt_call (ds->clnt, RPC_DEV_PING,
		    (xdrproc_t)xdr__dev_import_in,  (caddr_t) &dev_import_in,
		    (xdrproc_t)xdr__dev_import_out, (caddr_t) &dev_import_out, TIMEVAL(timeout));
	}
	else
	{
 	   /*
	    * An old device does not support ping !
	    */
		*error = DevErr_CommandNotImplemented;;
		return(DS_NOTOK);
	}


	/*
         * Check for errors on the RPC connection.
         */

	if ( dev_rpc_error (ds, clnt_stat, error) == DS_NOTOK )
	{
		return (DS_NOTOK);
	}
	/*
	 *  free dev_import_out 
	 */

	xdr_free ((xdrproc_t)xdr__dev_import_out,
	    (char *)&dev_import_out);

	/*
         * Free the variable arguments in the dev_import_out
         * structure, coming from the server.
         */

	xdr_free ((xdrproc_t)xdr_DevVarArgumentArray,
	    (char *)&(dev_import_out.var_argument));


/*
 * Return error code and status from device server.
 */

	*error = dev_import_out.error;
	return (dev_import_out.status);
}
