static char RcsId[] = "@(#)$Header: tango_api.cpp,v 8.14 2000/06/22 14:50:24 goetz Rel $";
/*+*******************************************************************

 File       :	tango_api.c

 Project    :	Interfacing TACO (based on SUN/RPC) to TANGO (based on CORBA)

 Description:	TACO to TANGO Application Programmer's Interface

		Source code to implement the taco interface to tango via
		the Device Server C API (DSAPI) using the TANGO C++ interface. 
		This means it will be possible to use the TACO dev_putget() 
		call from TACO programs which have been linked with C++
		to talk to TANGO device servers.
	
 Author(s)  :	Andy Goetz

 Original   :	December 1999

 $Revision: 8.14 $
 $Date: 2000/06/22 14:50:24 $

 $Author: goetz $

 $Log:	tango_api.cpp,v $
// Revision 8.14  2000/06/22  14:50:24  14:50:24  goetz (Andrew GOETZ)
// fixed bug in dev_event_listen() which caused server to crash
// 
 Revision 8.13  2000/06/02 21:56:42  goetz
 _DEVICE_H now _TANGO_DEVICE_H in Device.H; (char*)taco_tango

 Revision 8.12  2000/06/02 15:58:01  goetz
 dev_event_fire() now device specific; ported to SuSE V6.4

 Revision 8.11  2000/05/31 07:47:47  goetz
 tango_api has local copies of argc and argv, ported to HP-UX

// Revision 8.10  2000/05/30  10:16:33  10:16:33  goetz (Andrew GOETZ)
// tango_api.cpp now has same version as other DSAPI files (V8.10)
// 
// Revision 8.0  99/12/28  14:18:21  14:18:21  goetz (Andrew GOETZ)
// added TANGO support for TACO dev_xxx() calls via -DTANGO for C++
// 



 Copyleft (c) 1999 by European Synchrotron Radiation Facility,
                      Grenoble, France

********************************************************************-*/

#include <tango_ds.h>
#include <API.h>
#include <ApiP.h>
#include <stdio.h>
#include <assert.h>
#include <iostream.h>
#include <vector>
#include <string>
#ifdef linux
using namespace std;
#endif

#define D_IIOP	1111		/* TANGO native protocol - should be in API.h */

static long get_cmd_value (char*, char*, long*, long *error);
static long tango_to_taco_type(long );
static CORBA_Any tango_argin_to_any(long, long, void *, long *);    
static void tango_any_to_argout(long, long, CORBA_Any, void *, long *);    
static void tango_any_to_argout_raw(long, long, CORBA_Any, void *, long *);    


/*
 * TANGO/CORBA globals with file scope
 */

static CORBA_ORB *orb;
static Tango_Device *tango_dbase;
static long tango_dbase_init=0;
typedef struct _tango_device {
	long flag; 
	vector<string> cmd_name;
	vector<long> cmd_value;
	vector<long> argin_type;
	vector<long> argout_type;
	Tango_Device_var object;} _tango_device;
static vector<_tango_device> tango_device;
static long n_tango_device=0;

/*+**********************************************************************
 Function   :   extern long tango_db_import()

 Description:   import TANGO database

 Arg(s) In  :   char *db_name - name of the database
            :   char *db_host - database host
            :   long db_port  - database port

 Arg(s) Out :   none

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long tango_db_import(char *db_name, char *db_host, long db_port, long *error)
{

	cout4 << "tango_db_import(): called name = " << db_name << ", host = " << db_host << ", port = " << db_port << endl; 

	if (db_name == NULL) {
		*error = DevErr_DbImportFailed;
		return(-1);
	}

// Connect to TANGO database

	try {
		int _argc;
		char **_argv;

// Pass dummy arguments to init() because we don't have access to
// argc and argv
		_argc = 1;
		_argv = (char**)malloc(sizeof(char*));
		_argv[0] = (char*)"taco_tango";

		orb = CORBA_ORB_init(_argc, _argv);
	
// Try to connect to server using the get_inet_object call

		CORBA_Object_var obj;
		obj = orb->get_inet_object(db_host,db_port,db_name);
		tango_dbase = Tango_Device::_narrow(obj);
	}
	catch (CORBA_Exception  &e) {
		TangoUtil::print_exception(e); 
		*error = DevErr_DbImportFailed;
		return(DS_NOTOK);
	}
	
	tango_dbase_init = 1;

	return(DS_OK);
}

/*+**********************************************************************
 Function   :   extern long tango_dev_import()

 Description:   import TANGO device

 Arg(s) In  :   char *dev_name - name of device

 Arg(s) Out :   none

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long tango_dev_import(char *dev_name, long access, devserver *ds_ptr, long *error)
{

	_tango_device tango_device_tmp;

	cout4 << "tango_dev_import(): called name = " << dev_name  << endl; 

	if (!tango_dbase_init) 
	{
		try
		{
			tango_db_import("database","amber1",10000,error);
        	}
		catch (CORBA_Exception  &e)
		{
                	TangoUtil::print_exception(e);
			*error = DevErr_DbImportFailed;
                	return(DS_NOTOK);
        	}
	}

	if (dev_name == NULL) 
	{
		*error = DevErr_DeviceOfThisNameNotServed;
		return(DS_NOTOK);
	}

// Connect to TANGO database

	try 
	{
                CORBA_Any send;
                send <<= dev_name;

                CORBA_Any_var received = tango_dbase->command_inout("DbImportDevice",send);

                Tango_DevVarLongStringArray *dev_info;
                received >>= dev_info;

		cout4 << "Device name : " << (dev_info->svalue)[0] << endl;
		cout4 << "Device IOR : " << (dev_info->svalue)[1] << endl;

                if ((dev_info->lvalue)[0] == 0)
                {
			*error = DevErr_DeviceNotImportedYet;
                        return(DS_NOTOK);
                }
                CORBA_Object_var obj = orb->string_to_object((dev_info->svalue)[1]
);

                tango_device_tmp.object = Tango_Device::_narrow(obj);
                tango_device_tmp.flag = 1;
        }
        catch (CORBA_Exception  &e)
	{
                TangoUtil::print_exception(e);
		*error = DevErr_DeviceNotImportedYet;
                return(DS_NOTOK);
        }

        if (CORBA_is_nil(tango_device_tmp.object))
        {
		*error = DevErr_DeviceNotImportedYet;
                return(DS_NOTOK);
        }
        cout4 << "Connected to device" << endl;
//
// initialise command list for device
//
	try
	{
		Tango_DevCmdInfoList_var cmd_query = tango_device_tmp.object->command_list_query();
		Tango_DevInfo_var dev_info = tango_device_tmp.object->info();

		cout4 << "Number of command(s) : " << cmd_query->length() << endl;
		tango_device_tmp.cmd_name = vector<string>(cmd_query->length());
// Q: why can't I create a vector with an initialising value ?
		tango_device_tmp.cmd_value = vector<long>(cmd_query->length());
		tango_device_tmp.argin_type = vector<long>(cmd_query->length());
		tango_device_tmp.argout_type = vector<long>(cmd_query->length());
		for (long i = 0;i < cmd_query->length();i++)
		{
			long error;
			tango_device_tmp.cmd_name[i] = string((*cmd_query)[i].cmd_name);
			cout4 << "Command name : " << tango_device_tmp.cmd_name[i].c_str();
			get_cmd_value((char*)dev_info->dev_class,
			              (char*)tango_device_tmp.cmd_name[i].c_str(),
			              &tango_device_tmp.cmd_value[i],
			              &error);
			cout4 << "value " << tango_device_tmp.cmd_value[i];
			tango_device_tmp.argin_type[i] = (*cmd_query)[i].in_type;
			cout4 << " ->Input type : " << Tango_CmdArgTypeName[tango_device_tmp.argin_type[i]];
			tango_device_tmp.argout_type[i] = (*cmd_query)[i].out_type;
			cout4 << " ->Output type : " << Tango_CmdArgTypeName[tango_device_tmp.argout_type[i]] << endl;
		}
        	if ((*ds_ptr = (devserver)malloc(sizeof(struct _devserver))) == NULL)
        	{
                	*error = DevErr_InsufficientMemory;
                	return(DS_NOTOK);
        	}
		tango_device.push_back(tango_device_tmp);
        	sprintf((*ds_ptr)->device_name,"%s",dev_name);
        	sprintf((*ds_ptr)->device_class,"%s",(char*)dev_info->dev_class);
        	sprintf((*ds_ptr)->device_type,"TANGO_Device");
        	sprintf((*ds_ptr)->server_name,"Unknown");
        	sprintf((*ds_ptr)->server_host,"Unknown");
        	(*ds_ptr)->clnt             = NULL;
        	(*ds_ptr)->ds_id            = n_tango_device;
        	(*ds_ptr)->no_svr_conn      = 0;
        	(*ds_ptr)->rpc_conn_counter = 0;
        	(*ds_ptr)->dev_access       = access;
        	(*ds_ptr)->i_nethost        = 0;
        	(*ds_ptr)->rpc_protocol     = D_IIOP;
        	(*ds_ptr)->rpc_timeout.tv_sec  = 0;
        	(*ds_ptr)->rpc_timeout.tv_usec = 0;

	}
        catch (CORBA_Exception  &e)
        {
                TangoUtil::print_exception(e);
		*error = DevErr_DeviceNotImportedYet;
                return(DS_NOTOK);
        }
	n_tango_device++;

	return(0);
}

/*+**********************************************************************
 Function   :   extern long tango_dev_putget()

 Description:   execute a command on a TANGO device

 Arg(s) In  :   long id - device id
		long cmd - command to execute
		void *argin - pointer to argin
		long argin_type - argin type
		void *argout - pointer to argout
		long argout_type - argout type

 Arg(s) Out :   long *error - error code, in case of failure

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long tango_dev_putget(devserver ds, long cmd, void *argin, long argin_type,
                      void *argout, long argout_type, long *error)
{

	long i_cmd = -1, dev_id;
	char *cmd_name = NULL;

	cout4 << "tango_dev_putget(): called device id = " << dev_id  << " command = " << cmd << endl; 

	dev_id = ds->ds_id;
//
// check if command has already been executed, if not then
// read command string using its value from TACO database
//
// Q: why can't I use find() in a vector to return the index of the element ?
	for (long i=0; i<tango_device[dev_id].cmd_value.size(); i++)
	{
		if (tango_device[dev_id].cmd_value[i] == cmd)
		{
			cout4 << "tango_dev_putget(): found command at position "<< i;
			cout4 << " argin_type " << tango_device[dev_id].argin_type[i] ;
			cout4 << " argout_type " << tango_device[dev_id].argout_type[i] << endl;
			i_cmd = i;
			break;
		}
	}
	if (i_cmd == -1)
	{
		*error = DevErr_CommandNotImplemented;
		return(DS_NOTOK);
	}

	try 
	{

// insert input argument into Any type

		CORBA_Any send = tango_argin_to_any(argin_type, tango_device[dev_id].argin_type[i_cmd],
		                                    argin, error);

// execute command using command_inout() method

                CORBA_Any_var received = tango_device[dev_id].object->command_inout(tango_device[dev_id].cmd_name[i_cmd].c_str(),send);

// extract output argument from Any

		tango_any_to_argout(argout_type, tango_device[dev_id].argout_type[i_cmd],
		                    received, argout, error);
        }
        catch (CORBA_Exception  &e)                                                     {
                TangoUtil::print_exception(e);
// TODO - improve TANGO to TACO error handling here
		*error = DevErr_CommandFailed;
                return(DS_NOTOK);
        }

	return(0);
}

/*+**********************************************************************
 Function   :   extern long tango_dev_putget_raw()

 Description:   execute a command on a TANGO device and return the data in 
		DEV_OPAQUE_TYPE

 Arg(s) In  :   long id - device id
		long cmd - command to execute
		void *argin - pointer to argin
		long argin_type - argin type
		void *argout - pointer to argout
		long argout_type - argout type

 Arg(s) Out :   long *error - error code, in case of failure

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long tango_dev_putget_raw(devserver ds, long cmd, void *argin, long argin_type,
                      void *argout, long argout_type, long *error)
{

	long i_cmd = -1, dev_id;
	char *cmd_name = NULL;

	cout4 << "tango_dev_putget(): called device id = " << dev_id  << " command = " << cmd << endl; 

	dev_id = ds->ds_id;
//
// check if command has already been executed, if not then
// read command string using its value from TACO database
//
// Q: why can't I use find() in a vector to return the index of the element ?
	for (long i=0; i<tango_device[dev_id].cmd_value.size(); i++)
	{
		if (tango_device[dev_id].cmd_value[i] == cmd)
		{
			cout4 << "tango_dev_putget(): found command at position "<< i;
			cout4 << " argin_type " << tango_device[dev_id].argin_type[i] ;
			cout4 << " argout_type " << tango_device[dev_id].argout_type[i] << endl;
			i_cmd = i;
			break;
		}
	}
	if (i_cmd == -1)
	{
		*error = DevErr_CommandNotImplemented;
		return(DS_NOTOK);
	}

	try 
	{

// insert input argument into Any type

		CORBA_Any send = tango_argin_to_any(argin_type, tango_device[dev_id].argin_type[i_cmd],
		                                    argin, error);

// execute command using command_inout() method

                CORBA_Any_var received = tango_device[dev_id].object->command_inout(tango_device[dev_id].cmd_name[i_cmd].c_str(),send);

// extract output argument from Any

		tango_any_to_argout_raw(argout_type, tango_device[dev_id].argout_type[i_cmd],
		                    received, argout, error);
        }
        catch (CORBA_Exception  &e)                                                     {
                TangoUtil::print_exception(e);
// TODO - improve TANGO to TACO error handling here
		*error = DevErr_CommandFailed;
                return(DS_NOTOK);
        }

	return(0);
}


/*+**********************************************************************
Function   : 	extern long tango_dev_cmd_query()

 Description:	Returns a sequence of structures containing all
		available commands, names, input and output data types
		for a TANGO device. 
            :	Command names and data types are read from the command
		list in the device server by calling device->cmd_query.
            :	Command values are read from the TACO database
            :	Data type descriptions have to be specified as 
		CLASS resources as:
            :		CLASS/class_name/cmd_name/IN_TYPE:
            :		CLASS/class_name/cmd_name/OUT_TYPE:

 Arg(s) In  :	dev_id - device id in tango_device

 Arg(s) Out :	varcmdarr - sequence of DevCmdInfo structures.
		error     - Will contain an appropriate error code if the
			    corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long tango_dev_cmd_query (devserver ds, DevVarCmdArray *varcmdarr, long *error)
{
	long dev_id;

	dev_id = ds->ds_id;

	*error = 0;

/*
 * Allocate memory for a sequence of DevCmdInfo structures
 * returned with varcmdarr.
 */

	varcmdarr->length   = tango_device[dev_id].cmd_value.size();
	varcmdarr->sequence = (DevCmdInfo*)malloc(varcmdarr->length * sizeof (DevCmdInfo));
	if ( varcmdarr->sequence == NULL )
	{
		*error  = DevErr_InsufficientMemory;
		return (DS_NOTOK);
	}
	memset ((char *)varcmdarr->sequence,0,(varcmdarr->length * sizeof (DevCmdInfo)));

	for (long i = 0; i < varcmdarr->length; i++)
	{
		varcmdarr->sequence[i].cmd      = tango_device[dev_id].cmd_value[i];
		strncat(varcmdarr->sequence[i].cmd_name,tango_device[dev_id].cmd_name[i].c_str(),32);
		varcmdarr->sequence[i].in_type  = tango_to_taco_type(tango_device[dev_id].argin_type[i]);
		varcmdarr->sequence[i].in_name = (char*)" ";
		varcmdarr->sequence[i].out_type = tango_to_taco_type(tango_device[dev_id].argout_type[i]);
		varcmdarr->sequence[i].out_name = (char*)" ";
	}

	return (DS_OK);
}

/*+**********************************************************************
Function   : 	extern long tango_dev_ping()

 Description:	Ping a TANGO device

 Arg(s) In  :	devserver ds - device 

 Arg(s) Out :	varcmdarr - sequence of DevCmdInfo structures.
		error     - Will contain an appropriate error code if the
			    corresponding call returns a non-zero value.

 Return(s)  :	DS_OK or DS_NOTOK
***********************************************************************-*/

long tango_dev_ping (devserver ds, long *error)
{
	long dev_id;

	dev_id = ds->ds_id;

	*error = 0;

        try
        {
                tango_device[dev_id].object->ping();
        }
        catch (CORBA_Exception  &e)                                                     {                                                                                       TangoUtil::print_exception(e);
// TODO - improve TANGO to TACO error handling here
                *error = DevErr_CommandFailed;
                return(DS_NOTOK);
        } 

	return (DS_OK);
}

/*+**********************************************************************
 Function   :   extern long tango_argin_to_any()

 Description:   convert a TACO argin type to a TANGO type and return it in an Any

 Arg(s) In  :   long argin_type - argin type to convert
                void *argin - pointer to argin

 Arg(s) Out :   long *error - error code, in case of failure

 Return(s)  :   CORBA_Any - argin converted to Any
***********************************************************************-*/         

static CORBA_Any tango_argin_to_any(long argin_type, long tango_type, void *argin, long *error)
{
	CORBA_Any send;

	cout4 << "tango_argin_to_any(): convert argin_type " << argin_type << " to CORBA_Any" << endl;

	if (argin == NULL && argin_type != D_VOID_TYPE)
	{
		throw_exception((const char*)"argin is NULL ", 
		                (const char *)"tango_argin_to_any()",
		                DevErr_IncompatibleCmdArgumentTypes ,
		                Tango_APILIB);
	}

	switch(argin_type) 
	{
		case D_VOID_TYPE : 
		{
			if (tango_type != Tango_DEV_VOID)
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEV_VOID ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			break;
		}

		case D_SHORT_TYPE :
		case D_USHORT_TYPE :
		{
			if ((tango_type != Tango_DEV_SHORT) &&
			    (tango_type != Tango_DEV_USHORT))
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEV_SHORT ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			short *argin_short;
			argin_short = (short*)argin;
			send <<= *argin_short;
			break;
		}
			
		case D_LONG_TYPE :
		case D_ULONG_TYPE :
		{
			if ((tango_type != Tango_DEV_LONG) &&
			    (tango_type != Tango_DEV_ULONG))
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEV_LONG ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			long *argin_long;
			argin_long = (long*)argin;
			send <<= *argin_long;
			break;
		}
			
		case D_FLOAT_TYPE :
		{
			if (tango_type != Tango_DEV_FLOAT)
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEV_FLOAT ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			float *argin_float;
			argin_float = (float*)argin;
			send <<= *argin_float;
			break;
		}
			
		case D_DOUBLE_TYPE :
		{
			if (tango_type != Tango_DEV_DOUBLE)
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEV_DOUBLE ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			double *argin_double;
			argin_double = (double*)argin;
			send <<= *argin_double;
			break;
		}
			
		case D_STRING_TYPE :
		{
			char **argin_str;

			if (tango_type != Tango_DEV_STRING)
			{
				throw_exception((const char*)"tango argin type is not Tango_DEV_STRING", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			argin_str = (char**)argin;
			send <<= CORBA_string_dup(*argin_str);
			break;
		}

		case D_VAR_STRINGARR :
		{
			if (tango_type != Tango_DEVVAR_STRINGARRAY)
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEVVAR_STRINGARRAY ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			DevVarStringArray *argin_vsa;
			argin_vsa = (DevVarStringArray*)argin;
			Tango_DevVarStringArray *tango_vsa = new Tango_DevVarStringArray;
			tango_vsa->length(argin_vsa->length);
			for (long i=0; i<argin_vsa->length; i++)
			{
				(*tango_vsa)[i] = CORBA_string_dup(argin_vsa->sequence[i]);
			}
			send <<= tango_vsa;
			break;
		}
			
		case D_VAR_SHORTARR :
		case D_VAR_USHORTARR :
		{
			if ((tango_type != Tango_DEVVAR_SHORTARRAY) &&
			    (tango_type != Tango_DEVVAR_USHORTARRAY))
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEVVAR_SHORTARRAY ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			DevVarShortArray *argin_vsha;
			argin_vsha = (DevVarShortArray*)argin;
			Tango_DevVarShortArray *tango_vsha = new Tango_DevVarShortArray;
			tango_vsha->length(argin_vsha->length);
			for (long i=0; i<argin_vsha->length; i++)
			{
				(*tango_vsha)[i] = argin_vsha->sequence[i];
			}
			send <<= tango_vsha;
			break;
		}
			
		case D_VAR_LONGARR :
		case D_VAR_ULONGARR :
		{
			if ((tango_type != Tango_DEVVAR_LONGARRAY) &&
			    (tango_type != Tango_DEVVAR_ULONGARRAY))
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEVVAR_LONGARRAY ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			DevVarLongArray *argin_vla;
			argin_vla = (DevVarLongArray*)argin;
			Tango_DevVarLongArray *tango_vla = new Tango_DevVarLongArray;
			tango_vla->length(argin_vla->length);
			for (long i=0; i<argin_vla->length; i++)
			{
				(*tango_vla)[i] = argin_vla->sequence[i];
			}
			send <<= tango_vla;
			break;
		}
			
		case D_VAR_FLOATARR :
		{
			if (tango_type != Tango_DEVVAR_FLOATARRAY)
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEVVAR_FLOATARRAY ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			DevVarFloatArray *argin_vfa;
			argin_vfa = (DevVarFloatArray*)argin;
			Tango_DevVarFloatArray *tango_vfa = new Tango_DevVarFloatArray;
			tango_vfa->length(argin_vfa->length);
			for (long i=0; i<argin_vfa->length; i++)
			{
				(*tango_vfa)[i] = argin_vfa->sequence[i];
			}
			send <<= tango_vfa;
			break;
		}
			
		case D_VAR_DOUBLEARR :
		{
			if (tango_type != Tango_DEVVAR_DOUBLEARRAY)
			{
				throw_exception((const char*)"TANGO argin type not Tango_DEVVAR_DOUBLEARRAY ", 
				                (const char *)"tango_argin_to_any()",
				                DevErr_IncompatibleCmdArgumentTypes ,
						Tango_APILIB);
			}
			DevVarDoubleArray *argin_vda;
			argin_vda = (DevVarDoubleArray*)argin;
			Tango_DevVarDoubleArray *tango_vda = new Tango_DevVarDoubleArray;
			tango_vda->length(argin_vda->length);
			for (long i=0; i<argin_vda->length; i++)
			{
				(*tango_vda)[i] = argin_vda->sequence[i];
			}
			send <<= tango_vda;
			break;
		}
			
		default:
			throw_exception((const char*)"TACO type no supported", 
					(const char *)"tango_argin_to_any()",
			                DevErr_IncompatibleCmdArgumentTypes ,
					Tango_APILIB);
	}

	return(send);
}

/*+**********************************************************************
 Function   :   extern long tango_any_to_argout()

 Description:   extract a TANGO type from an Any and convert it to a TACO argout

 Arg(s) In  :   long argout_type - argout type to convert
		CORBA_Any received - Any returned by device server
                void *argout - pointer to argout

 Arg(s) Out :   long *error - error code, in case of failure

 Return(s)  :   void
***********************************************************************-*/         

static void tango_any_to_argout(long argout_type, long tango_type, CORBA_Any received, void *argout, long *error)
{
	cout4 << "tango_any_to_argout(): convert CORBA_Any to argout type " << argout_type << endl;

	if (argout == NULL && argout_type != D_VOID_TYPE)
	{
		throw_exception((const char*)"argout is NULL ", 
		                (const char *)"tango_any_to_argout()",
		                DevErr_IncompatibleCmdArgumentTypes ,
		                Tango_APILIB);
	}

	switch(argout_type) 
	{
		case D_VOID_TYPE : 
		{
			if (tango_type != Tango_DEV_VOID)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_VOID", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			break;
		}

		case D_SHORT_TYPE :
		case D_USHORT_TYPE :
		{
			short tango_short;
			short *argout_short;

			if ((tango_type != Tango_DEV_SHORT) &&
			    (tango_type != Tango_DEV_USHORT) &&
			    (tango_type != Tango_DEV_STATE))
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_SHORT", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_short;
			argout_short = (short*)argout;
			*argout_short = tango_short;
			break;
		}

		case D_LONG_TYPE :
		case D_ULONG_TYPE :
		{
			long tango_long;
			long *argout_long;

			if (tango_type != Tango_DEV_LONG)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_LONG", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_long;
			argout_long = (long*)argout;
			*argout_long = tango_long;
			break;
		}

		case D_FLOAT_TYPE :
		{
			float tango_float;
			float *argout_float;

			if (tango_type != Tango_DEV_FLOAT)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_FLOAT", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_float;
			argout_float = (float*)argout;
			*argout_float = tango_float;
			break;
		}

		case D_DOUBLE_TYPE :
		{
			double tango_double;
			double *argout_double;

			if (tango_type != Tango_DEV_DOUBLE)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_DOUBLE", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_double;
			argout_double = (double*)argout;
			*argout_double = tango_double;
			break;
		}

		case D_STRING_TYPE :
		{
			char *tango_str;
			char **argout_str;

			if (tango_type != Tango_DEV_STRING)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_STRING", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_str;
			cout4 << "tango_any_to_argout(): string received = " << tango_str << endl;
			argout_str = (char**)argout;
			*argout_str = (char*)malloc(strlen(tango_str)+1);
			strcpy(*argout_str,tango_str);
			break;
		}

		case D_VAR_STRINGARR :
		{
			DevVarStringArray *argout_vsa;
			Tango_DevVarStringArray *tango_vsa;

			if (tango_type != Tango_DEVVAR_STRINGARRAY)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_STRINGARRAY", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vsa;
			cout4 << "tango_any_to_argout(): string array received length = " << tango_vsa->length() << endl;
			argout_vsa = (DevVarStringArray*)argout;
			if (argout_vsa->sequence == NULL)
			{
				argout_vsa->sequence = (char**)malloc(tango_vsa->length()*sizeof(char*));
			}
			argout_vsa->length = tango_vsa->length();
			for (long i=0; i< tango_vsa->length(); i++)
			{
				argout_vsa->sequence[i] = (char*)malloc(strlen((*tango_vsa)[i])+1);
				strcpy(argout_vsa->sequence[i],(*tango_vsa)[i]);
			}
			break;
		}

		case D_VAR_SHORTARR :
		case D_VAR_USHORTARR :
		{
			DevVarShortArray *argout_vsha;
			Tango_DevVarShortArray *tango_vsha;

			if ((tango_type != Tango_DEVVAR_SHORTARRAY) &&
			    (tango_type != Tango_DEVVAR_USHORTARRAY))
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_SHORTARRAY", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vsha;
			argout_vsha = (DevVarShortArray*)argout;
			if (argout_vsha->sequence == NULL)
			{
				argout_vsha->sequence = (short*)malloc(tango_vsha->length()*sizeof(short));
			}
			argout_vsha->length = tango_vsha->length();
			for (long i=0; i< tango_vsha->length(); i++)
			{
				argout_vsha->sequence[i] = (*tango_vsha)[i];
			}
			break;
		}

		case D_VAR_LONGARR :
		case D_VAR_ULONGARR :
		{
			DevVarLongArray *argout_vla;
			Tango_DevVarLongArray *tango_vla;

			if ((tango_type != Tango_DEVVAR_LONGARRAY) &&
			    (tango_type != Tango_DEVVAR_ULONGARRAY))
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_LONGARRAY", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vla;
			argout_vla = (DevVarLongArray*)argout;
			if (argout_vla->sequence == NULL)
			{
				argout_vla->sequence = (long*)malloc(tango_vla->length()*sizeof(long));
			}
			argout_vla->length = tango_vla->length();
			for (long i=0; i< tango_vla->length(); i++)
			{
				argout_vla->sequence[i] = (*tango_vla)[i];
			}
			break;
		}

		case D_VAR_FLOATARR :
		{
			DevVarFloatArray *argout_vfa;
			Tango_DevVarFloatArray *tango_vfa;

			if (tango_type != Tango_DEVVAR_FLOATARRAY)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_FLOATARRAY", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vfa;
			argout_vfa = (DevVarFloatArray*)argout;
			if (argout_vfa->sequence == NULL)
			{
				argout_vfa->sequence = (float*)malloc(tango_vfa->length()*sizeof(float));
			}
			argout_vfa->length = tango_vfa->length();
			for (long i=0; i< tango_vfa->length(); i++)
			{
				argout_vfa->sequence[i] = (*tango_vfa)[i];
			}
			break;
		}

		case D_VAR_DOUBLEARR :
		{
			DevVarDoubleArray *argout_vda;
			Tango_DevVarDoubleArray *tango_vda;

			if (tango_type != Tango_DEVVAR_DOUBLEARRAY)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_DOUBLEARRAY", 
				                (const char *)"tango_any_to_argout()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vda;
			argout_vda = (DevVarDoubleArray*)argout;
			if (argout_vda->sequence == NULL)
			{
				argout_vda->sequence = (double*)malloc(tango_vda->length()*sizeof(double));
			}
			argout_vda->length = tango_vda->length();
			for (long i=0; i< tango_vda->length(); i++)
			{
				argout_vda->sequence[i] = (*tango_vda)[i];
			}
			break;
		}

		default:
			throw_exception((const char*)"taco argout type not supported", 
			                (const char *)"tango_any_to_argout()",
			                DevErr_IncompatibleCmdArgumentTypes ,
			                Tango_APILIB);
	}

	return;
}
	
/*+**********************************************************************
 Function   :   extern long tango_any_to_argout_raw()

 Description:   extract a TANGO type from an Any and convert it to a TACO argout
		of DEV_OPAQUE_TYPE

 Arg(s) In  :   long argout_type - argout type to convert
		CORBA_Any received - Any returned by device server
                void *argout - pointer to argout

 Arg(s) Out :   long *error - error code, in case of failure

 Return(s)  :   void
***********************************************************************-*/         

static void tango_any_to_argout_raw(long argout_type, long tango_type, CORBA_Any received, void *argout, long *error)
{
	dc_datacmd dc_datacmd;
	dc_error dc_error;
	DevOpaque *taco_opaque;
	long status;

	taco_opaque = (DevOpaque*)argout;

	cout4 << "tango_any_to_argout_raw(): convert CORBA_Any to argout type " << argout_type << endl;

	if (argout == NULL && argout_type != D_VOID_TYPE)
	{
		throw_exception((const char*)"argout is NULL ", 
		                (const char *)"tango_any_to_argout()",
		                DevErr_IncompatibleCmdArgumentTypes ,
		                Tango_APILIB);
	}

	switch(argout_type) 
	{
		case D_VOID_TYPE : 
		{
			void *taco_void = NULL;

			if (tango_type != Tango_DEV_VOID)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_VOID", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			dc_datacmd.argout_type = argout_type;
			dc_datacmd.argout = &taco_void;
			status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;
			break;
		}

		case D_SHORT_TYPE :
		case D_USHORT_TYPE :
		{
			short tango_short;
			short taco_short;

			if ((tango_type != Tango_DEV_SHORT) &&
			    (tango_type != Tango_DEV_USHORT) &&
			    (tango_type != Tango_DEV_STATE))
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_SHORT", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_short;
			taco_short = tango_short;
			dc_datacmd.argout_type = argout_type;
			dc_datacmd.argout = &taco_short;
			status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;
			break;
		}

		case D_LONG_TYPE :
		case D_ULONG_TYPE :
		{
			long tango_long;
			long taco_long;

			if (tango_type != Tango_DEV_LONG)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_LONG", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_long;
			taco_long = tango_long;
			dc_datacmd.argout_type = argout_type;
			dc_datacmd.argout = &taco_long;
			status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;
			break;
		}

		case D_FLOAT_TYPE :
		{
			float tango_float;
			float taco_float;

			if (tango_type != Tango_DEV_FLOAT)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_FLOAT", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_float;
			taco_float = tango_float;
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_float;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);       
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence; 
			break;
		}

		case D_DOUBLE_TYPE :
		{
			double tango_double;
			double taco_double;

			if (tango_type != Tango_DEV_DOUBLE)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_DOUBLE", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_double;
			taco_double = tango_double;
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_double;
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);       
			break;
		}

		case D_STRING_TYPE :
		{
			char *tango_str;
			char *taco_str;

			if (tango_type != Tango_DEV_STRING)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEV_STRING", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_str;
			cout4 << "tango_any_to_argout(): string received = " << tango_str << endl;
			taco_str = (char*)malloc(strlen(tango_str)+1);
			strcpy(taco_str,tango_str);
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_str;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);       
			taco_opaque->length = dc_datacmd.length;
			taco_opaque->sequence = dc_datacmd.sequence;
			free(taco_str);
			break;
		}

		case D_VAR_STRINGARR :
		{
			DevVarStringArray taco_vsa;
			Tango_DevVarStringArray *tango_vsa;
			long i;

			if (tango_type != Tango_DEVVAR_STRINGARRAY)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_STRINGARRAY", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vsa;
			cout4 << "tango_any_to_argout(): string array received length = " << tango_vsa->length() << endl;
			taco_vsa.sequence = (char**)malloc(tango_vsa->length()*sizeof(char*));
			taco_vsa.length = tango_vsa->length();
			for (i=0; i< tango_vsa->length(); i++)
			{
				taco_vsa.sequence[i] = (char*)malloc(strlen((*tango_vsa)[i])+1);
				strcpy(taco_vsa.sequence[i],(*tango_vsa)[i]);
			}
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_vsa;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;              
			for (i=0; i< tango_vsa->length(); i++)
			{
				free(taco_vsa.sequence[i]);
			}
			free(taco_vsa.sequence);
			break;
		}

		case D_VAR_SHORTARR :
		case D_VAR_USHORTARR :
		{
			DevVarShortArray taco_vsha;
			Tango_DevVarShortArray *tango_vsha;

			if ((tango_type != Tango_DEVVAR_SHORTARRAY) &&
			    (tango_type != Tango_DEVVAR_USHORTARRAY))
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_SHORTARRAY", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vsha;
			taco_vsha.sequence = (short*)malloc(tango_vsha->length()*sizeof(short));
			taco_vsha.length = tango_vsha->length();
			for (long i=0; i< tango_vsha->length(); i++)
			{
				taco_vsha.sequence[i] = (*tango_vsha)[i];
			}
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_vsha;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;              
			free(taco_vsha.sequence);
			break;
		}

		case D_VAR_LONGARR :
		case D_VAR_ULONGARR :
		{
			DevVarLongArray taco_vla;
			Tango_DevVarLongArray *tango_vla;

			if ((tango_type != Tango_DEVVAR_LONGARRAY) &&
			    (tango_type != Tango_DEVVAR_ULONGARRAY))
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_LONGARRAY", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vla;
			taco_vla.sequence = (long*)malloc(tango_vla->length()*sizeof(long));
			taco_vla.length = tango_vla->length();
			for (long i=0; i< tango_vla->length(); i++)
			{
				taco_vla.sequence[i] = (*tango_vla)[i];
			}
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_vla;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;              
			free(taco_vla.sequence);
			break;
		}

		case D_VAR_FLOATARR :
		{
			DevVarFloatArray taco_vfa;
			Tango_DevVarFloatArray *tango_vfa;

			if (tango_type != Tango_DEVVAR_FLOATARRAY)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_FLOATARRAY", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vfa;
			taco_vfa.sequence = (float*)malloc(tango_vfa->length()*sizeof(float));
			taco_vfa.length = tango_vfa->length();
			for (long i=0; i< tango_vfa->length(); i++)
			{
				taco_vfa.sequence[i] = (*tango_vfa)[i];
			}
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_vfa;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;              
			free(taco_vfa.sequence);
			break;
		}

		case D_VAR_DOUBLEARR :
		{
			DevVarDoubleArray taco_vda;
			Tango_DevVarDoubleArray *tango_vda;

			if (tango_type != Tango_DEVVAR_DOUBLEARRAY)
			{
				throw_exception((const char*)"tango argout type is not Tango_DEVVAR_DOUBLEARRAY", 
				                (const char *)"tango_any_to_argout_raw()",
				                DevErr_IncompatibleCmdArgumentTypes ,
				                Tango_APILIB);
			}
			received >>= tango_vda;
			taco_vda.sequence = (double*)malloc(tango_vda->length()*sizeof(double));
			taco_vda.length = tango_vda->length();
			for (long i=0; i< tango_vda->length(); i++)
			{
				taco_vda.sequence[i] = (*tango_vda)[i];
			}
                        dc_datacmd.argout_type = argout_type;
                        dc_datacmd.argout = &taco_vda;
                        status = dc_dataconvert(&dc_datacmd, 1, &dc_error);
                        taco_opaque->length = dc_datacmd.length;
                        taco_opaque->sequence = dc_datacmd.sequence;              
			free(taco_vda.sequence);
			break;
		}

		default:
			throw_exception((const char*)"taco argout type not supported", 
			                (const char *)"tango_any_to_argout_raw()",
			                DevErr_IncompatibleCmdArgumentTypes ,
			                Tango_APILIB);
	}

	return;
}
	
/*+**********************************************************************
 Function   :   static long tango_to_taco_type()

 Description:   convert a TANGO type code to a TACO type code

 Arg(s) In  :   long tango_type - TANGo type code

 Arg(s) Out :   none

 Return(s)  :   TACO type code
***********************************************************************-*/

static long tango_to_taco_type(long tango_type)
{
	long taco_type;

	switch (tango_type)
	{
		case Tango_DEV_VOID : 
			taco_type = D_VOID_TYPE;
			break;
		case Tango_DEV_BOOLEAN : 
			taco_type = D_BOOLEAN_TYPE;
			break;
		case Tango_DEV_STATE :
		case Tango_DEV_SHORT : 
			taco_type = D_SHORT_TYPE;
			break;
		case Tango_DEV_USHORT : 
			taco_type = D_USHORT_TYPE;
			break;
		case Tango_DEV_LONG : 
			taco_type = D_LONG_TYPE;
			break;
		case Tango_DEV_ULONG : 
			taco_type = D_ULONG_TYPE;
			break;
		case Tango_DEV_FLOAT : 
			taco_type = D_FLOAT_TYPE;
			break;
		case Tango_DEV_DOUBLE : 
			taco_type = D_DOUBLE_TYPE;
			break;
		case Tango_DEV_STRING : 
			taco_type = D_STRING_TYPE;
			break;
		case Tango_DEVVAR_CHARARRAY : 
			taco_type = D_VAR_CHARARR;
			break;
		case Tango_DEVVAR_SHORTARRAY : 
			taco_type = D_VAR_SHORTARR;
			break;
		case Tango_DEVVAR_USHORTARRAY : 
			taco_type = D_VAR_USHORTARR;
			break;
		case Tango_DEVVAR_LONGARRAY : 
			taco_type = D_VAR_LONGARR;
			break;
		case Tango_DEVVAR_ULONGARRAY : 
			taco_type = D_VAR_ULONGARR;
			break;
		case Tango_DEVVAR_STRINGARRAY : 
			taco_type = D_VAR_STRINGARR;
			break;
		default : taco_type = -1;
	}
	return(taco_type);
}

/*+**********************************************************************
 Function   :	static long get_cmd_value()

 Description:   Read the command value corresponding to the command string
		from the TACO resource database. This is the reverse of
		the function get_cmd_string() in util_api.c. 
		The resource name must follow the convention :
		CLASS/class_name/CMD/cmd_string: value
		e.g. class/database/cmd/dbinfo: 10000

		DS_WARNING is returned, if the function was
		executed correctly, but no command name
		value was found in the database.

 Arg(s) In  :   char *class_name - class name
		char *cmd_name   - command name

 Arg(s) Out :   long *cmd_value  - command value
		long *error   - Will contain an appropriate error
			        code if the corresponding call
		    	        returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK or DS_WARNING
***********************************************************************-*/

static long get_cmd_value (char *class_name, char *cmd_name, long *cmd_value, long *error)
{
	char		res_path[LONG_NAME_SIZE];
	char		res_name[SHORT_NAME_SIZE];
	char		*ret_str = NULL;
	db_resource 	res_tab;

	*error = 0;

	cout4 << "Class " << class_name << " command " << cmd_name << endl;
/*
 * Create the resource path and the resource structure.
 *
 * first check to see whether the device belongs to another
 * nethost domain i.e. i_nethost != 0
 */

/*
 * use default nethost
 */
	sprintf(res_path, "CLASS/%s/CMD", class_name);

	sprintf (res_name, "%s", cmd_name);

	res_tab.resource_name = res_name;
	res_tab.resource_type = D_STRING_TYPE;
	res_tab.resource_adr  = &ret_str;

/*
 * Read the command name string from the database.
 */

	if (db_getresource (res_path, &res_tab, 1, error) == DS_NOTOK)
	{
		return (DS_NOTOK);
	}

/*
 * If the variable ret_str is still NULL, no resource value was found
 * in the database, but the function was executed without error.
 * In this case return the value DS_WARNING.
 */
	if ( ret_str == NULL )
	{
		return (DS_WARNING);
	}

	sscanf(ret_str,"%d",cmd_value);

	return (DS_OK);
}
