static char RcsId[] = 
"@(#)$Header: gen_api.c,v 8.14 2000/06/22 14:50:15 goetz Rel $";
/*+*******************************************************************

 File       :	gen_api.c

 Project    :	Device Servers

 Description:	General Application Programmers Interface to the device
		servers and clients using the SUN-RPC.

 Author(s)  :	Jens Meyer

 Original   :	January 1991

 $Revision: 8.14 $
 $Date: 2000/06/22 14:50:15 $

 $Author: goetz $

 $Log:	gen_api.c,v $
 * Revision 8.14  2000/06/22  14:50:15  14:50:15  goetz (Andrew GOETZ)
 * fixed bug in dev_event_listen() which caused server to crash
 * 
 Revision 8.13  2000/06/02 21:56:33  goetz
 _DEVICE_H now _TANGO_DEVICE_H in Device.H; (char*)taco_tango

 Revision 8.12  2000/06/02 15:57:52  goetz
 dev_event_fire() now device specific; ported to SuSE V6.4

 Revision 8.11  2000/05/31 07:47:39  goetz
 tango_api has local copies of argc and argv, ported to HP-UX

 * Revision 8.10  2000/05/29  21:39:06  21:39:06  goetz (Andrew GOETZ)
 * fixed prototyping problems with C++ on HP-UX
 * 
 * Revision 8.9  2000/05/29  18:13:16  18:13:16  goetz (Andrew GOETZ)
 * fixed bug in dev_import_timeout() which prevented timeouts < 2s
 * 
 Revision 8.8  2000/05/02 15:32:00  goetz
 added prototype for tsleep() for OS9 C++

 * Revision 8.7  2000/05/02  14:19:24  14:19:24  goetz (Andy Goetz)
 * removed check for device server running on different host
 * 
 * Revision 8.6  2000/05/02  13:28:02  13:28:02  goetz (Andy Goetz)
 * added random sleep to gettransient + 3 retries
 * 
 Revision 8.5  2000/03/27 17:13:41  goetz
 declared pmap_getmaps() as external C function for OS9 C++

 Revision 8.4  2000/03/13 14:49:57  goetz
 import timeout now programmable using dev_import_timeout()

 Revision 8.3  2000/03/13 10:31:31  goetz
 added Jens' modifications to fix bug in local putget

 Revision 8.2  2000/03/10 17:01:18  goetz
 dev_synch() now excludes udp clients and new imports

 Revision 8.1  2000/01/18 16:48:33  goetz
 tango_dev_import() now only called in dev_import if -DTANGO

 Revision 8.0  1999/12/28 14:18:21  goetz
 added TANGO support for TACO dev_xxx() calls via -DTANGO for C++

 Revision 7.10  1999/11/25 08:27:09  goetz
 replaced fprintf(stderr) with printf; made startup() C++ compatible

 Revision 7.9  1999/11/22 20:18:55  goetz
 removed const from gettransient() prototype

 Revision 7.8  1999/11/21 20:45:34  goetz
 included all M.Diehl's patches (major changes to gettransient() + main())

 Revision 7.7  1999/11/18 22:28:49  goetz
 fixed event multi-client bug, free client handle after event_listen, timeout in dev_synch()

 Revision 7.6  1999/11/01 19:05:14  goetz
 make local copy of user name returned by getpwuid() in sec_api.c

 Revision 7.5  1999/08/06 17:43:57  goetz
 removed usage of varargs for g++ on solaris and hpux

 Revision 7.4  1999/07/09 05:15:25  goetz
 added M.Diehl's patch to DevServerSig.c to exit() after calling unregister_server()

 Revision 7.3  1999/06/07 15:26:47  goetz
 fixed bug with multi-nethost reimport, device name stored with nethost

 Revision 7.2  1999/05/12 15:21:44  goetz
 changed dev_event_fire() to void; fixed bug in dev_event_unlisten()

 * Revision 7.1  99/05/11  15:59:33  15:59:33  goetz (Andy Goetz)
 * replace static declaration of event_client[] array by malloc()
 * 
 Revision 7.0  1999/04/26 07:30:28  goetz
 implemented user events (added event_api.c)

 Revision 6.15  1999/03/27 09:00:08  goetz
 asynch_cleanup() only checks for asynch clients; C++ returns output arguments if DS_WARNING

 * Revision 6.14  99/02/27  15:21:06  15:21:06  goetz (Andy Goetz)
 * fixed reimport bug; disabled reimport for rpc version 1
 * 
 Revision 6.13  1999/02/05 00:50:14  goetz
 fixed reimport socket+memory leak; ported to aCC on HPUX; suppressed rpc errors

 * Revision 6.12  99/01/18  21:38:50  21:38:50  goetz (Andy Goetz)
 * suppressed test for RPC_TIMEDOUT in svc_check()
 * 
 * Revision 6.11  99/01/14  23:14:39  23:14:39  goetz (Andy Goetz)
 * dbase timeout = 4 s; unregister ver. 1; svc_check() handles RPC_TIMEDOUT
 * 
 * Revision 6.10  98/12/22  10:29:32  10:29:32  taurel (E.Taurel)
 * Added support for Solaris and HP C++ native compiler. Remove K_R C support
 * 
 * Revision 6.9  98/12/15  14:04:49  14:04:49  goetz (Andy Goetz)
 * unregister RPC; use pgm no. 1 as flag; dev_pending() checks timeout; extern C prototype
 * 
 * Revision 6.8  98/11/20  14:35:24  14:35:24  goetz (Andy Goetz)
 * unregister API_VERSION (4) correctly now + check return value of svc_register()
 * 
 * Revision 6.7  98/11/20  10:34:49  10:34:49  goetz (Andy Goetz)
 * improved gettrans_ut(); multi-config_flags=0; removde static client_data; fixed linux select() bug; only register pgm # 0,4,5; portmap address == 111
 * 
 * Revision 6.6  98/09/24  09:25:01  09:25:01  goetz (Andy Goetz)
 * fixed stupid type which reintroduced asynchronous memory leak
 * 
 * Revision 6.5  98/08/26  16:20:25  16:20:25  goetz (Andy Goetz)
 * added dynamic NETHOST allocation, fixed bug in db_setup_multi, ported to Irix 6.4
 * 
 * Revision 6.4  98/04/29  10:21:14  10:21:14  goetz (Andy Goetz)
 * asynchronous - memory leak fixed, OS9 bug fixed, stale handle cleanup, pending=300
 * 
 * Revision 6.3  97/12/10  15:28:22  15:28:22  meyer (J.Meyer)
 * Changed select() timeout in rpc_check_host() to 200ms
 * 
 * Revision 6.2  97/12/01  12:59:05  12:59:05  goetz (Andy Goetz)
 * intermediate checkin before opening 5.14.1 branch
 * 
 * Revision 6.1  97/11/20  21:49:15  21:49:15  goetz (Andy Goetz)
 * main() continue even if cannot register asynch rpc
 * 
 * Revision 6.0  97/11/03  17:22:16  17:22:16  goetz (Andy Goetz)
 * merged synchronous and asynchronous branches
 * 
 Revision 5.15  1997/10/24 15:26:55  klotz
 ported to NT NT 4.0 + NMSDK 5.0 ; tested with TextTalkds ; new DevServerMain.c

 * Revision 5.14  97/09/25  17:36:32  17:36:32  goetz (Andy Goetz)
 * fixed
 * 
 Revision 5.13  1997/09/03 17:34:25  meyer
 Added casting for ucc++ in DevServerSig.c

 * Revision 5.12  97/09/03  15:54:01  15:54:01  meyer (J.Meyer)
 * Changed OS9 signal handling from intercept() to signal().
 * 
 * Revision 5.11  97/08/05  14:19:14  14:19:14  goetz (Andy Goetz)
 * ported to VxWork 5.2, pgm. no. based on CRC, -m suppressed, -ve errors bug fix
 * 
 * Revision 5.10  97/03/14  17:26:05  17:26:05  dserver ()
 * fixed bug with open file handles after free
 * 
 * Revision 5.8.1.4  97/03/14  17:24:39  17:24:39  goetz (Andy Goetz)
 * fixed bug with open file handles after free
 * 
 * Revision 5.9  97/03/13  11:17:06  11:17:06  klotz (W.D. Klotz)
 * first part of NT4.0 port
 * 
 * Revision 5.8  97/01/31  12:06:14  12:06:14  goetz (Andy Goetz)
 * dev_protocol(), Linux+security bug fixes, dev_import() ignores timeout on NULL procedure, svc_run for OS9
 * 
 * Revision 5.7  97/01/14  08:57:13  08:57:13  goetz (Andy Goetz)
 * added stateless to dev_free(),_raw(),_asyn(),_protocol(),_timeout()
 * 
 * Revision 5.6  97/01/12  18:33:33  18:33:33  goetz (Andy Goetz)
 * dev_import() now stateless i.e. import works even if server is not running
 * 
 * Revision 5.5  97/01/08  17:02:19  17:02:19  goetz (Andy Goetz)
 * Ported to Linux (2.0.9)
 * 
 * Revision 5.4  96/11/27  09:58:06  09:58:06  goetz (Andy Goetz)
 * removed rpc admin interface from DevServerMain.c; putenv() suppressed for _UCC
 * 
 * Revision 5.3  96/11/26  16:24:33  16:24:33  goetz (Andy Goetz)
 * fixed bug (i_nethost=0) in dev_import_local()
 * 
 * Revision 5.2  96/11/26  13:30:54  13:30:54  goetz (Andy Goetz)
 * OICDevice (C++ wrapper for OIC) added + fixed multi-nethost memory leaks
 * 
 * Revision 5.1  96/10/31  18:02:20  18:02:20  goetz (Andy Goetz)
 * added multi-nethost support
 * 
 * Revision 5.0  96/10/31  17:59:11  17:59:11  goetz (Andy Goetz)
 * added multi-nethost support
 * 

 Copyright (c) 1990 by European Synchrotron Radiation Facility, 
                       Grenoble, France

********************************************************************-*/

#include <API.h>
#include <ApiP.h>
#include <API_xdr_vers3.h>

#include <DserverTeams.h>

#include <Admin.h>
#include <DevErrors.h>
#include <DevSignal.h>
#ifdef vxworks
#include <taskLib.h>
#endif /* vxworks */

/*
 * Include file for variable function arguments.
 */
#ifdef sun
#include <varargs.h>
#include <signal.h>
#else
#ifndef OSK
#include <stdarg.h>
#endif
#endif

static long setup_config 	(long *error);
extern _DLLFunc long setup_config_multi 	(char *nethost, long *error);
static void msg_write 		(_Int msg_type, char *msg_string);
static void msg_send 		(_Int msg_type);
static void msg_clear 		(_Int msg_type);
static long get_error_string 	(long error, char *error_str);

/*
 * UCC has not prototype for putenv() , define one ! 
 *
 * - andy 22nov96
 */
#ifdef _OSK
long putenv(char *);
#endif /* _OSK */


/****************************************
 *          Globals	                *
 ****************************************/

/*
 *  Types for global state flags for the current
 *  server or client status and for Message Server and Database
 *  Server information are defined in API.h
 */

/*
 *  Configuration flags - modified by ag 16sep94
 *
 *  configuration 	== false
 *  message_server 	== false
 *  database_server	== false
 *  startup		== false
 *  security		== false
 */


configuration_flags	config_flags =
{
	False,False,False,False,False, "", "", 0L};


/*
 *  Database Server globals
 */

dbserver_info		db_info;

/*
 *  Message Server globals
 */

msgserver_info		msg_info;
static _message_buffer 	message_buffer [NUMBER_OF_MSG_TYPES];

/*
 * multi-nethost support requires an array of nethost_info structures
 * to keep track of the multple nethosts and their databases and
 * message servers. The following structures is used for this.
 * The first element (nethost=0) is the default nethost, a copy 
 * of the (old) globals config_flags, db_info and msg_info.
 * Additional nethosts are stored starting from element 1 on.
 *
 * The maximum number of nethosts starts off at MIN_NETHOST.
 * Space is allocated as needed for more nethosts in slices of
 * MIN_NETHOST (10)
 *
 */

/*nethost_info 	multi_nethost[MIN_NETHOST];*/
/*long max_nethost = MIN_NETHOST;*/

nethost_info 	*multi_nethost;
long max_nethost = 0;

/*
 *  Debug globals
 */

long debug_flag = 0x0;

/*+**********************************************************************
 Function   :   extern long msg_import()

 Description:   imports a message service.
            :   DS_xxx parameters are used to name the
		approriate error file and to display
		a correct error window. 

 Arg(s) In  :   char *DS_name       - name of the device server
	    :   char *DS_host       - host of the device server
            :   long DS_prog_number - program number of the device server
	    :   char *DS_display    - display to use

 Arg(s) Out :   long *error         - Will contain an appropriate error
				      code if the corresponding call
		    		      returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc msg_import (char *DS_name, char *DS_host, long DS_prog_number,
char *DS_display, long *error)
{
	CLIENT		*clnt;
	char		*hstring;

	*error = 0;

	/*
	 * check wether a message server is already imported
	 */

	if (config_flags.message_server)
		return (0);

	/*
	 * check wether the system is already configured
	 */

	if ( !config_flags.configuration )
	{
		if ( (setup_config (error)) < 0 )
			return (-1);
	}


	/*
	 * Create message server client handle with data from
	 * global msgserver_info structure.
	 */

	clnt = clnt_create ( msg_info.conf->server_host,
	    msg_info.conf->prog_number,
	    msg_info.conf->vers_number,
	    "udp");
	if (clnt == NULL)
	{
		hstring = clnt_spcreateerror ("msg_import");
		dev_printerror (SEND,"%",hstring);
		*error = DevErr_MsgImportFailed;
		return (-1);
	}

	clnt_control (clnt, CLSET_RETRY_TIMEOUT, (char *) &msg_retry_timeout);
	clnt_control (clnt, CLSET_TIMEOUT, (char *) &msg_timeout);


	/* 
 	 * pass the information to the msgserver_info structure
 	 */

	sprintf (msg_info.conf->device_name,"MessageServer");
	msg_info.conf->clnt = clnt;
	msg_info.conf->ds_id = 0;
	msg_info.conf->no_svr_conn = 1;

	sprintf (msg_info.DS_name,    "%s", DS_name);
	sprintf (msg_info.DS_host,    "%s", DS_host);
	msg_info.DS_prog_number         =   DS_prog_number;
	sprintf (msg_info.DS_display, "%s", DS_display);

	config_flags.message_server = True;

	return (0);
}



/*
 * Attention: OS9 version 2.4 can not handle variable argument
 * lists. It will still use the old tricky function!!
 */
#if (!defined OSK) && (!defined _GNU_PP)
/*+**********************************************************************
 Function   :   extern void dev_printerror()

 Description:   Send device server error_string to a message 
		service or to stdout, if a message service
		is not imported.
            :   Error strings can be only 256 characters long.
		Longer texts can be send by storing short
		strings in the message buffer with the WRITE
		mode and send the whole buffer by using
		the SEND mode with the last string.

 Arg(s) In  :	a variable list of arguments which should contain 

	    :	short mode       - mode for message buffer : WRITE or SEND
	    :   char *fmt        - format string in printf() style.
            :   char *var_args   - variable argument list of variables to 
	    :                      print

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

_DLLFunc void dev_printerror (DevShort mode, char *fmt, ...)
{
	char		buffer [256];
	va_list 	args;

	if ( mode == CLEAR )
	{
		msg_clear ( ERROR_TYPE );
		return;
	}

	/*
 * get variable argument list pointer in order to pass it to vsprintf()
 */
#if (defined sun) || (defined irix)
	va_start(args);
#else
	va_start(args, fmt);
#endif /* sun */

	vsprintf (buffer,fmt,args);
	strcat (buffer,"\n");

	va_end(args);

	msg_write ( ERROR_TYPE, buffer );

	if ( mode != WRITE )
	{
		/*
            * send messages to message server if imported
            */

		if ( config_flags.message_server )
		{
			msg_send ( ERROR_TYPE );
		}
		else
		{
#ifdef _NT
			char msg [1024];
			sprintf ( msg, "\n\n%s\n", 
			    message_buffer[ERROR_TYPE].messages);
			conout(msg);
#else
			printf ( "\n\n%s\n", 
			    message_buffer[ERROR_TYPE].messages);
#endif
			msg_clear ( ERROR_TYPE );
		}
	}
}

/*
 * For OS9 version 2.4 and Solaris with g++
 */
#else
/*+**********************************************************************
 Function   :   extern void dev_printerror()

 Description:   Send device server error_string to a message 
		service or to stdout, if a message service
		is not imported.
            :   Error strings can be only 256 characters long.
		Longer texts can be send by storing short
		strings in the message buffer with the WRITE
		mode and send the whole buffer by using
		the SEND mode with the last string.

 Arg(s) In  :	short mode       - mode for message buffer : WRITE or SEND
	    :   char *fmt        - format string in printf() style.

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/
_DLLFunc void dev_printerror (DevShort mode,char *fmt, char *str)
{
	char  		buffer [256];

	if ( mode == CLEAR )
	{
		msg_clear ( ERROR_TYPE );
		return;
	}

	if (str != NULL)
	{
		sprintf (buffer,fmt,str);
	}
	else
	{
		sprintf (buffer,fmt);
	}
	strcat (buffer,"\n");

	msg_write ( ERROR_TYPE, buffer );

	if ( mode != WRITE )
	{
		/*
            * send messages to message server if imported
            */

		if ( config_flags.message_server )
		{
			msg_send ( ERROR_TYPE );
		}
		else
		{
#ifdef _NT
			char msg[1024];
			sprintf ( msg, "\n\n%s\n", 
			    message_buffer[ERROR_TYPE].messages);
			conout(msrg);
#else
			printf ( "\n\n%s\n", 
			    message_buffer[ERROR_TYPE].messages);
#endif
			msg_clear ( ERROR_TYPE );
		}
	}
}
#endif /* OSK */

/*+**********************************************************************
 Function   :	extern void *dev_printerror_no()

 Description:   retrieves the related error string for an error number
		from the list in DevErrors.h or the rsource database
		and sends the error string with the choosen mode 
		to dev_printerror().

 Arg(s) In  :	short mode     - mode for message buffer : WRITE or SEND
	    :	char *comment  - comment to be printed before 
				 error string.
	    :   long dev_errno - device server system error number.

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

void _DLLFunc dev_printerror_no (DevShort mode, char *comment, long dev_errno)
{
	char error_string [512];
	char err_string [256];

	/*
         * get error string from the resource database
         */

	get_error_string ( dev_errno, err_string );

	if (comment != NULL)
	{
		sprintf (error_string, "%s : %s", comment, err_string);
	}
	else
	{
		sprintf (error_string, "%s", err_string);
	}

	dev_printerror (mode,"%s",error_string);
}


/*+**********************************************************************
 Function   :	extern char *dev_error_str()

 Description:   retrieves the related error string for an error number
		from the list in DevErrors.h.

 Arg(s) In  :   long dev_errno - device server system error number.

 Arg(s) Out :   none

 Return(s)  :   error string, related to dev_errno.
***********************************************************************-*/

_DLLFunc char * dev_error_str (long dev_errno)
{
	char		err_string[256];
	char		*error_string = NULL;
	int	 	len;

	/*
	 * get error string from the resource database
	 */

	get_error_string ( dev_errno, err_string );

	len = strlen(err_string) + 1;
	error_string = (char *)malloc (len);
	sprintf (error_string,"%s", err_string);

	return (error_string);
}




/*
 * Attention: OS9 version 2.4 can not handle variable argument
 * lists. It will still use the old tricky function!!
 */
#if (!defined OSK) && (!defined _GNU_PP)
/*+**********************************************************************
 Function   :   extern void dev_printdebug()

 Description:   Send debug string to a message service
		or to stdout, if a message service
		is not imported.
            :   A debug string will be send if one of the debug_bits
		related to the string is set in the global debug_flag.
            :   Debug strings can br only 256 characters long.

 Arg(s) In  :	long debug_bits  - debug bits on which to send the
				   information.
	    :   char *fmt        - format string in printf() style.
            :   double a0,...,a9 - reserving memory for variables to print.

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

void _DLLFunc dev_printdebug (long debug_bits, char *fmt, ...)
{
	char		debug_string[256];
	va_list	args;
/*
 * get variable argument list pointer in order to pass it to vsprintf()
 */
#if (defined sun) || (defined irix)
	va_start(args);
#else
	va_start(args, fmt);
#endif /* sun */

	/*
	 *  is debuging switched on ?
	 */

	if ( debug_flag & DEBUG_ON_OFF )
	{
     	   /*
	    *  are the right debug bits set ?
	    */

		if ( (debug_flag & debug_bits) != 0 )
		{
			vsprintf (debug_string,fmt,args);
			/*printf("dev_printdebug(): strlen(fmt) %d strlen(debug_string) %d\n",strlen(fmt),strlen(debug_string));*/

	      		/*
	       		 *  is the message server imported ?
	       		 */

			if ( config_flags.message_server )
			{
				msg_write ( DEBUG_TYPE, debug_string );
				msg_send ( DEBUG_TYPE );
			}
			else
			{
#ifdef _NT
				conout( debug_string );
#else
				printf ( debug_string );
#endif
				msg_clear ( DEBUG_TYPE );
			}
		}
	}

	va_end(args);

}

/*
 * For OS9 version 2.4 and Solaris with g++
 */
#else
/*+**********************************************************************
 Function   :   extern void dev_printdebug()

 Description:   Send debug string to a message service
		or to stdout, if a message service
		is not imported.
            :   A debug string will be send if one of the debug_bits
		related to the string is set in the global debug_flag.
            :   Debug strings can br only 256 characters long.

 Arg(s) In  :	long debug_bits  - debug bits on which to send the
				   information.
	    :   char *fmt        - format string in printf() style.
            :   double a0,...,a9 - reserving memory for variables to print.

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/
void dev_printdebug (long debug_bits,char *fmt, char *str)
{
	char 	debug_string[256];

	/*
	 *  is debuging switched on ?
	 */

	if ( debug_flag & DEBUG_ON_OFF )
	{
		/*
	    *  are the right debug bits set ?
	    */

		if ( (debug_flag & debug_bits) != NULL )
		{
			if (str != NULL)
			{
				sprintf (debug_string,fmt,str);
			}
			else
			{
				sprintf (debug_string,fmt);
			}

			/*
	       *  is the message server imported ?
	       */

			if ( config_flags.message_server )
			{
				msg_write ( DEBUG_TYPE, debug_string );
				msg_send ( DEBUG_TYPE );
			}
			else
			{
#ifdef _NT
				conout ( debug_string );
#else
				printf ( debug_string );
#endif
				msg_clear ( DEBUG_TYPE );
			}
		}
	}

}
#endif /* OSK */



/*
 * Attention: OS9 version 2.4 can not handle variable argument
 * lists. It will still use the old tricky function!!
 */
#if (!defined OSK) && (!defined _GNU_PP)

/*+**********************************************************************
 Function   :   extern void dev_printdiag()

 Description:   Send device server diagnostic_string to a message 
		service or to stdout, if a message service
		is not imported.
            :   Diagnostic strings can be only 256 characters long.
		Longer texts can be send by storing short
		strings in the message buffer with the WRITE
		mode and send the whole buffer by using
		the SEND mode with the last string.

 Arg(s) In  :	short mode       - mode for message buffer : WRITE or SEND
	    :   char *fmt        - format string in printf() style.
            :   double a0,...,a9 - reserving memory for variables to print.

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

_DLLFunc void dev_printdiag (DevShort mode, char *fmt, ...)
{
	char  		buffer [256];
	va_list		args;

	if ( mode == CLEAR )
	{
		msg_clear ( DIAG_TYPE );
		return;
	}
	/*
 * get variable argument list pointer in order to pass it to vsprintf()
 */
#if (defined sun) || (defined irix)
	va_start(args);
#else
	va_start(args, fmt);
#endif /* sun */

	/*
 * vsprintf() gives a core dump at present (17sep94) replace it with
 * a simple sprintf() and ignore the variable arguments for the moment
 *
 	sprintf (buffer,fmt);
 */
	vsprintf (buffer,fmt,args);

	va_end(args);

	msg_write ( DIAG_TYPE, buffer );

	if ( mode != WRITE )
	{
		/*
            * send messages to message server if imported
            */

		if ( config_flags.message_server )
		{
			msg_send ( DIAG_TYPE );
		}
		else
		{
#ifdef _NT
			char msg[1024];
			sprintf (msg,"%s", message_buffer[DIAG_TYPE].messages);
			conout(msg);
#else
			printf ("%s", message_buffer[DIAG_TYPE].messages);
#endif
			msg_clear ( DIAG_TYPE );
		}
	}
}

/*
 * For OS9 version 2.4 and Solaris with g++
 *
 * NOTE : these platforms have difficulty with the variable arguments 
 *        therefore only a reduced form of dev_printdiag() is supported
 *        i.e. with only a single argument to print (char *fmt)
 *
 *        andy 5aug99
 */
#else

/**/
/*+**********************************************************************
 Function   :   extern void dev_printdiag()

 Description:   Send device server diagnostic_string to a message 
		service or to stdout, if a message service
		is not imported.
            :   Diagnostic strings can be only 256 characters long.
		Longer texts can be send by storing short
		strings in the message buffer with the WRITE
		mode and send the whole buffer by using
		the SEND mode with the last string.

 Arg(s) In  :	short mode       - mode for message buffer : WRITE or SEND
	    :   char *fmt        - format string in printf() style.

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/
void dev_printdiag (DevShort mode,char *fmt,char *str)
{
	char  		buffer [256];

	if ( mode == CLEAR )
	{
		msg_clear ( DIAG_TYPE );
		return;
	}

	if (str != NULL)
	{
		sprintf (buffer,fmt,str);
	}
	else
	{
		sprintf (buffer,fmt);
	}

	msg_write ( DIAG_TYPE, buffer );

	if ( mode != WRITE )
	{
		/*
            * send messages to message server if imported
            */

		if ( config_flags.message_server )
		{
			msg_send ( DIAG_TYPE );
		}
		else
		{
#ifdef _NT
			char msg[1024];
			sprintf (masg,"%s", message_buffer[DIAG_TYPE].messages);
			conout(msg);
#else
			printf ("%s", message_buffer[DIAG_TYPE].messages);
#endif
			msg_clear ( DIAG_TYPE );
		}
	}
}
#endif /* OSK */


/*+**********************************************************************
 Function   :	static void msg_write()

 Description:   writing messages into the message buffer  

 Arg(s) In  :   int msg_type     - type of message : error or debug
	    :	char *msg_string - message string

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

static void msg_write (_Int msg_type, char *msg_string)
{
	char 	*help;
	int  	len;
	int   	i;
	int		pid = 0;
	static short init_flg = 0;

	/*
   * initialise all message buffer
   */

	if ( init_flg == 0 )
	{
		for ( i=0; i<NUMBER_OF_MSG_TYPES; i++ )
		{
			message_buffer[i].init_flg = 0;
			message_buffer[i].nbytes = 0;
			message_buffer[i].messages = NULL;
		}
		init_flg = 1;
	}

	/*
    * allocate space for the first message
    */

	if ( message_buffer[msg_type].init_flg == 0 )
	{
		len = strlen(msg_string) + 1;
		if ( (message_buffer[msg_type].messages = (char *)malloc (len)) == NULL )
		{
#ifdef _NT
			conout("msg_write() : Insufficient memory for allocation !");
#else
			printf ("msg_write() : Insufficient memory for allocation !");
#endif
#if !defined (_NT)
#if !defined (vxworks)
			pid = getpid ();
#else  /* !vxworks */
			pid = taskIdSelf ();
#endif /* !vxworks */
			kill (pid,SIGQUIT);
#else
			raise(SIGTERM);
#endif	/* _NT */
		}

		message_buffer[msg_type].nbytes = len;
		strcpy ( message_buffer[msg_type].messages , msg_string);
		message_buffer[msg_type].init_flg = 1;
	}

	else
		/*
       * reallocate space for further messages
       */

		{
			len = message_buffer[msg_type].nbytes + strlen (msg_string);
			help = message_buffer[msg_type].messages;

			if ((help=(char *)realloc(message_buffer[msg_type].messages,len)) == NULL)
			{
#ifdef _NT
				{
				char msg[1024];
				conout("msg_write() : Insufficient memory for reallocation !");
				sprintf(msg,"message_buffer contents :\n%s",
				    message_buffer[msg_type].messages );
				conout(msg);
				}
#else
				printf("msg_write() : Insufficient memory for reallocation !");
				printf ("message_buffer contents :\n%s",
				    message_buffer[msg_type].messages );
#endif
#if !defined (_NT)
#if !defined (vxworks)
				pid = getpid ();
#else  /* !vxworks */
				pid = taskIdSelf ();
#endif /* !vxworks */
				kill (pid,SIGQUIT);
#else
				raise(SIGTERM);
#endif /* _NT */
			}

			message_buffer[msg_type].nbytes = len;
			message_buffer[msg_type].messages = help;
			strcat ( message_buffer[msg_type].messages , msg_string);
		}

}


/*+**********************************************************************
 Function   :	static void msg_send()

 Description:   sends the contents of the message buffer
		to the message server or stdout, if no
		message service is imported.

 Arg(s) In  :   int msg_type     - type of message : error or debug

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

static void msg_send (_Int msg_type)
{
	_msg_data 		msg_data;
	_msg_out		msg_out;
	enum clnt_stat	clnt_stat;
	int			pid = 0;


	if ( message_buffer[msg_type].init_flg == 1 )
	{
		/*
       * initialise message structur
       */

		msg_data.devserver_name  = msg_info.DS_name;
		msg_data.host_name       = msg_info.DS_host;
		msg_data.prog_number     = msg_info.DS_prog_number;
		msg_data.display         = msg_info.DS_display;
		msg_data.type_of_message = msg_type;
		msg_data.message_buffer  = message_buffer[msg_type].messages;

		/*
       * send message structur to message server
       */

		clnt_stat = clnt_call (msg_info.conf->clnt,RPC_MSG_SEND,
		    (xdrproc_t)xdr__msg_data, (caddr_t) &msg_data,
		    (xdrproc_t)xdr__msg_out, (caddr_t) &msg_out, TIMEVAL(timeout));
		if (clnt_stat != RPC_SUCCESS)
		{
#ifdef _NT
			{
			char msg[1024];
			conout(clnt_sperror (msg_info.conf->clnt,"msg_send()"));
			sprintf(msg, "message_buffer contents :\n%s",
			    message_buffer[msg_type].messages );
			conout(msg);
			}
#else
			clnt_perror (msg_info.conf->clnt,"msg_send()");
			printf ("message_buffer contents :\n%s",
			    message_buffer[msg_type].messages );
#endif

			/*
	  *  exit server if message server connection fails
	  */

#if !defined (_NT)
#if !defined (vxworks)
			pid = getpid ();
#else  /* !vxworks */
			pid = taskIdSelf ();
#endif /* !vxworks */
			kill (pid,SIGQUIT);
#else
			raise(SIGTERM);
#endif /* _NT */
			return;
		}

		if (msg_out.status < 0)
		{
			if (msg_type == ERROR_TYPE)
			{
#ifdef _NT
				{
				char msg[1024];
				sprintf (msg,"Message Server : error number %d",msg_out.error);
				conout(msg);
				sprintf (msg,"message_buffer contents :\n%s",
				    message_buffer[msg_type].messages );
				conout(msg);
				}
#else
				printf ("Message Server : error number %d",msg_out.error);
				printf ("message_buffer contents :\n%s",
				    message_buffer[msg_type].messages );
#endif

				/*
	     *  exit server if message server fails to
	     *  process an error message
	     */

#if !defined (_NT)
#if !defined (vxworks)
				pid = getpid ();
#else  /* !vxworks */
                        	pid = taskIdSelf ();
#endif /* !vxworks */
				kill (pid,SIGQUIT);
#else
				raise(SIGTERM);
#endif /* _NT */
				return;
			}

			dev_printerror_no (WRITE,"Message Server : ",msg_out.error);
			dev_printerror (SEND,"%s","msg_send : Message transfer failed");
			return;
		}

		/*
       * free message buffer
       */

		msg_clear ( msg_type );
	}
}


/*+**********************************************************************
 Function   :	static void msg_clear()

 Description:   frees the contents of the message buffer

 Arg(s) In  :   int msg_type     - type of message : error or debug

 Arg(s) Out :   none

 Return(s)  :   none
***********************************************************************-*/

static void msg_clear (_Int msg_type)
{
	if (message_buffer[msg_type].messages != NULL) 
		free ( message_buffer[msg_type].messages );
	message_buffer[msg_type].init_flg = 0;
	message_buffer[msg_type].nbytes = 0;
	message_buffer[msg_type].messages = NULL;
}


/*+**********************************************************************
 Function   :	extern long db_import()

 Description:   import the static database service

 Arg(s) In  :   none

 Arg(s) Out :   long *error - Will contain an appropriate error
			      code if the corresponding call
		    	      returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long _DLLFunc db_import (long *error)
{
	CLIENT		*clnt;
        enum clnt_stat  clnt_stat;
	char		*hstring;

	*error = 0;

	/*
	 * check wether a database server is already imported
	 */

	if (config_flags.database_server)
		return (DS_OK);

	/*
	 * check wether the system is already configured
	 */

	if ( !config_flags.configuration )
	{
		if ( (setup_config (error)) < 0 )
			return (DS_NOTOK);
	}


	/*
	 * Create message server client handle with data from
	 * global dbserver_info structure.
	 */

	/* Create a client handle for version 3! */
	clnt = clnt_create ( db_info.conf->server_host,
	    		     db_info.conf->prog_number,
	    		     DB_VERS_3,
	    		     "udp");
	if (clnt == NULL)
	{
		hstring = clnt_spcreateerror ("db_import");
		dev_printerror (SEND,"%s",hstring);
		*error = DevErr_DbImportFailed;
		return (DS_NOTOK);
	}

	clnt_control (clnt, CLSET_RETRY_TIMEOUT, (char *) &dbase_retry_timeout);
	clnt_control (clnt, CLSET_TIMEOUT, (char *) &dbase_timeout);

	/*
	 * Check the database server version. 
	 * If it is not a new version 3 server, create a handle to
	 * version 2!!!
	 */ 

        clnt_stat = clnt_call (clnt, NULLPROC, (xdrproc_t)xdr_void, NULL, 
			       (xdrproc_t)xdr_void, NULL, TIMEVAL(timeout));
        if (clnt_stat != RPC_SUCCESS)
           {
           if ( clnt_stat != RPC_PROGVERSMISMATCH )
              {
              clnt_perror (clnt,"db_import()");
              clnt_destroy (clnt);
              *error = DevErr_DbImportFailed;
              return (DS_NOTOK);
              }

           /*
            * If it was an old version 2 of the database server,
            * a version mismatch occured because the client handle
            * was created for version 3.
            * Destroy the handle and use version 2.
            */
	   else
	      {
              /*
               * Destroy version 3 handle.
               */
              clnt_destroy (clnt);

              /*
               * Set version number to 2 and recreate the
               * client handle.
               */

	      clnt = clnt_create ( db_info.conf->server_host,
	    		           db_info.conf->prog_number,
	    		           DB_VERS_2,
	    		           "udp");
	      if (clnt == NULL)
	         {
		 hstring = clnt_spcreateerror ("db_import()");
		 dev_printerror (SEND,"%s",hstring);
		 *error = DevErr_DbImportFailed;
		 return (DS_NOTOK);
	         }

	      db_info.conf->vers_number = DB_VERS_2;
	      }
	   }
	else
	   {
	   db_info.conf->vers_number = DB_VERS_3;
	   }


	/* 
 	 * pass the information to the database server_info structure
 	 */

	sprintf (db_info.conf->device_name,"DatabaseServer");
	db_info.conf->clnt = clnt;
	db_info.conf->ds_id = 0;
	db_info.conf->no_svr_conn = 1;

	config_flags.database_server = True;

/*
 * for multi-nethost support copy the default configuration
 * and database info to multi_nethost[0] in case one of the 
 * api routines makes a reference to it.
 */
	multi_nethost[0].config_flags = config_flags;
	multi_nethost[0].db_info = db_info.conf;

	return (DS_OK);
}


/*+**********************************************************************
 Function   :	extern long db_import_multi()

 Description:   import the static database service for a multi-nethost

 Arg(s) In  :   nethost - which nethost to import the static database
		          for

 Arg(s) Out :   long *error - Will contain an appropriate error
			      code if the corresponding call
		    	      returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/
long _DLLFunc db_import_multi (char *nethost, long *error)
{
	CLIENT		*clnt;
        enum clnt_stat  clnt_stat;
	char		*hstring;
	nethost_info	*nethost_i;
	long 		i_nethost;

	*error = 0;

	/*
	 * identify the nethost in the array of multi-nethosts
	 */
	if ((i_nethost = get_i_nethost_by_name(nethost,error)) < 0)
	{
        /*
         * if not try to configure the new nethost
         */
                if ( (setup_config_multi (nethost,error)) < 0 )
                        return (DS_NOTOK);
		i_nethost = get_i_nethost_by_name(nethost,error);
	}
	/* 
	 * which nethost to import the database from
	 */
	nethost_i = &(multi_nethost[i_nethost]);

	/*
	 * check wether a database server is already imported
	 */

	if (nethost_i->config_flags.database_server)
		return (DS_OK);

	/*
	 * Create message server client handle with data from
	 * global dbserver_info structure.
	 */

	/* Create a client handle for version 3! */
	clnt = clnt_create ( nethost_i->db_info->server_host,
	    		     nethost_i->db_info->prog_number,
	    		     DB_VERS_3,
	    		     "udp");
	if (clnt == NULL)
	{
		hstring = clnt_spcreateerror ("db_import");
		dev_printerror (SEND,"%s",hstring);
		*error = DevErr_DbImportFailed;
		return (DS_NOTOK);
	}

	clnt_control (clnt, CLSET_RETRY_TIMEOUT, (char *) &dbase_retry_timeout);
	clnt_control (clnt, CLSET_TIMEOUT, (char *) &dbase_timeout);

	/*
	 * Check the database server version. 
	 * If it is not a new version 3 server, create a handle to
	 * version 2!!!
	 */ 

        clnt_stat = clnt_call (clnt, NULLPROC, (xdrproc_t)xdr_void, NULL, 
			       (xdrproc_t)xdr_void, NULL, TIMEVAL(timeout));
        if (clnt_stat != RPC_SUCCESS)
           {
           if ( clnt_stat != RPC_PROGVERSMISMATCH )
              {
              clnt_perror (clnt,"db_import()");
              clnt_destroy (clnt);
              *error = DevErr_DbImportFailed;
              return (DS_NOTOK);
              }

           /*
            * If it was an old version 2 of the database server,
            * a version mismatch occured because the client handle
            * was created for version 3.
            * Destroy the handle and use version 2.
            */
	   else
	      {
              /*
               * Destroy version 3 handle.
               */
              clnt_destroy (clnt);

              /*
               * Set version number to 2 and recreate the
               * client handle.
               */

	      clnt = clnt_create ( nethost_i->db_info->server_host,
	    		           nethost_i->db_info->prog_number,
	    		           DB_VERS_2,
	    		           "udp");
	      if (clnt == NULL)
	         {
		 hstring = clnt_spcreateerror ("db_import()");
		 dev_printerror (SEND,"%s",hstring);
		 *error = DevErr_DbImportFailed;
		 return (DS_NOTOK);
	         }

	      nethost_i->db_info->vers_number = DB_VERS_2;
	      }
	   }
	else
	   {
	   nethost_i->db_info->vers_number = DB_VERS_3;
	   }


	/* 
 	 * pass the information to the database server_info structure
 	 */

	sprintf (nethost_i->db_info->device_name,"DatabaseServer");
	nethost_i->db_info->clnt = clnt;
	nethost_i->db_info->ds_id = 0;
	nethost_i->db_info->no_svr_conn = 1;

	nethost_i->config_flags.database_server = True;

	if (i_nethost == 0)
	{
/*
 * if this is the first nethost (i_nethost=0) then add mono-nethost 
 * support by calling db_import to import the "default" nethost
 */
		if (!config_flags.database_server)
		{
			if (db_import(error) != DS_OK)
			{
				return(DS_NOTOK);
			}
		}
	}

	return (DS_OK);
}


/*+**********************************************************************
 Function   :	static long setup_config()

 Description:   gets the necessary configuration information
		for a static database service and a message
		service from  a network manager.
            :   The host of the network manager must be 
	        specified by the environment variable NETHOST.    

 Arg(s) In  :   none

 Arg(s) Out :   long *error - Will contain an appropriate error
			      code if the corresponding call
		    	      returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

static long setup_config (long *error)
{
	char				nethost[SHORT_NAME_SIZE], *nethost_env;
	CLIENT				*clnt;
	enum clnt_stat			clnt_stat;
	int				pid;
	static _manager_data 		manager_data;
	static _register_data  		register_data;
	static char			host_name[SHORT_NAME_SIZE];
	static struct _devserver	msg_ds, db_ds;

#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\nsetup_config() : entering routine\n");
#endif /* EBUG */

	*error = 0;
	memset ((char *)&manager_data,0,sizeof(manager_data));

	/*
 	 *  read environmental variable NETHOST
  	 */

	if ( (nethost_env = (char *)getenv ("NETHOST")) == NULL )
	{
		*error = DevErr_NethostNotDefined;
		return (-1);
	}
	sprintf(nethost, "%s",nethost_env);

	/*
	 *  create registration information that is send to
	 *  the network manager and stored in the System.log file.
	 */

	gethostname (host_name,32);

#if !defined (_NT)
#if !defined (vxworks)
        pid = getpid ();
#else  /* !vxworks */
        pid = taskIdSelf ();
#endif /* !vxworks */
#else  /* !_NT */
        pid = _getpid ();
#endif /* !_NT */
	register_data.host_name   = host_name;
	register_data.prog_number = pid;
	register_data.vers_number = 0;

	/*
	 * Create network manager client handle 
	 */

	clnt = clnt_create ( nethost,
	    NMSERVER_PROG,
	    NMSERVER_VERS,
	    "udp");
	if (clnt == NULL)
	{
		clnt_pcreateerror ("setup_config");
		*error = DevErr_NoNetworkManagerAvailable;
		return (-1);
	}

	clnt_control (clnt, CLSET_RETRY_TIMEOUT, (char *) &api_retry_timeout);
	clnt_control (clnt, CLSET_TIMEOUT, (char *) &api_timeout);

	/*
       	 *   get configuration information from
       	 *   a network manager
       	 */

	clnt_stat = clnt_call (clnt, RPC_GET_CONFIG,
	    (xdrproc_t)xdr__register_data, (caddr_t) &register_data,
	    (xdrproc_t)xdr__manager_data, (caddr_t) &manager_data, TIMEVAL(timeout));
	if (clnt_stat != RPC_SUCCESS)
	{
		if ( clnt_stat != RPC_PROGVERSMISMATCH )
		{
			clnt_perror (clnt,"setup_config ()");
			clnt_destroy (clnt);
			*error = DevErr_NetworkManagerNotResponding;
			return (-1);
		}

	   /*
	    * If it was an old version 1 of the manager process,
	    * a version mismatch occured because the client handle
	    * was created for version 4.
	    * Destroy the handle and use version 1.
	    */
		else
		{
	/*
         * Destroy version 4 handle.
               */
	clnt_destroy (clnt);

	/*
               * Set version number to 1 and recreate the
               * client handle.
               */

	clnt = clnt_create (nethost,NMSERVER_PROG,NMSERVER_VERS_1,"udp");
	if (clnt == NULL)
	{
		clnt_pcreateerror ("setup_config");
		*error = DevErr_NoNetworkManagerAvailable;
		return (DS_NOTOK);
	}

	/*
       	       *   get configuration information from a network manager
	       *   running version 1.
       	       */

	clnt_stat = clnt_call (clnt, RPC_GET_CONFIG,
	    (xdrproc_t)xdr__register_data, (caddr_t) &register_data,
	    (xdrproc_t)xdr__manager_data_3, (caddr_t) &manager_data, TIMEVAL(timeout));
	if (clnt_stat != RPC_SUCCESS)
	{
		clnt_perror (clnt,"setup_config ()");
		clnt_destroy (clnt);
		*error = DevErr_NetworkManagerNotResponding;
		return (-1);
	}
}
	}


	if (manager_data.status < 0)
	{
		*error = manager_data.error;
/*
 * free memory allocated by xdr in manager_data (assume we have connected
 * to version 4 of the Manager
 */
		xdr_free((xdrproc_t)xdr__manager_data, (char *)&manager_data);
		
		clnt_destroy (clnt);
		return (-1);
	}


	/*
 	 * Initialise the XDR data type list with all data types
	 * specified in the Kernel of the system.
	 */

	if ( xdr_load_kernel (error) == DS_NOTOK )
	{
/*
 * free memory allocated by xdr in manager_data (assume we have connected
 * to version 4 of the Manager
 */
		xdr_free((xdrproc_t)xdr__manager_data, (char *)&manager_data);
		clnt_destroy (clnt);
		return (DS_NOTOK);
	}

	/*
	 *  put message server and database server configuration
	 *  into the global structures msgserver_info and
	 *  dbserver_info.
	 */

/*
 * do not allocate space for _devserver structure because this routine
 * could be called multiple times e.g. when database server has to be
 * reimported, simply point to a static _devserver structure which
 * is locally allocated in static space
 */
	msg_info.conf=(devserver)&msg_ds;
	db_info.conf =(devserver)&db_ds;

	sprintf (msg_info.conf->server_host, "%s", 
	    manager_data.msg_info.host_name);
	msg_info.conf->prog_number =  manager_data.msg_info.prog_number;
	msg_info.conf->vers_number =  manager_data.msg_info.vers_number;

	sprintf (db_info.conf->server_host, "%s", 
	    manager_data.db_info.host_name);
	db_info.conf->prog_number =  manager_data.db_info.prog_number;
	db_info.conf->vers_number =  manager_data.db_info.vers_number;

	/*
	 * Enable the security system if the Manager indicates 
	 * the use of the security system.
	 */

	if (manager_data.security == True)
	{
		config_flags.security =  True;
	}

/*
 * free memory allocated by xdr in manager_data (assume we have connected
 * to version 4 of the Manager
 */
	xdr_free((xdrproc_t)xdr__manager_data, (char *)&manager_data);
	clnt_destroy (clnt);
	config_flags.configuration = True;
/*
 * for multi-nethost support copy the default configuration,
 * database and message info to multi_nethost[0] in case one of the 
 * api routines makes a reference to it.
 *
 * first allocate space for a minimum no. of nethosts
 */
	if (max_nethost <= 0) nethost_alloc(error);

	strcpy(multi_nethost[0].nethost,nethost);
	multi_nethost[0].config_flags = config_flags;
	multi_nethost[0].db_info = db_info.conf;
	multi_nethost[0].msg_info = msg_info.conf;

	return (DS_OK);
}


/*struct _devserver msg_ds[MIN_NETHOST], db_ds[MIN_NETHOST];*/
struct _devserver *msg_ds, *db_ds;


/*+**********************************************************************
 Function   :	static long setup_config_multi()

 Description:   gets the necessary configuration information
		for a static database service and a message
		service from  a network manager in a multi-nethost
		environment.
            :   The host of the network manager is specified
	        by the nethost_name parameter.    

 Arg(s) In  :   nethost_name - name of nethost to configure

 Arg(s) Out :   long *error - Will contain an appropriate error
			      code if the corresponding call
		    	      returns a non-zero value.

 Return(s)  :   DS_OK or DS_NOTOK
***********************************************************************-*/

long setup_config_multi (char *nethost, long *error)
{
	_manager_data 		manager_data;
	_register_data  	register_data;
	static char		host_name[SHORT_NAME_SIZE];
	CLIENT			*clnt;
	enum clnt_stat		clnt_stat;
	int			pid;
	long			i_nethost, i;
#ifndef _UCC
	static char		nethost_env[80];
#endif

	*error = 0;
	memset ((char *)&manager_data,0,sizeof(manager_data));

#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
		    "\nsetup_config_multi() : entering routine\n");
#endif /* EBUG */

	/*
 	 *  read environmental variable NETHOST
  	 */

	if ( nethost == NULL )
	{
		*error = DevErr_NethostNotDefined;
		return (DS_NOTOK);
	}

	/*
	 * look for the nethost in the array of multi_nethosts[] and
	 * return its index; if the nethost is not found then return the
	 * index of the next free slot in the array
	 */

	if ((i_nethost = get_i_nethost_by_name(nethost,error)) < 0)
	{
/*
 * nethost is not defined in the list of imported nethosts, add it
 * to the next free slot in multi_nethosts
 *
 * but first check to see whether the $NETHOST environment variable is
 * defined. If so then set it up first as default nethost by calling 
 * setup_config(). The $NETHOST environment variable has precedance 
 * over the nethost defined in the device name
 *
 * only call setup_config() if it hasn't been called already - andy 19nov98
 */
		if ((char *)getenv("NETHOST") != NULL)
		{
			if ( !config_flags.configuration )
			{
				if ( (setup_config (error)) != DS_OK )
					return (DS_NOTOK);
			}
		}
		for (i=0; i< max_nethost; i++)
		{
			if (!multi_nethost[i].config_flags.configuration)
			{
				break;
			}
		}
		i_nethost = i;
#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
		    "\nsetup_config_multi() : add nethost %d\n",i_nethost);
#endif /* EBUG */
/*
 * if this is the first time a nethost is being defined or the nethost
 * then table is full therefore allocate space for another MIN_NETHOST 
 * nethosts i.e. there is no limit on the maximumu number of nethosts
 * 
 * andy 30/6/98
 */
		if ((i == 0) || (i >= max_nethost))
		{
			nethost_alloc(error);
		}
 	}
	/*
	 *  create registration information to be sent to
	 *  the network manager and stored in the System.log file.
	 */

	gethostname (host_name,32);

#if !defined (_NT)
#if !defined (vxworks)
        pid = getpid ();
#else  /* !vxworks */
        pid = taskIdSelf ();
#endif /* !vxworks */
#else /* !_NT */
	pid = _getpid ();
#endif	/* !_NT */
	register_data.host_name   = host_name;
	register_data.prog_number = pid;
	register_data.vers_number = 0;

	/*
	 * Create network manager client handle 
	 */

	clnt = clnt_create ( nethost,
	    NMSERVER_PROG,
	    NMSERVER_VERS,
	    "udp");
	if (clnt == NULL)
	{
		clnt_pcreateerror ("setup_config");
		*error = DevErr_NoNetworkManagerAvailable;
		return (-1);
	}

	clnt_control (clnt, CLSET_RETRY_TIMEOUT, (char *) &api_retry_timeout);
	clnt_control (clnt, CLSET_TIMEOUT, (char *) &api_timeout);

	/*
       	 *   get configuration information from
       	 *   a network manager
       	 */

	clnt_stat = clnt_call (clnt, RPC_GET_CONFIG,
	    (xdrproc_t)xdr__register_data, (caddr_t) &register_data,
	    (xdrproc_t)xdr__manager_data, (caddr_t) &manager_data, TIMEVAL(timeout));
	if (clnt_stat != RPC_SUCCESS)
	{
		if ( clnt_stat != RPC_PROGVERSMISMATCH )
		{
			clnt_perror (clnt,"setup_config ()");
			clnt_destroy (clnt);
			*error = DevErr_NetworkManagerNotResponding;
			return (-1);
		}

	   /*
	    * If it was an old version 1 of the manager process,
	    * a version mismatch occured because the client handle
	    * was created for version 4.
	    * Destroy the handle and use version 1.
	    */
		else
		{
	/*
         * Destroy version 4 handle.
               */
	clnt_destroy (clnt);

	/*
               * Set version number to 1 and recreate the
               * client handle.
               */

	clnt = clnt_create (nethost,NMSERVER_PROG,NMSERVER_VERS_1,"udp");
	if (clnt == NULL)
	{
		clnt_pcreateerror ("setup_config");
		*error = DevErr_NoNetworkManagerAvailable;
		return (DS_NOTOK);
	}

	/*
       	       *   get configuration information from a network manager
	       *   running version 1.
       	       */

	clnt_stat = clnt_call (clnt, RPC_GET_CONFIG,
	    (xdrproc_t)xdr__register_data, (caddr_t) &register_data,
	    (xdrproc_t)xdr__manager_data_3, (caddr_t) &manager_data, TIMEVAL(timeout));
	if (clnt_stat != RPC_SUCCESS)
	{
		clnt_perror (clnt,"setup_config ()");
		clnt_destroy (clnt);
		*error = DevErr_NetworkManagerNotResponding;
		return (-1);
	}
}
	}


	if (manager_data.status < 0)
	{
		*error = manager_data.error;
/*
 * free memory allocated by xdr in manager_data (assume we have connected
 * to version 4 of the Manager
 */
		xdr_free((xdrproc_t)xdr__manager_data, (char *)&manager_data);
		clnt_destroy (clnt);
		return (-1);
	}


	/*
	 *  put message server and database server configuration
	 *  into the global structures msgserver_info and
	 *  dbserver_info.
	 */

/*
 * do not allocate space for _devserver structure because this routine
 * could be called multiple times e.g. when database server has to be
 * reimported, simply point to a static _devserver structure which
 * is always allocated
 */
	multi_nethost[i_nethost].msg_info = (devserver)&msg_ds[i_nethost];
	multi_nethost[i_nethost].db_info = (devserver)&db_ds[i_nethost];

	sprintf (multi_nethost[i_nethost].msg_info->server_host, "%s", 
	    manager_data.msg_info.host_name);
	multi_nethost[i_nethost].msg_info->prog_number =  
                                 manager_data.msg_info.prog_number;
	multi_nethost[i_nethost].msg_info->vers_number =  
                                 manager_data.msg_info.vers_number;

	sprintf (multi_nethost[i_nethost].db_info->server_host, "%s", 
	    manager_data.db_info.host_name);
	multi_nethost[i_nethost].db_info->prog_number =  
                                 manager_data.db_info.prog_number;
	multi_nethost[i_nethost].db_info->vers_number =  
                                 manager_data.db_info.vers_number;

	/*
	 * Enable the security system if the Manager indicates 
	 * the use of the security system.
	 */

	if (manager_data.security == True)
	{
		multi_nethost[i_nethost].config_flags.security =  True;
	}
	else
	{
		multi_nethost[i_nethost].config_flags.security =  False;
	}

/*
 * free memory allocated by xdr in manager_data (assume we have connected
 * to version 4 of the Manager
 */
	xdr_free((xdrproc_t)xdr__manager_data, (char *)&manager_data);
	clnt_destroy (clnt);
	strcpy(multi_nethost[i_nethost].nethost,nethost);
/*
	printf( "setup_config_multi(): configured nethost[%d] %s\n",
	        i_nethost,multi_nethost[i_nethost].nethost);
 */
	multi_nethost[i_nethost].config_flags.configuration = True;

	if (i_nethost == 0)
	{
/*
 * if this is the first nethost (i_nethost=0) then add mono-nethost 
 * support by calling setup_config to initialise the default nethost
 * i.e. the present nethost
 */
		if (!config_flags.configuration)
		{
/* 
 * Ultra-C++ does not find the symbole putenv() (maybe the prototype
 * is wrong). Supress its use for Ultra-C++. This means Ultra-C++
 * programs have to specify $NETHOST
 *
 * - andy 27nov96
 */
#ifndef _UCC
			sprintf(nethost_env,"NETHOST=%s",nethost);
			putenv(nethost_env);
#endif /*!_UCC*/
			if (setup_config(error) != DS_OK)
			{
				return(DS_NOTOK);
			}
		}
	}

	return (DS_OK);
}


/*+**********************************************************************
 Function   :	static long get_error_string()

 Description:   Read the error string from the global error table
		or from the resource database.
		The rsource name is:
		ERROR/team_no/server_no/error_ident:
                DS_WARNING is returned, if the function was
                executed correctly, but no error
                string was found in the database.

 Arg(s) In  :   long error    - error number

 Arg(s) Out :   char *error_str - error string.

 Return(s)  :   DS_OK or DS_NOTOK or DS_WARNING
***********************************************************************-*/

static long get_error_string (long error, char *error_str)
{
	char				res_path[LONG_NAME_SIZE];
	char				res_name[SHORT_NAME_SIZE];
	char				*ret_str = NULL;
	db_resource 	res_tab;
	long				db_error;
	unsigned short 	error_number_mask = 0xfff;
	unsigned short 	team;
	unsigned short 	server;
	unsigned short 	error_ident;
	short		i;

	time_t		sec;
	char		*time_stamp;


#ifdef EBUG
	dev_printdebug (DBG_TRACE | DBG_API,
	    "\nget_error_string() : entering routine\n");
#endif /* EBUG */

	db_error = 0;

/*
 * Get the time stamp
 */
	time (&sec);
	time_stamp = ctime (&sec);
	time_stamp[24] = '\0';

/*
 * Negative errors are not supported, return an appropriate message
 * and a return the value DS_WARNING.
 */
        if ( error < 0 )
        {
                sprintf ( error_str,
                          "%s  No error message stored in the database for negative errors (%d)",
                          time_stamp, error );
                return (DS_WARNING);
        }

/*
 * Decode the error number into the fields:
 * team, server and error_ident.
 */
	team   = (_Int)(error >> DS_TEAM_SHIFT);
	team   = team & DS_TEAM_MASK;
	server = (_Int)(error >> DS_IDENT_SHIFT);
	server = server & DS_IDENT_MASK;
	error_ident = (_Int)(error & error_number_mask);


/*
 * Search the global error table first, if the error number
 * indicates team = 0 and server = 0.
 */

	if ( team == 0 && server == 0 )
	{
		for (i=0; i<(MAX_DEVERR); i++)
		{
			if ((_Int)(DevErr_List[i].dev_errno) == (_Int)error)
			{
				sprintf ( error_str, "%s  %s", time_stamp, 
				    DevErr_List[i].message );
				return (DS_OK);
			}
		}
	}

/*
 * Create the resource path and the resource structure.
 */

	sprintf (res_path, "ERROR/%d/%d", team, server);
	sprintf (res_name, "%d", error_ident);
#ifdef EBUG
	dev_printdebug (DBG_API,
	    "get_error_string() : res_path = %s\n", res_path);
	dev_printdebug (DBG_API,
	    "get_error_string() : res_name = %s\n", res_name);
#endif /* EBUG */

	res_tab.resource_name = res_name;
	res_tab.resource_type = D_STRING_TYPE;
	res_tab.resource_adr  = &ret_str;

/*
 * Read the error string from the database.
 */

	if (db_getresource (res_path, &res_tab, 1, &db_error) == DS_NOTOK)
	{
#ifdef EBUG
		dev_printdebug (DBG_API | DBG_ERROR,
		    "get_error_string() : db_getresource failed with error %d\n",
		    db_error);
#endif /* EBUG */

	   /*
 	    * If the database call returned an error, search the global
	    * error table for a description.
	    */

		for (i=0; i<(MAX_DEVERR); i++)
		{
			if ((_Int)(DevErr_List[i].dev_errno) == (_Int)db_error)
			{
				sprintf ( error_str, "%s  %s", time_stamp, 
				    DevErr_List[i].message );
				return (DS_NOTOK);
			}
		}

		sprintf ( error_str, "%s  db_getresource() failed with error (%d)", 
		    time_stamp, db_error );
		return (DS_NOTOK);
	}

	/*
 * If the variable ret_str is still NULL, no resource value was found
 * in the database, but the function was executed without error.
 * In this case return the value DS_WARNING.
 */
	if ( ret_str == NULL )
	{
		sprintf ( error_str, 
		    "%s  No error message stored in the database for error (%d/%d/%d)", 
		    time_stamp, team, server, error_ident );
		return (DS_WARNING);
	}

	/*
 * Found the error string in the database. Return the string with 
 * a time stamp.
 */
	sprintf ( error_str, "%s  %s", time_stamp, ret_str );
	free (ret_str);
	return (DS_OK);
}
