static char RcsId[] = "@(#)$Header: /segfs/dbase/res/clnt/RCS/browse_cli.c,v 6.9 1999/07/08 15:44:27 goetz Rel goetz $";

/*+*******************************************************************

 File       :   browse_cli.c

 Project    :   Static database

 Description:   Application Programmers Interface

            :   Interface to access static database for all its browsnig
                calls

 Author(s)  :   Emmanuel Taurel

 Original   :   December 1997


 * $Log: browse_cli.c,v $
 * Revision 6.9  1999/07/08 15:44:27  goetz
 * added J.Quick's patch to fix reconnection bug on Linux
 *
 * Revision 6.8  1999/06/07 15:48:14  goetz
 * replace name with name_copy in db_dev_import()
 *
 * Revision 6.7  1999/02/22 17:17:23  taurel
 * Fix another bug in reconnection. This bug increases open f. for db_putresource and db_getresource
 *
 * Revision 6.6  99/02/04  12:51:21  12:51:21  taurel (E.Taurel)
 * Add a clnt_destroy in reconnect function to avoid mem. and open file leak
 * 
 * Revision 6.5  98/12/01  13:26:51  13:26:51  taurel (E.Taurel)
 * Added call to init resource cache
 * 
 * Revision 6.4  98/10/26  16:33:48  16:33:48  taurel (E.Taurel)
 * Fix a bug in the db_delreslist call (setacc_cli.c file)
 * 
 * Revision 6.3  98/10/12  16:10:38  16:10:38  taurel (E.Taurel)
 * Added a new call db_delreslist. Reconnection in its own file. Fix miscellaneous small bugs
 * 
 * Revision 6.2  98/09/01  16:21:23  16:21:23  taurel (E.Taurel)
 * Adapted to any number of nethost
 * 
 * Revision 6.1  98/08/21  08:51:59  08:51:59  taurel (E.Taurel)
 * Fix a little bug in tools_cli.c file.
 * 
 * Revision 6.0  98/08/14  08:55:04  08:55:04  taurel (E.Taurel)
 * Added many calls to server in order to have all utilities server oriented
 * 
 
 *-*******************************************************************/

#define PORTMAP

#include <macros.h>
#include <db_setup.h>
#include <db_xdr.h>

#if defined(_NT)
#include <API.h>
#include <ApiP.h>
#include <DevErrors.h>
#include <nm_rpc.h>

#else
#include <API.h>
#include <private/ApiP.h>
#include <DevErrors.h>

#ifdef _OSK
#include <inet/socket.h>
#include <inet/netdb.h>
#include <strings.h>
#else
#include <string.h>
#include <sys/socket.h>
#ifndef vxworks
#include <netdb.h>
#else
#include <hostLib.h>
#include <taskLib.h>
#endif
#include <unistd.h>
#endif /* _OSK */
#endif	/* _NT */

#ifndef OSK
#include <stdlib.h>
#endif

#include <math.h>

#ifdef ALONE
extern CLIENT *cl;
extern int first;
#else
extern dbserver_info db_info;
#endif /* ALONE */


/* Static and Global variables */

static struct timeval timeout_browse={60,0};




/****************************************************************************
*                                                                           *
*		db_getdevdomainlist function code                           *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of devices  domain  name	    *
*                                                                           *
*    Argout : - p_domain_nb : Pointer for domain number			    *
*	      - ppp_list : Pointer for the domain name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getdevdomainlist(long *p_domain_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j;
	long error;
	long nb_domain;
	long exit_code = 0;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_domain_nb == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_domain_nb = 0;
	*p_error = 0;
			
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;
#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */


/* Call server */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif /* ALONE */

	recev = db_getdevdomain_1(local_cl,&error);

/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)NULL,(void **)&recev,&local_cl,
					(int)DB_GETDEVDOMAIN,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getdevdomain_1(local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Initialize domain name number */

	nb_domain = recev->res_val.arr1_len;
	*p_domain_nb = nb_domain;

/* Allocate memory for domain names array and copy them */

	if ((*ppp_list = (char **)calloc(nb_domain,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		exit_code = -1;
	}
	else 
	{	
		for (i = 0;i < nb_domain;i++)
		{
			if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
			{
				for (j = 0;j < i;j++)
					free((*ppp_list)[j]);
				free(*ppp_list);
				*p_error = DbErr_ClientMemoryAllocation;
				exit_code = -1;
				break;
			}
			else
				strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getdevfamilylist function code                           *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of devices family name when the   *
*		     domain name is known			    	    *
*                                                                           *
*    Argins : - domain : The domain name				    *
*                                                                           *
*    Argout : - p_family_nb : Pointer for family number			    *
*	      - ppp_list : Pointer for the family name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getdevfamilylist(char *domain,long *p_family_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_family;
	char *dom_sent;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_family_nb == NULL) || (domain == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_family_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;

#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	k = strlen(domain);
	if ((dom_sent = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(dom_sent,domain);
	for(i = 0;i < k;i++)
		dom_sent[i] = tolower(dom_sent[i]);

/* Call server */

	recev = db_getdevfamily_1(&dom_sent,local_cl,&error);

/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&dom_sent,(void **)&recev,&local_cl,
					(int)DB_GETDEVFAMILY,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getdevfamily_1(&dom_sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}

/* Free memory used to send data to server */

	free(dom_sent);
		
/* Initialize family name number */

	nb_family = recev->res_val.arr1_len;
	*p_family_nb = nb_family;

/* Allocate memory for domain names array and copy them */

	if (nb_family != 0)
	{
		if ((*ppp_list = (char **)calloc(nb_family,sizeof(char *))) == NULL)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = DbErr_ClientMemoryAllocation;
			exit_code = -1;
		}
		else 
		{	
			for (i = 0;i < nb_family;i++)
			{
				if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
				{
					for (j = 0;j < i;j++)
						free((*ppp_list)[j]);
					free(*ppp_list);
					*p_error = DbErr_ClientMemoryAllocation;
					exit_code = -1;
					break;
				}
				else
					strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
			}
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getdevmemberlist function code                           *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of devices member name when the   *
*		     domain and family names are known		    	    *
*                                                                           *
*    Argins : - domain : The domain name				    *
*	      - family : The family name				    *
*                                                                           *
*    Argout : - p_member_nb : Pointer for member number			    *
*	      - ppp_list : Pointer for the member name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getdevmemberlist(char *domain,char *family,long *p_member_nb, \
			 char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_member;
	db_res sent;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_member_nb == NULL) || 
	    (domain == NULL) || (family == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_member_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;

#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	sent.db_err = 0;
	sent.res_val.arr1_len = 2;
	
	if ((sent.res_val.arr1_val = (char **)calloc(2,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	
	k = strlen(domain);
	if ((sent.res_val.arr1_val[0] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[0],domain);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[0][i] = tolower(sent.res_val.arr1_val[0][i]);

	k = strlen(family);
	if ((sent.res_val.arr1_val[1] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[1],family);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[1][i] = tolower(sent.res_val.arr1_val[1][i]);
		
/* Call server */

	recev = db_getdevmember_1(&sent,local_cl,&error);	

/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&sent,(void **)&recev,&local_cl,
					(int)DB_GETDEVMEMBER,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getdevmember_1(&sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Free memory used to send data to server */

	free(sent.res_val.arr1_val[0]);
	free(sent.res_val.arr1_val[1]);
	free(sent.res_val.arr1_val);	
		
/* Initialize family name number */

	nb_member = recev->res_val.arr1_len;
	*p_member_nb = nb_member;

/* Allocate memory for domain names array and copy them */

	if (nb_member != 0)
	{
		if ((*ppp_list = (char **)calloc(nb_member,sizeof(char *))) == NULL)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = DbErr_ClientMemoryAllocation;
			exit_code = -1;
		}
		else 
		{	
			for (i = 0;i < nb_member;i++)
			{
				if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
				{
					for (j = 0;j < i;j++)
						free((*ppp_list)[j]);
					free(*ppp_list);
					*p_error = DbErr_ClientMemoryAllocation;
					exit_code = -1;
					break;
				}
				else
					strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
			}
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getresdomainlist function code                           *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of resources  domain  name	    *
*                                                                           *
*    Argout : - p_domain_nb : Pointer for domain number			    *
*	      - ppp_list : Pointer for the domain name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getresdomainlist(long *p_domain_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j;
	long error;
	long nb_domain;
	long exit_code = 0;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_domain_nb == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_domain_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;

#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Call server */

	recev = db_getresdomain_1(local_cl,&error);	

/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)NULL,(void **)&recev,&local_cl,
					(int)DB_GETRESDOMAIN,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getresdomain_1(local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Initialize domain name number */

	nb_domain = recev->res_val.arr1_len;
	*p_domain_nb = nb_domain;

/* Allocate memory for domain names array and copy them */

	if ((*ppp_list = (char **)calloc(nb_domain,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		exit_code = -1;
	}
	else 
	{	
		for (i = 0;i < nb_domain;i++)
		{
			if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
			{
				for (j = 0;j < i;j++)
					free((*ppp_list)[j]);
				free(*ppp_list);
				*p_error = DbErr_ClientMemoryAllocation;
				exit_code = -1;
				break;
			}
			else
				strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getresfamilylist function code                           *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of resources family name when the *
*		     domain name is known			    	    *
*                                                                           *
*    Argins : - domain : The domain name				    *
*                                                                           *
*    Argout : - p_family_nb : Pointer for family number			    *
*	      - ppp_list : Pointer for the family name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getresfamilylist(char *domain,long *p_family_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_family;
	char *dom_sent;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_family_nb == NULL) || (domain == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_family_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;

#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	k = strlen(domain);
	if ((dom_sent = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(dom_sent,domain);
	for(i = 0;i < k;i++)
		dom_sent[i] = tolower(dom_sent[i]);

/* Call server */

	recev = db_getresfamily_1(&dom_sent,local_cl,&error);	

/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&dom_sent,(void **)&recev,&local_cl,
					(int)DB_GETRESFAMILY,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			free(dom_sent);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getresfamily_1(&dom_sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				free(dom_sent);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		free(dom_sent);
		*p_error = recev->db_err;
		return(-1);
	}

/* Free memory used to send data to server */

	free(dom_sent);
		
/* Initialize family name number */

	nb_family = recev->res_val.arr1_len;
	*p_family_nb = nb_family;

/* Allocate memory for domain names array and copy them */

	if ((*ppp_list = (char **)calloc(nb_family,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		exit_code = -1;
	}
	else 
	{	
		for (i = 0;i < nb_family;i++)
		{
			if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
			{
				for (j = 0;j < i;j++)
					free((*ppp_list)[j]);
				free(*ppp_list);
				*p_error = DbErr_ClientMemoryAllocation;
				exit_code = -1;
				break;
			}
			else
				strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getresmemberlist function code                           *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of resources member name when the *
*		     domain and family names are known		    	    *
*                                                                           *
*    Argins : - domain : The domain name				    *
*	      - family : The family name				    *
*                                                                           *
*    Argout : - p_member_nb : Pointer for member number			    *
*	      - ppp_list : Pointer for the member name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getresmemberlist(char *domain,char *family,long *p_member_nb, \
			 char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_member;
	db_res sent;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_member_nb == NULL) || 
	    (domain == NULL) || (family == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_member_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;

#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	sent.db_err = 0;
	sent.res_val.arr1_len = 2;
	
	if ((sent.res_val.arr1_val = (char **)calloc(2,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	
	k = strlen(domain);
	if ((sent.res_val.arr1_val[0] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[0],domain);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[0][i] = tolower(sent.res_val.arr1_val[0][i]);

	k = strlen(family);
	if ((sent.res_val.arr1_val[1] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[1],family);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[1][i] = tolower(sent.res_val.arr1_val[1][i]);
		
/* Call server */

	recev = db_getresmember_1(&sent,local_cl,&error);
	
/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&sent,(void **)&recev,&local_cl,
					(int)DB_GETRESMEMBER,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			free(sent.res_val.arr1_val[0]);
			free(sent.res_val.arr1_val[1]);
			free(sent.res_val.arr1_val);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getresmember_1(&sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				free(sent.res_val.arr1_val[0]);
				free(sent.res_val.arr1_val[1]);
				free(sent.res_val.arr1_val);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		free(sent.res_val.arr1_val[0]);
		free(sent.res_val.arr1_val[1]);
		free(sent.res_val.arr1_val);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Free memory used to send data to server */

	free(sent.res_val.arr1_val[0]);
	free(sent.res_val.arr1_val[1]);
	free(sent.res_val.arr1_val);	
		
/* Initialize family name number */

	nb_member = recev->res_val.arr1_len;
	*p_member_nb = nb_member;

/* Allocate memory for domain names array and copy them */

	if ((*ppp_list = (char **)calloc(nb_member,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		exit_code = -1;
	}
	else 
	{	
		for (i = 0;i < nb_member;i++)
		{
			if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
			{
				for (j = 0;j < i;j++)
					free((*ppp_list)[j]);
				free(*ppp_list);
				*p_error = DbErr_ClientMemoryAllocation;
				exit_code = -1;
				break;
			}
			else
				strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getresresolist function code                             *
*               -------------------                                         *
*                                                                           *
*    Function rule : To retrieve the list of resources name when the 	    *
*		     domain, family and member names are known		    *
*                                                                           *
*    Argins : - domain : The domain name				    *
*	      - family : The family name				    *
*	      - member : The member name				    *
*                                                                           *
*    Argout : - p_reso_nb : Pointer for resources number		    *
*	      - ppp_list : Pointer for the member name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getresresolist(char *domain,char *family,char *member, \
		       long *p_reso_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_reso;
	db_res sent;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_reso_nb == NULL) || 
	    (domain == NULL) || (family == NULL) || (member == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_reso_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;
#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	sent.db_err = 0;
	sent.res_val.arr1_len = 3;
	
	if ((sent.res_val.arr1_val = (char **)calloc(3,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	
	k = strlen(domain);
	if ((sent.res_val.arr1_val[0] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[0],domain);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[0][i] = tolower(sent.res_val.arr1_val[0][i]);

	k = strlen(family);
	if ((sent.res_val.arr1_val[1] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[1],family);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[1][i] = tolower(sent.res_val.arr1_val[1][i]);

	k = strlen(member);
	if ((sent.res_val.arr1_val[2] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[2],member);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[2][i] = tolower(sent.res_val.arr1_val[2][i]);
				
/* Call server */

	recev = db_getresreso_1(&sent,local_cl,&error);	

/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&sent,(void **)&recev,&local_cl,
					(int)DB_GETRESRESO,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			for (i = 0; i < 3;i++)
				free(sent.res_val.arr1_val[i]);
			free(sent.res_val.arr1_val);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getresreso_1(&sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				for (i = 0; i < 3;i++)
					free(sent.res_val.arr1_val[i]);
				free(sent.res_val.arr1_val);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		for (i = 0; i < 3;i++)
			free(sent.res_val.arr1_val[i]);
		free(sent.res_val.arr1_val);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Free memory used to send data to server */

	for (i = 0; i < 3;i++)
	{
		free(sent.res_val.arr1_val[i]);
	}
	free(sent.res_val.arr1_val);	
		
/* Initialize family name number */

	nb_reso = recev->res_val.arr1_len;
	*p_reso_nb = nb_reso;

/* Allocate memory for domain names array and copy them */

	if (nb_reso != 0)
	{
		if ((*ppp_list = (char **)calloc(nb_reso,sizeof(char *))) == NULL)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = DbErr_ClientMemoryAllocation;
			exit_code = -1;
		}
		else 
		{	
			for (i = 0;i < nb_reso;i++)
			{
				if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
				{
					for (j = 0;j < i;j++)
						free((*ppp_list)[j]);
					free(*ppp_list);
					*p_error = DbErr_ClientMemoryAllocation;
					exit_code = -1;
					break;
				}
				else
					strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
			}
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getresresoval function code                              *
*               ----------------                                            *
*                                                                           *
*    Function rule : To retrieve resource value when domain,family,member   *
*		     and resource name are provided. Member and/or resource *
*		     names can be the wildcard "*". Resources are returned  *
*		     in the following syntax:				    *
*			domain/family/member/resource: resource_value	    *
*		     If the wildcard is not used, only one resource value   *
*		     is returned					    *
*		     If the wildcard is used, this call could return many   *
*		     data, use a TCP connection for this call     	    *
*                                                                           *
*    Argins : - domain : The domain name				    *
*	      - family : The family name				    *
*	      - member : The member name				    *
*	      - resource: The resource name				    *
*                                                                           *
*    Argout : - p_rval_nb : Pointer for resources number		    *
*	      - ppp_list : Pointer for the resource value list. Memory is   *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getresresoval(char *domain,char *family,char *member,char *resource, \
		      long *p_rval_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_reso;
	db_res sent;
	CLIENT *local_cl;
	CLIENT *cl_tcp;
	struct sockaddr_in serv_adr;
	int tcp_so;
#ifndef vxworks
	static struct hostent *ht;
#else
	static int host_addr;
#endif
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_rval_nb == NULL) || 
	    (domain == NULL) || (family == NULL) || (member == NULL) ||
	    (resource == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_rval_nb = 0;
	*p_error = 0;

/* Create RPC TCP connection */

#ifdef ALONE
	cl_tcp = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"tcp");
	if (cl_tcp == NULL)
	{
		*p_error = DbErr_CannotCreateClientHandle;
		return(-1);
	}
	clnt_control(cl_tcp,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(cl_tcp,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
#else /* ALONE */			
#ifndef vxworks
	ht = gethostbyname(db_info.conf->server_host);
	if (ht == NULL)
	{
		*p_error = DbErr_CannotCreateClientHandle;
		return(-1);
	}		
#else /* !vxworks */
	host_addr = hostGetByName(db_info.conf->server_host);
	if (host_addr == 0)
	{
		*p_error = DbErr_CannotCreateClientHandle;
		return(-1);
	}
#endif /* !vxworks */


	serv_adr.sin_family = AF_INET;
#ifndef vxworks
	memcpy((void *)(&serv_adr.sin_addr),ht->h_addr,(size_t)ht->h_length);
#else  /* !vxworks */
	memcpy((void *)(&serv_adr.sin_addr),(char*)&host_addr, 4);
#endif /* !vxworks */


	serv_adr.sin_port = 0;
	tcp_so = RPC_ANYSOCK;

	cl_tcp = clnttcp_create(&serv_adr,db_info.conf->prog_number,
				DB_VERS_3,&tcp_so,0,0);

	if (cl_tcp == NULL)
	{
		*p_error = DbErr_CannotCreateClientHandle;
		return(-1);
	}
		
	clnt_control(cl_tcp,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(cl_tcp,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);

#endif /* ALONE */

	local_cl = cl_tcp;

/* Initialize data sent to server */

	sent.db_err = 0;
	sent.res_val.arr1_len = 4;
	
	if ((sent.res_val.arr1_val = (char **)calloc(4,sizeof(char *))) == NULL)
	{
		clnt_destroy(cl_tcp);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	
	k = strlen(domain);
	if ((sent.res_val.arr1_val[0] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_destroy(cl_tcp);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[0],domain);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[0][i] = tolower(sent.res_val.arr1_val[0][i]);

	k = strlen(family);
	if ((sent.res_val.arr1_val[1] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_destroy(cl_tcp);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[1],family);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[1][i] = tolower(sent.res_val.arr1_val[1][i]);

	k = strlen(member);
	if ((sent.res_val.arr1_val[2] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_destroy(cl_tcp);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[2],member);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[2][i] = tolower(sent.res_val.arr1_val[2][i]);
		
	k = strlen(resource);
	if ((sent.res_val.arr1_val[3] = (char *)malloc(k + 1)) == NULL)
	{
		clnt_destroy(cl_tcp);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(sent.res_val.arr1_val[3],resource);
	for(i = 0;i < k;i++)
		sent.res_val.arr1_val[3][i] = tolower(sent.res_val.arr1_val[3][i]);
				
/* Call server */

	recev = db_getresresoval_1(&sent,local_cl,&error);
	
/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&sent,(void **)&recev,&local_cl,
					(int)DB_GETRESRESOVAL,(long)0,
					DB_TCP,&error);
		}
		if (error != 0)
		{
			clnt_destroy(cl_tcp);
			for (i = 0; i < 4;i++)
				free(sent.res_val.arr1_val[i]);
			free(sent.res_val.arr1_val);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getresresoval_1(&sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_destroy(local_cl);
				for (i = 0; i < 4;i++)
					free(sent.res_val.arr1_val[i]);
				free(sent.res_val.arr1_val);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		*p_error = recev->db_err;
		clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);
		clnt_destroy(local_cl);
		for (i = 0; i < 4;i++)
			free(sent.res_val.arr1_val[i]);
		free(sent.res_val.arr1_val);
		return(-1);
	}
	
/* Free memory used to send data to server */

	for (i = 0; i < 4;i++)
	{
		free(sent.res_val.arr1_val[i]);
	}
	free(sent.res_val.arr1_val);	
		
/* Initialize resource value number */

	nb_reso = recev->res_val.arr1_len;
	*p_rval_nb = nb_reso;

/* Allocate memory for domain names array and copy them */

	if (nb_reso != 0)
	{
		if ((*ppp_list = (char **)calloc(nb_reso,sizeof(char *))) == NULL)
		{
			clnt_destroy(local_cl);
			*p_error = DbErr_ClientMemoryAllocation;
			exit_code = -1;
		}
		else 
		{	
			for (i = 0;i < nb_reso;i++)
			{
				if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
				{
					for (j = 0;j < i;j++)
						free((*ppp_list)[j]);
					free(*ppp_list);
					clnt_destroy(local_cl);
					*p_error = DbErr_ClientMemoryAllocation;
					exit_code = -1;
					break;
				}
				else
					strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
			}
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_destroy(local_cl);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getdsserverlist function code                            *
*               ------------------                                          *
*                                                                           *
*    Function rule : To retrieve the list of devices server		    *
*                                                                           *
*    Argout : - p_server_nb : Pointer for server number			    *
*	      - ppp_list : Pointer for the server name list. Memory is      *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getdsserverlist(long *p_server_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j;
	long error;
	long nb_server;
	long exit_code = 0;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_server_nb == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_server_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;
#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Call server */

	recev = db_getdsserver_1(local_cl,&error);
	
/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)NULL,(void **)&recev,&local_cl,
					(int)DB_GETSERVER,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getdsserver_1(local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Initialize domain name number */

	nb_server = recev->res_val.arr1_len;
	*p_server_nb = nb_server;

/* Allocate memory for domain names array and copy them */

	if ((*ppp_list = (char **)calloc(nb_server,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		exit_code = -1;
	}
	else 
	{	
		for (i = 0;i < nb_server;i++)
		{
			if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
			{
				for (j = 0;j < i;j++)
					free((*ppp_list)[j]);
				free(*ppp_list);
				*p_error = DbErr_ClientMemoryAllocation;
				exit_code = -1;
				break;
			}
			else
				strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getdspersnamelist function code                          *
*               --------------------                                        *
*                                                                           *
*    Function rule : To retrieve the list of devices server personal name   *
*		     when the device server name is known		    *
*                                                                           *
*    Argins : - server : The device server name				    *
*                                                                           *
*    Argout : - p_pers_nb : Pointer for personal name number		    *
*	      - ppp_list : Pointer for the personal name list. Memory is    *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getdspersnamelist(char *server,long *p_pers_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j,k;
	long error;
	long exit_code = 0;
	long nb_pers;
	char *serv_sent;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_pers_nb == NULL) || (server == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_pers_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;
#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	k = strlen(server);
	if ((serv_sent = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(serv_sent,server);
	for(i = 0;i < k;i++)
		serv_sent[i] = tolower(serv_sent[i]);

/* Call server */

	recev = db_getdspers_1(&serv_sent,local_cl,&error);
	
/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)&serv_sent,(void **)&recev,&local_cl,
					(int)DB_GETPERS,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getdspers_1(&serv_sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}

/* Free memory used to send data to server */

	free(serv_sent);
		
/* Initialize family name number */

	nb_pers = recev->res_val.arr1_len;
	*p_pers_nb = nb_pers;

/* Allocate memory for domain names array and copy them */

	if (nb_pers != 0)
	{
		if ((*ppp_list = (char **)calloc(nb_pers,sizeof(char *))) == NULL)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = DbErr_ClientMemoryAllocation;
			exit_code = -1;
		}
		else 
		{	
			for (i = 0;i < nb_pers;i++)
			{
				if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
				{
					for (j = 0;j < i;j++)
						free((*ppp_list)[j]);
					free(*ppp_list);
					*p_error = DbErr_ClientMemoryAllocation;
					exit_code = -1;
					break;
				}
				else
					strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
			}
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}




/****************************************************************************
*                                                                           *
*		db_gethostlist function code                                *
*               --------------                                              *
*                                                                           *
*    Function rule : To retrieve the list of host where device server should*
*		     run						    *
*                                                                           *
*    Argout : - p_host_nb : Pointer for host number			    *
*	      - ppp_list : Pointer for the host name list. Memory is        *
*			   allocated by this function		    	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_gethostlist(long *p_host_nb,char ***ppp_list,long *p_error)
{
	db_res *recev;
	int i,j;
	long error;
	long nb_host;
	long exit_code = 0;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ppp_list == NULL) || (p_host_nb == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_host_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;
#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Call server */

	recev = db_gethost_1(local_cl,&error);
	
/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)NULL,(void **)&recev,&local_cl,
					(int)DB_GETHOST,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_gethost_1(local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Initialize domain name number */

	nb_host = recev->res_val.arr1_len;
	*p_host_nb = nb_host;

/* Allocate memory for domain names array and copy them */

	if ((*ppp_list = (char **)calloc(nb_host,sizeof(char *))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		exit_code = -1;
	}
	else 
	{	
		for (i = 0;i < nb_host;i++)
		{
			if (((*ppp_list)[i] = (char *)malloc(strlen(recev->res_val.arr1_val[i]) + 1)) == NULL)
			{
				for (j = 0;j < i;j++)
					free((*ppp_list)[j]);
				free(*ppp_list);
				*p_error = DbErr_ClientMemoryAllocation;
				exit_code = -1;
				break;
			}
			else
				strcpy((*ppp_list)[i],recev->res_val.arr1_val[i]);
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_res,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}



/****************************************************************************
*                                                                           *
*		db_getdsonhost function code                                *
*               --------------                                              *
*                                                                           *
*    Function rule : To a list of device server which should run on a host  *
* 									    *
*    Argin : - host : The host name					    *
*									    *                                                                          *
*    Argout : - p_ds_nb : Pointer or the device server list number	    *
*	      - ds_list : Pointer for the ds info structure array	    *
*             - p_error : Pointer for the error code in case of problems    *
*                                                                           *
*    In case of trouble, the function returns -1 and set the variable       *
*    pointed to by "p_error". Otherwise, the function returns 0             *
*                                                                           *
*****************************************************************************/


long db_getdsonhost(char *host,long *p_ds_nb,db_svc **ds_list,long *p_error)
{
	db_svcarray_net *recev;
	char *host_sent;
	int i,j,k;
	long error;
	long nb_ds;
	long exit_code = 0;
	struct timeval old_tout;
	CLIENT *local_cl;
#ifndef _OSK
	struct timeval tout;
#endif /* _OSK */
#ifdef ALONE
	char *serv_name = ALONE_SERVER_HOST;
#endif /* ALONE */

/* Try to verify function parameters */

	if ((ds_list == NULL) || (p_ds_nb == NULL))
	{
		*p_error = DbErr_BadParameters;
		return(-1);
	}

/* Miscellaneous init. */

	*p_ds_nb = 0;
	*p_error = 0;
		
#ifdef ALONE
/* Create RPC connection if it's the first call */

	if (!first)
	{
		cl = clnt_create(serv_name,DB_SETUPPROG,DB_VERS_3,"udp");
		if (cl == NULL)
		{
			*p_error = DbErr_CannotCreateClientHandle;
			return(-1);
		}
		clnt_control(cl,CLSET_TIMEOUT,(char *)&timeout_browse);
		clnt_control(cl,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
		first++;
	}
	old_tout = timeout_browse;
#else
/* Get client time-out value and change it to a larger value more suitable
   for browsing call */
   
   	clnt_control(db_info.conf->clnt,CLGET_TIMEOUT,(char *)&old_tout);
	clnt_control(db_info.conf->clnt,CLSET_TIMEOUT,(char *)&timeout_browse);
	clnt_control(db_info.conf->clnt,CLSET_RETRY_TIMEOUT,(char *)&timeout_browse);
			
#endif /* ALONE */

#ifdef ALONE
	local_cl = cl;
#else
	local_cl = db_info.conf->clnt;
#endif

/* Initialize data sent to server */

	k = strlen(host);
	if ((host_sent = (char *)malloc(k + 1)) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	strcpy(host_sent,host);
	for(i = 0;i < k;i++)
		host_sent[i] = tolower(host_sent[i]);

/* Call server */

	recev = db_getdsonhost_1(&host_sent,local_cl,&error);
	
/* Any problem with server ? If yes and if it is a time-out, try to reconnect
   to a new database server. */

	if(recev == NULL)
	{
		if(error == DevErr_RPCTimedOut)
		{
			to_reconnection((void *)NULL,(void **)&recev,&local_cl,
					(int)DB_GETDSHOST,(long)0,DB_UDP,&error);
		}
		if (error != 0)
		{
			clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
			clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
			*p_error = error;
			return(-1);
		}
	}

/* If the server is not connected to the database (because a database update
   is just going on), sleep a while (20 mS) and redo the call */

	if (recev->db_err == DbErr_DatabaseNotConnected)
	{
		for (i = 0;i < RETRY;i++)
		{

#ifdef _OSK
			tsleep(SLEEP_TIME);
#else
			tout.tv_sec = 0;
			tout.tv_usec = 20000;
			select(0,0,0,0,&tout);
#endif /* _OSK */
			recev = db_getdsonhost_1(&host_sent,local_cl,&error);
			if(recev == NULL)
			{
				clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
				clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
				*p_error = error;
				return(-1);
			}
			if ((recev->db_err == 0) || 
			    (recev->db_err != DbErr_DatabaseNotConnected))
				break;
		}
	}

/* Any problems during database access ? */

	if(recev->db_err != 0)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		*p_error = recev->db_err;
		return(-1);
	}
	
/* Initialize ds number */

	nb_ds = recev->length;
	*p_ds_nb = nb_ds;

/* Allocate memory for domain names array and copy them */

	if ((*ds_list = (db_svc *)calloc(nb_ds,sizeof(db_svc))) == NULL)
	{
		clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
		clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
		clnt_freeres(local_cl,(xdrproc_t)xdr_db_svcarray_net,(char *)recev);
		*p_error = DbErr_ClientMemoryAllocation;
		return(-1);
	}
	else 
	{	
		for (i = 0;i < nb_ds;i++)
		{
			strcpy((*ds_list)[i].server_name,recev->sequence[i].server_name);
			strcpy((*ds_list)[i].personal_name,recev->sequence[i].personal_name);
			strcpy((*ds_list)[i].host_name,recev->sequence[i].host_name);
			(*ds_list)[i].pid = recev->sequence[i].pid;
			(*ds_list)[i].program_num = recev->sequence[i].program_num;
		}
	}
	
/* Free memory allocated by XDR stuff */

	clnt_freeres(local_cl,(xdrproc_t)xdr_db_svcarray_net,(char *)recev);

/* Leave function */

	clnt_control(local_cl,CLSET_TIMEOUT,(char *)&old_tout);
	clnt_control(local_cl,CLSET_RETRY_TIMEOUT,(char *)&old_tout);
	return(exit_code);

}
