/*static char RcsId[] = "$Header: /segfs/dserver/classes/ccd/imagepro/src/RCS/ImagePro.c,v 1.11 2002/05/17 17:02:55 goetz Exp goetz $";*/
/*********************************************************************
 *
 *File:		ImagePro.c
 *
 *Project:	Beamline Control
 *
 *Description:	A device class to control the ImagePro image acquistion
		and analysis program via a device server. ImagePro is a sophisticated
		program for image analysis which has drivers for a number of
		cameras. All the ImagePro functions (known as Auto-Pro functions)
		can be called via the mouse, from a macros or from Visual C++.
		The device server provides remote client access to ImagePro using
		the visual C++ interface. Thereby enabling clients like SPEC to 
		control the beamline motors and send remote commands to ImagePro.

		This device server has originally been developed on ID11 to control
		the Photonics Science and Photometrics cameras. It is now used on
		ID22 and ID18 for controlling the Photonics Science XIOS II camera.
		It can be used to control any camera which has an ImagePro driver.
 *
 *Author(s):	Andy Gotz
 *
 *Original:	22 April 1999
 *
 *$Log: ImagePro.c,v $
 *Revision 1.11  2002/05/17 17:02:55  goetz
 *intermediate checkin with latest version from ID11
 *
 * Revision 1.10  2002/04/25  07:23:19  07:23:19  goetz (Andy Goetz)
 * intermediate checkin
 * 
 *Revision 1.9  2002/02/22 19:40:56  goetz
 *latest version from ID2, supports trigger mode from spec, reads sequences
 *
 *Revision 1.8  2002/01/16 11:55:37  goetz
 *latest version from JCL's lab; tested with SPEC V5
 *
 *Revision 1.6  2001/10/17 19:30:06  goetz
 *latest version from Frelon lab; uses Jean-Luc's functions to snap
 *
 *Revision 1.5  2001/09/07 08:54:27  goetz
 *latest version from id11; provides support for writing EDF format
 *
 *Revision 1.4  2001/07/11 08:47:45  goetz
 *IpWsSaveAs now accepts format as second input argument
 *
 *
 *			13/06/2000 - added IpAcqSnap and IpWsSaveAs commands
 *			             removed trailing zero from IpAcqTimed file name
 *
 *			20/07/2000 - added IpGetArea command
 *
 *Copyright(c) 1999 by European Synchrotron Radiation Facility, 
 *                     Grenoble, France
 *
 *
 *********************************************************************/

#include <API.h>
#include <DevServer.h>
#include <DevErrors.h>
#include <Admin.h>
#include <ApiP.h>
#include <DevServerP.h>
#include <ImageProP.h>
#include <ImagePro.h>
#include <acqopts.h>

#include <winbase.h>
#include <ipc.h>
#include <windows.h>
#include <process.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/timeb.h>

#include <DevCcdCmds.h>
#include <DevCcdStates.h>
#include <DevCcdErrors.h>
/*
 * public methods
 */

static long class_initialise(long *error);
static long object_create(char *name,DevServer *ds_ptr,long *error);
static long object_initialise(ImagePro ds,long *error);
static long state_handler(ImagePro ds,DevCommand cmd,long *error);

static	DevMethodListEntry methods_list[] = {
   	{DevMethodClassInitialise,	class_initialise},
   	{DevMethodInitialise,		object_initialise},
  	{DevMethodCreate,		object_create},
   	{DevMethodStateHandler,		state_handler},
   };


ImageProClassRec imageProClassRec = {
   /* n_methods */        sizeof(methods_list)/sizeof(DevMethodListEntry),
   /* methods_list */     methods_list,
   };

ImageProClass imageProClass = (ImageProClass)&imageProClassRec;

/*
 * public commands
 */

static long dev_status(ImagePro ds,DevVoid * argin,DevString *argout,long *error);
static long dev_state(ImagePro ds,DevVoid *argin,DevLong *argout,long *error);
static long dev_macro_run(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error);
static long dev_macro_stop(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error);
static long dev_macro_load(ImagePro ds,DevString *argin,DevVoid *argout,long *error);
static long dev_acq_snap(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error);
static long dev_acq_control(ImagePro ds,DevVarUShortArray *argin,DevVoid *argout,long *error);
static long dev_acq_timed(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_ws_save_as(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_set_variable(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_get_variable(ImagePro ds,DevString *argin,DevFloat *argout,long *error);
static long dev_get_image_size(ImagePro ds,DevVoid *argin,DevVarShortArray *argout,long *error);
static long dev_get_image(ImagePro ds,DevVoid *argin,DevOpaque *argout,long *error);
static long dev_get_area(ImagePro ds,DevVarLongArray *argin,DevOpaque *argout,long *error);
static long dev_ws_change_description(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_ws_save_as_edf(ImagePro ds,DevString *argin,DevVoid *argout,long *error);
static long dev_reset(ImagePro ds, DevVoid *argin, DevVoid *argout, long *error);
static long dev_pre_snap_macro(ImagePro ds,DevString *argin,DevVoid *argout,long *error);
static long dev_post_snap_macro(ImagePro ds,DevString *argin,DevVoid *argout,long *error);
static long dev_doc_id(ImagePro ds,DevVoid *argin,DevLong *argout,long *error);
static long dev_doc_select(ImagePro ds,DevShort *argin,DevVoid *argout,long *error);
static long dev_get_stats(ImagePro ds,DevVoid *argin,DevVarFloatArray *argout,long *error);
static long dev_check_dir(char *filename, long *error);
static long ip_error(int status, long *error);

/*
 * commands added for compatibility with other SPEC CCD device server
 */

static long dev_ccd_get_state(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_start(ImagePro ds, DevVoid *argin, DevVoid *argout, long *error);
static long dev_ccd_stop(ImagePro ds, DevVoid *argin, DevVoid *argout, long *error);
static long dev_ccd_read(ImagePro ds, DevVarLongArray *argin, DevOpaque *argout, long *error);
static long dev_ccd_set_exposure(ImagePro ds, DevFloat *argin, DevVoid *argout, long *error);
static long dev_ccd_get_exposure(ImagePro ds, DevVoid *argin, DevFloat *argout, long *error);
static long dev_ccd_set_bin(ImagePro ds, DevVarLongArray *argin, DevVoid *argout, long *error);
static long dev_ccd_get_bin(ImagePro ds, DevVoid *argin, DevVarLongArray *argout, long *error);
static long dev_ccd_set_trigger(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);
static long dev_ccd_get_trigger(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_set_file_par(ImagePro ds, DevVarStringArray *argin, DevVoid *argout, long *error);
static long dev_ccd_get_file_par(ImagePro ds, DevVoid *argin, DevVarStringArray *argout, long *error);
static long dev_ccd_write_file(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);
static long dev_ccd_x_size(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_y_size(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_depth(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_set_frames(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);
static long dev_ccd_get_frames(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_set_shutter(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);
static long dev_ccd_get_shutter(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_set_channel(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);
static long dev_ccd_get_channel(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_set_roi(ImagePro ds, DevVarLongArray *argin, DevVoid *argout, long *error);
static long dev_ccd_get_roi(ImagePro ds, DevVoid *argin, DevVarLongArray *argout, long *error);

static	DevCommandListEntry commands_list[] = {
   	{DevState, dev_state, D_VOID_TYPE, D_LONG_TYPE,0,"DevState"},
   	{DevStatus, dev_status, D_VOID_TYPE, D_STRING_TYPE,0,"DevStatus"},
   	{DevIpMacroRun, dev_macro_run, D_STRING_TYPE, D_VOID_TYPE,0,"DevIpMacroRun"},
   	{DevIpMacroStop, dev_macro_stop, D_STRING_TYPE, D_VOID_TYPE,0,"DevIpMacroStop"},
	{DevIpMacroLoad, dev_macro_load, D_STRING_TYPE, D_VOID_TYPE,0,"DevIpMacroLoad"},
	{DevIpAcqControl, dev_acq_control, D_VAR_SHORTARR, D_VOID_TYPE,0,"DevIpAcqControl"},
   	{DevIpAcqSnap, dev_acq_snap, D_VOID_TYPE, D_VOID_TYPE,0,"DevIpAcqSnap"},
   	{DevIpAcqTimed, dev_acq_timed, D_VAR_STRINGARR, D_VOID_TYPE,0,"DevIpAcqTimed"},
	{DevIpWsSaveAs, dev_ws_save_as, D_VAR_STRINGARR, D_VOID_TYPE,0,"DevIpWsSaveAs"},
	{DevIpSetVariable, dev_set_variable, D_VAR_STRINGARR, D_VOID_TYPE,0,"DevIpSetVariable"},
	{DevIpGetVariable, dev_get_variable, D_STRING_TYPE, D_FLOAT_TYPE,0,"DevIpGetVariable"},
	{DevIpGetImageSize, dev_get_image_size, D_VOID_TYPE, D_VAR_SHORTARR,0,"DevIpGetImageSize"},
	{DevIpGetImage, dev_get_image, D_VOID_TYPE, D_OPAQUE_TYPE,0,"DevIpGetImage"},
	{DevIpGetArea, dev_get_area, D_VAR_LONGARR, D_OPAQUE_TYPE,0,"DevIpGetArea"},
	{DevIpWsChangeDescription, dev_ws_change_description, D_VAR_STRINGARR, D_VOID_TYPE,0,"DevIpWsChangeDescription"},
	{DevReset, dev_reset, D_VOID_TYPE, D_VOID_TYPE,0,"DevReset"},
	{DevIpPreSnapMacro, dev_pre_snap_macro, D_STRING_TYPE, D_VOID_TYPE,0,"DevIpPreSnapMacro"},
	{DevIpPostSnapMacro, dev_post_snap_macro, D_STRING_TYPE, D_VOID_TYPE,0,"DevIpPostSnapMacro"},
	{DevIpMacroLoad, dev_macro_load, D_STRING_TYPE, D_VOID_TYPE,0,"DevIpMacroLoad"},
	{DevIpDocId, dev_doc_id, D_VOID_TYPE, D_LONG_TYPE,0,"DevIpDocId"},
	{DevIpDocSelect, dev_doc_select, D_SHORT_TYPE, D_VOID_TYPE,0,"DevIpDocSelect"},
	{DevIpGetStats, dev_get_stats, D_VOID_TYPE, D_VAR_FLOATARR,0,"DevIpGetStats"},

/*
 * commands added for compatibility with other SPEC CCD device servers
 */
	{DevCcdStart, dev_ccd_start, D_VOID_TYPE, D_VOID_TYPE,0,"DevCcdStart"},
	{DevCcdStop, dev_ccd_stop, D_VOID_TYPE, D_VOID_TYPE,0,"DevCcdStop"},
//	{DevCcdRead, dev_ccd_read, D_LONG_TYPE, D_OPAQUE_TYPE},
	{DevCcdRead, dev_ccd_read, D_VAR_LONGARR, D_OPAQUE_TYPE,0,"DevCcdRead"},
	{DevCcdSetExposure, dev_ccd_set_exposure, D_FLOAT_TYPE, D_VOID_TYPE,0,"DevCcdSetExposure"},
	{DevCcdGetExposure, dev_ccd_get_exposure, D_VOID_TYPE, D_FLOAT_TYPE,0,"DevCcdGetExposure"},
	{DevCcdSetBin, dev_ccd_set_bin, D_VAR_LONGARR, D_VOID_TYPE,0,"DevCcdSetBin"},
	{DevCcdGetBin, dev_ccd_get_bin, D_VOID_TYPE, D_VAR_LONGARR,0,"DevCcdGetBin"},
	{DevCcdSetTrigger, dev_ccd_set_trigger, D_LONG_TYPE, D_VOID_TYPE,0,"DevCcdSetTrigger"},
	{DevCcdGetTrigger, dev_ccd_get_trigger, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdGetTrigger"},
	{DevCcdSetFilePar, dev_ccd_set_file_par, D_VAR_STRINGARR, D_VOID_TYPE,0,"DevCcdSetFilePar"},
	{DevCcdGetFilePar, dev_ccd_get_file_par, D_VOID_TYPE, D_VAR_STRINGARR,0,"DevCcdGetFilePar"},
	{DevCcdWriteFile, dev_ccd_write_file, D_LONG_TYPE, D_VOID_TYPE,0,"DevCcdWriteFile"},
	{DevCcdXSize, dev_ccd_x_size, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdXSize"},
	{DevCcdYSize, dev_ccd_y_size, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdYSize"},
	{DevCcdDepth, dev_ccd_depth, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdDepth"},
	{DevCcdSetFrames, dev_ccd_set_frames, D_LONG_TYPE, D_VOID_TYPE,0,"DevCcdSetFrames"},
	{DevCcdGetFrames, dev_ccd_get_frames, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdGetFrames"},
	{DevCcdSetShutter, dev_ccd_set_shutter, D_LONG_TYPE, D_VOID_TYPE,0,"DevCcdSetShutter"},
	{DevCcdGetShutter, dev_ccd_get_shutter, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdGetShutter"},
	{DevCcdSetChannel, dev_ccd_set_channel, D_LONG_TYPE, D_VOID_TYPE,0,"DevCcdSetChannel"},
	{DevCcdGetChannel, dev_ccd_get_channel, D_VOID_TYPE, D_LONG_TYPE,0,"DevCcdGetChannel"},
	{DevCcdSetRoI, dev_ccd_set_roi, D_VAR_LONGARR, D_VOID_TYPE,0,"DevCcdSetRoI"},
	{DevCcdGetRoI, dev_ccd_get_roi, D_VOID_TYPE, D_VAR_LONGARR,0,"DevCcdGetRoI"},
};

static long n_commands = sizeof(commands_list)/sizeof(DevCommandListEntry);

/*
 * reserve space for a default copy of the ImagePro object
 */

static ImageProRec imageProRec;
static ImagePro imagePro = (ImagePro)&imageProRec;

/*
 * ImagePro resource tables used to access the static database
 *
 */

db_resource res_object[] = {
   {"simulation", D_LONG_TYPE, NULL},
   {"camera_type", D_LONG_TYPE, NULL},
   	};
int res_object_size = sizeof(res_object)/sizeof(db_resource);


db_resource res_class[] = {
   {"nada",       D_LONG_TYPE, NULL},
   	};
int res_class_size = sizeof(res_class)/sizeof(db_resource);

#define SIMULATION 1
#define CAMERA_STANDARD 	0
#define CAMERA_FRELON2000	1

long    (*IpAcquShow)(long show);
long    (*IpAcquSnap)(void);
long	(*IpAcquRefresh)(void);
long	(*IpAcquGetExposure)(void);
long	(*IpAcquSetExposure)(long exposure);
long	(*IpAcquGetHorizontalBinning)(void);
long	(*IpAcquSetHorizontalBinning)(long hbin);
long	(*IpAcquGetVerticalBinning)(void);
long	(*IpAcquSetVerticalBinning)(long vbin);
long	(*IpAcquGetRoi)(long* x1, long* y1, long *x2, long* y2);
long	(*IpAcquSetRoi)(long x1, long y1, long x2, long y2);
long	(*IpAcquGetNbImage)(void);
long	(*IpAcquSetNbImage)(long nb_image);
long	(*IpAcquGetTimingMode)(void);
long	(*IpAcquSetTimingMode)(long timing_mode);
long	(*IpAcquGetWindowMode)(void);
long	(*IpAcquSetWindowMode)(long window_mode);
long	(*IpAcquGetChannel)(void);
long	(*IpAcquSetChannel)(long channel);
long	(*IpAcquGetShutter)(void);
long	(*IpAcquSetShutter)(long shutter);
long	(*IpAcquGetShutterTime)(void);
long	(*IpAcquSetShutterTime)(long shutter_time);
long	(*IpAcquGetHddMode)(void);
long	(*IpAcquSetHddMode)(long hdd_out);
char* 	(*IpAcquGetHddFile)(void);
long	(*IpAcquSetHddFile)(char *file);
char* 	(*IpAcquGetHddDescr)(void);
long	(*IpAcquSetHddDescr)(char *description);
long	(*IpAcquGetLatency)(void);
long	(*IpAcquSetLatency)(long latency);
long	(*IpAcquGetTimeUnit)(void);
long	(*IpAcquSetTimeUnit)(long time_unit);
long	(*IpAcquGetAntiBlooming)(void);
long	(*IpAcquSetAntiBlooming)(long anti_blooming);
long	(*IpAcquGetTestImage)(void);
long	(*IpAcquSetTestImage)(long test_image);
long	(*IpAcquGetRingBuff)(void);
long	(*IpAcquSetRingBuff)(long ring_buff);
long	(*IpAcquGetCameraMode)(void);
long	(*IpAcquSetCameraMode)(long camera_mode);
long	(*IpAcquGetReset)(void);
long	(*IpAcquSetRefresh)(void);
long	(*IpAcquGetSerial)(void);


                                              
/*======================================================================
 Function:      static long class_initialise()

 Description:	routine to be called the first time a device is 
 		created which belongs to this class (or is a subclass
		thereof. This routine will be called only once.

 Arg(s) In:	none

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
 =======================================================================*/

static long class_initialise(long *error)
{
/*
 * ImageProClass is a subclass of the DevServerClass
 */

	HINSTANCE acqdll;

   imageProClass->devserver_class.superclass = devServerClass;
   imageProClass->devserver_class.class_name = (char*)malloc(sizeof("ImageProClass")+1);
   sprintf(imageProClass->devserver_class.class_name,"ImageProClass");
   imageProClass->devserver_class.class_inited = 1;
   imageProClass->devserver_class.n_commands = n_commands;
   imageProClass->devserver_class.commands_list = commands_list;

/*
 * initialise ImagePro with default values. these will be used
 * for every ImagePro object created.
 */

   imagePro->devserver.class_pointer = (DevServerClass)imageProClass;
   imagePro->imagepro.ccd_state = DEVON;


   imagePro->imagepro.cam_type = CAMERA_STANDARD;
   imagePro->imagepro.roix1 = 1;
   imagePro->imagepro.roix2 = 20;
   imagePro->imagepro.roiy1 = 1;
   imagePro->imagepro.roiy2 = 15;
   imagePro->imagepro.hbin = 1;
   imagePro->imagepro.vbin = 1;
   imagePro->imagepro.description[0] = 0;
   imagePro->imagepro.pre_snap_macro = NULL;
   imagePro->imagepro.post_snap_macro = NULL;
/*
 * Interrogate the static database for default values
 *
 */

   res_class[0].resource_adr	= &(imagePro->devserver.n_state);

   if(db_getresource("CLASS/ImagePro/DEFAULT", res_class, res_class_size, error))
   {
      /*return(DS_NOTOK);*/
   }

/*
 * ImagePro must be running when the AutoPro functions are executed
 *
   WORD wReturn = WinExec("c:\\ipwin3\\ipwin16",SW_SHOWNORMAL);

   dprintf("class_initialise(): WinExec(ipwin16.exe) returned %d\n",wReturn);
 */

/*
 * get frelon functions from sdk 
 */
#ifndef IPP45
   acqdll = LoadLibrary("C:\\IPWIN4\\acquisition.dll");
#else
   acqdll = LoadLibrary("C:\\IPWIN4\\acquisition.dll");
#endif

/* imageProClass->imagepro_class.acqdll = acqdll; */

   if( acqdll )
   {
    	IpAcquShow = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquShow");
        dprintf("class_initialise(): IpAcquShow 0x%08x",IpAcquShow);

    	IpAcquSnap = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquSnap");
        dprintf("class_initialise(): IpAcquSnap 0x%08x",IpAcquSnap);
    	IpAcquRefresh = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquRefresh");
        dprintf("class_initialise(): IpAcquRefresh 0x%08x",IpAcquRefresh);
    	IpAcquGetExposure = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetExposure");
        dprintf("class_initialise(): IpAcquGetExposure 0x%08x",IpAcquGetExposure);
    	IpAcquSetExposure = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetExposure");
        dprintf("class_initialise(): IpAcquSetExposure 0x%08x",IpAcquSetExposure);
    	IpAcquGetHorizontalBinning = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetHorizontalBinning");
        dprintf("class_initialise(): IpAcquGetHorizontalBinning 0x%08x",IpAcquGetHorizontalBinning);
    	IpAcquSetHorizontalBinning = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetHorizontalBinning");
        dprintf("class_initialise(): IpAcquSetHorizontalBinning 0x%08x",IpAcquSetHorizontalBinning);
    	IpAcquGetVerticalBinning = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetVerticalBinning");
        dprintf("class_initialise(): IpAcquGetVerticalBinning 0x%08x",IpAcquGetVerticalBinning);
    	IpAcquSetVerticalBinning = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetVerticalBinning");
        dprintf("class_initialise(): IpAcquSetVerticalBinning 0x%08x",IpAcquSetVerticalBinning);
    	IpAcquGetRoi = (int (__cdecl *)(long*,long*,long*,long*))GetProcAddress(acqdll, "IpAcquGetRoi");
        dprintf("class_initialise(): IpAcquGetRoi 0x%08x",IpAcquGetRoi);
    	IpAcquSetRoi = (int (__cdecl *)(long,long,long,long))GetProcAddress(acqdll, "IpAcquSetRoi");
        dprintf("class_initialise(): IpAcquSetRoi 0x%08x",IpAcquSetRoi);
   		IpAcquGetNbImage = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetNbImage");
        dprintf("class_initialise(): IpAcquGetNbImage 0x%08x",IpAcquGetNbImage);
  		IpAcquSetNbImage = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetNbImage");
        dprintf("class_initialise(): IpAcquSetNbImage 0x%08x",IpAcquSetNbImage);
   		IpAcquGetHddMode = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetHddMode");
        dprintf("class_initialise(): IpAcquGetHddMode 0x%08x",IpAcquGetHddMode);
  		IpAcquSetHddMode = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetHddMode");
        dprintf("class_initialise(): IpAcquSetHddMode 0x%08x",IpAcquSetHddMode);
   		IpAcquGetTimingMode = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetTimingMode");
        dprintf("class_initialise(): IpAcquGetTimingMode 0x%08x",IpAcquGetTimingMode);
  		IpAcquSetTimingMode = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetTimingMode");
        dprintf("class_initialise(): IpAcquSetTimingMode 0x%08x",IpAcquSetTimingMode);
   		IpAcquGetWindowMode = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetWindowMode");
        dprintf("class_initialise(): IpAcquGetWindowMode 0x%08x",IpAcquGetWindowMode);
  		IpAcquSetWindowMode = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetWindowMode");
        dprintf("class_initialise(): IpAcquSetWindowMode 0x%08x",IpAcquSetWindowMode);
   		IpAcquGetShutter = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetShutter");
        dprintf("class_initialise(): IpAcquGetShutter 0x%08x",IpAcquGetShutter);
  		IpAcquSetShutter = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetShutter");
        dprintf("class_initialise(): IpAcquSetShutter 0x%08x",IpAcquSetShutter);
   		IpAcquGetShutterTime = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetShutterTime");
        dprintf("class_initialise(): IpAcquGetShutterTime 0x%08x",IpAcquGetShutterTime);
  		IpAcquSetShutterTime = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetShutterTime");
        dprintf("class_initialise(): IpAcquSetShutterTime 0x%08x",IpAcquSetShutterTime);
   		IpAcquGetHddFile = (char* (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetHddFile");
        dprintf("class_initialise(): IpAcquGetHddFile 0x%08x",IpAcquGetHddFile);
  		IpAcquSetHddFile = (int (__cdecl *)(char*))GetProcAddress(acqdll, "IpAcquSetHddFile");
        dprintf("class_initialise(): IpAcquSetHddFile 0x%08x",IpAcquSetHddFile);
   		IpAcquGetHddDescr = (char* (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetHddDescr");
        dprintf("class_initialise(): IpAcquGetHddDescr 0x%08x",IpAcquGetHddDescr);
  		IpAcquSetHddDescr = (int (__cdecl *)(char*))GetProcAddress(acqdll, "IpAcquSetHddDescr");
        dprintf("class_initialise(): IpAcquSetHddDescr 0x%08x",IpAcquSetHddDescr);
   		IpAcquGetLatency = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetLatency");
        dprintf("class_initialise(): IpAcquGetLatency 0x%08x",IpAcquGetLatency);
  		IpAcquSetLatency = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetLatency");
        dprintf("class_initialise(): IpAcquSetLatency 0x%08x",IpAcquSetLatency);
   		IpAcquGetTimeUnit = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetTimeUnit");
        dprintf("class_initialise(): IpAcquGetTimeUnit 0x%08x",IpAcquGetTimeUnit);
  		IpAcquSetTimeUnit = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetTimeUnit");
        dprintf("class_initialise(): IpAcquSetShutter 0x%08x",IpAcquSetTimeUnit);
   		IpAcquGetAntiBlooming = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetAntiBlooming");
        dprintf("class_initialise(): IpAcquGetAntiBlooming 0x%08x",IpAcquGetAntiBlooming);
  		IpAcquSetAntiBlooming = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetAntiBlooming");
        dprintf("class_initialise(): IpAcquSetAntiBlooming 0x%08x",IpAcquSetAntiBlooming);
   		IpAcquGetTestImage = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetTestImage");
        dprintf("class_initialise(): IpAcquGetShutter 0x%08x",IpAcquGetShutter);
  		IpAcquSetTestImage = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetTestImage");
        dprintf("class_initialise(): IpAcquSetTestImage 0x%08x",IpAcquSetTestImage);
   		IpAcquGetRingBuff = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetRingBuff");
        dprintf("class_initialise(): IpAcquGetRingBuff 0x%08x",IpAcquGetRingBuff);
  		IpAcquSetRingBuff = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetRingBuff");
        dprintf("class_initialise(): IpAcquSetRingBuff 0x%08x",IpAcquSetRingBuff);
   		IpAcquGetCameraMode = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetCameraMode");
        dprintf("class_initialise(): IpAcquGetCameraMode 0x%08x",IpAcquGetCameraMode);
  		IpAcquSetCameraMode = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetCameraMode");
        dprintf("class_initialise(): IpAcquSetCameraMode 0x%08x",IpAcquSetCameraMode);
   		IpAcquGetSerial = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquGetSerial");
        dprintf("class_initialise(): IpAcquGetSerial 0x%08x",IpAcquGetSerial);
  		IpAcquRefresh = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquRefresh");
        dprintf("class_initialise(): IpAcquRefresh 0x%08x",IpAcquRefresh);
   }
   else
   {
      dprintf("class_initialise(): could not load library c:\\IPWIN4\\acquisition.dll");
      dprintf("class_initialise(): therefore POOR SUPPORT for FRELON 1000 and 2000 !");
   }

   return(DS_OK);
}

/*======================================================================
 Function:	static long object_create()

 Description:	routine to be called on creation of a device object

 Arg(s) In:	char *name - name to be given to device

 Arg(s) Out:	DevServer *ds_ptr - pointer to created device
		long *error - pointer to error code, in case routine fails
 =======================================================================*/

static long object_create(char *name,DevServer *ds_ptr,long *error)
{
   ImagePro ds;

   ds = (ImagePro)malloc(sizeof(ImageProRec));

/*
 * initialise device with default object
 */

   *(ImageProRec*)ds = *(ImageProRec*)imagePro;

/*
 * finally initialise the non-default values
 */

   ds->devserver.name = (char*)malloc(strlen(name)+1);
   sprintf(ds->devserver.name,"%s",name);

   *ds_ptr = (DevServer)ds;

   return(DS_OK);
}

/*============================================================================

Function:		static long object_initialise()

Description:	routine to be called on initialisation of a device object

Arg(s) In:		ImagePro ds	- object to initialise

Arg(s) Out:

		long *error     - pointer to error code, in case routine fails
=============================================================================*/
static long object_initialise(ImagePro ds,long *error)
{

   long simulation;

   simulation = ds->imagepro.mode = 0;
   res_object[0].resource_adr        = &simulation;
   res_object[1].resource_adr        = &(ds->imagepro.cam_type);
   if(db_getresource(ds->devserver.name, res_object, res_object_size, error))
   {
   	/*return(DS_NOTOK);*/
   }

   if (simulation) 
   {

	   ds->imagepro.mode = SIMULATION;

	   dprintf("object_initialise(): device is in SIMULATION mode");

   }
   ds->devserver.state = DEVON;
   ds->imagepro.ccd_state = DEVON;
   ds->imagepro.description[0] = 0;

   return(DS_OK);
}

/*======================================================================
 Function:      static long state_handler()

 Description:	this routine is reserved for checking wether the command
				requested can be executed in the present state.

 Arg(s) In:		ImagePro ds - device on which command is to executed
				DevCommand cmd - command to be executed

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
 =======================================================================*/

static long state_handler(ImagePro ds,DevCommand cmd,long *error)
{
   long int p_state, n_state, lcmd;
   long iret = DS_OK;
/*
 * Get here the real state of the physical device
 *
 *  example:
 *  ds->devserver.state=get_state();
 */
   dev_state(ds,NULL,&p_state,error);

   lcmd = cmd;
   p_state = ds->imagepro.ccd_state;
//   dprintf("state_handler(): p_state %d cmd %d ",p_state, lcmd);
/*
 * Before checking out the state machine assume that the state doesn't
 * change i.e. new state == old state
 *
 */

   n_state = p_state;

   switch (p_state)
   {
    
		case (DEVRUN) : 
		{
            dprintf("state_handler(): DEVRUN state ");
			switch (cmd) 
			{
			case (DevReset) :
			case (DevState):
			case (DevStatus): iret = DS_OK;
							  break;
			default :	      iret = -1;
							  (*error) = DevErr_AttemptToViolateStateMachine;
							  break;
			}
		}
		break;

		case (DevCcdInitializing) :
		case (DevCcdTaking) :
		{
            if (p_state == DevCcdInitializing) dprintf("state_handler(): DevCcdInitializing state ");
            if (p_state == DevCcdTaking) dprintf("state_handler(): DevCcdTaking state ");
			switch (cmd) 
			{
			case (DevCcdSetTrigger) : /* why does SPEC resend this after a CcdStart ? */
			case (DevCcdStop) :
			case (DevState):
			case (DevStatus): iret = DS_OK;
							  break;
			default :	      iret = -1;
							  (*error) = DevErr_AttemptToViolateStateMachine;
							  break;
			}
		}
		break; 
		
   		default: iret = DS_OK;
			     /*printf("state_handler(): default state ");*/
   		         break;
   }
   ds->devserver.n_state = n_state;

   return(iret);
}

/*============================================================================
 Function:      static long dev_state()

 Description:	return the state of the device

 Arg(s) In:		ImagePro 	ds 	- 
				DevVoid  	*argin  - none
   				  
 Arg(s) Out:	DevShort	*argout - returned state 
				long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_state(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

#ifdef IPP45
   int frelon_state;
#endif /* IPP45 */


   if (ds->imagepro.mode == SIMULATION)
   {
      dprintf2("dev_state(): SIMULATION mode");

      ds->imagepro.ccd_state = DEVON;
   }

   if (ds->imagepro.cam_type == CAMERA_FRELON2000)
   {

/*
 * interrogating the frelon driver while IPP is exposing is
 * only supported on IPP 4.5, on previous versions it cause
 * the famous error -6
 */
#ifdef IPP45
	   frelon_state = IpAcqControl(FRLCMD_STATE,0,0);
	   dprintf("updated Frelon ccd state %d", frelon_state);

	   switch (frelon_state) {

	   case (0) : ds->imagepro.ccd_state = DEVON;
			      break;

	   case (1) : ds->imagepro.ccd_state = DevCcdInitializing;
		          break;

	   case (2) : ds->imagepro.ccd_state = DevCcdTaking;
		          break;

	   default : dprintf("Frelon state %d not recognised !",frelon_state);

	   }
#endif /* IPP45 */
   }

   *argout	= (long)ds->imagepro.ccd_state;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_status()

 Description:	return state of the device as an ASCII string

 Arg(s) In:		ImagePro 	ds 	- 
				DevVoid  	*argin  - none
   				  
 Arg(s) Out:	DevString	*argout - contains string 
============================================================================*/

static long dev_status(ImagePro ds,DevVoid * argin,DevString *argout,long *error)
{
   static	char	str[1024];
   long				status;
   long			state;

   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * interrogate state
 */
      status = dev_state(ds,NULL,&state,error);
	  if (state == DEVRUN)
	  {
		 sprintf(str,"The camera is exposing");
	  }
	  else
	  {
         if (state == DEVON)
		 {
		    sprintf(str,"The camera has an exposed image");
		 }
	     else
	     {
		    sprintf(str,"The camera is not exposing ");
         }
	  }
   }
   else
   {
      sprintf(str,"The camera is in SIMULATION mode\n");
   }

   *argout = str;
   return (DS_OK);
}

/*============================================================================
 Function:      static long dev_macro_run()

 Description:	start macro

 Arg(s) In:	 ImagePro 	ds 	- 
			 DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevShort	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/
static long dev_macro_run(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
   char *macro_name;
/*
   char *script_file;
 */
   long status;

/*
   macro_name = argin->sequence[0];
   script_file = argin->sequence[1];
 */
   macro_name = *argin;

   dprintf("dev_macro_run(): macro name %s",
	        macro_name);

   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpMacroRun(macro_name,"");
		dprintf("dev_macro_run(): IpMacroRun() returned %d",status);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf2("dev_macro_run(): SIMULATION mode");
	  return(DS_OK);
   }
   return (DS_OK);
   
}

/*============================================================================
 Function:      static long dev_macro_stop()

 Description:	stop image exposure

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_macro_stop(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
   char message[256];
   long status;

   dprintf("dev_macro_stop(): message %s",argin);

   sprintf(message,"IpMacroStop() called from device server (reason=%s)",
           argin);
   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpMacroStop(message,0);
		dprintf("dev_macro_stop(): IpMacroRun() returned %d",status);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf2("dev_macro_stop(): SIMULATION mode");
   }

   return(DS_OK);
}


/*============================================================================
 Function:      static long dev_macro_load()

 Description:	load a macro script file

 Arg(s) In:	 ImagePro 	ds 	- 
			 DevString 	*argin  - macro script file
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/
static long dev_macro_load(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
   long status;

   dprintf("dev_macro_load(): script file %s",
	        *argin);

   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpMacroLoad(*argin);
		dprintf("dev_macro_load(): IpMacroLoad() returned %d",status);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf2("dev_macro_load(): SIMULATION mode");
	  return(DS_OK);
   }
   return (DS_OK);
   
}

/*============================================================================
 Function:      static long dev_doc_id()

 Description:	get active document id macro

 Arg(s) In:	 ImagePro 	ds 	- 
			 DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	 DevLong	*argout - doc id
		 long *error - 	pointer to error code, in case routine fails

============================================================================*/
static long dev_doc_id(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{
   int doc_id;
   long status;

   doc_id = 0;

   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpDocGet(GETACTDOC,0,&doc_id);
		dprintf("dev_doc_id(): IpDocGet() returned %d doc id %d",status,doc_id);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf("dev_doc_id(): SIMULATION mode");
	  return(DS_OK);
   }
   *argout = doc_id;
   return (DS_OK);
   
}

/*============================================================================
 Function:      static long dev_doc_select()

 Description:	set active document 

 Arg(s) In:	 ImagePro 	ds 	- 
		 DevLong  	*argin  - doc id
   				  
 Arg(s) Out: 	 DevVoid	*argout -  none
		 long *error - 	pointer to error code, in case routine fails

============================================================================*/
static long dev_doc_select(ImagePro ds,DevShort *argin,DevVoid *argout,long *error)
{
   short doc_id;
   long status;

   doc_id = *argin;

   dprintf("dev_doc_id(): set id %d to active window",doc_id);
   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpAppSelectDoc(doc_id);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf("dev_doc_id(): SIMULATION mode");
	  return(DS_OK);
   }
   return (DS_OK);
   
}

/*============================================================================
 Function:      static long dev_acq_control()

 Description:	set the acquisition parameters

 Arg(s) In:	ImagePro 	ds 	- 
			DevVarShortArray  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_acq_control(ImagePro ds,DevVarUShortArray *argin,DevVoid *argout,long *error)
{
   short cmd, param;
   long status;


   cmd = argin->sequence[0];
   param = argin->sequence[1];

   dprintf("dev_acq_control(): cmd %d param %d value %d",
	       cmd, param, argin->sequence[2]);
   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * if this is a "SET EXPOSURE" command and this is a FRELON2000 then
 * use the FRELON SDK function IpAcquSetExposure() instead of the 
 * IPAcqControl() function
 */
	  if ((cmd == 49) && (ds->imagepro.cam_type == CAMERA_FRELON2000) && (IpAcquSetExposure != NULL))
	  {
         if (argin->sequence[3] != 0)
		 {
			 ds->imagepro.exposure = argin->sequence[3]*65536 + argin->sequence[2];
		 }
		 else
		 {
			 ds->imagepro.exposure = argin->sequence[2];
		 }
		 status = IpAcquSetExposure(ds->imagepro.exposure);
		 dprintf("dev_acq_control(): FRELON IpAcquSetExposure(%d) returned %d",ds->imagepro.exposure, status);
	  }
	  else
	  {
         status = IpAcqControl(cmd, param,&(argin->sequence[2]));
         dprintf("dev_acq_control(): IpAcqControl() returned %d",status);
         if (status < 0) return (DS_NOTOK);
	  }
   }
   else
   {
      dprintf2("dev_acq_control(): SIMULATION mode");
   }

   return(DS_OK);
}

static char dir[80], prefix[80];
static long startnumber, frames, interval;
static ImagePro ip_acq_ds;

/*============================================================================
 Function:      static long ip_acq_timed()

 Description:	thread to acquire a sequence of frames and write them to disk
				will be called by the dev_acq_timed() command which is now
				non-blocking. the thread will block until it finished and then
				update the device state to indicate the result

 Arg(s) In:	char** 	sequence 	- pointer to sequence of input parameters as strings

 Arg(s) Out:  none

============================================================================*/

void ip_acq_timed(void *argin)
{
   long status;
   static char filename[256];

   dprintf("ip_acq_timed(): dir %s prefix %s start number %d frames %d interval %d",
	       dir,prefix,startnumber,frames,interval);

   ip_acq_ds = (ImagePro)argin;

/*
 * the IpAcqTimed() command does not display the result in a new
 * new workspace. Therefore replace it with a combination of IpAcqSnap(0
 * and IpWsSaveAs() - andy 27mar2000 
 *
   status = IpAcqTimed(dir,prefix,(short)startnumber,
                      (short)frames,(short)interval);
   dprintf("ip_acq_timed(): IpAcqTimed() returned %d",status);
 */

   status = IpAcqSnap(ACQ_NEW);
   dprintf("ip_acq_timed(): IpAcqSnap(ACQ_NEW) returned %d",status);
   sprintf(filename,"%s\\%s.tif",dir,prefix);
   dprintf("ip_acq_timed(): filename = %s",filename);
   status = IpWsSaveAs(filename,"TIF");
   dprintf("ip_acq_timed(): IpWsSaveAs() returned %d",status);

   ip_acq_ds->imagepro.ccd_state = DEVON;

   _endthread();
}

/*============================================================================
 Function:      static long ip_acq_snap()

 Description:	thread to acquire a snap shot in a new workspace.
				will be called by the dev_acq_timed() command which is now
				non-blocking. the thread will block until it finished and then
				update the device state to indicate the result

 Arg(s) In:	char** 	sequence 	- pointer to sequence of input parameters as strings

 Arg(s) Out:  none

============================================================================*/

void ip_acq_snap(void *argin)
{
   long status, elapsed, error;
   static char filename[256];
   struct _timeb before, after;

   ip_acq_ds = (ImagePro)argin;

   if (ip_acq_ds->imagepro.pre_snap_macro != NULL)

   {
	if (strlen(ip_acq_ds->imagepro.pre_snap_macro) > 0)
	{
	   status = dev_macro_run(ip_acq_ds, &(ip_acq_ds->imagepro.pre_snap_macro), NULL, &error);
	   dprintf("ip_acq_snap(): pre-snap macro %s returned %d",ip_acq_ds->imagepro.pre_snap_macro,status);
	}
   }
//   dprintf("ip_acq_snap(): dir %s prefix %s start number %d frames %d interval %d",
//	       dir,prefix,startnumber,frames,interval);
   _ftime(&before);
/*
 * for FRELON2000 type camera use the frelon sdk to snap and not the standard
 * snap
 */
   if ((ip_acq_ds->imagepro.cam_type == CAMERA_FRELON2000) && (IpAcquSnap != NULL))
   {
	    if (ip_acq_ds->imagepro.frames > 1)
		{
		   if (IpAcquSetWindowMode != NULL) status = IpAcquSetWindowMode(3);
		   dprintf("ip_acq_snap(): set window mode to sequence to take %d frames",ip_acq_ds->imagepro.frames);
		}
        status = IpAcquSnap();
        dprintf("ip_acq_snap(): FRELON IpAcquSnap() returned %d",status);
   }
   else
   {
	   if (ip_acq_ds->imagepro.frames > 1)
	   {
		   status = IpAcqMultiSnap(0, (short)ip_acq_ds->imagepro.frames, ACQ_SEQUENCE);
		   dprintf("ip_acq_snap(): IpAcqMultiSnap(0,%d, ACQ_SEQUENCE) returned %d",ip_acq_ds->imagepro.frames,status);
	   }
	   else
	   {
   			status = IpAcqSnap(ACQ_NEW);
			dprintf("ip_acq_snap(): IpAcqSnap(ACQ_NEW) returned %d",status);
	   }
   }

   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;
   dprintf("ip_acq_snap(): time elapsed %d milliseconds",elapsed);

   if (ip_acq_ds->imagepro.post_snap_macro != NULL)
   {
	   if (strlen(ip_acq_ds->imagepro.post_snap_macro) > 0)
	   {
		status = dev_macro_run(ip_acq_ds, &(ip_acq_ds->imagepro.post_snap_macro), NULL, &error);
		dprintf("ip_acq_snap(): post-snap macro %s returned %d",ip_acq_ds->imagepro.post_snap_macro,status);
	   }
   }
   ip_acq_ds->imagepro.ccd_state = DEVON;

   _endthread();
}

/*============================================================================
 Function:      static long dev_acq_snap()

 Description:	acquire a single frame in a new workspace

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_acq_snap(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{


   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * fork thread to do acquisition
 */   
	  dprintf("dev_acq_snap(): fork thread ip_acq_snap() to do acquisition ");
      _beginthread (ip_acq_snap, 0, ds);

	  ds->imagepro.ccd_state = DEVRUN;
      dprintf("dev_acq_snap(): new state %d",ds->imagepro.ccd_state);
   }
   else
   {
      dprintf2("dev_acq_snap(): SIMULATION mode");
   }

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_acq_timed()

 Description:	acquire a single frames and write them to disk

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevShort	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_acq_timed(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{

   if (argin->length < 5)
   {
      *error = DevErr_DeviceIllegalParameter;
	  return (DS_NOTOK);
   }

   sprintf(dir,"%s",argin->sequence[0]);
   sprintf(prefix,"%s",argin->sequence[1]);
   sscanf(argin->sequence[2],"%d",&startnumber);
   sscanf(argin->sequence[3],"%d",&frames);
   sscanf(argin->sequence[4],"%d",&interval);
   ip_acq_ds = ds;

   dprintf("dev_acq_timed(): dir %s prefix %s start number %d frames %d interval %d",
	       dir,prefix,startnumber,frames,interval);

   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * fork thread to do acquisition
 */   
	   if (_stricmp(prefix,"NEW") != 0)
	   {
	      dprintf("dev_acq_timed(): fork thread ip_acq_timed() to do acquisition ");
         _beginthread (ip_acq_timed, 0, ds);
	   }
	   else
	   {
	      dprintf("dev_acq_timed(): fork thread ip_acq_snap() to do acquisition ");
         _beginthread (ip_acq_snap, 0, ds);
	   }

	  ds->imagepro.ccd_state = DEVRUN;
      dprintf("dev_acq_timed(): new state %d",ds->imagepro.ccd_state);
   }
   else
   {
      dprintf2("dev_acq_timed(): SIMULATION mode");
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ws_save_as()

 Description:	save the current workspace to disk

 Arg(s) In:	ImagePro 	ds 	- 
		DevString  	*argin  - file name
   				  
 Arg(s) Out: DevVoid	*argout - none
	     long 	*error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ws_save_as(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{

   char *filename, *format;
   long status;

   filename = argin->sequence[0];
   format = argin->sequence[1];

//   if (dev_check_dir(filename) != DS_OK) return(DS_NOTOK);

   if (stricmp(format, "edf") == 0)
   {
      status = dev_ws_save_as_edf(ds, &filename, NULL, error);
   }
   else
   {
      dprintf("dev_ws_save_as(): filename = %s (format = %s)",filename, format);
      status = IpWsSaveAs(filename,format);
      dprintf("dev_ws_save_as(): IpWsSaveAs() returned %d",status);
   }

   if (status < 0) return(DS_NOTOK);
   
   return(DS_OK);
}

/*============================================================================
 Function:      static long save_as_edf_thread()

 Description:	thread to save an image as edf format
		the thread will save the image in ds->imagepro.image.sequence
		using the header in ds->imagepro.edf_header and the file
		in ds->imagepro.filename. when it is finished it will update
		the par_write_flag to signal that a new thread can be started

 Arg(s) In:	ImagePro ds - ImagePro device object

 Arg(s) Out:  none

============================================================================*/


void save_as_edf_thread(void *argin)
{
   int image_file;
   long status, elapsed;
   struct _timeb before, after;
   ImagePro ds;

   ds = (ImagePro)argin;

   _ftime(&before);
   image_file = _open(ds->imagepro.filename, _O_BINARY | _O_CREAT | _O_RDWR);
   if (image_file <= 0)
   {
	   ds->imagepro.par_write_flag = 0;
      dprintf("save_as_edf_thread(): failed to open file %s, image not saved !",ds->imagepro.filename);
      return;
   }
   status = _write(image_file, ds->imagepro.edf_header, sizeof(ds->imagepro.edf_header));
   status = _write(image_file, ds->imagepro.image.sequence, ds->imagepro.image.length);
   status = _close(image_file);
      
   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;

   dprintf("save_as_edf_thread(): open(), write(), close() took %d milliseconds",elapsed);
   dprintf("save_as_edf_thread(): saved image in EDF format to file %s",ds->imagepro.filename);
   ds->imagepro.par_write_flag = 0;
}

/*============================================================================
 Function:      static long dev_ws_save_as_edf()

 Description:	save the current workspace to disk as EDF format (internal
		ESRF format)

 Arg(s) In:	ImagePro 	ds 	- 
		DevString  	*argin  - file name
   				  
 Arg(s) Out: DevVoid	*argout - none
	     long 	*error - pointer to error code, in case routine fails

============================================================================*/


static long dev_ws_save_as_edf(ImagePro ds,DevString *argin,DevVoid *argout,long *error){

   char error_mess[1024];
   short *image_size;
   DevVarShortArray var_sharr;
   long status, elapsed;
   struct _timeb before, after;
   time_t ltime;
   int image_file;

   _ftime(&before);
   elapsed = 0;
   while (elapsed < 10 && ds->imagepro.par_write_flag != 0) {
	   Sleep(10);
	   _ftime(&after);
	   elapsed = after.time-before.time;
   }
   if (elapsed > 10)
   {
      dprintf("dev_ws_save_as_edf(): timed out waiting for parallel save to finish, image not saved !",ds->imagepro.filename);
      sprintf(error_mess,"Timed out waiting for parallel save to finish, file=%s not saved ! (hint: restart the device server ?)",ds->imagepro.filename);
      dev_error_push(error_mess);
      *error = DevErr_CommandFailed;
      return(DS_NOTOK);
   }
   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;  
   dprintf("ip_ws_save_as_edf(): wait for parallel write to finish took %d ms",elapsed);

   sprintf(ds->imagepro.filename,"%s",*argin); 
   image_file = _open(ds->imagepro.filename, _O_BINARY | _O_CREAT | _O_RDWR, _S_IWRITE);
   if (image_file <= 0)
   {
      dprintf("dev_ws_save_as_edf(): failed to open file %s, image not saved !",ds->imagepro.filename);
      sprintf(error_mess,"Cannot create file %s for image (status=%d), are you sure it doesn't exist already ?",ds->imagepro.filename, image_file);
      dev_error_push(error_mess);
      *error = DevErr_DeviceIllegalParameter;
      return(DS_NOTOK);
   }
   _close(image_file);
/*
 * first get image size
 */
   _ftime(&before);

   status = dev_get_image_size(ds, NULL, &var_sharr, error);
   image_size = var_sharr.sequence;

   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;
   dprintf("ip_ws_save_as_edf(): dev_get_image_size() took %d milliseconds",elapsed);

/*
 * then get pointer to image
 */
   _ftime(&before);

   status = dev_get_image(ds, NULL, &(ds->imagepro.image), error);

   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;
   dprintf("ip_ws_save_as_edf(): dev_get_image() took %d milliseconds",elapsed);
/*
 * create EDF header
 */
   memset(ds->imagepro.edf_header, ' ', 1024); 
   sprintf(ds->imagepro.edf_header, "{\n");
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "HeaderID       = EH:000001:000000:000000 ;\n");
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "Image          = 1 ;\n");
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "ByteOrder      = LowByteFirst ;\n");
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "DataType       = UnsignedShort ;\n" );
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "Dim_1          = %d ;\n", image_size[0]);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "Dim_2          = %d ;\n", image_size[1]);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "Size           = %d ;\n", image_size[0]*image_size[1]*2);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "count_time     = %d ;\n", ds->imagepro.exposure);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "point_no       = %d ;\n", 0);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "preset         = %g ;\n", 1.0);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "col_end        = %d ;\n", image_size[0]-1);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "col_beg        = %d ;\n", 0);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "row_end        = %d ;\n", image_size[1]-1);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "row_beg        = %d ;\n", 0);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "dir            = %s ;\n", ".");
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "suffix         = %s ;\n", "edf");
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "prefix         = %s ;\n", ds->imagepro.filename);
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "run            = %d ;\n", 1);
   ds->imagepro.description[511] = 0;
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "description    = %s ;\n", ds->imagepro.description);


   time( &ltime );
   sprintf(ds->imagepro.edf_header+strlen(ds->imagepro.edf_header), "title          = Frelon CCD with Imagepro on the %s ;\n",ctime( &ltime ));
   dprintf("dev_ws_save_as_edf(): edf_header length %d",strlen(ds->imagepro.edf_header));
/*
   add the dimensions of the image to the header in binary format so IPP can
   automatically detect them using the flat file format
 */
   ds->imagepro.edf_header[1018] = (image_size[0] & 0xff00) >> 8;
   ds->imagepro.edf_header[1019] = (image_size[0] & 0xff);
   ds->imagepro.edf_header[1020] = (image_size[1] & 0xff00) >> 8;
   ds->imagepro.edf_header[1021] = (image_size[1] & 0xff);

   sprintf(ds->imagepro.edf_header+1022,"}\n");

/*
 * start a thread to do the actual writing of the file
 */
   ds->imagepro.par_write_flag = 1;
   _beginthread(save_as_edf_thread,0,ds);
/*
 * old code for writing edf sequentially
 *
      status = _write(image_file, edf_header, sizeof(edf_header));
      status = _write(image_file, image.sequence, image.length);
      status = _close(image_file);
      
      _ftime(&after);
      elapsed = (after.time-before.time)*1000;
      elapsed = (after.millitm-before.millitm) + elapsed;
      dprintf("ip_ws_save_as_edf(): open(), write(), close() took %d milliseconds",elapsed);
      dprintf("dev_ws_save_as_edf(): saved image in EDF format to file %s",filename);
 */
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_check_dir()

 Description:	check if a directory exists and try to create it if not

 Arg(s) In:	char *filename - filename
   				  
 Arg(s) Out: 

============================================================================*/

static long dev_check_dir(char *filename)
{
   char dirname[256], *last_slash;

   sprintf(dirname,"%s",filename);
   last_slash = strrchr(filename,'\\');
   if (last_slash == NULL ) return(DS_OK);
   *last_slash = 0;
/*
 * try to open directory
 *
 * if failed then try to open first directory
 * 
 * if failed then try to create it exit if failed
 *
 * try second directory ...
 */
   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_set_variable()

 Description:	set a variable in MACROS.INI

 Arg(s) In:	ImagePro 	ds 	- 
			DevVarStringArray  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_set_variable(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{
   char *variable;
   float value;
   long status;


   variable = argin->sequence[0];
   sscanf(argin->sequence[1],"%f",&value);

   dprintf("dev_set_variable(): set variable %s to value %f",
	       variable,value);
   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpIniFile(SETFLOAT, variable ,&value);
      dprintf("dev_set_variable(): IpIniFile() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_set_variable(): SIMULATION mode");
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_get_variable()
 Description:	get a variable in MACROS.INI

 Arg(s) In:	ImagePro 	ds 	- 
			DevString  	*argin  - none
   				  
 Arg(s) Out: DevFloat	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_variable(ImagePro ds,DevString *argin,DevFloat *argout,long *error)
{
   char *variable;
   float value;
   long status;


   variable = *argin;

   dprintf("dev_get_variable(): get variable %s",
	       variable);
   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpIniFile(GETFLOAT, variable ,&value);
      dprintf("dev_get_variable(): IpIniFile() returned %d, value = %f",status,value);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_get_variable(): SIMULATION mode");
   }

   *argout = value;

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_get_image_size()

 Description:	get image size
 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - nothing
   				  
 Arg(s) Out: DevShortArray	*argout - image size (rows, columns)
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_image_size(ImagePro ds,DevVoid *argin,DevVarShortArray *argout,long *error)
{
   long status;
   IPDOCINFO info;
   static short image_size[2];


   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
      image_size[1] = info.Height;
      image_size[0] = info.Width;
      dprintf("dev_get_image_size(): IpDocGet() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_get_image_size(): SIMULATION mode");
      image_size[0] = 1024;
	  image_size[1] = 1024;
   }
   dprintf("dev_get_image_size(): rows=%d, columns=%d",
	       image_size[1], image_size[0]);

   argout->length = 2;
   argout->sequence = (short*)image_size;
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_get_image()

 Description:	get whole image 
 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - nothing
   				  
 Arg(s) Out: DevOpaque	*argout - area of image
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_image(ImagePro ds,DevVoid *argin,DevOpaque *argout,long *error)
{
   long status;
   RECT region;
   IPDOCINFO info;
   static short image[2048][2048];
#ifdef DEBUG
   short *pimage;
#endif /* DEBUG */


   if (ds->imagepro.mode != SIMULATION)
   {
	 IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
	 region.left = 0;
	 region.right = info.Width-1;
	 region.top = 0;
	 region.bottom = info.Height-1;
         dprintf("dev_get_image(): get area left=%d, right=%d, top=%d, bottom=%d",
	       region.left,region.right,region.top,region.bottom);
	 if (info.Width > 2048 )
	 {
		 dprintf("dev_get_image(): width %d too big, clip to 2048 !",info.Width);
         region.right = 2048-1;
	 }
	 if (info.Height > 2048 )
	 {
		 dprintf("dev_get_image(): height %d too big, clip to 2048 !",info.Height);
         region.right = 2048-1;
	 }
	 status = IpDocGetArea(DOCSEL_ACTIVE, &region, image, CPROG);
         dprintf("dev_get_image(): IpDocGetArea() returned %d",status);
         if (status < 0) return (DS_NOTOK);
   }
   else
   {
	  region.left = 0;
	  region.right = 1023;
	  region.top = 0;
	  region.bottom = 1023;
      dprintf2("dev_get_image(): SIMULATION mode");
   }

#ifdef DEBUG
   pimage = (short*)image;
   dprintf("dev_get_image(): image[0][0..8] %d %d %d %d %d %d %d %d",
	   *pimage, *(pimage+1), *(pimage+2), *(pimage+3), 
	   *(pimage+4), *(pimage+5), *(pimage+6), *(pimage+7)); 
   pimage = (short*)image+(region.right+1)*(region.bottom);
   dprintf("dev_get_image(): image[%d][0..8] %d %d %d %d %d %d %d %d",region.bottom,
	   *pimage, *(pimage+1), *(pimage+2), *(pimage+3), 
	   *(pimage+4), *(pimage+5), *(pimage+6), *(pimage+7)); 
#endif /* DEBUG */

/*
 * NOTE : hardcoded for 16 bit GRAY level images !
 */
   argout->length = (region.right+1)*(region.bottom+1)*2;
   argout->sequence = (char*)image;
   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_get_area()

 Description:	get a part of and image (part or whole)
 Arg(s) In:	ImagePro 	ds 	- 
			DevVarShortArray  	*argin  - top left and bottom right coordinates
   				  
 Arg(s) Out: DevOpaque	*argout - area of image
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_area(ImagePro ds,DevVarLongArray *argin,DevOpaque *argout,long *error)
{
   long status;
   RECT region;
   IPDOCINFO info;
   static short area[2048][2048];
#ifdef DEBUG
   short *parea;
#endif /* DEBUG */

   region.left = argin->sequence[0];
   region.right = argin->sequence[1];
   region.top = argin->sequence[2];
   region.bottom = argin->sequence[3];

/*
 * if region is (0,0)-(0,0) then return entire image
 */
   if ((region.left == 0) && (region.right == 0) && 
	   (region.top == 0) && (region.bottom == 0) && 
	   (ds->imagepro.mode != SIMULATION))
   {
	   IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
	   region.right = info.Width-1;
	   region.bottom = info.Height-1;
   }
/*
 * order left to right and top to bottom
 */
   else
   {
	   if (region.right < region.left)
	   {
		   region.left = argin->sequence[1];
		   region.right = argin->sequence[0];
	   }
	   if (region.bottom < region.top)
	   {
		   region.top = argin->sequence[3];
		   region.bottom = argin->sequence[2];
	   }
   }

   dprintf("dev_get_area(): get area left=%d, right=%d, top=%d, bottom=%d",
	       region.left,region.right,region.top,region.bottom);
   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpDocGetArea(DOCSEL_ACTIVE, &region, area, CPROG);
      dprintf("dev_getarea(): IpDocGetArea() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_get_area(): SIMULATION mode");
   }

#ifdef DEBUG
   parea = (short*)area;
   dprintf("dev_get_area(): area[%d][0..8] %d %d %d %d %d %d %d %d",region.top,
	   *parea, *(parea+1), *(parea+2), *(parea+3), 
	   *(parea+4), *(parea+5), *(parea+6), *(parea+7)); 
   parea = (short*)area+(region.right-region.left+1)*(region.bottom-region.top);
   dprintf("dev_get_area(): area[%d][0..8] %d %d %d %d %d %d %d %d",region.bottom,
	   *parea, *(parea+1), *(parea+2), *(parea+3), 
	   *(parea+4), *(parea+5), *(parea+6), *(parea+7)); 
#endif /* DEBUG */

   argout->length = (region.right-region.left+1)*(region.bottom-region.top+1)*2;
   argout->sequence = (char*)area;
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_get_stats()

 Description:	get stats for whole image
 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - nothing
   				  
 Arg(s) Out: DevVarFloatArray	*argout - statistics of image
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_stats(ImagePro ds,DevVoid *argin,DevVarFloatArray *argout,long *error)
{
   long status;
   static float stats[10];

   status = IpHstGet(GETSTATS, 0, stats);


   dprintf("dev_get_stats(): mean %.3f, stddev %.3f, sum %.3f",
	       stats[0],stats[1],stats[2]);
 
   argout->length = 5;
   argout->sequence = stats;

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_change_description()

 Description:	change the description of the current workspace

 Arg(s) In:	ImagePro 	ds 	- 
			DevVarStringArray  	*argin  - description
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ws_change_description(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{
   char *field, *description;
   short inf_field;
   long status;


   if (argin->length < 2)
   {

	   dev_error_push("you must specify two parametes to change the description, first one must be one of title, artist, date, description, name or range");
	   *error = DevErr_UnknownInputParameter;
	   return(DS_NOTOK);
   }

   field = argin->sequence[0];
   description = argin->sequence[1];
   dprintf("dev_ws_change_description(): set field %s",field);
//   dprintf("dev_ws_change_description(): set field %s to description %s",
//	       field,description);
   if (strlen(description) > 512) description[511] = 0; /* crop description to 512 characters */
   sprintf(ds->imagepro.description,"%s",description);
   inf_field = -1;
   if (stricmp(field,"title") == 0) inf_field = INF_TITLE;
   if (stricmp(field,"artist") == 0) inf_field = INF_ARTIST;
   if (stricmp(field,"date") == 0) inf_field = INF_DATE;
   if (stricmp(field,"description") == 0) inf_field = INF_DESCRIPTION;
   if (stricmp(field,"name") == 0) inf_field = INF_NAME;
   if (stricmp(field,"range") == 0) inf_field = INF_RANGE;
   if (inf_field == -1)
   {
	   dev_error_push("the first parameters must be one of title, artist, date, description, name or range");
	   *error = DevErr_UnknownInputParameter;
	   return(DS_NOTOK);
   }

   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpWsChangeDescription(inf_field, description);
      dprintf("dev_ws_change_description(): IpWsChangeDescription() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_ws_change_description(): SIMULATION mode");
   }


   if (inf_field == INF_DESCRIPTION && ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetHddDescr != NULL)
   {
//	   status = IpAcquSetHddDescr(ds->imagepro.description);
   }
 
   

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_reset()

 Description:	Reset device state to DEVON (useful to unblock state)

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_reset(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{
   ds->imagepro.ccd_state = DEVON;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_pre_snap_macro()

 Description:	define the pre-snap macro to execute i.e. before every snap

 Arg(s) In:	ImagePro 	ds 	- 
		DevString  	*argin  - maro name
   				  
 Arg(s) Out: DevVoid	*argout - none
	     long 	*error - pointer to error code, in case routine fails

============================================================================*/


static long dev_pre_snap_macro(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
	if (*argin != NULL)
	{
		ds->imagepro.pre_snap_macro = (char*)realloc(ds->imagepro.pre_snap_macro,strlen(*argin)+1);
		sprintf(ds->imagepro.pre_snap_macro,"%s",*argin);
	}

	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_post_snap_macro()

 Description:	define the post-snap macro to execute i.e. after every snap

 Arg(s) In:	ImagePro 	ds 	- 
		DevString  	*argin  - maro name
   				  
 Arg(s) Out: DevVoid	*argout - none
	     long 	*error - pointer to error code, in case routine fails

============================================================================*/


static long dev_post_snap_macro(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
	if (*argin != NULL)
	{
		ds->imagepro.post_snap_macro = (char*)realloc(ds->imagepro.post_snap_macro,strlen(*argin)+1);
		sprintf(ds->imagepro.post_snap_macro,"%s",*argin);
	}

	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_state()

 Description:	Get CCD state

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevLong	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_state(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

   *argout = ds->imagepro.ccd_state;

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_ccd_start()

 Description:	Start acquistion

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_start(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{

   dprintf("dev_ccd_start(): fork thread ip_acq_snap() to do acquisition ");
   _beginthread (ip_acq_snap, 0, ds);

   ds->imagepro.ccd_state = DevCcdTaking;
   dprintf("dev_ccd_start(): new state %d",ds->imagepro.ccd_state);
   
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_stop()

 Description:	Stop acquistion

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_stop(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{

   dprintf("dev_ccd_stop(): stop acquisition ");

/* PRESENT VERSION DOES NOTHING ! */

   ds->imagepro.ccd_state = DevCcdReady;
   
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_read()

 Description:	Read image

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevOpaque	*argout - image
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_read(ImagePro ds,DevVarLongArray *argin,DevOpaque *argout,long *error)
{
   int status, frame_no, no_frames;

   frame_no = argin->sequence[0];

   dprintf("dev_ccd_read(): frame %d image size %d",argin->sequence[0],argin->sequence[1]);

   ds->imagepro.sequence_extracted = 0;
   if (frame_no < 0) frame_no = 0;

   status = IpSeqGet(SEQ_NUMFRAMES, &no_frames);
   if (status > -1 && no_frames > 1)
   {
	 dprintf("dev_ccd_read(): this is a sequence, extract the %d image for reading.", frame_no);
	 if (frame_no >= no_frames)
	 {
		char error_mess[1024];
		sprintf(error_mess,"dev_ccd_read(): requested image %d exceeded maximum number of images taken %d, please specify a valid image number\n",no_frames,frame_no);
		dev_error_push(error_mess);
		*error = DevErr_CommandFailed;
		dprintf("dev_ccd_read(): exceeded maximum number of images in sequence !");
		return(DS_NOTOK);
	 }
     status = IpSeqExtractFrames(frame_no,1);
	 dprintf("dev_ccd_read(): extracted image %d from %d frames (status=%d)",frame_no,no_frames,status);
     if (status != -1) ds->imagepro.sequence_extracted = 1;
   }
   status = dev_get_image(ds,argin,argout,error);

   if ((long)argout->length > argin->sequence[1])
   {
		char error_mess[1024];
		sprintf(error_mess,"dev_ccd_read(): requested image size %d exceeded maximum size in client %d, please redimension your array (hint: type reconfig in spec)\n",
			argout->length,argin->sequence[1]);
		dev_error_push(error_mess);
		*error = DevErr_CommandFailed;
		dprintf("dev_ccd_read(): exceeded maximum client image size !");
		return(DS_NOTOK);
   }


   if (ds->imagepro.sequence_extracted)  IpDocClose();

   return(status);
}

/*============================================================================
 Function:      static long dev_ccd_set_exposure()

 Description:	Set exposure time in seconds

 Arg(s) In:	ImagePro 	ds 	- 
		DevFloat  	*argin  - exposure (in seconds)
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_exposure(ImagePro ds,DevFloat *argin,DevVoid *argout,long *error)
{
   long status;
   DevVarUShortArray var_shortarr;
   short params[4];

   ds->imagepro.exposure = (long)((*argin)*1000.); /* convert to milliseconds */

   dprintf("dev_ccd_set_exposure(): set exposure to %d ms",ds->imagepro.exposure);

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetExposure != NULL)
   {
      status = IpAcquSetExposure(ds->imagepro.exposure);
   }
   else
   {

	   params[0] = 49;
	   params[1] = 1;
	   if (ds->imagepro.exposure >65535)
	   {
		   params[2] = (short)(ds->imagepro.exposure - 65536);
		   params[3] = (short)(ds->imagepro.exposure/65536);
	   }
	   else
	   {
		   params[2] = (short)ds->imagepro.exposure;
		   params[3] = 0;
	   }
	   var_shortarr.length = 4;
	   var_shortarr.sequence = params;
	   status = dev_acq_control(ds,&var_shortarr, NULL,error);
   }
   
   if (status != 0) return(DS_NOTOK);

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_exposure()

 Description:	Get exposure time in seconds

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevFloat	*argout - exposure in seconds
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_exposure(ImagePro ds,DevVoid *argin,DevFloat *argout,long *error)
{

	if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetExposure != NULL)
   {
      ds->imagepro.exposure = IpAcquGetExposure();
   }

   *argout = (float)((ds->imagepro.exposure)/1000.); /* convert to seconds */

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_bin()

 Description:	Set binning

 Arg(s) In:	ImagePro 		ds 	- 
		DevVarLongArray  	*argin  - binning
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_bin(ImagePro ds,DevVarLongArray *argin,DevVoid *argout,long *error)
{
   int status;

   ds->imagepro.hbin = argin->sequence[0];
   ds->imagepro.vbin = argin->sequence[1];
   if(ds->imagepro.vbin <= 1) ds->imagepro.vbin = 1;
   if( ds->imagepro.vbin > 1 && ds->imagepro.vbin <= 2) ds->imagepro.vbin = 2;
   if( ds->imagepro.vbin > 2 && ds->imagepro.vbin <= 4) ds->imagepro.vbin = 3;
   if( ds->imagepro.vbin > 4 && ds->imagepro.vbin <= 8) ds->imagepro.vbin = 4;
   if( ds->imagepro.vbin > 8 && ds->imagepro.vbin <= 16) ds->imagepro.vbin = 5;
   if( ds->imagepro.vbin > 16 && ds->imagepro.vbin <= 32) ds->imagepro.vbin = 6;
   if( ds->imagepro.vbin > 32 && ds->imagepro.vbin <= 64) ds->imagepro.vbin = 7;
   if( ds->imagepro.vbin > 64 && ds->imagepro.vbin <= 128) ds->imagepro.vbin = 8;
   if( ds->imagepro.vbin > 128 && ds->imagepro.vbin <= 256) ds->imagepro.vbin = 9;
   if( ds->imagepro.vbin > 256 && ds->imagepro.vbin <= 512) ds->imagepro.vbin = 10;
   if( ds->imagepro.vbin > 512) ds->imagepro.vbin = 11;

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetHorizontalBinning != NULL)
   {
      status = IpAcquSetHorizontalBinning(ds->imagepro.hbin);
   }
   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetVerticalBinning != NULL)
   {
      status = IpAcquSetVerticalBinning(ds->imagepro.vbin);
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_bin()

 Description:	Get binning

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid		  	*argin  - none
   				  
 Arg(s) Out: 	DevVarLongArray		*argout - binning
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_bin(ImagePro ds,DevVoid *argin,DevVarLongArray *argout,long *error)
{
   static long binning[2];

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetHorizontalBinning != NULL)
   {
      ds->imagepro.hbin = IpAcquGetHorizontalBinning();
   }
   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetVerticalBinning != NULL)
   {
      ds->imagepro.vbin = IpAcquGetVerticalBinning();
   }

   binning[0] = ds->imagepro.hbin;
   binning[1] = ds->imagepro.vbin;

   argout->length = 2;
   argout->sequence = binning;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_trigger()

 Description:	Set trigger

 Arg(s) In:	ImagePro 	ds 	- 
		DevLong  	*argin  - trigger
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_trigger(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{

	long status;

   ds->imagepro.trigger = *argin;

   dprintf("dev_ccd_set_trigger(): set trigger to %d ms",ds->imagepro.trigger);

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetTimingMode != NULL)
   {
	   switch (*argin)
	   {
		case (0) : status = IpAcquSetTimingMode(0); /* soft internal trigger */
			       dprintf("dev_ccd_set_trigger(): set trigger mode to soft internal trigger");
				   break;
		case (1) : status = IpAcquSetTimingMode(2); /* hard external trigger */
			       dprintf("dev_ccd_set_trigger(): set trigger mode to hard external trigger");
				   break;
		default : dprintf("trigger mode %d not recognised, ignored !",*argin);
	   }
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_trigger()

 Description:	Get trigger

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - exposure in seconds
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_trigger(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

   *argout = ds->imagepro.trigger;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_file_par()

 Description:	Set file parameters

 Arg(s) In:	ImagePro 		ds 	- 
		DevVarStringArray  	*argin  - file parms
		sequence[0] - absolute root directory
		sequence[1] - filename prefix
		sequence[2] - filename suffix
		sequence[3] - image no.
		sequence[4] - printf() format for image number
		sequence[5] - overwrite flag, y=true, n=false
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_file_par(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{

//   char hdd_file[1024];

   if (argin->length != 6) 
   {
		dev_error_push("dev_ccd_set_file_par(): insufficient parameters, must specify exactly 6 i.e. directory, prefix, suffix, number, format + overwrite");
		*error = DevErr_CommandFailed;
   		return(DS_NOTOK);
   }

   dprintf("dev_ccd_set_file_par(): dir %s prefix %s suffix %s no. %s fmt %s overwrite %s",
	        argin->sequence[0],argin->sequence[1],argin->sequence[2],
			argin->sequence[3],argin->sequence[4],argin->sequence[5]);
   strcpy(ds->imagepro.ccd_file_dir, argin->sequence[0]);
   strcpy(ds->imagepro.ccd_file_prefix, argin->sequence[1]);
   strcpy(ds->imagepro.ccd_file_suffix, argin->sequence[2]);
   if (argin->sequence[3] != NULL) 
   {
	   sscanf(argin->sequence[3],"%ld", &(ds->imagepro.ccd_file_no));
   }
   else
   {
	   ds->imagepro.ccd_file_no = 0;
   }
   strcpy(ds->imagepro.ccd_file_no_fmt, argin->sequence[4]);
   strcpy(ds->imagepro.ccd_file_overwrite, argin->sequence[5]);

   /*
   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetHddFile != NULL)
   {
	   sprintf(hdd_file,"%s\\%s",ds->imagepro.ccd_file_dir,ds->imagepro.ccd_file_prefix);
	   IpAcquSetHddFile(hdd_file);
   }
   */
   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_ccd_get_file_par()

 Description:	Get file parameters

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid  		*argin  - none
   				  
 Arg(s) Out: 	DevVarStringArray	*argout - file parms
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_file_par(ImagePro ds,DevVoid *argin,DevVarStringArray *argout,long *error)
{
   static char ccd_file_no[256];
   static char *get_file_par[6];

   get_file_par[0] = ds->imagepro.ccd_file_dir;
   get_file_par[1] = ds->imagepro.ccd_file_prefix;
   get_file_par[2] = ds->imagepro.ccd_file_suffix;
   sprintf(ccd_file_no, "%d", ds->imagepro.ccd_file_no);
   get_file_par[3] = ccd_file_no;
   get_file_par[4] = ds->imagepro.ccd_file_no_fmt;
   get_file_par[5] = ds->imagepro.ccd_file_overwrite;
   argout->sequence = get_file_par;
   argout->length = 6;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_write_file()

 Description:	Write file

 Arg(s) In:	ImagePro 		ds 	- 
		DevLong  		*argin  - frame number to write
   				  
 Arg(s) Out: 	DevVoid			*argout - nothing
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_write_file(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{

   char file_name[1024], *file_name_ptr, *strarr[2];
   int ireturn, status, no_frames, frame_no, hdd_mode;
   DevVarStringArray var_strarr;

   dprintf("dev_ccd_write_file(): write image number %d",*argin);
/*
 * if this is a Frelon camera then first check to see if images are
 * being written "on-the-fly" in which case do not write the file again
 */
   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetHddMode != NULL)
   {
	   hdd_mode = IpAcquGetHddMode();
	   if (hdd_mode)
	   {
                   dprintf("dev_ccd_write_file(): images being written on the file, ignore request !");

		   dev_error_push("the images are already being written on-the-fly by ImagePro i.e. ignore write request !");
		   *error = DevErr_CommandFailed;
		   return(DS_NOTOK);
	   }
   }
   sprintf(file_name,"");
   if (ds->imagepro.ccd_file_dir != NULL) sprintf(file_name,"%s",ds->imagepro.ccd_file_dir);
   if (ds->imagepro.ccd_file_prefix != NULL) sprintf(file_name+strlen(file_name),"\\%s",ds->imagepro.ccd_file_prefix);
//
// ignore file_no_format sent by SPEC because it is wrong i.e. %03d instead of %04d !
//
//   if (ds->imagepro.ccd_file_no_fmt != NULL) sprintf(file_name+strlen(file_name),ds->imagepro.ccd_file_no_fmt,ds->imagepro.ccd_file_no++);
   if (ds->imagepro.ccd_file_no_fmt != NULL) sprintf(file_name+strlen(file_name),"%04d",ds->imagepro.ccd_file_no++);
   if (ds->imagepro.ccd_file_suffix != NULL) sprintf(file_name+strlen(file_name),".%s",ds->imagepro.ccd_file_suffix);

   file_name_ptr = file_name;
   frame_no = *argin;
   ds->imagepro.sequence_extracted = 0;
/*
 * if DOC is a sequence then only save the nth image (default is first image)
 */
   if (frame_no < 0) frame_no = 0;
   status = IpSeqGet(SEQ_NUMFRAMES, &no_frames);
   if (status > -1 && no_frames > 1)
   {
	 dprintf("dev_ccd_write_file(): this is a sequence, extract the %d image for writing.", frame_no);
	 if (frame_no >= no_frames) frame_no = no_frames-1;
         status = IpSeqExtractFrames(frame_no,1);
	 dprintf("dev_ccd_write_file(): extracted image %d from %d frames (status=%d)",frame_no,no_frames,status);
         if (status != -1) ds->imagepro.sequence_extracted = 1;
   }
   strarr[0] = file_name_ptr;
   strarr[1] = "edf";
   if (ds->imagepro.ccd_file_suffix != NULL)
   {
      if (stricmp(ds->imagepro.ccd_file_suffix,"tif") == 0)
      {
         strarr[1] = "tif";
      }
   }
   var_strarr.length = 2;
   var_strarr.sequence = strarr;
   ireturn = dev_ws_save_as(ds, &var_strarr, NULL, error);
/*
 * if an image was extracted then close the extracted image workspace
 */
   if (ds->imagepro.sequence_extracted) IpDocClose();
   return(ireturn);
}

/*============================================================================
 Function:      static long dev_ccd_x_size()

 Description:	Get X size

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - x size
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_x_size(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{
	IPDOCINFO info;

	IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
	*argout = info.Width;

#ifdef FRELON2000
	*argout = 2048;
#endif
#ifdef FRELON1000
	*argout = 1024;
#endif
	  
	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_y_size()

 Description:	Get Y size

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - y size
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_y_size(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{
	IPDOCINFO info;

	IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
	*argout = info.Height;

#ifdef FRELON2000
	*argout = 2048;
#endif
#ifdef FRELON1000
	*argout = 1024;
#endif

	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_depth()

 Description:	Get depth

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - bytes per pixel
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_depth(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

	*argout = 2;
	  
	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_frames()

 Description:	Set number of frames to take

 Arg(s) In:	ImagePro 	ds 	- 
		DevLong  	*argin  - no. of frames
   				  
 Arg(s) Out: 	DevVoid		*argout - none
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_frames(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{

	int status;

	if ((ds->imagepro.cam_type == CAMERA_FRELON2000) && (IpAcquSetNbImage != NULL))
	{
		status = IpAcquSetNbImage(*argin);
		if (ip_error(status, error) != DS_OK) return(DS_NOTOK);
	}
	
	ds->imagepro.frames = *argin;
	  
	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_frames()

 Description:	Get number of frames to take

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - no. of frames
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_frames(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

	int status;
			
	*argout = ds->imagepro.frames;

	if ((ds->imagepro.cam_type == CAMERA_FRELON2000) && (IpAcquGetNbImage != NULL))
	{
		status = IpAcquGetNbImage();
		if (status >= 0)
		{
			*argout = status;
			ds->imagepro.frames = status;
		}
		else
		{
			if (ip_error(status, error) != DS_OK) return(DS_NOTOK);
		}
	}
		  
	return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_shutter()

 Description:	Set shutter (1=external shutter enabled, 0=none)

 Arg(s) In:	ImagePro 		ds 	- 
		DevLong  	*argin  - shutter
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_shutter(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{
   int status;

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetShutter != NULL)
   {
      status = IpAcquSetShutter(*argin);
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_shutter()

 Description:	Get shutter

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid		  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - shutter
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_shutter(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetShutter != NULL)
   {
      *argout = IpAcquGetShutter();
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_channel()

 Description:	Set channel

 Arg(s) In:	ImagePro 		ds 	- 
		DevLong  	*argin  - channel
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_channel(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{
   int status;

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetChannel != NULL)
   {
      status = IpAcquSetChannel(*argin);
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_channel()

 Description:	Get channel

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid		  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - channel
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_channel(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetChannel != NULL)
   {
      *argout = IpAcquGetChannel();
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_roi()

 Description:	Set roi

 Arg(s) In:	ImagePro 		ds 	- 
		DevVarLongArray  	*argin  - roi
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_roi(ImagePro ds,DevVarLongArray *argin,DevVoid *argout,long *error)
{
   int status;

   ds->imagepro.roix1 = argin->sequence[0];
   ds->imagepro.roiy1 = argin->sequence[1];
   ds->imagepro.roix2 = argin->sequence[2];
   ds->imagepro.roiy2 = argin->sequence[3];


   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquSetRoi != NULL)
   {
      status = IpAcquSetRoi(ds->imagepro.roix1,ds->imagepro.roiy1,ds->imagepro.roix2,ds->imagepro.roiy2);
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_roi()

 Description:	Get roi

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid		  	*argin  - none
   				  
 Arg(s) Out: 	DevVarLongArray		*argout - roi
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_roi(ImagePro ds,DevVoid *argin,DevVarLongArray *argout,long *error)
{
   static long roi[4];
   int status;

   if (ds->imagepro.cam_type == CAMERA_FRELON2000 && IpAcquGetRoi != NULL)
   {
      status = IpAcquGetRoi(&ds->imagepro.roix1,&ds->imagepro.roiy1,&ds->imagepro.roix2,&ds->imagepro.roiy2);
   }

   roi[0] = ds->imagepro.roix1;
   roi[1] = ds->imagepro.roiy1;
   roi[2] = ds->imagepro.roix2;
   roi[3] = ds->imagepro.roiy2;

   argout->length = 4;
   argout->sequence = roi;

   return(DS_OK);
}
/*============================================================================
 Function:      static long ip_error()

 Description:	Check status returned by IPP for error

 Arg(s) In:	int	 	status - status returned by IPP 
   				  

============================================================================*/

static long ip_error(int status, long *error)
{

	int ireturn;

	ireturn = DS_OK;

	if (status < 0)
	{
		if (status == -1)
		{
			dev_error_push("ImagePro error -1. Image-Pro is not running. (hint: start ImagePro without restarting the device server !)");
			ireturn = DS_NOTOK;
		}
		if (status == -2)
		{
			dev_error_push("ImagePro error -2. Undocumented error ! (hint: exit and restart ImagePro without restarting the device server !)");
			ireturn = DS_NOTOK;
		}
		if (status == -3)
		{
			dev_error_push("ImagePro error -3. Can't find the DLL that implements the function. (hint: check your Image-Pro environment.)");
			ireturn = DS_NOTOK;
		}
		if (status == -4)
		{
			dev_error_push("ImagePro error -4. Can't find the function in the DLL. (hint: check your Image-Pro environment)");
			ireturn = DS_NOTOK;
		}
		if (status == -5)
		{
			dev_error_push("ImagePro error -5. Not applicable to the current image/situation. (hint: exit and restart ImagePro without restarting the device server !)");
			ireturn = DS_NOTOK;
		}
		if (status == -6)
		{
			dev_error_push("ImagePro error -6. There is no active workspace. (hint: exit and restart ImagePro without restarting the device server !)");
			ireturn = DS_NOTOK;
		}
		if (status == -7)
		{
			dev_error_push("ImagePro error -7. Invalid command arguments. (hint: check your command !)");
			ireturn = DS_NOTOK;
		}
		if (status == -8)
		{
			dev_error_push("ImagePro error -8. Insufficient memory. (hint: restart Image-Pro !)");
			ireturn = DS_NOTOK;
		}		if (status < -9)
		{
			char ip_error_mess[1024];
			sprintf(ip_error_mess,"ImagePro error %d. Check the camera, trigger signal and input parameters e.g. file name");
			dev_error_push(ip_error_mess);
			ireturn = DS_NOTOK;
		}
	}
	if (ireturn == DS_NOTOK) *error = DevErr_CommandFailed;
	
	return(ireturn);
}
