static char RcsId[] = "@(#) $Header: /segfs/dserver/classes++/beamline/id11/fitem/src/RCS/Fitem.cpp,v 1.1 2003/05/16 04:55:58 goetz Exp goetz $ ";

//+*********************************************************************
//
// File:	Fitem.C
//
// Project:	Device Servers in C++
//
// Description:	Code for implementing the Fitem analysis class in C++.
//
// Author(s);	Andy Gotz + Gavin Vaughan
//
// Original:	August 2001
//
// $Log: Fitem.cpp,v $
// Revision 1.1  2003/05/16 04:55:58  goetz
// Initial revision
//
//
// Copyright (c) 2001 by European Synchrotron Radiation Facility, 
//                       Grenoble, France
//
//
//
//-*********************************************************************

#include <stdlib.h>
#include <errno.h>
#include <pthread.h>
#include <iostream>
#include <fstream>
#include <string>
#include <strstream>
#include <API.h>
#include <DevServer.h>
#include <ApiP.h>
#include <BlcDsNumbers.h>
#include <Fitem.h>
#include <sps.h>
#include <math.h>

short Fitem::class_inited = 0;
short Fitem::class_destroyed = 0;

extern Fitem *device_ds[];
extern unsigned int n_devices;

//+======================================================================
// Function:    Fitem::GetResources()
//
// Description:	Interrogate the static database for Fitem resources 
//		for the specified device. This routine can also be used
//		to initialise the class default resources.
//
// Arg(s) In:	char *res_name - name in the database to interrogate
//
// Arg(s) Out:	long *error - pointer to error code if routine fails.
//-=======================================================================
long Fitem::GetResources (char *dev_name, long *error)
{
	static db_resource res_table[] = {
               {"fit_file",D_STRING_TYPE},
                             };
	static int res_tab_size = sizeof(res_table)/sizeof(db_resource);

	*error = 0;

	res_table[0].resource_adr = &(fit_file);

	if(db_getresource(dev_name,res_table,res_tab_size,error))
	{
   		cout << "Fitem::GetResources(): db_getresource() failed, error ";
		cout << *error << endl;
      		return(DS_NOTOK);
   	}
   	else
   	{
   		cout << "initial values after searching the static database for ";
		cout << dev_name << endl << endl;
//      		cout << "fit_file  D_STRING_TYPE  " << fit_file << endl;
   	}
	
	return(DS_OK);
}

//+======================================================================
// Function:    Fitem::ClassInitialise()
//
// Description:	Initialise the Fitem, is called once for
//		this class per process. ClassInitialise() will initialise
//		the class variables 
//
// Arg(s) In:	none
//
// Arg(s) Out:	long *error - pointer to error code if routine fails.
//-=======================================================================

long Fitem::ClassInitialise (long *error)
{
   	int iret=0;

   	Fitem::class_inited = 1;

   	cout << "Fitem::ClassInitialise(): returning" << endl;

   	return(iret);
}

//+======================================================================
// Function:    Fitem::Fitem()
//
// Description:	create a Fitem object. This involves allocating
//		memory for this object and initialising its name.
//
// Arg(s) In:	char *name - name of object.
//
// Arg(s) Out:	DevServer *ds_ptr - pointer to object created.
//		long *error - pointer to error code (in case of failure)
//-=====================================================================

Fitem::Fitem (char *name, long *error)
              :Device (name, error)
{
   	static Device::DeviceCommandListEntry commands_list[] = {
         	{DevState, (DeviceMemberFunction)(&Device::State), D_VOID_TYPE, D_SHORT_TYPE},
         	{DevStatus, (DeviceMemberFunction)(&Device::Status), D_VOID_TYPE, D_STRING_TYPE},
         	{FitemSinglePeak, (DeviceMemberFunction)&Fitem::FitSinglePeak, D_VAR_STRINGARR, D_VAR_DOUBLEARR},
         	{FitemSetParameters, (DeviceMemberFunction)&Fitem::SetParameters, D_VAR_DOUBLEARR, D_VOID_TYPE},
   	};
   	static long n_commands = sizeof(commands_list)/
			    	sizeof(DeviceCommandListEntry);
	long status;

   	cout << "Fitem::Fitem(): called with name " << name << endl;

//
// check to see if Fitem::ClassInitialise has been called
//
   	if (Fitem::class_inited != 1)
   	{
      		if (Fitem::ClassInitialise(error) != DS_OK)
      		{
         		return;
      		}
   	}

//
// Fitem is a subclass of Device
//

   	this->class_name = (char*)malloc(strlen("FitemClass")+1);
   	sprintf(this->class_name,"FitemClass");

//
// initialise the commands list to point to the commands list 
// implemented for the Fitem class
//

   	this->n_commands = n_commands;
   	this->commands_list = commands_list;

//
// read device resources
//

   	if (this->GetResources(this->name,error) != DS_OK)
   	{
      		cout << "Fitem::Fitem(): device GetResources() failed  (error=";
		cout << *error << ")" << endl;
      		return;
   	}

//
// create SPS arrays for binem input and output
//
	status = SPS_CreateArray("fitem", "fitin", 10000, 3, SPS_DOUBLE, SPS_IS_ARRAY);
	status = SPS_CreateArray("fitem", "fitout", 10000, 3, SPS_DOUBLE, SPS_IS_ARRAY);
   	cout << "Fitem::Fitem(): leaving and all OK" << endl;

   	return;
}

//+=====================================================================
// Function:	Fitem::~Fitem()
//
// Description:	destructor to destroy an object of the Fitem class
//
// input:	none
//
// Output:	none
//
//-=====================================================================
Fitem::~Fitem()
{
    	cout << "Fitem::~Fitem() called for " << this->name << endl;

//
// check to see if Fitem class has already been destroyed
//

   	if (Fitem::class_destroyed != 1)
   	{
		cout << "Destroying Fitem class" << endl;

		Fitem::class_destroyed = 1;
   	}
}

//+======================================================================
// Function:    Fitem::StateMachine()
//
// Description:	Check if the command to be executed does not violate
//		the present state of the device.
//
// Arg(s) In:	DevCommand cmd - command to be executed.
//
// Arg(s) Out:	long *error - pointer to error code (in case of failure).
//-=====================================================================
long Fitem::StateMachine (long cmd, long *error)
{
   	long iret = 0;
   	long int p_state, n_state;

   	p_state = this->state;

//
// before checking out the state machine assume that the state
// doesn't change i.e. new state == old state
//
   	n_state = p_state;

   	switch (p_state)
	{
//
// all commands allowed in all states !
//
   		default : break;
   	}

//
// update powersupply's private variable n_state so that other methods
// can use it too.
//

   	this->n_state = n_state;
   	this->xb = NULL;
   	this->yb = NULL;
   	this->sb = NULL;
	this->x_start = 0;
	this->x_end = 1;
	this->x_step = .1;

#ifdef DEBUG
//   	cout << "Fitem::StateMachine(): p_state " << p_state,
//	cout << " n_state " << n_state << ", iret " << iret << endl;
#endif
	
   	return(iret);
}

//+=====================================================================
// Function:    Fitem::FitSinglePeak()
//
// Description:	Fit a single peak to a regularly spaced rebinned scan 
//
// Arg(s) In:	DevVarStringArray *vargin - binning parameters
//
// Arg(s) Out:	DevVoid *vargout - none
//		long *error - pointer to error code (in the case of failure)
//-=====================================================================

long Fitem::FitSinglePeak (void *vargin, void *vargout, long *error)
{
	DevVarStringArray *argin;
	DevVarDoubleArray *argout;
	char *spec_version, *spec_array;
	int x_raw_col, y_raw_col, act_rows_x, act_rows_y, scan;
	float start, end, step; 
	static double fits[10];

#ifdef DEBUG
	cout << "Fitem::FitSinglePeak(" << name << ") called" << endl;
#endif

//
// get pointer to argin variable
//
	argin = (DevVarStringArray*)vargin;
	argout = (DevVarDoubleArray*)vargout;

	if (argin->length < 5)
	{
		ostrstream error_stream;

		error_stream << "Fitem::FitSinglePeak(): incorrect number of input arguments, needs at least 6 arguments" << endl << ends;
		dev_error_push(error_stream.str());
		error_stream.rdbuf()->freeze(false);
		*error = DevErr_CommandFailed;

		return(DS_NOTOK);
	}
//
// pickup input parameters
//

	spec_version = argin->sequence[0];
	spec_array = argin->sequence[1];
	rebin_file = argin->sequence[2];
	nbins = strtol(argin->sequence[3], NULL, 0);
	scan = strtol(argin->sequence[4], NULL, 0);
	if (argin->length >8)
	{
		start = strtod(argin->sequence[5], NULL);
		end = strtod(argin->sequence[6], NULL);
		step = strtod(argin->sequence[7], NULL);
	}
	else
	{
		start = x_start;
		end = x_end;
		step = x_step;
	}

//
// get copy of SPEC data from shared array 
//
  	xb = (double *)realloc(xb, (size_t)(nbins+1)*sizeof(double));
  	yb = (double *)realloc(yb, (size_t)(nbins+1)*sizeof(double));
  	sb = (double *)realloc(sb, (size_t)(nbins+1)*sizeof(double));
	
/*
	SPS_CopyColFromShared(spec_version, spec_array, xb, SPS_DOUBLE, 
	                      0, 0, &act_rows_x);
	SPS_CopyColFromShared(spec_version, spec_array, yb, SPS_DOUBLE, 
	                      1, 0, &act_rows_y);
	SPS_CopyColFromShared(spec_version, spec_array, sb, SPS_DOUBLE, 
	                      2, 0, &act_rows_y);
 */
//
// do the fitting 
//

	fit(rebin_file, start, step, end, scan, fits);

	argout->length = 9;
	argout->sequence = fits;
	
//
// copy result to output shared array so other programs (e.g. fitem)
// can continue the data reduction
//
	SPS_CopyColToShared("fitem", "fitout", xb, SPS_DOUBLE, 0, nbins, NULL);
	SPS_CopyColToShared("fitem", "fitout", yb, SPS_DOUBLE, 1, nbins, NULL);
	SPS_CopyColToShared("fitem", "fitout", sb, SPS_DOUBLE, 2, nbins, NULL);

	return (DS_OK);
}

//+=====================================================================
// Function:    Fitem::SetParameters()
//
// Description: Set the binning parameters for the next bin
//
// Arg(s) In:   DevVarDoubleArray *vargin - binning parameters
//
// Arg(s) Out:  DevVoid *vargout - none
//              long *error - pointer to error code (in the case of failure)
//-=====================================================================
 
long Fitem::SetParameters (void *vargin, void *vargout, long *error)
{
        DevVarDoubleArray *argin;
 
        argin = (DevVarDoubleArray*)vargin;
 
        if (argin->length < 3)
        {
                ostrstream error_stream;
 
                error_stream << "Fitem::SetParameters(): incorrect number of input arguments, needs 3 arguments" << endl << ends;
                dev_error_push(error_stream.str());
                error_stream.rdbuf()->freeze(false);
                *error = DevErr_CommandFailed;
 
                return(DS_NOTOK);
        }
 
        x_start = argin->sequence[0];
        x_end = argin->sequence[1];
        x_step = argin->sequence[2];
 
        cout << "Fitem::SetParameters(): start " << x_start << " end " << x_end << " step " << x_step << endl;
        return (DS_OK);
}
// fit() - call fitting program of Gavin Vaughan. 
//
// TODO - do fit in memory.

int Fitem::fit(char *file, float start, float end, float step, int scan, double *fits)
{
	int status;
	char fit_cmd[256];
	ifstream fits_file;

	sprintf(fit_cmd, "cat fits >>fits.old;rm fits;marqmake %s - 0 9999", file);

  	status = system(fit_cmd);

	fits_file.open("fits",std::ios::in);

	if (fits_file.is_open()) 
	{
		cout << "fits results ";
		for(int i=0; i<9; i++)
		{
			fits_file >> fits[i];
			cout << " " << fits[i];
		}
		cout << endl;
	}
	fits_file.close();

  	return(status);

} 
