static char RcsId[] = "@(#) $Header$ ";

//+*********************************************************************
//
// File:	Template.C
//
// Project:	Device Servers in C++
//
// Description:	Code for implementing the Template class in C++.
//
// Author(s);	Emmanuel Taurel, Andy Gotz
//
// Original:	January 1997
//
// $Log$
// 
//
// Copyright (c) 1997 by European Synchrotron Radiation Facility, 
//                       Grenoble, France
//
//
//
//-*********************************************************************

#include <stdlib.h>
#include <string.h>
#include <API.h>
#include <DevErrors.h>
#include <DevSignal.h>

#include <DevServer.h>
#include <Template.H>

#include <cglob.h>
#include <errno.h>
#include <modes.h>
#include <types.h>
#include <stddef.h>

short Template::class_inited = 0;
short Template::class_destroyed = 0;

extern Template *device_ds[];
extern unsigned int n_devices;

//+======================================================================
// Function:    Template::GetResources()
//
// Description:	Interrogate the static database for Template resources 
//		for the specified device. This routine can also be used
//		to initialise the class default resources.
//
// Arg(s) In:	char *res_name - name in the database to interrogate
//
// Arg(s) Out:	long *error - pointer to error code if routine fails.
//-=======================================================================
long Template::GetResources (char *dev_name, long *error)
{
   static db_resource res_table[] = {
               {"nada",D_SHORT_TYPE},
                             };
   static int res_tab_size = sizeof(res_table)/sizeof(db_resource);

   *error = 0;

   res_table[0].resource_adr = &(this->nada);

   if(db_getresource(dev_name,res_table,res_tab_size,error))
   {
   	printf("Template::GetResources(): db_getresource() failed, error %d\n",*error);
      	return(DS_NOTOK);
   }
   else
   {
   	printf("initial values after searching the static database for %s\n\n",dev_name);
      	printf("nada       D_SHORT_TYPE   %d\n",this->nada);
   }

   return(DS_OK);
}

//+======================================================================
// Function:    Template::ClassInitialise()
//
// Description:	Initialise the Template, is called once for
//		this class per process. ClassInitialise() will initialise
//		the class variables 
//
// Arg(s) In:	none
//
// Arg(s) Out:	long *error - pointer to error code if routine fails.
//-=======================================================================

long Template::ClassInitialise (long *error)
{
   int iret=0;

   printf ("Template::ClassInitialise() called\n");
  
// To force the compiler to include the RcsId string in the executable

   int l = strlen(RcsId);

   Template::class_inited = 1;

   printf("returning from Template::ClassInitialise()\n");

   return(iret);
}

//+======================================================================
// Function:    Template::Template()
//
// Description:	create a Template object. This involves allocating
//		memory for this object and initialising its name.
//
// Arg(s) In:	char *name - name of object.
//
// Arg(s) Out:	DevServer *ds_ptr - pointer to object created.
//		long *error - pointer to error code (in case of failure)
//-=====================================================================

Template::Template (char *name, long *error)
              :Device (name, error)
{
   static Device::DeviceCommandListEntry commands_list[] = {
         {DevState, &Device::State, D_VOID_TYPE, D_SHORT_TYPE},
         {DevStatus, (DeviceMemberFunction)(&Template::Status), D_VOID_TYPE, D_STRING_TYPE},
         {DevOn, (DeviceMemberFunction)(&Template::On), D_VOID_TYPE, D_VOID_TYPE},
         {DevOff, (DeviceMemberFunction)&Template::Off, D_VOID_TYPE, D_VOID_TYPE},
         {DevSetValue, (DeviceMemberFunction)(&Template::SetValue), D_SHORT_TYPE, D_VOID_TYPE},
         {DevReadValue, (DeviceMemberFunction)&Template::ReadValue, D_VOID_TYPE, D_SHORT_TYPE},
                                                           };
   static long n_commands = sizeof(commands_list)/
			    sizeof(DeviceCommandListEntry);
   char *tmp;

   printf("Template::Template called, name %s\n",name);

//
// check to see if Template::ClassInitialise has been called
//
   if (Template::class_inited != 1)
   {
      if (Template::ClassInitialise(error) != DS_OK)
      {
         return;
      }
   }

//
// Template is a subclass of Device
//

   this->class_name = (char*)malloc(strlen("TemplateClass")+1);
   sprintf(this->class_name,"TemplateClass");

//
// initialise the commands list to point to the commands list 
// implemented for the Template class
//

   this->n_commands = n_commands;
   this->commands_list = commands_list;

//
// read device resources
//

   if (this->GetResources(this->name,error) != DS_OK)
   {
      printf("Template::Template(): device GetResources() failed, error %d\n",*error);
      return;
   }

//
// The initial state of the device is DEVOFF
//

   this->state = DEVOFF;

   printf("leaving Template::Template() and all OK\n");

   return;
}

//+=====================================================================
// Function:	Template::~Template()
//
// Description:	destructor to destroy an object of the Template class
//
// input:	none
//
// Output:	none
//
//-=====================================================================
Template::~Template()
{
    printf("Template::~Template called for %s\n",this->name);

//
// check to see if Template class has already been destroyed
//

   if (Template::class_destroyed != 1)
   {
	printf("Destroying Template class\n");

	Template::class_destroyed = 1;
   }
}

//+======================================================================
// Function:    Template::StateMachine()
//
// Description:	Check if the command to be executed does not violate
//		the present state of the device.
//
// Arg(s) In:	DevCommand cmd - command to be executed.
//
// Arg(s) Out:	long *error - pointer to error code (in case of failure).
//-=====================================================================
long Template::StateMachine (long cmd, long *error)
{
   	long iret = 0;
   	long int p_state, n_state;

   	p_state = this->state;

//
// before checking out the state machine assume that the state
// doesn't change i.e. new state == old state
//
   	n_state = p_state;

   	switch (p_state)
	{
//
// all commands allowed in all states !
//
   		default : break;
   	}

//
// update powersupply's private variable n_state so that other methods
// can use it too.
//

   this->n_state = n_state;

#ifdef DEBUG
   printf("Template::StateMachine(): p_state %2d n_state %2d, iret %2d\n",
          p_state,n_state, iret);
#endif

   return(iret);
}

//+=====================================================================
// Function:    Template::Off()
//
// Description:	switch the device off.
//
// Arg(s) In:	DevVoid *argin - void.
//
// Arg(s) Out:	DevVoid *argout - void.
//		long *error - pointer to error code in case of failure.
//-=====================================================================

long Template::Off (void *vargin, void *vargout, long *error)
{

#ifdef DEBUG
	printf("Template::Off(%s) called\n",this->name);
#endif

	*error = 0;

	this->state = DEVOFF;

	return (DS_OK);
}
	
//+=====================================================================
// Function:    Template::On()
//
// Description:	switch the device ON
//
// Arg(s) In:	DevVoid *argin - void.
//
// Arg(s) Out:	DevVoid *argout - void.
//		long *error - pointer to error code (in case of failure)
//-=====================================================================

long Template::On (void *vargin, void *vargout, long *error)
{

#ifdef DEBUG
	printf("Template::On(%s) called\n",this->name);
#endif

	*error = 0;

   	this->state = DEVON;

	return (DS_OK);
}

//+=====================================================================
// Function:    Template::Status()
//
// Description:	Return the state as an ASCII string. Interprets the error
//		flag as well if the status is FAULT.
//
// Arg(s) In:	DevVoid *vargin - void.
//
// Arg(s) Out:	DevString *vargout - contains string.
//		long *error - pointer to error code (in the case of failure)
//-=====================================================================

long Template::Status ( void *vargin, void *vargout, long *error)
{
	static char mess[2048];

#ifdef DEBUG
	printf("Template::Status(%s) called\n",name);
#endif
	*error = 0;

	switch (this->state)
	{
	case (DEVOFF) : sprintf(mess,"%s","Off: Everything OK");
	                break;

	case (DEVON) : sprintf(mess,"%s","On: Everything OK");
	               break;

	default : sprintf(mess,"%s","Unknown");
	          break;
	}

//
// Change end of message for UNIX
//

        mess[strlen(mess)-1] = 0x0a;

        status = (char **)vargout;
        *status = mess;

        return(DS_OK);
}

//+=====================================================================
// Function:    Template::SetValue()
//
// Description:	Routine to set a dummy value for a template device.
//
// Arg(s) In:	DevFloat *vargin - dummy value to set
//
// Arg(s) Out:	DevVoid *argout - void.
//		long *error - pointer to error code (in the case of failure)
//-=====================================================================

long Template::SetValue (void *vargin, void *vargout, long *error)
{
	short *set_value;

#ifdef DEBUG
	printf("Template::SetValue(%s) called\n",name);
#endif

	*error = 0;

//
// get pointer to argin variable
//
	set_value = (short*)vargin;

//
// assign contents of argin variable to nada
//
	this->nada = *set_value;

	return (DS_OK);
}

//+=====================================================================
// Function:    Template::ReadValue()
//
// Description:	Routine to read a dummy value for a template device.
//
// Arg(s) In:	DevVoid *vargin - void
//
// Arg(s) Out:	DevFloat *argout - dummy value read
//		long *error - pointer to error code (in the case of failure)
//-=====================================================================

long Template::ReadValue (void *vargin, void *vargout, long *error)
{
	short *read_value;

#ifdef DEBUG
	printf("Template::ReadValue(%s) called\n",name);
#endif

	*error = 0;

//
// get pointer to argout variable
//
	read_value = (short*)vargout;

//
// assign nada to variable pointed to by argout
//
	*read_value = this->nada ;

	return (DS_OK);
}

