static char RcsId[] = "$Header: /segfs/dserver/classes/vacuum/rga/src/RCS/Rga.c,v 1.19 2004/06/14 12:55:06 goetz Exp $" ;
/*********************************************************************
 *
 * File:       rga.c
 *
 * Project:    Residual Gas Analizer (RGA) Device Server
 *
 * Purpose :   Implements RPC commands for the access to SATELLITE
 *             units from LEDA-MASS (RGA controllers)
 *
 * Author(s):  Vicente Rey
 *
 * $Date: 2004/06/14 12:55:06 $
 *
 * Copyright (c) 1992 European Synchrotron Radiation Facility,
 *                    Grenoble, France
 *
 *                   All Rights Reserved
 *
 **********************************************************************/
/*
 * $Log: Rga.c,v $
 * Revision 1.19  2004/06/14 12:55:06  goetz
 * port to linux
 *
 * Revision 1.18  2001/05/29  12:50:24  12:50:24  chaize (Jm.Chaize)
 * add ceiling under max_noise resource
 * 
 * 
 * Revision 1.17  2001/03/08  14:14:59  14:14:59  chaize (Jm.Chaize)
 * *** empty log message ***
 * 
 * Revision 1.16  00/12/01  11:45:23  11:45:23  pons ()
 * *** empty log message ***
 * 
 * Revision 1.15  2000/08/30  16:55:44  16:55:44  olivier (Olivier Martin)
 * Serial Line Class implemented
 * 
 * 
 * Revision 1.14  99/12/30  15:56:38  15:56:38  olivier (Olivier Martin)
 * Bug on Filament 2.
 * 
 * 
 * Revision 1.13  99/01/18  13:32:38  13:32:38  chaize (Jm.Chaize)
 * Modify state handler for DevReadSigValues command.
 * 
 * Revision 1.11  99/01/04  11:24:35  11:24:35  olivier (Olivier Martin)
 * Multi Signal Implemented
 * 
 * Revision 1.10  98/12/22  14:54:13  14:54:13  chaize (Jm.Chaize)
 * *** empty log message ***
 * 
 * Revision 1.9  97/12/09  15:54:12  15:54:12  chaize (Jm.Chaize)
 * *** empty log message ***
 * 
 * Revision 1.8  97/12/03  14:06:22  14:06:22  cardonne ()
 * End of modification
 * 
 * Revision 1.7  97/10/22  08:36:15  08:36:15  cardonne ()
 * Few bugs fixed + DevStatus command implemented
 * 
 * Revision 1.6  97/10/13  14:05:18  14:05:18  chaize (Jm.Chaize)
 * *** empty log message ***
 * 
 * Revision 1.5  97/07/04  13:16:47  13:16:47  chaize (Jm.Chaize)
 * *** empty log message ***
 * 
 * Revision 1.4  97/04/25  10:05:56  10:05:56  chaize (Jm.Chaize)
 * *** empty log message ***
 * 
 * Revision 1.2  96/08/29  17:30:48  17:30:48  chaize (Jm.Chaize)
 * new commands and automatic restart implemented
 * 
 * Revision 1.1  1992/08/19  10:57:38  rey
 * Initial revision
 *
 *
 */

#include <API.h>
#include <DevErrors.h>
#include <DevServer.h>
#include <DevServerP.h>
#include <Admin.h>

#include <Rga.h>
#include <RgaP.h>
#include <DevRgaCmds.h>
#include <DevRgaErrors.h>
#include <vgc_xdr.h>

#include <errno.h>
#ifndef __cplusplus
#include <modes.h>
#include <ctype.h>
#include <sgstat.h>
#include <scf.h>
#endif /* __cplusplus */

#include <math.h>

#include <SerialLine.h>
#ifndef __cplusplus
#include <SerialLineP.h>
#endif /* __cplusplus */

/* public methods */
static long     class_initialise(long *error);
static long     object_create(char *name, DevServer *ds_ptr, long *error);
static long     object_initialise(Rga ds,long *error);
static long     state_handler(Rga ds,DevCommand cmd,long *error);


static DevMethodListEntry method_list[] =
{
    {DevMethodClassInitialise, (DevMethodFunction)class_initialise},
    {DevMethodCreate,          (DevMethodFunction)object_create},
    {DevMethodInitialise,      (DevMethodFunction)object_initialise},
    {DevMethodStateHandler,    (DevMethodFunction)state_handler},
};

/*
 *  Definition of the class structure. This structure is partially
 *  initialised in order to allow the calls to the different methods.
 *  The initialisation will be completed by DevMethodClassInitialise.
 */
RgaClassRec  rgaClassRec = {
    sizeof(method_list) / sizeof(DevMethodListEntry),  /* n_methods */
    method_list,                                       /* method_list */
};

RgaClass     rgaClass = (RgaClass) & rgaClassRec;


/*
 *  Definition of a template copy of the device server structures that
 *  will be created by DevMethodCreate and iniatilised by DevMethodInitialise
 *  during DevMethodStartup.
 */
static RgaRec  rgaRec;
static Rga     rga = (Rga) &rgaRec;

/*
 *  Declaration of the command functions and definition of the table of
 *  available commands.
 */
static long     dev_state(Rga ga,DevVoid *argin,DevLong *argout,long *error);
static long     dev_status(Rga ga,DevVoid *argin,DevString *argout,long *error);
static long     dev_on(Rga ga,DevVoid *argin,DevVoid *argout,long *error);
static long     devfilamenton(Rga ga,DevShort *argin,DevVoid *argout,long *error);
static long     devfilamentstate(Rga ga,DevVoid *argin,DevVarShortArray *argout,long *error);
static long     devfilamentoff(Rga ga,DevVoid *argin,DevVoid *argout,long *error);
static long     devcommand(Rga ga,DevString *argin,DevVoid *argout,long *error);
static long     devrequest(Rga ga,DevString *argin,DevString *argout,long *error);
static long     devoperationmode(Rga ga,DevVoid *argin,DevVarShortArray *argout,long *error);
static long     devchannelparam(Rga ga,DevVoid *argin,DevVarShortArray *argout,long *error);
static long     devreset(Rga ga,DevVoid *argin,DevVoid *argout,long *error);
static long     devdescription(Rga ga,DevShort *argin,DevString *argout,long *error);
static long     auxgaugename(Rga ga,DevString *argin,DevString *argout,long *error);
static long     auxgaugecontrol(Rga ga,DevShort *argin,DevFloat *argout,long *error);
static long     devreadvalues(Rga ga,DevVoid *argin,DevVarFloatArray *argout,long *error);
static long     devsetmode(Rga ga,DevShort *argin,DevVoid *argout,long *error);
static long     devindex(Rga ga,DevVoid *argin,DevLong *argout,long *error);
static long     devsetparam(Rga ga,DevVarShortArray *argin,DevVoid *argout,long *error);
static long     devsetchannel(Rga ga,DevVarShortArray *argin,DevVoid *argout,long *error);
static long	devsetvoltage(Rga ga,DevShort *argin,DevVoid *argout,long *error);
static long     devsetmultipliergain(Rga ga,DevShort *argin,DevVoid *argout,long *error);
static long     devindicator(Rga ga,DevVoid *argin,DevShort *argout,long *error);
static long     devreadvoltage(Rga ga,DevVoid *argin,DevShort *argout,long *error);
static long     default_config(Rga ga,DevVoid *argin,DevVoid *argout,long *error);
static long     dev_update_config(Rga ga, DevVoid *argin, DevVoid *argout, long *error);
static long 	dev_signal_config (Rga ga,DevVoid *argin,DevVarStringArray *argout,long *error);


static DevCommandListEntry commands_list[] = {
  {DevState,               (DevCommandFunction)dev_state,        D_VOID_TYPE,   D_LONG_TYPE},
  {DevStatus,              (DevCommandFunction)dev_status,       D_VOID_TYPE,   D_STRING_TYPE},
  {DevFilamentState,       (DevCommandFunction)devfilamentstate,D_VOID_TYPE,   D_VAR_SHORTARR},
  {DevFilamentOn,          (DevCommandFunction)devfilamenton,   D_SHORT_TYPE,  D_VOID_TYPE},
  {DevFilamentOff,         (DevCommandFunction)devfilamentoff,  D_VOID_TYPE,   D_VOID_TYPE},
  {DevOn,                  (DevCommandFunction)dev_on,           D_VOID_TYPE,   D_VOID_TYPE},
  {DevOff,                 (DevCommandFunction)devfilamentoff,  D_VOID_TYPE,   D_VOID_TYPE},
  {DevSendCommand,         (DevCommandFunction)devcommand,      D_STRING_TYPE, D_VOID_TYPE},
  {DevSendRequest,         (DevCommandFunction)devrequest,      D_STRING_TYPE, D_STRING_TYPE},
  {DevGetOperationMode,    (DevCommandFunction)devoperationmode,D_VOID_TYPE,   D_VAR_SHORTARR},
  {DevGetChannelParameters,(DevCommandFunction)devchannelparam, D_VOID_TYPE,   D_VAR_SHORTARR},
  {DevReset,               (DevCommandFunction)devreset,        D_VOID_TYPE,   D_VOID_TYPE},
  {DevRestore,             (DevCommandFunction)default_config,  D_VOID_TYPE,   D_VOID_TYPE},
  {DevGetDescription,      (DevCommandFunction)devdescription,  D_SHORT_TYPE,  D_STRING_TYPE},
  {DevAuxGaugeName,        (DevCommandFunction)auxgaugename,    D_STRING_TYPE, D_STRING_TYPE},
  {DevAuxGaugeControl,     (DevCommandFunction)auxgaugecontrol, D_SHORT_TYPE,  D_FLOAT_TYPE},
  {DevReadValues,          (DevCommandFunction)devreadvalues,   D_VOID_TYPE,   D_VAR_FLOATARR},
  {DevReadNChannels,       (DevCommandFunction)devreadvalues,   D_VOID_TYPE,   D_VAR_FLOATARR},
  {DevReadSigValues,       (DevCommandFunction)devreadvalues,   D_VOID_TYPE,   D_VAR_FLOATARR},
  {DevSetMode,             (DevCommandFunction)devsetmode,      D_SHORT_TYPE,   D_VOID_TYPE},
  {DevSetParam,            (DevCommandFunction)devsetparam,     D_VAR_SHORTARR,   D_VOID_TYPE},
  {DevSetChannel,          (DevCommandFunction)devsetchannel,   D_VAR_SHORTARR,   D_VOID_TYPE},
  {DevSetVoltage,          (DevCommandFunction)devsetvoltage,   D_SHORT_TYPE,   D_VOID_TYPE},
  {DevSetChanGain,         (DevCommandFunction)devsetmultipliergain,      D_SHORT_TYPE,   D_VOID_TYPE},
  {DevIndicator,           (DevCommandFunction)devindicator,   D_VOID_TYPE,   D_SHORT_TYPE},
  {DevReadVoltage,         (DevCommandFunction)devreadvoltage,  D_VOID_TYPE,   D_SHORT_TYPE},
  {DevUpdateSigConfig,	   (DevCommandFunction)dev_update_config, D_VOID_TYPE, D_VOID_TYPE,  WRITE_ACCESS},
  {DevGetSigConfig,	   (DevCommandFunction)dev_signal_config, D_VOID_TYPE, D_VAR_STRINGARR, READ_ACCESS}		

};
static long  n_commands = sizeof(commands_list) / sizeof(DevCommandListEntry);


/*
 * Definition of a table of class resources, loaded by db_getresource
 * in DevMethodClassInitialise from the database.
 */
static db_resource class_tab[] = {
           {"Description",         D_STRING_TYPE, NULL},
           {"AuxGaugeMaxPressure", D_FLOAT_TYPE,  NULL},
       };
static long  class_tab_size = sizeof(class_tab) / sizeof(db_resource);


/*
 * Definition of a table of device resources, loaded by db_getresource
 * in DevMethodInitialise from the database.
 */
static db_resource obj_tab[] = {
    {"Description",      D_STRING_TYPE, NULL},
    /*{"SerialDriverName", D_STRING_TYPE, NULL},*/
    {"serial_driver_name", D_STRING_TYPE, NULL},
    {"MaxMass",          D_SHORT_TYPE,  NULL},
    {"AuxVacuumGauge",   D_STRING_TYPE, NULL},
    {"AuxGaugeCheck",    D_BOOLEAN_TYPE,NULL},
    {"FilamentUsed",     D_SHORT_TYPE,NULL},
    {"AutoScanMode",     D_SHORT_TYPE,NULL},
    {"ChanMasses",       D_VAR_SHORTARR,NULL},
    {"ChanAutoRange",    D_VAR_SHORTARR,NULL},
    {"ChanMultiplier",   D_VAR_SHORTARR,NULL},
    {"MaxNoise",         D_FLOAT_TYPE,NULL},
    {"Multiplier",    	 D_SHORT_TYPE,NULL},
    {"Multipliergain",   D_SHORT_TYPE,NULL},
    {"TotalPressure",    D_SHORT_TYPE,NULL},
    {"Accuracy",     	 D_SHORT_TYPE,NULL},
    {"FirstMass",   	 D_SHORT_TYPE,NULL},
    {"MassZoom",    	 D_SHORT_TYPE,NULL},
    {"Range",     	 D_SHORT_TYPE,NULL},
    {"AutoRange",        D_SHORT_TYPE,NULL},
    {"MultiplierVolt",   D_SHORT_TYPE,NULL},
    {"MultiplierBase",   D_SHORT_TYPE,NULL},
    {"reset_timeout",    D_SHORT_TYPE,NULL},
   
};
static long  obj_tab_size = sizeof(obj_tab) / sizeof(db_resource);


/*
 * Private functions.
 */
static long get_state(Rga ga,long *error);
static long set_up_comm(Rga ga,long *error);
static long send_request(Rga ga,char *request,char **answer,long *error);
static long send_string(Rga ga,char *string,long *error);
static long get_answer(Rga ga,long *error);
static long long_request(Rga ga,char *request,char **answer,long *error);
static long mode_update(Rga ga,long *error);
static long get_op_mode(Rga ga,long *error);
static long get_chann_param(Rga ga,long *error);
static long get_aux_defaults(Rga ga);
static long check_aux_pressure(Rga ga,long *error);
static long read_aux_gauge(Rga ga,long *error);
static long import_aux_gauge(Rga ga,long *error);
static long free_aux_gauge(Rga ga,long *error);
static long get_single_channel(Rga ga ,short param ,short channel ,short value ,long *error );
/*
 * Internal definitions
 */

/*
 * Commands.
 */
#define ACK_COMMAND               "A1"
#define RESET_COMMAND             "R"
#define ANSWER_TERMINATOR_COMMAND "O1,1"
#define ANSWER_DELIMITER_COMMAND  "O0,0"
#define FILAMENT_OFF_COMMAND      "F1,0"      /* This command switches off *
                                               * either filement           */ 
#define ACK_ANSWER   'A'
#define NACK_ANSWER  'N'

#define LINE_TERMINATOR '\n'

#define TIME_OUT  3                          /* Time out in seconds  */

/*
 * Requests.
 */
#define FILAMENT_STATUS_REQUEST       "f"
#define SCAN_MODE_REQUEST             "m"
#define MULTIPLIER_GAIN_REQUEST       "g"
#define TOTAL_PRESSURE_TRIP_REQUEST   "t1"
#define PARTIAL_PRESSURE_TRIP_REQUEST "t2"
#define RANGE_REQUEST                 "p6"
#define SPECTRUM_REQUEST              "e"
#define INDEX_REQUEST                 "i"
#define VOLT_REQUEST                 "v"
#define IS_NEW_SCAN                 "o"

#define FILAMENT_ON_PATTERN               "F%d,1"
#define MODE_PARAMETER_REQUEST_PATTERN    "p%d"
#define MODE_SET_PATTERN                  "M%d"
#define ALARM_STATUS_REQUEST_PATTERN      "a%d"
#define CHANNEL_PARAMETER_REQUEST_PATTERN "c%d,%d"
#define CHANNEL_SET_PATTERN              "C%d,%d,%d"
#define PARAMETER_SET_PATTERN            "P%d,%d"
#define RANGE_REQUEST_PATTERN             "c4,%d"
#define SET_MULTIPLIER_GAIN_PATTERN       "G%d"

#define MAX_DESCRIPTION_LENGTH  40
#define RGA_MAX_LENGTH         40
#define MAX_STRING_LENGTH      100
#define MAX_ANSWER_LENGTH     2400
#define OVER_RANGE	      1e-5
#define UNDER_RANGE	      1e-14

/* PPAS Define for test presence */
#define PPAS_CONTROL	"CONTROL"
#define PPAS_BYPASS	"BYPASS"
#define PPAS_SPECTRUM	"ec"
/*
 * Internal static variables
 */

static char answer_buff[MAX_ANSWER_LENGTH + 1];    /* input buffer */

static int  Number_Param[8] = {0,10,7,3,7,0,0,0};  /* number of parameters *
                                                    * in each scan mode    */

/*****************************************************
 * DevMethodClassInitialise  ( class_initialise() )
 *******************************************************************/
/*
 *  Purpose:  Initialises rgaClass and loads default values from
 *            the database.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_InsufficientMemory 
 *
 *            +  errors returned by db_getresource()
 *
 *******************************************************************/
static long class_initialise(long *error)

{
    long err;
    *error = FALSE;

#ifdef DEBUG_LVL1
   printf("\t<RGA class_initialise> %s\n", rgaClass->rga_class.class_svc_name);
#endif

    rga->devserver.class_pointer = (DevServerClass) rgaClass;

    /*
     * rgaClass is a subclass of devServerClass
     */

    rgaClass->devserver_class.superclass = devServerClass;
    rgaClass->devserver_class.class_name =
                                      (char *) malloc(sizeof(RGA_CLASS_NAME));
    if ( rgaClass->devserver_class.class_name == 0 ){
        *error = DevErr_InsufficientMemory;
        return (DS_NOTOK);
    }

    strcpy(rgaClass->devserver_class.class_name, RGA_CLASS_NAME);

#ifdef DEBUG_LVL2
    printf("\t - Reading class resources\n");
#endif

    rgaClass->rga_class.description = "residual gas analyser";
    rgaClass->rga_class.aux_max_pressure = 1e-5;
     
    class_tab[0].resource_adr = &(rgaClass->rga_class.description);
    class_tab[1].resource_adr = &(rgaClass->rga_class.aux_max_pressure);

  if ( db_getresource(rgaClass->rga_class.class_svc_name, class_tab,
              class_tab_size, error) != 0 )

        return(DS_NOTOK);
    printf(" resource for %s:\n",rgaClass->rga_class.class_svc_name);
    printf("\t\t\tDescription  : %s\n", rgaClass->rga_class.description);
    printf("\t\t\tMax. Pressure: %.2E\n",rgaClass->rga_class.aux_max_pressure);

    /*
     * commands implemented for the rga server
     */
    rgaClass->devserver_class.n_commands = n_commands;
    rgaClass->devserver_class.commands_list = commands_list;

    rgaClass->devserver_class.class_inited = 1;

    /*
     * Load VGC xdr types.
     */
    LOAD_VGC_STATUS(&err);
    LOAD_VGC_GAUGE(&err);
    LOAD_VGC_CONTROLLER(&err);

    return (DS_OK);
}

/*****************************************
 * DevMethodCreate  ( object_create() )
 **************************************************************/
/*
 *  Purpose:  Creates the new object and initialises it with
 *            the default object values
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_InsufficientMemory
 *
 **************************************************************/
static long object_create(DevString name,DevServer *ds_ptr,long *error)

{
    Rga    ga;

    *error = FALSE;

#ifdef DEBUG_LVL1
   printf("\t<RGA object_create> name: %s\n", name);
#endif

    /*
     * Allocate memory for this new device
     */

    if ((ga = (Rga) malloc(sizeof(RgaRec))) == 0) {
        *error = DevErr_InsufficientMemory;
        return (DS_NOTOK);
    }

    /*
     * Initialise device with default object and allocate
     * the required additional memory
     */

    *ga = *rga;
    ga->devserver.name = (char *) malloc(strlen(name));
    ga->rga.actual_aux_vacuum_gauge = (char *) malloc(RGA_MAX_LENGTH + 1);
    if ( ga->devserver.name == 0 ||
         ga->rga.actual_aux_vacuum_gauge == 0 ){
        *error = DevErr_InsufficientMemory;
        return (DS_NOTOK);
    }
    sprintf(ga->devserver.name, "%s", name);
    ga->devserver.class_pointer = (DevServerClass) rgaClass;
    ga->devserver.state = DEVUNKNOWN;


    *ds_ptr = (DevServer) ga;

    return (DS_OK);
}

/****************************************************
 * DevMethodInitialise  ( object_initialise() )
 **************************************************************/
/*
 *  Purpose:  Initialises the parameters of the device with the
 *            default values from the database and opens a
 *            file path for I/O operations.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceOpen
 *               DevErr_DriverNotFound
 *
 *            +  errors returned by db_getresource()
 *
 **************************************************************/
static long object_initialise(Rga ds,long *error)
{
    char *answer=NULL;	
    Rga         ga = (Rga) ds;
    short channel;
    long ret;
    DevVarShortArray ChanParam;
    *error = FALSE;
    
    ChanParam.sequence=(short *)malloc(3*sizeof(short));
    ChanParam.length=3;


#ifdef DEBUG_LVL1
    printf("\t<RGA object_initialise>\n");
#endif
    ga->rga.FilamentUsed=0;
    ga->rga.AutoScanMode=0;
    ga->rga.max_noise=1e-13;
    ga->rga.Multiplier = 1;
    ga->rga.MultiplierGain = 2;
    ga->rga.MultiplierVolt = 720;
    ga->rga.MultiplierBase = 700;
    ga->rga.TotalPressure = 0;
    ga->rga.Accuracy = 3;
    ga->rga.FirstMass = 1;
    ga->rga.MassZoom = 64;
    ga->rga.Range = 10;
    ga->rga.AutoRange = 0;
    ga->rga.max_mass = 100;
    ga->rga.reset_timeout = 1800; /* old 180*/
    ga->rga.serial_driver_name="none";
    ga->rga.serial_line_handle=0; /*NULL*/
    
   /*
     * Read object resources from database
     */

#ifdef DEBUG_LVL2
    printf("\t - Reading resources\n");
#endif
    
    obj_tab[0].resource_adr = &(ga->rga.description);
    obj_tab[1].resource_adr = &(ga->rga.serial_driver_name);
    obj_tab[2].resource_adr = &(ga->rga.max_mass);
    obj_tab[3].resource_adr = &(ga->rga.default_aux_vacuum_gauge);
    obj_tab[4].resource_adr = &(ga->rga.default_aux_gauge_check);
    obj_tab[5].resource_adr = &(ga->rga.FilamentUsed);
    obj_tab[6].resource_adr = &(ga->rga.AutoScanMode);
    obj_tab[7].resource_adr = &(ga->rga.ChanMasses);
    obj_tab[8].resource_adr = &(ga->rga.ChanAutoRange);
    obj_tab[9].resource_adr = &(ga->rga.ChanMultiplier);
    obj_tab[10].resource_adr = &(ga->rga.max_noise);
    obj_tab[11].resource_adr = &(ga->rga.Multiplier);
    obj_tab[12].resource_adr = &(ga->rga.MultiplierGain);
    obj_tab[13].resource_adr = &(ga->rga.TotalPressure);
    obj_tab[14].resource_adr = &(ga->rga.Accuracy);
    obj_tab[15].resource_adr = &(ga->rga.FirstMass);
    obj_tab[16].resource_adr = &(ga->rga.MassZoom);
    obj_tab[17].resource_adr = &(ga->rga.Range);
    obj_tab[18].resource_adr = &(ga->rga.AutoRange);
    obj_tab[19].resource_adr = &(ga->rga.MultiplierVolt);
    obj_tab[20].resource_adr = &(ga->rga.MultiplierBase);
    obj_tab[21].resource_adr = &(ga->rga.reset_timeout);

    if (db_getresource(ga->devserver.name, obj_tab, obj_tab_size, error) != 0)
        return(DS_NOTOK);

    printf("\t\t\tDescription      : %s\n", ga->rga.description);
    printf("\t\t\tSerial Drive Name: %s\n", ga->rga.serial_driver_name);
    printf("\t\t\tMax. Mass        : %d\n", ga->rga.max_mass);
    printf("\t\t\tAux. Vacuum Gauge: %s\n", ga->rga.default_aux_vacuum_gauge);
    printf("\t\t\tAux. Gauge Check : %d\n", ga->rga.default_aux_gauge_check);

    /*
     *  Load the auxiliary gauge variables with the default values
     */
    get_aux_defaults(ga);


#ifdef DEBUG_PRINT
   printf ("importing serial line: %s\n", ga->rga.serial_driver_name);
#endif

   if ( dev_import (ga->rga.serial_driver_name, WRITE_ACCESS, 
		    &ga->rga.serial_line_handle, error ) == DS_NOTOK )
      {
        printf ("cannot import serial line %s error %d\n", ga->rga.serial_driver_name, *error);
      	return (DS_NOTOK);
      }

#ifdef DEBUG_PRINT
   printf ("imported serial line %s\n", ga->rga.serial_line_handle->device_name);
#endif

   /*
    *	Test the PPAS presence or no
    */
    set_up_comm(ga, error);

    if(get_op_mode(ga,error)<0)
    {
    	    	printf("MODE UNKNOWN\n");
    }
    
    if(ga->rga.OpMode.ScanMode != CONTROL_MODE)
    {
    	time(&ga->rga.when_last_cmd);
    	if(default_config(ga,NULL,NULL, error)==DS_NOTOK) 
	{
        	printf("I continue even I have an error on the initialisation\n");
	}
    }	

 	/* Create the multisignal object */
   
   ga->rga.msignal_obj=NULL;
   
   if( ds__create (ga->devserver.name , mDSSignalClass,
                         &ga->rga.msignal_obj, error) == DS_NOTOK){
    	return (DS_NOTOK);
   }
   
   if( ds__method_finder (ga->rga.msignal_obj,
   			  DevMethodInitialise)
   			  (ga->rga.msignal_obj,rgaClass->devserver_class.class_name,error) == DS_NOTOK ){
   	return (DS_NOTOK);
   }
    return (DS_OK);
}

/***********************************************
 * DevMethodStateHandler  ( state_handler() )
 **************************************************************/
/*
 *  Purpose:  Updates the filament state before executing the
 *            device server command.
 *  
 *  Remarks: - If the device was previously in the DEVUNKNOWN
 *             state, an attempt to establish communication is
 *             done. If it is successful, the filament status
 *             is read and the device state updated.
 *           - If the device server command is DevGetDescription,
 *             DevState, DevAuxGaugeControl, DevAuxGaugeName,
 *             or DevReset, the previous sequence is skipped.
 *           - If there is any problem when reading the state,
 *             the apropiate error code is returned and the
 *             internal state is turned to DEVUNKNOWN.
 *
 * Returns:  DS_OK      If no error
 *           DS_NOTOK   If error
 *
 * Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_FilamentStateError
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long state_handler(Rga ga,long ds_cmd,long *error)
{
    short iret;
    time_t now;
    *error = FALSE;
    switch(ds_cmd)
    {
    	case DevIndicator:
	case DevGetSigConfig:
	case DevUpdateSigConfig:
	case DevGetDescription:
	case DevRestore:
 	case DevReset:
 	case DevAuxGaugeName:
 	case DevReadVoltage:
             break;
	default:
    	if(get_state(ga, error)==DS_NOTOK) iret=DS_NOTOK;
    	if (mode_update(ga, error) == DS_NOTOK) iret =DS_NOTOK;
    }
       	
    iret = 0;
    switch(ds_cmd)
    {
         case DevFilamentOn:
         case DevFilamentOff:
         case DevOn:
         case DevOff:
         case DevSendCommand:
         case DevSetMode:
         case DevSetParam:
         case DevSetChannel:
             time(&ga->rga.when_last_cmd);
             ga->rga.new_scan = True;
              break;
    }
    switch(ga->rga.OpMode.ScanMode)
    {
        case PEAK_JUMP_MODE:
             break;
        
	case CONTROL_MODE:
	     switch(ds_cmd)
	     {
	     	case DevFilamentOn:
		case DevFilamentOff:
		case DevOn:
         	case DevSetMode:
         	case DevOff:
         	case DevSetParam:
			*error=DevErr_AttemptToViolateStateMachine;
		iret=DS_NOTOK;
	     	break;
	      }	
            break;
	     
	default:
    /*
     * If a channel spectrum has been requested while the scanning mode is not Peak Jump,
     * we can consider that someone have changed the configuration.
     * If this problem continue, and no other command is sended after the reset_timeout,
     *  we reconfigure the Rga by executing the DevReset command.
     ***********************************************************************************/
               switch(ds_cmd)
	       {  
		 case DevReadSigValues:
                 case DevReadNChannels:
                      *error=DevErr_AttemptToViolateStateMachine;
                      printf("%s scanning mode not correct for peak jump\n",ga->devserver.name);
                     	iret=DS_NOTOK;
                	time(&now);
                	if(now - ga->rga.when_last_cmd > ga->rga.reset_timeout)
                	{
                     		printf("%s Automatic reset to peakjump mode in state handler\n",ga->devserver.name);
                     		printf("difftime=%d,timeout=%d\n",now - ga->rga.when_last_cmd,
                                                         ga->rga.reset_timeout);
                     		devreset(ga,NULL,NULL,error);
                	}
                      break;
		}
    }
    
    /*
     * If a spectrum has been requested while the filament is not ON, there is a problem.
     * If this problem continue, we reconfigure the Rga by executing the DevReset command.
     ***********************************************************************************/
    if(ga->devserver.state != DEVON && ds_cmd == DevReadSigValues)
    {
            *error=DevErr_NoInputVoltage;
             iret=DS_NOTOK;
             if(ga->rga.FilamentUsed != 0)
             {
                printf("%s  Filament is %s: not correct for peak jump reading\n",
                                                         ga->devserver.name,
                                                         DEVSTATES[ga->devserver.state]);
                time(&now);
                if(now - ga->rga.when_last_cmd > ga->rga.reset_timeout)
                {
                   printf("%s Automatic reset to peakjump mode\n",ga->devserver.name);
                   printf("difftime=%d,timeout=%d\n",now - ga->rga.when_last_cmd,
                                                         ga->rga.reset_timeout);
                   devreset(ga,NULL,NULL,error);
                }
             }
    }
    return(iret);
}

/******************************
 *  DevState  ( dev_state() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client the filament status of
 *            the SATELLITE unit.
 *
 *  Remarks: - If the device was previously in the DEVUNKNOWN
 *             state, an attempt to establish communication is
 *             done. If it is successful, the filament status
 *             is read and the device state updated.
 *           - If the instrument is not connected or there is
 *             a hardware failure, the associated error is
 *             returned and the internal state is turned
 *             to DEVUNKNOWN.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error (state = DEVUNKNOWN)
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_FilamentStateError
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long dev_state(Rga ga,DevVoid *argin,DevLong *argout,long *error)

/*Rga          ga;         In:  Pointer to device server structure */
/*DevVoid     *argin;           Dummy         */
/*DevLong     *argout;     Out: State, the possible values are:         *
                         *        DEVOFF     - Filament off              *
                         *        DEVON      - Filament on               *
                         *        DEVFAULT   - Filament fail             *
                         *        DEVTRIPPED - Total pressure, partial   * 
                         *                     pressure or external trip */
/*long        *error;      Out: If error contains the error code */
{
    *error = FALSE;

#ifdef DEBUG_LVL1
    printf("\t<RGA devstate>\n");
#endif
    *argout = (DevLong) ga->devserver.state;
    if (ga->devserver.state == DEVUNKNOWN)
        return(DS_NOTOK);

    return(DS_OK);
}

/*************************************************
 *  DevStatus  ( dev_status() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client the status of the device.
 *            The status is a string made of the informations 
 *	      returnde by "devoperationmode".
 *   
 *                            
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:              
 *
 **************************************************************/
static long dev_status(Rga ga,DevVoid *argin,DevString *argout,long *error)
{
    static char status_str[200], mass_str[200];
    char pres_str[200], header_str[25], tmp[15];
    DevVarShortArray short_array, sht_arr;
    DevVarFloatArray float_array;
    DevRgaOpMode *opmode_ptr;
    DevRgaChannParam *param_ptr;    
    long ret_readval, ret_channparam;
    int i=0;
    
    *error = FALSE;

    /* Get operation mode using devoperationmode() */
    if (devoperationmode(ga, NULL, &short_array, error) == DS_NOTOK)       
    	return(DS_NOTOK);
    
    opmode_ptr = (DevRgaOpMode *)(short_array.sequence);
  
    sprintf(header_str, "Filament %d is %s", 
    	    ga->rga.FilamentUsed, 
    	    DEVSTATES[ga->devserver.state]);
   
    /* Build up status string according to Rga mode */
    switch(opmode_ptr->ScanMode)
    {    	
    	case LEAK_CHECK_MODE :  
    	
    		sprintf(status_str, 
    			"%s\012Leak Check Mode\012Multiplier = %d\012Speed = %d\012Mass = %d\012MassTune = %d\012Range = %d\012AutoRange = %d\012High Alarm = %d\012Low Alarm  = %d\012Audio = %d",
    			header_str,
    			opmode_ptr->Parameter.LeakCheck.Multiplier,
    			opmode_ptr->Parameter.LeakCheck.Speed,
    			opmode_ptr->Parameter.LeakCheck.Mass,
    			opmode_ptr->Parameter.LeakCheck.MassTune,
    			opmode_ptr->Parameter.LeakCheck.Range,
    			opmode_ptr->Parameter.LeakCheck.AutoRange,
    			opmode_ptr->Parameter.LeakCheck.HighAlarm,
    			opmode_ptr->Parameter.LeakCheck.LowAlarm,
    			opmode_ptr->Parameter.LeakCheck.Audio);
    			
    			break;
    				
  	case BAR_CHART_MODE : 
  	
  		sprintf(status_str, 
    			"%s\012Bar Chart Mode\012Multiplier = %d\012TotalPressure = %d\012Speed = %d\012FirstMass = %d\012LastMass = %d\012Range = %d\012AutoRange = %d",
    			header_str,
    			opmode_ptr->Parameter.BarChart.Multiplier,
    			opmode_ptr->Parameter.BarChart.TotalPressure,
    			opmode_ptr->Parameter.BarChart.Speed,
    			opmode_ptr->Parameter.BarChart.FirstMass,
    			opmode_ptr->Parameter.BarChart.LastMass,
    			opmode_ptr->Parameter.BarChart.Range,
    			opmode_ptr->Parameter.BarChart.AutoRange);
    			
    			break; 
    				
	case PEAK_JUMP_MODE : 
			
		/* Get masses array using devchannelparam() */
		ret_channparam = devchannelparam(ga, NULL, &sht_arr, error);
			
		/* Get pressures array using devreadvalues() */
		float_array.sequence = 0;
		ret_readval = devreadvalues(ga, NULL, &float_array, error);
			
		param_ptr = (DevRgaChannParam *)(sht_arr.sequence);
		
		if(param_ptr->Channel[0].Parameter.number[1] == True)/*0*/
		{
			if (ret_channparam == DS_OK)
				sprintf(mass_str, 
					"Mass     :%10d", 				 
				param_ptr->Channel[0].Parameter.number[2]);/*0*/
			else
				sprintf(mass_str, 
					"Mass     : no data, scan in process");	
			
			if (ret_readval == DS_OK)
				sprintf(pres_str, 
					"Pressure : %.1e  ", 				
					float_array.sequence[0]);/*0*/
			else
				sprintf(pres_str, 
					"Pressure : no data, scan in process");
		}				
		
		for (i=1;i < RGA_NUM_OF_CHANNELS  ;i++)
		{
			/* Get infos only for valid channels */
			if(param_ptr->Channel[i].Parameter.number[1] == True )
			{
				if (ret_channparam == DS_OK)
				{
					sprintf(tmp, "%13d", 
			        	param_ptr->Channel[i].Parameter.number[2]);				
					strcat(mass_str, tmp); 
				}
			
				if (ret_readval == DS_OK)
				{
					sprintf(tmp, "%.1e  ",
			        	 	float_array.sequence[i]);			
					strcat(pres_str, tmp); 
				}
			}
									     
		}
		
		sprintf(status_str, 
    			"%s\012Peak Jump Mode\012TotalPressure = %d\012Speed = %d\012%s\012%s",
    			header_str,
    			opmode_ptr->Parameter.PeakJump.TotalPressure,
    			opmode_ptr->Parameter.PeakJump.Speed,
    			mass_str,
    			pres_str);    				
    		break;
	
	case ANALOG_MODE : 
	
		sprintf(status_str, 
    			"%s\012Analog Mode\012Multiplier = %d\012TotalPressure = %d\012Speed = %d\012FirstMass = %d\012MassZoom = %d\012Range = %d\012AutoRange = %d",
    			header_str,
    			opmode_ptr->Parameter.Analog.Multiplier,
    			opmode_ptr->Parameter.Analog.TotalPressure,
    			opmode_ptr->Parameter.Analog.Speed,
    			opmode_ptr->Parameter.Analog.FirstMass,
    			opmode_ptr->Parameter.Analog.MassZoom,
    			opmode_ptr->Parameter.Analog.Range,
    			opmode_ptr->Parameter.Analog.AutoRange);
    				
    		break;  
    				
	case NO_SCAN_MODE : 
			
		sprintf(status_str, 
			"%s\012%d = Scan Off",
			header_str,
    			opmode_ptr->ScanMode);
						
		break;
										
	case SINGLE_PEAK_MODE : 
				
		sprintf(status_str, 
			"%s\012%d = Single Peak Mode",
			header_str,
    			opmode_ptr->ScanMode);
						
		break;
						
	case FAST_SCAN_MODE : 
						
		sprintf(status_str, 
			"%s\012%d = Fast Scan Mode",
			header_str,
    			opmode_ptr->ScanMode);
						
		break;
						
	case DEGAS_MODE : 
				
		sprintf(status_str, 
			"%s\012%d = Degas Mode",
			header_str,
    			opmode_ptr->ScanMode);
						
		break;
	
	case CONTROL_MODE :
		
		/* Get pressures array using devreadvalues() */
		float_array.sequence = 0;
		ret_readval = devreadvalues(ga, NULL, &float_array, error);
		
		for (i=0;i < RGA_NUM_OF_CHANNELS ;i++)  /*RGA_NUM_OF_CHANNELS - 1*/
		{				
			if (ret_readval == DS_OK)
			{
				sprintf(tmp, "%.1e  ",
			       	 	float_array.sequence[i]);				
				strcat(pres_str, tmp); 
			}
		}						     
		
		
		sprintf(status_str, 
			"%s\012%d = Control Mode \012",
			header_str,
    			opmode_ptr->ScanMode);
		
		strcat(status_str,pres_str);
		
		break;
						
	default : sprintf(status_str, 
			  "%s\012%d = Unknown Mode",
			  header_str,
    			  opmode_ptr->ScanMode);										    				   					    				    				   				
    }
   
    *argout = status_str;
    
    return(DS_OK);
}

/*******************************
 *  DevFilamentState  ( devstate() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client the filament status of
 *            the SATELLITE unit.
 *
 *  argin none;
 * argout D_VAR_SHORTARR sequence[0]=filament number;
 *                       sequence[1]= filament status;
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error (state = DEVUNKNOWN)
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_FilamentStateError
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devfilamentstate(Rga ga,DevVoid *argin,DevVarShortArray *argout,long *error)
{
    char   *status=NULL;
    static short tabshort[2];
    
    argout->length=2;
    argout->sequence=tabshort;
    argout->sequence[0] = ga->rga.FilamentUsed;
    argout->sequence[1] = ga->devserver.state;
    return(DS_OK);
}
/*******************************
 *  DevIndicator  ( devindicator() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client the scan index
 *            .
 *
 *  argin none;
 * argout D_SHORT_TYPE    mass beiing scanned;
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error (state = DEVUNKNOWN)
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_FilamentStateError
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devindicator(Rga ga,DevVoid *argin,DevShort *argout,long *error)
{
    char   *status=NULL;
    long result;
    if(send_request(ga,INDEX_REQUEST,&status,error) == DS_NOTOK)
    {
        return(DS_NOTOK);
    }
    status++;
    result = atoi(status);
    *argout = (short)result;
    return(DS_OK);
}
/*******************************
 *  DevReadVoltage  ( devreadvoltage() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client multiplier voltage
 *            .
 *
 *  argin none;
 * argout D_SHORT_TYPE    multiplier voltage;
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error (state = DEVUNKNOWN)
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_FilamentStateError
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devreadvoltage(Rga ga,DevVoid *argin,DevShort *argout,long *error)
{
    char   *status=NULL;
    float result;
    if(send_request(ga,VOLT_REQUEST,&status,error) == DS_NOTOK)
    {
        return(DS_NOTOK);
    }
    status++;
    result = atof(status);
    *argout = ga->rga.MultiplierBase + (short)(2.87 * result);
    printf("devreadvoltage: %s -> %d\n",status,*argout);
    return(DS_OK);
}

/***************************************
 * DevFilamentOn  ( devfilamenton() )
 **************************************************************/
/*
 *  Purpose:  Switchs the selected filament on.
 * 
 *  Remarks: - If the auxiliary gauge check flag is set, before
 *             switching the filament on, the function verifies
 *             that the pressure reading from the auxiliary
 *             gauge is below the maximum pressure.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_ValueOutOfBounds
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_AuxiliaryGaugeOff
 *               DevErr_AuxPressureTooHigh
 *
 *             + errors returned by: dev_import()
 *                                   dev_putget()
 *
 *             + errors returned by the commands DevState and
 *                 DevGaugeReadOut implemented in the device
 *                 server that controls the auxiliary gauge.
 *
 **************************************************************/
static long devfilamenton(Rga ga,DevShort *argin,DevVoid *argout,long *error)
{
    short filament;
    char  command[10];
    short channel;
    short tabparam[3];
    DevVarShortArray ChanParam;
    long	devstatus;

    ChanParam.sequence=tabparam;
    ChanParam.length=3;
    filament = *argin;
    *error = FALSE;

    /*
     *  Check filament value
     */
    if (filament != 1 && filament != 2)
    {
        *error = DevErr_ValueOutOfBounds;
          return(DS_NOTOK);
    }
    /*
     *  If necessary, check the auxiliary gauge pression
     */
    if (ga->rga.actual_aux_gauge_check != 0)
        if (check_aux_pressure(ga, error) == DS_NOTOK)
            return(DS_NOTOK);

    /*
     * Make up the command (Fn,1) , n = filament
     */
    
    sprintf(command, FILAMENT_ON_PATTERN, filament);

    /*
     * Send the command (Fn,1) to the SATELLITE
     */
    
    if (send_string(ga,command, error) == DS_NOTOK )
          return(DS_NOTOK);
    /*
     * and wait for acknowledgement.
     * The timeout is enlarged  because switching on the filaments
     * takes more than two seconds.
     */
    ChanParam.sequence=tabparam;
    ChanParam.length=2;
    
    
    ChanParam.sequence[0]=SL_TIMEOUT;
    ChanParam.sequence[1]= 2000;
    if(dev_putget(ga->rga.serial_line_handle, DevSerSetParameter,
  			&ChanParam, D_VAR_SHORTARR,
  			NULL, D_VOID_TYPE,
  			error)==DS_NOTOK)
			return(DS_NOTOK);
    printf("read with long timeout ");
    fflush(stdout);
    if ( get_answer(ga, error) == DS_NOTOK)
    {
         return(DS_NOTOK);
    }
    ChanParam.sequence[0]=SL_TIMEOUT;
    ChanParam.sequence[1]= 200;
    if(dev_putget(ga->rga.serial_line_handle, DevSerSetParameter,
  			&ChanParam, D_VAR_SHORTARR,
  			NULL, D_VOID_TYPE,
  			error)==DS_NOTOK)
			return(DS_NOTOK);
     /*
      * If we are in peak jump mode, it is necessary to reasign the multiplier
      * for the channels in order to have readings (BUG in leda-mass)
      */     
      ChanParam.length=3;
      if(ga->rga.OpMode.ScanMode==PEAK_JUMP_MODE) 
      {
         for(channel=0;channel<ga->rga.ChanMultiplier.length;channel++)
         {
            ChanParam.sequence[1]=channel+1;
            ChanParam.sequence[0]=CHAN_MULTIPLIER;
            ChanParam.sequence[2]=ga->rga.ChanMultiplier.sequence[channel];
            if(devsetchannel(ga,&ChanParam,NULL,error) == DS_NOTOK)
                return (DS_NOTOK);
         }
      }
    return(DS_OK);
}

/***************************************
 * DevOn  ( dev_on() )
 **************************************************************/
/*
 *  Purpose:  Switches the currently used filament on.
 * 
 *  Remarks: - If the auxiliary gauge check flag is set, before
 *             switching the filament on, the function verifies
 *             that the pressure reading from the auxiliary
 *             gauge is below the maximum pressure.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:     same as devfilamenton()          
 *
 **************************************************************/
static long dev_on(Rga ga,DevVoid *argin,DevVoid *argout,long *error)
{	
	*error = FALSE;
	
	if (devfilamenton(ga, &ga->rga.FilamentUsed, NULL, error) == DS_NOTOK)
		return(DS_NOTOK);
	
	return(DS_OK);
}

/*****************************************
 * DevFilamentOff  ( devfilamentoff() )
 **************************************************************/
/*
 *  Purpose:  Switchs the filaments off.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *
 **************************************************************/
static long devfilamentoff(Rga ga,DevVoid *argin,DevVoid *argout,long *error)
{
    char *answer;
    *error = FALSE;
    /*
     *  Send the command (F1,0) to SATELLITE
     *  (this command switches off either filament),
     */
    if (send_request(ga,FILAMENT_OFF_COMMAND,&answer,error)==DS_NOTOK) return(DS_NOTOK);
       return(DS_OK);
}

/**************************************
 *  DevSendCommand  ( devcommand() )
 **************************************************************/
/*
 *  Purpose:  Sends a command to the SATELLITE unit.
 *
 *  Remarks: - The command is an ASCII string without brackets.
 *           - Only those commands starting with a letter
 *             included in the string RGA_COMMANDS are actually
 *             sent to the instrument.
 *           - The commands for filament control and reset must
 *             be sent through the special device server
 *             commands implemented for such purposes.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_DeviceSyntaxError
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devcommand(Rga ga,DevString *argin,DevVoid *argout,long *error)
{
    char  *command = (char *) *argin;
    char *answer;
    *error = FALSE;
    /*
     *  send the command string to the instrument,
     */
    if (send_request(ga,command,&answer,error)==DS_NOTOK) return(DS_NOTOK);
    /*
     *  If the command was successful, the variables that contain the
     *  mode of operation of the instrument are updated.
     */
    if (mode_update(ga, error) == DS_NOTOK)
             return (DS_NOTOK);

    return(DS_OK);
}

/**************************************
 *  DevSendRequest  ( devrequest() )
 **************************************************************/
/*
 *  Purpose:  Sends a request to the SATELLITE unit and
 *            returns the answer to the client.
 *
 *  Remarks: - The request is an ASCII string without brackets.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_DeviceSyntaxError
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devrequest(Rga ga,DevString *argin,DevString *argout,long *error)
{
    char  *request = (char *) *argin;
    char  *answer=NULL;

     *error = FALSE;
    /*
     * Send request string and get the answer
     */
    if (send_request(ga, request, &answer, error) == DS_NOTOK )
               return(DS_NOTOK);
    *argout = (DevString) answer;
    return(DS_OK);
}

/**************************************
 *  DevReadValues  ( devreadvalues() )
 **************************************************************/
/*
 *  Purpose:  return to the client the last read spectrum
 *
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_DeviceSyntaxError
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devreadvalues(Rga ga,void *argin,DevVarFloatArray *argout,long *error)
{
     char  *answer=NULL;
     char *ptnext=NULL;
     time_t now;
     short i;
     static float floattab[256];
     float range_factor,over_range, under_range;
     *error = FALSE;
    /*
     * If a parameter has been modifyed, then wait for a new scan available before returning a spectra
     */
    if(ga->rga.new_scan)
    {
         if (send_request(ga,IS_NEW_SCAN , &answer, error) == DS_NOTOK )
             return(DS_NOTOK);
         else
         {
            if(answer[0] == '0')
            {
               *error = DevErr_NoData;
               return(DS_NOTOK);
            }
         }
    }   
    time(&now);
    argout->sequence=floattab;
    switch(ga->rga.OpMode.ScanMode)
    {
    	case(BAR_CHART_MODE):
         	printf("Read values in Bar Chart Mode\n");
       		range_factor = (float)pow((double)10,-(double)(ga->rga.OpMode.Parameter.BarChart.Range));
       		over_range = 10 * range_factor;
       		under_range = 0.05 * range_factor;
    		if (long_request(ga,SPECTRUM_REQUEST , &answer, error) == DS_NOTOK )
		{
			return(DS_NOTOK);
		}	 
		break;
    	case(ANALOG_MODE):
       		range_factor = (float)pow((double)10,-(double)(ga->rga.OpMode.Parameter.Analog.Range));
       		over_range = 10 * range_factor;
       		under_range = 0.05 * range_factor;
       		printf("Read values in analog mode (%d) range = %d\n",ga->rga.OpMode.ScanMode,ga->rga.OpMode.Parameter.Analog.Range);
    		if (long_request(ga,SPECTRUM_REQUEST, &answer, error) == DS_NOTOK )
		{
			return(DS_NOTOK);
		}
		break;
    	case(PEAK_JUMP_MODE):
     		over_range = OVER_RANGE;
     		under_range = ga->rga.max_noise;
        	printf("Read values in peak jump mode (%d)\n",ga->rga.OpMode.ScanMode);
    		if (send_request(ga,SPECTRUM_REQUEST , &answer, error) == DS_NOTOK )
		{
        	    if(now - ga->rga.when_last_cmd > ga->rga.reset_timeout)
        	    {
        	       printf("Automatic reset to peakjump mode in command\n");
        	       printf("now=%d , last=%d,timeout=%d\n",now,
                                                      ga->rga.when_last_cmd,
                                                      ga->rga.reset_timeout);
        	       devreset(ga,NULL,NULL,error);
        	    }
		    return(DS_NOTOK);
		}
	 	argout->length=ga->rga.ChanMasses.length+1;/* */
	 	argout->sequence[argout->length-1]=ga->rga.OpMode.ScanMode;
         	ga->rga.when_last_cmd = now;
		break;
    	case(CONTROL_MODE):
     		over_range = OVER_RANGE;
     		under_range = ga->rga.max_noise;
     	        printf("Read values in CONTROL MODE (%d)\n",ga->rga.OpMode.ScanMode);
    		if (send_request(ga,PPAS_SPECTRUM , &answer, error) == DS_NOTOK )
		{
			return(DS_NOTOK);
		}
	 	argout->length=ga->rga.ChanMasses.length+1;/* */
	 	argout->sequence[argout->length-1]=ga->rga.OpMode.ScanMode;
         	ga->rga.when_last_cmd = now;
		break;
    }
    
    /*
     *  Increment answer to the Acknowledge charactere
     */
    
    answer++;
    for(ptnext=answer,i=0;ptnext!=(char *)1;ptnext=(char *)index(ptnext,',')+1,i++)
    {
        switch(ptnext[0])
        {
           case '>':  /* then the RGA indicate an over range, we should put a typical value */
                    floattab[i] = over_range;
                    break;
           case '*':  /* then the RGA indicate an under range, we should put a typical value */
                    floattab[i] = under_range;
                    break;
           
           default:   /* then the value is valid */
                    sscanf(ptnext,"%e",&floattab[i]);

                    if(floattab[i]<under_range) floattab[i]=under_range;
                    break;
        }
    }
	/*
	 *  ANALOG_MODE
	 */
    if(ga->rga.OpMode.ScanMode==ANALOG_MODE||ga->rga.OpMode.ScanMode==BAR_CHART_MODE)
    {
       argout->length=i;
    }
    ga->rga.new_scan = False; 
    return(DS_OK);
}

/*****************************************
 * DevSetParam  ( devsetparam() )
 **************************************************************/
/*
 *  Purpose:  set a parameter value.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *  
 *  argin: DevVarShortArray
 *
 *  argout:
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *
 **************************************************************/
static long devsetparam(Rga ga,DevVarShortArray *argin,DevVoid *argout,long *error)
{
    static char cmdstring[24];
    char *answer;
    short saved_param;
    *error = FALSE;

    printf("RGA devsetparam(%d,%d)\n",argin->sequence[0],argin->sequence[1]);
    if(ga->rga.OpMode.ScanMode==ANALOG_MODE)
    {
    /* save the present value of the requested parameter in order to be able
       to put it back in the parameter structure in case of failure.
    */
       saved_param = ga->rga.OpMode.Parameter.number[argin->sequence[0]-1];
    /* 
       Then put the requested value of the parameter in the structure in order to
       build the command string
     */
       ga->rga.OpMode.Parameter.number[argin->sequence[0]-1] = argin->sequence[1];
       
        sprintf(cmdstring,"M%d,%d,%d,%d,%d,%d,%d,%d",ga->rga.OpMode.ScanMode,
 					ga->rga.OpMode.Parameter.Analog.Multiplier,
 					ga->rga.OpMode.Parameter.Analog.TotalPressure,
  					ga->rga.OpMode.Parameter.Analog.Speed,
					ga->rga.OpMode.Parameter.Analog.FirstMass,
 					ga->rga.OpMode.Parameter.Analog.MassZoom,
 					ga->rga.OpMode.Parameter.Analog.Range,
 					ga->rga.OpMode.Parameter.Analog.AutoRange
 					);
       
    /* restore the present value of the requested parameter 
       back in the parameter structure in case of failure of the next command.
    */
      ga->rga.OpMode.Parameter.number[argin->sequence[0]-1] = saved_param;
     /*
     *  Send the command to SATELLITE
     */
     if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         return(DS_NOTOK);
    }
    else
    {
/*********************************
 * A bug in the ledamass firmware do not accept any change in the mass zoom
 * parameter (number 5) if the first mass parameter (number 4) is equal to 1.
 * To work around this problem, if this is the case, we first put the first
 * mass to 2, then we change the mass zoom, then we change back the first mass.
 ************************************************************************/
     if(argin->sequence[0]==5 && ga->rga.OpMode.Parameter.number[3] == 1)
     {
        sprintf(cmdstring,PARAMETER_SET_PATTERN,4,2);
     	if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         	return(DS_NOTOK);
     }
     sprintf(cmdstring,PARAMETER_SET_PATTERN,argin->sequence[0],argin->sequence[1]);
    /*
     *  Send the command to SATELLITE
     */
     if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         	return(DS_NOTOK);
		
     ga->rga.OpMode.Parameter.number[argin->sequence[0]-1] = argin->sequence[1];
     if(argin->sequence[0]==5 && ga->rga.OpMode.Parameter.number[3] == 1)
     {
         sprintf(cmdstring,PARAMETER_SET_PATTERN,4,1);
     	 if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         	return(DS_NOTOK);
		
     }
    }
    if (get_op_mode(ga, error) == DS_NOTOK)
             return (DS_NOTOK);
    return(DS_OK);
}
/*****************************************
 * DevSetChannel  ( devsetchannel() )
 **************************************************************/
/*
 *  Purpose:  set a channel  parameter value.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *  
 *  argin: DevVarShortArray sequence[0]=channel number
 *                         sequence[1]=parameter number
 *                         sequence[2]=value
 *
 *  argout:
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *
 **************************************************************/
static long devsetchannel(Rga ga,DevVarShortArray *argin,DevVoid *argout,long *error)
{
    static char cmdstring[24];
    char *answer;
    *error = FALSE;
    
    sprintf(cmdstring,CHANNEL_SET_PATTERN,argin->sequence[0],
                                           argin->sequence[1],
                                           argin->sequence[2]);
    /*
     *  Send the command to SATELLITE
     */
     if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         return(DS_NOTOK);
	 
   if ( get_single_channel(ga,argin->sequence[0],argin->sequence[1],argin->sequence[2],error)== DS_NOTOK )
        return(DS_NOTOK);
    return(DS_OK);
}
/*****************************************
 * DevSetMultiplierGain  ( devsetmultipliergain() )
 **************************************************************/
static long devsetmultipliergain(Rga ga,DevShort *argin,DevVoid *argout,long *error)
{
    static  char cmdstring[24];
    char *answer;
    *error = FALSE;
     if (*argin != 2 && *argin != 3)
     {
        *error=DevErr_ValueOutOfBounds;
        return(DS_NOTOK);
     }

     sprintf(cmdstring,SET_MULTIPLIER_GAIN_PATTERN,*argin);
     if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         return(DS_NOTOK);
 }

/*****************************************
 * DevSetMode  ( devsetmode() )
 **************************************************************/
/*
 *  Purpose:  set a parameter value.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *  
 *  argin: DevShort
 *
 *  argout:
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *
 **************************************************************/
static long devsetmode(Rga ga,DevShort *argin,DevVoid *argout,long *error)
{
    static  char cmdstring[24];
    char *answer;
    *error = FALSE;

     switch (*argin)
     {
          case ANALOG_MODE:
  	ga->rga.OpMode.Parameter.Analog.Multiplier = ga->rga.Multiplier; 
 	ga->rga.OpMode.Parameter.Analog.TotalPressure = ga->rga.TotalPressure;
 	ga->rga.OpMode.Parameter.Analog.Speed = ga->rga.Accuracy;
	ga->rga.OpMode.Parameter.Analog.FirstMass = ga->rga.FirstMass;
 	ga->rga.OpMode.Parameter.Analog.MassZoom = ga->rga.MassZoom;
 	ga->rga.OpMode.Parameter.Analog.Range = ga->rga.Range;
 	ga->rga.OpMode.Parameter.Analog.AutoRange = 0;

        sprintf(cmdstring,"M%d,%d,%d,%d,%d,%d,%d,%d",*argin,
 					ga->rga.Multiplier,
 					ga->rga.TotalPressure,
  					ga->rga.Accuracy,
					ga->rga.FirstMass,
 					ga->rga.MassZoom,
 					ga->rga.Range,
 					ga->rga.AutoRange
 					);
 	  break;
          case PEAK_JUMP_MODE:
         sprintf(cmdstring,"M%d,%d,%d",*argin,
 					ga->rga.TotalPressure,
  					ga->rga.Accuracy
 					);
 	  break;
          case BAR_CHART_MODE:
  	ga->rga.OpMode.Parameter.BarChart.Multiplier = ga->rga.Multiplier; 
 	ga->rga.OpMode.Parameter.BarChart.TotalPressure = ga->rga.TotalPressure;
 	ga->rga.OpMode.Parameter.BarChart.Speed = ga->rga.Accuracy;
	ga->rga.OpMode.Parameter.BarChart.FirstMass = ga->rga.FirstMass;
 	ga->rga.OpMode.Parameter.BarChart.LastMass = ga->rga.max_mass;
 	ga->rga.OpMode.Parameter.BarChart.Range = ga->rga.Range;
 	ga->rga.OpMode.Parameter.BarChart.AutoRange = 0;

        sprintf(cmdstring,"M%d,%d,%d,%d,%d,%d,%d,%d",*argin,
 					ga->rga.Multiplier,
 					ga->rga.TotalPressure,
  					ga->rga.Accuracy,
					ga->rga.FirstMass,
 					ga->rga.max_mass,
 					ga->rga.Range,
 					ga->rga.AutoRange
 					);
 	  break;
 	  default:
          sprintf(cmdstring,"M%d",*argin);
 	  break;
     }
    /*
     *  Send the command to SATELLITE
     */
     printf(" setmode: %s\n",cmdstring);

     if (send_request(ga,cmdstring , &answer, error) == DS_NOTOK )
         return(DS_NOTOK);

    if (get_op_mode(ga, error) == DS_NOTOK)
             return (DS_NOTOK);
	     
    return(DS_OK);
}
/*************************************************
 *  DevGetOperationMode  ( devoperationmode() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client the gain of the multiplier,
 *            the state of the pressure trips, the scan mode
 *            and the corresponding parameters.
 *   
 *  Remarks: - The values are not actually read from the
 *             instrument unless the scan mode was Degas in
 *             the previous device server command. Instead a
 *             copy stored in the internal variables of the
 *             device server is returned. This copy is updated
 *             when the communication is set up or when a
 *             command is successfully sent to the instrument.
 *           - In Leak Cheak, Bar Chart and Analog Mode if the
 *             autorange feature is activated, the actual ranges
 *             are always read from the SATELLITE.
 *                            
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devoperationmode(Rga ga,DevVoid *argin,DevVarShortArray *argout,long *error)
{
    char  *answer=NULL;

    *error = FALSE;

#ifdef DEBUG_LVL1
    printf("\t<RGA devoperationmode> scan mode = %d\n",
                                             ga->rga.OpMode.ScanMode);
#endif

    switch(ga->rga.OpMode.ScanMode)
    {
        case DEGAS_MODE:
            break;
        case LEAK_CHECK_MODE:
        case BAR_CHART_MODE:
        case ANALOG_MODE:
                 if(send_request(ga, RANGE_REQUEST, &answer, error) == DS_NOTOK)
                     return(DS_NOTOK);
                 answer++;
		 ga->rga.OpMode.Parameter.number[5] = (short) atoi(answer) % 128;
#ifdef DEBUG_LVL4
    printf("\t\t\tRange = %d\n", ga->rga.OpMode.Parameter.number[5]);
#endif
    }
    argout->length   = RGA_NUM_OF_MODE_PARAMETERS;
    argout->sequence = (short *)&(ga->rga.OpMode);

    return(DS_OK);
}

/****************************************************
 *  DevGetChannelParameters  ( devchannelparam() )
 **************************************************************/
/*
 *  Purpose:  Returns the parameters of all the channels for
 *            scan modes Peak Jump and Fast Scan.
 *            If the scan mode is Peak Jump returns also the
 *            alarm status of the channels.
 *  
 *  Remarks: - The values are not actually read from the
 *             instrument. Instead a copy stored in the internal
 *             variables of the device server is returned.
 *             This copy is updated when the communication is
 *             set up or when a command is successfully sent to
 *             the instrument.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long devchannelparam(Rga ga,DevVoid *argin,DevVarShortArray *argout,long *error)
{
    char   generic_rq[6];
    char  *answer=NULL;
    short  nchan;

    *error = FALSE;

#ifdef DEBUG_LVL1
    printf("\t<RGA devchannelparam>\n");
#endif

    switch(ga->rga.OpMode.ScanMode)
    {
        case FAST_SCAN_MODE:
            break;

        case CONTROL_MODE:
	    break;
	
	case PEAK_JUMP_MODE:

            for (nchan =0 ; nchan < RGA_NUM_OF_CHANNELS ; nchan++)
	    {
                /*
                sprintf(generic_rq, ALARM_STATUS_REQUEST_PATTERN, nchan + 1);

                /*
                 *  Send the request (anchan) and store the answer
                if(send_request(ga, generic_rq, &answer, error) == DS_NOTOK)
                    return(DS_NOTOK);
		answer++;
                ga->rga.ChParam.Channel[nchan].AlarmStatus =
                                                     (short) atoi(answer);

                 */
#ifdef DEBUG_LVL4
    printf("\t\t\tAlarm status channel %d = %d\n", nchan + 1,
                                   ga->rga.ChParam.Channel[nchan].AlarmStatus);
#endif

                /*
                 *  Update the gain range if necessary
                 */
		 
                if (ga->rga.ChParam.Channel[nchan].Parameter.PeakJump.AutoRange
                                                                         == 1)
		{
                    /*
                     *  Make up the request (c4,nchan)
                     */
                    sprintf(generic_rq, RANGE_REQUEST_PATTERN, nchan + 1);
                    /*
                     *  Send the request (anchan) and store the answer
                     */
                    if(send_request(ga, generic_rq, &answer, error) == DS_NOTOK)
                        return(DS_NOTOK);
                    answer++;
		    ga->rga.ChParam.Channel[nchan].Parameter.PeakJump.Range =
                                                    (short) atoi(answer) % 128;
#ifdef DEBUG_LVL4
    printf("\t\t\tRange channel %d = %d\n", nchan,
                       ga->rga.ChParam.Channel[nchan].Parameter.PeakJump.Range);
#endif
                }
            }
            break;

        default:
            *error = DevErr_NotAcknowledged;
             return(DS_NOTOK);
    }

    /* The length is the number of channel parameter + the alarm status for all the channels*/
    argout->length   = (1 + RGA_NUM_OF_CHANNEL_PARAMETERS) * RGA_NUM_OF_CHANNELS; 
    argout->sequence = (short *)&(ga->rga.ChParam);

    return(DS_OK);
}

/******************************
 *  DevReset  ( devreset() )
 **************************************************************/
/*
 *  Purpose:  Sends a reset command to the SATELLITE unit.
 *  
 *  Remarks: - The device state is turned to DEVUNKNOWN and
 *             no attempt for setting up communication again
 *             is done.
 *           - The communication with the instrument will be
 *             re-established by DevMethodStateHandler if
 *             necessary in the next device server command
 *             call.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *
 **************************************************************/
static long devreset(Rga ga,DevVoid *argin,DevVoid *argout,long *error)
{
    char *str;
    *error = FALSE;
    /*
     * Send command (R) to SATELLITE,
     */

    if (send_request( ga, RESET_COMMAND,&str, error) == DS_NOTOK )
    {
              printf("error on reset, I continue\n");
    }
   /*
    *  get the resources
    */
    obj_tab[0].resource_adr = &(ga->rga.description);
    obj_tab[1].resource_adr = &(ga->rga.serial_driver_name);
    obj_tab[2].resource_adr = &(ga->rga.max_mass);
    obj_tab[3].resource_adr = &(ga->rga.default_aux_vacuum_gauge);
    obj_tab[4].resource_adr = &(ga->rga.default_aux_gauge_check);
    obj_tab[5].resource_adr = &(ga->rga.FilamentUsed);
    obj_tab[6].resource_adr = &(ga->rga.AutoScanMode);
    obj_tab[7].resource_adr = &(ga->rga.ChanMasses);
    obj_tab[8].resource_adr = &(ga->rga.ChanAutoRange);
    obj_tab[9].resource_adr = &(ga->rga.ChanMultiplier);
    obj_tab[10].resource_adr = &(ga->rga.max_noise);
    obj_tab[11].resource_adr = &(ga->rga.Multiplier);
    obj_tab[12].resource_adr = &(ga->rga.MultiplierGain);
    obj_tab[13].resource_adr = &(ga->rga.TotalPressure);
    obj_tab[14].resource_adr = &(ga->rga.Accuracy);
    obj_tab[15].resource_adr = &(ga->rga.FirstMass);
    obj_tab[16].resource_adr = &(ga->rga.MassZoom);
    obj_tab[17].resource_adr = &(ga->rga.Range);
    obj_tab[18].resource_adr = &(ga->rga.AutoRange);
    obj_tab[19].resource_adr = &(ga->rga.MultiplierVolt);
    obj_tab[20].resource_adr = &(ga->rga.MultiplierBase);
    obj_tab[21].resource_adr = &(ga->rga.reset_timeout);
    dev_xdrfree(D_VAR_SHORTARR,&(ga->rga.ChanMasses),error);
    dev_xdrfree(D_VAR_SHORTARR,&(ga->rga.ChanAutoRange),error);
    dev_xdrfree(D_VAR_SHORTARR,&(ga->rga.ChanMultiplier),error);
    if (db_getresource(ga->devserver.name, obj_tab, obj_tab_size, error) != 0)
        return(DS_NOTOK);

    ga->devserver.state = DEVUNKNOWN;
    time(&ga->rga.when_last_cmd);
#ifndef __cplusplus
    tsleep(500);
#else
    sleep(1);
#endif /* __cplusplus */
    set_up_comm(ga, error);
     /*
      * Get the information from the device
      */
    if (get_state(ga, error) == DS_NOTOK)
    {
#ifndef __cplusplus
       tsleep(500);
#else
       sleep(1);
#endif /* __cplusplus */
       if (get_state(ga, error) == DS_NOTOK)
       {
           ga->devserver.state = DEVUNKNOWN;
	}
    }
    if(default_config(ga,NULL,NULL, error)==DS_NOTOK) 
        return(DS_NOTOK);
    return(DS_OK);
}

/*********************************************
 *  DevGetDescription  ( devdescription() )
 **************************************************************/
/*
 *  Purpose:  Returns to the client string with a description
 *            or the maximum valid mass number.
 *  
 *  Remarks: - The input parameter determines if the return
 *             string contains the class description, the
 *             instrument description, or the maximum mass 
 *             number that the instrument accepts.
 *           - All these strings are defined in the resource
 *             file.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_ValueOutOfBounds
 *
 **************************************************************/
static long devdescription(Rga ga,DevShort *argin,DevString *argout,long *error)

/*      In:  Type of description, values: *
      *         RGA_CLASS_DESCRIPTION     *
      *         RGA_DEVICE_DESCRIPTION    *
      *         RGA_MAX_MASS              */
{
    *error = FALSE;
    switch(*argin)
    {
        case RGA_CLASS_DESCRIPTION:
            *argout = (DevString)
               ((RgaClass)ga->devserver.class_pointer)->rga_class.description;
            break;
        case RGA_DEVICE_DESCRIPTION:
            *argout = (DevString) ga->rga.description;
            break;
        case RGA_MAX_MASS:
            sprintf(answer_buff, "%d", ga->rga.max_mass);
            *argout = (DevString) answer_buff;
            break;
        default:
            *error = DevErr_ValueOutOfBounds;
            return(DS_NOTOK);
    }
    return(DS_OK);
}

/****************************************
 * DevAuxGaugeName  ( auxgaugename() )
 **************************************************************/
/*
 *  Purpose:  Lets the client read and modify the device name
 *            of the auxiliary gauge.
 *
 *  Remarks: - If the input string is empty, the name is not
 *             modified and the current name is returned.
 *           - If the input string contains a new name, the
 *             name is changed and the same name is returned.
 *           - The validity of the new name is not verified.
 *           - The maximum length of the name is RGA_MAX__LENGTH.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_StringTooLong
 *
 *             + errors returned by dev_free()
 *
 **************************************************************/
static long auxgaugename(Rga ga,DevString *argin,DevString *argout,long *error)
{
    short   length;
    char   *newname = (char *) *argin;

    *error = FALSE;

    if ((length = strlen(newname)) > 0)
    {
        if (length > RGA_MAX_LENGTH)
	{
            *error = DevErr_StringTooLong;
            return(DS_NOTOK);
        }
        if (ga->rga.aux_dserver_flag != 0)
            free_aux_gauge(ga, error);

        strcpy(ga->rga.actual_aux_vacuum_gauge, newname);
    }
    *argout = (DevString) ga->rga.actual_aux_vacuum_gauge;
    return(DS_OK);
}

/**********************************************
 * DevAuxGaugeControl  ( auxgaugecontrol() )
 **************************************************************/
/*
 *  Purpose:  Performs different actions that are related to 
 *            the auxiliary vaccum gauge.
 *            The possible actions are:
 *
 *             AUX_GAUGE_RESET_TO_DEFAULTS: replaces the device
 *               name and the pressure checking feature flag
 *               with the values loaded from the database.
 *
 *             AUX_GAUGE_GET_CHECK_FLAG: returns the pressure
 *               checking feature flag.
 *
 *             AUX_GAUGE_ENABLE_CHECK: activates the pressure
 *               checking feature.
 *
 *             AUX_GAUGE_DISABLE_CHECK: disables the pressure
 *               checking feature.
 *
 *             AUX_GAUGE_READ_PRESSURE: returns the current
 *               pressure from the gauge.
 *
 *             AUX_GAUGE_CHECK_PRESSURE: checks if the current
 *               pressure is below the maximum value defined in
 *               the database.
 *
 *             AUX_GAUGE_GET_MAX_PRESSURE: returns the maximum
 *               pressure value defined in the database.
 *
 *  Remarks: - The pressure checking feature flag follows the
 *             usual conventions for boolean variables.
 *           - The pressure checking feature only afects to the
 *             command DevFilamentOn. The access to the auxiliary
 *             gauge is always allowed in the DevAuxGaugeControl
 *             command.
 *           - Only gauges controlled by PG_ARUN device servers
 *             are accepted.
 *           - The gauge settings are not modified, only 
 *             information (state and pressure) is requested.
 *             
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_ValueOutOfBounds
 *               DevErr_AuxiliaryDeviceNotValid
 *               DevErr_AuxiliaryGaugeOff
 *               DevErr_AuxPressureTooHigh
 *
 *             + errors returned by: dev_import()
 *                                   dev_putget()
 *                                   dev_free()
 *
 *             + errors returned by the commands DevState and
 *                 DevGaugeReadOut implemented in the device
 *                 server that controls the auxiliary gauge.
 *
 *************************************************************/
static long auxgaugecontrol(Rga ga,DevShort *argin,DevFloat *argout,long *error)


/*        In:   Selects the action:                  *
                    *          AUX_GAUGE_RESET_TO_DEFAULTS       *
                    *          AUX_GAUGE_GET_CHECK_FLAG          *
                    *          AUX_GAUGE_ENABLE_CHECK            *
                    *          AUX_GAUGE_DISABLE_CHECK           *
                    *          AUX_GAUGE_READ_PRESSURE           *
                    *          AUX_GAUGE_CHECK_PRESSURE          *
                    *          AUX_GAUGE_GET_MAX_PRESSURE        */
{
    *error = FALSE;
    *argout = 0.0;

    switch(*argin){
        case AUX_GAUGE_RESET_TO_DEFAULTS:
            if (ga->rga.aux_dserver_flag != 0)
                free_aux_gauge(ga, error);
            get_aux_defaults(ga);
            break;

        case AUX_GAUGE_GET_CHECK_FLAG:
            *argout = (DevFloat) ga->rga.actual_aux_gauge_check;
            break;

        case AUX_GAUGE_ENABLE_CHECK:
            if (ga->rga.aux_dserver_flag != 0)
                free_aux_gauge(ga, error);
            ga->rga.actual_aux_gauge_check = 1;
            break;

        case AUX_GAUGE_DISABLE_CHECK:
            if (ga->rga.aux_dserver_flag != 0)
                free_aux_gauge(ga, error);
            ga->rga.actual_aux_gauge_check = 0;
            break;

        case AUX_GAUGE_READ_PRESSURE:
            if (read_aux_gauge(ga, error) == DS_NOTOK)
                return(DS_NOTOK);
            *argout = (DevFloat) ga->rga.aux_pressure;
            break;

        case AUX_GAUGE_CHECK_PRESSURE:
            if (check_aux_pressure(ga, error) == DS_NOTOK)
                return(DS_NOTOK);
            break;

        case AUX_GAUGE_GET_MAX_PRESSURE:
            *argout = (DevFloat) ((RgaClass)ga->devserver.class_pointer)->
                                          rga_class.aux_max_pressure;
            break;

        default:
            *error = DevErr_ValueOutOfBounds;
            return(DS_NOTOK);
    }
    return(DS_OK);
}

/****************
 * get_state()
 **************************************************************/
/*
 *  Purpose:  Updates the device state variable with the
 *            current filament status of the instrument.
 *
 *  Remarks: - If an error occurs while reading the status,
 *             the device state is turned to DEVUNKNOWN.
 *
 *  Returns:  DS_OK      if the status is successfully read
 *            DS_NOTOK   if the device state is DEVUNKNOWN
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *               DevErr_FilamentStateError
 *
 **************************************************************/
static long get_state(Rga ga,long *error)

/*Rga          ga;         In:   Pointer to device server structure  *
                         * Out:  Device state in ga->devserver.state *
                         *       States:    DEVUNKNOWN               *
                         *                  DEVOFF                   *
                         *                  DEVON                    *
                         *                  DEVFAULT                 *
                         *                  DEVTRIPPED               */

{
    char   *status=NULL;
    char   filament_state;

    if(send_request(ga, FILAMENT_STATUS_REQUEST, &status, error) == DS_NOTOK )
    {
        ga->devserver.state = DEVUNKNOWN;
        return(DS_NOTOK);
    }
    filament_state=status[1];
    
    status++;
    switch(status[2]){

        case '0':                             /* Filament Off */
            ga->devserver.state = DEVOFF;
            break;

        case '1':                             /* Filament On */
    	    /* Store currently used filament */
    	    ga->rga.FilamentUsed = status[0] - 0x30;
            ga->devserver.state = DEVON;
            break;

        case '2':                             /* Filament Failure */
            ga->devserver.state = DEVFAULT;
            break;

        case '3':case '4':case '5':           /* Filament Trip */
            ga->devserver.state = DEVTRIPPED;
            break;

        default:
            ga->devserver.state = DEVUNKNOWN;
            *error = DevErr_FilamentStateError;
    }
    return(DS_OK);
}

/*******************
 *  set_up_comm()
 **************************************************************/
/*
 *  Purpose:  Initialises the communication with the instrument,
 *            setting up the acknowledgement protocol and the
 *            terminator and delimiter characters.
 *
 *  Remarks: - If the communication is succesfully established,
 *             the device server variables that contain the
 *             mode of operation are initialised with current
 *             values taken from the instrument.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long set_up_comm(Rga ga,long *error)
{
    long ret;
    char *answer;

    /*
     *  The following command and the sleep() are necessary because
     *  of an anomalous behaviour of the SATELLITE after be reset.
     */
    if (send_request(ga, ACK_COMMAND,&answer, error) == DS_NOTOK)
    {
       	return (DS_NOTOK);
    }
    /*
     *  The actual initialisation starts now
     */

    if (send_request(ga, ANSWER_TERMINATOR_COMMAND,&answer, error) == DS_NOTOK)
             return (DS_NOTOK);

    if (send_request(ga, ANSWER_DELIMITER_COMMAND,&answer, error) == DS_NOTOK)
             return (DS_NOTOK);

    /*
     *  Update mode of operation
    if (mode_update(ga, error) == DS_NOTOK)
             return (DS_NOTOK);

     */

    return (DS_OK);
}

/********************
 *  send_request()
 **************************************************************/
/*
 *  Purpose:  Sends a request to the SATELLITE and receives
 *            the answer from it.
 *
 *           - This function is analogous to devrequest, but
 *             it is called by other private functions.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_DeviceSyntaxError
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long send_request(Rga ga,char *request,char **answer,long *error)
{
    extern char  answer_buff[];  
    /*
     * Send request,
     */   
    if (send_string(ga, request, error) == DS_NOTOK )
    {
               return(DS_NOTOK);
    }
    
    if (get_answer(ga, error) == DS_NOTOK )
    {
      	return(DS_NOTOK);
    }
    *answer = answer_buff;
   return(DS_OK);
}

/********************
 *  long_request()
 **************************************************************/
/*
 *  Purpose:  request to the SATELLITE and receives
 *            the answer from it. this function work around a bug in the os9 readln function
 *            which make impossible to read in one goal a buffer >1024char.
 *            It does several read in raw mode until end of line is returned.
 *
 *           - This function is analogous to devrequest, but
 *             it is called by other private functions.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_DeviceSyntaxError
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long long_request(Rga ga,char *request,char **answer,long *error)
{
    extern char  answer_buff[];
  short	mode[]={SL_NEWLINE,0};
  DevVarShortArray parameter; 
  long devstatus; 
  int lenght;
  long  sl_mode;
  static char *str;
  
  static char charbuff[3000];
  charbuff[0]='\0';
    
    str=NULL;
    /*
     * Send request,
     */   
    if (send_string(ga, request, error) == DS_NOTOK )
    {
               return(DS_NOTOK);
    }
    
    /* read the first character in SL_NCHAR mode in order to have a timeout output if the
    device is not connected */
    sl_mode = SL_NCHAR + (1 << 8);
    str=NULL;
    if(dev_putget( ga->rga.serial_line_handle , DevSerReadString,
 	      		&sl_mode, D_LONG_TYPE,
 	      		&str, D_STRING_TYPE,
 	      		error) == DS_NOTOK )
		return (DS_NOTOK);
				
	charbuff[0]=*str;
	charbuff[1]=0;
	printf("--> %s \n",charbuff);
	fflush(stdout);
    sl_mode=SL_RAW;
    lenght=1;
    while( lenght > 0)
    {
#ifndef __cplusplus
	tsleep(50);
#else
	sleep(1);
#endif /* __cplusplus */
	dev_xdrfree(D_STRING_TYPE, &str, error);
	str=NULL;
	if(dev_putget( ga->rga.serial_line_handle , DevSerReadString,
 		      		&sl_mode, D_LONG_TYPE,
 		      		&str, D_STRING_TYPE,
 		      		error) == DS_NOTOK )
		return (DS_NOTOK);
				
	strcat(charbuff,str);
	lenght=strlen(str);
	printf(" %d ",lenght);
	fflush(stdout);
	if(str[lenght-1] == '\x0d')
	{
		lenght=0;
		printf("end\n");
		fflush(stdout);
		dev_xdrfree(D_STRING_TYPE, &str, error);
	} 
   }	
   *answer = charbuff;
   return(DS_OK);
}

/*******************
 *  send_string()
 **************************************************************/
/*
 *  Purpose:  Sends a string to the instrument.
 *
 *  Remarks: - Before the write operation, the input buffer is
 *             cleared and the string is bracketed in order to
 *             follow the syntax conventions of the SATELLITE.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long send_string(Rga ga,char *string,long *error)
{
    long   nbytes;
    char *str;
    static char in_buff[128];
    long  sl_mode=SL_RAW;
    sprintf(in_buff,"(%s)",string);
    string=in_buff;
    printf("Rga send \"%s\"\t",in_buff);
    fflush(stdout);

    str=answer_buff;
   if(dev_putget( ga->rga.serial_line_handle , DevSerReadString,
 		      &sl_mode, D_LONG_TYPE,
 		      &str, D_STRING_TYPE,
 		      error) == DS_NOTOK)
		      return (DS_NOTOK);
    if(strlen(str)!=0) printf("unexpected char %s",str);
    
    if (dev_putget(ga->rga.serial_line_handle, DevSerWriteString, &string, D_STRING_TYPE, 
		&nbytes, D_LONG_TYPE, error ) == DS_NOTOK)
	return (DS_NOTOK);	  	  	

    return(DS_OK);
}

/******************
 *  get_answer()
 **************************************************************/
/*
 *  Purpose:  Receives a answer string from the instrument.
 *
 *  Remarks: - The end of the answer string is determined by
 *             a LINE_TERMINATOR character, but never more 
 *             than MAX_ANSWER_LENGTH characters are read.
 *           - The answer is converted into a NULL terminated
 *             string by eliminating all the non printable
 *             characters at the end and appending a '\0'.
 *           - The resulting string is stored in the static
 *             buffer answer_buff.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long get_answer(Rga ga,long *error)
{
    extern  char  answer_buff[];
    char *str;
    char c;
    long  sl_mode=SL_LINE;
    
    str=answer_buff;
   if(dev_putget( ga->rga.serial_line_handle , DevSerReadString,
 		      &sl_mode, D_LONG_TYPE,
 		      &str, D_STRING_TYPE,
 		      error) == DS_NOTOK)
		      return (DS_NOTOK);
    c=answer_buff[0];
    
    if (c != ACK_ANSWER)
    {
        if (c == NACK_ANSWER)
            *error = DevErr_NotAcknowledged;
        else
	{
            *error = DevErr_AcknowledgementFailed;
	    printf(" Bad acknowledgment character: %c 0x%x\n",c,c);
#ifndef __cplusplus
	    tsleep(100);
#else
            sleep(1);
#endif /* __cplusplus */
	    set_up_comm(ga, error);

        }
        return(DS_NOTOK);
    }
    printf("\t--> %s",str);
    fflush(stdout);
    return(DS_OK);
}

/*******************
 *  mode_update()
 **************************************************************/
/*
 *  Purpose:  Updates the variables that contain the mode of
 *            operation of the instrument and, if the scan mode
 *            is Peak Jump or Fast Scan, the channel parameters too.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long mode_update(Rga ga,long *error)
{
    char *answer=NULL;
    *error = FALSE;
    if(send_request(ga, SCAN_MODE_REQUEST, &answer, error) == DS_NOTOK)
        return(DS_NOTOK);
    answer++;
    ga->rga.OpMode.ScanMode = (short) atoi(answer);    
    return(DS_OK);
}

/*******************
 *  get_op_mode()
 **************************************************************/
/*
 *  Purpose:  Updates the variables in the structure rga.OpMode
 *            with the current mode of operation.
 *
 *  Remarks: - The mode of operation comprises the scan mode,
 *             the multiplier gain, the status of the total
 *             and partial pressure trips, and the parameters
 *             associated to the particular scan mode.
 *           - All this information is obtained with the
 *             requests (m), (g), (t) and (p).
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long get_op_mode(Rga ga,long *error)
{
    extern DevRgaOpMode operation_mode;
    char  param_rq[4];
    char *answer=NULL;
    int   npar;

    *error = FALSE;
    /*
     *  Send the request (m) and store the answer
     */
    if(send_request(ga, SCAN_MODE_REQUEST, &answer, error) == DS_NOTOK)
        return(DS_NOTOK);
    answer++;
    ga->rga.OpMode.ScanMode = (short) atoi(answer);    

    if(ga->rga.OpMode.ScanMode != CONTROL_MODE)
    { 	
    
    /*
     *  Send the request (g) and store the answer
     */
    if(send_request(ga, MULTIPLIER_GAIN_REQUEST, &answer, error)
                                                          == DS_NOTOK )
        return(DS_NOTOK);
    
    answer++;
    ga->rga.OpMode.MultiplierGain = (short) atoi(answer);

    /*
     *  Request mode parameters
     */
    for (npar = 0 ; npar < Number_Param[ga->rga.OpMode.ScanMode] ; npar++ )
    {

        if (!(ga->rga.OpMode.ScanMode == 1 && npar + 1 == 2 ||
              ga->rga.OpMode.ScanMode == 3 && npar + 1 == 1 ))
	{
     
            /*
             *  Make up the request (pnpar)
             */
            sprintf(param_rq, MODE_PARAMETER_REQUEST_PATTERN, npar +1 );

            /*
             *  Send the request (pn) and store the answer
             */
            if(send_request(ga, param_rq, &answer, error) == DS_NOTOK)
                return(DS_NOTOK);
	    answer++;
            ga->rga.OpMode.Parameter.number[npar] = (short) atoi(answer);
       }
    }
   /*
    * Work around a bug in the LEDAMASS software for the first mass specification
    * If we ask for n, actually n-1 is rreturned by the satelite.
    * So, we add 1 before sending the result.
    * This is true for analog mode but not for Bar chart mode.
    ******************************************************************************/
        if(ga->rga.OpMode.ScanMode == ANALOG_MODE)
	   ga->rga.OpMode.Parameter.number[3]++;    
    
    }
    return(DS_OK);
}

/***********************
 *  get_chann_param()
 **************************************************************/
/*
 *  Purpose:  Updates the structure rga.ChParam with the
 *            current parameters of all the channels.
 *
 *  Remarks: - The channel parameters are the values returned by
 *             the request (s).
 *           - This function should be called only if the
 *             current scan mode is Peak Jump or Fast Scan.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long get_chann_param(Rga ga,long *error)
{
    char   generic_rq[6];
    char  *answer=NULL;
    short  npar;
    short  nchan;

    *error = FALSE;
    for (nchan = 0 ; nchan < RGA_NUM_OF_CHANNELS ; nchan++ )
    {
        for (npar = 0 ; npar < RGA_NUM_OF_CHANNEL_PARAMETERS ; npar++ )
	{

            /*
             *  Make up the request (cnpar,nchan)
             */
            sprintf(generic_rq, CHANNEL_PARAMETER_REQUEST_PATTERN,
                           npar + 1, nchan + 1);

            /*
             *  Send the request (cnpar,nchan) and store the answer
             */
            if(send_request(ga, generic_rq, &answer, error) == DS_NOTOK)
                return(DS_NOTOK);
		
	    answer++;
            ga->rga.ChParam.Channel[nchan].Parameter.number[npar] =
                            (short) atoi(answer);
        }
    }
    return(DS_OK);
}

/***********************
 *  get_single_channel()
 **************************************************************/
/*
 *  Purpose:  Updates the structure rga.ChParam with the
 *            current parameters of one channels.
 *
 *  Remarks: - The channel parameters are the values returned by
 *             the request (s).
 *           - This function should be called only if the
 *             current scan mode is Peak Jump or Fast Scan.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_DeviceWrite
 *               DevErr_DeviceRead
 *               DevErr_DeviceTimedOut
 *               DevErr_NotAcknowledged
 *               DevErr_AcknowledgementFailed
 *               DevErr_StringTooLong
 *
 **************************************************************/
static long get_single_channel(Rga ga ,short param ,short channel ,short value ,long *error )
{
    *error = FALSE;
            /*
             *  Make up the request (cnpar,nchan) MODIF
             */
           /* sprintf(generic_rq, CHANNEL_PARAMETER_REQUEST_PATTERN,
                           param, channel);*/
		
            /*
             *  Send the request (cnpar,nchan) and store the answer
             */
           /* if(send_request(ga, generic_rq, &answer, error) == DS_NOTOK)
	    {
                return(DS_NOTOK);
            }
	    
    
	    answer++;*/
	    ga->rga.ChParam.Channel[channel-1].Parameter.number[param-1] = value;
                            /*(short) atoi(answer); value;*/
    return(DS_OK);
}

/************************
 *  get_aux_defaults()
 **************************************************************/
/*
 *  Purpose:  Replaces the device name of the auxiliary gauge
 *            and the pressure checking feature flag with the
 *            values originally loaded from the database.
 *  
 *  Returns:  DS_OK
 *
 **************************************************************/
static long get_aux_defaults(Rga ga)
{
    strcpy(ga->rga.actual_aux_vacuum_gauge,
                            (char *)ga->rga.default_aux_vacuum_gauge);
    ga->rga.actual_aux_gauge_check = (char)ga->rga.default_aux_gauge_check;

#ifdef DEBUG_LVL3
    printf("\t\t\tAux. Vacuum Gauge: %s\n", ga->rga.actual_aux_vacuum_gauge);
    printf("\t\t\tAux. Gauge Check : %d\n", ga->rga.actual_aux_gauge_check);
#endif

    return(DS_OK);
}

/**************************
 *  check_aux_pressure()
 **************************************************************/
/*
 *  Purpose:  Checks if the pressure from the auxiliary gauge 
 *            is below or above the maximum value defined in
 *            the database.
 *            
 *  Remarks: - If the pressure is higher than the maximum value
 *             the error DevErr_AuxPressureTooHigh is returned.
 *
 *  Returns:  DS_OK      If no error and the pressure is OK
 *            DS_NOTOK   If error or the pressure is too high
 *
 *  Error Codes:
 *               DevErr_AuxiliaryDeviceNotValid
 *               DevErr_AuxiliaryGaugeOff
 *               DevErr_AuxPressureTooHigh
 *
 *             + errors returned by: dev_import()
 *                                   dev_putget()
 *
 *             + errors returned by the commands DevState and
 *                 DevGaugeReadOut implemented in the device
 *                 server that controls the auxiliary gauge.
 *
 **************************************************************/
static long check_aux_pressure(Rga ga,long *error)
{
    if (read_aux_gauge(ga, error) == DS_NOTOK)
        return(DS_NOTOK);

    if (ga->rga.aux_pressure > ((RgaClass)ga->devserver.class_pointer)->
                                               rga_class.aux_max_pressure)
    {
        *error = DevErr_AuxPressureTooHigh;
        return(DS_NOTOK);
    }
    
    return(DS_OK);
}

/**********************
 *  read_aux_gauge()
 **************************************************************/
/*
 *  Purpose:  Returns the pressure reading from the auxiliary
 *            vacuum gauge.
 *
 *  Remarks: - If necessary, the device is imported.
 *           - The device state is checked before trying to
 *             read the pressure.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_AuxiliaryDeviceNotValid
 *               DevErr_AuxiliaryGaugeOff
 *
 *             + errors returned by: dev_import()
 *                                   dev_putget()
 *
 *             + errors returned by the commands DevState and
 *                 DevGaugeReadOut implemented in the device
 *                 server that controls the auxiliary gauge.
 *
 **************************************************************/
static long read_aux_gauge(Rga ga,long *error)
{
    DevStateFloatReadPoint vgc_gauge;
    DevLong      state;

    if (ga->rga.aux_dserver_flag == 0)
        if (import_aux_gauge(ga, error) == DS_NOTOK)
            return(DS_NOTOK);

    if (dev_putget(ga->rga.aux_PG_dserver, DevUpdate,
                  NULL, D_VOID_TYPE, &vgc_gauge, D_STATE_FLOAT_READPOINT, error) == DS_NOTOK)
    {   
        printf("cannot read auxiliary gauge %s, %d\n",ga->rga.actual_aux_vacuum_gauge);
        return(DS_NOTOK);
    }
    if (vgc_gauge.state != DEVON)
    {
         *error = DevErr_AuxiliaryGaugeOff;
         return(DS_NOTOK);
    }
    ga->rga.aux_pressure = vgc_gauge.read;

    printf("\t\t\tPressure on %s = %.1E\n",ga->rga.actual_aux_vacuum_gauge, ga->rga.aux_pressure);
    
    return(DS_OK);
}

/************************
 *  import_aux_gauge()
 **************************************************************/
/*
 *  Purpose:  Imports the device that controls the auxiliary
 *            gauge.
 *
 *  Remarks: - If the device is succesfully imported the flag
 *             ga->rga.aux_dserver_flag is set to 1, and the 
 *             pointer ga->rga.aux_PG_dserver is initialised.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               DevErr_AuxiliaryDeviceNotValid
 *
 *             + errors returned by dev_import()
 *
 **************************************************************/
static long import_aux_gauge(Rga ga,long *error)
{

    if ( dev_import(ga->rga.actual_aux_vacuum_gauge, 0,
                               &(ga->rga.aux_PG_dserver), error) == DS_NOTOK )
    {
       printf("can't import auxiliary gauge %s, error %d\n",ga->rga.actual_aux_vacuum_gauge,*error);
        return(DS_NOTOK);
    }
    ga->rga.aux_dserver_flag = 1;
    return(DS_OK);
}

/**********************
 *  free_aux_gauge()
 **************************************************************/
/*
 *  Purpose:  Closes the connection to the device
 *            
 *  Remarks: - Always ga->rga.aux_dserver_flag is set to 0.
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               errors returned by dev_free()
 *
 **************************************************************/
static long free_aux_gauge(Rga ga,long *error)
{
    ga->rga.aux_dserver_flag = 0;

    if ( dev_free(ga->rga.aux_PG_dserver, error) == DS_NOTOK )
        return(DS_NOTOK);

    return(DS_OK);
}

/**********************
 *  DevSetVoltage()
 **************************************************************/
/*
 *  Purpose:  configure the device as requested in the initial resources
 *            
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               errors returned by dev_free()
 *
 **************************************************************/
static long devsetvoltage(Rga ga,short *argin, void *argout,long *error)
{
   
    char request[5], *answer;
    double voltage;
    *error = FALSE;

     voltage = (abs((int)(*argin)) - ga->rga.MultiplierBase)/2.87;
     sprintf(request,"V%.0f", voltage);
     
     if (send_request(ga, request,&answer, error) == DS_NOTOK )
               return(DS_NOTOK);

     return(DS_OK);
}
/**********************
 *  default_config()
 **************************************************************/
/*
 *  Purpose:  configure the device as requested in the initial resources
 *            
 *
 *  Returns:  DS_OK      If no error
 *            DS_NOTOK   If error
 *
 *  Error Codes:
 *               errors returned by dev_free()
 *
 **************************************************************/
static long default_config(Rga ga,void *argin, void *argout,long *error)

{
    short channel;
    DevVarShortArray ChanParam;
    short tabparam[3];
    char *answer;
    double voltage;
    *error = FALSE;
    ChanParam.sequence=tabparam;
    ChanParam.length=3;
     /*
      * Set the multiplier gain
      */
     if (devsetmultipliergain(ga,&ga->rga.MultiplierGain , NULL, error) == DS_NOTOK)
                return (DS_NOTOK);
     
     /*
      * Set the multiplier voltage 
      */
     
     if(devsetvoltage(ga,&ga->rga.MultiplierVolt, NULL, error) == DS_NOTOK)
     		return (DS_NOTOK);
     
     
      if (send_request(ga,"@5651,,1,0", &answer, error) == DS_NOTOK )
               return(DS_NOTOK);

     /*
      * If requested in resources, set the correct scanning mode
      */
         if(ga->rga.AutoScanMode != NO_SCAN_MODE)
         if (devsetmode(ga,&ga->rga.AutoScanMode , NULL, error) == DS_NOTOK)
                return (DS_NOTOK);

     /*
      * If requested in resources, set the channel parameters
      */
      if(ga->rga.AutoScanMode == PEAK_JUMP_MODE)
      {
         for(channel=0;channel<ga->rga.ChanAutoRange.length;channel++)
         {
            ChanParam.sequence[1] = channel + 1;
            ChanParam.sequence[0] = CHAN_AUTORANGE;
            ChanParam.sequence[2] = ga->rga.ChanAutoRange.sequence[channel];
            if(devsetchannel(ga,&ChanParam,NULL,error) == DS_NOTOK)
                return (DS_NOTOK);
         }
         for(channel=0;channel < ga->rga.ChanMasses.length;channel++)
         {
            if(ga->rga.ChanMasses.sequence[channel]!=0)
            { 

               ChanParam.sequence[1]=channel+1;
               ChanParam.sequence[0]=CHAN_MASS;
               ChanParam.sequence[2]=ga->rga.ChanMasses.sequence[channel];
               if(devsetchannel(ga,&ChanParam,NULL,error) == DS_NOTOK)
                   return (DS_NOTOK);
               ChanParam.sequence[1]=channel+1;
               ChanParam.sequence[0]=CHAN_ENABLED;
               ChanParam.sequence[2]=1;
               if(devsetchannel(ga,&ChanParam,NULL,error) == DS_NOTOK)
                   return (DS_NOTOK);
            }
         }
         for(channel=0;channel<ga->rga.ChanMultiplier.length;channel++)
         {
            ChanParam.sequence[1]=channel+1;
            ChanParam.sequence[0]=CHAN_MULTIPLIER;
            ChanParam.sequence[2]=ga->rga.ChanMultiplier.sequence[channel];
            if(devsetchannel(ga,&ChanParam,NULL,error) == DS_NOTOK)
                return (DS_NOTOK);
         }
         ga->rga.new_scan = True;
      }
     /*
      * If requested in resources, switch ON the Filament MODIF
      */
      if(ga->rga.FilamentUsed != 0)
         if (devfilamenton(ga,&ga->rga.FilamentUsed , NULL, error) == DS_NOTOK)
                return (DS_NOTOK);
}

/*======================================================================
 Function:      static long dev_signal_config()

 Description:	Read the properties of all signals specified 
		

 Arg(s) In:	Rga ds : pointer to object
		void  *argin : no input arguments

 Arg(s) Out:	DevVarStringArray  *argout : Array of signal properties.
		long *error : pointer to error code, in case routine fails
 =======================================================================*/
static long dev_signal_config (Rga ga,DevVoid *argin,DevVarStringArray *argout,long *error)
{
	*error = 0;
     if (ds__method_finder(ga->rga.msignal_obj,DevMethodReadProperties)
			  (ga->rga.msignal_obj, argout, error) == DS_NOTOK)
     {
       return(DS_NOTOK);
     }
  return (DS_OK);
}

/*======================================================================
 Function:      static long dev_update_config()

 Description:	Reinitialises signal resources.

 Arg(s) In:	Rga ga : pointer to object
		void  *argin : no input arguments

 Arg(s) Out:	void  *argout : no outgoing arguments
		long *error   : pointer to error code, in case routine fails
 =======================================================================*/
static long dev_update_config(Rga ga,DevVoid *argin,DevVoid *argout,long *error)
{
   *error = 0;
  /*
   * 	Reset all signals in order to reinitialize their resources.
   */

  	if (ds__method_finder(ga->rga.msignal_obj,DevMethodSignalsReset)
				(ga->rga.msignal_obj,error) == DS_NOTOK)
   	{
    	 return(DS_NOTOK);
  	}
   return(DS_OK);
}
