/*static char RcsId[] = "$Header: /segfs/dserver/classes++/motor/oregon/src/RCS/OregonMaxe.cpp,v 1.6 2000/03/16 23:04:25 goetz Exp goetz $";*/
//*********************************************************************
//
//File:		OregonMaxe.cpp
//
//Project:	Fast stepper motor scanning in VME/PC-104
//
//Description:	Implements the Maxe class compatible list of commands
//		for the Oregon stepper motor card controllers VME-58
//		and PC-68 (PC-104) in C++. Each card is an object i.e.
//		4 or 8 motors. The class is intended to run under Linux
//		and talk to the hardware via the oms device driver
//		written by Richard Hirst (richard@sleepie.demon.co.uk).
//
//		NOTE : this class has been cloned from the OmsMaxe class
//		in C by Lucille Rosier (Lure) which was written for a
//		previous version of the card (VME-8). The main differences
//		are that the class has been converted to C++, the library
//		calling the OMS card has been encapsulated in an object
//		(of class Oms) and (most) of the French comments have been
//		translated to English.
//
//		It has been written so that it can serve as a template for
//		future Maxe-compliant motor device servers. It can either
//		be converted to another concrete MyMotorMaxe class or
//		to an abstract Maxepp class which serves as super-class
//		for MyMotorMaxe. In the latter case this class (OregonMaxe)
//		should be derived from Maxepp and suitably adapted.
//
//Author(s):	Andy Gotz (goetz@esrf.fr),  
//		Lucile Roussier - original C version (roussier@lure.fr)
//
//Original:	March 1999
//
//$Log: OregonMaxe.cpp,v $
//Revision 1.6  2000/03/16 23:04:25  goetz
//multiple commands now treat only nb_axes of channels[]
//
//Revision 1.5  1999/12/15 17:14:16  goetz
//latest version from ID11; improved debugging messages
//
//Revision 1.4  1999/09/03 13:30:39  goetz
//latest version from site - modifications tested on ID10B
//
//Revision 1.3  1999/05/19 18:47:47  goetz
//fixed bug passing state and position parameters to dev_event_fire()
//
//Revision 1.2  1999/05/19 15:12:29  goetz
//added event thread to detect and fire state or pos change events
//
//Revision 1.1  1999/04/13 19:27:33  goetz
//Initial revision
//
//
//Copyright(c) 1999 by European Synchrotron Radiation Facility, 
//                     Grenoble, France
//*********************************************************************
 
#ifdef EVENTS
#include <pthread.h>
#endif /* EVENTS */

#include <API.h>
#include <ApiP.h>
#include <DevServer.h>
#include <DevErrors.h>
#include <Admin.h>
#include <BlcDsNumbers.h>

#include <Device.H>
#include <Oms.h>
#include <OregonMaxe.h>

#include <Maxe.h>
#include <maxe_xdr.h>

short OregonMaxe::class_inited = 0;
short OregonMaxe::class_destroyed = 0;

#ifdef EVENTS
/*
 * mutexes for controlling access to state and position
 */
static pthread_mutex_t oms_state_mutex=PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t oms_position_mutex=PTHREAD_MUTEX_INITIALIZER;
#endif

/*
 * Omsmaxe resource tables used to access the static database
 *
 */
	/*  Object resources i.e. for one Oregon card */
db_resource res_object[] = {
   {"nb_axes",		D_LONG_TYPE, NULL},
   {"descriptor",	D_STRING_TYPE, NULL},
   {"events",           D_LONG_TYPE,  NULL},
   	};
static int res_object_size = sizeof(res_object)/sizeof(db_resource);

	/*  Axes name resources  */
static db_resource res_axe_name[] = {
    {NULL, D_STRING_TYPE, NULL},
};
static int  res_axe_name_size = sizeof(res_axe_name) / sizeof(db_resource);
	
	/*  Motor and encoder resources for one axe  */
static db_resource res_axe[] = {
    {"channel",       D_LONG_TYPE,  NULL},
    {"velocity",      D_LONG_TYPE,   NULL},
    {"firststeprate", D_LONG_TYPE,   NULL},
    {"acceleration",  D_LONG_TYPE,   NULL},
    {"accelslope",    D_LONG_TYPE,  NULL},
    {"backlash",      D_LONG_TYPE,   NULL},
    {"powerauto",     D_LONG_TYPE,  NULL},
/*
 * no support for encoders at present
 *
 *    {"enc_type",      D_STRING_TYPE, NULL},
 *    {"enc_ident",     D_STRING_TYPE, NULL},
 *    {"enc_steps_mm",  D_FLOAT_TYPE,  NULL},
 *    {"enc_factor",  	D_FLOAT_TYPE,  NULL},
 */
};		
static int  res_axe_size = sizeof(res_axe) / sizeof(db_resource);

/*======================================================================
 Function:      void *oms_state_pos_change_thread ()

 Description:   routine which will run as separate thread to update
		the states and positions of all axes motors of an oms object 
		every 100 ms and compare it with their previous state. If any 
		of the states have changed OR position change and not moving
		then fire a D_EVENT_OMS_STATE_CHANGE event sending all states 
		and positions to the client. One thread will be started for 
		each OregonMaxe object. 

		NOTE - if the motor is moving for less than 100 ms then 
		the state change will not be detected except if the position
		changes. This seems reasonable. If however clients insist 
		on being informed of every state change then the OregonMaxe
		class will have to be modified to queue every state change
		so that the event thread can fire events.

 Arg(s) In:     void *arg - pointer to OregonMaxe object to monitor

 Arg(s) Out:    long *error- pointer to error code, in case routine fails

 Error Codes:   none

 Returns:       void
=======================================================================*/

void * oms_state_pos_change_event_thread(void * arg)
{
  OregonMaxe *omaxe;
  long event_type = D_EVENT_OMS_STATE_POS_CHANGE;
  long state_pos_change, counter=0, i;
  struct timespec t100ms;
  long old_state[8], old_position[8], new_position[8], state_pos[16];
  long error;
  DevVarLongArray vlongarr;

  omaxe = (OregonMaxe*)arg;
  fprintf(stderr, "\noms_state_pos_change_event_thread(): starting thread for oms object %s\n", (char *) omaxe->name);

#ifdef EVENTS
  pthread_mutex_lock(&oms_state_mutex);
#endif
  for(i=0; i<omaxe->nb_axes; i++)
  {
     old_state[i] = omaxe->axes[i].state;
  }
#ifdef EVENTS
  pthread_mutex_unlock(&oms_state_mutex);
#endif

  for (;;)
  {

// update all motors states

#ifdef EVENTS
     pthread_mutex_lock(&oms_state_mutex);
#endif
     for (i=0; i<omaxe->nb_axes; i++)
     {
        omaxe->oms->ReadStatus(omaxe->axes[i].channel,
                               &(omaxe->axes[i].state),&error);
     }
#ifdef EVENTS
     pthread_mutex_unlock(&oms_state_mutex);
#endif

// update all motor positions

#ifdef EVENTS
     pthread_mutex_lock(&oms_position_mutex);
#endif
     omaxe->oms->ReadMultiplePositions(omaxe->nb_axes,omaxe->channels,new_position,&error);
#ifdef EVENTS
     pthread_mutex_unlock(&oms_position_mutex);
#endif

// check for state or position change and not moving

     state_pos_change = FALSE;
     for (i=0; i<omaxe->nb_axes; i++)
     {
        if ((omaxe->axes[i].state != old_state[i]) ||
            ((old_position[i] != new_position[i]) &&
             (omaxe->axes[i].state != DEVMOVING)))
        {
           state_pos_change = TRUE;
           old_state[i] = omaxe->axes[i].state;
           old_position[i] = new_position[i];
        }
     }

// state or position changed detected 

     if (state_pos_change == TRUE)
     {
        printf("\noms_state_pos_change_event_thread(): state change detected, firing event (counter=%d)\n",
                  counter);

// send new states and positions with event

        for (i=0; i<omaxe->nb_axes; i++)
        {
           state_pos[i*2] = old_state[i];
           state_pos[i*2+1] = new_position[i];
           printf("oms_state_pos_change_event_thread(): new state[%d] = %d new position[%d] = %d\n",
               i,state_pos[i*2],i,state_pos[i*2+1]);
        }
	vlongarr.length = 16;
	vlongarr.sequence = state_pos;
        dev_event_fire(event_type,&vlongarr,D_VAR_LONGARR,DS_OK,0);
        counter++;
     }

     t100ms.tv_sec = 0;
     t100ms.tv_nsec =  90000000;
     nanosleep(&t100ms, NULL);
  }
  return NULL;
}

/*======================================================================
 Function:      static long class_initialise ()

 Description:	routine to be called the first time a device is 
 		created which belongs to this class (or is a subclass
		thereof. This routine will be called only once.

 Arg(s) In:	none

 Arg(s) Out:	long *error- pointer to error code, in case routine fails
 
 Error Codes:	DevErr_InsufficientMemory
	      + errors returned by db_getresource()
	      
 Returns:  	DS_OK      If no error
           	DS_NOTOK   If error
=======================================================================*/

long OregonMaxe::ClassInitialise(long *error)
{

  *error= 0;

   OregonMaxe::class_inited = 1;

   return(DS_OK);
}

/*======================================================================
 Function:	long OregonMaxe::OregonMaxe ()

 Description:	OregonMaxe constructor which will create an object of
		the class OregonMaxe
             	
 Arg(s) In:	char *name - name to be given to device

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
             	
 Error Codes:	

 Returns:  	DS_OK      If no error
            	DS_NOTOK   If error
 =======================================================================*/

OregonMaxe::OregonMaxe (char *name,long *error) 
			: Device (name, error)
{
   static Device::DeviceCommandListEntry om_commands_list[] = {
   	{DevState, &Device::State, D_VOID_TYPE,	D_SHORT_TYPE},
   	{DevStatus, &Device::Status, D_VOID_TYPE, D_STRING_TYPE},
   	
   	{DevReadState,(DeviceMemberFunction)(&OregonMaxe::ReadState), D_LONG_TYPE, D_VAR_LONGARR},
   	{DevReadSwitches, (DeviceMemberFunction)(&OregonMaxe::ReadSwitches), D_LONG_TYPE, D_LONG_TYPE},
   	
   	{DevReadPosition, (DeviceMemberFunction)(&OregonMaxe::ReadPosition), D_LONG_TYPE, D_FLOAT_TYPE},
   	{DevReadMulPos, (DeviceMemberFunction)(&OregonMaxe::ReadMultiplePositions), D_VAR_LONGARR, D_VAR_FLOATARR},
   	{DevLoadPosition, (DeviceMemberFunction)(&OregonMaxe::LoadPosition), D_MOTOR_FLOAT, D_VOID_TYPE},
   	
   	{DevReset, (DeviceMemberFunction)(&OregonMaxe::Reset), D_VOID_TYPE, D_VOID_TYPE},
   	{DevAbortCommand, (DeviceMemberFunction)(&OregonMaxe::AbortCommand), D_LONG_TYPE, D_VOID_TYPE},
   	
   	{DevMoveRelative, (DeviceMemberFunction)(&OregonMaxe::MoveRelative), D_MOTOR_FLOAT, D_VOID_TYPE},
   	{DevMoveAbsolute, (DeviceMemberFunction)(&OregonMaxe::MoveAbsolute), D_MOTOR_FLOAT, D_VOID_TYPE},
   	{DevMoveMultiple, (DeviceMemberFunction)(&OregonMaxe::MoveMultiple), D_MULMOVE_TYPE, D_VOID_TYPE},
   	{DevMoveReference, (DeviceMemberFunction)(&OregonMaxe::MoveReference), D_MOTOR_LONG, D_VOID_TYPE},
   	{DevSetContinuous, (DeviceMemberFunction)(&OregonMaxe::SetContinuous), D_MOTOR_LONG, D_VOID_TYPE},
   	
   	{DevSetVelocity, (DeviceMemberFunction)(&OregonMaxe::SetVelocity), D_MOTOR_FLOAT, D_VOID_TYPE},
   	{DevSetAcceleration, (DeviceMemberFunction)(&OregonMaxe::SetAcceleration), D_MOTOR_FLOAT, D_VOID_TYPE},
   	{DevSetFirstStepRate, (DeviceMemberFunction)(&OregonMaxe::SetFirstStepRate), D_MOTOR_FLOAT, D_VOID_TYPE},
   	{DevSetBacklash, (DeviceMemberFunction)(&OregonMaxe::SetBacklash), D_MOTOR_FLOAT, D_VOID_TYPE},
   	
   	{DevReadVelocity, (DeviceMemberFunction)(&OregonMaxe::ReadVelocity), D_LONG_TYPE, D_FLOAT_TYPE},
   	{DevReadAcceleration, (DeviceMemberFunction)(&OregonMaxe::ReadAcceleration), D_LONG_TYPE, D_FLOAT_TYPE},
   	{DevReadFStepRate, (DeviceMemberFunction)(&OregonMaxe::ReadFirstStepRate), D_LONG_TYPE, D_FLOAT_TYPE},
   	{DevReadBacklash, (DeviceMemberFunction)(&OregonMaxe::ReadBacklash), D_LONG_TYPE, D_FLOAT_TYPE},
   	
   	{DevReadEncPos, (DeviceMemberFunction)(&OregonMaxe::ReadEncoderPosition), D_LONG_TYPE, D_FLOAT_TYPE},
   	{DevLoadEncPos, (DeviceMemberFunction)(&OregonMaxe::LoadEncoderPosition), D_MOTOR_FLOAT, D_VOID_TYPE},
   	{DevReadMulEncPos, (DeviceMemberFunction)(&OregonMaxe::ReadMultipleEncoderPositions), 	D_VAR_LONGARR, 	D_VAR_FLOATARR},
   	{DevReadUnits, (DeviceMemberFunction)(&OregonMaxe::ReadUnits), D_LONG_TYPE, D_LONG_TYPE},
   	{DevSetStepMode, (DeviceMemberFunction)(&OregonMaxe::SetStepMode), D_MOTOR_LONG, D_VOID_TYPE},
   	{DevReadStepMode, (DeviceMemberFunction)(&OregonMaxe::ReadStepMode), D_LONG_TYPE, D_LONG_TYPE},
};
#ifdef EVENTS
  pthread_t event_thread; 
#endif

  commands_list = om_commands_list;
  n_commands = sizeof(om_commands_list)/sizeof(DeviceCommandListEntry); 

#ifdef OMS_DS_DEBUG
   printf("OregonMaxe::OregonMaxe(): create an object of name: %s\n", name);
#endif /*OMS_DS_DEBUG*/

/*
 * if class has not been initialised then call ClassInitialise
 */
   if (OregonMaxe::class_inited != 1)
   {
      if (OregonMaxe::ClassInitialise(error) != DS_OK)
      {
         return;
      }
   }

   *error = 0;

/*
 * initialise this device with its default resources from the static database
 */
   GetResources(name,error);

/*
 * create an Oms sub-object for the VME-58 card, there will be one
 * per OregonMaxe object
 */
   oms = new Oms(descriptor,error);

   if (oms == NULL)
   {
      printf("OregonMaxe::OregonMaxe() could not create Oms object, aborting !\n");
      return;
   }

   if (oms->ResetCard (error) != DS_OK)
   {
      printf("OregonMaxe::OregonMaxe() failed to reset card, continuing\n");
   }

   if (InitialiseMotors (error) != DS_OK)
   {
      printf("OregonMaxe::OregonMaxe() failed to initialise motors, continuing\n");
   }

#ifdef EVENTS
   if (events == 1)
   {
      printf("OregonMaxe::OregonMaxe() events supported - create event thread\n");
      pthread_create(&event_thread, (pthread_attr_t *)NULL, (void*)oms_state_pos_change_event_thread,
                     this);
   }
   else
#endif
   {
      printf("OregonMaxe::OregonMaxe() events not supported - do not create event thread\n");
   }

   return;
}
/*============================================================================

Function:	long OregonMaxe::GetResources()

Description:	method to be called to initialise the fields of
		an OregonMaxe object with resources from the database.

Arg(s) In:	

Arg(s) Out:	long *error     - pointer to error code, in case routine fails

Error Codes:	DevErr_DeviceOpen
		DevErr_DriverNotFound
		+ errors returned by db_getresource()
 
Returns:  	DS_OK      If no error
	        DS_NOTOK   If error
=============================================================================*/
long OregonMaxe::GetResources (char *dev_name, long *error)
{
   short i;
   char axename[80], *axeid = NULL ;

   *error = 0;

#ifdef OMS_DS_DEBUG
    printf("OregonMaxe::GetResources\n");
#endif /*OMS_DS_DEBUG*/

	
   descriptor =  "" ;
   nb_axes = OMS_MAX_MOTORS;
   events = 0;

   res_object[0].resource_adr        = &nb_axes;
   res_object[1].resource_adr        = &descriptor;
   res_object[2].resource_adr        = &events;
   
   if(db_getresource(dev_name, res_object, res_object_size, error))
   {
   	printf("OregonMaxe::GetResources(): error detected reading nb_axes resources in database (%d)\n",
   		*error);
   	return(DS_NOTOK);
   }

   if (nb_axes > OMS_MAX_MOTORS)
   {
   	printf("OregonMaxe::GetResources() maximum no. of axes supported is %d !\n",
   		OMS_MAX_MOTORS);
        nb_axes = OMS_MAX_MOTORS;
   }

    for ( i = 0; i < nb_axes; i++) 
    {
        sprintf(axename,"axe_ident%d",i);

        axeid="";
        res_axe_name[0].resource_adr  = &(axeid);
        res_axe_name[0].resource_name = axename;

	if (db_getresource(dev_name, res_axe_name, res_axe_name_size, error) != 0)
    	{
   		printf("OregonMaxe::GetResources():  error detected reading name of axe %d (%d)\n",
   		i, *error);
        	return(DS_NOTOK);
    	}
    	
// program default values per axis
    	
    	axes[i].channel = 0;
    	axes[i].units = STEPS;
    	axes[i].velocity = 0;
    	axes[i].firststeprate = 0;
    	axes[i].acceleration = 0;
    	axes[i].accelslope = OMS_ACC_LINEAR;
    	axes[i].backlash = 0;
    	axes[i].powerauto = 0;		
   	axes[i].enc_steps_mm = 1.0;
    	axes[i].enc_factor = 1.0;
	axes[i].enc_type = "NONE";
    	axes[i].enc_position = 0;
    	axes[i].position = 0;
    	axes[i].position_ok = FALSE;
    	axes[i].direction = OMS_POSITIVE;
    	
    		/* Lecture dans database des parametres de l'axe */
    		
        res_axe[0].resource_adr = &(axes[i].channel);
        res_axe[1].resource_adr = &(axes[i].velocity);
        res_axe[2].resource_adr = &(axes[i].firststeprate);
        res_axe[3].resource_adr = &(axes[i].acceleration);
        res_axe[4].resource_adr = &(axes[i].accelslope);
        res_axe[5].resource_adr = &(axes[i].backlash);
        res_axe[6].resource_adr = &(axes[i].powerauto);
//
// encoders not supported at present
//
//      res_axe[7].resource_adr = &(axes[i].enc_type);
//      res_axe[8].resource_adr = &(axes[i].enc_ident);
//      res_axe[9].resource_adr = &(axes[i].enc_steps_mm);
//      res_axe[10].resource_adr = &axes[i].enc_factor);
        
    	if (db_getresource(axeid, res_axe, res_axe_size, error) != 0)
    	{
   		printf("OregonMaxe::GetResources(): error detected reading resources from database for axe %d (error=%d)\n",
   		i, *error);
        	return(DS_NOTOK);
    	}

    	printf("OregonMaxe::GetResources(): static database resources for axis %d : \n",i);
    	printf("	%s	channel           %d\n",axeid,axes[i].channel);
    	printf("	%s	velocity          %d\n",axeid,axes[i].velocity);
    	printf("	%s	firststeprate     %d\n",axeid,axes[i].firststeprate);
    	printf("	%s	acceleration      %d\n",axeid,axes[i].acceleration);
    	printf("	%s	accelerationslope %d\n",axeid,axes[i].accelslope);
    	printf("	%s	backlash          %d\n",axeid,axes[i].backlash);
    	printf("	%s	powerauto         %d\n",axeid,axes[i].powerauto);
//
// encoders not supported at present
//
//  	printf("enc_type %s\n"		, ds->omsmaxe.ptabAxes[i].enc_type);
//  	printf("enc_ident %s\n"		, ds->omsmaxe.ptabAxes[i].enc_ident);
//  	printf("enc_steps_mm %f\n"	, ds->omsmaxe.ptabAxes[i].enc_steps_mm);
//  	printf("enc_factor%f\n"		, ds->omsmaxe.ptabAxes[i].enc_factor);

	channels[i] = axes[i].channel;

    }
   
   return(DS_OK);
}

/*============================================================================

Function:	long OregonMaxe::InitialiseMotors()

Description:	method to be called to initialise all motors of 
		an OregonMaxe object by downloading values to the
		hardware e.g. velocity, acceleration etc. This
		method will first initialise the card.

Arg(s) In:	

Arg(s) Out:	long *error     - pointer to error code, in case routine fails

Error Codes:	DevErr_DeviceOpen
		DevErr_DriverNotFound
		+ errors returned by db_getresource()
 
Returns:  	DS_OK      If no error
	        DS_NOTOK   If error
=============================================================================*/
long OregonMaxe::InitialiseMotors (long *error)
{

   *error = 0;
   long i;
    
// intialise the Oms (VME58/PC68) card so that we can send requests to it

   if (oms->InitialiseCard( error) == DS_OK)
   {
   	state = DEVON ;
	printf("OregonMaxe::Initialise(): Oms card initialised correctly\n");
   }
   else
   {
       	state = DEVFAULT ;
	printf("OregonMaxe::Initialise(): problem initialising Oms card (error=%ld\n", *error);
       	return(DS_NOTOK);
   }

    
#ifdef EVENTS
   pthread_mutex_lock(&oms_state_mutex);
#endif
   for ( i = 0; i < nb_axes; i++) 
   {
	if (oms->InitialiseMotor (axes[i].channel,
				  axes[i].velocity,
				  axes[i].firststeprate,
				  axes[i].acceleration,
				  axes[i].accelslope,
				  axes[i].powerauto,
				  error) 
	    == DS_OK) 
        {
		axes[i].state = DEVON;
    		printf("OregonMaxe::Initialise(): motor channel %d of axis %d\n",
    	   		axes[i].channel,i);
	    
	}
	else 
	{
	 	axes[i].state = DEVFAULT;
    		printf("OregonMaxe::Initialise(): motor channel %d of axis %d is in DEVFAULT state (error=%d) !\n",
    	   		axes[i].channel,i,*error);
	}
   }
#ifdef EVENTS
   pthread_mutex_unlock(&oms_state_mutex);
#endif

   return(DS_OK);
}

/*======================================================================
 Function:      long StateMachine ()

 Description:	this routine is reserved for checking wether the command
		requested can be executed in the present state.
		
		Because the device is a card and because we do not know
		for which axis the command is intended the StateMachine
		cannot determine if the command should be refused or
		not. All it does therefore is update the state of all
		the motors and return DS_OK. The command will be checked 
		based on the axis requested and its present state in
		the actual command method. 

		note : this is a limitation of the way the StateMachine 
		is implemented in the Device Server Model.

 Arg(s) In:	Omsmaxe ds - device on which command is to executed
		DevCommand cmd - command to be executed

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
  
 Error Codes:	

 Returns:  	DS_OK      If no error
 		DS_NOTOK   If error
=======================================================================*/

long OregonMaxe::StateMachine(DevCommand cmd,long *error)
{
   *error = 0;
   long i, status_all[OMS_MAX_MOTORS];

//
// update state for all motors
//
#ifdef EVENTS
   pthread_mutex_lock(&oms_state_mutex);
#endif
//
// replace with single command to speed up response time for VME58 - andy 3/9/99
//
#if PC68
     for (i=0; i<nb_axes; i++)
     {
        oms->ReadStatus(axes[i].channel,&(axes[i].state),error);
     }
#else /* PC68 */
// preserve present state e.g. DEVMOVING
   for (i=0; i<nb_axes; i++)
   {
      status_all[i] = axes[i].state;
   }
// read multiple states in one go
   oms->ReadMultipleStatus(nb_axes,channels,status_all,error);
   for (i=0; i<nb_axes; i++)
   {
#ifdef OMS_DS_DEBUG
      if (axes[i].state != status_all[i])
      {
         printf("OregonMaxe::StateMachine(): state changed on axe %d, new state %d\n",
                 i, status_all[i]);
      }
#endif /* OMS_DS_S_DEBUG */
      axes[i].state = status_all[i];
   }
#endif /* PC68 */
#ifdef EVENTS
   pthread_mutex_unlock(&oms_state_mutex);
#endif

   return(DS_OK);
}


/*======================================================================
 Function:      long CommandAllowedOnMotor ()

 Description:	This method is used to check whether the requested
		command can be executed on a motor. This depends
		on the motor state and command. This method implements
		the real StateMachine(). This cannot be done in the
		StateMachine because it has no information about which
		motor to exact the command on. It assumes however that
		the motor state has been updated by the StateMachine.
		
 Arg(s) In:	DevCommand cmd - command to be executed (long)
		long i_axe - motor to execute the command on

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
  
 Error Codes:	DevErr_AttemptToViolateStateMachine
 		DevErr_DeviceIllegalParameter
 		DevErr_UnrecognisedState

 Returns:  	DS_OK      If no error
 		DS_NOTOK   If error
=======================================================================*/
long OregonMaxe::CommandAllowedOnMotor (DevCommand cmd, long i_axe, long *error)
{
   short axe_state = 0;
   long iret = DS_OK;

   *error = FALSE;

// is i_axe valid ?

   if ( i_axe < 0 || i_axe > (nb_axes - 1) ) 
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

   axe_state = axes[i_axe].state ;	
	
   switch (axe_state)
   {
   	case (DEVON) 	:	
   	   switch (cmd)
   	   {

// the following Command(s) are not implemented 

                 case DevMoveReference	:	*error = DevErr_CommandNotImplemented ;
                 				iret = DS_NOTOK;
                     				break;
 		
                 default		:	break;   	   
           } 
   	   break ;	// all commands allowed

   	case (DEVMOVING):	
   	{
   	   switch (cmd)
   	   {
   			// allowed commands
                 case DevState		:	break;
                 case DevStatus		:	break;
                 case DevReadState	:	break;
                 case DevReadSwitches	:	break;
                 case DevReadPosition	:	break;
                 case DevReadMulPos	:	break;
                 case DevReset		:	break;
                 case DevAbortCommand	:	break;
                 case DevReadAcceleration:	break;
                 case DevReadVelocity	:	break;
                 case DevReadFStepRate	:	break;
                 case DevReadHomeSlewRate:	break;
                 case DevReadBacklash	:	break;
                 case DevReadMulEncPos	:	break;
                 case DevReadEncPos	:	break;
 
                 default		:	*error = DevErr_AttemptToViolateStateMachine ;
                 				iret = DS_NOTOK;
                     				break;
   	   } 
   	   break;
   	}
   	
   	case (DEVFAULT) :	
   	{
   	   switch (cmd)
   	   {
   			// allowed commands
                 case DevState		:	break;
                 case DevStatus		:	break;
                 case DevReadState	:	break;
                 case DevReadSwitches	:	break;
                 case DevReadPosition	:	break;
                 case DevReadMulPos	:	break;
                 case DevReset		:	break;
                 case DevAbortCommand	:	break;
                 case DevReadAcceleration:	break;
                 case DevReadVelocity	:	break;
                 case DevReadFStepRate	:	break;
                 case DevReadHomeSlewRate:	break;
                 case DevReadBacklash	:	break;
                 case DevReadMulEncPos	:	break;
                 case DevReadEncPos	:	break;
                 case DevMoveRelative	:	break;
                 case DevMoveAbsolute	:	break;
                 case DevMoveMultiple	:	break;
 
                 default		:	*error = DevErr_AttemptToViolateStateMachine ;
                 				iret = DS_NOTOK;
                     				break;
   	   } 
   	   break;
   	}
   	   
   	default		:
   	{
   	   *error = DevErr_UnrecognisedState;
   	   iret = DS_NOTOK;
   	   break;
   	}
   } 

   return(iret);
}


/*============================================================================
 Function:      long AbortCommand ()

 Description:	 Abort a movement on a motor and all commands queued
   	
 Arg(s) In:	 DevLong  	*argin  - motor number
   				  
 Arg(s) Out:	 DevVoid	*argout - nothing
		 long		*error	- pointer to error code, in case
		 			routine fails.
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_CommandFailed
 
 Returns:  	DS_OK      If no error
            	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::AbortCommand(void *vargin,void *vargout,long *error)
{
   long i_axe ;
   DevLong *argin;
   
   *error = 0;
   argin = (DevLong*)vargin;

   i_axe = *argin ;
   
   if (CommandAllowedOnMotor(DevAbortCommand, i_axe, error) != DS_OK)
   {	
   	printf("OregonMaxe::AbortCommand(): i_axe %d exits with error (%d)\n", i_axe, *error);
	return (DS_NOTOK);
   }		

   if (oms->Abort (axes[i_axe].channel, error) != DS_OK)
   {	
   	printf("OregonMaxe::AbortCommand(): i_axe %d exits with error (%d)\n", i_axe, *error);
	return (DS_NOTOK);
   }		
   
//
// don't set state automatically - wait for card to set DONE bit
//
// pthread_mutex_lock(&oms_state_mutex);
// axes[i_axe].state = DEVON;
// pthread_mutex_unlock(&oms_state_mutex);

   return(DS_OK);
}


/*============================================================================
 Function:      long LoadEncoderPosition ()

 Description:	Load the encoder position with the specified value
   	
 Arg(s) In:	 DevMotorFloat  *argin  - Structure 2 valeurs : 
		 			  - (long) numero d'axe
					  - (float) position absolue souhaitee
   				  
 Arg(s) Out:	 DevVoid	*argout - rien
		 long		*error	- pointer to error code, in case
		 			  routine fails.
		 			
 Error code(s):	 DevErr_DeviceIllegalParameter
		 DevErr_DeviceWrite
 		 DevErr_CommandIgnored
 	
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::LoadEncoderPosition(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long ret = DS_OK;
   long position = 0;
   long i_axe;
   
   *error = DevErr_CommandNotImplemented;

   return(DS_NOTOK);
}


/*============================================================================
 Function:      long LoadPosition ()

 Description:	Load the position counter of the motor with the 
		specified value
   	
 Arg(s) In:	 DevMotorFloat  *argin  - Structure 2 valeurs : 
					     - (long) motor number
					     - (float) position
   				  
 Arg(s) Out:	 DevVoid	*argout - rien
		 long		*error	- pointer to error code, in case
		 			  routine fails. 
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		 			
 Returns:  	DS_OK      If no error
            	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::LoadPosition(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe ;
   long position = 0 ;

   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   
   if (CommandAllowedOnMotor(DevLoadPosition, i_axe, error) != DS_OK)
   {	
   	printf("OregonMaxe::LoadPosition(): command for i_axe %d not allowed (error=%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }		

   position = (long)argin->value ;
#ifdef EVENTS
   pthread_mutex_lock(&oms_position_mutex);
#endif
   if ( oms->LoadPosition (axes[i_axe].channel, position, error )
	!= DS_OK )	
   {	
#ifdef EVENTS
   	pthread_mutex_unlock(&oms_position_mutex);
#endif
   	printf("OregonMaxe::LoadPosition(): i_axe %d exits with error (%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }		

   axes[i_axe].position = position ;	
   axes[i_axe].position_ok = TRUE ;
#ifdef EVENTS
   pthread_mutex_unlock(&oms_position_mutex);
#endif
   	
   return(DS_OK);
}


/*============================================================================
 Function:      long MoveAbsolute ()

 Description:	Move a motor to absolute position (in steps)
   	
 Arg(s) In:	DevMotorFloat 	*argin  - Structure 2 values :
					   - (long) motor number 
					   - (float) absolute position 
   				  
 Arg(s) Out:	DevVoid		*argout - nothing
		long		*error	- pointer to error code, in case
		 			routine fails. 
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		  
 Returns:  	DS_OK      If no error
            	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::MoveAbsolute(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe = 0 ;
   long abs_position = 0;
   
   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   abs_position = (long)argin->value ;
   
   if (CommandAllowedOnMotor(DevMoveAbsolute, i_axe, error) != DS_OK)
   {	
   	printf("OregonMaxe::MoveAbsolute(): command for i_axe %d not allowed (error=%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }		

   if (UpdateDirection(i_axe, OMS_ABSOLUTE_MOVE, abs_position, error) != DS_OK)
   {	
   	printf("OregonMaxe::MoveAbsolute(): SaveDirection(i_axe=%d) exits with error (%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }
   if (oms->MoveAbsolute (axes[i_axe].channel, abs_position, 
                          axes[i_axe].backlash, 
                          axes[i_axe].direction, error) != DS_OK)
   {	
   	printf("OregonMaxe::MoveAbsolute(): oms->MoveAbsolute(i_axe=%d) exits with error (%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }		
   	  
// manage state

#ifdef EVENTS
   pthread_mutex_lock(&oms_state_mutex);
#endif
   axes[i_axe].state = DEVMOVING ;
#ifdef EVENTS
   pthread_mutex_unlock(&oms_state_mutex);
#endif
   axes[i_axe].position_ok = FALSE ;
   
   return(DS_OK);
}
/*============================================================================
 Function:      long MoveMultiple()

 Description:	Move multiple motors. Absolute and relative movements are
		sent separately.
   	
 Arg(s) In:	DevMulMove  	*argin  - Structure 3 arrays of 8 values :
					- (long) type of movement
					   (NO_ACTION (0), ABSOLUTE_MOVE (1),
					    RELATIVE_MOVE (2)) 
					- (float) value to move
					- (long) delay (ms) ... not used
   				  
 Arg(s) Out:	DevVoid	*argout - rien
		long		*error	- pointer to error code, in case
		 			routine fails. 
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
   
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 
 WARNING:   The structure defined in maxe_xdr.h for DevMulMove
            allows a maximum of 8 motors. No more can be moved
            simultaneously.
============================================================================*/
long  OregonMaxe::MoveMultiple(void *vargin,void *vargout,long *error)
{
   DevMulMove *argin;
   long	i, j;
   enum { REL, ABS } ;  
   short todo[2];
   static short action[OMS_MAX_MOTORS] ;
   static long position[OMS_MAX_MOTORS] ;
   static long direction[OMS_MAX_MOTORS] ;
   static long backlash[OMS_MAX_MOTORS] ;
   long ret = DS_OK ;
   
   *error = 0;
   argin = (DevMulMove*)vargin;

   for (i = 0; i < nb_axes; i++)
   {
   		action[i] = OMS_MOT_NOTHING ;
   		position[i] = 0 ;
   }
	
   todo[REL] = todo[ABS] = 0 ;
   for (i = 0; i < nb_axes; i++)
   {

   	if (   (argin->action[i] == RELATIVE_MOVE || argin->action[i] == ABSOLUTE_MOVE) 
   	    && (i < nb_axes))
   	{
//#ifdef OMS_DS_DEBUG
//   		printf ("i= %d, k= %d, action %d, position %d (%f)\n",
//   			 i, k, argin->action[i], (long) argin->position[i], 
//		         argin->position[i] ) ;
//#endif /*OMS_DS_DEBUG*/
   		if (CommandAllowedOnMotor(DevMoveMultiple, i, error) != DS_OK)
   		{
   			ret = DS_NOTOK ;
   			break;
   		}
   		action[i] = (short) argin->action[i] ;
   		position[i] = (long) argin->position[i] ;
   		
   		if (UpdateDirection (i, action[i], position[i], error) != DS_OK)
   		{
   			ret = DS_NOTOK ;
   			break;
   		}
		direction[i] = axes[i].direction;
		backlash[i] = axes[i].backlash;

   		if (action[i] == ABSOLUTE_MOVE)
		{
			todo[ABS] = ABSOLUTE_MOVE ;
		}
   		else 
		{
			if (action[i] == RELATIVE_MOVE)
			{
				todo[REL] = RELATIVE_MOVE ;
			}
		}

   	}
   }
   if (ret == DS_NOTOK)
   {	
   	printf("OregonMaxe::MoveMultiple(): exits with error (error=%d)\n", *error);
	return  (DS_NOTOK);
   }		

   	
   for (j = 0; j < 2 ; j++)
   {
   	if (todo[j])	
   	{
   	    if (oms->MoveMultiple (todo[j], nb_axes, channels, action, position, 
	                           direction, backlash, error) != DS_OK)
   	    {	
   		printf("OregonMaxe::MoveMultiple(): oms->MoveMultiple() returned error (error=%d)\n", *error);
		return  (DS_NOTOK);
   	    }		
	    for (i = 0; i < nb_axes; i++)
	    {
		if ((argin->action[i] == todo[j]) && (i < nb_axes))
	   	{
#ifdef EVENTS
   			pthread_mutex_lock(&oms_state_mutex);
#endif
			axes[i].state = DEVMOVING ;
#ifdef EVENTS
   			pthread_mutex_unlock(&oms_state_mutex);
#endif
			axes[i].position_ok = FALSE ;
#ifdef OMS_DS_DEBUG
   			printf("OregonMaxe::MoveMultiple(): axe %d MOVING state %d\n", i,axes[i].state);
#endif /*OMS_DS_DEBUG*/
	  	}	
	    }
	}
   }

   return(DS_OK) ;
}


/*============================================================================
 Function:      long MoveReference ()

 Description:	Move a motor to its reference
   	
 Arg(s) In:	DevMotorLong  	*argin  - Structure 2 values : 
		 			   - (long) number of motor
		 			   - (long) value (0,1 ou -1):
						force or default direction
   				  
 Arg(s) Out:	DevVoid	*argout - rien
		long	*error	- pointer to error code, in case
	 			  routine fails. 

 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		DevErr_DeviceRead
 ============================================================================*/

long  OregonMaxe::MoveReference(void *vargin,void *vargout,long *error)
{
   DevMotorLong *argin;
   long i_axe ;
   long direction ;
   
   *error = 0;
   argin = (DevMotorLong*)vargin;

   i_axe = argin->axisnum ;
   direction = argin->value ;
   
   if (CommandAllowedOnMotor(DevMoveReference, i_axe, error) != DS_OK)
   {	
   	printf("OregonMaxe::MoveReference(): command not allowed on motor %d (error=%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }		

   if (UpdateDirection (i_axe, OMS_REFERENCE_MOVE, direction, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }

   if (oms->MoveReference(axes[i_axe].channel, direction, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		
   	  
#ifdef EVENTS
   pthread_mutex_lock(&oms_state_mutex);
#endif
   axes[i_axe].state = DEVMOVING ;
#ifdef EVENTS
   pthread_mutex_unlock(&oms_state_mutex);
#endif
   axes[i_axe].position_ok = FALSE ;

   return(DS_OK);
}


/*============================================================================
 Function:      long MoveRelative ()

 Description:	Move a motor relative to its current position
   	
 Arg(s) In:	Omsmaxe 	ds 	- 
		DevMotorFloat 	*argin  - Structure 2 values : 
		 			    - (long) motor number
		 			    - (float) value
   				  
 Arg(s) Out:	DevVoid	*argout - nothing
		long	*error	- pointer to error code, in case
		 		  routine fails. 
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
 ============================================================================*/

long  OregonMaxe::MoveRelative(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe = 0 ;
   long position = 0 ;
   
   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   position = (long) argin->value ;
   
   if (CommandAllowedOnMotor(DevMoveRelative, i_axe, error) != DS_OK)
   {	
   	printf("OregonMaxe::MoveRelative(): command not allowed on motor %d (error=%d)\n", i_axe, *error);
	return  (DS_NOTOK);
   }		

   if (UpdateDirection (i_axe, OMS_RELATIVE_MOVE, position, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		
   
   if (oms->MoveRelative (axes[i_axe].channel, position, 
                          axes[i_axe].backlash, 
                          axes[i_axe].direction, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		
   	  
#ifdef EVENTS
   pthread_mutex_lock(&oms_state_mutex);
#endif
   axes[i_axe].state = DEVMOVING ;
#ifdef EVENTS
   pthread_mutex_unlock(&oms_state_mutex);
#endif

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadAcceleration ()

 Description:	Read acceleration for this axis, return last value sent
		by device server i.e. do not reread the board
   	
 Arg(s) In:	DevLong  	*argin  - axis number
   				  
 Arg(s) Out:	DevFloat	*argout - acceleration
		long		*error	- pointer to error code, in case
		 			routine fails.

 Error code(s):	DevErr_DeviceIllegalParameter

 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::ReadAcceleration(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevFloat *argout;
   long i_axe = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevFloat*)vargout;

   i_axe = *argin ;

   	   	
   if ( i_axe < 0 || i_axe > nb_axes - 1) 
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

   *argout = (float) axes[i_axe].acceleration ;  	  

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadBacklash ()

 Description:	Read backlash. Return last value sent to the device server
		i.e. do not reread the card.
   	
 Arg(s) In:	DevLong  	*argin  - axis number
   				  
 Arg(s) Out:	DevFloat	*argout - backlash 
		 long		*error	- pointer to error code, in case
		 			  routine fails.
		 			
 Error code(s):	 DevErr_DeviceIllegalParameter
  
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::ReadBacklash(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevFloat *argout;
   long i_axe = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevFloat*)vargout;

   i_axe = *argin ;

   if ( i_axe < 0 || i_axe > (nb_axes - 1)) 
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

    *argout = (float) axes[i_axe].backlash ;  	  

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadEncoderPosition ()

 Description:	 Lecture position du codeur associe a un moteur
   	
 Arg(s) In:	 DevLong  	*argin  - Numero d'axe
   				  
 Arg(s) Out:	 DevFloat	*argout - Position codeur
		 long		*error	- pointer to error code, in case
		 			routine fails.
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_CommandIgnored
		DevErr_DeviceRead
 
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::ReadEncoderPosition(void *vargin,void *vargout,long *error)
{
   long i_axe = 0 ;
   long position = 0 ;
   
   *error = DevErr_CommandNotImplemented;
 
   return (DS_NOTOK);
}


/*============================================================================
 Function:      long ReadFirstStepRate ()

 Description:	Read first step rate. Return last value sent to device server
		i.e. do not read the card.
   	
 Arg(s) In:	DevLong  	*argin  - axis number
   				  
 Arg(s) Out:	DevFloat	*argout - first step rate
		long		*error	- pointer to error code, in case
		 			  routine fails.
		 			
 Error code(s):	 DevErr_DeviceIllegalParameter
 
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::ReadFirstStepRate(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevFloat *argout;
   long i_axe = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevFloat*)vargout;

   i_axe = *argin ;

   if ( i_axe < 0 || i_axe > (nb_axes - 1))
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

   *argout = (float) axes[i_axe].firststeprate ;  	  

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadMultipleEncoderPositions ()

 Description:	Read multiple encoder positions
   	
 Arg(s) In:	DevVarLongArray  	*argin  - Structure 2 values : 
		 			- (u_int) number of axes
					- (long) array of axes numbers
   				  
 Arg(s) Out:	 DevVarFloatArray	*argout - Structure 2 values : 
 					- number of axes 
					- array of encoder positions
		 long		*error	- pointer to error code, in case
		 			  routine fails.
		 			
 Error code(s):	 DevErr_DeviceIllegalParameter
		 DevErr_CommandIgnored
		 DevErr_DeviceRead
  
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
============================================================================*/
long  OregonMaxe::ReadMultipleEncoderPositions(void *vargin,void *vargout,long *error)
{
   DevVarLongArray *argin;
   DevVarFloatArray *argout;
   short i;
   long i_axe, ret, position, localerror;
   static float outarray [OMS_MAX_MOTORS];
   
   *error = DevErr_CommandNotImplemented;

   return(DS_NOTOK);
	
}


/*============================================================================
 Function:      long ReadMultiplePositions ()

 Description:	Read multiple motor positions from card. Values returned
		in motor steps.
   	
 Arg(s) In:	DevVarLongArray  	*argin  - Structure 2 values 
						- number of axes
					        - array of axes numbers 
   				  
 Arg(s) Out:	DevVarFloatArray	*argout - Structure 2 values 
					        - number of axes
					        - array of positions
 		long		*error	- pointer to error code, in case
		 			  routine fails.
  
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		DevErr_DeviceRead
		
Returns:  	DS_OK      If no error
            	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::ReadMultiplePositions(void *vargin,void *vargout,long *error)
{

   DevVarLongArray *argin;
   DevVarFloatArray *argout;
   static long  pos_all[OMS_MAX_MOTORS];
   static float pos_out[OMS_MAX_MOTORS];
   long i, j;
   long i_axe;

   *error = 0;
   argin = (DevVarLongArray*)vargin;
   argout = (DevVarFloatArray*)vargout;
    
   for (i = 0; i < nb_axes ; i++)
   {
   	   pos_all[i] =0 ;
   	   pos_out[i] = 0.0 ;	
   }
   if ( oms->ReadMultiplePositions (nb_axes, channels, pos_all, error) != DS_OK )	
   {
	return  (DS_NOTOK);
   }		

   for (i = 0; i < nb_axes; i++)
   {
       axes[i].position = pos_all[i] ;
       axes[i].position_ok = TRUE ;
   }
    
   for (i = 0; i < argin->length;i++) 
   {
      	i_axe = argin->sequence[i] ;
   	if ( i_axe < 0 || i_axe > (nb_axes - 1) )
   	{
   		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
   	}
       	pos_out[i] = (float) axes[i_axe].position;
   }
    
   argout->length   = argin->length;
   argout->sequence = pos_out;

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadPosition ()

 Description:	Read position of a single motor
   	
 Arg(s) In:	DevLong  	*argin  - axis number
   				  
 Arg(s) Out:	DevFloat	*argout - motor position in steps
		long		*error	- pointer to error code, in case
		 			  routine fails. 
           	
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		DevErr_DeviceRead
		 			
 Returns:  	DS_OK      If no error
           	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::ReadPosition(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevFloat *argout;
   long i_axe = 0 ;
   long position = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevFloat*)vargout;
   
   i_axe = *argin ;
   
   if (oms->ReadPosition (axes[i_axe].channel, &position, error) == DS_NOTOK)
   	return (DS_NOTOK);
  
   *argout = (float) position ;
   
   return(DS_OK);
}


/*============================================================================
 Function:      long ReadState ()

 Description:	Read motor state. Can be DEVON, DEVMOVING, DEVFAULT, or DEVOFF 
   	
 Arg(s) In:	DevLong  	*argin  - axis number
   				  
 Arg(s) Out:	DevVarLongArray *argout - Length = 2 : state of motor + encoder
		long		 *error	- pointer to error code, in case
		 			  routine fails.
		 			 
 Error code(s):	DevErr_DeviceIllegalParameter
 		DevErr_DeviceRead
 		 			
 Returns:  	DS_OK      If no error
 		DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::ReadState(void *vargin, void *vargout,long *error)
{
   DevLong *argin;
   DevVarLongArray *argout;
   static long  mot_state[2];
   long i_axe, ret;
    
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevVarLongArray*)vargout;

   i_axe = *argin ;
   if ( i_axe < 0 || i_axe > (nb_axes - 1) )
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

   mot_state[0] = axes[i_axe].state ;
   mot_state[1] = DEVON; /* encoders not supported yet */
   	
   argout->length      = 2;
   argout->sequence    = mot_state;

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadSwitches ()

 Description:	Read state of limit switches of motor 
   	
 Arg(s) In:	DevLong *argin - axis number
   				  
 Arg(s) Out:	DevLong	*argout - status of limt switch (LIMITSOFF,
				  NEGATLIMIT, POSITLIMIT).
		long	*error	- pointer to error code, in case
		 		  routine fails. 

Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		DevErr_DeviceRead
		 			
 Returns:  	DS_OK      If no error
 		DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::ReadSwitches(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevLong *argout;
   long  i_axe = 0 ;
   short status = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevLong*)vargout;

   i_axe = *argin ;

   if (CommandAllowedOnMotor(DevReadSwitches, i_axe, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

/*
 * command not implemented in Oms class, limits can be deduced from 
 * axis state and direction
 *
 *   if ( oms->ReadLimits (axes[i_axe].channel, &status, error ) != DS_OK )
 *   {	
 *	return  (DS_NOTOK);
 *   }		
 *   *argout = (long) (status & 3) ; * mask off info about DONE and HOME *
 */

   if (axes[i_axe].state == DEVFAULT)
   {
      if (axes[i_axe].direction == OMS_POSITIVE)
         *argout = POSITLIMIT;
      else
         *argout = NEGATLIMIT;
   }
   else
   {
      *argout = LIMITSOFF;
   }

   return(DS_OK);
}


/*============================================================================
 Function:      long ReadVelocity ()

 Description:	Read velocity of motor. Return last value sent to device
		server i.e. do not read the card.
   	
 Arg(s) In:	DevLong 	*argin  - axis number
   				  
 Arg(s) Out:	DevFloat	*argout - velocity
		long		*error	- pointer to error code, in case
		 			routine fails. 
 
 Error code(s):	 DevErr_DeviceIllegalParameter
						
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::ReadVelocity(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevFloat *argout;
   long i_axe = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevFloat*)vargout;

   i_axe = *argin ;
   
   if ( i_axe < 0 || i_axe > (nb_axes - 1) )
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

   *argout = (float) axes[i_axe].velocity ;  	  

   return(DS_OK);
}

/*============================================================================
 Function:      long ReadUnits ()

 Description:	Read units for motor. Because the OregonMaxe server
		only supports steps it will always be STEPS
   	
 Arg(s) In:	DevLong 	*argin  - axis number
   				  
 Arg(s) Out:	DevLong		*argout - units
		long		*error	- pointer to error code, in case
		 			routine fails. 
 
 Error code(s):	 DevErr_DeviceIllegalParameter
						
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::ReadUnits(void *vargin,void *vargout,long *error)
{
   DevLong *argin;
   DevLong *argout;
   long i_axe = 0 ;
   
   *error = 0;
   argin = (DevLong*)vargin;
   argout = (DevLong*)vargout;

   i_axe = *argin ;
   
   if ( i_axe < 0 || i_axe > (nb_axes - 1) )
   {
   	*error = DevErr_DeviceIllegalParameter;
	return (DS_NOTOK);
   }

   *argout = axes[i_axe].units ;  	  

   return(DS_OK);
}


/*============================================================================
 Function:      long Reset ()

 Description:	Reset the card. All movements are stopped and parameters
		can be lost.
   	
 Arg(s) In:	DevVoid *argin  - rien
   				  
 Arg(s) Out:	DevVoid	*argout - rien
		long	*error	- pointer to error code, in case
		 	 	  routine fails.
             	
 Error code(s):	DevErr_CommandFailed

 Returns:  	DS_OK      If no error
            	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::Reset(void *vargin,void *vargout,long *error)
{
   
   *error = 0;
   
   if (oms->ResetCard (error) != DS_OK) return (DS_NOTOK) ;   

// reinitialise motors to present settings

   if (InitialiseMotors (error) != DS_OK) return (DS_NOTOK) ;   

   return(DS_OK);
}


/*============================================================================
 Function:      long SetAcceleration ()

 Description:	Set the acceleration of a motor
   	
 Arg(s) In:	DevMotorFloat  *argin  - Structure 2 values : 
					   - (long) axis number
					   - (float) acceleration (steps/s/s)
					
  Arg(s) Out:	DevVoid		*argout - nothing
		long		*error	- pointer to error code, in case
		 			  routine fails.
		 					 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
    
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::SetAcceleration(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe = 0 ;
   long acceleration = 0 ;
   
   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   acceleration = (long) argin->value ;
   
   if (CommandAllowedOnMotor(DevSetAcceleration, i_axe, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		
  
   if (oms->SetAcceleration (axes[i_axe].channel, acceleration, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

   axes[i_axe].acceleration =  acceleration ;  	  

   return(DS_OK);
}


/*============================================================================
 Function:      long SetBacklash ()

 Description:	Set backlash for a single motor
   	
 Arg(s) In:	DevMotorFloat  *argin  - Structure 2 values : 
					   - (long) number of axis
					   - (float) backlash
					
 Arg(s) Out:	DevVoid		*argout - nothing
		long		*error	- pointer to error code, in case
		 			  routine fails.
		 					 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
    
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::SetBacklash(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe = 0 ;
   long backlash = 0 ;
   
   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   backlash = (long) argin->value ;
      
   if (CommandAllowedOnMotor(DevSetBacklash, i_axe, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

   axes[i_axe].backlash =  backlash ;  	  
   
   return(DS_OK);
}


/*============================================================================
 Function:      long SetContinuous ()

 Description:	Start a continuous motion of a motor. Uses the default 
             	parameters for speed and acceleration. The movement is stopped
             	by a DevAbortCommand or by reaching a switch limit.
 	
 Arg(s) In:	DevLong  	*argin  - Structure 2 values : 
		 			  - (long) axis number
		 			  - (long) direction=sign 
   				  
 Arg(s) Out:	DevVoid		*argout - nothing
		long		*error	- pointer to error code, in case
		 			  routine fails. 
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
		DevErr_DeviceRead
    
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error

 ===========================================================================*/
long  OregonMaxe::SetContinuous(void *vargin,void *vargout,long *error)
{
   DevMotorLong *argin;
   long i_axe = 0 ;
   long direction = 0 ;
   
   *error = 0;
   argin = (DevMotorLong*)vargin;

   i_axe = argin->axisnum ;
   direction = argin->value ;
   
   if (CommandAllowedOnMotor(DevSetContinuous, i_axe, error) != DS_OK)
   {	
	return (DS_NOTOK);
   }		

   if (UpdateDirection (i_axe, OMS_CONTINUOUS_MOVE, direction, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }

   if (oms->MoveContinuous (axes[i_axe].channel, direction, 
   			    axes[i_axe].velocity, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		
   	  
#ifdef EVENTS
   pthread_mutex_lock(&oms_state_mutex);
#endif
   axes[i_axe].state = DEVMOVING ;
#ifdef EVENTS
   pthread_mutex_unlock(&oms_state_mutex);
#endif
   axes[i_axe].position_ok = FALSE ;

   return(DS_OK);
}


/*============================================================================
 Function:      long SetFirstStepRate ()

 Description:	Fixation de la vitesse de base de la rampe
 		d'acceleration
   	
 Arg(s) In:	DevMotorFloat  *argin  - Structure 2 values : 
					   - (long) axis number
					   - (float) first step rate
					
  Arg(s) Out:	DevVoid		*argout - nothing
		long		*error	- pointer to error code, in case
		 			  routine fails.
		 					 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
    
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
 ============================================================================*/

long  OregonMaxe::SetFirstStepRate(void *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe = 0 ;
   long firststeprate = 0 ;
   
   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   firststeprate = (long) argin->value ;
      
   if (CommandAllowedOnMotor(DevSetFirstStepRate, i_axe, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

   if (oms->SetFirstStepRate (axes[i_axe].channel, firststeprate, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

   axes[i_axe].firststeprate =  firststeprate ;  	  

   return(DS_OK);
}


/*============================================================================
 Function:      long SetVelocity ()

 Description:	Set the slewing velocity of a motor 
   	
 Arg(s) In:	DevMotorFloat  	*argin  - Structure 2 values : 
					   - (long) axis number
					   - (float) slew rate (steps/sec)
   				  
 Arg(s) Out:	DevVoid		*argout - nothing
		long		*error	- pointer to error code, in case
		 			  routine fails. 
		 			
 Error code(s):	DevErr_DeviceIllegalParameter
		DevErr_DeviceWrite
 		DevErr_DeviceRead
    
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
============================================================================*/

long  OregonMaxe::SetVelocity(void  *vargin,void *vargout,long *error)
{
   DevMotorFloat *argin;
   long i_axe = 0 ;
   long velocity = 0 ;
   
   *error = 0;
   argin = (DevMotorFloat*)vargin;

   i_axe = argin->axisnum ;
   velocity = (long) argin->value ;   
   
   if (CommandAllowedOnMotor (DevSetVelocity, i_axe, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

   if (oms->SetVelocity (axes[i_axe].channel, velocity, error) != DS_OK)
   {	
	return  (DS_NOTOK);
   }		

   axes[i_axe].velocity =  velocity ;  	  

   return(DS_OK);
}


/*======================================================================
 Function:    	long UpdateDirection ()

 Description:	Update the direction register of the requested move.
		This information is useful for detecting the end of
		motion.
		
 Arg(s) In:	long 		i_axe  	- index of axis in "axes[]" array
		int		movement- type of movement (RELATIF, ABSOLU)
 
 Error code(s):	DevErr_DeviceIllegalParameter
		 
 Returns:  	DS_OK      If no error
             	DS_NOTOK   If error
============================================================================*/

long OregonMaxe::UpdateDirection (long i_axe, long movement, long position, long *error) 
{
	float pos_now = 0.;
	struct timeval tout_1ms = {0,1000};

	*error = 0;

//
// if axis is in DEVFAULT state then send a dummy move in the opposite
// direction to the last move to desactivate the limit switch and
// enable the motor to move. not doing so will mean the new move will
// not be executed - a strange "feature" of the VME58 again - andy 12apr99
//
	if (axes[i_axe].state == DEVFAULT)
	{
		if (axes[i_axe].direction == OMS_POSITIVE)
		{
			oms->MoveRelative(axes[i_axe].channel,-1,0,0,error);
			printf("OregonMaxe::UpdateDirection(): move -1 step off limit\n");
		}
		else
		{
			oms->MoveRelative(axes[i_axe].channel,1,0,0,error);
			printf("OregonMaxe::UpdateDirection(): move +1 step off limit\n");
		}
//
// sleep 1 ms to give OMS card time to digest this command
//
		select(0,NULL,NULL,NULL,&tout_1ms);
	}

	switch (movement)
	{
	    case OMS_RELATIVE_MOVE	:
	    case OMS_REFERENCE_MOVE :
	    case OMS_CONTINUOUS_MOVE:

// direction is the sign of the requested position to move to

	    	if (position >= 0)
		{
	    		axes[i_axe].direction = OMS_POSITIVE;
		}
	    	else 
		{
			axes[i_axe].direction = OMS_NEGATIVE;
		}
	    	break;

	    case OMS_ABSOLUTE_MOVE	:

// update the motor position reading if necessary

	    	if (axes[i_axe].position_ok == FALSE)
	    	{
	    	    if (ReadPosition ((DevVoid*)&i_axe, (DevVoid*)&pos_now, error) != DS_OK)
	    	    {
	    	    	return (DS_NOTOK) ;
	    	    }
	    	}
	    	if (position >= axes[i_axe].position)
	    		axes[i_axe].direction = OMS_POSITIVE;
	    	else
	    		axes[i_axe].direction = OMS_NEGATIVE ;
	    	
	    	break;

	    default		:
	    	*error = DevErr_DeviceIllegalParameter ;
	    	return DS_NOTOK ;
	}
	return (DS_OK);
}

/*============================================================================
 Function:      long ReadStepMode ()

 Description:   Read step mode on a motor - always returns the same value 

 Arg(s) In:     DevLong    *argin  - (long) number of motor

 Arg(s) Out:    DevLong    *argout - step mode
                long       *error  - pointer to error code, in case
                                  routine fails.

 Error code(s): DevErr_DeviceIllegalParameter
                DevErr_DeviceWrite
                DevErr_DeviceRead
 ============================================================================*/

long  OregonMaxe::ReadStepMode(void *vargin,void *vargout,long *error)
{
   DevLong *argin, *argout;
   long i_axe ;
   long stepmode ;

   *error = 0; 
   argin = (DevLong*)vargin;
   argout = (DevLong*)vargout;

   i_axe = *argin ;

   if ( i_axe < 0 || i_axe > nb_axes - 1)
   {
        *error = DevErr_DeviceIllegalParameter;
        return (DS_NOTOK);
   }

   *argout = axes[i_axe].stepmode ;

   return(DS_OK);
}

/*============================================================================
 Function:      long SetStepMode ()

 Description:   Set step mode on a motor 

 Arg(s) In:     DevMotorLong    *argin  - (long) number of motor

 Arg(s) Out:    DevVoid         *argout - nothing
                long            *error  - pointer to error code, in case
                                          routine fails.

 Error code(s): DevErr_DeviceIllegalParameter
                DevErr_DeviceWrite
                DevErr_DeviceRead
 ============================================================================*/

long  OregonMaxe::SetStepMode(void *vargin,void *vargout,long *error)
{
   DevMotorLong *argin;
   long i_axe ;
   long stepmode ;

   *error = 0; 
   argin = (DevMotorLong*)vargin;

   i_axe = argin->axisnum ;
   stepmode = (long) argin->value ;

   if ( i_axe < 0 || i_axe > nb_axes - 1)
   {
        *error = DevErr_DeviceIllegalParameter;
        return (DS_NOTOK);
   }

   axes[i_axe].stepmode = stepmode;

   return(DS_OK);
}
