/******************************************************************************
 *
 * File:     	Oms.cpp
 *
 * Project:     Fast micro-stepping stepper motors
 *
 * Purpose :    Implements low-level C++ object for Oregon card. This
 *		object will implement the OMS ascii protocol to send
 *		commands to the card. It is intended to be called from
 *		a higher level class which implements the Maxe set
 *		standard commands. The card is accessed via the oms
 *		device driver for the Oregon VME-58 and PC-68 cards
 *		written by Richard Hirst (richard@sleepie.demon.co.uk). 
 *		This software is intended to run on Linux/m68k and Linux/x86.
 *
 *		The source has been derived from the C version (bibliOregon.c)
 *		for the	previous generation of the VME board (VME8) written
 *		by Danielle Betaille (betaille@lure.fr)
 *
 * Author(s):	Andy Gotz (goetz@esrf.fr) 
 *		Daniel Betaille (betaille@lure.fr) - original C version
 *
 * Date:	March 1999
 *
 ******************************************************************************/
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <API.h>
#include <DevServer.h>
#include <DevErrors.h>
#include <DevStates.h>
#include <Maxe.h>

#include <Oms.h>
#include <oms.h>
#include <omslib.h>

//---------------------------------------------------------------------------*/
//  - Oms::MotorToAxis 
//  
// - Convert a motor number (1 - 8) to an ascii name which identifies the axis
//   of the form of "Ax ", where x = letter identifying axis (X,Y,Z,T,U,V,R,S)
//
//   If the number of axis is zero the x=A i.e. for all axes
//---------------------------------------------------------------------------*/
long Oms::MotorToAxis (long number, char *name, long *error)
{
	static char axes[]="AXYZTUVRS";
	
	*error = 0;
	
	if (number < 0 || number > OMS_MAX_MOTORS)
	{
		*error= DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
	
	name[0]='A';
	name[1]=axes[number];
	name[2]=' ';
	name[3]='\0';

	return (DS_OK);	
	
} 


//-----------------------------------------------------------------------------
//  - Oms::Oms 
//
//  - Create an object of the Oms class, open the device descriptor
//
//-----------------------------------------------------------------------------

Oms::Oms (char *name, long *error) 
{
	device_name = (char*)malloc(strlen(name)+1);
	sprintf(device_name,"%s",name);
	
	device_fd = open(name, O_RDWR, 0x666);
	if (device_fd < 0)
	{
		printf("Oms::Oms(): error opening oms device %s\n",name);
		*error = -1;
	}
}

//-----------------------------------------------------------------------------
//  - Oms::ResetCard -
//
//  Reset card, all movements are stopped
//  
//-----------------------------------------------------------------------------

long Oms::ResetCard (long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "KL RS;" ;
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;
	
	*error = 0 ;
//
// OMS_RESET is not implemented (yet) i.e. return without calling anything
//
	return(0);

	memset(&oms_req, 0, sizeof(oms_req));
	
#ifdef OMS_DS_DEBUG
	printf("Oms::ResetCard(): call ioctl(OMS_RESET)\n");
#endif /* OMS_DS_DEBUG */
	result = ioctl(device_fd, OMS_RESET, &oms_req);

//	oms_req.cmdstr = cmd_str;
//	oms_req.cmdlen = cmd_len;
//
//#ifdef OMS_DS_DEBUG
//	printf("Oms::ResetCard(): command %s\n", cmd_str);
//#endif /* OMS_DS_DEBUG */
//
//	result = ioctl(device_fd, OMS_CMD_RET, &oms_req);

	if (result != 0)
	{
		printf("Oms::ResetCard(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

	return(result);
}

//---------------------------------------------------------------------------
//  - Oms::InitialiseCard -
//
//  - Initialise oms card with default parameters
//
//---------------------------------------------------------------------------

long Oms::InitialiseCard (long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "KL;" ;	/* flush all axes */
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;
  
  	*error = 0 ;

	memset(&oms_req, 0, sizeof(oms_req));

	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = cmd_len;

#ifdef OMS_DS_DEBUG
	printf("Oms::InitialiseCard(): command %s\n", cmd_str);
#endif /* OMS_DS_DEBUG */

	if ((result = ioctl(device_fd, OMS_CMD_RET, &oms_req)) != 0)
	{
		printf("Oms::InitialiseCard(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

	return (result) ;
}

//-------------------------------------------------------------------------
//  - Oms::SoftLimits -
//
//  - Change the behaviour of the limit inputs to ramp down instead of
//    stopping brutally
//
//-------------------------------------------------------------------------

long Oms::SoftLimits (long motor, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "" ;
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;
	
	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if (Oms::MotorToAxis(motor,cmd_str,error) != DS_OK )
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
	
	strcat(cmd_str,"SL;");
	
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::SoftLimits(): command %s\n", cmd_str);
#endif /* OMS_DS_DEBUG */

	if ((result = ioctl(device_fd, OMS_CMD_RET, &oms_req)) != 0)
	{
		printf("Oms::SoftLimits(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::InitialiseMotor -
//
//  - Initialise all parameters of a motor
//
//-----------------------------------------------------------------------------

long Oms::InitialiseMotor (long motor, long velocity, long firststeprate,
		           long acceleration, short mode, short autopower, 
                           long *error)
{
	*error = 0 ;

	if (motor < 1 || motor > OMS_MAX_MOTORS) 
  	{
  		*error = DevErr_DeviceIllegalParameter ;
  		return (DS_NOTOK) ;
	}
  
	if (SetVelocity(motor,velocity,error) == DS_NOTOK) 
	{
		printf("Oms::InitiliaseMotor(): SetVelocity() failed !\n");
		return (DS_NOTOK);
	}

	if (SetAcceleration(motor,acceleration,error) == DS_NOTOK)
	{
		printf("Oms::InitiliaseMotor(): SetAcceleration() failed !\n");
		return (DS_NOTOK);
	}

	if (SetFirstStepRate(motor,firststeprate,error) == DS_NOTOK)
	{
		printf("Oms::InitiliaseMotor(): SetFirstStepRate() failed !\n");
		return (DS_NOTOK);
	}

  	if (SoftLimits(motor,error) == DS_NOTOK )
  	{
  		printf("Oms::InitiliaseMotor(): SoftLimits() failed !\n");
  		return (DS_NOTOK);
  	}

//
// not supported at present - andy 13apr99
//
//	if (AutoPower(motor,autopower,error) == DS_NOTOK )
//	{
//		return (DS_NOTOK);
//	}
//

  	return (DS_OK);
}

		     
//----------------------------------------------------------------------------
// - Oms::AutoPower
//
// - Set automatic power mode
//
//----------------------------------------------------------------------------

long Oms::AutoPower(long motor, short autopower, long *error)
{
	static char cmd_str[OMS_MAX_BUF] = "", buffer[OMS_MAX_BUF];
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0;

	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
		
	if ( (autopower<0) || (autopower >999) )
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
	if (autopower == 0)
		sprintf (buffer, "AN;");
	else
		sprintf (buffer,"PA0 SE%d;",autopower);

	strcat(cmd_str,buffer);
	
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::AutoPower(): command %s\n", cmd_str);
#endif /* OMS_DS_DEBUG */

	if ((result = ioctl(device_fd, OMS_CMD_RET, &oms_req)) != 0)
	{
		printf("Oms::AutoPower(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

	return (DS_OK);
}

//-----------------------------------------------------------------------------
//  - Oms::SetVelocity -
//
//  - Set slewing velocity for motor
//
//-----------------------------------------------------------------------------

long Oms::SetVelocity (long motor, long velocity, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "", buffer[OMS_MAX_BUF];
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != DS_OK ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
	if (velocity < 0 || velocity > OMS_MAX_VELOCITY)
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

	sprintf(buffer,"VL%d;",velocity);
	strcat (cmd_str,buffer);

	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::SetVelocity(): command %s\n", cmd_str);
#endif /* OMS_DS_DEBUG */

	if ((result = ioctl(device_fd, OMS_CMD_RET,&oms_req)) != 0)
	{
		printf("Oms::SetVelocity(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::SetFirstStepRate -
//
//  - Set firststeprate for motor
//-----------------------------------------------------------------------------

long Oms::SetFirstStepRate (long motor, long firststeprate, long *error)
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

	if (firststeprate < 0 || firststeprate > OMS_MAX_VELOCITY)
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
		
	sprintf(cmd_str+strlen(cmd_str),"VB%d;",firststeprate);

	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::SetFirstStepRate(): command %s\n", cmd_str);
#endif /* OMS_DS_DEBUG */
		
	if ((result= ioctl(device_fd, OMS_CMD_RET, &oms_req)) != 0)
	{
		printf("Oms::SetFirstStepRate(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

	return (DS_OK);
	
}

//-----------------------------------------------------------------------------
//  - Oms::SetAcceleration -
//
//  - Set acceleration for motor
//-----------------------------------------------------------------------------

long Oms::SetAcceleration (long motor, long acceleration, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
	
	if (acceleration < 0 || acceleration > OMS_MAX_ACCELERATION)
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

	sprintf(cmd_str+strlen(cmd_str),"AC%d;",acceleration);


	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::SetAcceleration(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if ((result = ioctl(device_fd, OMS_CMD_RET,&oms_req)) != 0)
	{
		printf("Oms::SetAcceleration(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  

  	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::SetAccelerationMode -
//
//  - Set acceleration mode for all motors (not implemented yet)
//
//-----------------------------------------------------------------------------

long Oms::SetAccelerationMode (long motor, long mode, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = DevErr_CommandNotImplemented ;
	return (DS_NOTOK);
  
/* Pour le cas ou ... */
	
	if (motor < 1 || motor > OMS_MAX_MOTORS )
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

	if (mode == OMS_ACC_LINEAR)
		sprintf(cmd_str,"PF");
	else if (mode == OMS_ACC_COSINE)
		sprintf(cmd_str,"CN");
	else if (mode == OMS_ACC_PARABOLIC)
		sprintf(cmd_str,"PN5");
	else 	
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}
	
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::SetAccelerationMode(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::SetAccelerationMode(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}  
	
	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms:LoadPosition -
//
//  - Load the position counter for a motor
//
//-----------------------------------------------------------------------------

long Oms::LoadPosition (long motor, long position, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

	sprintf(cmd_str+strlen(cmd_str)," LP%d",position);

	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::LoadPosition(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::LoadPosition(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}
//
// reread position to update permanent store
//
     	if (ReadPosition(motor, &position, error) != DS_OK)
	 	printf ("Oms::LoadPosition(): error rereading position, permanenet store not updated\n");
		
  	return (DS_OK) ;
}


//-----------------------------------------------------------------------------
//  - Oms::MoveRelative -
//
//  - Move a motor to a new position relative to the present position
//-----------------------------------------------------------------------------
long Oms::MoveRelative (long motor, long position, long backlash, long direction, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

//
// interpret backlash as follows - if movement has a different sense
// than the sign of the backlash then do a backlash movement i.e.
// move backlash steps extra and then move back backlash to compensate
// for mechanical play - andy 20jul99
//
	if (((direction == OMS_POSITIVE) && (backlash < 0)) ||
	    ((direction == OMS_NEGATIVE) && (backlash > 0)))
	{
		sprintf(cmd_str+strlen(cmd_str),"MR%d;GD;MR%d;GD;ID",position-backlash,backlash);
	}
	else
	{
		sprintf(cmd_str+strlen(cmd_str),"MR%d;GD;ID",position);
	}
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::MoveRelative(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::MoveRelative(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}

	return (DS_OK) ;
}

//----------------------------------------------------------------------------
//  - Oms::MoveAbsolute -
//
//  - Move a motor to an absolute position
//
//-----------------------------------------------------------------------------

long Oms::MoveAbsolute (long motor, long position, long backlash, long direction, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));
  
	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

//
// interpret backlash as follows - if movement has a different sense
// than the sign of the backlash then do a backlash movement i.e.
// move backlash steps extra and then move back backlash to compensate
// for mechanical play - andy 20jul99
//
	if (((direction == OMS_POSITIVE) && (backlash < 0)) ||
	    ((direction == OMS_NEGATIVE) && (backlash > 0)))
	{
		sprintf(cmd_str+strlen(cmd_str),"MA%d;GD;MR%d;GD;ID",position-backlash,backlash);
	}
	else
	{
		sprintf(cmd_str+strlen(cmd_str),"MA%d;GD;ID",position);
	}
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::MoveAbsolute(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::MoveAbsolute(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}

  	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::MoveReference -
//
//  - Move a motor to its reference position
//
//-----------------------------------------------------------------------------

long Oms::MoveReference (long motor, long value, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));
  
	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

	sprintf(cmd_str+strlen(cmd_str),"HM0;GD;ID",value);

	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::MoveReference(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::MoveReference(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}

  	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::MoveContinuous -
//
//  - Start a continuous motion on a motor
//
//-----------------------------------------------------------------------------
long Oms::MoveContinuous (long motor, long direction, long speed, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));
   
	if ((motor == 0) || (MotorToAxis(motor,cmd_str,error) != 0 ))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

/*
 * NOTE - direction is ignored
 */
	sprintf(cmd_str+strlen(cmd_str), "JG%d",speed);
	
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::MoveContinuous(): command %s\n",cmd_str);
#endif /* OMS_DS_DEBUG */

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::MoveContinuous(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}

	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::MoveMultiple -
//
//  - Move up to 8 motors simultaneously 
//
//  TODO - pass the channel numbers along with the movements so as to
//         identify the motor axes uniquely. The present code will only
//         work correctly if the server handles all 8 axes of a card
//         i.e.subsets are not supported
//
//-----------------------------------------------------------------------------

long Oms::MoveMultiple(short move_type, long nb_axes, long channels[],
                       short action[],long position[],
                       long direction[], long backlash[], long *error)
{
	static char cmd_str[OMS_MAX_BUF]="";
	static char backlash_str[OMS_MAX_BUF]="";
	static char cmd_str_rel[] = "AA MR";
	static char cmd_str_abs[] = "AA MA";
	static char cmd_str_backlash[OMS_MAX_BUF];
	static char cmd_str_clear[OMS_MAX_BUF];
	static char *cmd_clear[] ={
		"AX CA,","AY CA,","AZ CA,","AT CA,",
		"AU CA,","AV CA,","AR CA,","AS CA,"};
	static OMS_req_t oms_req;
	int i, j, i_channel, result, todo = 0, i_todo = 0, backlash_todo = 0, status;
	
	*error = 0;
	memset(&oms_req, 0, sizeof(oms_req));
	backlash_str[0] = 0;

	switch (move_type)
	{
		case ABSOLUTE_MOVE :
			strcpy( cmd_str,cmd_str_abs);
			break;
		case RELATIVE_MOVE :
			strcpy( cmd_str,cmd_str_rel);
			break;
		default :
			*error = DevErr_DeviceIllegalParameter ;
			return (DS_NOTOK);
	}

	for (i=0;i<OMS_MAX_MOTORS;i++)
	{
		i_channel = -1;
		for (j=0;j<nb_axes;j++)
		{
			if (i == channels[j]-1) i_channel = j;
		}
//
// if backlash needs to be added then add it to the global movement
//
		if ((i_channel != -1) && (move_type == action[i_channel]))
		{
			j = i_channel;
	        	if (((direction[j] == OMS_POSITIVE) && (backlash[j] < 0)) ||
            		    ((direction[j] == OMS_NEGATIVE) && (backlash[j] > 0)))
        		{
                		sprintf(cmd_str+strlen(cmd_str),"%d,",position[j]-backlash[j]);
			}
			else
			{
				sprintf(cmd_str+strlen(cmd_str),"%d,",position[j]);
			}
//			sprintf(cmd_str_clear,cmd_clear[j]);
			i_todo = j;
			todo++;
		} 
		else
		{
			sprintf(cmd_str+strlen(cmd_str),",");
		}
	}
/*
 * if only single move to execute use AX MA or AX MR instead
 * of AA MA or AA MR command
 */
	if (todo == 1)
	{
		if (move_type == ABSOLUTE_MOVE)
		{
			status = MoveAbsolute(channels[i_todo],position[i_todo], 
			                      backlash[i_todo], direction[i_todo],
			                      error);
		}
		else
		{
			status = MoveRelative(channels[i_todo],position[i_todo], 
			                      backlash[i_todo], direction[i_todo],
			                      error);
		}
		return(status);
	}
	cmd_str[strlen(cmd_str) - 1] = '\0';
        sprintf(cmd_str+strlen(cmd_str),";GD");
//
// now construct a global movement with only the backlash compensations in it
// using a global relative movement
//
	sprintf(backlash_str,cmd_str_rel);
	for (i=0;i<OMS_MAX_MOTORS;i++)
	{
		i_channel = -1;
		for (j=0;j<nb_axes;j++)
		{
			if (i == channels[j]-1) i_channel = j;
		}
		if ((i_channel != -1) && (move_type == action[i_channel]))
		{
			j = i_channel;
			printf("Oms::MoveMultiple(): action for axe %d\n",j);
	        	if (((direction[j] == OMS_POSITIVE) && (backlash[j] < 0)) ||
            		    ((direction[j] == OMS_NEGATIVE) && (backlash[j] > 0)))
        		{
                		sprintf(backlash_str+strlen(backlash_str),"%d,",backlash[j]);
				backlash_todo = 1;
			}
			else
			{
				sprintf(backlash_str+strlen(backlash_str),",");
			}
		} 
		else
		{
			sprintf(backlash_str+strlen(backlash_str),",");
		}
	}
// check if any backlash movement to do
	if (backlash_todo)
	{
		sprintf(cmd_str+strlen(cmd_str),";%s",backlash_str);
		strcat(cmd_str, ";GD;ID;");
	}
	else
	{
		sprintf(cmd_str+strlen(cmd_str),";ID;");
	}

	if (todo == 0)
	{
		printf("Oms::MoveMultiple(): no command !\n");
		return (DS_OK);
	}
//	cmd_str[strlen(cmd_str) - 1] = '\0';
//	strcat(cmd_str_clear,cmd_str);
//	strcpy(cmd_str,cmd_str_clear);

#ifdef OMS_DS_DEBUG
	printf("Oms::MoveMultiple(): command %s\n",cmd_str);
#endif

	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

	if ((result = ioctl(device_fd, OMS_CMD_RET, &oms_req) != 0))
	{
		printf("Oms::MoveMultiple(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}
	return (DS_OK);
}


//-----------------------------------------------------------------------------
//  - Oms::Abort 
//
//  - Abort any pending commands
//
//-----------------------------------------------------------------------------
long Oms::Abort (long motor, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static OMS_req_t oms_req;
	static long result;

	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	if (MotorToAxis(motor,cmd_str,error) != 0 )
  	{
  		*error = DevErr_DeviceIllegalParameter ;
  		return (DS_NOTOK) ;
	}

//
// reactivate the following line in order to get the DONE bit set after aborting
// a movement - andy 20jul99
//
// Q: why was this code disabled ?
//
	strcat (cmd_str, "ST;MR0;GD;ID;");

// "brutal" method i.e. STOP and then set state to DEVON
//
//	strcat (cmd_str, "ST");
	  
	oms_req.cmdstr = cmd_str;
	oms_req.cmdlen = strlen(cmd_str);

#ifdef OMS_DS_DEBUG
	printf("Oms::Abort(): command %s\n",cmd_str);
#endif

	if (ioctl(device_fd, OMS_CMD_RET, &oms_req) == -1)
	{
		printf("Oms::Abort(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}

  	return (DS_OK) ;
}


//-----------------------------------------------------------------------------
//  - Oms::ReadStatus -
//
//  - Read status of motor
//
//-----------------------------------------------------------------------------

long Oms::ReadStatus (long motor, long *status, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static char cmd_resp[80];
	static OMS_req_t oms_req;
	static long result;
	unsigned char motor_bit;
	
	*error = 0 ;
        memset(&oms_req, 0, sizeof(oms_req));
  
	if ((motor < 1) || (motor > 8))
	{
		*error = DevErr_DeviceIllegalParameter;
		return (DS_NOTOK);
	}

//
// WAITING FOR MORE INFORMATON FROM OMS BEFORE IMPLEMENTING THIS
//
// due to problems with the limit interrupt not going away when we
// hit a limit the driver will not detect the limit bit correctly
// replacing this call to the driver with the QA command to the
// board for this axes. QA replies with <lf><cr><cr>PDLH<lf><cr><cr>
// (cf. page 5-53 of VME58 User's Manual) - andy 12apr99
//
// read status flags for all axes
//
//	oms_req.axes = 0xff;
//	if ((result = ioctl(device_fd, OMS_READ_AXES_INFO, &oms_req)) != 0)
//	{
//		*error = DevErr_DeviceWrite;
//		return (DS_NOTOK);
//	}
//
//	motor_bit = 0x01 << (motor-1);
//
//	if (oms_req.f_limit & motor_bit) *status |= OMS_LIMIT;
//	if (oms_req.f_done & motor_bit) *status |= OMS_DONE;
//
// no home bit apparently - how to detect home then ?
//
//	if (oms_req.f_home & motor_bit) *status |= OMS_HOME;
//
// QA code
//
// query axis status using the QA command and interpret the string returned
//

	if (MotorToAxis(motor,cmd_str,error) != 0 )
  	{
  		*error = DevErr_DeviceIllegalParameter ;
  		return (DS_NOTOK) ;
	}

	strcat (cmd_str, "QA;");
        oms_req.cmdstr = cmd_str;
        oms_req.cmdlen = strlen(cmd_str);
	oms_req.rspstr = cmd_resp;
	oms_req.rsplen = 80;
	oms_req.timeout = 1;
	  
	if (ioctl(device_fd, OMS_CMD_RESP, &oms_req) == -1)
	{
		printf("Oms::ReadStatus(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}
//
// decode QA response now
//
	cmd_resp[0] = cmd_resp[1] = cmd_resp[2] = ' ';
//#ifdef OMS_DS_DEBUG
//	printf("Oms::ReadStatus(axis=%d): QA response %s",motor,cmd_resp);
//#endif /* OMS_DS_DEBUG */

	if (cmd_resp[5] == 'L') *status = DEVFAULT;
	if (cmd_resp[4] == 'D') *status = DEVON;

	return (DS_OK) ;
}

//-----------------------------------------------------------------------------
//  - Oms::ReadMultipleStatus -
//
//  - Read status of all motors
//
//-----------------------------------------------------------------------------

long Oms::ReadMultipleStatus (long nb_axes, long channels[], long *status, long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static char cmd_resp[256];
	static OMS_req_t oms_req;
	static long result, i, offset;
	unsigned char motor_bit;
	
	*error = 0 ;
        memset(&oms_req, 0, sizeof(oms_req));
        memset(cmd_str, 0, sizeof(cmd_str));
  
	for (i=0; i< nb_axes; i++)
	{

//
// WAITING FOR MORE INFORMATON FROM OMS BEFORE IMPLEMENTING THIS
//
// due to problems with the limit interrupt not going away when we
// hit a limit the driver will not detect the limit bit correctly
// replacing this call to the driver with the QA command to the
// board for this axes. QA replies with <lf><cr><cr>PDLH<lf><cr><cr>
// (cf. page 5-53 of VME58 User's Manual) - andy 12apr99
//
// read status flags for all axes
//
//	oms_req.axes = 0xff;
//	if ((result = ioctl(device_fd, OMS_READ_AXES_INFO, &oms_req)) != 0)
//	{
//		*error = DevErr_DeviceWrite;
//		return (DS_NOTOK);
//	}
//
//	motor_bit = 0x01 << (motor-1);
//
//	if (oms_req.f_limit & motor_bit) *status |= OMS_LIMIT;
//	if (oms_req.f_done & motor_bit) *status |= OMS_DONE;
//
// no home bit apparently - how to detect home then ?
//
//	if (oms_req.f_home & motor_bit) *status |= OMS_HOME;
//
// QA code
//
// query axis status using the QA command and interpret the string returned
//

		if (MotorToAxis(channels[i],cmd_str+strlen(cmd_str),error) != 0 )
  		{
  			*error = DevErr_DeviceIllegalParameter ;
  			return (DS_NOTOK) ;
		}

		strcat (cmd_str, "QA;");
	}
//#ifdef OMS_DS_DEBUG
//	printf("Oms::ReadMultipleStatus(): QA query %s",cmd_str);
//#endif /* OMS_DS_DEBUG */
        oms_req.cmdstr = cmd_str;
        oms_req.cmdlen = strlen(cmd_str);
	oms_req.rspstr = cmd_resp;
	oms_req.rsplen = 256;
	oms_req.timeout = 1;
	  
	if (ioctl(device_fd, OMS_CMD_RESP, &oms_req) == -1)
	{
		printf("Oms::ReadMultipleStatus(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite ;
		return (DS_NOTOK);
	}
//
// decode QA response now
//
	for (i=0; i< nb_axes; i++)
	{
		offset = i*10;
//#ifdef OMS_DS_DEBUG
//		cmd_resp[offset+0] = cmd_resp[offset+1] = cmd_resp[offset+2] = ' ';
//		printf("Oms::ReadMultipleStatus(): QA response for motor %d %c %c\n",
//		        channels[i],cmd_resp[offset+5],cmd_resp[offset+4]);
//#endif /* OMS_DS_DEBUG */

		if (cmd_resp[offset+5] == 'L') status[i] = DEVFAULT;
		if (cmd_resp[offset+4] == 'D') status[i] = DEVON;
	}

	return (DS_OK) ;
}


//----------------------------------------------------------------------------
//
// - EndOfMovement -
//
// - Detect end of movement
//
//----------------------------------------------------------------------------

long Oms::EndOfMovement (short *todo, long nb_axes, long channels[], short movement[], short endofmove[], long *error) 
{
	static char cmd_str[OMS_MAX_BUF] = "";
	static int  cmd_len = strlen(cmd_str);
	static char cmd_resp[80];
	static OMS_req_t oms_req;
	static long result, position;
	int   i ;
	 
	*error = 0 ;
	*todo = 0;
	
//
// WAITING FOR MORE INFORMATON FROM OMS BEFORE IMPLEMENTING THIS
//
// due to problems with the limit interrupt not going away when we
// hit a limit the driver will not detect the limit bit correctly
// replacing this call to the driver with the QA command to the
// board for this axes. QA replies with <lf><cr><cr>PDLH<lf><cr><cr>
// (cf. page 5-53 of VME58 User's Manual) - andy 12apr99
//
// read status flags for all axes
//
//	oms_req.axes = 0xff;
//	if ((result = ioctl(device_fd, OMS_READ_AXES_INFO, &oms_req)) != 0)
//	{
//		*error = DevErr_DeviceWrite;
//		return (DS_NOTOK);
//	}

	for (i=0; i< nb_axes; i++)
	{
//
// replace global call to READ_AXES_INFO with a call to QA per axes
//
// QA code
//
// query axis status using the QA command and interpret the string returned
//

        	if (MotorToAxis(channels[i],cmd_str,error) != 0 )
        	{
                	*error = DevErr_DeviceIllegalParameter ;
                	return (DS_NOTOK) ;
        	}

        	strcat (cmd_str, "QA;");
        	oms_req.cmdstr = cmd_str;
        	oms_req.cmdlen = strlen(cmd_str);
        	oms_req.rspstr = cmd_resp;
        	oms_req.rsplen = 80;
        	oms_req.timeout = 1;

        	if (ioctl(device_fd, OMS_CMD_RESP, &oms_req) == -1)
        	{
			printf("Oms::EndOfMoveMent(): ioctl() error (reason=%s)\n",
		                oms_strerror(result));
			*error = DevErr_DeviceWrite ;
                	return (DS_NOTOK);
		}
//
// decode QA response now
//
        	cmd_resp[0] = cmd_resp[1] = cmd_resp[2] = ' ';
//#ifdef OMS_DS_DEBUG
//        	printf("Oms::EndOfMovement(axis=%d): QA response %s",i,cmd_resp);
//#endif /* OMS_DS_DEBUG */

		if ( (cmd_resp[4] == 'D') && (movement[i] != OMS_MOT_NOT_MOVING) )
		{
			endofmove[i] = OMS_MOT_FINISHED;
			movement[i] = OMS_MOT_NOT_MOVING;
			*todo = 1;
		}
		else
		{
			endofmove[i] = OMS_MOT_NOTHING;
		}

// in the case of a limit switch the DONE flag is not set 

		if (movement[i] != OMS_MOT_NOT_MOVING)
		{
			if (cmd_resp[5] == 'L')
			{
				endofmove[i]= OMS_MOT_FINISHED;
				movement[i] = OMS_MOT_NOT_MOVING;
				*todo = 1;
			}
		}
	}
	
// reread position to update permanent store

	for (i = 0; i < nb_axes;i++)
	{	
		if (endofmove[i] == OMS_MOT_FINISHED)
		{
		     if (ReadPosition( i+1, &position, error) != 0) return (DS_NOTOK);
		}
	}
		
	return (DS_OK);
}

//-----------------------------------------------------------------------------
//  - ReadPosition -
//
//  - Read motor position 
//
//-----------------------------------------------------------------------------

long Oms::ReadPosition (long motor, long *position, long *error) 
{
	static OMS_req_t oms_req;
	static long result;
	
	*error = 0 ;
	memset(&oms_req, 0, sizeof(oms_req));

	oms_req.axes = 0xff;

	if ((result = ioctl(device_fd, OMS_READ_POSITIONS, &oms_req)) != 0)
	{
		printf("Oms::ReadPosition(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite;
		return (DS_NOTOK);
	}

	*position = oms_req.positions[motor-1];

//
// save position in permanent store (not implement yet)
//
//	_oms_save(card,motor,*position);

	return (DS_OK);
}

//-----------------------------------------------------------------------------
//  - Oms::ReadMultiplePositions 
//
//  - Read positions of all motors
//
//-----------------------------------------------------------------------------

long Oms::ReadMultiplePositions (long nb_axes, long channels[], long positions[], long *error) 
{
	static OMS_req_t oms_req;
	static long result;
	int i;
	
	*error = 0 ;

	memset(&oms_req, 0, sizeof(oms_req));
	oms_req.axes = 0xff;

	if ((result = ioctl(device_fd, OMS_READ_POSITIONS, &oms_req)) != 0)
	{
		printf("Oms::ReadMultiplePosition(): ioctl() error (reason=%s)\n",
		        oms_strerror(result));
		*error = DevErr_DeviceWrite;
		return (DS_NOTOK);
	}

	for (i=0; i<nb_axes; i++)
	{
		positions[i] = oms_req.positions[channels[i]-1];

//
// save position in permanent store (not implement yet)
//
//		_oms_save(card,motor,position[i]);
	}

  	return (DS_OK);
}
