/*
 * This a a threads based test program for the Oregon VME58/PC68 driver
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "omslib.h"

/* Axes are referenced as 0-7, for XYZTUVRS.  We always use AM mode */

char *ammr_spec[] = {
	"ammr",		/* X */
	"ammr,",	/* Y */
	"ammr,,",	/* Z */
	"ammr,,,",	/* T */
	"ammr,,,,",	/* U */
	"ammr,,,,,",	/* V */
	"ammr,,,,,,",	/* R */
	"ammr,,,,,,,"	/* S */
};


/* PC68 has axes X, Y, Z, T */
/* My VME58 has U, V, R, S set up for stepper outputs */

#ifdef VME58
#define BASE_AXIS		4
#elif defined (PC68)
#define BASE_AXIS		0
#endif
#define CMD_BLK_THREADS		4
#define READ_POSITION_THREADS	4
#define WHO_ARE_YOU_THREADS	0

int fd;

pthread_t cmd_blk_threads[CMD_BLK_THREADS];
pthread_t read_position_threads[READ_POSITION_THREADS];
pthread_t who_are_you_threads[WHO_ARE_YOU_THREADS];


/* 'arg' is an axis number, 0=X, 1=Y, etc */

void *
cmd_blk (void *arg)
{
	int axis = (int)arg;
	char axis_name = oms_axis_name(axis);
	struct timeval ts, te;
	double dt, rate;
	int cnt = 0;
	int res;
	char cmdstr[80];
	int step = 50;
	int mr;
	int max_move = 75000;	/* Make this smaller, or timeout bigger
				 * if you don't want errors
				 */
	srand(getpid());

	printf ("%c: CmdBlk starting as pid %d\n", axis_name, getpid());

	res = oms_ack_all_ints(fd, axis);
	res = oms_stop_axis(fd, axis, 4);
	if (res)
	{
		oms_log("%c: Initial STOP failed, aborting: %s\n",
				axis_name, oms_strerror(res));
		return (void *)1;
	}
	gettimeofday(&ts, NULL);
	for (;;)
	{
		/* This uses multi-axis mode, but seemed to work equally well
		 * using single axis mode
		 */

		mr = (rand() % max_move) - max_move/2;
		sprintf (cmdstr, "%s%d;gdid", ammr_spec[axis], mr);
#if 0
		fprintf (stderr, "cmdstr='%s'\n", cmdstr);
#endif
		res = oms_cmd_blk(fd, axis, cmdstr, 2);
		if (res == OMS_ERR_SLIP)
		{
			/* Slip recovery...
			 */
		}
		else if (res == OMS_ERR_LIMIT)
		{
			/* Limit recovery.  I don't know.  Maybe we have
			 * to turn limit detection off and move off the
			 * limit stop.  For now we will just wait until
			 * the limit switch is released.  Normal move
			 * commands will fail while limit is active.
			 */
			while (oms_check_limit(fd, axis))
			{
				oms_log("%c: Limit active\n", axis_name);
				sleep(1);
			}
			/* Clear any outstanding interrupt flags */
			res = oms_ack_all_ints(fd, axis);
			/* XXX Should check result really */
			res = oms_stop_axis(fd, axis, 4);
			if (res)
			{
				oms_log("%c: Failed to stop, aborting!\n",
					axis_name);
				return (void *)1;
			}
		}
		else if (res == OMS_ERR_TIMEOUT)
		{
			/* A command failed to complete within the
			 * timeout.  Issue a ST ID for the axis, and
			 * wait for the interrupt.  Should then
			 * acknowledge any further queued ints.
			 */
			res = oms_stop_axis(fd, axis, 4);
			if (res)
			{
				oms_log("%c: Failed to stop, aborting!\n",
					axis_name);
				return (void *)1;
			}
		}
		else if (res)
		{
			/* Unexpected error...
			 */
			return (void *)1;
		}
		else
		{
#ifdef DO_RQ
			/* Issue an RQ and check that queue space for
			 * this axis is 200, indicating no commands
			 * outstanding.
			 */
			char rsp[80];

			res = oms_cmd_resp(fd, "rq", rsp, 80);
			if (res)
			{
				return (void *)1;
			}
			if (sscanf(rsp+2,"%ld,%ld,%ld,%ld",
				pos.positions + 0,
				pos.positions + 1,
				pos.positions + 2,
				pos.positions + 3) != 4)
			{
				printf ("rq='%s'\n", rsp);
			}
			else if (pos.positions[axis] != 200)
			{
				printf ("Error axis %c\n", axis_name);
				exit (1);
			}
#endif
		}

		cnt++;
		if (!(cnt % step))
		{
			gettimeofday(&te, NULL);
			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			rate = dt / (double)cnt;
			fprintf (stderr, "%c: CmdBlk, %d cycles in %.3fs;"
					" average %.3fms per request\n",
					axis_name, cnt, dt/1000000, rate/1000);
		}
	}
	return NULL;
}


void *
who_are_you (void *arg)
{
	int thr = (int)arg;
	struct timeval ts, te;
	double dt, rate;
	int cnt = 0;
	int res;
	char resp[80];

	printf ("Who Are You thread %d starting as pid %d\n", thr, getpid());

	gettimeofday(&ts, NULL);
	for (;;)
	{
		res = oms_cmd_resp(fd, "wy", resp, 80);
			continue;
		cnt++;
		if (!(cnt % 1000))
		{
			gettimeofday(&te, NULL);
			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			rate = dt / (double)cnt;
			fprintf (stderr, "%d: WhoAreYou, %d cycles in %.3fs;"
					" average %.3fms per request\n",
					thr, cnt, dt/1000000, rate/1000);
		}
	}
	return NULL;
}


void *
read_position (void *arg)
{
	int thr = (int)arg;
	struct timeval ts, te;
	double dt, rate;
	int cnt = 0;
	int res;
	long positions[8];

	printf ("Read Position thread %d starting as pid %d\n", thr, getpid());

	gettimeofday(&ts, NULL);
	for (;;)
	{
		res = oms_read_positions(fd, positions);
		if (res)
			continue;
		cnt++;
		if (!(cnt % 1000))
		{
			gettimeofday(&te, NULL);
			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			rate = dt / (double)cnt;
			fprintf (stderr, "%d: ReadPosition, %d cycles in %.3fs;"
					" average %.3fms per request\n",
					thr, cnt, dt/1000000, rate/1000);
		}
	}
	return NULL;
}


int
main (int argc, char **argv)
{
	int i, res;

	fd = open("/dev/oms0", O_RDWR);

	if (fd < 0)
	{
		perror ("open");
		exit (1);
	}

	res = ioctl(fd, OMS_DEBUG, (void *)1);
	if (res)
	{
		oms_log("Set debug level failed: %s\n", oms_strerror(res));
		exit (1);
	}

	for (i = BASE_AXIS; i < CMD_BLK_THREADS+BASE_AXIS; i++)
	{
		res = pthread_create(&cmd_blk_threads[i], NULL,
				cmd_blk, (void *)i);

		if (res)
		{
			fprintf (stderr, "pthread_create() returned %d\n",
					res);
			exit(1);
		}
	}
	for (i = 0; i < READ_POSITION_THREADS; i++)
	{
		res = pthread_create(&read_position_threads[i], NULL,
				read_position, (void *)i);

		if (res)
		{
			fprintf (stderr, "pthread_create() returned %d\n",
					res);
			exit(1);
		}
	}
	for (i = 0; i < WHO_ARE_YOU_THREADS; i++)
	{
		res = pthread_create(&who_are_you_threads[i], NULL,
				who_are_you, (void *)i);

		if (res)
		{
			fprintf (stderr, "pthread_create() returned %d\n",
					res);
			exit(1);
		}
	}
	for (;;)
		sleep(1);
	return 0;
}

