/***********************************************************************
 *  Test harness for the Oregon Micro System's PC58, VME58, PC68
 *  Intelligent Motor Controllers device driver.
 *  Linux 2.0 Operating System.
 * --------------------------------------------------------------------
 *  This code is based on a test harness for the PC58, written by
 *     Tony Denault
 *     Institute for Astronomy, University of Hawaii
 *
 *  Modified for VME58 and PC68 by richard@sleepie.demon.co.uk
 ***********************************************************************
 */

/*--------------------------
**  include files
**--------------------------
*/
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>

#include "omslib.h" 

#define FALSE		0
#define TRUE		1

#define ERR_NONE	0
#define ERR_INV_KW	1
#define ERR_INV_RNG	2
#define ERR_INV_FORMAT	3
#define ERR_NOT_READY	4
#define ERR_FILE_READ	5
#define ERR_NIF		6
#define ERR_BAD_AXES	7

char *Error_string[] = {
	"No error!",
	"Invalid keyword",
	"Invalid range",
	"Invalid format",
	"Not ready",
	"File read error",
	"Function not implemented",
	"Bad axes specification",
};

/*---------------------------
** function prototypes
**---------------------------
*/
int cmd_execute( char * cmd_buf);
int cmd_search_selection_list( char * item, char ** list);

int do_beep(char *par, int soakcnt);
int do_do(char *par, int soakcnt);
int do_help(char *par, int soakcnt);
int do_quit(char *par, int soakcnt);
int do_sleep(char *par, int soakcnt);
int do_reset(char *par, int soakcnt);
int do_reg_handler(char *par, int soakcnt);
int do_read_ints(char *par, int soakcnt);
int do_ack_ints(char *par, int soakcnt);
int do_read_positions(char *par, int soakcnt);
int do_cmd_blk(char *par, int soakcnt);
int do_cmd_force(char *par, int soakcnt);
int do_cmd_sig(char *par, int soakcnt);
int do_cmd_sig_ack(char *par, int soakcnt);
int do_cmd_ret(char *par, int soakcnt);
int do_cmd_resp(char *par, int soakcnt);
int do_dump(char *par, int soakcnt);
int do_debug(char *par, int soakcnt);
int do_timeout(char *par, int soakcnt);
int do_log_io(char *par, int soakcnt);

/*-------------------------------------
*  Global variables
*-------------------------------------
*/
 
struct PARSE_TABLE
{
	char * kw;	/* The opcode string  */
	int (* pfi)();	/* pfi is pointer to function returning an integer */
	char * desc;	/* Description of command */
};
 
struct PARSE_TABLE parse_table[] =
{
	{ "beep",       do_beep,         "BEEP                   Outputs the bell characters" },
	{ "do",         do_do,           "DO <filename>          Execute command from filename" },
	{ "help",       do_help,         "HELP                   Prints this message" },
	{ "reset",	do_reset,	 "RESET                  Reset and reinitialise the driver" },
	{ "reg",	do_reg_handler,	 "REG <axis> <sig>       Register a signal handler for <axis>" },
	{ "ri",		do_read_ints,	 "RI <axes>              Read outstanding interrupts for <axes>" },
	{ "ra",		do_ack_ints,	 "RA <axes>              Read and ack. interrupts fpr <axes>" },
	{ "rp",		do_read_positions,"RP                    Read current position for all axes" },
	{ "wb",		do_cmd_blk,	 "WB <axis> <str>        Write command and block" },
	{ "wf",		do_cmd_force,	 "WF <axis> <str>        Force command and block" },
	{ "ws",		do_cmd_sig,	 "WS <axis> <sig> <str>  Write command and signal completion" },
	{ "wa",		do_cmd_sig_ack,  "WA <axis> <sig> <str>  Write command, signal and ack when done" },
	{ "wr",		do_cmd_ret,	 "WR <str>               Write command and return" },
	{ "ww",		do_cmd_resp,	 "WW <str>               Write command and await response" },
	{ "dump",	do_dump,	 "DUMP                   Ask driver to dump workspace" },
	{ "debug",	do_debug,	 "DEBUG <level>          Set debug level in driver" },
	{ "Quit",       do_quit,         "QUIT                   Quits the application" },
	{ "sleep",      do_sleep,        "SLEEP <sec>            Pauses the program of <sec> seconds" },
	{ "soak",	NULL,		 "SOAK <cnt> ...         Repeat rest of line <cnt> times" },
	{ "timeout",	do_timeout,	 "TIMEOUT <secs>         Timeout to apply to subsequent cmds" },
	{ "logio",	do_log_io,	 "LOGIO <file>           Write io log buffer to <file>" },
};
 
int num_parse_item = { sizeof(parse_table) / sizeof(struct PARSE_TABLE) };

int Quit,
    Fd;

int timeout = 0;

void
skip_space(char **p)
{
	while (**p == ' ')
		(*p)++;
}

int get_dec(char **p)
{
	int n;

	if (!p || !*p || !**p)
		return -1;
	skip_space(p);
	if (**p < '0' || **p > '9')
		return -1;
	n = 0;
	while (**p >= '0' && **p <= '9')
	{
		n = n * 10 + **p - '0';
		(*p)++;
	}
	skip_space(p);
	return n;
}


int
parse_axes (char **p)
{
	int axes = 0;

	if (!p || !*p || !**p)
		return -1;

	if (!strcasecmp(*p, "all"))
		return 0xff;

	while (**p && **p != ' ')
	{
		switch (**p)
		{
		case 'X': case 'x': axes |= OMS_X_AXIS; break;
		case 'Y': case 'y': axes |= OMS_Y_AXIS; break;
		case 'Z': case 'z': axes |= OMS_Z_AXIS; break;
		case 'T': case 't': axes |= OMS_T_AXIS; break;
		case 'U': case 'u': axes |= OMS_U_AXIS; break;
		case 'V': case 'v': axes |= OMS_V_AXIS; break;
		case 'R': case 'r': axes |= OMS_R_AXIS; break;
		case 'S': case 's': axes |= OMS_S_AXIS; break;
		default:
			return -1;
		}
		(*p)++;
	}
	if (axes)
		return axes;
	else
		return -1;
}


char *
axes2str (u8 axes)
{
	static char res[9];
	int i = 0;
	u8 mask = 0x80;

	while (mask)
	{
		if (axes & mask)
			res[i] = "SRVUTZYX"[i];
		else
			res[i] = '-';
		i++;
		mask >>= 1;
	}
	res[8] = '\0';
	return res;
}

void
strxcpy (char *dst, char *src, int len)
{
	strncpy(dst, src, len);
	dst[len-1] = '\0';
	len = strlen(dst);
	while (dst[len-1] == '\n')
		dst[--len] = '\0';
}

void
unpad (char *str, char c)
{
	char *p = str;

	while (*p == c)
		p++;

	memmove(str, p, strlen(p)+1);

	p = str + strlen(str) - 1;

	while (p > str && *p == c)
		*p-- = '\0';
}

int
my_atol(char *p, long *v)
{
	char *end;

	*v = strtol(p, &end, 0);

	if (*v < 0 || *v == LONG_MAX || !*p || *end)
		return -1;
	else
		return 0;
}

void
str_rws(char *str, int len)
{
	char *p = str;

	str[len - 1] = '\0';

	while (*p == '\r' || *p == '\n')
		p++;

	memmove(str, p, strlen(p)+1);

	p = str + strlen(str) - 1;

	while (p > str && (*p == '\r' || *p == '\n'))
		*p-- = '\0';
}


/*---------------------------------
** main()
**---------------------------------
*/
int main( int argc, char * argv[])
{
	char buf[80];

	/*----------------------
	**  open device driver & Read config.
	*/
	if (argc > 1)
		Fd = open(argv[1], O_RDWR, 0x666);
	else
		Fd = open("/dev/oms0", O_RDWR, 0x666);
	if (Fd < 0)
	{
		printf("Error opening oms0 device driver \n");
		exit(0);
	}

	/* main loop for commands */
	Quit = FALSE;
	while( !Quit )
	{
		/* Get command */
		printf("Enter Command: ");
		fgets(buf, 80, stdin);

		if (buf[0])
			buf[strlen(buf)-1] = '\0';
		cmd_execute( buf );
	}

	close( Fd );

	return 0;
}

/*-------------------------------------------------------------
** cmd_execute() - Executes a command line instruction
**-------------------------------------------------------------
*/
int cmd_execute( cmd_buf )
   char * cmd_buf;
{
	char copy_buf[80];
	int error, found, i, soakcnt = 1;

	char * kw, * par;

	/* Special case for 'soak' prefix */
	if (!strncasecmp(cmd_buf, "soak ", 5))
	{
		cmd_buf += 5;
		if ((soakcnt = get_dec(&cmd_buf)) < 1)
		{
			printf ("Bad soak count\n");
			return ERR_INV_KW;
		}
	}

	/*
	**  Search table for keyword
	*/
	strxcpy( copy_buf, cmd_buf, sizeof(copy_buf));

	kw  = strtok(copy_buf, " ");
	if( (kw == NULL) || (*kw=='#') )
		return( ERR_NONE );

	if( (par = strtok( (char*)NULL, "")) != NULL )
		unpad( par, ' ' );

	/*
	**  Search table for keyword. If found call function.
	*/
	error = ERR_INV_KW;
	for(i=0, found = FALSE; i<num_parse_item && found == FALSE; )
	{
		if( !strcasecmp( kw, parse_table[i].kw ) )
		{
			struct timeval ts, te;
			double dt;

			gettimeofday(&ts, NULL);
			found = TRUE;
			error = (parse_table[i].pfi)(par, soakcnt);
			gettimeofday(&te, NULL);

			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			dt /= 1000000;
			if (soakcnt > 1)
				printf ("Time taken %.3f seconds\n", dt);
		}
		else
			i++;
	}
	/*
	**  Provide user feedback
	*/

	if(oms_strerror(error))
		printf("%s - Driver error: %s\n", cmd_buf, oms_strerror(error));
	else if (error != ERR_NONE)
		printf("%s - %s\n", cmd_buf, Error_string[abs(error)]);

	return( error );
}

/*--------------------------------------------------------------
**  cmd_search_selection_list() - search a list for item.
**     Return:  >= 0         =  the index of the matching item.
**              ERR_INV_RNG  =  Item is not in list.
**--------------------------------------------------------------
*/
int cmd_search_selection_list( item, list)
   char * item;    /* Item to locate in list            */
   char ** list;   /* This list must be NULL terminaled */
{
	int i;

	if( item == NULL )
		return ERR_INV_RNG;

	for( i=0; list[i] != NULL; i++)
	{
		if( !strcasecmp( item, list[i] ))
			return i;
	}
	return ERR_INV_RNG;
}
 
/********************************************************************
**                      COMMAND FUNCTIONS                          **
********************************************************************/

/*----------------------------------------------------------------------
**  do_beep() - output the beep character
**----------------------------------------------------------------------
*/
int do_beep( char *par, int cnt )
{
	printf("\007\n");
		return ERR_NONE;
}

/*----------------------------------------------------------------------
**  do_do() - execute a file.
**----------------------------------------------------------------------
*/
int do_do( char *par, int cnt )
{
	char filename[40];
	long reps;
	int  i;
	char *cptr;
	char buf[80];

	static FILE *fp = NULL;

	if( fp != NULL)
		return ERR_NOT_READY;

	if( (cptr = strtok( par, " ")) == NULL )
		return ERR_INV_FORMAT;
	strxcpy( filename, cptr, sizeof(filename));
	unpad( filename, ' ');

	if( -1 == my_atol( strtok(NULL, " "), &reps) )
		reps = 1;

	for( i=0; i<reps; i++)
	{
		int line;
		printf("Executing file %s. Reps %d of %ld\n", filename, i+1, reps+1);
		printf("========================================================\n");

		if( NULL == ( fp = fopen( filename, "r")) )
			return ERR_FILE_READ;

		line = 1;
	   	while( fgets( buf, sizeof(buf), fp) != NULL)
		{
			str_rws( buf, sizeof(buf));
			unpad( buf, ' ');
			printf("%d. Command=[%s]\n", line++, buf);

			cmd_execute( buf );
		}
		fclose( fp );
		fp = NULL;
	}
	return ERR_NONE;
}

/*----------------------------------------------------------------------
**  do_help() 
**----------------------------------------------------------------------
*/
int do_help( char *par, int cnt )
{
	int i;

	printf("\n"
	   "This program communicates with the OMS driver. It was written\n"
	   "as a test application for the oms device driver.\n" 
	   "Summary of commands:\n\n");

	for( i=0; i<num_parse_item; i++)
	{
		printf("   %s\n", parse_table[i].desc);
	}
	printf("\n");
	return ERR_NONE;
}

/*----------------------------------------------------------------------
**  do_cmd_ret() - send a command to the oms
**     Syntax: Puts  command
**----------------------------------------------------------------------
*/
int do_cmd_ret( char *par, int cnt )
{
	OMS_req_t req;
	int res = 0;

	memset(&req, 0, sizeof(req));

	req.cmdstr = par;
	req.cmdlen = strlen(par);
	res = ioctl(Fd, OMS_CMD_RET, &req);

	return res;
}


/*----------------------------------------------------------------------
**  do_dump() - 
**----------------------------------------------------------------------
*/
int do_dump( char *par, int cnt )
{
	int res;

	res = ioctl(Fd, OMS_DUMP);

	return res;
}


/*----------------------------------------------------------------------
**  do_timeout() - 
**----------------------------------------------------------------------
*/
int do_timeout( char *par, int cnt )
{
	int t = get_dec (&par);

	if (t < 0)
		return ERR_INV_FORMAT;

	timeout = t;

	return ERR_NONE;
}


/*----------------------------------------------------------------------
**  do_debug() - Makes the driver dump info to the console
**----------------------------------------------------------------------
*/
int do_debug( char *par, int cnt )
{
	int res;
	int level = get_dec (&par);

	if (level < 0)
		return ERR_INV_FORMAT;

	res = ioctl(Fd, OMS_DEBUG, level);

	return res;
}


/*----------------------------------------------------------------------
**  do_quit() - set Quit flag to TRUE.
**----------------------------------------------------------------------
*/
int do_quit( char *par, int cnt )
{
	Quit = TRUE;
	return 0;
}

/*----------------------------------------------------------------------
**  do_sleep() -  Pauses for sec seconds.
**     Syntax: sleep sec
**----------------------------------------------------------------------
*/
int do_sleep( char *par, int cnt )
{
	int i;
	long reps;
 
	if( -1 == my_atol( par, &reps) )
		return ERR_INV_FORMAT;
 
	printf("   Sleeping...");
	for( i=0; i< reps; i++ )
	{
		printf("z"); fflush(stdout);
		sleep(1);
	}
	printf("\n");
	return ERR_NONE;
}

/*----------------------------------------------------------------------
 *  do_reset()								
 *---------------------------------------------------------------------*/

int do_reset(char *par, int cnt)
{
	return ERR_NIF;
}

/*----------------------------------------------------------------------
 *  do_reg_handler()								
 *---------------------------------------------------------------------*/

int do_reg_handler(char *par, int cnt)
{
	return ERR_NIF;
}

/*----------------------------------------------------------------------
 *  do_read_ints()								
 *---------------------------------------------------------------------*/

int do_read_ints(char *par, int cnt)
{
	OMS_req_t req;
	int res = ERR_NONE;
	int axes;

	memset(&req, 0, sizeof(req));
	if ((axes = parse_axes(&par)) < 0)
		return ERR_BAD_AXES;

	req.axes = (u8)axes;

	if ((res = ioctl(Fd, OMS_READ_INTS, &req)))
		return res;

	printf ("         SRVUTZYX\n");
	printf ("Done:    %s\n", axes2str(req.f_done));
	printf ("Slip:    %s\n", axes2str(req.f_slip));
	printf ("Limit:   %s\n", axes2str(req.f_limit));
	printf ("CmdErr:  %s\n", axes2str(req.f_cmderr));

	return res;
}

/*----------------------------------------------------------------------
 *  do_ack_ints()								
 *---------------------------------------------------------------------*/

int do_ack_ints(char *par, int cnt)
{
	OMS_req_t req;
	int res = ERR_NONE;
	int axes;

	memset(&req, 0, sizeof(req));
	if ((axes = parse_axes(&par)) < 0)
		return ERR_BAD_AXES;

	req.axes = (u8)axes;

	if ((res = ioctl(Fd, OMS_ACK_INTS, &req)))
		return res;

	printf ("         SRVUTZYX\n");
	printf ("Done:    %s\n", axes2str(req.f_done));
	printf ("Slip:    %s\n", axes2str(req.f_slip));
	printf ("Limit:   %s\n", axes2str(req.f_limit));
	printf ("CmdErr:  %s\n", axes2str(req.f_cmderr));

	return res;
}

/*----------------------------------------------------------------------
 *  do_read_positions()								
 *---------------------------------------------------------------------*/

int do_read_positions(char *par, int cnt)
{
	OMS_req_t req;
	int res = ERR_NONE;
	int i;

	memset(&req, 0, sizeof(req));
	req.timeout = timeout;

	while (cnt && (res = ioctl(Fd, OMS_READ_POSITIONS, &req)) == 0)
		cnt--;

	if (res)
		return res;

	printf ("Positions:");
	for (i = 0; i < 8; i++)
	{
		if (i == 4)
			printf ("\n          ");
		printf (" %11ld", req.positions[i]);
	}
	putchar ('\n');
	return res;
}

/*----------------------------------------------------------------------
 *  do_cmd_blk()								
 *---------------------------------------------------------------------*/

int do_cmd_blk(char *par, int cnt)
{
	OMS_req_t req;
	int res = 0, axes;

	memset(&req, 0, sizeof(req));

	if ((axes = parse_axes(&par)) < 0)
		return ERR_BAD_AXES;
	req.axes   = axes;
	req.cmdstr = par;
	req.cmdlen = strlen(par);
	req.timeout = timeout;
	res = ioctl(Fd, OMS_CMD_BLK, &req);

	return res;
}

/*----------------------------------------------------------------------
 *  do_cmd_force()								
 *---------------------------------------------------------------------*/

int do_cmd_force(char *par, int cnt)
{
	OMS_req_t req;
	int res = 0, axes;

	memset(&req, 0, sizeof(req));

	if ((axes = parse_axes(&par)) < 0)
		return ERR_BAD_AXES;
	req.axes   = axes;
	req.cmdstr = par;
	req.cmdlen = strlen(par);
	req.timeout = timeout;
	res = ioctl(Fd, OMS_CMD_FORCE, &req);

	return res;
}

/*----------------------------------------------------------------------
 *  do_cmd_sig()								
 *---------------------------------------------------------------------*/

int do_cmd_sig(char *par, int cnt)
{
	return ERR_NIF;
}

/*----------------------------------------------------------------------
 *  do_cmd_sig_ack()								
 *---------------------------------------------------------------------*/

int do_cmd_sig_ack(char *par, int cnt)
{
	return ERR_NIF;
}

/*----------------------------------------------------------------------
 *  do_cmd_resp()								
 *---------------------------------------------------------------------*/

int do_cmd_resp(char *par, int cnt)
{
	OMS_req_t req;
	int res = 0;
	char rsp[80], *p;

	memset(&req, 0, sizeof(req));

	req.cmdstr = par;
	req.cmdlen = strlen(par);
	req.rspstr = rsp;
	req.rsplen = 80;
	req.timeout = timeout;
	while (cnt && (res = ioctl(Fd, OMS_CMD_RESP, &req)) == 0)
		cnt--;

	if (res)
		return res;

	p = rsp;
	printf ("Response: '");
	while (*p)
	{
		if (*p < ' ' || *p > 0x7e)
			printf ("<%02x>", (u8)*p++);
		else
			putchar(*p++);
	}
	printf ("'\n");
	return 0;
}

/*----------------------------------------------------------------------
 *  do_log_io()
 *---------------------------------------------------------------------*/

#ifndef OMS_LOG_IO

int do_log_io(char *par, int cnt)
{
	printf ("This command is only available if the driver and test\n");
	printf ("program are rebuild with OMS_LOG_IO defined in omslib.h\n");
	return ERR_NIF;
}

#else

char *
reg2name (unsigned char r)
{
	switch (r & 0x7f)
	{
	case 0x00: return "DATA";
	case 0x01: return "DONE";
	case 0x02: return "ICTL";
	case 0x03: return "STAT";
	default:   return "????";
	}
}

int do_log_io(char *par, int cnt)
{
	struct {
		int entries;
		int index;
		struct {
			unsigned char reg;
			unsigned char val;
		} log[16000];
	} buf;
	int res, i;
	FILE *fp;
	unsigned char r, v;
	char *s;

	res = ioctl(Fd, OMS_LOG_IO, &buf);
	if (res)
		return res;
	fp = fopen(par, "w");
	if (!fp)
		return errno;
	printf ("Entries: %d, index %d\n", buf.entries, buf.index);
	for (i = buf.index; i < buf.entries; i++)
	{
		r = buf.log[i].reg;
		v = buf.log[i].val;
		s = reg2name(r);
		fprintf (fp, "%c %s %02x %c\n", r & 0x80 ? 'W' : 'R', s, v,
			v >= ' ' && v < 0x7f ? v : '.');
	}
	for (i = 0; i < buf.index; i++)
	{
		r = buf.log[i].reg;
		v = buf.log[i].val;
		s = reg2name(r);
		fprintf (fp, "%c %s %02x %c\n", r & 0x80 ? 'W' : 'R', s, v,
			v >= ' ' && v < 0x7f ? v : '.');
	}
	fclose(fp);
	return 0;
}

#endif
