/***********************************************************************
 *  Device driver for Oregon Micro System's PC58, VME58, PC68
 *  Intelligent Motor Controllers.
 *  Linux 2.0 Operating System.
 * --------------------------------------------------------------------
 * This file is a userland library to simplify communications with
 * the driver.
 *
 *  richard@sleepie.demon.co.uk
 ***********************************************************************
 */

/*----------------------------------------------------------------------
 * Include files
 *---------------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <sys/ioctl.h>
#include <stdarg.h>

#include "omslib.h" 


/*----------------------------------------------------------------------
 * Local data
 *---------------------------------------------------------------------*/

static char *OMS_Error_Strings[] = {
	"No error!",
	"Invalid function",
	"Multiple axes specified",
	"Axis is busy",
	"Non-implemented function",
	"Timeout",
	"No axes specified",
	"Bad axis specification",
	"DONE interrupt pending",
	"Response overload",
	"String too long",
	"Bad position data from board",
	"Insufficient output buffer space",
	"Interrupt queued",
	"Bad response to status request",
	"Slip detected",
	"Limit detected",
	"Command error",
};

static unsigned char axis_mask[] = { 0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80 };
static char axis_name[] = "XYZTUVRS";


/*----------------------------------------------------------------------
 * Function: oms_strerror
 *
 * Map the supplied error code ot a string description.  Values which
 * fall in the range of valid OMS_ERR_ codes are translated to a local
 * string, otherwise a system error code is assumed.  Error 0 (success)
 * maps to a null pointer.
 *---------------------------------------------------------------------*/

char *
oms_strerror(int error)
{
	if( error >= OMS_ERR_BASE_VALUE &&
			error <= OMS_ERR_BASE_VALUE + OMS_ERR_RANGE)
		return OMS_Error_Strings[error - OMS_ERR_BASE_VALUE];
	else if (error)
		return strerror(error);
	else
		return NULL;
}


/*----------------------------------------------------------------------
 * Function: oms_log
 *
 * Log function for library and user code (if you wish).  This currently
 * logs to stderr, and doesn't have a concept of log levels.  Could
 * be enhanced.
 *---------------------------------------------------------------------*/

void
oms_log (char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

/*----------------------------------------------------------------------
 * Function: oms_log_str
 *
 * Private function for logging bad response strings from the driver.
 *---------------------------------------------------------------------*/

static void
oms_log_str (int axis, char *cmd, char *rsp, int len)
{
	char buf[80], *p = buf;

	if (len > 20)
		len = 20;
	*p++ = '\'';
	for (; len; len--, rsp++)
	{
		if (*rsp < ' ' || *rsp > 0x7e)
		{
			sprintf(p, "<%02x>", (unsigned char)*rsp);
			p += 4;
		}
		else
			*p++ = *rsp;
	}
	*p++ = '\'';
	oms_log ("%c: %s failed: %s\n", axis_name[axis], cmd, buf);
}


/*----------------------------------------------------------------------
 * Function: oms_cmd_blk
 *
 * Issue a blocking command to the driver.  Handles error returns and
 * interogates the driver/board so as to return a more specific error
 * code to the caller if possible.
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *	int  axis	Axis (0=X, 1=Y, etc)
 *	char *cmd	Command string to send
 *	int  timeout	Timeout (in seconds) to apply
 *
 * Returns:
 *	0 for success, or one of OMS_ERR_*
 *---------------------------------------------------------------------*/

int
oms_cmd_blk(int fd, int axis, char *cmd, int timeout)
{
	OMS_req_t req;
	int res;
	int slipped;

retry:
	slipped = 0;
	memset(&req, 0, sizeof(req));
	req.axes = axis_mask[axis];
	req.cmdstr = cmd;
	req.cmdlen = strlen(cmd);
	req.timeout = timeout;
	res = ioctl(fd, OMS_CMD_BLK, &req);
	if (!res)
		return res;
	else if (res == OMS_ERR_INT_QUEUED || res == OMS_ERR_TIMEOUT)
	{

		/* INT_QUEUED means there is a DONE, SLIP or LIMIT interrupt
		 * unacknowledged on this axis.  In the case of SLIP or
		 * LIMIT, this should be taken as an indication that
		 * slip or limit has occurred on some axis, but not
		 * necessarily this axis.
		 * TIMEOUT means we never got a DONE - perhaps we are
		 * in a LIMIT or SLIP state.
		 * Read and acknowledge all interrupts, then check
		 * this axes slip and limit status if necessary; if
		 * there aren't any errors relevant to this axis, just
		 * issue the command again if it was INT_QUEUED, or
		 * otherwsie return the TIMEOUT error.
		 */

		OMS_req_t ints;
		int timedout = (res == OMS_ERR_TIMEOUT);

		memset(&ints, 0, sizeof(ints));
		ints.axes = req.axes;
		res = ioctl(fd, OMS_ACK_INTS, &ints);
		if (res)
		{
			oms_log("%c: ACK_INTS failed: %s\n",
				axis_name[axis], oms_strerror(res));
			return res;
		}
		if (ints.f_done)
			oms_log("Ignoring queued DONE flag");
		if (ints.f_slip)
		{
			OMS_req_t slip;
			char resp[40];
			char reqs[40];

			memset(&slip, 0, sizeof(slip));
			slip.axes = req.axes;
			sprintf(reqs, "a%ceaam", axis_name[axis]);
			slip.cmdstr = reqs;
			slip.cmdlen = strlen(reqs);
			slip.rspstr = resp;
			slip.rsplen = 40;
			res = ioctl(fd, OMS_CMD_RESP, &slip);
			if (res)
			{
				oms_log("%c: Slip request failed: %s\n",
					axis_name[axis], oms_strerror(res));
				return res;
			}
			if (strlen(resp) != 12)
			{
				oms_log_str(axis, "EA", resp, slip.rsplen);
				return OMS_ERR_BAD_RESP;
			}
			if (resp[5] == 'S')
			{
				oms_log("%c: SLIP detected\n", axis_name[axis]);
				slipped = 1;
			}
		}
		if (ints.f_limit)
		{
			OMS_req_t limit;
			char resp[40];
			char reqs[40];

			memset(&limit, 0, sizeof(limit));
			limit.axes = req.axes;
			sprintf(reqs, "a%craam", axis_name[axis]);
			limit.cmdstr = reqs;
			limit.cmdlen = strlen(reqs);
			limit.rspstr = resp;
			limit.rsplen = 40;
			res = ioctl(fd, OMS_CMD_RESP, &limit);
			if (res)
			{
				oms_log("%c: Limit request failed: %s\n",
					axis_name[axis], oms_strerror(res));
				return res;
			}
			if (strlen(resp) != 10)
			{
				oms_log_str(axis, "RA", resp, limit.rsplen);
				return OMS_ERR_BAD_RESP;
			}
			if (resp[5] == 'L')
			{
				oms_log("%c: LIMIT detected\n",
						axis_name[axis]);
				return OMS_ERR_LIMIT;
			}
		}
		if (slipped)
			return OMS_ERR_SLIP;
		if (ints.f_cmderr)
		{
			oms_log("%c: Command Error\n", axis_name[axis]);
			return OMS_ERR_CMDERR;
		}
		if (timedout)
		{
			oms_log("%c: Timeout\n", axis_name[axis]);
			return OMS_ERR_TIMEOUT;
		}
		/* Still here?  Issue the command again. */
		goto retry;
	}
	else
	{
		oms_log("%c: cmd_blk failed: %s\n", axis_name[axis],
				oms_strerror(res));
		return res;
	}
}


/*----------------------------------------------------------------------
 * Function: oms_read_positions
 *
 * Reads the current position of all axes, and returns values in
 * supplied buffer.
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *	long *ppos	Pointer to array of 8 longs for results
 *
 * Returns:
 *	Axis position data in users buffer
 *	0 for success, or one of OMS_ERR_*
 *---------------------------------------------------------------------*/

int
oms_read_positions(int fd, long *ppos)
{
	OMS_req_t req;
	int res;

	memset(&req, 0, sizeof(req));
	res = ioctl(fd, OMS_READ_POSITIONS, &req);
	memcpy(ppos, req.positions, 8 * sizeof(long));
	if (res)
		oms_log("Read Positions failed: %s\n", oms_strerror(res));
	return res;
}

/*----------------------------------------------------------------------
 * Function: oms_cmd_resp
 *
 * Issue a response generating command and return the response.
 * XXX Maybe should allow a timeout to be specified (will currently
 * XXX use driver default).
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *	char *cmdstr	Command string to send
 *	char *rspstr	Buffer to hold response string
 *	int  maxlen	Length of response buffer
 *
 * Returns:
 *	Null terminated response string in users buffer
 *	0 for success, or one of OMS_ERR_*
 *---------------------------------------------------------------------*/

int
oms_cmd_resp (int fd, char *cmdstr, char *rspstr, int maxlen)
{
	OMS_req_t req;
	int res;

	memset(&req, 0, sizeof(req));

	req.cmdstr = cmdstr;
	req.cmdlen = strlen(cmdstr);
	req.rspstr = rspstr;
	req.rsplen = maxlen;

	res = ioctl(fd, OMS_CMD_RESP, &req);

	if (res)
		oms_log("oms_cmd_resp failed: %s\n", oms_strerror(res));
	return res;
}


/*----------------------------------------------------------------------
 * Function: oms_stop_axis
 *
 * Stop the specified axis
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *	int  axis	Axis (0=X, 1=Y, etc)
 *	int  timeout	Timeout (in seconds) to apply
 *
 * Returns:
 *	0 for success, or one of OMS_ERR_*
 *---------------------------------------------------------------------*/

int
oms_stop_axis(int fd, int axis, int timeout)
{
	char cmd[40];
	OMS_req_t req;
	int res;

	sprintf(cmd, "a%cstwqidam", axis_name[axis]);
	memset(&req, 0, sizeof(req));
	req.axes = axis_mask[axis];
	req.cmdstr = cmd;
	req.cmdlen = strlen(cmd);
	req.timeout = timeout;
	res = ioctl(fd, OMS_CMD_FORCE, &req);
	if (res)
		oms_log("oms_stop_axis failed: %s\n", oms_strerror(res));
	return res;
}

/*----------------------------------------------------------------------
 * Function: oms_check_limit
 *
 * Check state of limit switch for specified axis.
 * XXX Should the return indicate which limit switch triggered?
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *	int  axis	Axis (0=X, 1=Y, etc)
 *
 * Returns:
 *	0 for success, OMS_ERR_LIMIT if limit detected, or one of OMS_ERR_*
 *
 *---------------------------------------------------------------------*/

int
oms_check_limit(int fd, int axis)
{
	char cmd[40];
	char rsp[40];
	int res;

	sprintf(cmd, "a%craam", axis_name[axis]);
	res = oms_cmd_resp(fd, cmd, rsp, 40);

	if (res)
	{
		oms_log("oms_check_limit failed: %s\n", oms_strerror(res));
		return res;
	}
	if (strlen(rsp) != 10)
	{
		oms_log_str(axis, "RA", rsp, strlen(rsp));
		return OMS_ERR_BAD_RESP;
	}
	if (rsp[5] == 'L')
		return OMS_ERR_LIMIT;
	else
		return 0;
}

/*----------------------------------------------------------------------
 * Function: oms_axis_name
 *
 * Maps an axis ID (0 - 7) to a character name for the axis (0 => X,
 * 1 => Y, etc).
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *
 * Returns:
 *	'?' for bad input, or one of 'X', 'Y', etc
 *
 *---------------------------------------------------------------------*/

char
oms_axis_name(int axis)
{
	if (axis < 0 || axis > 8)
		return '?';
	else
		return axis_name[axis];
}

/*----------------------------------------------------------------------
 * Function: oms_ack_all_ints
 *
 * Acknowledge all outstanding interrupts on the specified axis.  Useful
 * when tidying up after an error.
 *
 * Parameters:
 *	int  fd		Open file descriptor for board
 *	int  axis	Axis (0=X, 1=Y, etc)
 *
 * Returns:
 *	0 for success, one of OMS_ERR_*
 *
 *---------------------------------------------------------------------*/

int
oms_ack_all_ints(int fd, int axis)
{
	OMS_req_t ints;
	int res;

	memset(&ints, 0, sizeof(ints));
	ints.axes = axis_mask[axis];
	res = ioctl(fd, OMS_ACK_INTS, &ints);
	if (res)
	{
		oms_log("%c: ACK_INTS failed: %s\n",
			axis_name[axis], oms_strerror(res));
		return res;
	}
	if (ints.f_done)
		oms_log("%c: Ignoring DONE interrupt\n", axis_name[axis]);
	if (ints.f_slip)
		oms_log("%c: Ignoring SLIP interrupt\n", axis_name[axis]);
	if (ints.f_limit)
		oms_log("%c: Ignoring LIMIT interrupt\n", axis_name[axis]);
	if (ints.f_cmderr)
		oms_log("%c: Ignoring CMDERR interrupt\n", axis_name[axis]);

	return res;
}

/*----------------------------------------------------------------------
 * End of file
 *---------------------------------------------------------------------*/

