/***********************************************************************
 *  Device driver private header file for PC58/VME58/PC68
 * --------------------------------------------------------------------
 *  This code is based on a driver for the PC58, written by
 *     Tony Denault
 *     Institute for Astronomy, University of Hawaii
 *
 *  Modified for VME58 and PC68 by richard@sleepie.demon.co.uk
 ***********************************************************************
 */

#ifndef _LINUX_OMS_H
#define _LINUX_OMS_H

#include "omslib.h"	/* Get public structures and definitions	*/

/* 'me' is used as the driver name in kernel error messages; this will
 * probably change in the future.
 */

#ifdef PC58
#define me		"PC58"
#elif defined(VME58)
#define me		"VME58"
#elif defined(PC68)
#define me		"PC68"
#else
#error  "OMS board type undefined"
#endif

/*----------------------------------------------------------------------
 * OMS_dram_t - how the Dual ported ram is organized for VME58
 *---------------------------------------------------------------------*/

/* NOTE: For PC68 we emulate this 4K memory segment in the driver local
 * workspace, so the i/o quue handling can be common between all boards.
 */

/* Input and output queue size in the shared memory.  if this value
 * changes from 256, then the drvier code which accesses the queues
 * will also have to change.
 */

#define OMS_BUF_SIZE     256

/* First the structure of the axis parameters				*/

/* XXX This looks somewhat different to the manual - perhaps it
 * XXX maps to the ISA card???
 */

typedef struct {
	long  encoder_pos;
	long  command_pos;
	long  command_vel;
	long  accel;
	long  max_vel;
	long  base_vel;
	float filter_gain;
	long  filter_pole;
	long  filter_zero;
} OMS_dram_axis_t, *OMS_dram_axis_ptr;

/* Wrap axis structure in a union to pad to 128 entries			*/

union OMS_axis_u 
{ 
	char status_block[128];
	OMS_dram_axis_t axis;
};

/* Now the structure of the 4096 byte shared memory area		*/

/* Note: 'input' and 'output' are relative to the host, not the OMS
 *       board.  These confusing names are chosen to match the OMS
 *	 manual.
 */

#define vu8	volatile unsigned char
#define v16	volatile short
#define v32	volatile long

typedef struct {
	v16 input_put_index;		/* End of inbuf[]. Where OMS
					 * writes new char		*/
	v16 output_get_index;		/* Start of outbuf[]. Where
					 * OMS reads next char		*/
	v16 inbuf[OMS_BUF_SIZE];	/* Input Buffer			*/

	short unused_1[254];

	volatile union OMS_axis_u motor_status[8];	/* Axis params.	*/

	v16 output_put_index;		/* End of outbuf[]. Where host
					 * writes new char		*/
	v16 input_get_index;		/* Start of inbuf[]. where host
					 * reads next char		*/
	v16 outbuf[OMS_BUF_SIZE];	/* Output Buffer		*/

	short unused_2[706];

	v32 mailbox;			/* Mailbox for direct cmnd i/p	*/

	short unused_3[42];

	/* Control/status registers, as they appear on a VME58 board	*/

	char pad0; vu8 cntl_reg;	/* Control register		*/
	char pad1; vu8 status_reg;	/* Status register		*/
	char pad2; vu8 dio_lo_reg;	/* User I/O bits 0 to 7		*/
	char pad3; vu8 slip_reg;	/* Slip flags register		*/
	char pad4; vu8 done_reg;	/* Done flags register		*/
	char pad5; vu8 dio_hi_reg;	/* User I/O bits 8 to 13	*/
	char pad6; vu8 limit_reg;	/* Limit flags register		*/
	char pad7; vu8 home_reg;	/* Home flags register		*/
	char pad8; vu8 intvec_reg;	/* Interrupt vector register	*/

	short unused_4[7];

} OMS_dram_t, *OMS_dram_ptr;

#ifdef PC68
/*----------------------------------------------------------------------
 * Number of I/O ports used in the io memory map of Intel processors
 *---------------------------------------------------------------------*/

#define OMS_NUM_IOPORTS		4

#endif

#if defined(VME58) || defined(PC58)
/*----------------------------------------------------------------------
 * Register names and offsets; these are used for PC58 and VME58
 *---------------------------------------------------------------------*/

#define OMS_CNTL_REG            0	/* Control Register   - Read/Write */

#define OMS_STATUS_REG          1	/* Status Register     - Read only */
#define OMS_DIO_LO_REG          2	/* User DIO bit 0-7    - Read only */
#define OMS_SLIP_REG            3	/* Slip Flags          - Read only */
#define OMS_DONE_REG            4	/* Done Flags          - Read only */
#define OMS_DIO_HI_REG          5	/* User DIO bit 8-13   - Read only */
#define OMS_LIMIT_REG           6	/* Limit Switch Status - Read only */
#define OMS_HOME_REG            7	/* Home Switch Status  - Read only */
#define OMS_INTVEC_REG		8	/* Interrupt vector (VME58 only)   */
#define OMS_MAILBOX		0xff8	/* Mailbox register (VME58 only)   */

/* Write only registers, used to locate dual port RAM on PC58 boards	   */

#define OMS_DPRAM_LO_REG        6	/* Dual Port RAM Addr (A15-A12)    */
#define OMS_DPRAM_HI_REG        7	/* Dual Port RAM Addr (A23-A16)    */

#elif defined(PC68)
/*----------------------------------------------------------------------
 * Register names and offsets; these are used for PC68
 *---------------------------------------------------------------------*/

#define OMS_DATA_REG            0	/* Data register       - read/write */
#define OMS_DONE_REG            1	/* Done Flags          - Read only  */
#define OMS_CNTL_REG            2	/* Control Register    - Read/Write */
#define OMS_STATUS_REG          3	/* Status Register     - Read only  */

#endif
/*----------------------------------------------------------------------
 * Bit field definitions for the control register
 *---------------------------------------------------------------------*/

#ifdef VME58

#define OMS_CNTL_REG_data_area_update_request   0x01    
#define OMS_CNTL_REG_unused_1                   0x02    
#define OMS_CNTL_REG_encoder_slip_int_enb       0x04    
#define OMS_CNTL_REG_limit_register_int_enb     0x08    
#define OMS_CNTL_REG_done_reg_int_enb           0x10
#define OMS_CNTL_REG_int_request                0x20
#define OMS_CNTL_REG_io_0_and_1_int_enb         0x40
#define OMS_CNTL_REG_int_request_enb            0x80

#elif defined(PC68)

#define OMS_CNTL_REG_done_reg_int_enb           0x10
#define OMS_CNTL_REG_rxbuf_full_enb		0x20
#define OMS_CNTL_REG_txbuf_empty_enb		0x40
#define OMS_CNTL_REG_int_request_enb            0x80

#endif

/*----------------------------------------------------------------------
 * Bit field definitions for the status register
 *---------------------------------------------------------------------*/

#ifdef VME58

#define OMS_STATUS_REG_command_error            0x01    
#define OMS_STATUS_REG_initialized              0x02    
#define OMS_STATUS_REG_encoder_slip             0x04    
#define OMS_STATUS_REG_overtravel               0x08    
#define OMS_STATUS_REG_done                     0x10
#define OMS_STATUS_REG_int_req_status_to        0x20
#define OMS_STATUS_REG_dir_int_req_from         0x40	/* Unused on VME58 */
#define OMS_STATUS_REG_int_req_status           0x80

#elif defined(PC68)

#define OMS_STATUS_REG_command_error            0x01    
#define OMS_STATUS_REG_initialized              0x02    
#define OMS_STATUS_REG_encoder_slip             0x04    
#define OMS_STATUS_REG_overtravel               0x08    
#define OMS_STATUS_REG_done                     0x10
#define OMS_STATUS_REG_rxbuf_full		0x20
#define OMS_STATUS_REG_txbuf_empty		0x40
#define OMS_STATUS_REG_int_req_status           0x80

#endif

/*----------------------------------------------------------------------
 * OMS_table_t structure - one of these exists for each board in the
 * system.
 *---------------------------------------------------------------------*/

/* Values for OMS_table_t.busy[] - anything non-zero basically means
 * there is an interrupt pending on the axis.
 */

#define OMS_BUSY_NOT	 0	/* No interrupt pending on the axis	*/
#define	OMS_BUSY_BLK	 1	/* Axis is waiting for an interrupt to
				 * terminate a blocking command		*/
#define OMS_BUSY_SIG	 2	/* Axis is busy waiting to signal on
				 * the next DONE interrupt		*/
#define OMS_BUSY_SIG_ACK 3	/* Axis will signal and clear the DONE
				 * flag on the next interrupt		*/
#define OMS_BUSY_PENDING 4	/* Axis has errored in some way (eg.
				 * timeout) and DONE still expected.	*/

#define MAX_BOARDS	4	/* Max 4 OMS boards in a system		*/
#define MAX_RESP	8	/* Max number of response generating
				 * commands outstanding at a time	*/
#define MAX_STR		250	/* Maximum command length allowed	*/

typedef struct {
	u32 minor;		/* Minor number for this board		*/
	u32 base;		/* base I/O address of board		*/
	OMS_dram_ptr address;	/* Shared memory address for board	*/
	u8 vector;		/* Interrupt vector allocated to board	*/

	u8 i_done;		/* Unreported Done interrupts		*/
	u8 i_slip;		/* Unreported Slip interrupts		*/
	u8 i_limit;		/* Unreported Limit interrupts		*/
	u8 i_cmderr;		/* Unreported CmdErr interrupts		*/

	struct wait_queue *resp_waitq;	/* Wait queue for response cmds	*/
	struct wait_queue *blk_waitq;	/* Wait queue for OMS_CMD_BLK	*/
	u8 busy[8];		/* Interrupt pending on this axis?	*/
	u32 seq_queue[MAX_RESP];/* Sequence of queued CMD_RESP requests	*/
	u32 aarp_seq;		/* Sequence of last AA RP command	*/
	u32 seq_no;		/* Last sequence number used		*/
	char buf[MAX_STR+1];	/* Buffer for string i/o to board	*/
	long positions[8];	/* Latest position data			*/
#ifdef PC68
	OMS_dram_t pc68_dram;	/* Simulated shared memory for PC68	*/
#endif
} OMS_table_t, *OMS_table_ptr;

#endif /*  _LINUX_OMS_H  */

