/***********************************************************************
 *  Device driver for Oregon Micro System's PC58, VME58, PC68
 *  Intelligent Motor Controllers.
 *  Linux 2.0 Operating System.
 * --------------------------------------------------------------------
 *  This code is based on a driver for the PC58, written by
 *     Tony Denault
 *     Institute for Astronomy, University of Hawaii
 *
 *  Modified for VME58 and PC68 by richard@sleepie.demon.co.uk
 ***********************************************************************
 */

/* TBD:
 *
 * Probe for boards on startup.
 *
 * Handle error recovery better in probe code.
 *
 * Implement the OMS_RESET as a means to recover from errors, and wake
 * all threads when OMS_RESET received.
 *
 * Leave axes busy following timeout, until interrupt or reset.
 *
 * Implement signalling functions.
 *
 * Implement async timeouts on signalling functions.
 *
 * Make the semaphore per-board, not per-driver.
 *
 * Decide what to do about 'command queue full' and 'command bigger
 * than buffer'
 *
 * Enabling certain VME IRQ singals in to the MVME16x board should not be
 * a function of this driver.  That should be decided at a system level
 * and controlled by some higher level board/system configuration module.
 *
 * Various minor things - see XXX in code.
 *
 * Lots of other things, I am sure.
 */


/* Reading position:
 *  The PC58 and VME58 boards provide 4K of shared memory interface,
 * while the PC68 is accessed via I/O registers only.  Current position
 * information can be read via command request interface, or via the
 * shared memory.  The driver uses shared memory for VME58, and the
 * command/response interface for PC68.
 */

#if defined(VME58) || defined(PC58)
#define GET_POSITIONS_FROM_SHARED_MEMORY
#endif

/*----------------------------------------------------------------------
 * Include files
 *---------------------------------------------------------------------*/

#define MODULE			/* Always a module */

#include <linux/module.h>
#include <linux/version.h>

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/malloc.h>
#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>

#include "omslib.h"		/* Driver public data			*/
#include "oms.h"		/* Driver private data			*/


/*----------------------------------------------------------------------
 * Macros
 *---------------------------------------------------------------------*/

/* GETLONG_D16() reads a 32 bit value from shared memory as two 16 bit
 * accesses.  This is necessary for the VME58, which does not support
 * D32 access to short I/O space.
 */

#define GETLONG_D16(ptr) (*(u16 *)(ptr) << 16 | *((u16 *)(ptr)+1))


/*----------------------------------------------------------------------
 * Prototypes for forward referenced functions
 *---------------------------------------------------------------------*/

static int  oms_configure (OMS_table_ptr tp);
static int  oms_open (struct inode * inode, struct file * file);
static void oms_release (struct inode * inode, struct file * file);
static int  oms_ioctl (struct inode *inode, struct file *file,
		unsigned int cmd, unsigned long arg);
static void oms_sleep (int csec);
static int  oms_putusr (OMS_table_ptr tp, OMS_req_ptr rp);
static int  oms_puts (OMS_table_ptr tp);
static int  oms_getusr (OMS_table_ptr tp, OMS_req_ptr rp);
static int  oms_gets (OMS_table_ptr tp);

/*----------------------------------------------------------------------
 * Local variables
 *---------------------------------------------------------------------*/

/* The driver creates an array, one entry per board, for its local
 * data at startup time.  oms_table holds the base of that array
 */

static OMS_table_ptr oms_table;

/*
 * NOTE:  The board does not currently probe for boards, but expects
 *        parameters to be passed to it at startup to define the
 *        available boards.  That scheme will change in the future, to
 *        one where the driver probes for boards at a set of commonly
 *        used addresses or ioports.
 *
 * Module parameters.  oms_major is the device major number to use, and
 * oms_params is the base, address, and vector values to use for each
 * configured board.  For example,
 *
 *   insmod vme58.o oms_params=0,0xffff1000,0x8d,0,0xffff2000,0x8e
 *
 * would define two boards.  The base value is the i/o register base
 * used on PC68 boards.
 *
 * Defaults are currently major 26, and one board at 0xffff1000, vector
 * 0x8d.  For PC68 we default to base 0x300, IRQ 5.
 */

static int oms_major  = 26;		/* Major device number to use	*/
static char *oms_params = NULL;		/* pointer to any module
					 * parameters specified		*/

static int oms_boards = 1;		/* Default number of boards	*/
#ifdef VME58
static int oms_level  = 2;		/* Default VMEBUS IRQ level	*/
#endif

/* Structure used to hold results of parsing oms_params, with default
 * values of one VME58 board at 0xffff1000
 */

static struct {
	u32 base;
	u32 address;
	u32 vector;
} oms_board_defs[MAX_BOARDS] = {
#ifdef VME58
	{ 0, 0xffff1000, 0x8d }
#elif defined(PC68)
	{ 0x230, 0, 5 }
#endif
};

/* The driver supports various debug output levels; the higher the value
 * of oms_debug, the more output is produced.  The debug level can be
 * adjusted at runtime with an appropriate ioctl call.
 */

int oms_debug = 0;			/* Initial default debug level	*/

/* Define a semaphore to prevent multiple processes interleaving their
 * accesses to the shared memory area when accessing the i/o queues.
 */

/* XXX If we need this, I guess it needs to be per-board */

static struct semaphore Sem = MUTEX;


/*----------------------------------------------------------------------
 * Define device driver entries points
 *---------------------------------------------------------------------*/

struct file_operations oms_fops =
{
	NULL,			/* _seek		*/
	NULL,			/* _read		*/
	NULL,			/* _write		*/
	NULL,			/* _readdir		*/
	NULL,			/* _select		*/
	oms_ioctl,		/* _ioctl		*/
	NULL,			/* _mmap		*/
	oms_open,		/* _open		*/
	oms_release,		/* _release		*/
};


#ifdef OMS_LOG_IO

/*----------------------------------------------------------------------
 * Function: log_io
 *
 * This is only available if OMS_LOG_IO is defined in omslib.h.  It
 * is intended as a debugging aid.
 *---------------------------------------------------------------------*/

#define LOG_IO_ENTRIES	16000

typedef struct {
	u8 reg;
	u8 val;
} log_io_t;

static log_io_t *log_io_buf = NULL;
static int log_io_index = 0;

static void
log_io (u8 reg, u8 value)
{
	unsigned long flags;

	save_flags(flags);
	cli();
	log_io_buf[log_io_index].reg = reg;
	log_io_buf[log_io_index].val = value;
	if (++log_io_index == LOG_IO_ENTRIES)
		log_io_index = 0;
	restore_flags(flags);
}

#endif

/*----------------------------------------------------------------------
 * Function: read_oms_reg()
 *
 * Read and return the contents of the specified register from the board
 *---------------------------------------------------------------------*/

static u8
read_oms_reg(OMS_table_ptr tp, int reg)
{
#if defined(PC58) || defined(PC68)
	u8 v = inb_p (tp->base+reg);

	if (oms_debug > 4)
		printk("oms: 0x%04x ==> 0x%02x\n", tp->base + reg, v);
#elif defined(VME58)
	unsigned char v;
	OMS_dram_ptr dp = tp->address;

	v = *(&dp->cntl_reg + reg * 2);
	if (oms_debug > 4)
		printk("oms: 0x%p ==> 0x%02x\n", &dp->cntl_reg + reg * 2, v);
#endif
#ifdef OMS_LOG_IO
	log_io(reg, v);
#endif
	return v;
}

/*----------------------------------------------------------------------
 * Function: write_oms_reg()
 *
 * Write to a register in the board.
 *---------------------------------------------------------------------*/

static void
write_oms_reg(OMS_table_ptr tp, int reg, unsigned char value)
{
#if defined(PC58) || defined(PC68)
	if (oms_debug > 4)
		printk("oms: 0x%02x ==> 0x%04x\n", value, tp->base + reg);
	outb_p (value, tp->base + reg);
#elif defined(VME58)
	OMS_dram_ptr dp = tp->address;

	if (oms_debug > 4)
		printk("oms: 0x%02x ==> 0x%p\n", value, &dp->cntl_reg + reg*2);
	*(&dp->cntl_reg + reg * 2) = value;
#endif
#ifdef OMS_LOG_IO
	log_io(reg | 0x80, value);
#endif
}

/*----------------------------------------------------------------------
 * Function: oms_sleep()
 *
 * Causes the current process to sleep for the specified number of
 * ticks (regardless of tick period!!).  This is mainly used when
 * waiting for responses to OMS_READ_POSITIONS or OMS_CMD_RESP.
 *---------------------------------------------------------------------*/

static void
oms_sleep (int csec)
{
	current->state = TASK_INTERRUPTIBLE;
	current->timeout = jiffies + csec;
	schedule();
}

/*----------------------------------------------------------------------
 * Function: oms_puts()
 *
 * Write a string from the drivers board specific local buffer to
 * the board.  This currently only handles commands which will fit
 * entirely within the boards command buffer, and errors if there is
 * insufficient space for the whole command.
 *
 * For PC68 we simply add the string to the output buffer and enable
 * transmit buffer empty interrupts.
 *
 * XXX May have to sleep while we wait for space in the future, and
 * XXX may have to cope with commands longer that 256 chars....
 *---------------------------------------------------------------------*/

static int
oms_puts (OMS_table_ptr tp)
{
	int inx, space;
	char *s = tp->buf;

	OMS_dram_ptr dp = tp->address;

	space = (u8)(dp->output_get_index - dp->output_put_index) - 2;
	if (space < strlen(tp->buf))
		return OMS_ERR_NO_SPACE;

	inx = dp->output_put_index;  /* get start of output buffer */

	while (*s)
	{
		dp->outbuf[inx] = *s++;    /* write char to buffer */

		if (++inx >= OMS_BUF_SIZE) /* inc. index to next location */
			inx = 0;
#if 0
		/* if the buffer is full, just wait (but not too long) */
		cnt = 0;
		while (inx == dp->output_get_index)
		{
			oms_sleep (1);
			if (++cnt > 5)
				return OMS_ERR_TIMEOUT;
		}
		/* update output_put_index (notifies OMS a char is ready) */
		dp->output_put_index = inx;
#endif
	}
	dp->output_put_index = inx;
#ifdef PC68
	/* Ensure transmit interrupts are enabled */
	{
		unsigned long flags;

		save_flags(flags);
		cli();
		write_oms_reg (tp, OMS_CNTL_REG,
			read_oms_reg(tp, OMS_CNTL_REG) |
				OMS_CNTL_REG_txbuf_empty_enb);
		restore_flags(flags);
	}
#endif

	return OMS_ERR_NONE;
}


/*----------------------------------------------------------------------
 * Function: oms_putusr
 *
 * Write a command string from the user supplied buffer to the boards
 * command queue.  Will error if there is insufficient space for the
 * command.
 *---------------------------------------------------------------------*/

static int
oms_putusr (OMS_table_ptr tp, OMS_req_ptr rp)
{
	int res;

	/* Copy the command string over from user-land */

	res = verify_area(VERIFY_READ, rp->cmdstr, rp->cmdlen);
	if (res)
		return res;
	memcpy_fromfs (tp->buf, rp->cmdstr, rp->cmdlen);
	tp->buf[rp->cmdlen] = '\0';

	return oms_puts (tp);
}


/*----------------------------------------------------------------------
 * Function: oms_gets
 *
 * Get a string from the boards response buffer, and store it in the
 * board local data private buffer.  This should only be called when
 * a complete command string is available (see oms_resp_ready()).
 *
 * XXX Jon Branch at OMS advises 'reading the characters one at a time'
 * XXX here, which I think means we should update the index in increments
 * XXX or 1.  He hasn't explained why yet, so I havn't changed the
 * XXX code.
 *---------------------------------------------------------------------*/

static int
oms_gets (OMS_table_ptr tp)
{
	int inx, len;
	OMS_dram_ptr dp = tp->address;
	char *p = tp->buf;

	len = 0; 
	inx = dp->input_get_index;  /* get index of next char to read */

	while (1)
	{
		if (inx == dp->input_put_index)   /* queue is empty, return */
		{
			/* terminate string */
			*p++ = '\0';
			return OMS_ERR_NONE;
		}

		/* read char into buffer (if we can), and increment index */
		if (len < MAX_STR)
		{
			*p++ = dp->inbuf[inx];
			len++;
		}

		if (++inx >= OMS_BUF_SIZE) /* inc. index to next location */
			inx = 0;

		dp->input_get_index = inx;  /* keep shared memory up-to-date */
	}
}


/*----------------------------------------------------------------------
 * Function: oms_getusr()
 *
 * Read a response from the board in to the users supplied response
 * string buffer.
 *---------------------------------------------------------------------*/

static int
oms_getusr (OMS_table_ptr tp, OMS_req_ptr rp)
{
	int res;

	oms_gets (tp);
	rp->rsplen = strlen(tp->buf);

	res = verify_area(VERIFY_WRITE, rp->rspstr, rp->rsplen+1);

	if (res)
		return res;

	memcpy_tofs (rp->rspstr, tp->buf, rp->rsplen + 1);

	return res;
}


/*----------------------------------------------------------------------
 * Function: oms_resp_ready()
 *
 * This function is used when handling response generating commands.
 * It analyses the available string in the shared memory buffer an
 * returns non-zero if the string is complete.
 *
 * The extensive use of (u8) casts is to take advantage of the fact
 * that the queues are 256 entries long, and indices can be automatically
 * wrapped by keping them in 8 bit data types.  Beware that
 * dp->inbuf[index+1] when index is a u8 of value 255 will acesss
 * element 256, not zero!!
 *
 * Responses always have a matching pre- and post-fix of a single line
 * feed, followed by one or two carriage return characters.
 *---------------------------------------------------------------------*/

static int
oms_resp_ready(OMS_table_ptr tp)
{
	OMS_dram_ptr dp = tp->address;
	int cnt = (dp->input_put_index - dp->input_get_index) & 0xff;
	u8 index = (u8)dp->input_get_index;
	int two_crs;

	if (cnt < 4)
		return 0;
	if (dp->inbuf[index] != '\n' || dp->inbuf[(u8)(index+1)] != '\r')
		return 0;
	two_crs = (dp->inbuf[(u8)(index+2)] == '\r');
	index += cnt;
	if (dp->inbuf[(u8)(--index)] != '\r')
		return 0;
	if (two_crs && dp->inbuf[(u8)(--index)] != '\r')
		return 0;
	if (dp->inbuf[(u8)(--index)] != '\n')
		return 0;
	return 1;
}


/*----------------------------------------------------------------------
 * Function: parse_aarp()
 *
 * This function parses the response string from an multi-axes Read
 * Position request, and updates the latest position info in the
 * boards local data.  The response string is expected to have been
 * copied from the board to the board specific local buffer before
 * calling this.  Returns 0 if parsed successfully.
 *---------------------------------------------------------------------*/

#ifndef GET_POSITIONS_FROM_SHARED_MEMORY
static int
parse_aarp (OMS_table_ptr tp)
{
	int i;
	char *p = tp->buf;
	long n;
	char sign;

	for (i = 0; i < 8; i++)
		tp->positions[i] = 0x7fffffff;

	if (p[0] != '\n' || p[1] != '\r')
		return OMS_ERR_POS_DATA;
	p += 2;
	for (i = 0; i < 8; i++)
	{
		n = 0;
		if (*p == '-')
			sign = *p++;
		else
			sign = '+';
		if (*p < '0' || *p > '9')
			break;
		while (*p >= '0' && *p <= '9')
			n = n * 10 + *p++ - '0';
		if (sign == '-')
			n = 0 - n;
		tp->positions[i] = n;
		if (*p != ',')
			break;
		p++;
	}
	if (i < 2 || p[0] != '\n' || p[1] != '\r')
		return OMS_ERR_POS_DATA;
	return OMS_ERR_NONE;
}
#endif


/*----------------------------------------------------------------------
 * Function: set_seq()
 *
 * Response generating requests to the driver are allocated a sequence
 * number to ensure that they are serviced fairly.  To understand why
 * this is necessary, consider the case of two processes both making
 * requests as quickly as possible.  Process A issues a request, the
 * driver forwards the request to the board and puts A to sleep for
 * 10ms.  Now process B makes a request; the driver realises the board
 * is busy, and adds B to a wait queue.  The 10ms expires, the driver
 * marks B as really to run, and returns to A.  Process A runs and
 * issues a new request; the board is not busy, as B has not had chance
 * to run yet, so the driver sends the request to the board and puts
 * A to sleep for 10ms.  B now gets a chance to run, sees the board
 * still busy, and gets put back on the wait queue.  So far as the
 * kernel scheduler is concerned, both processes run every 10ms, so
 * it is behaving fairly.
 *
 * This is overcome by allocating a sequence number to each request,
 * and keeping a list of sequence numbers of outstanding requests.  Then
 * whenever the driver runs on behalf of a process, it checks this
 * queue and will add itself to a wait queue regardless of whether
 * the board is busy, if there is an entry in the list with a lower
 * value than its own sequence number.
 *
 * Whenever a process request completes (successfully, or due to a
 * timeout, ctrl-C, or whatever), it clears its entry in the list and
 * causes the kernel to re-scan the wait queue.
 *
 * This particular function locates a free entry in the list, sets
 * it to the next available sequence number, and returns a pointer
 * to it.  An added twist is that the sequence number counter could
 * overflow within a few months of continuous running, so the code
 * checks for that and adjusts the sequence numbers downwards.  Because
 * of that, users of the list must always reference their sequence
 * number via the returned pointer, and not keep the value locally.
 * The function returns NULL if the list is full, indicating too many
 * respone generating commands are queued.
 *
 * This sequence number scheme also helps with processing Read Positions
 * requests, in that we record the sequence number when we get the
 * position data, and then any request whose sequence number is lower
 * than the sequence number of the position data is current, and can
 * make use of it.
 *---------------------------------------------------------------------*/

static u32 *
set_seq(OMS_table_ptr tp)
{
	int i;

	if (tp->seq_no > 0xf0000000)
	{
		/* Recalibrate to avoid wrap-around */
		for (i = 0; i < MAX_RESP; i++)
			if (tp->seq_queue[i])
				tp->seq_queue[i] -= 0xe0000000;
		tp->seq_no -= 0xe0000000;
		if (tp->aarp_seq > 0xe0000000)
			tp->aarp_seq -= 0xe0000000;
		else
			tp->aarp_seq = 0;
	}

	for (i = 0; i < MAX_RESP; i++)
	{
		if (tp->seq_queue[i] == 0)
		{
			tp->seq_queue[i] = ++tp->seq_no;
			return tp->seq_queue + i;
		}
	}
	return NULL;
}

/*----------------------------------------------------------------------
 * Function: youngest_seq()
 *
 * Scan the list of sequence numbers representing outstanding response
 * generating requests, and return non-zero if the supplied sequence
 * number is the youngest available (ie. the caller should be the next
 * process to issue a request to the board).
 *---------------------------------------------------------------------*/

static int
youngest_seq(OMS_table_ptr tp, u32 *seq_ptr)
{
	int i;

	for (i = 0; i < MAX_RESP; i++)
	{
		if (tp->seq_queue[i] && tp->seq_queue[i] < *seq_ptr)
			return 0;
	}
	return 1;
}


/*----------------------------------------------------------------------
 * Function: oms_dump()
 *
 * This is a simple debugging aid; it dumps various potentialy useful
 * driver data to the console in response to an OMS_DUMP ioctl request.
 *---------------------------------------------------------------------*/

static void
oms_dump (void)
{
	int i,j;
	OMS_table_ptr tp;

	for (i = 0, tp = oms_table; i < oms_boards; i++, tp++)
	{
		printk ("OMS Driver dump, board %d\n", i);
		printk (" i_done   = %02x\n", tp->i_done);
		printk (" i_slip   = %02x\n", tp->i_slip);
		printk (" i_limit  = %02x\n", tp->i_limit);
		printk (" i_cmderr = %02x\n", tp->i_cmderr);
		printk (" busy = ");
		for (j = 7; j >= 0; j--)
		{
			switch (tp->busy[j])
			{
			case 0:			printk (" --- "); break;
			case OMS_BUSY_BLK:	printk (" BLK "); break;
			case OMS_BUSY_SIG:	printk (" SIG "); break;
			case OMS_BUSY_SIG_ACK:	printk (" SAK "); break;
			case OMS_BUSY_PENDING:	printk (" PEN "); break;
			default:		printk (" ??? "); break;
			}
		}
		printk ("\n");
		printk ("seq_no=%d, seq_queue", tp->seq_no);
		for (j = 0; j < MAX_RESP; j++)
			printk ("%c%d", j ? ',' : '=', tp->seq_queue[j]);
		printk ("\n");
		printk ("aarp_seq=%d\n", tp->aarp_seq);
		printk ("InputPutIndex=%02x, InputGetIndex=%02x\n",
			tp->address->input_put_index, tp->address->input_get_index);
		printk ("OutputGetIndex=%02x, OutputPutIndex=%02x\n",
			tp->address->output_get_index, tp->address->output_put_index);
	} 
}


/*----------------------------------------------------------------------
 * Function: bit_set()
 *
 * Analyse an axes bit mask; if exactly one bit is set then return
 * the bit number, otherwise return (-1).  This translates from
 * OMS_x_AXIS bit masks to OMS_x_POS definitions.
 *---------------------------------------------------------------------*/

static int
bit_set (u8 axes)
{
	int cnt = 0, axis = 0;
	u8 mask = 1, i = 0;

	while (mask)
	{
		if (axes & mask)
		{
			axis = i;
			cnt++;
		}
		mask <<= 1;
		i++;
	}
	return (cnt != 1) ? -1 : axis;
}

/*----------------------------------------------------------------------
 * Function: oms_interrupt()
 *
 * This is the interrupt handler for all boards.  The 'dev_id' parameter
 * identifies the relevant OMS_table[] entry.  Basically we just grab the
 * current interrupt status from the board and wake up or signal
 * processes as appropriate.
 *---------------------------------------------------------------------*/

static void
oms_interrupt (int irq, void *dev_id, struct pt_regs *regs)
{
	u8 status;
	u8 control;
	OMS_table_ptr tp = (OMS_table_ptr)dev_id;
	int i;
	u8 mask;

/*
 * WAITING FOR INPUT FROM OMS BEFORE IMPLEMENTING THIS
 *
 * save the control register in case a limit has been hit and it needs
 * to be changed - andy 11apr99
 */
 	control = read_oms_reg(tp, OMS_CNTL_REG);

	while ((status = read_oms_reg(tp, OMS_STATUS_REG)) & 0x80)
	{
		if (oms_debug > 3)
			printk("%s: Got IRQ 0x%02x, status 0x%02x", me, irq, status);

#ifdef VME58
		if ((status & (OMS_STATUS_REG_int_req_status |
				OMS_STATUS_REG_encoder_slip  |
				OMS_STATUS_REG_overtravel    |
				OMS_STATUS_REG_done          |
				OMS_STATUS_REG_command_error)) ==
					OMS_STATUS_REG_int_req_status)
		{
			u8 d1;

			if (oms_debug > 0)
				printk ("%s: I/O bits 0 or 1 (assumed)", me);
			/* Hmm, what do we do with these? */
			 	d1 = read_oms_reg(tp, OMS_DIO_LO_REG);  /* Unlatch interrupt */
				if (oms_debug > 3) printk (" value : %02x",d1);

		}
#endif

		if (status & OMS_STATUS_REG_command_error)
		{
			printk (KERN_ALERT "%s: Command error\n", me);
			tp->i_cmderr = 0xff;		/* Flag on all axes	*/
		}
		if ((status & OMS_STATUS_REG_encoder_slip))
		{
			u8 d1;

			if (oms_debug > 3) printk (" - Slip");
#ifdef VME58
 			d1 = read_oms_reg(tp, OMS_SLIP_REG);
			if (oms_debug > 3) printk (" value : %02x",d1);
			tp->i_slip |= d1;
#else
			tp->i_slip = 0xff;
#endif /* VME58 */
		}
		if ((status & OMS_STATUS_REG_overtravel))
		{
			u8 d1;

			if (oms_debug > 3) printk (" - Limit");
#ifdef VME58
 			d1 = read_oms_reg(tp, OMS_LIMIT_REG);
			if (oms_debug > 3) printk (" value : %02x",d1);
			tp->i_limit |= d1;
#else
			tp->i_limit = 0xff;
#endif /* VME58 */
/* 
 * WAITING FOR INPUT FROM OMS BEFORE IMPLEMENTING THIS
 *
 * disable limit interrupt, wait for host to drive off limit and
 * tell driver to reenable limit interrupt - andy 11apr99
 */
#ifdef VME58
 			control &= ~OMS_CNTL_REG_limit_register_int_enb;
 			write_oms_reg(tp, OMS_CNTL_REG, control);
#endif /* VME58 */

		}
		if ((status & OMS_STATUS_REG_done))
		{
			u8 d1, d2;

			if (oms_debug > 3) printk (" - Done");

			d1 = read_oms_reg(tp, OMS_DONE_REG);
			d2 = read_oms_reg(tp, OMS_DONE_REG);

			if ((d1 & d2) && oms_debug > 2)
				printk("%s: Duplicate DONEs: %02x, %02x\n",
						me, d1, d2);
			tp->i_done |= d1 | d2;
		}
#ifdef PC68
		/* Handle char I/O */
		if (status & OMS_STATUS_REG_rxbuf_full)
		{
			OMS_dram_ptr dp = tp->address;
			int c = read_oms_reg(tp, OMS_DATA_REG);
	
			if (oms_debug > 3)
				printk (" - RxFull");
			dp->inbuf[dp->input_put_index] = c;
			dp->input_put_index = (dp->input_put_index + 1) %
					OMS_BUF_SIZE;
			if (oms_debug > 5)
				printk("%s: Read char '%c' (%02x)\n", me,
					(c >= ' ' && c < 0x7f) ? c : '.', c);
		}
		if (status & OMS_STATUS_REG_txbuf_empty)
		{
			OMS_dram_ptr dp = tp->address;
			int c = dp->outbuf[dp->output_get_index];
	
			if (oms_debug > 3)
				printk (" - TxEmpty");
			if (dp->output_get_index != dp->output_put_index)
			{
				write_oms_reg(tp, OMS_DATA_REG, c);
				dp->output_get_index =
				     (dp->output_get_index + 1) % OMS_BUF_SIZE;
				if (oms_debug > 5)
					printk("%s: Wrote char '%c' (%02x)\n",
					    me, (c >= ' ' && c < 0x7f) ?
						c : '.', c);
				if (dp->output_get_index ==
						dp->output_put_index)
				{
					write_oms_reg (tp, OMS_CNTL_REG,
					    read_oms_reg(tp, OMS_CNTL_REG) &
						~OMS_CNTL_REG_txbuf_empty_enb);
				}
			}
		}
#endif
		if (oms_debug > 3)
			printk("\n");
	}

	/* Wake any processes waiting on OMS_CMD_BLK */

	wake_up_interruptible(&tp->blk_waitq);

	/* Signal any processes waiting on OMS_CMD_SIG or OMS_CMD_SIG_ACK */

	/* Signal any processes who have registered error handlers */

	/* Now any axes which are BUSY_PENDING for which we have queued
	 * a DONE interrupt, can be set not busy and we can clear the
	 * DONE flag.  This happens if a command is killed while a DONE
	 * is outstanding.
	 */
	for (i = 0, mask = 1; i < 8; i++, mask <<= 1)
	{
		if ((tp->i_done & mask) && tp->busy[i] == OMS_BUSY_PENDING)
		{
			tp->busy[i] = 0;
			tp->i_done ^= mask;
		}
	}
}


/*----------------------------------------------------------------------
 * Function: init_module()
 *
 * This is called when the driver is loaded, and creates and
 * initialises the oms_table[] array of local board data.
 *
 * XXX This needs to probe for the boards in future.
 *---------------------------------------------------------------------*/

int
init_module (void)
{
	int err, minor;
	OMS_table_ptr tp;

	/* register device & request a major number */

	err = register_chrdev (oms_major, me, &oms_fops);
	if (err < 0)
	{
		printk (KERN_NOTICE "%s: oms_init(): error %d\n", me, -err);
		return err;
	}
	if (oms_major == 0)
		oms_major = err;	/* Dynamic allocation */

	printk(KERN_INFO "Oregon Micro System %s Motor Controller: major %d\n",
		me, oms_major);

#ifdef OMS_LOG_IO
	log_io_buf = (log_io_t *)
		kmalloc(sizeof(log_io_t) * LOG_IO_ENTRIES, GFP_KERNEL);
	if (!log_io_buf)
		goto err_unreg_chrdev;
	memset(log_io_buf, 0xff, sizeof(log_io_t) * LOG_IO_ENTRIES);
#endif
#ifdef VME58
	/* Enable VME interrupts on level 'oms_level' */
	/* XXX This is not a driver function.  It should be a MVME16x
         * XXX board level service */
	{
		volatile u32 v, *p = (volatile u32 *)0xfff40084;

		v = *p;
		v &= ~(0x0f << (oms_level - 1) * 4);
		v |= (1 << (oms_level - 1) * 4);
		*p = v;

		p = (volatile u32 *)0xfff4006c;
		*p |= (1 << (oms_level - 1));
	}
#endif
	/* Analyse the oms_params string, if given */

	if (oms_params)
        {
		/* First determine how many boards are specified */
		int i = 0;
		char *p = oms_params;

		while (*p)
			if (*p++ == ',')
				i++;
		i = i + 1;
		if (i < 3 || i > 12 || (i % 3))
		{
			printk (KERN_NOTICE "%s: Bad oms_params\n", me);
			goto err_unreg_chrdev;
		}
		p = oms_params;
		oms_boards = i/3;
		/* now fill in oms_board_defs[] with the supplied params */
		for (i = 0; i < oms_boards; i++)
		{
			oms_board_defs[i].base = simple_strtoul(p,NULL,0);
			if ((p = strchr(p, ',')) == NULL)
			{
				printk (KERN_NOTICE "%s: Bad oms_params\n", me);
				goto err_unreg_chrdev;
			}
			oms_board_defs[i].address = simple_strtoul(p,NULL,0);
			if ((p = strchr(p, ',')) == NULL)
			{
				printk (KERN_NOTICE "%s: Bad oms_params\n", me);
				goto err_unreg_chrdev;
			}
			oms_board_defs[i].vector = simple_strtoul(p,NULL,0);
		}
	}
	oms_table = (OMS_table_ptr)kmalloc(sizeof(OMS_table_t) * oms_boards,
			GFP_KERNEL);
	if (oms_table == NULL)
		goto err_unreg_chrdev;
	memset (oms_table, 0, sizeof(OMS_table_t) * oms_boards);
	for (tp = oms_table, minor=0; minor < oms_boards; tp++, minor++)
	{
		tp->minor      = minor;
		tp->base       = oms_board_defs[minor].base;
		tp->address    = (OMS_dram_ptr)oms_board_defs[minor].address;
		tp->vector     = oms_board_defs[minor].vector;
#ifdef PC68
		/*
		 * For PC68 we simulate the shared DRAM in our local
		 * workspace, so the string i/o to the board is the
		 * same for all board types.
		 */
		tp->address    = &tp->pc68_dram;
#endif

		err = oms_configure(tp);

		if (err < 0)
		{
			printk("%s: minor %d - Initialization ERROR\n",
				me, minor);
			goto err_unreg_chrdev;
		}
		else if (oms_debug > 0)
			printk("%s: minor %d OK base=0x%x shared_mem=0x%p\n",
				me, minor, tp->base,
				tp->address);
	}
	return 0;

	/* XXX Needs to cope with an error part way through initialising
	 * XXX several boards.
	 */
err_unreg_chrdev:
	unregister_chrdev(oms_major, me);
	if (oms_table)
		kfree(oms_table);
#ifdef OMS_LOG_IO
	if (log_io_buf)
		kfree(log_io_buf);
#endif
	return -ENOMEM;
}

/*----------------------------------------------------------------------
 * Function: cleanup_module()
 *
 * Called when the driver is unloaded; it will only be called when
 * there are no processes with 'open' calls to the board so doesn't
 * have to do much by way of tidying up.
 *
 * XXX It should at least disable interrupts from the board!!
 *---------------------------------------------------------------------*/

void
cleanup_module (void)
{
	int i;

	printk(KERN_INFO "%s driver unloaded\n", me);

	for (i = 0; i < oms_boards; i++)
	{
		free_irq(oms_table[i].vector, oms_table + i);
#ifdef PC68
		release_region (oms_table[i].base, OMS_NUM_IOPORTS);
#endif
	}

	kfree(oms_table);

	unregister_chrdev(oms_major, me);
#ifdef OMS_LOG_IO
	kfree(log_io_buf);
#endif
}

/*----------------------------------------------------------------------
 * Function: oms_configure()
 *
 * Called to configure each minor device (ie. each board).  Doesn't
 * do much, and will probably be absorbed in to the board probing code
 * once that is written.
 *---------------------------------------------------------------------*/

static int
oms_configure (OMS_table_ptr tp)
{
	int cnt, ready;

#ifdef PC68
	/* identify the IOport used by the device  */
	request_region (tp->base, OMS_NUM_IOPORTS, me);
#endif
	if (request_irq(tp->vector, oms_interrupt, 0, me, tp))
	{
		printk (KERN_ALERT "%s: Cannot claim IRQ 0x%02x\n",
				me, tp->vector);
#ifdef PC68
		release_region (tp->base, OMS_NUM_IOPORTS);
#endif
		return -EBUSY;
	}

	/*-------------------------------------------------
	** wait until the board's intialized flag is set (clear for PC68)
	** XXX Should already be intialized - havn't just done a reset
	*/
	cnt   = 0;
	ready = read_oms_reg(tp, OMS_STATUS_REG) &
			OMS_STATUS_REG_initialized;
#ifdef PC68
	ready = !ready;
#endif
	while  (!ready)
	{
		if (++cnt > 10)
		{
			if (oms_debug > 3)
				printk("%s: Board not initialised\n", me);
			free_irq(tp->vector, tp);
#ifdef PC68
			release_region (tp->base, OMS_NUM_IOPORTS);
#endif
			return -EBUSY;
		}
		oms_sleep (1);
		ready = read_oms_reg(tp, OMS_STATUS_REG) &
				OMS_STATUS_REG_initialized;
	}
	if (oms_debug > 3)
		printk ("%s: oms_config(): cnt = %d \n", me, cnt);

	/*-------------------------------------------------
	** Configure OMS - interrupts & shared memory address
	*/
#ifdef VME58
	write_oms_reg(tp, OMS_INTVEC_REG, tp->vector);
#endif
	write_oms_reg(tp, OMS_CNTL_REG,
/*
 * Due to problems with limit interrupts on VME58 I have decided
 * to disable them permanently for this card. Only the PC68 will
 * have these interrupts enabled.
 *
 * a.gotz (1/6/99)
 */
			OMS_CNTL_REG_int_request_enb |
/*#if defined(VME58) || defined(PC58)
 *			OMS_CNTL_REG_encoder_slip_int_enb | 
 *			OMS_CNTL_REG_limit_register_int_enb | 
 *#elif defined(PC68)*/
#if defined(PC68)
			OMS_CNTL_REG_rxbuf_full_enb |
#endif
			OMS_CNTL_REG_done_reg_int_enb);

	/* set shared memory address */
#ifdef PC58
	write_oms_reg(tp, OMS_DPRAM_HI_REG, (tp->shared_mem >> 16) & 0xff);
	write_oms_reg(tp, OMS_DPRAM_LO_REG, (tp->shared_mem >>  8) & 0xff);
#endif
	/*---------------------------------------------------
	** set shared memory variable related to io buffer.
	** sometimes they come up funny (so PC58 driver said, anyway).
	*/
	{
		OMS_dram_ptr dp = tp->address;

		dp->output_put_index =  dp->output_get_index; 
		dp->input_get_index   = dp->input_put_index; 
	}

	return 0;
}

/*----------------------------------------------------------------------
 * Function: oms_open()
 *
 * Called when someone opens the /dev/whatever device file.  Just
 * update the usage count so the module doesn't get unloaded while
 * it is in use.
 *---------------------------------------------------------------------*/

static int
oms_open (struct inode *inode, struct file *file)
{
	if (oms_debug > 3)
		printk("%s: oms_open()\n", me);
	MOD_INC_USE_COUNT;
	return 0;
}

/*----------------------------------------------------------------------
 * Function: oms_release()
 *
 * Called when a process closes /dev/whatever, or when the process
 * exits.  XXX This needs to take care to cancel any outstanding timeouts
 * or signals which may be pending.
 *---------------------------------------------------------------------*/

static void
oms_release (struct inode *inode, struct file *file)
{
	if (oms_debug > 3)
		printk("%s: oms_release()\n", me);
	MOD_DEC_USE_COUNT;
}

/*----------------------------------------------------------------------
 * Function: oms_ioctl()
 *
 * This is the ioctl() handling routine, where all the real work is
 * done.
 *---------------------------------------------------------------------*/

static int
oms_ioctl (struct inode *inode, struct file *file,
                     unsigned int cmd, unsigned long arg)
{
	OMS_req_t req;		/*  driver's copy of user's parameters	*/
	OMS_table_ptr tp;
	int minor, res = OMS_ERR_NONE;
	int axis;
	u32 *seq_ptr;
	int i;
	int ret_req = 0;	/* !0 if OMS_req_t to be copied back	*/

	minor = MINOR(inode->i_rdev);	/* Minor = 0 for first brd, etc.*/
	if (minor >= oms_boards)
		return -ENODEV;

	tp = oms_table + minor;		/* Initialise local data ptr	*/

	/* Handle special cases, where there isn't an OMS_req_t struct	*/

	if (cmd == OMS_DEBUG)
	{
		oms_debug = (int)arg;	/* Set new debug level		*/
		return res;
	}
	else if (cmd == OMS_DUMP)
	{
		oms_dump();		/* Dump workspace to console	*/
		return res;
	}
#ifdef OMS_LOG_IO
	else if (cmd == OMS_LOG_IO)
	{
		int i = LOG_IO_ENTRIES;
		char *p = (char *)arg;

		res = verify_area(VERIFY_WRITE, (void *)arg,
			sizeof(int) * 2 + sizeof(log_io_t) * LOG_IO_ENTRIES);
		if (res)
			return res;
		memcpy_tofs (p, &i, sizeof(int));
		p += sizeof(int);
		i = log_io_index;
		memcpy_tofs (p, &i, sizeof(int));
		p += sizeof(int);
		memcpy_tofs (p, log_io_buf, LOG_IO_ENTRIES * sizeof(log_io_t));
		return 0;
	}
#endif

	/* Copy the request structure over from user-land		*/

	res = verify_area(VERIFY_READ, (void *)arg, sizeof(OMS_req_t));
	if (res)
		return res;
	memcpy_fromfs (&req, (OMS_req_ptr)arg, sizeof(OMS_req_t));

	/* Validate command string length; must fit in local buffer	*/

	if (req.cmdlen > MAX_STR)
	{
		if (oms_debug > 0)
			printk ("%s: Command string too long\n", me);
		return OMS_ERR_TOO_LONG;
	}

	/* Force a default timeout if none specified			*/

	if (!req.timeout)
		req.timeout = 4;	/* 4 seconds			*/

	/* Analyse request type and take the appropriate action		*/

	switch  (cmd)
	{
	case OMS_RESET:
#if defined(VME58)
		/* recover from a serious error on the board using the  */
		/* Mailbox feature to flush the command queue - andy 8dec99*/
		{
		u8 control;
		long *mailbox;
		mailbox = (long*)(tp->base + OMS_MAILBOX);
		*mailbox = 0x0001;
 		control = read_oms_reg(tp, OMS_CNTL_REG);
		control |= OMS_CNTL_REG_int_request;
		write_oms_reg(tp, OMS_CNTL_REG, control);
		printk ("oms : reset mailbox=0001 and set bit 5\n");
		}
#endif /* VME58 */
		break;
	case OMS_REG_HANDLER:
		/* Register a signal to send in case of an error	*/
		res = OMS_ERR_NIF;
		break;
	case OMS_READ_INTS:
		/* Read current interrupt status for specified axes	*/
		req.f_done   = tp->i_done   & req.axes;
		req.f_slip   = tp->i_slip   & req.axes;
		req.f_limit  = tp->i_limit  & req.axes;
		req.f_cmderr = tp->i_cmderr & req.axes;
		ret_req = 1;
		break;
	case OMS_ACK_INTS:
		/* Read and acknowledge int. status (ie. clear flags)	*/
		cli();
		req.f_done   = tp->i_done   & req.axes;
		req.f_slip   = tp->i_slip   & req.axes;
		req.f_limit  = tp->i_limit  & req.axes;
		req.f_cmderr = tp->i_cmderr & req.axes;

		tp->i_done   &= ~req.axes;
		tp->i_slip   &= ~req.axes;
		tp->i_limit  &= ~req.axes;
		tp->i_cmderr &= ~req.axes;
		sti();
		ret_req = 1;
		break;
	case OMS_READ_POSITIONS:
		/* Read current position for all axes.
		 * We use the dreaded goto here to ensure all exit paths
		 * do the right thing with regard to wait queues, etc.
		 */
		current->timeout = jiffies + req.timeout * HZ;
		cli();
		if ((seq_ptr = set_seq(tp)) == NULL)
		{
			sti();
			if (oms_debug > 0)
				printk ("%s: Response queue overload\n", me);
			return OMS_ERR_RESP_OVERLOAD;
		}
		while (!youngest_seq(tp, seq_ptr))
		{
			interruptible_sleep_on(&tp->resp_waitq);
			if (oms_debug > 2)
				printk ("%s: Woken for OMS_CMD_RESP\n", me);
			/* If positions have been read since we went to
			 * sleep then use those values now
			 */
			if (tp->aarp_seq > *seq_ptr)
			{
				memcpy(req.positions, tp->positions,
						sizeof(u32) * 8);
				goto exit_read_positions;
			}
			if (current->signal & ~current->blocked)
			{
				res = -ERESTARTSYS;
				goto exit_read_positions;
			}
			if (!current->timeout)
			{
				res = OMS_ERR_TIMEOUT;
				goto exit_read_positions;
			}
		}
		/* OK, it is our turn to have a go!! */
		sti();
#ifdef GET_POSITIONS_FROM_SHARED_MEMORY
		write_oms_reg(tp, OMS_CNTL_REG, read_oms_reg(tp, OMS_CNTL_REG)
				| OMS_CNTL_REG_data_area_update_request);
#else
		if (tp->address->input_get_index != tp->address->input_put_index)
		{
			printk (KERN_ALERT "%s: Premature response!\n", me);
			tp->address->input_get_index = tp->address->input_put_index;
		}
		down (&Sem);
		strcpy(tp->buf, "amrp");
		res = oms_puts(tp);
		up (&Sem);
		if (res)
			goto exit_read_positions;
#endif
		/* Should have a response quickly to this; wait half a second
		 * waking every tick.
		 */
		/* XXX do we have to worry about being killed here? */
#ifdef GET_POSITIONS_FROM_SHARED_MEMORY
		for (i = 0; (read_oms_reg(tp, OMS_CNTL_REG) &
			OMS_CNTL_REG_data_area_update_request) && i < HZ/2; i++)
			oms_sleep(1);
		if (read_oms_reg(tp, OMS_CNTL_REG) &
                        OMS_CNTL_REG_data_area_update_request)
		{
			res = OMS_ERR_TIMEOUT;
			goto exit_read_positions;
		}
		for (i = 0; i < 8; i++)
			tp->positions[i] = GETLONG_D16(&tp->address->motor_status[i].axis.command_pos);
		tp->aarp_seq = ++tp->seq_no;
		memcpy(req.positions, tp->positions, sizeof(u32) * 8);
#else
		for (i = 0; !oms_resp_ready(tp) && i < req.timeout * HZ; i++)
			oms_sleep(1);

		if (oms_debug > 0 && !oms_resp_ready(tp))
		{
			/* Seems to have failed; let's dump the input buffer */
			OMS_dram_ptr dp = tp->address;
			int c = (u8)(dp->input_put_index - dp->input_get_index);
			u8 i = (u8)dp->input_get_index;

			printk ("%s: RP failed, ibuf[%02x]='", me, i);
			while (c--)
			{
				if (dp->inbuf[i] > ' ' && dp->inbuf[i] < 0x7f)
					printk ("%c", dp->inbuf[i++]);
				else
					printk ("<%02x>", dp->inbuf[i++]);
			}
			printk ("'\n");
		}
		if (tp->address->input_get_index == tp->address->input_put_index)
		{
			res = OMS_ERR_TIMEOUT;
			goto exit_read_positions;
		}
		res = oms_gets(tp);
		if (res)
			goto exit_read_positions;
		if ((res = parse_aarp(tp)) == OMS_ERR_NONE)
		{
			tp->aarp_seq = ++tp->seq_no;
			memcpy(req.positions, tp->positions, sizeof(u32) * 8);
		}
#endif
exit_read_positions:
		*seq_ptr = 0;
		wake_up_interruptible(&tp->resp_waitq);
		ret_req = 1;
		break;
	case OMS_READ_AXES_INFO:
		/* Read axes parameters from shared memory		*/
#ifdef VME58
		res = OMS_ERR_NIF;
#else
		res = OMS_ERR_NIF;
#endif
		break;
	case OMS_CMD_BLK:
	case OMS_CMD_FORCE:
		/* Issue a command and block until done			*/
		if ((axis = bit_set(req.axes)) < 0)
		{
			if (oms_debug > 0)
				printk ("%s: Bad axis specification 0x%02x\n",
					me, req.axes);
			return OMS_ERR_BAD_AXES;
		}
		cli();
		if (cmd == OMS_CMD_FORCE)
		{
			;  /* Don't worry about the following checks	*/
		}
		else if (tp->busy[axis] == OMS_BUSY_PENDING)
		{
			if (oms_debug > 0)
				printk ("%s: Axis has DONE interrupt pending\n", me);
			sti();
			return OMS_ERR_DONE_PENDING;
		}
		else if (tp->busy[axis])
		{
			if (oms_debug > 0)
				printk ("%s: Axis already busy\n", me);
			sti();
			return OMS_ERR_AXIS_BUSY;
		}
		else if ((tp->i_done | tp->i_slip | tp->i_limit |
				tp->i_cmderr) & req.axes)
		{
			if (oms_debug > 0)
				printk ("%s: Interrupt queued\n", me);
			sti();
			return OMS_ERR_INT_QUEUED;
		}
		tp->busy[axis] = OMS_BUSY_BLK;
		sti();
		down (&Sem);
		res = oms_putusr(tp, &req);
		up (&Sem);
		if (res)
			break;
		current->timeout = jiffies + req.timeout * HZ;
		cli();
		while (!(tp->i_done & req.axes))
		{
			interruptible_sleep_on(&tp->blk_waitq);
			if (oms_debug > 2)
				printk ("%s: Woken for OMS_CMD_BLK on %d\n",
						me, axis);
			if (current->signal & ~current->blocked)
			{
				tp->busy[axis] = OMS_BUSY_PENDING;
				return -ERESTARTSYS;
			}
			if (tp->i_done & req.axes)
				break;
			if (!current->timeout)
			{
				tp->busy[axis] = OMS_BUSY_PENDING;
				return OMS_ERR_TIMEOUT;
			}
		}
		tp->i_done &= ~req.axes;
		tp->busy[axis] = 0;
		break;
	case OMS_CMD_SIG:
		/* Issue a command and trigger a signal on completion	*/
		if ((axis = bit_set(req.axes)) < 0)
		{
			if (oms_debug > 0)
				printk ("%s: Bad axis specification\n", me);
			return OMS_ERR_BAD_AXES;
		}
		res = OMS_ERR_NIF;
		break;
	case OMS_CMD_SIG_ACK:
		/* Issue a command; signal and clear DONE when complete	*/
		if ((axis = bit_set(req.axes)) < 0)
		{
			if (oms_debug > 0)
				printk ("%s: Bad axis specification\n", me);
			return OMS_ERR_BAD_AXES;
		}
		res = OMS_ERR_NIF;
		break;
	case OMS_CMD_RET:
		/* Issue a command and return immediately		*/
		down (&Sem);
		res = oms_putusr(tp, &req);
		up (&Sem);
		/* XXX if (res) ... */
		break;
	case OMS_CMD_RESP:
		/* Issue a command and wait for a response.
		 * We use the dreaded goto here to ensure all exit paths
		 * do the right thing with regard to wait queues, etc.
		 */
		current->timeout = jiffies + req.timeout * HZ;
		cli();
		if ((seq_ptr = set_seq(tp)) == NULL)
		{
			sti();
			if (oms_debug > 0)
				printk ("%s: Response queue overload\n", me);
			return OMS_ERR_RESP_OVERLOAD;
		}
		while (!youngest_seq(tp, seq_ptr))
		{
			interruptible_sleep_on(&tp->resp_waitq);
			if (oms_debug > 2)
				printk ("%s: Woken for OMS_CMD_RESP\n", me);
			if (current->signal & ~current->blocked)
			{
				res = -ERESTARTSYS;
				goto exit_cmd_resp;
			}
			if (!current->timeout)
			{
				res = OMS_ERR_TIMEOUT;
				goto exit_cmd_resp;
			}
		}
		/* OK, it is our turn to have a go!! */
		sti();
		if (tp->address->input_get_index != tp->address->input_put_index)
		{
			printk (KERN_ALERT "%s: Premature response!\n", me);
			tp->address->input_get_index = tp->address->input_put_index;
		}
		down (&Sem);
		res = oms_putusr(tp, &req);
		up (&Sem);
		if (res)
			goto exit_cmd_resp;
		/* XXX do we have to worry about being killed here? */
		for (i = 0; !oms_resp_ready(tp) && i < req.timeout * HZ; i++)
			oms_sleep(1);
		if (tp->address->input_get_index == tp->address->input_put_index)
		{
			res = OMS_ERR_TIMEOUT;
			goto exit_cmd_resp;
		}
		res = oms_getusr(tp, &req);
exit_cmd_resp:
		*seq_ptr = 0;
		wake_up_interruptible(&tp->resp_waitq);
		ret_req = 1;
		break;
	default:
		res = OMS_ERR_INV_FUN;
		break;
	}

	if (res)
		return res;

	if (ret_req)
	{
		res = verify_area(VERIFY_WRITE, (void *)arg, sizeof(OMS_req_t));

		if (res)
			return res;

		memcpy_tofs ((OMS_req_ptr)arg, &req, sizeof(OMS_req_t));
	}

	return res;
}

