/*
 * This a a threads based test program for the Oregon VME58/PC68 driver
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "omslib.h"

#define CMD_BLK_THREADS		4	/* Use S, then R, then V, then U */
#define READ_POSITION_THREADS	4
#define WHO_ARE_YOU_THREADS	0

int fd;

pthread_t cmd_blk_threads[CMD_BLK_THREADS];
pthread_t read_position_threads[READ_POSITION_THREADS];
pthread_t who_are_you_threads[WHO_ARE_YOU_THREADS];

void
ioctl_err (int res)
{
	if (oms_strerror(res))
		fprintf (stderr, "%d: driver error: %s\n",
				getpid(), oms_strerror(res));
	else
		fprintf (stderr, "%d: %s\n", getpid(), strerror(errno));
}

void
ack_ints (char *id, char axis_name, u8 axis)
{
	OMS_req_t req;
	int res;

	memset(&req, 0, sizeof(req));
	req.axes = axis;
	res = ioctl(fd, OMS_ACK_INTS, &req);

	if (res)
	{
		fprintf (stderr, "OMS_ACK_INTS failed!!!\n");
		return;
	}
	if (req.f_slip)
		fprintf (stderr, "%d: %s, axis %c, SLIP interrupt\n", getpid(), id, axis_name);
	if (req.f_limit)
		fprintf (stderr, "%d: %s, axis %c, LIMIT interrupt\n", getpid(), id, axis_name);
	if (req.f_done)
		fprintf (stderr, "%d: %s, axis %c, DONE interrupt\n", getpid(), id, axis_name);
	if (req.f_cmderr)
		fprintf (stderr, "%d: %s, axis %c, CMDERR interrupt\n", getpid(), id, axis_name);
}


/* 'arg' is an axis number, reverse mapping, with 0=S, 1=R, ettc. */

void *
cmd_blk (void *arg)
{
	int thr = (int)arg;
	char axis_name= "SRVUTZYX"[thr];
	struct timeval ts, te;
	double dt, rate;
	int cnt = 0;
	OMS_req_t req;
	int res;
	char cmdstr[80];
	char *errstr;
	int i = thr;
	int step = 50;
	int mr;
	int max_move = 75000;	/* Make this smaller, or timeout bigger
				 * if you don't want errors
				 */
	char ammr_cmd[80];

	srand(getpid());

	memset(&req, 0, sizeof(req));
	req.axes = 0x80;
	while (i-- > 0)
		req.axes >>= 1;
	req.timeout = 2;	 /* Two second timeout */

	/* This uses multi-axis mode, but seemed to work equally well
	 * using single axis mode
	 */

	strcpy (ammr_cmd, "ammr,,,,,,,,");
	ammr_cmd[11-thr] = '\0'; /* Get right number of commas on am cmd */

	printf ("CmdBlk thread %d starting as pid %d\n", thr, getpid());

	gettimeofday(&ts, NULL);
	for (;;)
	{
		mr = (rand() % max_move) - max_move/2;
		sprintf (cmdstr, "%s%d;gdid", ammr_cmd, mr);
#if 0
		fprintf (stderr, "cmdstr='%s'\n", cmdstr);
#endif
		req.cmdstr = cmdstr;
		req.cmdlen = strlen(cmdstr);
		res = ioctl(fd, OMS_CMD_BLK, &req);
		if (res == OMS_ERR_INT_QUEUED)
		{
			/* Can happen if an earlier test got killed,
			 * or if we have a LIMIT interrupt, for
			 * example.
			 */
			errstr = "Interrupt Queued";
		}
		else if (res == OMS_ERR_DONE_PENDING)
		{
			/* Something sent a command which should
			 * have resulted in a DONE interrupt, 
			 * but died or timedout out before it was
			 * received.  DONE int still outstanding.
			 * Command could have died because of
			 * limit interrupt, so DONE will never
			 * happen.
			 */
			errstr = "DONE Pending";
		}
		else if (res == OMS_ERR_TIMEOUT)
		{
			/* A command failed to complete within the
			 * timeout.  Issue a ST ID for the axis, and
			 * wait for the interrupt.  Should then
			 * acknowledge any further queued ints.
			 */
			errstr = "TIMEOUT";
		}
		else if (res)
		{
			ioctl_err(res);
			return (void *)1;
		}
		else
			errstr = NULL;

		if (errstr)
		{
			fprintf(stderr, "%d: CmdBlk, Axis %c, %s on cycle %d\n",
					getpid(), axis_name, errstr, cnt);
			ack_ints("CmdBlk", axis_name, req.axes);
			sprintf (cmdstr, "a%cstid", axis_name);
			req.cmdstr = cmdstr;
			req.cmdlen = strlen(cmdstr);
			while ((res = ioctl(fd, OMS_CMD_FORCE, &req)))
			{
				fprintf(stderr, "%d: CmdBlk, Axis %c, FAILED TO STOP",
					getpid(), axis_name);
				errstr = oms_strerror(res);
				fprintf(stderr, ", res=%d, %s\n", res, errstr ? errstr : "???");
			}
			continue;
		}
		cnt++;
		if (!(cnt % step))
		{
			gettimeofday(&te, NULL);
			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			rate = dt / (double)cnt;
			fprintf (stderr, "%d: CmdBlk, %d cycles in %.3fs;"
					" average %.3fms per request\n",
					getpid(), cnt, dt/1000000, rate/1000);
		}
	}
	return NULL;
}


void *
who_are_you (void *arg)
{
	int thr = (int)arg;
	struct timeval ts, te;
	double dt, rate;
	int cnt = 0;
	OMS_req_t req;
	int res;
	char *who = "wy ";
	char resp[80];

	memset(&req, 0, sizeof(req));

	req.cmdstr = who;
	req.cmdlen = strlen(who);
	req.rspstr = resp;
	req.rsplen = 80;

	printf ("Who Are You thread %d starting as pid %d\n", thr, getpid());

	gettimeofday(&ts, NULL);
	for (;;)
	{
		res = ioctl(fd, OMS_CMD_RESP, &req);
		if (res)
		{
			ioctl_err(res);
			return (void *)1;
		}
		cnt++;
		if (!(cnt % 1000))
		{
			gettimeofday(&te, NULL);
			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			rate = dt / (double)cnt;
			fprintf (stderr, "%d: WhoAreYou, %d cycles in %.3fs;"
					" average %.3fms per request\n",
					getpid(), cnt, dt/1000000, rate/1000);
		}
	}
	return NULL;
}


void *
read_position (void *arg)
{
	int thr = (int)arg;
	struct timeval ts, te;
	double dt, rate;
	int cnt = 0;
	OMS_req_t req;
	int res;

	memset(&req, 0, sizeof(req));

	printf ("Read Position thread %d starting as pid %d\n", thr, getpid());

	gettimeofday(&ts, NULL);
	for (;;)
	{
		res = ioctl(fd, OMS_READ_POSITIONS, &req);
		if (res)
		{
			ioctl_err(res);
			return (void *)1;
		}
		cnt++;
		if (!(cnt % 1000))
		{
			gettimeofday(&te, NULL);
			dt = ((double)te.tv_sec * 1000000 + te.tv_usec) -
				((double)ts.tv_sec * 1000000 + ts.tv_usec);
			rate = dt / (double)cnt;
			fprintf (stderr, "%d: ReadPosition, %d cycles in %.3fs;"
					" average %.3fms per request\n",
					getpid(), cnt, dt/1000000, rate/1000);
		}
	}
	return NULL;
}


int
main (int argc, char **argv)
{
	int i, res;

	fd = open("/dev/oms0", O_RDWR);

	if (fd < 0)
	{
		perror ("open");
		exit (1);
	}

	res = ioctl(fd, OMS_DEBUG, (void *)1);
	if (res)
	{
		ioctl_err(res);
		exit (1);
	}

	for (i = 0; i < CMD_BLK_THREADS; i++)
	{
		res = pthread_create(&cmd_blk_threads[i], NULL,
				cmd_blk, (void *)i);

		if (res)
		{
			fprintf (stderr, "pthread_create() returned %d\n",
					res);
			exit(1);
		}
	}
	for (i = 0; i < READ_POSITION_THREADS; i++)
	{
		res = pthread_create(&read_position_threads[i], NULL,
				read_position, (void *)i);

		if (res)
		{
			fprintf (stderr, "pthread_create() returned %d\n",
					res);
			exit(1);
		}
	}
	for (i = 0; i < WHO_ARE_YOU_THREADS; i++)
	{
		res = pthread_create(&who_are_you_threads[i], NULL,
				who_are_you, (void *)i);

		if (res)
		{
			fprintf (stderr, "pthread_create() returned %d\n",
					res);
			exit(1);
		}
	}
	for (;;)
		sleep(1);
	return 0;
}

