/*static char RcsId[] = "@(#) $Header: McaAIM.c,v 1.2 2000/07/26 11:30:33 beteva Rel $ ";*/
/*********************************************************************
 * File: McaAIM
 * Project: Multi Channel Analyser
 * Description:
 * Author(s); A.Beteva
 * Original: April 1998
 * $Log:	McaAIM.c,v $
 * Revision 1.2  2000/07/26  11:30:33  11:30:33  beteva (A.Beteva)
 * added giDebugLevel initialisation at startup
 * 
 * Revision 1.1  2000/07/24  19:47:05  19:47:05  beteva (A.Beteva)
 * Initial revision
 * European Synchrotron Radiation Facility, Grenoble, France
 *********************************************************************/

#include <API.h>
#include <DevServer.h>
#include <DevErrors.h>
#include <DevServerP.h>
#include <McaAIM.h>
#include <McaAIMP.h>
#include <McaDevErrors.h>
#include <McaDevCmds.h>
#include <signal.h>
/* elapsedtime measurment */
#include <Time.h>
#include <sys/types.h>
#include <sys/timeb.h>
/* C run time library includes 
#include <tchar.h>
#include <stdio.h> */

#include <mmp.h>
/* Genie-2000 include files */
#include <crackers.h>
#include <citypes.h> /* typedefs */
#include <citypes.h>
#include <spasst.h>  /* definitions - HMEM ... */
#include <sad.h>
#include <get_put.h>
#include <driver.h>
#include <ci_files.h>
#include <campdef.h>
#include <cam_n.h>
#include <utility.h> /* library routines function prototypes */
#include <anadefs.h>
#include <sad_nest.h>

#include <process.h>

/* public methods */
static long class_initialise (long *);
static long object_create (char *, DevServer *, long *);
static long object_initialise (McaAIM, long *);
static long state_handler (McaAIM, DevCommand, long *);

/* internal routines */
static long init_module (McaAIM);
static long init_icb (McaAIM, short);
void mcastopthread(McaAIM ds);

void mcaspecialthreadacq(McaAIM ds);


static DevMethodListEntry methods_list[] = {
 {DevMethodClassInitialise, class_initialise},
 {DevMethodCreate, object_create},
 {DevMethodInitialise, object_initialise},
 {DevMethodStateHandler, state_handler},
};

McaAIMClassRec mcaAIMClassRec = {
   /* n_methods */     sizeof(methods_list)/sizeof(DevMethodListEntry),
   /* methods_list */  methods_list,
};

McaAIMClass mcaAIMClass = (McaAIMClass)&mcaAIMClassRec;

/* public commands */
static long  mcaselectadc (McaAIM, DevShort *, DevVoid *, long*);
static long  mcasetmemorygroupsize (McaAIM, DevLong *,
									DevVarLongArray *, long *);
static long  mcaselectmemorygroup (McaAIM, DevShort *, DevVoid *, long *);
static long  mcaselecttimemode (McaAIM, DevShort *, DevVoid *, long *);
static long  mcaselectstopmode (McaAIM, DevVarShortArray *, DevVoid *, long *);
static long  mcapresettime (McaAIM, DevVarDoubleArray *, DevVoid *, long *);
static long  mcapreset (McaAIM, DevVarDoubleArray *, DevVoid *, long *);
static long  mcapresetcountvalue (McaAIM, DevLong *, DevLong *, long *);
static long  mcastartacq (McaAIM, DevVoid *, DevVoid *, long *);
static long  mcastopacq (McaAIM, DevVoid *, DevVoid *, long *);
static long  mcaclearmemory (McaAIM, DevVoid *, DevVoid *, long *);
static long  mcareset (McaAIM, DevVoid *, DevVoid *, long *);
static long  mcareaddata (McaAIM, DevVarLongArray *, DevVarULongArray *, long *);
static long  mcagetadcinfo (McaAIM, DevVoid *, DevVarDoubleArray *, long *);
static long  mcagetstatus (McaAIM, DevVoid *, DevVarShortArray *, long *);
static long  mcasendicb (McaAIM, DevVarULongArray *, DevVoid *, long *);
static long  mcarecvicb (McaAIM, DevVarULongArray *,  DevVarULongArray *, long *);
static long  mcaseticbaddr (McaAIM, DevShort *, DevVoid *, long *);
static long  mcaspecialacq (McaAIM, DevVarULongArray *, DevVarULongArray *, long *);
static long  mcaspecialsetup (McaAIM, DevVarULongArray *, DevVoid *, long *);
static long  mcaspecialstart (McaAIM, DevVoid *, DevVoid *, long *);
static long  mcaspecialread (McaAIM, DevVoid *, DevVarULongArray *, long *);
static long  mcaspecialstop (McaAIM, DevVoid *, DevVoid *, long *);
static long  dev_status (McaAIM, DevVoid *, DevString *, long *);
static long  dev_state (McaAIM, DevVoid *, DevShort *, long *);

static DevCommandListEntry commands_list[] = {
	{DevMcaSelectAdc,mcaselectadc, D_SHORT_TYPE, D_VOID_TYPE},
	{DevMcaSetMemoryGroupSize, mcasetmemorygroupsize, D_LONG_TYPE,D_VAR_LONGARR},
  {DevMcaSelectMemoryGroup, mcaselectmemorygroup, D_SHORT_TYPE, D_VOID_TYPE},
  {DevMcaSelectTimeMode, mcaselecttimemode, D_SHORT_TYPE, D_VOID_TYPE},
  {DevMcaSelectStopMode, mcaselectstopmode, D_VAR_SHORTARR, D_VOID_TYPE},
  {DevMcaPresetTime, mcapresettime, D_VAR_DOUBLEARR, D_VOID_TYPE},
  {DevMcaPreset, mcapreset, D_VAR_DOUBLEARR, D_VOID_TYPE},
  {DevMcaPresetCountValue, mcapresetcountvalue, D_LONG_TYPE, D_LONG_TYPE},
  {DevMcaStartAcq, mcastartacq, D_VOID_TYPE, D_VOID_TYPE},
  {DevMcaStopAcq, mcastopacq, D_VOID_TYPE, D_VOID_TYPE},
  {DevMcaClearMemory, mcaclearmemory, D_VOID_TYPE, D_VOID_TYPE},
  {DevReset, mcareset, D_VOID_TYPE, D_VOID_TYPE},
  {DevMcaReadData, mcareaddata, D_VAR_LONGARR, D_VAR_ULONGARR},
  {DevMcaGetAdcInfo, mcagetadcinfo, D_VOID_TYPE, D_VAR_DOUBLEARR},
  {DevMcaGetStatus, mcagetstatus, D_VOID_TYPE, D_VAR_SHORTARR},
  {DevMcaSendICB, mcasendicb, D_VAR_ULONGARR, D_VOID_TYPE},
  {DevMcaReadICB, mcarecvicb, D_VAR_ULONGARR, D_VAR_ULONGARR},
  {DevMcaSetICBAddr, mcaseticbaddr, D_SHORT_TYPE, D_VOID_TYPE},
  {DevMcaSpecialAcq, mcaspecialacq, D_VAR_ULONGARR, D_VAR_ULONGARR},
  {DevMcaSpecialSetup, mcaspecialsetup, D_VAR_ULONGARR, D_VOID_TYPE},
  {DevMcaSpecialStart, mcaspecialstart, D_VOID_TYPE, D_VOID_TYPE},
  {DevMcaSpecialRead, mcaspecialread, D_VOID_TYPE, D_VAR_ULONGARR},
  {DevMcaSpecialStop, mcaspecialstop, D_VOID_TYPE, D_VOID_TYPE},
  {DevStatus, dev_status, D_VOID_TYPE, D_STRING_TYPE},
  {DevState, dev_state, D_VOID_TYPE, D_SHORT_TYPE},
};

static long n_commands = sizeof(commands_list)/sizeof(DevCommandListEntry);

/* reserve space for a default copy of the mcaAIM object */
static McaAIMRec mcaAIMRec;
static McaAIM mcaAIM = (McaAIM)&mcaAIMRec;

/* template resource table used to access the static database */
  
db_resource res_object[] = {
  {"aim_file",     D_STRING_TYPE,  NULL},
  {"channel_num",  D_LONG_TYPE,   NULL},
  {"curr_adc",	   D_SHORT_TYPE,  NULL},
  {"mode",         D_SHORT_TYPE,  NULL},
  {"time",         D_DOUBLE_TYPE, NULL},
  {"horz_scale",   D_FLOAT_TYPE,  NULL},
  {"zb_offset",    D_SHORT_TYPE,  NULL},
  {"icb_address",  D_SHORT_TYPE,  NULL},
};
int res_object_size = sizeof(res_object)/sizeof(db_resource);

/* some global variables */
static long part1_ch_num = 0;
static DevShort tmp_aim_addr = 0;
static short adc_num = 0;
static short dev_num = 0;
static DevVarDoubleArray darray_argin;
double double_table[8];

char err_msg[512];
FetchFrom_T stData; /* get some info */
AcqPage_T stAcq; /* presets & elapsed */
INT iReturn;  /* return value from VDM calls - 0 or error */
SHORT sReturn; /* return value from SAD calls - 0 or error */

/* AIM display error */
ULONG ulError; /* error code for the AIM calls */
SHORT sDum1; /* dummy */
USHORT usDum2; /* dummy */
DSQuery_T stInfo;

/* Special Acquisition */
unsigned long	*SpDataBuff = (unsigned long *)NULL;
unsigned long	*SpNewStart = (unsigned long *)NULL;
unsigned short	SpReadMaxSize = 16384;
int				SpSpectrumIndex = 0;
int				SpLastSend = -1;
int				SpNbCycle = 0;
unsigned short	SpNbCh = 0;
int				SpStopThread = 0;
int				SpThreadStopped = 1;
int				SpMode = 0;
int				SpPPLevel = 0;
int             ppdata;

#ifdef ELAPSEDTIME_1
struct _timeb start, finish;
#endif
#ifdef ELAPSEDTIME
double elapsed_time;
long beg, end;
#endif

/* _________________________________________________________________________
 Function:      static long class_initialise()

 Description:	routine to be called the first time a device is 
 		created which belongs to this class.
		This routine will be called only once.
 Arg(s) In:	none

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
  _________________________________________________________________________*/

static long class_initialise (long *error)
{
  dprintf2("class_initialise enter");

  /* McaAIMClass is a subclass of the DevServerClass */
  mcaAIMClass->devserver_class.superclass = devServerClass;
  mcaAIMClass->devserver_class.class_name =
     (char*)malloc(strlen("McaAIMClass")+1);

  HWALK(mcaAIMClass->devserver_class.class_name);
  sprintf(mcaAIMClass->devserver_class.class_name,"McaAIMClass");


  /* commands implemented for the McaAIM class */
  mcaAIMClass->devserver_class.n_commands = n_commands;
  mcaAIMClass->devserver_class.commands_list = commands_list;

  mcaAIMClass->devserver_class.class_inited = 1;

  /* initialise the template powersupply so that DevMethodCreate has
     default values for creating a McaAIM - these values will be
     overridden by the static database (if defined there)  */
  mcaAIM->devserver.class_pointer = (DevServerClass)mcaAIMClass;

 mcaAIM->devserver.state = DEVON;

  /* Allocates memory to store the acquisition data */

  mcaAIM->mcaAIM.acq_mem = 0;
  mcaAIM->mcaAIM.acq_mem = (unsigned char *)malloc(65536*sizeof(unsigned long));
  if (mcaAIM->mcaAIM.acq_mem == 0)
    {
      *error = DevErr_InsufficientMemory;
      return(DS_NOTOK);
    }

  dprintf("class_initialise exit");
  return (DS_OK);
}

/*________________________________________________________________________

 Function:    static long object_create()
 Description: routine to be called on creation of a device object
 Arg(s) In:   char *name - name to be given to device
 Arg(s) Out:  DevServer *ds_ptr - pointer to created device
              long *error - pointer to error code, in case routine fails
 _________________________________________________________________________*/

static long object_create(char *name, DevServer *ds_ptr, long *error)
{
  McaAIM ds;

  dprintf2("object_create enter");

  *error = 0;
  ds = (McaAIM)malloc(sizeof(McaAIMRec));
  HWALK(ds);

  /* initialise device with default object */
  *(McaAIMRec*)ds = *(McaAIMRec*)mcaAIM;

  /* finally initialise the non-default values */
  ds->devserver.name = (char*)malloc(strlen(name)+1);
  HWALK(mcaAIMClass->devserver_class.class_name);
  sprintf(ds->devserver.name,"%s",name);

  *ds_ptr = (DevServer)ds;

  dprintf("object_create exit");
  return(DS_OK);
}

/*_________________________________________________________________________

  Function:    static long object_initialise()
  Description: routine to be called on initialisation of a device object
  Arg(s) In:   McaAIM ds - object to initialise
  Arg(s) Out:  long *error - pointer to error code, in case routine fails
 _________________________________________________________________________*/

static long object_initialise(McaAIM ds, long *error)
{
  /* all definitions used for setup the acquisition */
  int  response=0;
  int  cmd_size=0;
  int  actual=0;
  int  s=0;
  double  time = 0.;
  InfoICB_T stInfoICB;
  
  dprintf2("object_initialise enter");
  
  ds->mcaAIM.aim_file = "UNDEFINED";
  ds->mcaAIM.horz_scale = 0;
  ds->mcaAIM.zb_offset = 0;
  ds->mcaAIM.channel_num = 8192;
  ds->mcaAIM.curr_mem_gr = 1;
  ds->mcaAIM.curr_adc = 1;
  ds->mcaAIM.mode = 3;
  ds->mcaAIM.ch_per_group = 8192;
  ds->mcaAIM.nb_groups = 1;
  ds->mcaAIM.first_channel = 0;
  ds->mcaAIM.last_channel = 8191;
  ds->mcaAIM.pres_live_time = 0.;
  ds->mcaAIM.pres_real_time = 0.;
  ds->mcaAIM.total_counts = 0; 
  /* zero is not accepted from the MCA,
      therefore put the highest number possible
  ds->mcaAIM.total_counts = (unsigned long)21474836; */

  ds->mcaAIM.icb_address = 0x88; /* no ICB module */

 /* initialise mcaAIM with values defined in database */
  res_object[0].resource_adr = &(ds->mcaAIM.aim_file);
  res_object[1].resource_adr = &(ds->mcaAIM.channel_num);
  res_object[2].resource_adr = &(ds->mcaAIM.curr_adc);
  res_object[3].resource_adr = &(ds->mcaAIM.mode);
  res_object[4].resource_adr = &(time);
  res_object[5].resource_adr = &(ds->mcaAIM.horz_scale);
  res_object[6].resource_adr = &(ds->mcaAIM.zb_offset);
  res_object[7].resource_adr = &(ds->mcaAIM.icb_address);
  
  if (db_getresource(ds->devserver.name, res_object, res_object_size, error))
    return(DS_NOTOK);

  /* first check if aim_address defined */
  if (*ds->mcaAIM.aim_file == 'U')
    {
      *error = DevErr_McaAIMADCNotSetup;
      /*return (DS_NOTOK); */
    }

  /* increase the number of devices */
  dev_num++;

  /* initialize environment settings from the registry
   first compulsory call before using the library routines */
  vG2KEnv(); 
  /* Establish a connection to the local VDM */

  iReturn = iUtlCreateFileDSC2(&ds->mcaAIM.hDSC, 0, 0);
  if (iReturn != 0)
    {
     dprintf1("Error creating a VDM connection: %u", iReturn);
     return(DS_OK);
    }
  /* up to here nothing sent on the network */

  if (ds->mcaAIM.icb_address == 0x88) {

    /* Call init_module method to open the network connection to the module */
    if (init_module(ds) != DS_OK) {
      ds->devserver.state = DEVFAULT;
      return (DS_OK);
    }
  } else {
    /* Open the ICB datasource */
    sReturn = SadOpenICB(ds->mcaAIM.hDSC, ds->mcaAIM.aim_file,
			 ds->mcaAIM.icb_address,
			 ACC_Exclusive|ACC_SysWrite|ACC_ReadWrite, "");
    if (sReturn == CSI_Verify) {
      dprintf1("Error failed to verify %s ICB settings.", ds->mcaAIM.aim_file);
      return(DS_OK);
    } else if (sReturn != 0) {
      sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf("Error opening ICB: %lx", ulError);
      return(DS_OK);
    }

    /* get the ICB parameters */
    sReturn = SadInfoICB(ds->mcaAIM.hDSC, &stInfoICB);
    if (sReturn != 0) {     
      SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf("Failed to get ICBinfo %lx", ulError);
      return (DS_OK);
    }
    dprintf("ICB module: model %d", stInfoICB.bModel);
  }

  /*stop the acquisition */
  sReturn = SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_AbortAcq);
  if (sReturn != 0)
  {
   dprintf1("Failed to stop acquisition");
   return(DS_OK);
  }

  /* get some parameters: */
  /* the filename used */
  sReturn = SadQueryDSC(ds->mcaAIM.hDSC, DSC_NameOfDS, &stData);
  if (sReturn) {
    dprintf1("Failed the query");
    return (DS_OK);
  }
  dprintf2 ("name %s", stData.szDSName);

  /* current member of a multigroup */
  sReturn = SadQueryDSC(ds->mcaAIM.hDSC, DSC_CurrentMem, &stData);
  if (sReturn) {
    dprintf1("Failed the query");
    return (DS_OK);
  }
   
  ds->mcaAIM.curr_mem_gr = stData.usCurrent;
  /* total number of groups  */
  sReturn = SadQueryDSC(ds->mcaAIM.hDSC, DSC_CountMem, &stData);
  if (sReturn) {
	  dprintf1("Failed the query");
	  return (DS_OK);
  }
   
  ds->mcaAIM.conf_nb_groups = stData.usInputs;

/************************************************/
  /* the number of channels configured */
  sReturn = SadGetParam (ds->mcaAIM.hDSC, CAM_L_CHANNELS,
			 1, 1, &ds->mcaAIM.ch_per_group, sizeof(long));
  if (sReturn != 0) {     
    SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
    dprintf1("Failed to get nb of channels %lx", ulError);
    return (DS_OK);
  }
  ds->mcaAIM.channel_num = ds->mcaAIM.ch_per_group *
    ds->mcaAIM.conf_nb_groups;


  /* save the channel number for the first ADC */
  if (ds->mcaAIM.curr_adc == 1) {
    part1_ch_num = ds->mcaAIM.channel_num;
    ds->mcaAIM.first_channel = (unsigned long)0;
    ds->mcaAIM.last_channel = (unsigned long)ds->mcaAIM.ch_per_group;
  }

  /* check if 2 ADCs on the same AIM module */
  /* check if the 2 ADCs have different ordinal numbers */
  if (adc_num == ds->mcaAIM.curr_adc) {
    *error = DevErr_McaAIMADCNotSetup;
    return (DS_NOTOK);
  }
  /* check if the memory asked for the ADCs is in the limits of the AIM */
  if (ds->mcaAIM.curr_adc == 2) {
	 if ((part1_ch_num + ds->mcaAIM.channel_num) > 65536) {
       *error = DevErr_McaAIMADCNotSetup;
       return (DS_NOTOK);
	 } else {
       ds->mcaAIM.first_channel = (unsigned long)part1_ch_num;
       ds->mcaAIM.last_channel = (unsigned long)(part1_ch_num +
					    ds->mcaAIM.ch_per_group);
	 }
  }

  ds->devserver.state = DEVON;
  ds->devserver.n_state = DEVON;

  dprintf2("Resources:");
  dprintf2(" aim_file: %s", ds->mcaAIM.aim_file);
  dprintf2(" nb_groups: %d", ds->mcaAIM.conf_nb_groups);
  dprintf2(" ch_per_group %d", ds->mcaAIM.ch_per_group);
  dprintf2(" channel_num: %d", ds->mcaAIM.channel_num);
  dprintf2(" curr_adc: %d", ds->mcaAIM.curr_adc);
  dprintf2(" curr_mem_gr: %d", ds->mcaAIM.curr_mem_gr);
/*  dprintf2(" dev_num: %d", dev_num);
  dprintf2(" tmp_aim_address: %x", tmp_aim_addr);
  dprintf2(" part1_ch_num: %d", part1_ch_num); */

  double_table[0] = ds->mcaAIM.mode;
  double_table[1] = ds->mcaAIM.total_counts;
  double_table[2] = ds->mcaAIM.pres_live_time;
  double_table[3] = ds->mcaAIM.pres_real_time;
  darray_argin.length = 4;
  darray_argin.sequence = double_table;
  /* init the acquisition with the resource values */
  mcapreset (ds, &darray_argin, NULL, error);
  sReturn = SadFlush(ds->mcaAIM.hDSC);
  sReturn = SadQueryDataSource(ds->mcaAIM.hDSC, DSQ_Status, &stInfo);
  if (sReturn) {
    dprintf1("Error hardware query failed.");
    return(DS_NOTOK);
  }

  /* Cleanup and exit */
  /*  sReturn = SadCloseDataSource(ds->mcaAIM.hDSC);
    sReturn = SadDeleteDSC(ds->mcaAIM.hDSC); */
  dprintf("object_initialise exit");
  return (DS_OK);
}


/*_________________________________________________________________________

  Function:      static long state_handler()
  Description:	this routine is reserved for checking wether the command
		requested can be executed in the present state.
  Arg(s) In:	McaAIM ds - device on which command is to executed
		DevCommand cmd - command to be executed
  Arg(s) Out:	long *error - pointer to error code, in case routine fails
 ___________________________________________________________________________*/

static long state_handler(McaAIM ds, DevCommand cmd, long *error)
{
   long iret = DS_OK;
   long p_state, n_state;

   p_state = ds->devserver.state;
   *error = 0;
/*
 * before checking out the state machine assume that the state
 * doesn't change i.e. new state == old state
 */
   n_state = p_state;

   dprintf2 ("enter cmd %d, state %d", cmd, p_state);

   switch (p_state) {
   /* Device is on, all the commands can be sent */
   case (DEVON) :
      switch (cmd) {
        /* Allowed Commands - all */
	    case (DevMcaStartAcq): n_state = DEVRUN; break;
        case (DevMcaSpecialStart): n_state = RUNNING; break;
        default : break;
	  }
	break;
   /* DEVINIT - not enough memory */
   case (DEVINIT) :
      switch (cmd) {
        /* Allowed Commands - all except SpecialStart */
	   case (DevMcaSpecialSetup): n_state = DEVON; break;
 	   /* Forbidden command(s) */
	   case (DevMcaSpecialStart):
	     *error = DevErr_AttemptToViolateStateMachine;
	     iret = DS_NOTOK;
	   break;
	  }
	break;

   /* Device is fault when it can't transmit message */
   case (DEVFAULT) :
	 switch (cmd) {
	     /* Allowed Command(s) */
	   case (DevReset):  n_state = DEVON; break;
	   case (DevStatus):
	   case (DevState):
	   case (DevMcaStopAcq):
	   case (DevMcaClearMemory):
	   case (DevMcaReadData):
	   case (DevMcaGetAdcInfo):
	   case (DevMcaGetStatus):
	   case (DevMcaSpecialStop):
	     break;

	   /* Forbidden command(s) */
	   case (DevMcaSpecialSetup):
       case (DevMcaSpecialStart):
       case (DevMcaSpecialRead):
	   case (DevMcaSetMemoryGroupSize):
	   case (DevMcaSelectMemoryGroup):
	   case (DevMcaSelectTimeMode):
	   case (DevMcaSelectStopMode):
	   case (DevMcaPresetTime):
	   case (DevMcaPresetCountValue):
	   case (DevMcaPreset):
	   case (DevMcaStartAcq):
	     *error = DevErr_AttemptToViolateStateMachine;
	     iret = DS_NOTOK;
	   break;
	   }
	break;

	/* start acquisition has been launched */
   case (DEVRUN):
	  switch (cmd) {
	      /* Allowed Command(s) */
	    case (DevReset):
	    case (DevMcaStopAcq): n_state = DEVON; break;
	       
	       /* Forbidden Command(s) */
	    case (DevMcaSetMemoryGroupSize):
	    case (DevMcaSelectMemoryGroup):
	    case (DevMcaSelectTimeMode):
	    case (DevMcaSelectStopMode):
	    case (DevMcaPresetTime):
		case (DevMcaPresetCountValue):
	    case (DevMcaPreset):
          dprintf1(" DEVRUN violation");
	      *error = DevErr_AttemptToViolateStateMachine;
	      iret = DS_NOTOK;
	    break;
	}
	break;
       	/* special acquisition has been launched */
   case (RUNNING):
	  switch (cmd) {
	      /* Allowed Command(s) */
	    case (DevReset):
	    case (DevMcaStopAcq):
	    case (DevMcaSpecialStop): n_state = DEVON; break;
	       
	       /* Forbidden Command(s) */
	    case (DevMcaSpecialSetup):
        case (DevMcaSpecialStart):
        case (DevMcaSetMemoryGroupSize):
	    case (DevMcaSelectMemoryGroup):
	    case (DevMcaSelectTimeMode):
	    case (DevMcaSelectStopMode):
	    case (DevMcaPresetTime):
		case (DevMcaPresetCountValue):
	    case (DevMcaPreset):
	    case (DevMcaStartAcq):
		case (DevMcaClearMemory):
	    case (DevMcaReadData):
	    case (DevMcaSendICB):
		case (DevMcaReadICB):
	    case (DevMcaSetICBAddr):
          dprintf1("RUNNING violation");
	      *error = DevErr_AttemptToViolateStateMachine;
	      iret = DS_NOTOK;
	    break;
	}
	break;
      default:
	*error = DevErr_UnrecognisedState;
	iret = DS_NOTOK;
      break;
   }

/*  if (iret == DS_NOTOK) */
  dprintf2 ("exit cmd %d, state %d", cmd, n_state);

  ds->devserver.n_state = n_state;

   return(iret);
}

/*________________________________________________________________________
 Function:      static long mcareset()

 Description:	Reset Mca to a well known state.

 Arg(s) In:	McaAIM ps - object on which to execute command.
		DevVoid *argin - void.

 Arg(s) Out:	DevVoid *argout - void.
		long *error - pointer to error in the case of failure.
 __________________________________________________________________________*/
static long mcareset (McaAIM ds, DevVoid *argin, DevVoid *argout,
					   long *error)
{
  /* Cleanup */
  sReturn = SadCloseDataSource(ds->mcaAIM.hDSC);
  sReturn = SadDeleteDSC(ds->mcaAIM.hDSC);

  /* Establish a connection to the local VDM */
  iReturn = iUtlCreateFileDSC2(&ds->mcaAIM.hDSC, 0, 0);
  if (iReturn != 0){
     dprintf1("Error creating a VDM connection: %u", iReturn);
     return(DS_OK);
  }

  /* up to here nothing sent on the network */

  if (ds->mcaAIM.icb_address == 0x88) {

     /* Call init_module method to open the network connection to the module */
     if (init_module(ds) != DS_OK) {
      ds->devserver.state = DEVFAULT;
      return (DS_OK);
     }
  } else {
   /* Open the ICB datasource */
     sReturn = SadOpenICB(ds->mcaAIM.hDSC, ds->mcaAIM.aim_file,ds->mcaAIM.icb_address,
              ACC_Exclusive|ACC_SysWrite|ACC_ReadWrite, "");
     if (sReturn == CSI_Verify) {
       dprintf1("Error failed to verify %s ICB settings.", ds->mcaAIM.aim_file);
       return(DS_OK);
	 } else if (sReturn != 0) {
       sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
       dprintf1("Error opening ICB: %lx", ulError);
       return(DS_OK);
	 }
  }


  /*stop the acquisition */
  sReturn = SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_AbortAcq);
  if (sReturn != 0) {
   dprintf1("Failed to stop acquisition");
   return(DS_OK);
  }

  double_table[0] = ds->mcaAIM.mode;
  double_table[1] = ds->mcaAIM.total_counts;
  double_table[2] = ds->mcaAIM.pres_live_time;
  double_table[3] = ds->mcaAIM.pres_real_time;
  darray_argin.length = 4;
  darray_argin.sequence = double_table;
  /* init the acquisition with the resource values */
  mcapreset (ds, &darray_argin, NULL, error);
  ds->devserver.state =  ds->devserver.n_state;
  return (DS_OK);
}

/*_________________________________________________________________________
 Function:      static long mcaselectadc()

 Description:	 Select the current adc
   	
 Arg(s) In:	 McaAIM   ds 	  - 
		 DevShort *argin  - current adc (1 or 2)
 Arg(s) Out:	 DevVoid  *argout - None
		 long	  *error  - pointer to error code, in case
		                    routine fails. Error code(s):
				    DevErr_DeviceIllegalParameter
 _________________________________________________________________________ */

static long mcaselectadc(McaAIM ds, DevShort *argin, DevVoid  *argout,
			     long *error)
{
 /* command not used any more */
  return(DS_OK);
}

/*_________________________________________________________________________

 Function:    static long mcasetmemorygroupsize()
 Description: Select number of channels per memory group 
 Arg(s) In:   McaAIM            ds      -
              DevLong          *argin  - asked number of channels per group
 Arg(s) Out:  DevVarUShortArray  *argout - 
                                [0] - real number of channels per group
				[1] - calculated number of memory groups
              long              *error  - pointer to error code, in case
                                          routine fails. Error code(s):
					  DevErr_DeviceIllegalParameter
 __________________________________________________________________________*/

static long mcasetmemorygroupsize (McaAIM ds, DevLong *argin,
				       DevVarLongArray *argout, long *error)
{
  static long array[2];  /* for the argout */
  DevShort dummy;        /* used for intermediate calculations */
  long n;                
  
  dprintf2("McaAIM, mcasetmemorygroupsize(), enter");

  if ((*argin > ds->mcaAIM.channel_num) || (*argin <= 0))
    {
      *error = DevErr_DeviceIllegalParameter;
      return (DS_NOTOK);
    }
  
  /* forsing the number of channels per group to be on the basis of 512 */
  if ((*argin <= 511) && (*argin > 0))
    ds->mcaAIM.ch_per_group = 512;
  else if (*argin <= ds->mcaAIM.channel_num)
    {
      n = 16;
      dummy = FALSE;
      while(dummy == FALSE)
	{
	  if ((*argin > (long)(1 << (n-1))) && 
	      (*argin <= (long)(1 << n)))
	    {
	      ds->mcaAIM.ch_per_group = (long)(1 << n);
	      dummy = TRUE;
	    }
	  else
	    n--;
	}
    }

  /* calculate the possible number of groups */
  ds->mcaAIM.nb_groups = (ds->mcaAIM.channel_num /
	   (ds->mcaAIM.ch_per_group - 1));

  array[0] = ds->mcaAIM.ch_per_group;
  array[1] = ds->mcaAIM.nb_groups;

  argout->sequence = array;
  argout->length = 2;
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcasetmemorygroupsize(), exit, %d", ds->devserver.state);

  return(DS_OK);
}

/*_________________________________________________________________________

 Function:    static long mcaselectmemorygroup()
 Description: Select the current memory group
 Arg(s) In:   McaAIM   ds      -
              DevShort *argin  - number of the memory group
 Arg(s) Out:  DevVoid  *argout - none
              long     *error  - pointer to error code, in case
                                 routine fails. Error code(s):
                                 DevErr_DeviceIllegalParameter 
 __________________________________________________________________________*/

static long mcaselectmemorygroup (McaAIM ds, DevShort *argin,
				      DevVoid  *argout, long *error)
{

  dprintf2("McaAIM, mcaselectmemorygroup(), enter %d", ds->devserver.state);

  *error = 0;

  if (ds->mcaAIM.nb_groups == 0)
    {
      *error = DevErr_DeviceIllegalParameter;
      return (DS_NOTOK);
    }
  if ((*argin <=0) || (*argin > (short)ds->mcaAIM.nb_groups))
    {
      *error = DevErr_DeviceIllegalParameter;
      return (DS_NOTOK);
    }
  if (ds->mcaAIM.curr_adc == 1)
    ds->mcaAIM.first_channel = (*argin - 1) * ds->mcaAIM.ch_per_group;
  else
    ds->mcaAIM.first_channel = part1_ch_num + (*argin -1) *
	                           ds->mcaAIM.ch_per_group;
  ds->mcaAIM.last_channel = ds->mcaAIM.first_channel +
	  ds->mcaAIM.ch_per_group;
  ds->mcaAIM.curr_mem_gr = (USHORT)*argin;

  dprintf3 ("curr_adc %d", ds->mcaAIM.curr_adc);
  dprintf3 ("first_channel %d",ds->mcaAIM.first_channel);
  dprintf3 ("last_channel %d",ds->mcaAIM.last_channel);

  /* do the selection only if more than one group */
  if (ds->mcaAIM.conf_nb_groups > 1) {
    sReturn = SadSetCurrent(ds->mcaAIM.hDSC,0,ds->mcaAIM.curr_mem_gr);
    if (sReturn) {
      dprintf1("Error failed to set current group");
      sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf1("Error setCurrent: %lx", ulError);
      *error = DevErr_McaAIMModNotSetup;
      return(DS_NOTOK);
    }
  }

  /* current member of a multigroup */
  sReturn = SadQueryDSC(ds->mcaAIM.hDSC, DSC_CurrentMem, &stData);
  if (sReturn) {
    dprintf1("Failed the query");
    return (DS_OK);
  }
  dprintf3 ("current group %d", stData.usCurrent);

  /* the filename used */
  sReturn = SadQueryDSC(ds->mcaAIM.hDSC, DSC_NameOfDS, &stData);
  if (sReturn) {
    dprintf1("Failed the query");
    return (DS_OK);
  }

  dprintf3 ("name %s", stData.szDSName);

  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcaselectmemorygroup(), exit %d", ds->devserver.state);

  return (DS_OK);
}

/*_________________________________________________________________________

 Function:    static long mcaselecttimemode()
 Description: dummy command for complience with the Silena MCA only
 Arg(s) In:   McaAIM    ds     -
              DevShort  *argin - 1 (live time), 2 (real time)
 Arg(s) Out:  DevVoid   *argout - none
              long      *error  - pointer to error code, in case
                                  routine fails. Error code(s):
 __________________________________________________________________________*/

static long mcaselecttimemode(McaAIM ds, DevShort *argin,
				  DevVoid *argout, long *error)
{
  dprintf2("McaAIM, mcaselecttimemode(), enter");

  *error = 0;
  if ((*argin < 1) || (*argin >2))
    {
      *error = DevErr_DeviceIllegalParameter;
      return(DS_NOTOK);
    }
  ds->mcaAIM.mode = *argin;
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcaselecttimemode(), exit");

  return(DS_OK);
}

/*_________________________________________________________________________

 Function:    static long mcaselectstopmode()
 Description: dummy command for complience with the Silena MCA only
 Arg(s) In:   McaAIM           ds      -
              DevVarShortArray *argin  -
	                       [0] - time: 0 (not active), 1 (active)
			       [1] - counts: 0 (not active), 1 (active)
 Arg(s) Out:  DevVoid          *argout - none
              long             *error  - pointer to error code, in case
                                         routine fails. Error code(s):
                                    DevErr_DeviceIllegalParameter
 __________________________________________________________________________*/

static long mcaselectstopmode(McaAIM ds, DevVarShortArray *argin,
				  DevVoid *argout, long *error)
{
  dprintf2("McaAIM, mcaselectstopmode(), enter");

  *error = 0;

  if ((argin->sequence[0] > 1) || (argin->sequence[1] > 1) ||
      (argin->sequence[0] < 0) || (argin->sequence[1] < 0))
    {
      *error = DevErr_DeviceIllegalParameter;
      return(DS_NOTOK);
    }

  /* time mode only */
  if ((argin->sequence[0] == 1) && (argin->sequence[1] == 0))
    {
     ds->mcaAIM.total_counts = (unsigned long)0;
     switch(ds->mcaAIM.mode)
	 {
	   case 1:
	     ds->mcaAIM.pres_real_time = (unsigned long)0;
	   break;
 	  case 2:
	   ds->mcaAIM.pres_live_time = (unsigned long)0;
	   break;
 	default:
 	  *error = DevErr_DeviceIllegalParameter;
 	  return(DS_NOTOK);
 	  break;
	 }

  }
  else if ((argin->sequence[1] == 1) && (argin->sequence[0] == 0))
  {
   ds->mcaAIM.mode = TOT_COUNTS; 
   ds->mcaAIM.pres_live_time = (unsigned long)0;
   ds->mcaAIM.pres_real_time = (unsigned long)0;
  }
  else if ((argin->sequence[0] == 0) && (argin->sequence[1] == 0))
  {
   ds->mcaAIM.mode = ANY_MODE;
   ds->mcaAIM.total_counts = (unsigned long)0;
   ds->mcaAIM.pres_live_time = (unsigned long)0;
   ds->mcaAIM.pres_real_time = (unsigned long)0;
  }
  double_table[0] = ds->mcaAIM.mode;
  double_table[1] = ds->mcaAIM.total_counts;
  double_table[2] = ds->mcaAIM.pres_live_time;
  double_table[3] = ds->mcaAIM.pres_real_time;
  darray_argin.length = 4;
  darray_argin.sequence = double_table;
  /* init the acquisition with the resource values */
  mcapreset (ds, &darray_argin, NULL, error);
 ds->devserver.state = ds->devserver.n_state;

  dprintf2("McaAIM, mcaselectstopmode(), exit");
  return(DS_OK);
}

/*_________________________________________________________________________

 Function:    static long mcapresettime()
 Description: preset time [s] when stop mode is time
 Arg(s) In:   McaAIM            ds      -
              DevVarDoubleArray *argin  - 
	                         [0] - preset live time [s]
	                         [1] - preset real time [s]
 Arg(s) Out:  DevVoid           *argout - none
              long              *error  - none
 __________________________________________________________________________*/

static long mcapresettime (McaAIM ds, DevVarDoubleArray *argin,
			      DevVoid *argout, long *error)

{
  dprintf2("McaAIM, mcapresettime(), enter");

  ds->mcaAIM.pres_live_time = argin->sequence[0];
  ds->mcaAIM.pres_real_time = argin->sequence[1];

  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcapresettime(), exit");
  return(DS_OK);
}

/*_________________________________________________________________________

 Function:    static long mcapreset()
 Description: preset the MCA
 Arg(s) In:   McaAIM            ds      -
              DevVarDoubleArray *argin  -
	                         [0] - counting mode (0-counts,
							       1-live time, 2-real time, 3-any)
	                         [1] - preset counts [nb]
	                         [2] - preset live time [s]
	                         [3] - preset real time [s]
 Arg(s) Out:  DevVoid           *argout - none
              long              *error  - none
 __________________________________________________________________________*/

static long mcapreset (McaAIM ds, DevVarDoubleArray *argin,
			      DevVoid *argout, long *error)

{
  dprintf2("McaAIM, mcapreset(), enter");

  ds->mcaAIM.mode = (short)argin->sequence[0];
  ds->mcaAIM.total_counts = (unsigned long)argin->sequence[1];
  ds->mcaAIM.pres_live_time = argin->sequence[2];
  ds->mcaAIM.pres_real_time = argin->sequence[3];

  dprintf3 ("%d, %d, %f, %f", ds->mcaAIM.mode, ds->mcaAIM.total_counts,
	    ds->mcaAIM.pres_live_time, ds->mcaAIM.pres_real_time);

  /* prepare the acq parameters */
  /*
    FillMemory (&stPSet, sizeof(Preset_T), '\0'); */
  switch (ds->mcaAIM.mode)
    {
    case 0:
      /*   mode = CAM_M_PMOD_TOTAL; */
      ds->mcaAIM.pres_live_time = 0;
      ds->mcaAIM.pres_real_time = 0; 
      break;
    case 1:
      /*   mode = CAM_M_PMOD_LIVE;
	   stPSet.unTime.dPsetTime = ds->mcaAIM.pres_live_time; */
      ds->mcaAIM.total_counts = 0;
      ds->mcaAIM.pres_real_time = 0;
      break;
    case 2:
      /*    mode = CAM_M_PMOD_REAL;
	    stPSet.unTime.dPsetTime = ds->mcaAIM.pres_real_time; */
      ds->mcaAIM.total_counts = 0;
      ds->mcaAIM.pres_live_time = 0;
      break;
    case 3:
      /*    mode = CAM_M_PMOD_REAL;
	    stPSet.unTime.dPsetTime = ds->mcaAIM.pres_real_time; */
      ds->mcaAIM.total_counts = 0;
      break;
    default:
      *error = DevErr_DeviceIllegalParameter;
      return(DS_NOTOK);
    }
  /*
    stPSet.flPsetMode = mode;
    sReturn = SadPutStruct(ds->mcaAIM.hDSC, WST_SetPreset, 1, 1, &stPSet, sizeof(Preset_T));
    if (sReturn) {
    dprintf1("Failed to set acq mode");
    *error = DevErr_McaAIMInvalidAcqMode;
    return (DS_NOTOK);
    } */
  /* add here the total counts */
  stAcq.dPsetLive = ds->mcaAIM.pres_live_time;
  stAcq.dPsetTrue = ds->mcaAIM.pres_real_time;
 
  sReturn = SadPutStruct (ds->mcaAIM.hDSC, WST_AcqPage, 1, 1, &stAcq, sizeof(AcqPage_T));
  if (sReturn) {
    dprintf1("Error failed to set acq");
    *error = DevErr_McaAIMModNotSetup;
    return(DS_NOTOK);
  } 
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcapreset(), %d exit", ds->devserver.state);
  return(DS_OK);
}



/*_________________________________________________________________________

 Function:    static long mcapresetcountvalue()
 Description: Preset the number of counts
 Arg(s) In:   McaAIM    ds      -
              DevULong  *argin  - number of counts asked
 Arg(s) Out:  DevULong  *argout - real number of counts
              long      *error  - pointer to error code, in case
                                  routine fails. Error code(s):
                                  
 __________________________________________________________________________*/

static long mcapresetcountvalue (McaAIM ds, DevLong *argin,
				     DevLong *argout, long *error)
{
  dprintf2("McaAIM, mcapresetcountvalue(), enter");

  ds->mcaAIM.total_counts = *argin;
  ds->mcaAIM.mode = 0;
  *argout = ds->mcaAIM.total_counts;
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcapresetcountvalue(), exit");
  return(DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcastartacq()
  Description:	 Initiates Data Acquisition of the ADC
  Arg(s) In:	 McaAIM  ds 	 - 
		 DevVoid *argin  - None
  Arg(s) Out:	 DevVoid *argout - None
		 long	 *error	 - pointer to error code, in case
		                   routine fails. Error code(s):
				   NMC__READIOERR
 _________________________________________________________________________*/

static long  mcastartacq(McaAIM ds, DevVoid *argin,
			 DevVoid *argout, long *error)
{
  dprintf2("McaAIM, mcastartacq(), enter");

  if (ds->devserver.state == DEVRUN)
    mcastopacq (ds, NULL, NULL, error);

  /* clear the buffer */
  /*  mcaclearmemory (ds, NULL, NULL, error); */

  sReturn = SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_StartAcq);
  if (sReturn != 0) { 
    ds->devserver.state = DEVFAULT;
    sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
    dprintf1("Failed to start acquisition: %lx", ulError);
    return(DS_NOTOK);
  }
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcastartacq(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:    static long mcastopacq()
  Description: Stops Data Acquisition of the ADC
  Arg(s) In:   McaAIM  ds      - 
	       DevVoid *argin  - None
  Arg(s) Out:  DevVoid *argout - None
	       long    *error  - pointer to error code, in case
		 		 routine fails. Error code(s):
				 NMC__READIOERR
 _________________________________________________________________________*/

static long  mcastopacq(McaAIM ds, DevVoid *argin, DevVoid *argout,
			long *error)
{
  dprintf2("McaAIM, mcastopacq(), enter");

  if (ds->devserver.state != RUNNING) {
    /* stop the acquisition */
    sReturn = SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_AbortAcq);
    if (sReturn != 0) {
      sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf1("Failed to stop acquisition: %lx", ulError);
      return(DS_NOTOK);
    }
  } else {
    if (SpThreadStopped == 0) {
      SpStopThread = 1;
      while (SpThreadStopped == 0);
    }
  }
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcastopacq(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:    static long mcaclearmemory()
  Description: Clears the memory assigned to the current ADC
               (only the selected group)
  Arg(s) In:   McaAIM  ds      - 
	       DevVoid *argin  - None
  Arg(s) Out:  DevVoid *argout - None
	       long    *error  - pointer to error code, in case
		 	 	 routine fails. Error code(s):
				 NMC__READIOERR
 _________________________________________________________________________*/

static long  mcaclearmemory(McaAIM ds, DevVoid *argin,
				DevVoid *argout, long *error)
{
  sReturn = SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_ClearData);
  if (sReturn != 0)
  {
      dprintf1("Failed to clear acquisition memory");
	  return (DS_NOTOK);
  }
  ds->devserver.state =  ds->devserver.n_state;
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcareaddata()
  Description:	 Read the MCA memory segment selected by the memory group.
  Arg(s) In:	 McaAIM 	  ds 	  - 
		 DevVarLongArray *argin   - 
		                  [0] the first channel in the group
		                  [1] the last channel in the group
				  [2] the memory group to read
 Arg(s) Out:	 DevVarULongArray *argout - transfered data
		 long		  *error  - pointer to error code, in case
		 			    routine fails. Error code(s):
					    DevErr_DeviceIllegalParameter
 _________________________________________________________________________*/

static long  mcareaddata(McaAIM ds, DevVarLongArray *argin,
			     DevVarULongArray *argout, long *error)
{
  static unsigned long *mem_buffer;
  USHORT startCh, offsetCh;
  unsigned long cur_chans,   /* current num of returned channel */
    chans_left,  /* number of channel to be returned */
    max_chans,   /* Max channel to be returned */
    address,    /* start address of returned memory */
    size;      /* totals channel to be returned */
  short flag = 0;  		    
  *error = 0;

  dprintf2("McaAIM, mcareaddata(), enter");

  /* if ((argin->sequence[0] > (ds->mcaAIM.ch_per_group - 1)) ||
      (argin->sequence[1] > (ds->mcaAIM.ch_per_group - 1)) ||
      (argin->sequence[0] < 0) || (argin->sequence[1] < 0) ||
      (argin->sequence[0] > argin->sequence[1])) {
    *error = DevErr_DeviceIllegalParameter;
    return (DS_NOTOK);
  } */

  /* check if the new group is correct */
  if (argin->length >= 3) {
    if ((argin->sequence[2] > ds->mcaAIM.nb_groups) ||
	(argin->sequence[2] <= 0)) {
      dprintf1 ("%d", argin->sequence[2]);
      *error = DevErr_DeviceIllegalParameter;
      return (DS_NOTOK);
    }
    dprintf3 ("curr %d, %d", ds->mcaAIM.curr_mem_gr, argin->sequence[2]);
    /* check if the new group is not the same as current */
    if (ds->mcaAIM.curr_mem_gr != argin->sequence[2]) {
      /* change the group */
      if (ds->mcaAIM.conf_nb_groups > 1) {
		 flag = 1;
	dprintf("changing the groups");
	if (SadSetCurrent(ds->mcaAIM.hDSC, 0, (unsigned short)argin->sequence[2]) != 0) {
	  dprintf1("Error failed to set current group");
	  sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
	  dprintf1("Error setCurrent: %lx", ulError);
	  *error = DevErr_McaAIMModNotSetup;
	  return(DS_NOTOK);
	}
      }
    }
  }
#ifdef COM
  /* check if a new group is asked */
  if (argin->length < 3)
    address = (unsigned long)(ds->mcaAIM.first_channel + argin->sequence[0] + 1);
  else if (ds->mcaAIM.curr_adc == 1)
    address = (unsigned long)(((argin->sequence[2] - 1) * ds->mcaAIM.ch_per_group)
			      + argin->sequence[0] +1);
  else
    address = (unsigned long)(((argin->sequence[2] - 1) * ds->mcaAIM.ch_per_group)
			      + argin->sequence[0] + part1_ch_num +1);
#endif 
  address = argin->sequence[0] + 1;
 
 
  max_chans = 4000;
  size = (USHORT)(argin->sequence[1] - argin->sequence[0] +1);
  chans_left = size;

  dprintf3 ("address %d, %d",address,size);

  /*calculate the memory buffer pointer*/
  argout->sequence = (unsigned long *)ds->mcaAIM.acq_mem +
    ds->mcaAIM.first_channel + argin->sequence[0]; 
  mem_buffer = (unsigned long *)argout->sequence; 
  while (chans_left > 0) {
    cur_chans = chans_left;
    if (cur_chans > max_chans)
      cur_chans = max_chans;
    startCh = (USHORT)(address + (size - chans_left));
    offsetCh = (USHORT)cur_chans;
    sReturn = SadGetSpectrum(ds->mcaAIM.hDSC, startCh, offsetCh,
			     FALSE, mem_buffer);
    if (sReturn != 0) {
      sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf1("Error failed to read spectrum %lx", ulError);
      return(DS_NOTOK);
    }
    chans_left -= offsetCh;
    mem_buffer += offsetCh;
  }
  argout->length = argin->sequence[1] - argin->sequence[0] + 1;
  ds->devserver.state =  ds->devserver.n_state;

  /* change back the group, if needed */
  if (flag == 1) {
	dprintf3("changing back the groups");
	if (SadSetCurrent(ds->mcaAIM.hDSC, 0, (unsigned short)ds->mcaAIM.curr_mem_gr ) != 0) {
	  dprintf1("Error failed to set current group");
	  sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
	  dprintf1("Error setCurrent: %lx", ulError);
	  *error = DevErr_McaAIMModNotSetup;
	  return(DS_NOTOK);
	}
  }
  dprintf2("McaAIM, mcareaddata(), exit");
  return (DS_OK);
}
/*_________________________________________________________________________

  Function:    static long mcagetadcinfo()
  Description: Gives information about the acquisition in process
  Arg(s) In:   McaAIM 	         ds      - 
	       DevVoid           *argin  - None
  Arg(s) Out:  DevVarDoubleArray *argout - 
 				 [0] - current ADC
				 [1] - first channel
				 [2] - last channel
				 [3] - preset live time (s)
				 [4] - preset real time (s) 
				 [5] - preset totals counts 
				 [6] - elapsed real time (s)
				 [7] - elapsed real time (s)
		 long		 *error	 - pointer to error code, in case
		                           routine fails. Error code(s):

 _________________________________________________________________________*/
static long  mcagetadcinfo(McaAIM ds, DevVoid *argin,
			       DevVarDoubleArray *argout, long *error)
{
  static double argout_table[10];
  dprintf2("McaAIM, mcagetadcinfo(), enter");
  *error = 0;

  sReturn = SadGetParam(ds->mcaAIM.hDSC, CAM_X_ELIVE, 0, 0,
	  &ds->mcaAIM.live_time, sizeof(double));
  if (sReturn != 0)
  {
      SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf1("Failed to read time%lx", ulError);
	  return (DS_NOTOK);
  }
  sReturn = SadGetParam(ds->mcaAIM.hDSC, CAM_X_EREAL, 0, 0,
	  &ds->mcaAIM.real_time, sizeof(double));
  if (sReturn != 0)
  {
      SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
      dprintf1("Failed to read time%lx", ulError);
	  return (DS_NOTOK);
  }

  argout_table[0] = (double)ds->mcaAIM.curr_adc;
  argout_table[1] = (double)ds->mcaAIM.first_channel;
  argout_table[2] = (double)ds->mcaAIM.last_channel;
  argout_table[3] = (double)ds->mcaAIM.pres_live_time;
  argout_table[4] = (double)ds->mcaAIM.pres_real_time;
  argout_table[5] = (double)ds->mcaAIM.total_counts;
  argout_table[6] = ds->mcaAIM.live_time;
  argout_table[7] = ds->mcaAIM.real_time;
  
  argout->sequence = argout_table;
  argout->length = 8;

  ds->devserver.state =  ds->devserver.n_state;
 
  dprintf2("McaAIM, mcagetadcinfo(), exit");
  return (DS_OK);
}
/*_________________________________________________________________________

  Function:      static long mcagetstatus()
  Description:	 Gives the MCA ststus
  Arg(s) In:	 McaAIM 	  ds 	  - 
		 DevVoid  	  *argin  - None
  Arg(s) Out:	 DevVarShortArray *argout - 
				  [0] - mode: idle (0) or counting (1)
				  [1] - stop conditions (0-4)
		 long		  *error  - pointer to error code, in case
		                            routine fails. Error code(s):
					    NMC__READIOERR
 _________________________________________________________________________*/

static long  mcagetstatus(McaAIM ds, DevVoid *argin,
			      DevVarShortArray *argout, long *error)
{
  static short table[2];  /* prepare the argout */
  dprintf2("mcagetstatus() enter");
  sReturn = SadQueryDataSource(ds->mcaAIM.hDSC, DSQ_Status, &stInfo);
  if (sReturn) {
      dprintf1("Error hardware query failed.");
      return(DS_NOTOK);
  }
  dprintf3("busy %x", stInfo.stDS.fsStatus);

  if ((stInfo.stDS.fsStatus & DSS_Busy) == DSS_Busy) {
	  if (ds->devserver.state != DEVRUN) {
		  if ((ds->devserver.state != RUNNING) && (SpThreadStopped == 0))
			  ds->devserver.state = DEVFAULT;
	  }
  }
  else if (ds->devserver.state == DEVRUN)
	  ds->devserver.state = DEVON;

  table[0] = (short)(stInfo.stDS.fsStatus & DSS_Busy);
  table[1] = (short)0;  /* not used for this server */
  argout->length = 2;
  argout->sequence = table;

  dprintf2("mcagetstatus() %d", ds->devserver.state);
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:    static long mcasendicb()
  Description: Send commands on the ICB bus
  Arg(s) In:   McaAIM 	        ds
               DevVarULongArray *argin -
	                        [0]  - number of address/data pairs to write
				[1]  - ICB address
				[2]  - data to write
				...
				[64] - ICB address
				[65] - data to write
  Arg(s) Out:  DevVoid          *arout - None
	       long		*error - pointer to error code, in case
		                         routine fails. Error code(s):

 _________________________________________________________________________*/

static long mcasendicb (McaAIM ds, DevVarULongArray *argin,
			DevVoid *argout, long *error)
{
  int i;
  BYTE bRegister, bData;

  dprintf2("McaAIM mcasendicb() enter");

  *error = 0;

  /* write to a register */
  for (i=0; i<(int)*argin->sequence; i++)
  {
	  bRegister = (BYTE)*(argin->sequence + 2*i + 1);
	  bData = (BYTE)*(argin->sequence + (2*i + 1) + 1);
	  sReturn = SadPutICB(ds->mcaAIM.hDSC, bRegister, bData);
	  if (sReturn == CSI_Verify)
	  {
		  dprintf1("Error failed to verify %s ICB settings.",
			  ds->mcaAIM.aim_file);
		  return(DS_NOTOK);
	  }
	  else if (sReturn != 0)
	  {
		  SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
		  dprintf1("Error sending to ICB: %lx", ulError);
		  return(DS_NOTOK);
	  }
  }

  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM mcasendicb() exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:    static long mcarecvicb()
  Description: Read data from the ICB bus
  Arg(s) In:   McaAIM 	        ds
               DevVarULongArray *argout -
	            [0]  - number of addresses to read
				[1]  - ICB address from which data to read
				...
				[65] - ICB address from which data to read
   Arg(s) Out: DevVarULongArray *argout -
	                        [0]  - data 1
				[1]  - data 2
				...
				[64] - data 64
	       long		*error	- pointer to error code, in case
		                          routine fails. Error code(s):

 _________________________________________________________________________*/

static long mcarecvicb (McaAIM ds, DevVarULongArray *argin,
						DevVarULongArray *argout, long *error)
{
  BYTE bRegister;
  static unsigned long out[64];
  int actual=0;
  int i=0;

  dprintf2("McaAIM mcarecvicb() enter");

  *error = 0;
  for (i=0; i<(int)*argin->sequence; i++)
  {
	  bRegister = (unsigned char)*(argin->sequence + i + 1);
	  sReturn = SadGetICB(ds->mcaAIM.hDSC, bRegister, (unsigned char *)&out[i]);
	  if (sReturn == CSI_Verify)
	  {
		  dprintf1("Error failed to verify %s ICB settings.",
			  ds->mcaAIM.aim_file);
		  return(DS_NOTOK);
	  }
	  else if (sReturn != 0)
	  {
		  SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
		  dprintf1("Error reading ICB: %lx %d", ulError, sReturn);
		  return(DS_NOTOK);
	  }
  }

 argout->sequence = out;
 argout->length = *argin->sequence;

  ds->devserver.state =  ds->devserver.n_state; 

  dprintf2("McaAIM mcarecvicb() exit");
  return (DS_OK);
}


/*_________________________________________________________________________

  Function:      static long dev_state()
  Description:	 Gives the MCA state
  Arg(s) In:	 McaAIM  ds 	  - object on which to execute command.
		         DevVoid  *argin  - none
  Arg(s) Out:	 DevShort *argout - state
		         long	  *error  - set eventually in mcagetstatus
 _________________________________________________________________________*/

static long dev_state (McaAIM ds,  DevVoid *argin,
					   DevShort *argout, long *error)
{
  static DevVarShortArray longs;

  dprintf2("McaAIM dev_state() enter");

/* this command can be always executed independent of the state  */
  if (ds->devserver.state != DEVFAULT)
	{
	 if (mcagetstatus(ds, NULL, &longs, error) != DS_OK)
	 return (DS_NOTOK);
    }
  *argout = (short)ds->devserver.state;

  dprintf2("McaAIM dev_state() exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long dev_status()
  Description:	 Gives the MCA state as ASCII string
  Arg(s) In:	 McaAIM    ds 	  - object on which to execute command.
		         DevVoid   *argin  - none
  Arg(s) Out:	 DevString *argout - state
		         long	   *error  - none
 _________________________________________________________________________*/

static long dev_status (McaAIM ds, DevVoid *argin,
						DevString *argout, long *error)
{
	static char mess[1024];
	long p_state;

	dprintf2("McaAIM dev_status() enter");

	p_state = ds->devserver.state;

	switch (p_state) {
	case (DEVON) : sprintf(mess,"%s","On");
	                    break;
	case (DEVINIT) : sprintf(mess,"%s","SpecialSetup failed");
	                    break;
	case (DEVRUN) : sprintf(mess,"%s","Acquiring");
	                  break;
	case (RUNNING) : sprintf(mess,"%s","SpecialAcq");
	                  break;
	case (DEVFAULT) : sprintf(mess,"%s","Fault");
	                  break;
	default : sprintf(mess,"%s","Unknown");
	          break;
	}

	*argout = mess;

	dprintf2("McaAIM dev_state() exit");
	return(DS_OK);
}

/*_________________________________________________________________________

  Function:    static long mcaseticbaddr()
  Description: Open the ICB bus
  Arg(s) In:   McaAIM 	        ds
               DevShort *argin - the ICB bus address (hex)
  Arg(s) Out:  DevVoid          *arout - None
	       long		*error - pointer to error code, in case
		                         routine fails. Error code(s):

 _________________________________________________________________________*/

static long mcaseticbaddr (McaAIM ds, DevShort *argin,
			DevVoid *argout, long *error)
{
  /* check if the ICB datasource has been already open */
  if (ds->mcaAIM.icb_address != *argin)
  {
	dprintf2("opening %x, %x", *argin, ds->mcaAIM.icb_address);
	if ((*error = init_icb(ds, *argin)) != 0) {
		init_module (ds);
		return (DS_NOTOK);
	}
  }
  ds->mcaAIM.icb_address = *argin;
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:     static long init_module()
  Description:  routine to be called on initialisation of the AIM module.
  Arg(s) In:    McaAIM ds       - object
  Arg(s) Out:   long *error     - pointer to error code, in case routine fails
  _________________________________________________________________________*/
static long init_module (McaAIM ds)
{
  /* Open the McaAIM datasource */
   sReturn = SadOpenDataSource(ds->mcaAIM.hDSC, ds->mcaAIM.aim_file, CIF_Detector,
             ACC_Exclusive|ACC_SysWrite|ACC_ReadWrite, TRUE, "");
   if (sReturn == CSI_Verify)
	 {
      dprintf1("Error failed to verify %s parameter settings.", ds->mcaAIM.aim_file);
      return(DevErr_McaAIMIOError);
	}
   else if (sReturn != 0)
	{ 
     sReturn = SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
     dprintf1("Error opening %s: %lx", ds->mcaAIM.aim_file, ulError);
     return(DevErr_McaTimedOut);
	}
  return (DS_OK);
}
/*_________________________________________________________________________

  Function:     static long init_icb()
  Description:  routine to be called on initialisation of the AIM module.
                with ICB
  Arg(s) In:    McaAIM ds  - object
  Return:       long       - error code or DS_OK
  _________________________________________________________________________*/

static long init_icb (McaAIM ds, short address)
{
  /* close the existing link to the module */
  sReturn = SadCloseDataSource(ds->mcaAIM.hDSC);
  if (sReturn == CSI_Verify)
  {
	dprintf1("Error failed to verify %s ICB settings.", ds->mcaAIM.aim_file);
	return(DevErr_McaAIMVerifICB);
  }
  else if (sReturn != 0)
  {
	SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
	dprintf1("Error closing ICB: %lx", ulError);
	return(DevErr_McaAIMCloseDataS);
  }

  /* open the ICB datasource */
  sReturn = SadOpenICB(ds->mcaAIM.hDSC, ds->mcaAIM.aim_file,
	 address, ACC_Exclusive|ACC_SysWrite|ACC_ReadWrite, "");
  if (sReturn == CSI_Verify)
  {
   dprintf1("Error failed to verify %s ICB settings.",
	   ds->mcaAIM.aim_file);
   return(DevErr_McaAIMVerifICB);
  }
  else if (sReturn != 0)
  {
   SadGetStatus(ds->mcaAIM.hDSC, &ulError, &sDum1, &usDum2);
   dprintf1("Error opening ICB: %lx", ulError);
   return(DevErr_McaAIMOpenICB);
  }
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcaspecialacq()
  Description:	 Performs Special Data Acquisition cycle
  Arg(s) In:	 McaAIM  ds 	 - 
		 DevShort *argin  - Nb of times to perform the cycle
  Arg(s) Out:	 DevVoid *argout - None
		 long	 *error	 - pointer to error code, in case
		                   routine fails. Error code(s):
				   NMC__READIOERR
 _________________________________________________________________________*/

static long  mcaspecialacq(McaAIM ds, DevVarULongArray *argin,
						  DevVarULongArray *argout, long *error)
{
  int i;
  unsigned long *ulong_buff;
  unsigned short read_max_size;
  long err;
  long end;

  dprintf2("McaAIM, mcaspecialacq(), enter");

  /* open the LPT1 (address 0x378), using the MMP package functions */
  err = MMPOpen();

  dprintf3("opening LPT1");
  if (err)
	  dprintf1("Cannot open the LPT1");

  argout->sequence = calloc(2*argin->sequence[0]*argin->sequence[1]+1, 
	                        sizeof(unsigned long));
  ulong_buff=argout->sequence;

  /* start the first acquisition */
  read_max_size = 16384;

  /* set CB25 pin 2*/
  MMPOutp(0x378, MMPInp(0x378) | 0x01);

  stAcq.dPsetLive = 0;
  stAcq.dPsetTrue = 0;
  sReturn = SadPutStruct (ds->mcaAIM.hDSC, WST_AcqPage, 1, 1, &stAcq, 
	  sizeof(AcqPage_T));
 
  /* start acquisition */
  SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_StartAcq);
  /* give 35ms to acquire some data */
  Sleep (35);
#ifdef ELAPSEDTIME_1
  _ftime(&start);
#endif
#ifdef ELAPSEDTIME
   beg = clock();
#endif
  /* cycle for predefined nb of times */
  for (i=0; i<(int)argin->sequence[0]; i++) {
	  /* clear CB25 pin 2 */
	  MMPOutp(0x378, MMPInp(0x378) & 0xfe);
      sReturn = SadGetSpectrum(ds->mcaAIM.hDSC, 1,
		  (unsigned short)argin->sequence[1], FALSE, ulong_buff);
	  ulong_buff += argin->sequence[1];
	  dprintf4("i= %d, read %d, 0x%x", i, sReturn, ulong_buff);

     /* set CB25 pin 2*/
     MMPOutp(0x378, MMPInp(0x378) | 0x01);

     sReturn = SadGetSpectrum(ds->mcaAIM.hDSC,
		 read_max_size+1, (unsigned short)argin->sequence[1],
		 FALSE, ulong_buff);
     ulong_buff += argin->sequence[1];
     dprintf4("i= %d, read %d 0x%x", i, sReturn, ulong_buff);
  }
#ifdef ELAPSEDTIME
  end = clock() - beg;
  dprintf("%d ms", end);
#endif
#ifdef ELAPSEDTIME_1
  _ftime(&finish);
  if (start.millitm > finish.millitm) {
	  finish.millitm += 1000;
	  finish.time--;
  }
  dprintf("%d s, %d ms",(finish.time-start.time),
	  (finish.millitm - start.millitm));
#endif

   /*Sleep(1000);*/
  /* finaly stop the acquisition */
  SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_AbortAcq);
  /* clear CB25 pin 2 */
  MMPOutp(0x378, MMPInp(0x378) & 0xfe);

  MMPClose();
  argout->length = 2*argin->sequence[0]*argin->sequence[1]+1;
  argout->sequence[argout->length-1] = (unsigned long)end;
  dprintf4("%d, %ld", argout->length, argout->sequence[argout->length-1]);
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcaspecialacq(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcaspecialsetup()
  Description:	 Setup nb cycle, nb channel and allocate memory
  Arg(s) In:	 McaAIM  ds
                 DevVarULongArray *argin :
				        Nb of times to perform the cycle +
                        Nb of channel per spectrum +
						Mode
  Arg(s) Out:	 DevVoid  *argout :
                        None
		         long	  *error :
				        Pointer to error code, in case
		                routine fails. Error code(s):
 _________________________________________________________________________*/

static long  mcaspecialsetup(McaAIM ds, DevVarULongArray *argin,
							 DevVoid *argout, long *error)
{
  dprintf2("McaAIM, mcaspecialsetup(), enter");

  *error = 0;

  if (SpThreadStopped == 1) {

	  SpNbCycle = argin->sequence[0];
	  if (SpNbCycle <= 0) {
		  dprintf1("  Illegal Parameter : Nb Cycle");
		  *error = DevErr_DeviceIllegalParameter;
		  return (DS_NOTOK);
	  }

	  SpNbCh = (unsigned short) argin->sequence[1];
	  if ((SpNbCh <= 0) || (SpNbCh > 4000)) {
		  dprintf1(" Illegal Parameter : Nb Channel");
		  *error = DevErr_DeviceIllegalParameter;
		  return (DS_NOTOK);
	  }

	  /* mode : 0 -> take exactly SpNbCycle spectrum and stop
	            1 -> wait for parallel port bit 3 to be 1
				     take SpNbCycle spectrum and stop
			    2 -> wait for parallel port bit 3 to be 1
				     take spectrum untill paralell port bit 3
					 go down to 0.
					 if at the end of the SpNbCycle allocated array,
					 go back to the beginning and overwrite data */
	  SpMode = argin->sequence[2];
	  if ((SpMode < 0) || (SpMode > 3)) {
		  dprintf1(" Illegal Parameter : Mode");
		  *error = DevErr_DeviceIllegalParameter;
		  return (DS_NOTOK);
	  }

	  dprintf3(" Memory allocation");
	  
	  if (SpDataBuff != (unsigned long *)NULL) {
		  free(SpDataBuff);
		  SpDataBuff = (unsigned long *)NULL;
	  }
	  SpDataBuff = (unsigned long *)malloc(argin->sequence[0]*argin->sequence[1]*
	                        sizeof(unsigned long));
	  if (SpDataBuff == (unsigned long *)NULL) {
		  dprintf1(" Cannot allocate SpDataBuff");
		  *error = DevErr_InsufficientMemory;
		  ds->devserver.state = DEVINIT;
		  return(DS_NOTOK);
	  }

	  /* open lpt1 */
	  MMPOpen();
	  /* clear PP pin 3 : enable ISG card to look for address change */
	  MMPOutp(0x378, MMPInp(0x378) & 0xfc);
	  /* clear PP pin 2 : set MCA acquisition on low address */
	  MMPOutp(0x378, MMPInp(0x378) & 0xfe);
	  /* set PP pin 3 : diable ISG card to look for address change */
	  MMPOutp(0x378, MMPInp(0x378) | 0x03);
	  /* close lpt1 */
	  MMPClose();
	  SpPPLevel = 0;

  } else {
	  dprintf1(" Special Acquisition Already Running");
	  *error = DevErr_AttemptToViolateStateMachine;
	  return(DS_NOTOK);
  }
  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcaspecialsetup(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcaspecialstart()
  Description:	 Start Data Acquisition cycle
  Arg(s) In:	 McaAIM  ds
                 DevVoid *argin :
				        none
  Arg(s) Out:	 DevVoid  *argout :
                        None
		         long	  *error :
				        Pointer to error code, in case
		                routine fails. Error code(s):
 _________________________________________________________________________*/

static long  mcaspecialstart(McaAIM ds, DevVoid *argin,
							 DevVoid *argout, long *error)
{
  long err;

  dprintf2("McaAIM, mcaspecialstart(), enter");
  
  if (SpThreadStopped == 1) {
	  dprintf3("  opening LPT1");
	  /* open the LPT1 (address 0x378), using the MMP package functions */
	  err = MMPOpen();

	  if (err) {
		  dprintf1("Cannot open the LPT1 %d",err);
		  *error = DevErr_McaAIMLPTNotOpen;
		  return (DS_NOTOK);
	  }
	  dprintf3("  Select first part of the memory");
	  /* set CB25 pin 2 : select first part of the memory*/
	  MMPOutp(0x378, MMPInp(0x378) | 0x01);

	  /* read the paralel port - pin13 (select) base+1, bit 4*/
      ppdata = MMPInp(0x379);
	  dprintf4("paralel port state: %x, %x", ppdata, ppdata&0x10);
	  dprintf3(" Clear Mca Memory");

	  if (mcaclearmemory (ds, NULL, NULL, error) == DS_NOTOK) {
		  ds->devserver.state = DEVON;
		  return(DS_NOTOK);
      }

	  dprintf3("  Start the acquisition");
	  /* set mca acquisition time to 0 = infinity counting */
	  stAcq.dPsetLive = 0;
	  stAcq.dPsetTrue = 0;
	  sReturn = SadPutStruct (ds->mcaAIM.hDSC, WST_AcqPage, 1, 1, &stAcq, 
	                          sizeof(AcqPage_T));
	  dprintf3("  Start Thread");
	  /* start the acquisition thread */
	  SpStopThread = 0;
	  _beginthread(mcaspecialthreadacq, 0, (void *)ds);

  } else {
	  dprintf1("  Special Acquisition Already Running");
	  *error = DevErr_AttemptToViolateStateMachine;
	  return(DS_NOTOK);
  }

  ds->devserver.state =  ds->devserver.n_state;

  dprintf2("McaAIM, mcaspecialstart(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcaspecialread()
  Description:	 Read newly acquired spectrums
  Arg(s) In:	 McaAIM  ds :
		         DevVoid *argin :
				         None
  Arg(s) Out:	 DevVarUlongArray *argout :
                         spectrum data
		 long	 *error :
		                 pointer to error code, in case
		                 routine fails. Error code(s):
 _________________________________________________________________________*/

static long  mcaspecialread(McaAIM ds, DevVoid *argin, 
							DevVarULongArray *argout,long *error)
{
  int SpLastAcq, SpNbSend;

  dprintf2("McaAIM, mcaspecialread(), enter");

  SpLastAcq = SpSpectrumIndex-1;
  if (SpLastAcq >= SpLastSend) {
	  SpNbSend = SpLastAcq - SpLastSend;
	  argout->length = SpNbSend * SpNbCh;
	  argout->sequence = SpDataBuff + (SpLastSend+1) * SpNbCh;
	  SpLastSend = SpLastAcq;
  } else {
	  SpNbSend = SpNbCycle - SpLastSend - 1;
	  argout->length = SpNbSend * SpNbCh;
	  argout->sequence = SpDataBuff + (SpLastSend+1) * SpNbCh;
	  SpLastSend = -1;
  }
  /* ds->devserver.state =  ds->devserver.n_state; */
  dprintf2("McaAIM, mcaspecialread(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      static long mcaspecialstop()
  Description:	 Stop cycle acquisition
  Arg(s) In:	 McaAIM  ds :
		         DevVoid *argin :
				         None
  Arg(s) Out:	 DevVoid *argout :
                         none
		 long	 *error :
		                 pointer to error code, in case
		                 routine fails. Error code(s):
 _________________________________________________________________________*/

static long  mcaspecialstop(McaAIM ds, DevVoid *argin, DevVoid *argout,long *error)
{
  dprintf2("McaAIM, mcaspecialstop(), enter");
  
  if (SpThreadStopped == 0) {
	  SpStopThread = 1;
	  while (SpThreadStopped == 0);
  }
  ds->devserver.state =  ds->devserver.n_state;
  dprintf2("McaAIM, mcaspecialstop(), exit");
  return (DS_OK);
}

/*_________________________________________________________________________

  Function:      void mcaspecialthreadacq()
  Description:	 Stop cycle acquisition
  Arg(s) In:	 McaAIM  ds :
		         DevVoid *argin :
				         None
  Arg(s) Out:	 DevVoid *argout :
                         none

 _________________________________________________________________________*/
void mcaspecialthreadacq(McaAIM ds)
{
  unsigned short addr;
  unsigned long *ptr;
  int ppb3;
  
  SpThreadStopped = 0;
  dprintf3(" Start of Thread");
  ppdata = MMPInp(0x379);
  dprintf4("paralel port state: %x, %x", ppdata, (ppdata&0x10)>>4);
  
  SpSpectrumIndex=0;
  SpLastSend=-1;
  ptr=SpDataBuff;
  
  /* clear CB25 pin 3 : enable ISG card to detect address change */
  MMPOutp(0x378, MMPInp(0x378) & 0xfc);

  if ((SpMode == 1)||(SpMode == 2)) {
	  for (ppb3=(MMPInp(0x379)&0x10)>>4;ppb3==0;ppb3=(MMPInp(0x379)&0x10)>>4) {
		  if (SpStopThread == 1)
			  mcastopthread(ds);
	  }
  }

#ifdef ELAPSEDTIME
  beg = clock();
#endif

  /* start acquisition */
  SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_StartAcq);

  if ((SpMode == 0) || (SpMode == 1)) {
	  for (;SpSpectrumIndex<SpNbCycle;SpSpectrumIndex++,ptr+=SpNbCh) {
		  if (SpPPLevel == 0) {
			  /* set CB25 pin 2*/
			  MMPOutp(0x378, MMPInp(0x378) | 0x01);
			  SpPPLevel = 1;
			  addr = 1;
		  } else {
			  /* clear CB25 pin 2 */
			  MMPOutp(0x378, MMPInp(0x378) & 0xfe);
			  SpPPLevel = 0;
			  addr = SpReadMaxSize+1;
		  }
		  sReturn = SadGetSpectrum(ds->mcaAIM.hDSC, addr, SpNbCh, FALSE, ptr);
		  
		  if (SpStopThread == 1)
			  mcastopthread(ds);
	  }
  }

  if (SpMode == 2) {
	  ppb3=1;
	  while (ppb3 == 1) {
		  ppb3=(MMPInp(0x379)&0x10)>>4;
		  if (SpPPLevel == 0) {
			  /* set CB25 pin 2*/
			  MMPOutp(0x378, MMPInp(0x378) | 0x01);
			  SpPPLevel = 1;
			  addr = 1;
		  } else {
			  /* clear CB25 pin 2 */
			  MMPOutp(0x378, MMPInp(0x378) & 0xfe);
			  SpPPLevel = 0;
			  addr = SpReadMaxSize+1;
		  }
		  sReturn = SadGetSpectrum(ds->mcaAIM.hDSC, addr, SpNbCh, FALSE, ptr);
		  dprintf4("  addr=%d channel=%d ptr=0x%x ptr[0]=%d ptr[88]=%d", addr, SpNbCh, ptr, *ptr, *(ptr+88));
		  
		  if (SpStopThread == 1)
			  mcastopthread(ds);
		  SpSpectrumIndex++;
		  if (SpSpectrumIndex == SpNbCycle) {
			  SpSpectrumIndex = 0;
			  ptr=SpDataBuff;
		  } else {
			  ptr+=SpNbCh;
		  }
	  } /* end while */
	  dprintf4("MODE = 2  SPECTRUM = %d", SpSpectrumIndex);
  }

#ifdef ELAPSEDTIME
  end = clock() - beg;
  dprintf ("%d ms", end);
#endif

  mcastopthread(ds);
}

void mcastopthread(McaAIM ds)
{
  /* set CB25 pin 3 : disable ISG card to detect address change */
  MMPOutp(0x378, MMPInp(0x378) | 0x03);

  /* stop the acquisition */
  SadControlDSC(ds->mcaAIM.hDSC, MCA, CTL_AbortAcq);
			
  MMPClose();

  SpThreadStopped = 1;

  ds->devserver.state = DEVON;
  ds->devserver.n_state = DEVON;

  dprintf3("     End of Thread, %d", ds->devserver.n_state);
	
  _endthread();
}
