/*static char RcsId[] = "$Header: /segfs/dserver/classes/ccd/imagepro/src/RCS/ImagePro.c,v 1.7 2001/10/24 08:01:26 goetz Exp $";*/
/*********************************************************************
 *
 *File:		ImagePro.c
 *
 *Project:	Beamline Control
 *
 *Description:	A device class to control the ImagePro image acquistion
		and analysis program via a device server. ImagePro is a sophisticated
		program for image analysis which has drivers for a number of
		cameras. All the ImagePro functions (known as Auto-Pro functions)
		can be called via the mouse, from a macros or from Visual C++.
		The device server provides remote client access to ImagePro using
		the visual C++ interface. Thereby enabling clients like SPEC to 
		control the beamline motors and send remote commands to ImagePro.

		This device server has originally been developed on ID11 to control
		the Photonics Science and Photometrics cameras. It is now used on
		ID22 and ID18 for controlling the Photonics Science XIOS II camera.
		It can be used to control any camera which has an ImagePro driver.
 *
 *Author(s):	Andy Gotz
 *
 *Original:	22 April 1999
 *
 *$Log: ImagePro.c,v $
 *Revision 1.7  2001/10/24 08:01:26  goetz
 *added commands for SPEC CCD compatibility; tested on Frelon1000 on ID11
 *
 *Revision 1.6  2001/10/17 19:30:06  goetz
 *latest version from Frelon lab; uses Jean-Luc's functions to snap
 *
 *Revision 1.5  2001/09/07 08:54:27  goetz
 *latest version from id11; provides support for writing EDF format
 *
 *Revision 1.4  2001/07/11 08:47:45  goetz
 *IpWsSaveAs now accepts format as second input argument
 *
 *
 *			13/06/2000 - added IpAcqSnap and IpWsSaveAs commands
 *			             removed trailing zero from IpAcqTimed file name
 *
 *			20/07/2000 - added IpGetArea command
 *
 *Copyright(c) 1999 by European Synchrotron Radiation Facility, 
 *                     Grenoble, France
 *
 *
 *********************************************************************/

#include <API.h>
#include <DevServer.h>
#include <DevErrors.h>
#include <Admin.h>

#include <DevServerP.h>
#include <ImageProP.h>
#include <ImagePro.h>

#include <winbase.h>
#include <ipc.h>
#include <windows.h>
#include <process.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/timeb.h>

#include <DevCcdCmds.h>
#include <DevCcdStates.h>
#include <DevCcdErrors.h>
/*
 * public methods
 */

static long class_initialise(long *error);
static long object_create(char *name,DevServer *ds_ptr,long *error);
static long object_initialise(ImagePro ds,long *error);
static long state_handler(ImagePro ds,DevCommand cmd,long *error);

static	DevMethodListEntry methods_list[] = {
   	{DevMethodClassInitialise,	class_initialise},
   	{DevMethodInitialise,		object_initialise},
  	{DevMethodCreate,		object_create},
   	{DevMethodStateHandler,		state_handler},
   };


ImageProClassRec imageProClassRec = {
   /* n_methods */        sizeof(methods_list)/sizeof(DevMethodListEntry),
   /* methods_list */     methods_list,
   };

ImageProClass imageProClass = (ImageProClass)&imageProClassRec;

/*
 * public commands
 */

static long dev_status(ImagePro ds,DevVoid * argin,DevString *argout,long *error);
static long dev_state(ImagePro ds,DevVoid *argin,DevShort *argout,long *error);
static long dev_macro_run(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error);
static long dev_macro_stop(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error);
static long dev_macro_load(ImagePro ds,DevString *argin,DevVoid *argout,long *error);
static long dev_acq_snap(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error);
static long dev_acq_control(ImagePro ds,DevVarShortArray *argin,DevVoid *argout,long *error);
static long dev_acq_timed(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_ws_save_as(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_set_variable(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_get_variable(ImagePro ds,DevString *argin,DevFloat *argout,long *error);
static long dev_get_image_size(ImagePro ds,DevVoid *argin,DevVarShortArray *argout,long *error);
static long dev_get_image(ImagePro ds,DevVoid *argin,DevOpaque *argout,long *error);
static long dev_get_area(ImagePro ds,DevVarLongArray *argin,DevOpaque *argout,long *error);
static long dev_ws_change_description(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error);
static long dev_ws_save_as_edf(ImagePro ds,DevString *argin,DevVoid *argout,long *error);
static long dev_reset(ImagePro ds, DevVoid *argin, DevVoid *argout, long *error);

/*
 * commands added for compatibility with other SPEC CCD device server
 */

static long dev_ccd_get_state(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_start(ImagePro ds, DevVoid *argin, DevVoid *argout, long *error);
static long dev_ccd_stop(ImagePro ds, DevVoid *argin, DevVoid *argout, long *error);
static long dev_ccd_read(ImagePro ds, DevLong *argin, DevOpaque *argout, long *error);
static long dev_ccd_set_exposure(ImagePro ds, DevFloat *argin, DevVoid *argout, long *error);
static long dev_ccd_get_exposure(ImagePro ds, DevVoid *argin, DevFloat *argout, long *error);
static long dev_ccd_set_bin(ImagePro ds, DevVarLongArray *argin, DevVoid *argout, long *error);
static long dev_ccd_get_bin(ImagePro ds, DevVoid *argin, DevVarLongArray *argout, long *error);
static long dev_ccd_set_trigger(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);
static long dev_ccd_get_trigger(ImagePro ds, DevVoid *argin, DevLong *argout, long *error);
static long dev_ccd_set_file_par(ImagePro ds, DevVarStringArray *argin, DevVoid *argout, long *error);
static long dev_ccd_get_file_par(ImagePro ds, DevVoid *argin, DevVarStringArray *argout, long *error);
static long dev_ccd_write_file(ImagePro ds, DevLong *argin, DevVoid *argout, long *error);

static	DevCommandListEntry commands_list[] = {
   	{DevState, dev_state, D_VOID_TYPE, D_SHORT_TYPE},
   	{DevStatus, dev_status, D_VOID_TYPE, D_STRING_TYPE},
   	{DevIpMacroRun, dev_macro_run, D_VAR_STRINGARR, D_VOID_TYPE},
   	{DevIpMacroStop, dev_macro_stop, D_VAR_STRINGARR, D_VOID_TYPE},
	{DevIpMacroLoad, dev_macro_load, D_STRING_TYPE, D_VOID_TYPE},
	{DevIpAcqControl, dev_acq_control, D_VAR_SHORTARR, D_VOID_TYPE},
   	{DevIpAcqSnap, dev_acq_snap, D_VOID_TYPE, D_VOID_TYPE},
   	{DevIpAcqTimed, dev_acq_timed, D_VAR_STRINGARR, D_VOID_TYPE},
	{DevIpWsSaveAs, dev_ws_save_as, D_VAR_STRINGARR, D_VOID_TYPE},
	{DevIpSetVariable, dev_set_variable, D_VAR_STRINGARR, D_VOID_TYPE},
    {DevIpGetVariable, dev_get_variable, D_STRING_TYPE, D_FLOAT_TYPE},
	{DevIpGetImageSize, dev_get_image_size, D_VOID_TYPE, D_VAR_SHORTARR},
	{DevIpGetImage, dev_get_image, D_VOID_TYPE, D_OPAQUE_TYPE},
	{DevIpGetArea, dev_get_area, D_VAR_LONGARR, D_OPAQUE_TYPE},
	{DevIpWsChangeDescription, dev_ws_change_description, D_VAR_STRINGARR, D_VOID_TYPE},
	{DevReset, dev_reset, D_VOID_TYPE, D_VOID_TYPE},
/*
 * commands added for compatibility with other SPEC CCD device servers
 */
	{DevCcdStart, dev_ccd_start, D_VOID_TYPE, D_VOID_TYPE},
	{DevCcdStop, dev_ccd_stop, D_VOID_TYPE, D_VOID_TYPE},
	{DevCcdRead, dev_ccd_read, D_LONG_TYPE, D_OPAQUE_TYPE},
	{DevCcdSetExposure, dev_ccd_set_exposure, D_FLOAT_TYPE, D_VOID_TYPE},
	{DevCcdGetExposure, dev_ccd_get_exposure, D_VOID_TYPE, D_FLOAT_TYPE},
	{DevCcdSetBin, dev_ccd_set_bin, D_VAR_LONGARR, D_VOID_TYPE},
	{DevCcdGetBin, dev_ccd_get_bin, D_VOID_TYPE, D_VAR_LONGARR},
	{DevCcdSetTrigger, dev_ccd_set_trigger, D_LONG_TYPE, D_VOID_TYPE},
	{DevCcdGetTrigger, dev_ccd_get_trigger, D_VOID_TYPE, D_LONG_TYPE},
	{DevCcdSetFilePar, dev_ccd_set_file_par, D_VAR_STRINGARR, D_VOID_TYPE},
	{DevCcdGetFilePar, dev_ccd_get_file_par, D_VOID_TYPE, D_VAR_STRINGARR},
	{DevCcdWriteFile, dev_ccd_write_file, D_LONG_TYPE, D_VOID_TYPE},
};

static long n_commands = sizeof(commands_list)/sizeof(DevCommandListEntry);

/*
 * reserve space for a default copy of the ImagePro object
 */

static ImageProRec imageProRec;
static ImagePro imagePro = (ImagePro)&imageProRec;

/*
 * ImagePro resource tables used to access the static database
 *
 */

db_resource res_object[] = {
   {"simulation", D_LONG_TYPE, NULL},
   {"camera_type", D_LONG_TYPE, NULL},
   	};
int res_object_size = sizeof(res_object)/sizeof(db_resource);


db_resource res_class[] = {
   {"nada",       D_LONG_TYPE, NULL},
   	};
int res_class_size = sizeof(res_class)/sizeof(db_resource);

#define SIMULATION 1
#define CAMERA_STANDARD 	0
#define CAMERA_FRELON2000	1

long    (*IpAcquShow)(long show);
long    (*IpAcquSnap)(void);
long	(*IpAcquRefresh)(void);
long	(*IpAcquSetExposure)(long exposure);
long	(*IpAcquSetHorizontalBinning)(long hbin);
long	(*IpAcquSetVerticalBinning)(long vbin);

                                              
/*======================================================================
 Function:      static long class_initialise()

 Description:	routine to be called the first time a device is 
 		created which belongs to this class (or is a subclass
		thereof. This routine will be called only once.

 Arg(s) In:	none

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
 =======================================================================*/

static long class_initialise(long *error)
{
/*
 * ImageProClass is a subclass of the DevServerClass
 */

	HINSTANCE acqdll;

   imageProClass->devserver_class.superclass = devServerClass;
   imageProClass->devserver_class.class_name = (char*)malloc(sizeof("ImageProClass")+1);
   sprintf(imageProClass->devserver_class.class_name,"ImageProClass");
   imageProClass->devserver_class.class_inited = 1;
   imageProClass->devserver_class.n_commands = n_commands;
   imageProClass->devserver_class.commands_list = commands_list;

/*
 * initialise ImagePro with default values. these will be used
 * for every ImagePro object created.
 */

   imagePro->devserver.class_pointer = (DevServerClass)imageProClass;
   imagePro->devserver.state = DEVON;


   imagePro->imagepro.cam_type = CAMERA_STANDARD;
   imagePro->imagepro.roix1 = 1;
   imagePro->imagepro.roix2 = 20;
   imagePro->imagepro.roiy1 = 1;
   imagePro->imagepro.roiy2 = 15;
   imagePro->imagepro.hbin = 1;
   imagePro->imagepro.vbin = 1;
/*
 * Interrogate the static database for default values
 *
 */

   res_class[0].resource_adr	= &(imagePro->devserver.n_state);

   if(db_getresource("CLASS/ImagePro/DEFAULT", res_class, res_class_size, error))
   {
      return(DS_NOTOK);
   }

/*
 * ImagePro must be running when the AutoPro functions are executed
 *
   WORD wReturn = WinExec("c:\\ipwin3\\ipwin16",SW_SHOWNORMAL);

   dprintf("class_initialise(): WinExec(ipwin16.exe) returned %d\n",wReturn);
 */

/*
 * get frelon functions from sdk 
 */
   acqdll = LoadLibrary("C:\\IPWIN4\\acquisition.dll");
/* imageProClass->imagepro_class.acqdll = acqdll; */

   if( acqdll )
   {
    	IpAcquShow = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquShow");
        dprintf("class_initialise(): IpAcquShow 0x%08x",IpAcquShow);

    	IpAcquSnap = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquSnap");
        dprintf("class_initialise(): IpAcquSnap 0x%08x",IpAcquSnap);
    	IpAcquRefresh = (int (__cdecl *)(void))GetProcAddress(acqdll, "IpAcquRefresh");
        dprintf("class_initialise(): IpAcquRefresh 0x%08x",IpAcquRefresh);
    	IpAcquSetExposure = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetExposure");
        dprintf("class_initialise(): IpAcquSetExposure 0x%08x",IpAcquSetExposure);
    	IpAcquSetHorizontalBinning = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetHorizontalBinning");
        dprintf("class_initialise(): IpAcquSetHorizontalBinning 0x%08x",IpAcquSetHorizontalBinning);
    	IpAcquSetVerticalBinning = (int (__cdecl *)(long))GetProcAddress(acqdll, "IpAcquSetVerticalBinning");
        dprintf("class_initialise(): IpAcquSetVerticalBinning 0x%08x",IpAcquSetVerticalBinning);
   }
   else
   {
      dprintf("class_initialise(): could not load library c:\\IPWIN4\\acquisition.dll");
      dprintf("class_initialise(): therefore NO SUPPORT (or poor) for FRELON 2000 !");
   }

   return(DS_OK);
}

/*======================================================================
 Function:	static long object_create()

 Description:	routine to be called on creation of a device object

 Arg(s) In:	char *name - name to be given to device

 Arg(s) Out:	DevServer *ds_ptr - pointer to created device
		long *error - pointer to error code, in case routine fails
 =======================================================================*/

static long object_create(char *name,DevServer *ds_ptr,long *error)
{
   ImagePro ds;

   ds = (ImagePro)malloc(sizeof(ImageProRec));

/*
 * initialise device with default object
 */

   *(ImageProRec*)ds = *(ImageProRec*)imagePro;

/*
 * finally initialise the non-default values
 */

   ds->devserver.name = (char*)malloc(strlen(name)+1);
   sprintf(ds->devserver.name,"%s",name);

   *ds_ptr = (DevServer)ds;

   return(DS_OK);
}

/*============================================================================

Function:		static long object_initialise()

Description:	routine to be called on initialisation of a device object

Arg(s) In:		ImagePro ds	- object to initialise

Arg(s) Out:

		long *error     - pointer to error code, in case routine fails
=============================================================================*/
static long object_initialise(ImagePro ds,long *error)
{

   long simulation;

   simulation = ds->imagepro.mode = 0;
   res_object[0].resource_adr        = &simulation;
   res_object[1].resource_adr        = &(ds->imagepro.cam_type);
   if(db_getresource(ds->devserver.name, res_object, res_object_size, error))
   {
   	return(DS_NOTOK);
   }

   if (simulation) 
   {

	   ds->imagepro.mode = SIMULATION;

	   dprintf("object_initialise(): device is in SIMULATION mode");

   }
   ds->devserver.state = DEVON;

   return(DS_OK);
}

/*======================================================================
 Function:      static long state_handler()

 Description:	this routine is reserved for checking wether the command
				requested can be executed in the present state.

 Arg(s) In:		ImagePro ds - device on which command is to executed
				DevCommand cmd - command to be executed

 Arg(s) Out:	long *error - pointer to error code, in case routine fails
 =======================================================================*/

static long state_handler(ImagePro ds,DevCommand cmd,long *error)
{
   long int p_state, n_state, lcmd;
   long iret = DS_OK;
/*
 * Get here the real state of the physical device
 *
 *  example:
 *  ds->devserver.state=get_state();
 */
   lcmd = cmd;
   p_state = ds->devserver.state;
   dprintf("state_handler(): p_state %d cmd %d ",p_state, lcmd);
/*
 * Before checking out the state machine assume that the state doesn't
 * change i.e. new state == old state
 *
 */

   n_state = p_state;

   switch (p_state)
   {
    
		case (DEVRUN) : 
		{
            dprintf("state_handler(): DEVRUN state ");
			switch (cmd) 
			{
			case (DevReset) :
			case (DevState):
			case (DevStatus): iret = DS_OK;
							  break;
			default :	      iret = -1;
							  (*error) = DevErr_AttemptToViolateStateMachine;
							  break;
			}
		}
		break;
	
   		default: iret = DS_OK;
			     dprintf("state_handler(): default state ");
   		         break;
   }
   ds->devserver.n_state = n_state;

   return(iret);
}

/*============================================================================
 Function:      static long dev_state()

 Description:	return the state of the device

 Arg(s) In:		ImagePro 	ds 	- 
				DevVoid  	*argin  - none
   				  
 Arg(s) Out:	DevShort	*argout - returned state 
				long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_state(ImagePro ds,DevVoid *argin,DevShort *argout,long *error)
{

   if (ds->imagepro.mode != SIMULATION)

   {
/* not implemented yet ! */ 
   }
   else
   {
      dprintf2("dev_state(): SIMULATION mode");

      ds->devserver.state = DEVON;
   }

   *argout	= (short)ds->devserver.state;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_status()

 Description:	return state of the device as an ASCII string

 Arg(s) In:		ImagePro 	ds 	- 
				DevVoid  	*argin  - none
   				  
 Arg(s) Out:	DevString	*argout - contains string 
============================================================================*/

static long dev_status(ImagePro ds,DevVoid * argin,DevString *argout,long *error)
{
   static	char	str[1024];
   long				status;
   short			state;

   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * interrogate state
 */
      status = dev_state(ds,NULL,&state,error);
	  if (state == DEVRUN)
	  {
		 sprintf(str,"The camera is exposing");
	  }
	  else
	  {
         if (state == DEVON)
		 {
		    sprintf(str,"The camera has an exposed image");
		 }
	     else
	     {
		    sprintf(str,"The camera is not exposing ");
         }
	  }
   }
   else
   {
      sprintf(str,"The camera is in SIMULATION mode\n");
   }

   *argout = str;
   return (DS_OK);
}

/*============================================================================
 Function:      static long dev_macro_run()

 Description:	start macro

 Arg(s) In:	 ImagePro 	ds 	- 
			 DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevShort	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/
static long dev_macro_run(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{
   char *macro_name, *script_file;
   long status;

   macro_name = argin->sequence[0];
   script_file = argin->sequence[1];

   dprintf("dev_macro_run(): macro name %s, script file %s",
	        macro_name,script_file);

   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpMacroRun(macro_name,script_file);
		dprintf("dev_macro_run(): IpMacroRun() returned %d",status);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf2("dev_macro_run(): SIMULATION mode");
	  return(DS_OK);
   }
   return (DS_OK);
   
}

/*============================================================================
 Function:      static long dev_macro_stop()

 Description:	stop image exposure

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_macro_stop(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
   char message[256];
   long status;

   dprintf("dev_macro_stop(): message %s",argin);

   sprintf(message,"IpMacroStop() called from device server (reason=%s)",
           argin);
   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpMacroStop(message,0);
		dprintf("dev_macro_stop(): IpMacroRun() returned %d",status);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf2("dev_macro_stop(): SIMULATION mode");
   }

   return(DS_OK);
}


/*============================================================================
 Function:      static long dev_macro_load()

 Description:	load a macro script file

 Arg(s) In:	 ImagePro 	ds 	- 
			 DevString 	*argin  - macro script file
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/
static long dev_macro_load(ImagePro ds,DevString *argin,DevVoid *argout,long *error)
{
   long status;

   dprintf("dev_macro_load(): script file %s",
	        *argin);

   if (ds->imagepro.mode != SIMULATION)
   {
		status = IpMacroLoad(*argin);
		dprintf("dev_macro_load(): IpMacroLoad() returned %d",status);
		if (status < 0) return(DS_NOTOK);
   }
   else
   {
      dprintf2("dev_macro_load(): SIMULATION mode");
	  return(DS_OK);
   }
   return (DS_OK);
   
}

/*============================================================================
 Function:      static long dev_acq_control()

 Description:	set the acquisition parameters

 Arg(s) In:	ImagePro 	ds 	- 
			DevVarShortArray  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_acq_control(ImagePro ds,DevVarShortArray *argin,DevVoid *argout,long *error)
{
   short cmd, param;
   long status, exposure;


   cmd = argin->sequence[0];
   param = argin->sequence[1];

   dprintf("dev_acq_control(): cmd %d param %d value %d",
	       cmd, param, argin->sequence[2]);
   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * if this is a "SET EXPOSURE" command and this is a FRELON2000 then
 * use the FRELON SDK function IpAcquSetExposure() instead of the 
 * IPAcqControl() function
 */
	  if ((cmd == 49) && (ds->imagepro.cam_type == CAMERA_FRELON2000) && (IpAcquSetExposure != NULL))
	  {
         if (argin->sequence[3] != 0)
		 {
			 exposure = argin->sequence[3]*65536 + argin->sequence[2];
		 }
		 else
		 {
			 exposure = argin->sequence[2];
		 }
		 status = IpAcquSetExposure(exposure);
		 dprintf("dev_acq_control(): FRELON IpAcquSetExposure(%d) returned %d",exposure, status);
	  }
	  else
	  {
         status = IpAcqControl(cmd, param,&(argin->sequence[2]));
         dprintf("dev_acq_control(): IpAcqControl() returned %d",status);
         if (status < 0) return (DS_NOTOK);
	  }
   }
   else
   {
      dprintf2("dev_acq_control(): SIMULATION mode");
   }

   return(DS_OK);
}

static char dir[80], prefix[80];
static long startnumber, frames, interval;
static ImagePro ip_acq_ds;

/*============================================================================
 Function:      static long ip_acq_timed()

 Description:	thread to acquire a sequence of frames and write them to disk
				will be called by the dev_acq_timed() command which is now
				non-blocking. the thread will block until it finished and then
				update the device state to indicate the result

 Arg(s) In:	char** 	sequence 	- pointer to sequence of input parameters as strings

 Arg(s) Out:  none

============================================================================*/

void ip_acq_timed(void *argin)
{
   long status;
   static char filename[256];

   dprintf("ip_acq_timed(): dir %s prefix %s start number %d frames %d interval %d",
	       dir,prefix,startnumber,frames,interval);

   ip_acq_ds = (ImagePro)argin;

/*
 * the IpAcqTimed() command does not display the result in a new
 * new workspace. Therefore replace it with a combination of IpAcqSnap(0
 * and IpWsSaveAs() - andy 27mar2000 
 *
   status = IpAcqTimed(dir,prefix,(short)startnumber,
                      (short)frames,(short)interval);
   dprintf("ip_acq_timed(): IpAcqTimed() returned %d",status);
 */

   status = IpAcqSnap(ACQ_NEW);
   dprintf("ip_acq_timed(): IpAcqSnap(ACQ_NEW) returned %d",status);
   sprintf(filename,"%s\\%s.tif",dir,prefix);
   dprintf("ip_acq_timed(): filename = %s",filename);
   status = IpWsSaveAs(filename,"TIF");
   dprintf("ip_acq_timed(): IpWsSaveAs() returned %d",status);

   ip_acq_ds->devserver.state = DEVON;

   _endthread();
}

/*============================================================================
 Function:      static long ip_acq_snap()

 Description:	thread to acquire a snap shot in a new workspace.
				will be called by the dev_acq_timed() command which is now
				non-blocking. the thread will block until it finished and then
				update the device state to indicate the result

 Arg(s) In:	char** 	sequence 	- pointer to sequence of input parameters as strings

 Arg(s) Out:  none

============================================================================*/

void ip_acq_snap(void *argin)
{
   long status, elapsed;
   static char filename[256];
   struct _timeb before, after;

   ip_acq_ds = (ImagePro)argin;

//   dprintf("ip_acq_snap(): dir %s prefix %s start number %d frames %d interval %d",
//	       dir,prefix,startnumber,frames,interval);
   _ftime(&before);
/*
 * for FRELON2000 type camera use the frelon sdk to snap and not the standard
 * snap
 */
   if ((ip_acq_ds->imagepro.cam_type == CAMERA_FRELON2000) && (IpAcquSnap != NULL))
   {
        status = IpAcquSnap();
        dprintf("ip_acq_snap(): FRELON IpAcquSnap() returned %d",status);
   }
   else
   {
   		status = IpAcqSnap(ACQ_NEW);
        dprintf("ip_acq_snap(): IpAcqSnap(ACQ_NEW) returned %d",status);
   }

   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;
   dprintf("ip_acq_snap(): time elapsed %d milliseconds",elapsed);

   ip_acq_ds->devserver.state = DEVON;

   _endthread();
}

/*============================================================================
 Function:      static long dev_acq_snap()

 Description:	acquire a single frame in a new workspace

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_acq_snap(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{


   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * fork thread to do acquisition
 */   
	  dprintf("dev_acq_snap(): fork thread ip_acq_snap() to do acquisition ");
      _beginthread (ip_acq_snap, 0, ds);

	  ds->devserver.state = DEVRUN;
      dprintf("dev_acq_snap(): new state %d",ds->devserver.state);
   }
   else
   {
      dprintf2("dev_acq_snap(): SIMULATION mode");
   }

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_acq_timed()

 Description:	acquire a single frames and write them to disk

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevShort	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_acq_timed(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{

   if (argin->length < 5)
   {
      *error = DevErr_DeviceIllegalParameter;
	  return (DS_NOTOK);
   }

   sprintf(dir,"%s",argin->sequence[0]);
   sprintf(prefix,"%s",argin->sequence[1]);
   sscanf(argin->sequence[2],"%d",&startnumber);
   sscanf(argin->sequence[3],"%d",&frames);
   sscanf(argin->sequence[4],"%d",&interval);
   ip_acq_ds = ds;

   dprintf("dev_acq_timed(): dir %s prefix %s start number %d frames %d interval %d",
	       dir,prefix,startnumber,frames,interval);

   if (ds->imagepro.mode != SIMULATION)
   {
/*
 * fork thread to do acquisition
 */   
	   if (_stricmp(prefix,"NEW") != 0)
	   {
	      dprintf("dev_acq_timed(): fork thread ip_acq_timed() to do acquisition ");
         _beginthread (ip_acq_timed, 0, ds);
	   }
	   else
	   {
	      dprintf("dev_acq_timed(): fork thread ip_acq_snap() to do acquisition ");
         _beginthread (ip_acq_snap, 0, ds);
	   }

	  ds->devserver.state = DEVRUN;
      dprintf("dev_acq_timed(): new state %d",ds->devserver.state);
   }
   else
   {
      dprintf2("dev_acq_timed(): SIMULATION mode");
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ws_save_as()

 Description:	save the current workspace to disk

 Arg(s) In:	ImagePro 	ds 	- 
		DevString  	*argin  - file name
   				  
 Arg(s) Out: DevVoid	*argout - none
	     long 	*error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ws_save_as(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{

	char *filename, *format;
	long status;

   filename = argin->sequence[0];
   format = argin->sequence[1];

   if (strcmp(format, "edf") == 0)
   {
      status = dev_ws_save_as_edf(ds, &filename, NULL, error);
   }
   else
   {
      dprintf("dev_ws_save_as(): filename = %s (format = %s)",filename, format);
      status = IpWsSaveAs(filename,format);
      dprintf("dev_ws_save_as(): IpWsSaveAs() returned %d",status);
   }


   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ws_save_as_edf()

 Description:	save the current workspace to disk as EDF format (internal
		ESRF format)

 Arg(s) In:	ImagePro 	ds 	- 
		DevString  	*argin  - file name
   				  
 Arg(s) Out: DevVoid	*argout - none
	     long 	*error - pointer to error code, in case routine fails

============================================================================*/


static long dev_ws_save_as_edf(ImagePro ds,DevString *argin,DevVoid *argout,long *error){

   char *filename, edf_header[1024];
   short *image_size;
   DevVarShortArray var_sharr;
   DevOpaque image;
   int image_file;
   long status, elapsed;
   struct _timeb before, after;
   time_t ltime;

   filename = *argin; 
/*
 * first get image size
 */
   _ftime(&before);

   status = dev_get_image_size(ds, NULL, &var_sharr, error);
   image_size = var_sharr.sequence;

   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;
   dprintf("ip_ws_save_as_edf(): dev_get_image_size() took %d milliseconds",elapsed);

/*
 * then get pointer to image
 */
   _ftime(&before);

   status = dev_get_image(ds, NULL, &image, error);

   _ftime(&after);
   elapsed = (after.time-before.time)*1000;
   elapsed = (after.millitm-before.millitm) + elapsed;
   dprintf("ip_ws_save_as_edf(): dev_get_image() took %d milliseconds",elapsed);
/*
 * create EDF header
 */
   memset(edf_header, ' ', 1024); 
   sprintf(edf_header, "{\n");
   sprintf(edf_header+strlen(edf_header), "HeaderID       = EH:000001:000000:000000 ;\n");
   sprintf(edf_header+strlen(edf_header), "Image          = 1 ;\n");
   sprintf(edf_header+strlen(edf_header), "ByteOrder      = LowByteFirst ;\n");
   sprintf(edf_header+strlen(edf_header), "DataType       = UnsignedShort ;\n" );
   sprintf(edf_header+strlen(edf_header), "Dim_1          = %d ;\n", image_size[0]);
   sprintf(edf_header+strlen(edf_header), "Dim_2          = %d ;\n", image_size[1]);
   sprintf(edf_header+strlen(edf_header), "Size           = %d ;\n", image_size[0]*image_size[1]*2);
   sprintf(edf_header+strlen(edf_header), "count_time     = %g ;\n", 1.0);
   sprintf(edf_header+strlen(edf_header), "point_no       = %d ;\n", 0);
   sprintf(edf_header+strlen(edf_header), "preset         = %g ;\n", 1.0);
   sprintf(edf_header+strlen(edf_header), "col_end        = %d ;\n", image_size[0]-1);
   sprintf(edf_header+strlen(edf_header), "col_beg        = %d ;\n", 0);
   sprintf(edf_header+strlen(edf_header), "row_end        = %d ;\n", image_size[1]-1);
   sprintf(edf_header+strlen(edf_header), "row_beg        = %d ;\n", 0);
   sprintf(edf_header+strlen(edf_header), "dir            = %s ;\n", ".");
   sprintf(edf_header+strlen(edf_header), "suffix         = %s ;\n", "edf");
   sprintf(edf_header+strlen(edf_header), "prefix         = %s ;\n", filename);
   sprintf(edf_header+strlen(edf_header), "run            = %d ;\n", 1);
   time( &ltime );
   sprintf(edf_header+strlen(edf_header), "title          = Frelon CCD with Imagepro on the %s ;\n",ctime( &ltime ));
   dprintf("dev_ws_save_as_edf(): edf_header length %d",strlen(edf_header));
   sprintf(edf_header+1022,"}\n");

   _ftime(&before);
   image_file = _open(filename, _O_BINARY | _O_RDWR | _O_CREAT, _S_IREAD, _S_IWRITE);
   if (image_file >= 0)
   {
      status = _write(image_file, edf_header, sizeof(edf_header));
      status = _write(image_file, image.sequence, image.length);
      status = _close(image_file);
      
	  _ftime(&after);
      elapsed = (after.time-before.time)*1000;
      elapsed = (after.millitm-before.millitm) + elapsed;
      dprintf("ip_ws_save_as_edf(): open(), write(), close() took %d milliseconds",elapsed);
      dprintf("dev_ws_save_as_edf(): saved image in EDF format to file %s",filename);
   }
   else
   {
      dprintf("dev_ws_save_as_edf(): failed to open file %s, image not saved !",filename);
      *error = DevErr_DeviceIllegalParameter;
      return(DS_NOTOK);
   }
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_set_variable()

 Description:	set a variable in MACROS.INI

 Arg(s) In:	ImagePro 	ds 	- 
			DevVarStringArray  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_set_variable(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{
   char *variable;
   float value;
   long status;


   variable = argin->sequence[0];
   sscanf(argin->sequence[1],"%f",&value);

   dprintf("dev_set_variable(): set variable %s to value %f",
	       variable,value);
   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpIniFile(SETFLOAT, variable ,&value);
      dprintf("dev_set_variable(): IpIniFile() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_set_variable(): SIMULATION mode");
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_get_variable()
 Description:	get a variable in MACROS.INI

 Arg(s) In:	ImagePro 	ds 	- 
			DevString  	*argin  - none
   				  
 Arg(s) Out: DevFloat	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_variable(ImagePro ds,DevString *argin,DevFloat *argout,long *error)
{
   char *variable;
   float value;
   long status;


   variable = *argin;

   dprintf("dev_get_variable(): get variable %s",
	       variable,value);
   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpIniFile(GETFLOAT, variable ,&value);
      dprintf("dev_get_variable(): IpIniFile() returned %d, value = %f",status,value);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_get_variable(): SIMULATION mode");
   }

   *argout = value;

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_get_image_size()

 Description:	get image size
 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - nothing
   				  
 Arg(s) Out: DevShortArray	*argout - image size (rows, columns)
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_image_size(ImagePro ds,DevVoid *argin,DevVarShortArray *argout,long *error)
{
   long status;
   IPDOCINFO info;
   static short image_size[2];


   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
      image_size[0] = info.Height;
      image_size[1] = info.Width;
      dprintf("dev_get_image_size(): IpDocGet() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_get_image_size(): SIMULATION mode");
      image_size[0] = 1024;
	  image_size[1] = 1024;
   }
   dprintf("dev_get_image_size(): rows=%d, columns=%d",
	       image_size[0], image_size[1]);

   argout->length = 2;
   argout->sequence = (short*)image_size;
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_get_image()

 Description:	get whole image 
 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - nothing
   				  
 Arg(s) Out: DevOpaque	*argout - area of image
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_image(ImagePro ds,DevVoid *argin,DevOpaque *argout,long *error)
{
   long status;
   RECT region;
   IPDOCINFO info;
   static short image[2048][2048];
#ifdef DEBUG
   short *pimage;
#endif /* DEBUG */


   if (ds->imagepro.mode != SIMULATION)
   {
	  IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
	  region.left = 0;
	  region.right = info.Width-1;
	  region.top = 0;
	  region.bottom = info.Height-1;
      dprintf("dev_get_image(): get area left=%d, right=%d, top=%d, bottom=%d",
	       region.left,region.right,region.top,region.bottom);      
	  status = IpDocGetArea(DOCSEL_ACTIVE, &region, image, CPROG);
      dprintf("dev_get_image(): IpDocGetArea() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
	  region.left = 0;
	  region.right = 1023;
	  region.top = 0;
	  region.bottom = 1023;
      dprintf2("dev_get_image(): SIMULATION mode");
   }

#ifdef DEBUG
   pimage = (short*)image;
   dprintf("dev_get_image(): image[0][0..8] %d %d %d %d %d %d %d %d",
	   *pimage, *(pimage+1), *(pimage+2), *(pimage+3), 
	   *(pimage+4), *(pimage+5), *(pimage+6), *(pimage+7)); 
   pimage = (short*)image+(region.right+1)*(region.bottom);
   dprintf("dev_get_image(): image[%d][0..8] %d %d %d %d %d %d %d %d",region.bottom,
	   *pimage, *(pimage+1), *(pimage+2), *(pimage+3), 
	   *(pimage+4), *(pimage+5), *(pimage+6), *(pimage+7)); 
#endif /* DEBUG */

/*
 * NOTE : hardcoded for 16 bit GRAY level images !
 */
   argout->length = (region.right+1)*(region.bottom+1)*2;
   argout->sequence = (char*)image;
   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_get_area()

 Description:	get a part of and image (part or whole)
 Arg(s) In:	ImagePro 	ds 	- 
			DevVarShortArray  	*argin  - top left and bottom right coordinates
   				  
 Arg(s) Out: DevOpaque	*argout - area of image
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_get_area(ImagePro ds,DevVarLongArray *argin,DevOpaque *argout,long *error)
{
   long status;
   RECT region;
   IPDOCINFO info;
   static short area[2048][2048];
#ifdef DEBUG
   short *parea;
#endif /* DEBUG */

   region.left = argin->sequence[0];
   region.right = argin->sequence[1];
   region.top = argin->sequence[2];
   region.bottom = argin->sequence[3];

/*
 * if region is (0,0)-(0,0) then return entire image
 */
   if ((region.left == 0) && (region.right == 0) && 
	   (region.top == 0) && (region.bottom == 0) && 
	   (ds->imagepro.mode != SIMULATION))
   {
	   IpDocGet(GETDOCINFO, DOCSEL_ACTIVE, &info);
	   region.right = info.Width-1;
	   region.bottom = info.Height-1;
   }
/*
 * order left to right and top to bottom
 */
   else
   {
	   if (region.right < region.left)
	   {
		   region.left = argin->sequence[1];
		   region.right = argin->sequence[0];
	   }
	   if (region.bottom < region.top)
	   {
		   region.top = argin->sequence[3];
		   region.bottom = argin->sequence[2];
	   }
   }

   dprintf("dev_get_area(): get area left=%d, right=%d, top=%d, bottom=%d",
	       region.left,region.right,region.top,region.bottom);
   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpDocGetArea(DOCSEL_ACTIVE, &region, area, CPROG);
      dprintf("dev_getarea(): IpDocGetArea() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_get_area(): SIMULATION mode");
   }

#ifdef DEBUG
   parea = (short*)area;
   dprintf("dev_get_area(): area[%d][0..8] %d %d %d %d %d %d %d %d",region.top,
	   *parea, *(parea+1), *(parea+2), *(parea+3), 
	   *(parea+4), *(parea+5), *(parea+6), *(parea+7)); 
   parea = (short*)area+(region.right-region.left+1)*(region.bottom-region.top);
   dprintf("dev_get_area(): area[%d][0..8] %d %d %d %d %d %d %d %d",region.bottom,
	   *parea, *(parea+1), *(parea+2), *(parea+3), 
	   *(parea+4), *(parea+5), *(parea+6), *(parea+7)); 
#endif /* DEBUG */

   argout->length = (region.right-region.left+1)*(region.bottom-region.top+1)*2;
   argout->sequence = (char*)area;
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_change_description()

 Description:	change the description of the current workspace

 Arg(s) In:	ImagePro 	ds 	- 
			DevVarStringArray  	*argin  - description
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ws_change_description(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{
   char *field, *description;
   short inf_field;
   long status;


   if (argin->length < 2)
   {
	   *error = DevErr_UnknownInputParameter;
	   return(DS_NOTOK);
   }

   field = argin->sequence[0];
   description = argin->sequence[1];
   dprintf("dev_ws_change_description(): set field %s to description %s",
	       field,description);
   inf_field = -1;
   if (strcmp(field,"title") == 0) inf_field = INF_TITLE;
   if (strcmp(field,"artist") == 0) inf_field = INF_ARTIST;
   if (strcmp(field,"date") == 0) inf_field = INF_DATE;
   if (strcmp(field,"description") == 0) inf_field = INF_DESCRIPTION;
   if (strcmp(field,"name") == 0) inf_field = INF_NAME;
   if (strcmp(field,"range") == 0) inf_field = INF_RANGE;
   if (inf_field == -1)
   {
	   *error = DevErr_UnknownInputParameter;
	   return(DS_NOTOK);
   }

   if (ds->imagepro.mode != SIMULATION)
   {
      status = IpWsChangeDescription(inf_field, description);
      dprintf("dev_ws_change_description(): IpWsChangeDescription() returned %d",status);
      if (status < 0) return (DS_NOTOK);
   }
   else
   {
      dprintf2("dev_ws_change_description(): SIMULATION mode");
   }

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_reset()

 Description:	Reset device state to DEVON (useful to unblock state)

 Arg(s) In:	ImagePro 	ds 	- 
			DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
			 long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_reset(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{
   ds->devserver.state = DEVON;

   return(DS_OK);
}

/*
 * SPEC CCD compatible device server commands
 */

/*============================================================================
 Function:      static long dev_ccd_get_state()

 Description:	Get CCD state

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevLong	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_state(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

   *argout = ds->imagepro.ccd_state;

   return(DS_OK);
}
/*============================================================================
 Function:      static long dev_ccd_start()

 Description:	Start acquistion

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_start(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{

   dprintf("dev_ccd_start(): fork thread ip_acq_snap() to do acquisition ");
   _beginthread (ip_acq_snap, 0, ds);

   ds->imagepro.ccd_state = DevCcdTaking;
   dprintf("dev_acq_snap(): new state %d",ds->devserver.state);
   
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_stop()

 Description:	Stop acquistion

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_stop(ImagePro ds,DevVoid *argin,DevVoid *argout,long *error)
{

   dprintf("dev_ccd_stop(): stop acquisition ");

/* PRESENT VERSION DOES NOTHING ! */

/*   ds->imagepro.ccd_state = DevCcdReady;*/
   
   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_read()

 Description:	Read image

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevOpaque	*argout - image
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_read(ImagePro ds,DevLong *argin,DevOpaque *argout,long *error)
{
   int status;

   dprintf("dev_ccd_read(): argin %d",*argin);

   status = dev_get_image(ds,argin,argout,error);

   return(status);
}

/*============================================================================
 Function:      static long dev_ccd_set_exposure()

 Description:	Set exposure time in seconds

 Arg(s) In:	ImagePro 	ds 	- 
		DevFloat  	*argin  - exposure (in seconds)
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_exposure(ImagePro ds,DevFloat *argin,DevVoid *argout,long *error)
{
   long status;

   ds->imagepro.exposure = (long)((*argin)*1000.); /* convert to milliseconds */

   dprintf("dev_ccd_set_exposure(): set exposure to %d ms",ds->imagepro.exposure);

   if (IpAcquSetExposure != NULL)
   {
      status = IpAcquSetExposure(ds->imagepro.exposure);
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_exposure()

 Description:	Get exposure time in seconds

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: DevFloat	*argout - exposure in seconds
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_exposure(ImagePro ds,DevVoid *argin,DevFloat *argout,long *error)
{

   *argout = (float)((ds->imagepro.exposure)/1000.); /* convert to seconds */

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_bin()

 Description:	Set binning

 Arg(s) In:	ImagePro 		ds 	- 
		DevVarLongArray  	*argin  - binning
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_bin(ImagePro ds,DevVarLongArray *argin,DevVoid *argout,long *error)
{
   int status;

   ds->imagepro.hbin = argin->sequence[0];
   ds->imagepro.vbin = argin->sequence[1];

   if (IpAcquSetHorizontalBinning != NULL)
   {
      status = IpAcquSetHorizontalBinning(ds->imagepro.hbin);
   }
   if (IpAcquSetVerticalBinning != NULL)
   {
      status = IpAcquSetVerticalBinning(ds->imagepro.hbin);
   }

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_bin()

 Description:	Get binning

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid		  	*argin  - none
   				  
 Arg(s) Out: 	DevVarLongArray		*argout - binning
  	     	long *error - pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_bin(ImagePro ds,DevVoid *argin,DevVarLongArray *argout,long *error)
{
   static long binning[2];

   binning[0] = ds->imagepro.hbin;
   binning[1] = ds->imagepro.vbin;

   argout->length = 2;
   argout->sequence = binning;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_trigger()

 Description:	Set trigger

 Arg(s) In:	ImagePro 	ds 	- 
		DevLong  	*argin  - trigger
   				  
 Arg(s) Out: DevVoid	*argout - none
  	     long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_trigger(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{

   ds->imagepro.trigger = *argin;

   dprintf("dev_ccd_set_exposure(): set trigger to %d ms",ds->imagepro.trigger);

/* PRESENT VERSION DOES NOTHING ! */

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_trigger()

 Description:	Get trigger

 Arg(s) In:	ImagePro 	ds 	- 
		DevVoid  	*argin  - none
   				  
 Arg(s) Out: 	DevLong		*argout - exposure in seconds
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_trigger(ImagePro ds,DevVoid *argin,DevLong *argout,long *error)
{

   *argout = ds->imagepro.trigger;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_set_file_par()

 Description:	Set file parameters

 Arg(s) In:	ImagePro 		ds 	- 
		DevVarStringArray  	*argin  - file parms
		sequence[0] - absolute root directory
		sequence[1] - filename prefix
		sequence[2] - filename suffix
		sequence[3] - image no.
		sequence[4] - printf() format for image number
		sequence[5] - overwrite flag, y=true, n=false
   				  
 Arg(s) Out: 	DevVoid			*argout - none
  	     	long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_set_file_par(ImagePro ds,DevVarStringArray *argin,DevVoid *argout,long *error)
{

   if (argin->length != 6) {
   	return(DS_NOTOK);
   }

   strcpy(ds->imagepro.ccd_file_dir, argin->sequence[0]);
   strcpy(ds->imagepro.ccd_file_prefix, argin->sequence[1]);
   strcpy(ds->imagepro.ccd_file_suffix, argin->sequence[2]);
   sscanf(argin->sequence[3],"%ld", ds->imagepro.ccd_file_no);
   strcpy(ds->imagepro.ccd_file_no_fmt, argin->sequence[4]);
   strcpy(ds->imagepro.ccd_file_overwrite, argin->sequence[5]);

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_get_file_par()

 Description:	Get file parameters

 Arg(s) In:	ImagePro 		ds 	- 
		DevVoid  		*argin  - none
   				  
 Arg(s) Out: 	DevVarStringArray	*argout - file parms
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_get_file_par(ImagePro ds,DevVoid *argin,DevVarStringArray *argout,long *error)
{
   static char ccd_file_no[256];

   argout->length = 6;
   argout->sequence[0] = ds->imagepro.ccd_file_dir;
   argout->sequence[1] = ds->imagepro.ccd_file_prefix;
   argout->sequence[2] = ds->imagepro.ccd_file_suffix;
   sprintf(ccd_file_no, "%d", ds->imagepro.ccd_file_no);
   argout->sequence[3] = ccd_file_no;
   argout->sequence[4] = ds->imagepro.ccd_file_no_fmt;
   argout->sequence[5] = ds->imagepro.ccd_file_overwrite;

   return(DS_OK);
}

/*============================================================================
 Function:      static long dev_ccd_write_file()

 Description:	Write file

 Arg(s) In:	ImagePro 		ds 	- 
		DevLong  		*argin  - dunno !
   				  
 Arg(s) Out: 	DevVoid			*argout - nothing
  	        long *error - 	pointer to error code, in case routine fails

============================================================================*/

static long dev_ccd_write_file(ImagePro ds,DevLong *argin,DevVoid *argout,long *error)
{

   char file_name[1024];

   sprintf(file_name,"%s\\%s",ds->imagepro.ccd_file_dir,ds->imagepro.ccd_file_prefix);
   sprintf(file_name+strlen(file_name),ds->imagepro.ccd_file_no_fmt,ds->imagepro.ccd_file_no);
   sprintf(file_name+strlen(file_name),".%s",ds->imagepro.ccd_file_suffix);

   return(dev_ws_save_as_edf(ds, (char**)&file_name, NULL, error));
}
